/*
 * confg.c
 *
 * Read and understanding everything about the options 
 * & (dynamic) configuration of a2ps.
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97, 98 Akim Demaille, Miguel Santana
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * $Id: confg.c,v 1.68 1998/03/02 08:56:43 demaille Exp $
 */

/************************************************************************/
/*									*/
/*			I n c l u d e   f i l e s			*/
/*                                                                      */
/************************************************************************/
#include "a2ps.h"
#include "routines.h"
#include "msg.h"
#include "media.h"
#include "getopt.h"
#include "jobs.h"
#include "getshline.h"
#include "pathwalk.h"
#include "confg.h"
#include "useropt.h"
#include "pathconcat.h"
#include "printers.h"
#include "options.h"
#include "metaseq.h"

/*
 * Hooks used
 */
config_hook delegation_hook = NULL;
config_hook toc_entry_hook = NULL;

#define GET_TOKEN(from) (strtok ((from), " \t\n"))
#define GET_LINE_TOKEN(from) (strtok ((from), "\n"))
 
#define CHECK_TOKEN() 							\
  if (token2 == NULL) 							\
    error_at_line (1, 0, fname, firstline,				\
		   _("missing argument for `%s'"), token);

/*
 * Read the configuration file
 */
int
a2_read_config (a2ps_job * job, const char *path, const char *file)
{
  FILE *fp;
  char * fname;
  char *buf = NULL;
  size_t bufsiz = 0;
  char *token, *token2;
  int firstline = 0, lastline = 0;

  fname = path_concat (path, file, NULL);

  fp = fopen (fname, "r");
  if (fp == NULL) {
    FREE (fname);
    return 0;
  }

  message (msg_opt | msg_file, (stderr, "Reading configuration file `%s'\n", fname));

  while (getshline_numbered (&firstline, &lastline, &buf, &bufsiz, fp) != -1)
    {
      token = GET_TOKEN (buf);

      /* Blank line, but not empty */
      if (!token)
	continue;

      if (strequ (token, "Options:"))
	{
	  a2ps_handle_string_options (job, GET_LINE_TOKEN (NULL));
	}
      /* Default PPD file */
      else if (strequ (token, "DefaultPPD:"))
        {
          token2 = GET_LINE_TOKEN (NULL);
          CHECK_TOKEN ();
          a2ps_printers_default_ppdkey_set (job->printers, token2);
        }
      /* Handling of the printers */
      else if (strequ (token, "Printer:"))
	{
	  char * name;

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  name = token2;

	  /* The command must have leading spaces */
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  if (!a2ps_printers_add (job->printers, name, token2))
	    error_at_line (1, 0, fname, firstline,
			   _("invalid definition for printer `%s': %s"),
			   name, token2);
	}
      else if (strequ (token, "UnknownPrinter:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  if (!a2ps_printers_add (job->printers, _("Unknown Printer"), token2))
	    error_at_line (1, 0, fname, firstline,
			   _("invalid definition for printer `%s': %s"),
			   _("Unknown Printer"), token2);
	}
      else if (strequ (token, "DefaultPrinter:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  if (!a2ps_printers_add (job->printers, _("Default Printer"), token2))
	    error_at_line (1, 0, fname, firstline,
			   _("invalid definition for printer `%s': %s"),
			   _("Default Printer"), token2);
	}
      else if (strequ (token, "Delegation:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  /* This is only for a2ps the program.  Read this
	   * only if there's a reader */
	  if (delegation_hook)
	    delegation_hook (fname, firstline, token2);
	}
      else if (strequ (token, "UserOption:"))
	{
	  char * name, * command;

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  name = token2;
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  command = token2;
	  user_option_add (job, name, command);
	}
      else if (strequ (token, "OutputFirstLine:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  xustrcpy (job->status->magic_number, token2);
	}
      else if (strequ (token, "PageLabelFormat:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  xustrcpy (job->status->page_label_format, token2);
	}
      else if (strequ (token, "Medium:")
	       || /* Convenience for enscript users */
	       strequ (token, "Media:"))
	{
	  char *name;
          int w, h, llx, lly, urx, ury;
	  
          token2 = GET_TOKEN (NULL);
          CHECK_TOKEN ();
          name = token2;
	  
          token2 = GET_TOKEN (NULL);
          CHECK_TOKEN ();
          w = atoi (token2);
	  
          token2 = GET_TOKEN (NULL);
          CHECK_TOKEN ();
          h = atoi (token2);
	  
          token2 = GET_TOKEN (NULL);

	  if (token2 != NULL) {
	    /* BBox is also given */
	    llx = atoi (token2);
	    
	    token2 = GET_TOKEN (NULL);
	    CHECK_TOKEN ();
	    lly = atoi (token2);
	    
	    token2 = GET_TOKEN (NULL);
	    CHECK_TOKEN ();
	    urx = atoi (token2);
	    
	    token2 = GET_TOKEN (NULL);
	    CHECK_TOKEN ();
	    ury = atoi (token2);
	  } else {
	    /* A short hand has been used: use 24 points as a margin
	     * all around */
	    llx = lly = 24;
	    urx = w - 24;
	    ury = h - 24;
	  }
          add_medium (job, name, w, h, llx, lly, urx, ury);
	}
      else if (strequ (token, "Variable:")
	       	       || strequ (token, "MacroMetaSequence:"))
	{
	  char * key, * value;
	  
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  key = token2;
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  value = token2;
	  if (!macro_meta_sequence_add (job, key, value))
	    error_at_line (1, 0, fname, firstline,
			   _("invalid macro identifier `%s'"), key);
	}
      /* Handling of the library path */
      else if (strequ (token, "LibraryPath:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  XFREE (job->common.path);
	  job->common.path = pw_string_to_path (token2, PATH_SEP);
	}
      else if (strequ (token, "AppendLibraryPath:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  job->common.path = pw_append_string_to_path (job->common.path,
						       token2, PATH_SEP);
	}
      else if (strequ (token, "PrependLibraryPath:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  job->common.path = pw_prepend_string_to_path (job->common.path,
							token2, PATH_SEP);
	}
      /* Prefered tmp dir */
      else if (strequ (token, "TemporaryDirectory:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  xstrcpy (job->common.tmpdir, token2);
	}
      /* How to call file */
      else if (strequ (token, "FileCommand:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  xstrcpy (job->file_command, token2);
	}
      else if (strequ (token, "Pattern:")
	       || strequ (token, "PassThrough:"))
	{
	  /* TRANS: The following message says that in a2ps.cfg there
	     is an Entry (such as `Pattern:', or `PassThrough:') which
	     is no longer used */
	  error_at_line (0, 0, fname, firstline, 
			 _("obsolete `%s' entry.  Ignored"), token);
	}
      else
	error_at_line (1, 0, fname, firstline,
		       _("illegal option: `%s'"), token);
    }
  XFREE (fname);
  fclose (fp);
  /* BUF has been allocated by getshlinenumbered */
  FREE (buf);
  return 1;
}

/* Global config. 
 *    This is really not an easy thing because, people may want
 *    to check the package before the installation.  The worst
 *    case is when an older a2ps is yet installed.  So we _must_
 *    have a special way to deal with the tests.  This is why
 *    I introduced an env-var: A2PS_CONFIG, which
 *    points to a2ps.cfg
 *    Note that it also improves the robustness of `make distcheck'
 */
void
a2_read_sys_config (a2ps_job * job)
{
  char * config_file;

  if ((config_file = getenv ("A2PS_CONFIG")))
    {
      if (a2_read_config (job, NULL, config_file))
	return;
      error (0, errno,
	     _("cannot open configuration file `%s'"), config_file);
    }
  else
    {
      /* Well, we are not being tested, so the config file should
       * be in SYSCONFDIR.  I see no reason to end a2ps here
       * if the file is not found: other files follow.  Just say it.
       */
      if (a2_read_config (job, SYSCONFDIR, "a2ps.cfg"))
	return;
      config_file = ALLOCA (char, 
			    strlen (SYSCONFDIR) + strlen ("a2ps.cfg") + 2);
      sprintf (config_file, "%s/%s", SYSCONFDIR, "a2ps.cfg");
      error (0, errno,
	     _("cannot open configuration file `%s'"), config_file);
    }
}
