/*
 * Electric(tm) VLSI Design System
 *
 * File: ioecado.c
 * Input/output aid: ecad output
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

/*
 * ----------------- FORMAT ----------------------
 * ADD SO14 'IC1' R0 (0 0)
 * ADD SO16 'IC2' R0 (0 0)
 * ADD SO24L 'IC3' R0 (0 0)
 * ;
 * Signal 'A0'       'IC1'      '5' \
 *                   'IC2'      '10' \
 *                   'IC3'      '8' \
 * ;
 * Signal 'A1'       'IC1'      '6' \
 *                   'IC2'      '11' \
 *                   'IC3'      '5' \
 * ;
 */
/*
 * ----------------- ECAD NETLIST FORMAT ----------------------
 .cmp (t=pkg_type) ref_des (x=0, y=0)


  	.cmp (t=LED) LED1
 	.cmp (t=RES100) R1

Net_name : ref_des pad; \
           ref_des pad;

 	A0 : LED1 1; R1 2;
 	A1 : LED1 2; R1 1;

*/

#include "config.h"
#if IOECAD

#include "global.h"
#include "eio.h"

#define NOECADNET ((ECADNET *)-1)

typedef struct Inextecadnet
{
	char *netname;
	char *packagename;
	char *pinname;
	struct Inextecadnet *nextecadnet;
} ECADNET;

static ECADNET *io_ecadfirstnet;

/* prototypes for local routines */
static void io_ecaddumpfacet(NODEPROTO *curfacet, char *prefix);
static void io_ecadmarkfacets(NODEPROTO *curfacet);

INTSML io_writeecadlibrary(LIBRARY *lib)
{
	char file[100], *truename;
	REGISTER INTBIG i, sorted, maxnetname, maxpackagename;
	REGISTER char *name;
	REGISTER NODEPROTO *curfacet;
	REGISTER ECADNET *en, *lasten, *nexten;
	REGISTER PORTPROTO *pp;

	/* create the proper disk file for the ECAD */
	curfacet = lib->curnodeproto;
	if (curfacet == NONODEPROTO)
	{
		ttyputerr(_("Must be editing a facet to generate ECAD output"));
		return(1);
	}
	(void)strcpy(file, curfacet->cell->cellname);
	(void)strcat(file, ".enl");
	name = truepath(file);
	io_fileout = xcreate(name, io_filetypeecad, _("ECAD File"), &truename);
	if (io_fileout == NULL)
	{
		if (truename != 0) ttyputerr(_("Cannot write %s"), truename);
		return(1);
	}

	/* recurse through the circuit, gathering network connections */
	io_ecadfirstnet = NOECADNET;
	for(pp = curfacet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		pp->temp1 = 0;
	io_ecaddumpfacet(curfacet, "");
	xprintf(io_fileout, "| end of part list\n");  // cm - removed before netlist

	/* sort the signals */
	sorted = 0;
	while (sorted == 0)
	{
		sorted = 1;
		lasten = NOECADNET;
		for(en = io_ecadfirstnet; en != NOECADNET; en = en->nextecadnet)
		{
			nexten = en->nextecadnet;
			if (nexten == NOECADNET) break;
			if (namesame(en->netname, nexten->netname) > 0)
			{
				if (lasten == NOECADNET) io_ecadfirstnet = nexten; else
					lasten->nextecadnet = nexten;
				en->nextecadnet = nexten->nextecadnet;
				nexten->nextecadnet = en;
				en = nexten;
				sorted = 0;
			}
			lasten = en;
		}
	}

	/* determine longest name */
	maxnetname = maxpackagename = 0;
	for(en = io_ecadfirstnet; en != NOECADNET; en = en->nextecadnet)
	{
		i = strlen(en->netname);
		if (i > maxnetname) maxnetname = i;
		i = strlen(en->packagename);
		if (i > maxpackagename) maxpackagename = i;
	}
	maxnetname += 4;
	maxpackagename += 4;

	/* write networks */
	lasten = NOECADNET;
	for(en = io_ecadfirstnet; en != NOECADNET; en = en->nextecadnet)
	{
		if (lasten == NOECADNET || namesame(lasten->netname, en->netname) != 0)
		   {
			if (lasten != NOECADNET)
            xprintf(io_fileout, ";\n");
			xprintf(io_fileout, _("%s : "), en->netname);            // cm - write netname on first line of net
		   }
      else
		   {
			xprintf(io_fileout, "      ");                           // cm - was net name on each line
		   }
		for(i=strlen(en->netname); i<maxnetname; i++)
			xprintf(io_fileout, " ");

		/* write "Part" column */
		xprintf(io_fileout, "%s", en->packagename);                 // cm - removed single quotes
		for(i=strlen(en->packagename); i<maxpackagename; i++)
			xprintf(io_fileout, " ");

		/* write "Pad" column (an attribute with a number) */
		xprintf(io_fileout, "%s \\\n", en->pinname);                   // cm - removed single quotes
//		if (lasten == NOECADNET || namesame(lasten->netname, en->netname) != 0)
//		   {
//			if (lasten != NOECADNET)
 //           xprintf(io_fileout, ";\n");
//         else
//            xprintf(io_fileout, "\\\n");
//         }

      lasten = en;
	}

	if (lasten != NOECADNET) xprintf(io_fileout, ";    |end of net section.\n"); // cm - end of net section

	/* free memory */
	while (io_ecadfirstnet != NOECADNET)
	   {
		en = io_ecadfirstnet;
		io_ecadfirstnet = en->nextecadnet;
		efree(en->netname);
		efree(en->packagename);
		efree(en->pinname);
		efree((char *)en);
	   }

	/* clean up */
	xclose(io_fileout);

	/* tell the user that the file is written */
	ttyputmsg(_("%s written"), truename);
	return(0);
}

/*
 * Routine to write out all facets in the current library that are marked (temp1 != 0).
 */
void io_ecaddumpfacet(NODEPROTO *curfacet, char *prefix)
{
	char *name;
	char line[50], *newprefix, *tempname;
	REGISTER NODEPROTO *np, *subnp, *subcnp;
	REGISTER NETWORK *net;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *varpart, *var, *varpackagetype;
	REGISTER PORTARCINST *pi;
	REGISTER PORTPROTO *pp;
	REGISTER ECADNET *en;

	/* mark those facets that will be included in this dump */
	io_ecadmarkfacets(curfacet);

	/* write the symbols */
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (np->temp1 == 0) continue;

		/* if this is the main entry facet, pickup net names */
		for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
			net->temp1 = 0;
		if (np == curfacet)
		{
			for(pp = curfacet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				if (pp->temp1 == 0) continue;
				pp->network->temp1 = pp->temp1;
			}
		}

		/* dump symbols in this facet */
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			/* see if this node has a part name */
			if (ni->proto->primindex != 0) continue;
			subnp = ni->proto;

			/* ignore recursive references (showing icon in contents) */
			if (subnp->cell == np->cell) continue;

			varpart = getval((INTBIG)ni, VNODEINST, -1, "ATTR_ref_des");   // cm - changed for ecad
			if (varpart == NOVARIABLE)
			{
				subcnp = contentsview(subnp);
				if (subcnp != NONODEPROTO) subnp = subcnp;

				/* compute new prefix for nets in the subfacet */
				(void)initinfstr();
				(void)addstringtoinfstr(prefix);
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
				if (var == NOVARIABLE)
				{
					sprintf(line, "NODE%ld.", (INTBIG)ni);
					(void)addstringtoinfstr(line);
				} else
				{
					(void)addstringtoinfstr((char *)var->addr);
					(void)addtoinfstr('.');
				}
				allocstring(&newprefix, returninfstr(), el_tempcluster);

				/* store net names on ports for use inside the subfacet */
				for(pp = subnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					pp->temp1 = 0;
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				{
					pp = equivalentport(ni->parent, pi->proto, subnp);
					if (pp == NOPORTPROTO || pp->temp1 != 0) continue;
					net = pi->conarcinst->network;
					(void)initinfstr();
					(void)addstringtoinfstr(prefix);
					(void)addstringtoinfstr(net->netname);
					allocstring(&tempname, returninfstr(), io_tool->cluster);
					pp->temp1 = (INTBIG)tempname;
				}

				/* recurse */
				io_ecaddumpfacet(subnp, newprefix);

				/* remark the facets at the current level and clean up memory */
				io_ecadmarkfacets(curfacet);
				efree(newprefix);
				for(pp = subnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					if (pp->temp1 != 0) efree((char *)pp->temp1);
			} else
			{
				/* found package at bottom of hierarchy */
				(void)allocstring(&name, describevariable(varpart, -1, -1), el_tempcluster);
				varpackagetype = getval((INTBIG)ni, VNODEINST, -1, "ATTR_pkg_type");
				if (varpackagetype == NOVARIABLE)
				{
//					xprintf(io_fileout, ".cmp_t %s '%s%s'", ni->proto->cell->cellname,            // cm - changed
//						prefix, name);
					xprintf(io_fileout, "Error no pkg_type attribute for %s%s", prefix, name);
				} else
				{
					xprintf(io_fileout, ".cmp (t=%s) %s%s (x=0, y=0)", describevariable(varpackagetype, -1, -1),
						prefix, name);
				}
				xprintf(io_fileout, "\n");

				/* record all networks */
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				{
					/* get "Pad" information */
					(void)initinfstr();
					(void)addstringtoinfstr("ATTRP_");
					(void)addstringtoinfstr(pi->proto->protoname);
					(void)addstringtoinfstr("_pin");
					var = getval((INTBIG)ni, VNODEINST, -1, returninfstr());
					if (var == NOVARIABLE)
                  {
						/* "Pad" not found on instance: look for it inside of prototype */
						var = getval((INTBIG)pi->proto, VPORTPROTO, -1, "ATTR_pin");
						if (var == NOVARIABLE) continue;
                  }
					/* create a new net connection */
					en = (ECADNET *)emalloc(sizeof (ECADNET), io_tool->cluster);
					if (en == 0) return;
					net = pi->conarcinst->network;
					if (net->temp1 != 0)
					{
						(void)allocstring(&en->netname, (char *)net->temp1, io_tool->cluster);
					} else if (net->namecount > 0)
					{
						(void)initinfstr();
						(void)addstringtoinfstr(prefix);
						(void)addstringtoinfstr(net->netname);
						(void)allocstring(&en->netname, returninfstr(), io_tool->cluster);
					} else
					{
						(void)initinfstr();
						(void)formatinfstr("%sNET%ld", prefix, (INTBIG)net);
						(void)allocstring(&en->netname, returninfstr(), io_tool->cluster);
					}
					(void)initinfstr();
					(void)addstringtoinfstr(prefix);
					(void)addstringtoinfstr(name);
					(void)allocstring(&en->packagename, returninfstr(), io_tool->cluster);
					(void)allocstring(&en->pinname, describevariable(var, -1, -1), io_tool->cluster);
					en->nextecadnet = io_ecadfirstnet;
					io_ecadfirstnet = en;
				}
				efree((char *)name);
			}
		}
	}
}

/*
 * Routine to mark the facets in the current library that should be included
 * along with "curfacet".  The facets are marked by setting their "temp1" field
 * nonzero.  These facets include all that are part of a multipage schematic.
 */
void io_ecadmarkfacets(NODEPROTO *curfacet)
{
	REGISTER NODEPROTO *np;

	/* mark those facets that will be included in this dump */
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->temp1 = 0;
		if ((curfacet->cellview->viewstate&MULTIPAGEVIEW) != 0 ||
			namesamen(curfacet->cellview->viewname, "schematic-page-", 15) == 0)
		{
			/* original facet is a multipage: allow all multipage in this cell */
			if (curfacet->cell != np->cell) continue;
		} else
		{
			/* original is not multipage: allow only it */
			if (curfacet != np) continue;
		}
		np->temp1 = 1;
	}
}

#endif  /* IOECAD - at top */
