/* Copyright 2020 The Chromium OS Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

/*
 * Routines for accessing data objects store in the .init_rom region.
 * Enabled with the CONFIG_CHIP_INIT_ROM_REGION config option. Data
 * objects are placed into the .init_rom region using the __init_rom attribute.
 */

#ifndef __CROS_EC_INIT_ROM_H
#define __CROS_EC_INIT_ROM_H

#include "stdbool.h"

#ifdef CONFIG_CHIP_INIT_ROM_REGION
/**
 * Get the memory mapped address of an .init_rom data object.
 *
 * @param offset	Address of the data object assigned by the linker.
 *			This is effectively a flash offset when
 *			CONFIG_CHIP_INIT_ROM_REGION is enabled, otherwise
 *			it is a regular address.
 * @param size	        Size of the data object.
 *
 * @return Pointer to data object in memory. Return NULL if the object
 * is not memory mapped.
 */
const void *init_rom_map(const void *addr, int size);

/**
 * Unmaps an .init_rom data object. Must be called when init_rom_map() is
 * successful.
 *
 * @param offset	Address of the data object assigned by the linker.
 * @param size	        Size of the data object.
 */
void init_rom_unmap(const void *addr, int size);

/**
 * Copy an .init_rom data object into a RAM location. This routine must be used
 * if init_rom_get_addr() returns NULL. This routine automatically handles
 * locking of the flash.
 *
 * @param offset	Flash offset of the data object.
 * @param size	        Size of the data object.
 * @param data		Destination buffer for data.
 *
 * @return 0 on success.
 */
int init_rom_copy(int offset, int size, char *data);
#else
static inline const void *init_rom_map(const void *addr, int size)
{
	return addr;
}

static inline void init_rom_unmap(const void *addr, int size)
{
}

static inline int init_rom_copy(int offset, int size, char *data)
{
	return 0;
}
#endif

#endif /* __CROS_EC_INIT_ROM_H */
