/*
 * Copyright (C) 1999, 2000, 2001, 2002  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

// generatorfactory.cc

#include "generatorfactory.h"

#include "decorators.h"
#include "tags.h"
#include "keys.h"
#include "linenogen.h"
#include "docgenerator.h"

// global
#include "generators.h"
#include "globalcolormap.h"

GeneratorFactory::GeneratorFactory(bool line_number, const string &t) :
  char_translator (0), line_number_option (line_number), title (t)
{
}

template <class Decorator>
void
GeneratorFactory::decorate_generator(TextGenerator *&decorated)
{
  decorated = new Decorator (decorated, char_translator);
}

template <class Decorator>
void
GeneratorFactory::decorate_generators()
{
  decorate_generator<Decorator> (NormalGenerator);
  decorate_generator<Decorator> (KeywordGenerator);
  decorate_generator<Decorator> (CommentGenerator);
  decorate_generator<Decorator> (StringGenerator);
  decorate_generator<Decorator> (TypeGenerator);
  decorate_generator<Decorator> (NumberGenerator);
  decorate_generator<Decorator> (PreProcGenerator);
  decorate_generator<Decorator> (SymbolGenerator);
  decorate_generator<Decorator> (FunctionGenerator);
  decorate_generator<Decorator> (CBracketGenerator);
}

void
GeneratorFactory::createGenerators()
{
  char_translator = createCharTranslator ();
  MapOfColors = createColorMap ();

  GlobalGenerator = new TextGenerator (char_translator);

  TextGenerator *final_decorator;

  if (line_number_option)
    {
      GlobalGenerator = new LineNumberDecorator (GlobalGenerator, char_translator);
      final_decorator = new TextGenerator (char_translator);
    }
  else
    {
      final_decorator = GlobalGenerator;
    }

  NormalGenerator = createGenerator (final_decorator, NORMAL);
  KeywordGenerator = createGenerator (final_decorator, KEYWORD ) ;
  CommentGenerator = createGenerator (final_decorator, COMMENT ) ;
  StringGenerator = createGenerator (final_decorator, STRING ) ;
  TypeGenerator = createGenerator (final_decorator, TYPE ) ;
  NumberGenerator = createGenerator (final_decorator, NUMBER ) ;
  PreProcGenerator = createGenerator (final_decorator, PREPROC ) ;
  SymbolGenerator = createGenerator (final_decorator, SYMBOL ) ;
  FunctionGenerator = createGenerator (final_decorator, FUNCTION ) ;
  CBracketGenerator = createGenerator (final_decorator, CBRACKET ) ;

  if (line_number_option)
    decorate_generators <LineNumberDecorator> ();

  GlobalDocGenerator = createDocGenerator ();
}

#define GET_DECORATED( d ) ( d ? d : final_dec )

TextGenerator *
GeneratorFactory::createGenerator( TextGenerator *final_dec, const string &key )
{
  Tag *tag = getTag( key ) ;
  TextGenerator *dec = NULL ;

  if ( ! tag ) // no options
    return final_dec ;

  const string &color = tag->GetColor() ;
  if ( color.size () )
    {
     string c;
     if ( color[0] == '#' && color.size () == 7 )
       c = color; // FIXME: handle direct colors not only for HTML
     else
       c = MapOfColors->getColor (color);

     dec = createColorDecorator( final_dec, c ) ;
    }

  if ( tag->IsBold() )
    dec = createBoldDecorator( GET_DECORATED( dec ) ) ;
  if ( tag->IsItalic() )
    dec = createItalicDecorator( GET_DECORATED( dec ) ) ;
  if ( tag->IsUnderline() )
    dec = createUnderlineDecorator( GET_DECORATED( dec ) ) ;

  return GET_DECORATED( dec ) ;
  // There should be some options, but it's not ncessary ...
  // so this is just to be safe
}

DocGenerator *
GeneratorFactory::createDocGenerator()
{
  return new DocGenerator (title);
}
