/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 * @author Nic Eigel
 */

type HashCode = string;
type EddsaPublicKey = string;
type EddsaSignature = string;
type WireTransferIdentifierRawP = string;
type RelativeTime = TalerProtocolDuration;
type ImageDataUrl = string;
type AuditorUserType = "business" | "individual";


export interface WithId {
  id: string;
}

interface Timestamp {
  // Milliseconds since epoch, or the special
  // value "forever" to represent an event that will
  // never happen.
  t_s: number | "never";
}

interface TalerProtocolDuration {
  d_us: number | "forever";
}

interface Duration {
  d_ms: number | "forever";
}

interface WithId {
  id: string;
}

type Amount = string;
type UUID = string;
type Integer = number;

export namespace AuditorBackend {
  interface DepositConfirmation {
    // identifier
    deposit_confirmation_serial_id: number;

    h_contract_terms: string;

    h_policy: string;

    h_wire: string;

    exchange_timestamp: string;

    refund_deadline: string;

    wire_deadline: string;

    total_without_fee: string;

    coin_pubs: string;

    coin_sigs: string;

    merchant_pub: string;

    merchant_sig: string;

    exchange_pub: string;

    exchange_sig: string;

    suppressed: string;

    ancient: string;
  }

  interface Config {
    name: string;
    version: string;
    implementation: string;
    currency: string;
    auditor_public_key: string;
    exchange_master_public_key: string;
  }

  interface ErrorDetail {
    // Numeric error code unique to the condition.
    // The other arguments are specific to the error value reported here.
    code: number;

    // Human-readable description of the error, i.e. "missing parameter", "commitment violation", ...
    // Should give a human-readable hint about the error's nature. Optional, may change without notice!
    hint?: string;

    // Optional detail about the specific input value that failed. May change without notice!
    detail?: string;

    // Name of the parameter that was bogus (if applicable).
    parameter?: string;

    // Path to the argument that was bogus (if applicable).
    path?: string;

    // Offset of the argument that was bogus (if applicable).
    offset?: string;

    // Index of the argument that was bogus (if applicable).
    index?: string;

    // Name of the object that was bogus (if applicable).
    object?: string;

    // Name of the currency than was problematic (if applicable).
    currency?: string;

    // Expected type (if applicable).
    type_expected?: string;

    // Type that was provided instead (if applicable).
    type_actual?: string;
  }

  interface VersionResponse {
    // libtool-style representation of the Merchant protocol version, see
    // https://www.gnu.org/software/libtool/manual/html_node/Versioning.html#Versioning
    // The format is "current:revision:age".
    // Name of the protocol.

    name: "taler-auditor";
    version: string;

    // Default (!) currency supported by this backend.
    // This is the currency that the backend should
    // suggest by default to the user when entering
    // amounts. See currencies for a list of
    // supported currencies and how to render them.
    implementation: string;
    currency: string;
    auditor_public_key: string;
    exchange_master_public_key: string;

    // How services should render currencies supported
    // by this backend.  Maps
    // currency codes (e.g. "EUR" or "KUDOS") to
    // the respective currency specification.
    // All currencies in this map are supported by
    // the backend.  Note that the actual currency
    // specifications are a *hint* for applications
    // that would like *advice* on how to render amounts.
    // Applications *may* ignore the currency specification
    // if they know how to render currencies that they are
    // used with.
    //currencies: { currency: CurrencySpecification };

    // Array of exchanges trusted by the merchant.
    // Since protocol v6.
    // exchanges: ExchangeConfigInfo[];
  }

  export interface TokenResponse {
    null;
  }

  namespace Default {
    interface ObjectResponse {
      object: AnyEntry[];
    }
  }

  namespace AmountArithmeticInconsistency  {

    class ClassAmountArithmeticInconsistency {
      data: AmountArithmeticInconsistencyDetail[];
    }

    interface SummaryResponse {
      amount_arithmetic_inconsistency: AmountArithmeticInconsistencyDetail[];
    }

    interface AmountArithmeticInconsistencyDetail {
      row_id: number;
      operation: string;
      exchange_amount: string;
      auditor_amount: string;
      profitable: boolean;
      suppressed: boolean;
    }
  }

  namespace BadSigLoss  {
    class ClassBadSigLoss {
      data: BadSigLossDetail[];
    }

    interface SummaryResponse {
      amount_arithmetic_inconsistency: BadSigLossDetail[];
    }

    interface BadSigLossDetail
    {
      row_id: number;
      operation: string;
      loss: string;
      operation_specific_pub: string;
      suppressed: boolean;
    }
  }

  namespace Balance {

    class ClassBalance {
      // List of products that are present in the inventory
      data: BalanceDetail[];
    }

    interface SummaryResponse {
      // List of products that are present in the inventory
      balances: BalanceDetail[];
    }

    interface BalanceDetail {
      // identifier
      row_id: number;

      balance_key: string;

      balance_value: string;

      suppressed: boolean;
    }
  }

  namespace ClosureLag  {
    class ClassClosureLag {
      // List of products that are present in the inventory
      data: ClosureLagDetail[];
    }

    interface SummaryResponse {
      // List of products that are present in the inventory
      closure_lags: ClosureLagDetail[];
    }

    interface ClosureLagDetail {
      row_id: number;
      amount: string;
      deadline: number;
      wtid: number;
      account: string;
      suppressed: boolean;
    }
  }

  namespace CoinInconsistency  {
    class ClassCoinInconsistency {
      data: CoinInconsistencyDetail[];
    }

    interface SummaryResponse {
      amount_arithmetic_inconsistency: CoinInconsistencyDetail[];
    }

    interface CoinInconsistencyDetail
    {
      row_id: number;
      operation: string;
      exchange_amount: string;
      auditor_amount: string;
      coin_pub: string;
      profitable: boolean;
      suppressed: boolean;
    }
  }

  namespace DenominationKeyValidityWithdrawInconsistency  {
    class ClassDenominationKeyValidityWithdrawInconsistency {
      data: DenominationKeyValidityWithdrawInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: DenominationKeyValidityWithdrawInconsistencyDetail[];
    }

    interface DenominationKeyValidityWithdrawInconsistencyDetail
    {
      row_id: number;
      operation: string;
      loss: string;
      operation_specific_pub: string;
      suppressed: boolean;
    }
  }

  namespace DenominationPending  {
    class ClassDenominationPending {
      data: DenominationPendingDetail[];
    }

    interface SummaryResponse {
      responseData: DenominationPendingDetail[];
    }

    interface DenominationPendingDetail
    {
      denom_pub_hash: string;
      denom_balance: string;
      denom_loss: string;
      num_issued: number;
      denom_risk: string;
      recoup_loss: string;
      suppressed: boolean;
    }
  }

  namespace DenominationsWithoutSigs  {
    class ClassDenominationsWithoutSigs {
      data: DenominationsWithoutSigsDetail[];
    }

    interface SummaryResponse {
      responseData: DenominationsWithoutSigsDetail[];
    }

    interface DenominationsWithoutSigsDetail
    {
      row_id: number;
      denompub_h: string;
      value: string;
      start_time: number;
      end_time: number;
      suppressed: boolean;
    }
  }

  namespace DepositConfirmation {
    class ClassDepositConfirmation{
      data: DepositConfirmationDetail[];
    }

    interface SummaryResponse {
      responseData: DepositConfirmationDetail[];
    }

    interface DepositConfirmationDetail {
      deposit_confirmation_serial_id: number;
      h_contract_terms: string;
      h_policy: string;
      h_wire: string;
      exchange_timestamp: string;
      refund_deadline: string;
      wire_deadline: string;
      total_without_fee: string;
      coin_pubs: string;
      coin_sigs: string;
      merchant_pub: string;
      merchant_sig: string;
      exchange_pub: string;
      exchange_sig: string;
      suppressed: string;
        ancient: string;
    }
  }

  namespace Emergency  {
    class ClassEmergency{
      data: EmergencyDetail[];
    }

    interface SummaryResponse {
      responseData: EmergencyDetail[];
    }

    interface EmergencyDetail
    {
      row_id: number;
      denompub_h: string;
      denom_risk: string;
      denom_loss: string;
      deposit_start: number;
      deposit_end: number;
      value: string;
    }
  }

  namespace EmergencyByCount  {
    class ClassEmergencyByCount{
      data: EmergencyByCountDetail[];
    }

    interface SummaryResponse {
      responseData: EmergencyByCountDetail[];
    }

    interface EmergencyByCountDetail
    {
      row_id: number;
      denompub_h: string;
      num_issued: number;
      num_known: number;
      risk: string;
      start: number;
      deposit_end: number;
      value: string;
      suppressed: boolean;
    }
  }

  namespace FeeTimeInconsistency  {
    class ClassFeeTimeInconsistency{
      data: FeeTimeInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: FeeTimeInconsistencyDetail[];
    }

    interface FeeTimeInconsistencyDetail
    {
      row_id: number;
      type: string;
      time: string;
      diagnostic: string;
      suppressed: boolean;
    }
  }

  namespace HistoricDenominationRevenue  {
    class ClassHistoricDenominationRevenue {
      data: HistoricDenominationRevenueDetail[];
    }

    interface SummaryResponse {
      responseData: HistoricDenominationRevenueDetail[];
    }

    interface HistoricDenominationRevenueDetail
    {
      denom_pub_hash: string;
      revenue_timestamp: number;
      revenue_balance: string;
      loss_balance: string;
      suppressed: boolean;
    }
  }

  namespace HistoricReserveSummary  {
    class ClassHistoricReserveSummary {
      data: HistoricReserveSummaryDetail[];
    }

    interface SummaryResponse {
      responseData: HistoricReserveSummaryDetail[];
    }

    interface HistoricReserveSummaryDetail
    {
      denom_pub_hash: string;
      revenue_timestamp: number;
      revenue_balance: string;
      loss_balance: string;
      suppressed: boolean;
    }
  }

  namespace MisattributionInInconsistency  {
    class ClassMisattributionInInconsistency {
      data: MisattributionInInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: MisattributionInInconsistencyDetail[];
    }

    interface MisattributionInInconsistencyDetail
    {
      row_id: number;
      amount: string;
      bank_row: number;
      reserve_pub: string;
      suppressed: boolean;
    }
  }

  namespace Progress  {
    class ClassProgress {
      data: ProgressDetail[];
    }

    interface SummaryResponse {
      responseData: ProgressDetail[];
    }

    interface ProgressDetail
    {
      progress_key: string;
      progress_offset: number;
      suppressed: boolean;
    }
  }

  namespace PurseNotClosedInconsistency  {
    class ClassPurseNotClosedInconsistency {
      data: PurseNotClosedInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: PurseNotClosedInconsistencyDetail[];
    }

    interface PurseNotClosedInconsistencyDetail
    {
      row_id: number;
      purse_pub: string,
      amount: string;
      expiration_date: number;
      suppressed: boolean;
    }
  }

  namespace Purses  {
    class ClassPurses {
      data: PursesDetail[];
    }

    interface SummaryResponse {
      responseData: PursesDetail[];
    }

    interface PursesDetail
    {
      auditor_purses_rowid: number;
      purse_pub: string;
      balance: string;
      target: string,
      expiration_date: number;
      suppressed: boolean;
    }
  }

  namespace RefreshesHanging  {
    class ClassRefreshesHanging {
      data: RefreshesHangingDetail[];
    }

    interface SummaryResponse {
      responseData: RefreshesHangingDetail[];
    }

    interface RefreshesHangingDetail
    {
      row_id: number;
      amount: string;
      coin_pub: string;
      suppressed: boolean;
    }
  }

  namespace ReserveBalanceInsufficientInconsistency  {
    class ClassReserveBalanceInsufficientInconsistency {
      data: ReserveBalanceInsufficientInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: ReserveBalanceInsufficientInconsistencyDetail[];
    }

    interface ReserveBalanceInsufficientInconsistencyDetail
    {
      row_id: number;
      reserve_pub: string;
      inconsistency_gain: boolean;
      inconsistency_amount: string;
      suppressed: boolean;
    }
  }

  namespace ReserveBalanceSummaryWrongInconsistency  {
    class ClassReserveBalanceSummaryWrongInconsistency {
      data: ReserveBalanceSummaryWrongInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: ReserveBalanceSummaryWrongInconsistencyDetail[];
    }

    interface ReserveBalanceSummaryWrongInconsistencyDetail
    {
      row_id: number;
      reserve_pub: string;
      exchange_amount: string;
      auditor_amount: string;
      suppressed: boolean;
    }
  }

  namespace ReserveInInconsistency  {
    class ClassReserveInInconsistency {
      data: ReserveInInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: ReserveInInconsistencyDetail[];
    }

    interface ReserveInInconsistencyDetail
    {
      row_id: number;
      amount_exchange_expected: string;
      amount_wired: string;
      reserve_pub: string;
      timestamp: number;
      account: string;
      diagnostic: string;
      suppressed: boolean;
    }
  }

  namespace ReserveNotClosedInconsistency  {
    class ClassReserveNotClosedInconsistency {
      data: ReserveNotClosedInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: ReserveNotClosedInconsistencyDetail[];
    }

    interface ReserveNotClosedInconsistencyDetail
    {
      row_id: number;
      reserve_pub: string;
      balance: string;
      expiration_time: number;
      diagnostic: string;
      suppressed: boolean;
    }
  }

  namespace Reserves  {
    class ClassReserves{
      data: ReservesDetail[];
    }

    interface SummaryResponse {
      responseData: ReservesDetail[];
    }

    interface ReservesDetail
    {
      auditor_reserves_rowid: number;
      reserve_pub: string;
      reserve_balance: string;
      reserve_loss: string;
      withdraw_fee_balance: string;
      close_fee_balance: string;
      purse_fee_balance: string;
      open_fee_balance: string;
      history_fee_balance: string;
      expiration_date: number;
      origin_account: string;
      suppressed: boolean;
    }
  }

  namespace RowInconsistency  {
    class ClassRowInconsistency {
      data: RowInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: RowInconsistencyDetail[];
    }

    interface RowInconsistencyDetail
    {
      row_id: number;
      row_table: string;
      diagnostic: string;
      suppressed: boolean;
    }
  }

  namespace RowMinorInconsistency  {
    class ClassRowMinorInconsistency {
      data: RowMinorInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: RowMinorInconsistencyDetail[];
    }

    interface RowMinorInconsistencyDetail
    {
      row_id: number;
      row_table: string;
      diagnostic: string;
      suppressed: boolean;
    }
  }

  namespace WireFormatInconsistency  {
    class ClassWireFormatInconsistency {
      data: WireFormatInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: WireFormatInconsistencyDetail[];
    }

    interface WireFormatInconsistencyDetail
    {
      row_id: number;
      amount: string;
      wire_offset: string;
      diagnostic: string;
      suppressed: boolean;
    }
  }

  namespace WireOutInconsistency  {
    class ClassWireOutInconsistency{
      data: WireOutInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: WireOutInconsistencyDetail[];
    }

    interface WireOutInconsistencyDetail
    {
      row_id: number;
      destination_account: string;
      expected: string;
      claimed: string;
      suppressed: boolean;
    }
  }
}