;;; $Author: CBmgr $
;;; $Source: /export/kaplan/stable/sun4.os4.1/cb-2.0/src/data/node62.text,v $
;;; $Date: 92/03/16 08:19:56 $
;;; $Revision: 1.0 $
;;; $Locker:  $
;;; This file is part of the
;;;
;;;      Delta Project  (ConversationBuilder)
;;;      Human-Computer Interaction Laboratory
;;;      University of Illinois at Urbana-Champaign
;;;      Department of Computer Science
;;;      1304 W. Springfield Avenue
;;;      Urbana, Illinois 61801
;;;      USA
;;;
;;;      c 1989,1990,1991 Board of Trustees
;;;              University of Illinois
;;;              All Rights Reserved
;;;
;;;      This file is distributed under license and is confidential
;;;
;;;      Author:  Simon Kaplan (kaplan@cs.uiuc.edu)
;;;               Alan Carroll (carroll@cs.uiuc.edu)
;;;
;;;      Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
;;;      Direct enquiries to the project leader please.
;;;

;;; Message Bus functions
;;;
;;; ------------------------------------------------------------------------
;;; NOTE:  Epoch 4.0 (Emacs 18.57) changes what process-status returns.
;;; Hence the value to check for is a list of either 'run or 'open.  Thank
;;; you Free Software Foundation (NOT!)
;;;

(defun mbus:bus-close (process)

"Close the communications channel to the message bus."

  (if (mbus:bus-open-p process) (delete-process process))
)

(defun mbus:bus-open-p (process)

"Returns t if the PROCESS is a currently open network connect, nil
otherwise"

  (and
    (processp process)
    (memq (process-status process) '(run open))
))

;;;

(defun mbus:bus-open (name buff host port notify)

"Open a connection to a message bus.  NAME is the process name.  BUFF
is the process buffer.  HOST, PORT are the hostname and port number of
the bus.  PROPERTY is the property to set when a complete message
arrives."

  (setq buff (get-buffer buff))
  (if (not (resourcep notify)) (setq notify (intern-atom notify)))

  (let ((process (open-network-stream name buff host port)))
    (when (mbus:bus-open-p process)
      (save-excursion (set-buffer buff)
	(widen)
	(erase-buffer)		;clear out any cruft
	(lisp-mode)			;get a better mode for scanning sexps
	(set-marker mbus:*server-mark* (point-min))
	(set-marker
	  (set (make-local-variable 'mbus:*bus-sexp-mark*) (make-marker))
	  (point-min)
	)
	(set (make-local-variable 'mbus:*bus-atom*) notify)
	(set-process-filter process 'mbus:bus-filter)
      )
      (process-send-string process
	(format
	  "\n(clear)\n(id \"Text Server for %s\")\n(accept \".*\" (%s))\n"
	  mbus:*user-name*
	  mbus:*bus-domains*
    )))
    process				;return value
))

;;;
;;; ------------------------------------------------------------------------
;;;

(defun mbus:bus-strip-message (start end)

  "Strip the wrapper from a Bus message. The first argument should be
the starting location of the message, and the second argument the
ending location."

  (save-excursion
    (goto-char end)
    (forward-char -1)			;backup over (assumed) close paren
    (if (looking-at ")") (delete-char 1)) ; if there, delete it

    (goto-char start)
    (when (search-forward "(" nil t)
      ;; This leaves point _after_ the open paren.
      (forward-sexp 2)			;skip domains and tag
      (delete-region start (point))	;get rid of that stuff
)))

;;;
;;; ------------------------------------------------------------------------
;;;

;;; Mbus process filter
;;; Stick the data into the buffer, and then check to see if
;;; there's a complete sexp

(defun mbus:bus-filter (process data)
  "Accept messages from the Message Bus, and execute them when complete"
  (let ((mbus:bus-filter-match-data (match-data)))
    (unwind-protect
      (save-excursion (set-buffer (process-buffer process))
	(goto-char (point-max))
	(insert data)
	(while				; can get multiple messages
	  (let*
	    (
	      (start-loc (progn (goto-char mbus:*bus-sexp-mark*) (point)))
	      (end-loc
		(condition-case errval
		  (scan-sexps (point) 1)
		  ;; error tags
		  (end-of-file 'eof)
		  (invalid-read-syntax 'invalid)
		  (error 'error)
	    )))
	    (if (numberp end-loc)
	      (progn
		;; move marker to the correct location first
		(set-marker mbus:*bus-sexp-mark* end-loc)
		;; strip the bus wrapper
		(mbus:bus-strip-message start-loc end-loc)
		;; Need to tell the dispatcher that it should run.
		(set-property
		  mbus:*bus-atom*
		  (marker-position mbus:*bus-sexp-mark*)
		  (minibuf-screen)
		)
		t			;do it again
	      )
	      ;; else
	      nil			;exit while loop
      ))))
      ;; unwind forms
      (store-match-data mbus:bus-filter-match-data)
)))

;;;
;;; ------------------------------------------------------------------------
;;; General sexp reader. Normally this will move start to be past the just
;;; read sexp, but not if start == current-buffer, because we have to do the
;;; save-excursion.

(defun mbus:bus-safe-read-sexp (start &optional limit)

  "Read an sexp from the marker START, without reading past the
optional LIMIT. Returns the sexp if successfully read, 'eof if an
incomplete sexp, and 'invalid if the sexp has an invalid read syntax.
START is moved past the read portion of the buffer."

  (if (not (markerp start)) (setq start (get-buffer start)))
  (save-excursion
    (set-buffer (if (bufferp start) start (marker-buffer start)))
    (save-restriction
      (if limit (narrow-to-region (point-min) limit))
      (condition-case errval
	;; form
	(read start)
	;; error tags
	(end-of-file 'eof)
	(invalid-read-syntax 'invalid)
	(error 'error)
))))

;;;
;;; ------------------------------------------------------------------------
;;; The goal here is to be able to move sexp's from the message bus buffer
;;; into some other buffer without reading the sexp. This is desirable, for
;;; instance, if we are moving text from the server into a node.
;;;

(defun mbus:bus-move-sexp (src dst &optional limit post-process)

  "Move an sexp from marker SOURCE to marker DESTINATION. An optional
LIMIT constrains the amount of text removed from the SOURCE buffer.
After being moved, the optional function POST is called with two
arguments, the start and end of the moved sexp in the DESTINATION
buffer, with that buffer current.

This function is more efficient than using mbus:bus-read-sexp and insert."

  ;; I tried to generalize the arguments, but it's too much hassle, so
  ;; screw it.
  (if (not (markerp src)) (setq src (get-buffer src)))
  (if (not (markerp dst)) (setq dst (get-buffer dst)))

  (let
    (
      (src-buff (if (markerp src) (marker-buffer src) src))
      (dst-buff (if (markerp dst) (marker-buffer dst) dst))
    )
    (save-excursion (set-buffer src-buff)
      (let*
	(
	  (start-loc (if (markerp src) (marker-position src) (point)))
	  (end-loc
	    (condition-case errval
	      (scan-sexps start-loc 1)
	      ;; error tags
	      (end-of-file 'eof)
	      (invalid-read-syntax 'invalid)
	      (error 'error)
	)))
	(cond
	  ((numberp end-loc)		;successfully scanned sexp
	    (set-buffer dst-buff)
	    (save-excursion
	      (if (markerp dst) (goto-char dst))
	      (let ((here (point)))
		(insert-buffer-substring src-buff start-loc end-loc)
		(if post-process
		  (funcall post-process here (point))
	    )))
	    (set-buffer src-buff)
	    (delete-region start-loc end-loc)
	    t				;return value
	  )
	  (t end-loc)			;return value
)))))
;;;
;;; ------------------------------------------------------------------------
;;;

(defun mbus::replace-string (this that &optional buff)

  "Similar to replace-string, but without the annoying messages.
Replaces every occurence of FROM with TO in the optional BUFFER.
Preserves the match data."

  (save-excursion (if buff (set-buffer buff))
    (let
      (
	(save-match-data (match-data))	;preserve this
      )
      (goto-char (point-min))
      (while (search-forward this nil t)
	(goto-char (match-end 0))
	(delete-region (match-beginning 0) (match-end 0))
	(insert that)
      )
      (store-match-data save-match-data)
)))

(defun mbus:quotify-region (&optional start end buffer)

"Change the text from START to END in BUFFER to be quoted. This
modifies the buffer in place. START defaults to the start of the
buffer, END to end of buffer, and BUFFER to the current buffer."

  (or (and buffer (get-buffer buffer)) (setq buffer (current-buffer)))
  (save-excursion
    (set-buffer buffer)
    (or start (setq start (point-min)))
    (or end (setq end (point-max)))
    (save-restriction
      (narrow-to-region start end)
      (goto-char (point-min))
      (mbus::replace-string "\\" "\\\\")
      (goto-char (point-min))
      (mbus::replace-string "\"" "\\\"")
      (goto-char (point-min))
      (insert "\"")
      (goto-char (point-max))
      (insert "\"")
)))

(defun mbus:unqotify-region (&optional start end buffer)

"Change the text from START to END in BUFFER to be unquoted. This
modifies the buffer in place. START defaults to the start of the
buffer, END to end of buffer, and BUFFER to the current buffer."

  (or (and buffer (get-buffer buffer)) (setq buffer (current-buffer)))
  (save-excursion
    (set-buffer buffer)
    (or start (setq start (point-min)))
    (or end (setq end (point-max)))
    (save-restriction
      (narrow-to-region start end)
      (goto-char (- (point-max) 1))
      (if (looking-at "\"") (delete-char 1))
      (goto-char (point-min))
      (if (looking-at "\"") (delete-char 1))
      (mbus::replace-string "\\\"" "\"")
      (goto-char (point-min))
      (mbus::replace-string "\\\\" "\\")
)))
;;;
;;; ------------------------------------------------------------------------
;;;
