/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:	 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <mbus/api.h>
#include <mbus/keyword.h>
#include "mbus-parse.h"
#include "extern.h"


#define ADD 0
#define UPDATE 1
#define REMOVE 2
#define TYPE 3
#define COLOR 4
#define FONT 5
#define CLEARCOMMON 6
#define CLEARNODE 7
#define CLEAREDGE 8
#define TABLE 9
#define NOCLOBBER 10

static struct keyword_entry_struct DisplayTableKeywords[] =
{
  { ":add", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":update", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":remove", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":type", NULL, KEYWORD_COOKED},
  { ":color", NULL, KEYWORD_COOKED},
  { ":font", NULL, KEYWORD_COOKED},
  { ":clearcommon", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":clearnodes", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":clearedges", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":table", NULL, KEYWORD_COOKED},
  { ":noclobber", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
};

/*
 *  FreeFont takes a display table pointer and a widget (which is just the
 *  toplevel widget for now), and frees the font in that structure.
*/
void
FreeFont(widget, display_ptr)
     Widget widget;
     struct DisplayTable *display_ptr;
{
  Display *dpy;

  dpy = XtDisplay(widget);
  XFreeFont(dpy, display_ptr->font);
  display_ptr->font = NULL;
}

/*
 *  ClearDisplayTable takes a display table pointer as an argument and
 *  frees all of the space allocated to that table.  It then sets the
 *  table to NULL.
*/
void
ClearDisplayTable(table)
     struct DisplayTable **table;
{
  struct DisplayTable *entry, *tmp;

  for (entry = *table; entry; ) {
    tmp = entry->next;
    free(entry->type);
    free(entry);
    entry = tmp;
  }
  *table = NULL;
}

/*
 *  FindType takes a pointer to a display table, and a type, and tries to
 *  find whether or not the type exists in the table or not.  If so, then a
 *  pointer to the structure containing the type in returned.  Else, NULL is
 *  returned.
*/
struct DisplayTable *
FindType(type, flag)
     char *type;
     short flag;
{
  struct DisplayTable *entry;

/*
 *  Check the graph dependent tables.  Use the flag to determine what
 *  tables to look in.  Notice that the common table always gets checked.
*/
  if (flag & NODEMASK) {
    for (entry = graph->display_table->nodes; entry; entry = entry->next)
      if (!strcmp(type, entry->type))
	return entry;
  }
  else if (flag & EDGEMASK)
    for (entry = graph->display_table->edges; entry; entry = entry->next)
      if (!strcmp(type, entry->type))
	return entry;
/*
 *  Now, look at the common table
*/
  for (entry = graph->display_table->common; entry; entry = entry->next)
    if (!strcmp(type, entry->type))
      return entry;
/*
 *  Check the browser wide tables.
*/
  if (flag & NODEMASK) {
    for (entry = BrowserDisplays->nodes; entry; entry = entry->next)
      if (!strcmp(type, entry->type))
	return entry;
  }
  else if (flag & EDGEMASK)
    for (entry = BrowserDisplays->edges; entry; entry = entry->next)
      if (!strcmp(type, entry->type))
	return entry;
/*
 *  Look at the browser wide common table
*/
  for (entry = BrowserDisplays->common; entry; entry = entry->next)
    if (!strcmp(type, entry->type))
      return entry;

  return NULL;
}

/*
 *  The function InDisplayList determines if a type is contained in
 *  a certain table.  The table and type are passed in, and if the
 *  type exists, a pointer to that structure is returned, otherwise
 *  NULL is returned.
*/

struct DisplayTable *
InDisplayList(type, table)
     char *type;
     struct DisplayTable *table;
{
  struct DisplayTable *entry;

  for (entry = table; entry; entry = entry->next) {
    if (!strcmp(type, entry->type))
      return entry;
  }
  return NULL;
}

/*
 *  GetColor takes a character string that is the color that the user
 *  desires and a widget (which is always the toplevel widget for us
 *  at least for now, and returns the unsigned long integer representing
 *  the color that the user has selected.  If, for whatever reason, the
 *  color cannot be parsed, then the default foreground color is returned.
*/
unsigned long
GetColor (widget, color)
     Widget widget;
     char *color;
{
  XColor c;
  Colormap Cmap;
  Display *dpy;

  dpy = XtDisplay(widget);

  Cmap = DefaultColormap(dpy, DefaultScreen(dpy));
  
  if (color && XParseColor(dpy, Cmap, color, &c)) {
    c.flags = DoRed + DoGreen + DoBlue;
    if (!XAllocColor(dpy,Cmap,&c))
	return graph->foreground;
  }
  if (c.pixel == graph->background)
  {
    return graph->foreground;
  }
  return c.pixel;
}

/*
 *  The routine GetFont takes a character string that is the name of
 *  a font, and tries to get a font structure for this font.
*/
XFontStruct *
GetFont(widget, font)
     Widget widget;
     char *font;
{
  Display *dpy;
  XFontStruct *new_font = NULL;

  dpy = XtDisplay(widget);
  if (font)
      new_font = XLoadQueryFont(dpy, font);
  if (new_font == NULL)
      return graph->font;
  else
      return new_font;
}

void
AddEntry(type, color_name, font_name, color, font, noclobber, table)
     char *type, *color_name, *font_name;
     unsigned long color;
     XFontStruct *font;
     int noclobber;
     struct DisplayTable **table;
{
  struct DisplayTable *new_entry;

  new_entry = (struct DisplayTable *)malloc(sizeof(struct DisplayTable));
  new_entry->type = type;
  new_entry->color_name = color_name;
  new_entry->font_name = font_name;
  new_entry->noclobber = noclobber;
  new_entry->color = color;
  new_entry->font = font;
  new_entry->next = *table;
  *table = new_entry;
}  

void
RemoveEntry(entry, table)
     struct DisplayTable *entry, **table;
{
  struct DisplayTable *tmp;

  if (*table == entry)
    *table = entry->next;
  else {
    for (tmp = *table; tmp->next != entry; tmp = tmp->next) ;
    tmp->next = entry->next;
  }
  free(entry->type);
  if (entry->color_name)
    free(entry->color_name);
  if (entry->font_name) {
    free(entry->font_name);
    FreeFont(toplevel_widget, entry);
  }
  free(entry);
}


/*
 *  AddDisplay is a routine that takes a s-exp that is a list of displays
 *  and creates a structure that represents that display structure.
 *  The s-exp is passed in along with a pointer that will point to the new
 *  structure that is created.  This way, we can pass in either the pointer
 *  to the browser's type-color entries, or a pointer to an individual graph's
 *  entries.
*/
int
AddDisplay(display_list, table, noclobber, ws)
     struct mb_object *display_list;
     struct DisplayTable **table;
     short noclobber, ws;
{
  int change_made = FALSE;
  struct mb_object *d_entry;
  char *type, *color, *font;
  unsigned long color_val;
  struct DisplayTable *new_entry, *displayptr;
  XFontStruct *new_font;

  for (; MBconsp(display_list); display_list = MBcdr(display_list)) {
    d_entry = MBcar(display_list);
    type = MBCstring(MBnth(d_entry, 0));
    MBparse_keywords(MBnthcdr(d_entry,1), DisplayTableKeywords,
		     sizeof(DisplayTableKeywords) /
		     sizeof(struct keyword_entry_struct));
/*
 *  Get the color and font from the keywords.  We use AssignStrVal so that
 *  the space will get malloced properly.
*/
    AssignStrVal(&color, DisplayTableKeywords[COLOR].result, "");
    AssignStrVal(&font, DisplayTableKeywords[FONT].result, "");
/*
 *  First, try and locate the type in the display table so we can check to
 *  see if we need to update or remove this display entry from the proper
 *  table
*/
    if ((displayptr = InDisplayList(type, *table))) {
/*
 *  Might as well check to be sure that we are not trying to update or
 *  remove an entry that can't be clobbered.
*/
      if ((displayptr->noclobber == TRUE) && (ws == FALSE))
	break;
/*
 *  Okay.  check for update first.
*/
      if ((t_keyword_flag)DisplayTableKeywords[UPDATE].result ==
	  KEYWORD_TRUE) {
/*
 *  Get the color value and change it if neccessary
*/
	if (color)
	{
	  color_val = GetColor(toplevel_widget, color);
	  if (color_val != displayptr->color) {
	    change_made = TRUE;
	    displayptr->color = color_val;
	    displayptr->color_name = color;
	  }
	}
/*
 *  Now we check the font.
*/
	if (font)
	{
	  new_font = GetFont(toplevel_widget, font);
	  if (new_font != NULL) {
	    change_made = TRUE;
	    displayptr->font = new_font;
	    displayptr->font_name = font;
	  }
	}
      }
      /*
       *  Here, we check to see if we need to remove the display table entry.
       */
      else if ((t_keyword_flag)DisplayTableKeywords[REMOVE].result ==
	       KEYWORD_TRUE) {
/*
 *  Find the structure in the table, and remove it.
*/
	change_made = TRUE;
	RemoveEntry(displayptr, table);
      }
      free(type);			/* Get rid of the alloced space */
    }
    else {				/* Didn't find the type */
      if ((t_keyword_flag)DisplayTableKeywords[ADD].result ==
	  KEYWORD_TRUE) {
	AddEntry(type, color, font,
		 GetColor(toplevel_widget, color),
		 GetFont(toplevel_widget, font), noclobber, table);
	change_made = TRUE;
      }
    }
  }
  return(change_made);
}

int
ParseDisplayTable(color_def, common_list, node_list, edge_list, noclobber, ws)
     struct mb_object *color_def;
     struct DisplayTable **common_list, **node_list, **edge_list;
     short noclobber, ws;
{
  extern Widget toplevel_widget;
  Display *dpy;
  int change_made = FALSE;
  
  struct mb_object *node_display, *edge_display, *common_display;

  dpy = XtDisplay(toplevel_widget);
  
  common_display = MBnth(color_def, 0);
  node_display = MBnth(color_def, 1);
  edge_display = MBnth(color_def, 2);
  MBparse_keywords(MBnthcdr(color_def,3),DisplayTableKeywords,
		   sizeof(DisplayTableKeywords) /
		   sizeof(struct keyword_entry_struct));
/*
 *  Check and see if any of the keywords to clear out the color table
 *  entries are present.
*/
  if ((t_keyword_flag)DisplayTableKeywords[CLEARCOMMON].result == KEYWORD_TRUE)
  {
    change_made = TRUE;
    ClearDisplayTable(common_list);
  }
  if ((t_keyword_flag)DisplayTableKeywords[CLEARNODE].result == KEYWORD_TRUE)
  {
    change_made = TRUE;
    ClearDisplayTable(node_list);
  }
  if ((t_keyword_flag)DisplayTableKeywords[CLEAREDGE].result == KEYWORD_TRUE)
  {
    change_made = TRUE;
    ClearDisplayTable(edge_list);
  }
  
/*
 *  Now, apply the given type-color pairs to the proper lists.
*/
  if (AddDisplay(common_display, common_list, noclobber, ws))
    change_made = TRUE;
  if (AddDisplay(node_display, node_list, noclobber, ws))
    change_made = TRUE;
  if (AddDisplay(edge_display, edge_list, noclobber, ws))
    change_made = TRUE;
/*
 * If we have nodes in the graph, then we return the value of change_made
 * otherwise, it doesn't matter what we did.  I have to make a change in this
 *  because graph will not always be defined.  That is, the first time through,
 *  when the browsers first get started up, then graph=NULL, and this line
 *  segmentation faults.  Ouch.  I really don't like to do this, but I really
 *  think that it is needed.   
*/
  if (graph != NULL)
    return (graph->v ? change_made : FALSE);
  else
    return FALSE;
}

void
ParseTableEntry(definition)
     struct mb_object *definition;
{
  char *table_type, *type, *color, *font;
  int noclobber;
  struct DisplayTable **table, *entry, *tmp;
  
  MBparse_keywords(definition,DisplayTableKeywords,
		   sizeof(DisplayTableKeywords) /
		   sizeof(struct keyword_entry_struct));

  type = DisplayTableKeywords[TYPE].result;
  color = DisplayTableKeywords[COLOR].result;
  font = DisplayTableKeywords[FONT].result;
  noclobber = ((t_keyword_flag)DisplayTableKeywords[NOCLOBBER].result ==
	       KEYWORD_TRUE);
  table_type = DisplayTableKeywords[TABLE].result;

  if (!strcmp(table_type, "graph-common"))
    table = &(graph->display_table->common);
  else if (!strcmp(table_type, "graph-edges"))
    table = &(graph->display_table->edges);
  else if (!strcmp(table_type, "graph-nodes"))
    table = &(graph->display_table->nodes);
  else if (!strcmp(table_type, "browser-common"))
    table = &(BrowserDisplays->common);
  else if (!strcmp(table_type, "browser-edges"))
    table = &(BrowserDisplays->edges);
  else if (!strcmp(table_type, "browser-nodes"))
    table = &(BrowserDisplays->nodes);
  else
    return;

  entry = InDisplayList(type, *table);
  
  if (entry &&
      ((t_keyword_flag)DisplayTableKeywords[REMOVE].result == KEYWORD_TRUE)) {
    RemoveEntry(entry, table);
    RemoveDisplayEntry(table_type, type);
  }

  if (!entry &&
      ((t_keyword_flag)DisplayTableKeywords[ADD].result == KEYWORD_TRUE)) {
    AddEntry(type, color, font, GetColor(toplevel_widget, color),
	     GetFont(toplevel_widget, font), noclobber, table);
    AddDisplayEntry(*table, table_type);
  }
}
