/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	Author:  Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

#include "graph.h"
#include "extern.h"
#include "elision.h"
#include "edge.h"

int
EdgeConnects (edge, node, mask)
  Edge *edge;
  Vertex *node;
  int mask;
{
  Link *link;
  for (link = edge->links; link != NULL; link = link->n_edge_adj_node) {
    if ((mask & TO_NODE) && (link->tnode == node))
    {
      return TRUE;
    }
    if ((mask & FROM_NODE) && (link->fnode == node))
    {
      return TRUE;
    }
  }
  return FALSE;
}

/*
 * This routine actually does several things:
 * 1) It removes all edges that no longer point to any nodes.  When
 *    we are removing nodes, we don't make any checks for this, thus
 *    it is possible to end up with edges going nowhere.
 * 2) It computes the drawing position of all the edges.
 */
RecomputeEdgePos ()
{
  Edge *edge, *prev = NULL, *tmp;
  EdgeList *edge_list, *edge_ptr;
  coord_t EdgeOffset;
  int edge_count;
  
  edge = graph->edges;
/*
 *  This is the code that cleans up the edge list.  We do this here
 *  because it is more efficient to do so.
*/
  while (edge) {
    if (edge->node_count < 2)
    {
      tmp = edge->next;
      if (!prev)
	graph->edges = edge->next;
      else
	prev->next = edge->next;
      FreeEdge(edge);
      edge = tmp;
    }
    else {
      edge->offset = 0.0;	/* Set the offsets to known values */
      prev = edge;
      edge = edge->next;
    }
  }
  edge = graph->edges;
  graph->have_hyper_edge = FALSE;
  for (; edge; edge = edge->next)
  {
    if (EdgeHidden(edge))
	continue;
    if (edge->offset == 0.0)	/* Haven't seen this edge yet */
    {
      /* Check if we have a hyper edge in the graph.  If not, some computations
       * in GraphReassign don't have to happen.
       */
      if (edge->node_count > 2) graph->have_hyper_edge = TRUE;
      FindSameEdges(edge, &edge_list, &edge_count);
      edge_ptr = edge_list;
      if (edge_count & 1)
      {
	/* Nothing to do.  If it is an odd count, we know that edge->offset
	 * is 0.0 so it is the edge running between the two nodes directly
	 * (i.e., no arc).
	 */
	edge_ptr = edge_ptr->next;
      }
      EdgeOffset = ARC_OFFSET;
      /* Now for all the even number of edges */
      while (edge_ptr)
      {
	/* One above */
	edge_ptr->edge->offset = EdgeOffset;
	edge_ptr = edge_ptr->next;
	/* One below */
	edge_ptr->edge->offset = -EdgeOffset;
	edge_ptr = edge_ptr->next;
	EdgeOffset += ARC_OFFSET;
      }
      FreeEdgeList(&edge_list);
    }
  }
}  

/*
 * EdgeSelected will return TRUE or FALSE depending on whether the Edge
 * is selected or not.
 */
int
EdgeSelected(edge)
  Edge *edge;
{
  return ((edge->values->status & SELECTED) ? TRUE : FALSE);
}
  
/*
 * SelectEdge will select the Edge.
 */
void
SelectEdge(edge)
  Edge *edge;
{
  if (edge)
      edge->values->status |= SELECTED;
}

/*
 * UnselectEdge will mark a Edge as not selected.
 */
void UnselectEdge(edge)
  Edge *edge;
{
  if (edge)
      edge->values->status &= ~SELECTED;
}

/*
 * EdgeMarked will return TRUE or FALSE depending on whether the edge
 * is markedor not.
 */
int
EdgeMarked(edge, mark)
  Edge *edge;
  int mark;
{
  return ((edge->values->status & mark) ? TRUE : FALSE);
}

/*
 * MarkEdge will mark the edge.
 */
void
MarkEdge(edge, mark)
  Edge *edge;
  int mark;
{
  if (edge)
      edge->values->status |= mark;
}

/*
 * UnmarkEdge will mark a edge as not marked.
 */
void UnmarkEdge(edge, mark)
  Edge *edge;
  int mark;
{
  if (edge)
      edge->values->status &= ~mark;
}

/*
 * EdgeHidden will return TRUE or FALSE depending on whether the edge
 * is hidden or not.
 */
int
EdgeHidden(edge)
  Edge *edge;
{
  return ((edge->values->status & HIDDEN) ? TRUE : FALSE);
}

/*
 * HideEdge will Hide the edge.
 */
void
HideEdge(edge)
  Edge *edge;
{
  if (edge) {
    edge->values->status |= HIDDEN;
    UnmarkEdge(edge, MARKED | VISITED);
  }
}

/*
 * UnhideEdge will mark a edge as not hidden.
 */
void UnhideEdge(edge)
  Edge *edge;
{
  if (edge) 
      edge->values->status &= ~HIDDEN;
}


