/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:	 Mark Allender (allender@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

#include <stdio.h>
#include <string.h>
#include <X11/X.h>
#include <mbus/api.h>
#include <mbus/keyword.h>
#include "extern.h"
#include "graph.h"
#include "callback.h"
#include "elision.h"

/*
 *  GetPolicyValue takes a policy name, and returns in appropriate value.
*/
int
GetPolicyValue(policy)
     char *policy;
{
  if (policy == NULL)
    return -1;
  if (!strcmp(policy, "hide_edge"))
    return 0;
  if (!strcmp(policy, "connect_to_sn"))
    return 1;
  if (!strcmp(policy, "hide_node"))
    return 2;
  if (!strcmp(policy, "hide_subgraph"))
    return 3;
  if (!strcmp(policy, "unhide_node"))
    return 4;
  if (!strcmp(policy, "unhide_subgraph"))
    return 5;
  if (!strcmp(policy, "hide_in_from_node"))
    return 2;
  if (!strcmp(policy, "hide_in_to_node"))
    return 4;
  return -1;
}

/*
 *  This routine parses out the elision policy.  It takes three arguments
 *  corresponding with the polciy to follow for edges inbound to an elision
 *  region, outbound edges from a region, and undirected edge to and from
 *  a region.
*/
void
ParseElision(inbound, infinal, outbound, outfinal, undirected, unfinal, label)
     t_sexp inbound, infinal, outbound, outfinal, undirected, unfinal, label;
{
  short elision_policy;
  char *policy_name, *label_policy;

/*
 *  get the string containing the name and lookup the policy name.  We will
 *  then and with certain masks to determine if the policy value that was
 *  returned is valid for that elision type.  Notice that for a mask value
 *  we use the value of the policy that contains 1's in all of the bit
 *  locations that correspond to the policy we are currently parsing.
*/
  label_policy = MBCstring(label);
  if (label_policy) {
    if (!strcmp (label_policy, "query_for_label"))
      graph->label_policy = QUERY_FOR_LABEL;
    else
      graph->label_policy = ROOT_IS_LABEL;
  }

  policy_name = MBCstring(inbound);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name);
    if ((elision_policy >= 0) && !(elision_policy == HIDE_NODE) &&
	!(elision_policy == HIDE_SUB)) {
      graph->elision_policy &= (OUTBOUND | UNDIRECTED | INFINAL | OUTFINAL |
				UNFINAL);
      graph->elision_policy |= elision_policy;
    }
  }
  free(policy_name);
  
  policy_name = MBCstring(outbound);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name) << 3;
    if ((elision_policy >= 0) && !(elision_policy == UNHIDE_NODE) &&
	!(elision_policy == UNHIDE_SUB)) {
      graph->elision_policy &= (INBOUND | UNDIRECTED | INFINAL | OUTFINAL |
				UNFINAL);
      graph->elision_policy |= elision_policy;
    }
  }
  free(policy_name);
  
  policy_name = MBCstring(undirected);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name) << 6;
    if (elision_policy >= 0) {
      graph->elision_policy &= (INBOUND | OUTBOUND | INFINAL | OUTFINAL |
				UNFINAL);
      graph->elision_policy |= elision_policy;
    }
  }
  free(policy_name);

  policy_name = MBCstring(infinal);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name) << 9;
    if (elision_policy >= 0) {
      graph->elision_policy &= (INBOUND | OUTBOUND | UNDIRECTED | OUTFINAL |
				UNFINAL);
      graph->elision_policy |= elision_policy;
    }
  }
  free(policy_name);

  policy_name = MBCstring(outfinal);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name) << 10;
    if (elision_policy >= 0) {
      graph->elision_policy &= (INBOUND | OUTBOUND | UNDIRECTED | INFINAL |
				UNFINAL);
      graph->elision_policy |= elision_policy;
    }
  }
  free(policy_name);

  policy_name = MBCstring(unfinal);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name) << 11;
    if (elision_policy >= 0) {
      graph->elision_policy &= (INBOUND | OUTBOUND | UNDIRECTED | INFINAL |
				OUTFINAL);
      graph->elision_policy |= elision_policy;
    }
  }
  free(policy_name);
}  

/*
 *  This routine parses out the antielision policy.  It takes three arguments
 *  corresponding with the polciy to follow for edges inbound to an elision
 *  region, outbound edges from a region, and undirected edge to and from
 *  a region.
*/
void
ParseAntiElision(inbound, outbound, undirected, diff_set)
     t_sexp inbound, outbound, undirected, diff_set;
{
  short elision_policy;
  char *policy_name;
/*
 *  get the string containing the name and lookup the policy name.  We will
 *  then and with certain masks to determine if the policy value that was
 *  returned is valid for that elision type.  Notice that for a mask value
 *  we use the value of the policy that contains 1's in all of the bit
 *  locations that correspond to the policy we are currently parsing.
*/
  policy_name = MBCstring(inbound);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name);
    if ((elision_policy >= 0) && !(elision_policy == HIDE_NODE) &&
	!(elision_policy == HIDE_SUB)) {
      graph->unelision_policy &= (0x38 | 0x1c0 | 0xe00);
      graph->unelision_policy |= elision_policy;
    }
  }
  free(policy_name);
  
  policy_name = MBCstring(outbound);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name) << 3;
    if ((elision_policy >= 0) && !(elision_policy == UNHIDE_NODE) &&
	!(elision_policy == UNHIDE_SUB)) {
      graph->unelision_policy &= (0x07 | 0x1c0 | 0xe00);
      graph->unelision_policy |= elision_policy;
    }
  }
  free(policy_name);
  
  policy_name = MBCstring(undirected);
  if (policy_name) {
    elision_policy = GetPolicyValue(policy_name) << 6;
    if (elision_policy >= 0) {
      graph->unelision_policy &= (0x07 | 0x38 | 0xe00);
      graph->unelision_policy |= elision_policy;
    }
  }
  free(policy_name);
/*
 *  This is kind of gross code that will parse out the antielision mechansism
 *  pertaining to an edge that points between nodes in different elision sets.
 *  We just do this here because of the special case that is required
 *  when we have an edge like this.  There are still no mechanisms that deal
 *  with undirected edges which fall into this category.
*/
  policy_name = MBCstring(diff_set);
  if (policy_name) {
    if (!strcmp(policy_name, "connect_to_sn"))
      elision_policy = 1 << 9;
    else if (!strcmp(policy_name, "hide_in_from_node"))
      elision_policy = 2 << 9;
    else if (!strcmp(policy_name, "hide_in_to_node"))
      elision_policy = 4 << 9;
    else if (!strcmp(policy_name, "connect & hide_in_from_node"))
      elision_policy = 3 << 9;
    else if (!strcmp(policy_name, "connect & hide_in_to_node"))
      elision_policy = 5 << 9;
    else
      elision_policy = 2 << 9;
    graph->unelision_policy &= (0x07 | 0x38 | 0x1c0);
    graph->unelision_policy |= elision_policy;
  }
  free(policy_name);
}  

/*
 *  The routine HideSubgraph take a supernode, and a list of node and
 *  edges that are to be hidden under this supernode.
*/
void
HideSubgraph(super_node, nlist, elist)
     Vertex *super_node;
     NodeList *nlist;
     EdgeList *elist;
{
  struct SuperNodeList *snlist;
  NodeList *nodes;
  EdgeList *edges;

/*
 *  First, find the location of the supernode in the graph's list of
 *  supernodes.
*/
  for (snlist = graph->super_nodes; snlist; snlist = snlist->next)
    if (snlist->super_node == super_node)
      break;

  if (!snlist)
    return;
/*
 *  We will then attach the list of nodes and edges passed
 *  in onto this list.  What we really want to do though, is append
 *  these lists together because we may be adding some edges or some
 *  nodes into an already existing supernode that already contains nodes
 *  and/or edges.
*/
  MergeEdgeLists(&(snlist->edges), elist);
  MergeNodeLists(&(snlist->nodes), nlist);
/*
 *  Mark all of the nodes and edges as hidden
*/
  for (nodes = nlist; nodes != NULL; nodes = nodes->next) {
    HideNode(nodes->node);
    nodes->node->values->super_node = super_node;
  }

  for (edges = elist; edges != NULL; edges = edges->next) {
    HideEdge(edges->edge);
    edges->edge->values->super_node = super_node;
  }
}

/*
 *  The routine UnHideSubgraph takes a supernode, and a list of nodes and
 *  edges as parameters.  Those nodes and edges are then removed from
 *  the supernode's node and edge lists.
*/
void
UnHideSubgraph(super_node, nlist, elist)
     Vertex *super_node;
     NodeList *nlist;
     EdgeList *elist;
{
  NodeList *nodes;
  EdgeList *edges;
  Vertex *node;
  Edge *edge;
  struct SuperNodeList *snlist;


/*
 *  First find the supernode and related items in the graph's supernode list
*/
  for (snlist = graph->super_nodes; snlist; snlist = snlist->next)
    if (snlist->super_node == super_node)
      break;
  if (!snlist)
    return;
/*
 *  Now, just move down the two lists and remove those items from the
 *  supernode's lists.
*/
  while (nodes = nlist) {
    nlist = nodes->next;
    node = nodes->node;
    UnhideNode(node);
    node->values->super_node == NULL;
    RemoveNodefromList(&(snlist->nodes), node);
  }
  while (edges = elist) {
    elist = edges->next;
    edge = edges->edge;
    UnhideEdge(edge);
    edge->values->super_node = NULL;
    edge->super_edge = NULL;
    RemoveEdgefromList(&(snlist->edges), edge);
  }
}

/*
 *  This routine duplicates and edge because we may want to hook this
 *  new edge up between a node in the elided region, and a node outside
 *  of the region.  It takes on exactly the same values as before.
*/
Edge *
DuplicateEdge(oldedge)
     Edge *oldedge;
{
  Edge *newedge;
  char *newi_name, *new_label, *new_type, *slabel, *llabel;
/*
 *  First, we must create some new labels and types for the new
 *  edges that we are creating.
*/
  newi_name = (char *)malloc(6);
  sprintf(newi_name, "SE%d", SuperCounter++);
  AssignStrVal(&new_label, NULL, "SuperEdge");
  AssignStrVal(&new_type, NULL, "SEType");
  AssignStrVal(&slabel, NULL, "");
  AssignStrVal(&llabel, NULL, "");
  newedge = MakeEdge(newi_name, new_label, new_type, slabel, llabel, NULL);
  if (oldedge->directed == TRUE)
    newedge->directed = TRUE;
  else
    newedge->directed = FALSE;
/*
 *  We are also sure to mark the edge being duplicated with the new edge
 *  that will be in its place.
*/
  oldedge->super_edge = newedge;

  return newedge;
}

/*
 *  MakeSuperEdges takes a super node and an in and out list of edges from
 *  this supernode, and correctly links up the structures.
*/
void
MakeSuperEdges(super_node, out_list, in_list, u_list, edge_list)
     Vertex *super_node;
     EdgeList *in_list, *out_list, *u_list, **edge_list;
{
  EdgeList *edges;
  Edge *new_edge;
  Link *link;
/*
 *  Now, fix up the out_list of edges and make the from node of those
 *  edges the new super node that was created.  We will need to create
 *  new edges here because we will have no way to fix up the structure
 *  when the elided region is unelided
*/
  for (edges = out_list; edges != NULL; edges = edges->next) {
    new_edge = DuplicateEdge(edges->edge);
    for (link = edges->edge->links; link != NULL; link = link->n_edge_adj_node)
      AddLink(super_node, link->tnode, new_edge);
  }
  MergeEdgeLists(edge_list, out_list);
/*
 *  We do the same for the incoming edges, except we modify the tnode field
*/
  for (edges = in_list; edges != NULL; edges = edges->next) {
    new_edge = DuplicateEdge(edges->edge);
    for (link = edges->edge->links; link != NULL; link = link->n_edge_adj_node)
      AddLink(link->fnode, super_node, new_edge);
  }
  MergeEdgeLists(edge_list, in_list);
/*
 *  Now, we will deal with the undirected edges.  This is done similar to
 *  the above two loops, but we must now be careful of what we attach the
 *  edge to becuase of the way that undirected edges are stored.
*/
  for (edges = u_list; edges != NULL; edges = edges->next) {
    new_edge = DuplicateEdge(edges->edge);
    for (link = edges->edge->links; link != NULL; link=link->n_edge_adj_node) {
      if (NodeHidden(link->tnode) | NodeMarked(link->tnode, MARKED))
	  AddLink(link->fnode, super_node, new_edge);
      else
	  AddLink(super_node, link->tnode,new_edge);
    }
  }
  MergeEdgeLists(edge_list, u_list);
}

/*
 *  The function DepthFirstSearch does just that.  Given a node as
 *  a paramter. It will mark all children of that node in a depth
 *  first fashion.
*/
void
DepthFirstSearch(node, node_list, edge_list, marking, search_for_hidden)
     Vertex *node;
     NodeList **node_list;
     EdgeList **edge_list;
     short marking, search_for_hidden;
{
  Vertex *child;
  Link *link;
  NodeList *new_node;
  EdgeList *new_edge;

/*
 *  We add this node onto a node list.  This list will be scanned later
 *  to determine the nodes that will actually get hidden due to user
 *  setable parameters.
*/
  AddNodetoList(node_list, node);
  MarkNode(node, marking);

  for (link = node->links; link != NULL; link = link->n_adj_node) {
/*
 *  We must first check to be sure that the edge we are on is not
 *  hidden.  If the edge is hidden, then we are moving into an
 *  area that is already elided.  We need to back off, and try the
 *  next link.
*/
    if (!EdgeMarked(link->edge, marking)) {
      if ((EdgeHidden(link->edge) && search_for_hidden) ||
	  (!EdgeHidden(link->edge) && !search_for_hidden)) {
	MarkEdge(link->edge, marking);
	/*
	 *  We will put the edge on this list because the edges will have to
	 *  get removed.
	 */
	AddEdgetoList(edge_list, link->edge);
    
	child = link->tnode;
	if (!NodeMarked(child, marking))
	  if ((NodeHidden(child) && search_for_hidden) ||
	      (!NodeHidden(child) && !search_for_hidden))
	    DepthFirstSearch(child, node_list, edge_list, marking,
			     search_for_hidden);
      }
    }
  }
}

/*
 *  The routine FindExtraEdges will find any edges that point from
 *  unmarked nodes in the graph to marked nodes in the marked region,
 *  and any edges from the marked out region to unmarked nodes in 
 *  the graph.  Notice that in the case of node-elision, there will
 *  be no edges that can go from the elided region to an unmarked node in
 *  the graph.
*/
void
FindExtraEdges(out_edges, in_edges, u_edges, edge_list)
     EdgeList **out_edges, **in_edges, **u_edges, **edge_list;
{
  Edge *edge;
  Link *link;

  for (edge = graph->edges; edge != NULL; edge = edge->next) 
  {
/*    if (EdgeMarked(edge, MARKED)) */
    if (EdgeMarked(edge, MARKED) || EdgeHidden(edge))
	continue;
    for (link = edge->links; link != NULL; link = link->n_edge_adj_node) {
/*
 *  otherwise, we need to check, in the case of a region elision, to see if
 *  an edges points between two marked nodes.  If so, then add these
 *  nodes to the edgelist (which is the list of potential edges to
 *  be hidden
*/
      if (NodeMarked(link->fnode, MARKED) && NodeMarked(link->tnode, MARKED)) {
	MarkEdge(edge, MARKED);
	AddEdgetoList(edge_list, edge);
      }
/*
 *  check to see if this edge goes from an unmarked node to
 *  a marked node.  If so, then add this edge to the inlist
*/
/*     else if (!NodeHidden(link->fnode) && NodeMarked(link->tnode, MARKED)) */
      else if (!NodeHidden(link->fnode) && (NodeMarked(link->tnode, MARKED) ||
					    NodeHidden(link->tnode)))
	if (edge->directed == TRUE)
	  AddEdgetoList(in_edges, edge);
	else
	  AddEdgetoList(u_edges, edge);
/*
 *  And if the edge points from a marked node to an unmarked node,
 *  then add the edge to the outlist.
*/
/*   else if (NodeMarked(link->fnode, MARKED) && !NodeHidden(link->tnode))*/
      else if ((NodeMarked(link->fnode, MARKED) || NodeHidden(link->fnode)) &&
	       !NodeHidden(link->tnode))
	  if (edge->directed == TRUE)
	      AddEdgetoList(out_edges,edge);
	  else
	      AddEdgetoList(u_edges, edge);
    }
  }
}

/*
 *  ProcessExtraEdges is the routine which gets called after the inbound,
 *  outbound, and undirected edges have been found a _SECOND_ time.  Since
 *  we don't want to do elision recursively, this routine will handle those
 *  edges which are found on the second pass of FindExtraEdges.  There
 *  are two mechanisms available which handle these situations.
*/
void
ProcessExtraEdges(supernode)
     Vertex *supernode;
{
  EdgeList *in_edges, *out_edges, *un_edges, *edge_list, *edge;
  short policy;

  in_edges = out_edges = un_edges = edge_list = NULL;
  FindExtraEdges(&out_edges, &in_edges, &un_edges, &edge_list);

  policy = (graph->elision_policy & INFINAL) >> 9;
  while (edge = in_edges) {
    in_edges = edge->next;
    if (policy == HIDE_EDGE) {
      AddEdgetoList(&edge_list, edge->edge);
      RemoveEdgefromList(&in_edges, edge->edge);
    }
    free(edge);
  }

  policy = (graph->elision_policy & OUTFINAL) >> 10;
  while (edge = out_edges) {
    out_edges = edge->next;
    if (policy == HIDE_EDGE) {
      AddEdgetoList(&edge_list, edge->edge);
      RemoveEdgefromList(&out_edges, edge->edge);
    }
    free(edge);
  }

  policy = (graph->elision_policy & UNFINAL) >> 11;
  while (edge = un_edges) {
    un_edges = edge->next;
    if (policy == HIDE_EDGE) {
      AddEdgetoList(&edge_list, edge->edge);
      RemoveEdgefromList(&un_edges, edge->edge);
    }
    free(edge);
  }
  MakeSuperEdges(supernode, out_edges, in_edges, un_edges, &edge_list);
  HideSubgraph(supernode, NULL, edge_list);
}
  
/*
 *  The routine RedetermineEdges takes an edge list and recomputes
 *  which edges and node may become hidden as a result of the addition
 *  of these edges into the graph.  This is done when a node is unelided
 *  and when any new edges are added into the graph.
*/
void
ReAddEdges(add_edges)
     EdgeList **add_edges;
{
  EdgeList *this_edge, *edge_list, *in_edges, *out_edges, *undir_edges;
  NodeList *node_list;
  Edge *edge, *new_edge;
  Link *link;
  Vertex *supernode, *fnode, *tnode, *hidden_node, *unhidden_node;
  short epolicy;

  while (this_edge = *add_edges) {
/*
 *  isolate the pointer to this EdgeList entry.  We will make it's next
 *  pointer NULL so that we can do some slick tricks with manipulation
 *  of this entry.
*/
    *add_edges = this_edge->next;
    edge = this_edge->edge;
/*
 *  get the supernode and then make the edge_list NULL so that we can
 *  add things onto them.
*/
    node_list = NULL;
    in_edges = out_edges = undir_edges = edge_list = NULL;
    for (link = edge->links; link; link = link->n_edge_adj_node) {
      fnode = link->fnode;
      tnode = link->tnode;
/*
 *  make the check for an edge between two nodes in different elision
 *  sets.  This should be done first to avoid redundant or messy code
 *  when parsing out undirected edges since no action is taken
 *  here in this case.
*/
      if (NodeHidden(fnode) && NodeHidden(tnode))
      {
	epolicy = (graph->unelision_policy & BETWEENSETS) >> 9;
	if (fnode->values->super_node == tnode->values->super_node) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(fnode->values->super_node, NULL, edge_list);
	}
	else if (epolicy == HIDE_IN_FROM_NODE) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(fnode->values->super_node, NULL, edge_list);
	}
	else if (epolicy == HIDE_IN_TO_NODE) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(tnode->values->super_node, NULL, edge_list);
	}
	else if (epolicy == (CONNECT_TO_SUPERNODE | HIDE_IN_FROM_NODE)) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(fnode->values->super_node, NULL, edge_list);
	  new_edge = DuplicateEdge(edge);
	  for (link = edge->links; link != NULL; link = link->n_edge_adj_node)
	    AddLink(fnode->values->super_node,
		    tnode->values->super_node, new_edge);
	}
	else if (epolicy == (CONNECT_TO_SUPERNODE | HIDE_IN_TO_NODE)) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(tnode->values->super_node, NULL, edge_list);
	  new_edge = DuplicateEdge(edge);
	  for (link = edge->links; link != NULL; link = link->n_edge_adj_node)
	    AddLink(fnode->values->super_node,
		    tnode->values->super_node, new_edge);
	}
      }
/*
 *  check for an undirected edge.  I believe that this needs to be
 *  done first because of the way that the undirected edges are stored
 *  internally.
*/
      else if (edge->directed == FALSE) {
	epolicy = (graph->unelision_policy & UNDIRECTED) >> 6;
	if (NodeHidden(tnode))
	{
	  hidden_node = tnode;
	  unhidden_node = fnode;
	}
	else {
	  hidden_node = fnode;
	  unhidden_node = tnode;
	}
/*
 *  Down and dirty check to see if we need to process this edge at all.
*/
	supernode = hidden_node->values->super_node;
	if (!NodeHidden(hidden_node) && !NodeHidden(unhidden_node)) {
	  AddEdgetoList(&edge_list, edge);
	  UnHideSubgraph(supernode, NULL, edge_list);
	  break;
	}
	if (epolicy == HIDE_EDGE) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(supernode, NULL, edge_list);
	}
/*
 *  make the edge now point from the supernode
*/
	else if (epolicy == CONNECT_TO_SUPERNODE) {
	  MakeSuperEdges(supernode, NULL, this_edge, NULL, edge_list);
	  HideSubgraph(supernode, NULL, edge_list);
	}
/*
 *  The node and the edge will now appear
*/
	else if (epolicy == UNHIDE_NODE) {
	  AddNodetoList(&node_list, hidden_node);
	  AddEdgetoList(&edge_list, edge);
	  UnHideSubgraph(supernode, node_list, edge_list);
	  ProcessExtraEdges(supernode);
	}
/*
 *  The node and its subgraph along with the edge will now appear.
*/
	else if (epolicy == UNHIDE_SUB) {
	  DepthFirstSearch(hidden_node, &node_list, &edge_list, MARKED, TRUE);
	  UnHideSubgraph(supernode, node_list, edge_list);
	  ProcessExtraEdges(supernode);
	}
	else if (epolicy == HIDE_NODE) {
	  AddNodetoList(&node_list, unhidden_node);
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(supernode, node_list, edge_list);
	  ProcessExtraEdges(supernode);
	}
	else if (epolicy == HIDE_SUB) {
	  AddEdgetoList(&edge_list, edge);
	  DepthFirstSearch(unhidden_node, &node_list, &edge_list,MARKED,FALSE);
	  HideSubgraph(supernode, node_list, edge_list);
	  ProcessExtraEdges(supernode);
	}
	break;
      }
/*
 *  We have an inbound edge to a region that has been elided.
*/
      else if (!NodeHidden(fnode) && NodeHidden(tnode))
      {
	epolicy = graph->unelision_policy & INBOUND;
	supernode = tnode->values->super_node;
/*
 *  Here we just hide the edge
*/
	if (epolicy == HIDE_EDGE) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(supernode, NULL, edge_list);
	}
/*
 *  make the edge now point from the supernode
*/
	else if (epolicy == CONNECT_TO_SUPERNODE) {
	  MakeSuperEdges(supernode, NULL, this_edge, NULL, &edge_list);
	  HideSubgraph(supernode, NULL, edge_list);
	}
/*
 *  The node and the edge will now appear
*/
	else if (epolicy == UNHIDE_NODE) {
	  AddEdgetoList(&edge_list, edge);
	  AddNodetoList(&node_list, tnode);
	  UnHideSubgraph(supernode, node_list, edge_list);
	  ProcessExtraEdges(supernode);
	}
/*
 *  The node and its subgraph along with the edge will now appear.
*/
	else if (epolicy == UNHIDE_SUB) {
	  AddEdgetoList(&edge_list, edge);
	  DepthFirstSearch(tnode, &node_list, &edge_list, MARKED, TRUE);
	  UnHideSubgraph(supernode, node_list, edge_list);
	  ProcessExtraEdges(supernode);
	}
	break;
      }
/*
 *  We have an outbound edge from an elided region pointing to an unhidden
 *  node.  We will follow the same basic pattern as above except that some
 *  of the routines are slightly different because we are now working with
 *  an outbound edge from a region.  We will also have a couple of more
 *  options to deal with.
*/
      else if (NodeHidden(fnode) && !NodeHidden(tnode))
      {
	epolicy = (graph->unelision_policy & OUTBOUND) >> 3;
	supernode = fnode->values->super_node;
/*
 *  Now, just check all of the options as before.
*/
	if (epolicy == HIDE_EDGE) {
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(supernode, NULL, edge_list);
	}
	else if (epolicy == CONNECT_TO_SUPERNODE) {
	  MakeSuperEdges(supernode, this_edge, NULL, NULL, &edge_list);
	  HideSubgraph(supernode, NULL, edge_list);
	}
	else if (epolicy == HIDE_NODE) {
	  AddNodetoList(&node_list, tnode);
	  AddEdgetoList(&edge_list, edge);
	  HideSubgraph(supernode, node_list, this_edge);
	  ProcessExtraEdges(supernode);
	}
	else if (epolicy == HIDE_SUB) {
	  AddEdgetoList(&edge_list, edge);
	  DepthFirstSearch(tnode, &node_list, &edge_list, MARKED, FALSE);
	  HideSubgraph(supernode, node_list, edge_list);
	  ProcessExtraEdges(supernode);
	}
	break;
      }
/*
 *  We can just simply add the edge back into the graph.
*/
      else {
	AddEdgetoList(&edge_list, edge);
	UnHideSubgraph(edge->values->super_node, NULL, edge_list);
      }
    }
  }
}

/*
 *  RemoveSubgraph takes node and edges lists of some subgraph.  Given the
 *  node supplied as a parameter, we do a depth first search at that node.
 *  we get back a resulting node and edge list which will be the node and
 *  edges that are to be removed from the the initial subgraph.
*/
void
RemoveSubgraph(old_nodes, old_edges, node)
     NodeList **old_nodes;
     EdgeList **old_edges;
     Vertex *node;
{
  NodeList *remove_nodes, *this_node;
  EdgeList *remove_edges, *this_edge;

/*
 *  Do a depth first search to determine the nodes and edges in the
 *  subgraph to be removed.
*/
  remove_nodes = NULL;
  remove_edges = NULL;
  DepthFirstSearch(node, &remove_nodes, &remove_edges, VISITED, FALSE);

/*
 *  Remove the nodes and edges from the passed in node and edge lists.  Be
 *  sure to free space.
*/
  while (this_node = remove_nodes) {
    remove_nodes = this_node->next;
    UnhideNode(this_node->node);
    RemoveNodefromList(old_nodes, this_node->node);
    UnmarkNode(this_node->node, VISITED | MARKED);
  }
  while (this_edge = remove_edges) {
    remove_edges = this_edge->next;
    UnhideEdge(this_edge->edge);
    RemoveEdgefromList(old_edges, this_edge->edge);
    UnmarkEdge(this_edge->edge, VISITED | MARKED);
  }
}

/*
 *  FilterEdges is the function that will remove any edges and nodes
 *  from the edge_list and node_list that need to be removed due to
 *  the elision policy that the user specifies.  It will also add new
 *  edges to the in_list and out_list depending on what new edges needed
 *  to be added to the graph that will now point to the supernode.
*/
int
FilterEdges(out_list, in_list, u_edges, edge_list, node_list, root,
	    elision_policy, elision_type)
     EdgeList **out_list, **in_list, **u_edges, **edge_list;
     NodeList **node_list;
     Vertex *root;
     short elision_policy, elision_type;
{
  Edge *edge;
  Vertex *marked_node, *unmarked_node, *node;
  EdgeList *edges, *tmp;
  short this_policy;
  int change_made;

  change_made = FALSE;
/*
 *  First, we check all of the edges in the incoming list to see
 *  if there are any edges and nodes that need to be removed due
 *  to the user's elision policy.  The thing to note here is the
 *  special case given to the root node when we have NODE_ELISION.
 *  We will want to flag all of those edges and keep them in the
 *  incoming list no matter what the policy because these are the
 *  edges that are needed to attach the upcoming supernode to the
 *  rest of the graph.
*/
  edges = *in_list;
  this_policy = elision_policy & INBOUND;
  while (edges != NULL) {
    tmp = edges->next;
    edge = edges->edge;
    node = edge->links->tnode;

    if (this_policy == UNHIDE_SUB && node != root) {
      RemoveSubgraph(node_list, edge_list, node);
      UnmarkEdge(edge, MARKED | VISITED);
      RemoveNodefromList(node_list, node);
      RemoveEdgefromList(in_list, edge);
      change_made = TRUE;
    }
    else if (this_policy == UNHIDE_NODE && node != root) {
      UnmarkNode(node, MARKED | VISITED);
      UnmarkEdge(edge, MARKED | VISITED);
      RemoveNodefromList(node_list, node);
      RemoveEdgefromList(in_list, edge);
      change_made = TRUE;
    }
    else if ((this_policy == HIDE_EDGE) && (node != root)) {
      AddEdgetoList(edge_list, edge);
      RemoveEdgefromList(in_list, edge);
    }
    edges = tmp;
  }
/*
 *  Now, check all of the edges in the out_going list to see if
 *  we need to remove any of the edges and nodes from their respective
 *  lists to keep them from getting hidden.
*/
  edges = *out_list;
  this_policy = (elision_policy & OUTBOUND) >> 3;
  while (edges != NULL) {
    tmp = edges->next;
    edge = edges->edge;
    node = edge->links->fnode;
    if (this_policy == HIDE_SUB) {
      DepthFirstSearch(edge->links->tnode, node_list, edge_list,MARKED,FALSE);
      RemoveEdgefromList(out_list, edge);
      AddEdgetoList(edge_list, edge);
      change_made = TRUE;
    }
    else if (this_policy == HIDE_NODE) {
      AddNodetoList(node_list, edge->links->tnode);
      MarkNode(edge->links->tnode, MARKED);
      RemoveEdgefromList(out_list, edge);
      AddEdgetoList(edge_list, edge);
      MarkEdge(edge, MARKED);
      change_made = TRUE;
    }
    else if (this_policy == HIDE_EDGE) {
      AddEdgetoList(edge_list, edge);
      RemoveEdgefromList(out_list, edge);
    }
    edges = tmp;
  }
/*
 *  Now, we will work with the undirected list in much the same
 *  manner as we did before.  We must still keep into consideration
 *  that if an undirected edge is attached to the root node in
 *  NODE_ELISION, then this undirected edge must still remain as an
 *  undirected edge, and not be removed from the graph no matter what
 *  the elision policy.
*/
  edges = *u_edges;
  this_policy = (elision_policy & UNDIRECTED) >> 6;
  while (edges != NULL) {
    tmp = edges->next;
    edge = edges->edge;
/*
 *  First we must find the actual node on this undirected edge that is
 *  marked.  this is just due to the fact about the way that undirected
 *  edges are stored.
*/
    if (NodeMarked(edge->links->tnode, MARKED)) {
      marked_node = edge->links->tnode;
      unmarked_node = edge->links->fnode;
    }
    else {
      marked_node = edge->links->fnode;
      unmarked_node = edge->links->tnode;
    }

    if (this_policy == UNHIDE_SUB && marked_node != root) {
      RemoveSubgraph(node_list, edge_list, marked_node);
      UnmarkEdge(edge, MARKED | VISITED);
      RemoveNodefromList(node_list, marked_node);
      RemoveEdgefromList(u_edges, edge);
      change_made = TRUE;
    }
    else if (this_policy == UNHIDE_NODE && marked_node != root) {
      UnmarkNode(marked_node, MARKED | VISITED);
      UnmarkEdge(edge, MARKED | VISITED);
      RemoveNodefromList(node_list, marked_node);
      RemoveEdgefromList(u_edges, edge);
      change_made = TRUE;
    }
    if (this_policy == HIDE_SUB) {
      DepthFirstSearch(unmarked_node, node_list, edge_list, MARKED, FALSE);
      RemoveEdgefromList(u_edges, edge);
      AddEdgetoList(edge_list, edge);
      change_made = TRUE;
    }
    else if (this_policy == HIDE_NODE) {
      MarkNode(unmarked_node, MARKED);
      AddNodetoList(node_list, unmarked_node);
      RemoveEdgefromList(u_edges, edge);
      AddEdgetoList(edge_list, edge);
      change_made = TRUE;
    }
    else if (this_policy == HIDE_EDGE && node != root) {
      AddEdgetoList(edge_list, edge);
      RemoveEdgefromList(u_edges, edge);
    }
    edges = tmp;
  }
  return change_made;
}

/*
 *  This next routine actually creates the super node that will replace the
 *  elided region.  It is passed a list of nodes and edges it is replacing,
 *  a name, all types of labels, a list of edges that now point away from it
 *  (outgoing_to_list), a list of edges now pointing to it (incoming_from_list)
 *  and an undirected list (currently unused).
*/
Vertex *
CreateSuperNode(i_name, label, short_l, long_l, type, xpos, ypos, etype)
     char *i_name, *label, *short_l, *long_l, *type;
     coord_t xpos, ypos;
     short etype;
{
  Vertex *super_node;
  EdgeList *edges;
  Edge *new_edge;
  Link *link;
  struct SuperNodeList *new_sn;

/*
 *  Call MakeVertex to create the new SuperNode.  MakeVertex will put this
 *  node onto the graph's nodelist for us.
*/
  MakeVertex(i_name, label, type, short_l, long_l, FALSE, NULL, NULL, NULL);
  super_node = graph->v;		/* this is the new node */
/*
 *  set up the some of the parameters that this node will need as
 *  well.
*/
  super_node->pos[0] = super_node->x = xpos;
  super_node->pos[1] = super_node->y = ypos;

  AssignNodeLabel(super_node);
/*
 *  Now, put this super node onto the graph's super node list.  We will rely
 *  on the fact the MakeVertex puts this new node at the head of the
 *  graph's node list, so that when adding this supernode into the supernode
 *  list, we can get the vertex pointer from graph->v.  Don't change the
 *  order of this code to prevent possible complications.
*/
  new_sn = (struct SuperNodeList *)malloc(sizeof(struct SuperNodeList));
  new_sn->super_node = super_node;	/* we are assured new node here... */
  new_sn->nodes = NULL;
  new_sn->edges = NULL;
  new_sn->next = graph->super_nodes;
  graph->super_nodes = new_sn;

/*
 *  Now, we must query the user for what the labels and type of the
 *  supernode just created should be.  We will call the widget server
 *  and use it to get our values.  The widget server will send back
 *  a message in the form of a graph update so that this is taken care
 *  of automatically for us.
*/
  if ((etype == NODE_ELISION && graph->label_policy & QUERY_FOR_LABEL) ||
      (etype == SWEEP_ELISION))
    QueryforNode(i_name, label, type, short_l, long_l, graph);

  return super_node;
}

void
Elision(node, node_list, edge_list, elision_type)
  Vertex *node;
  NodeList *node_list;
  EdgeList *edge_list;
  short elision_type;
{
  int status;
  EdgeList *out_edges, *in_edges, *u_edges;
  short elision_policy;
  char *base_name, *long_name, *short_name, *type, *i_name;
  coord_t xpos, ypos;
  Vertex *super_node;

  elision_policy = graph->elision_policy;
/*
 *  Create the supernode.  First, we must malloc out space for the new
 *  supernode's labels.  It should be assumed that the CreateSuperNode
 *  routine shouldn't have to worry about doing things like this.  Because
 *  here we have a node elision, we will either get the values of the
 *  supernode (i.e. labels, etc), from the root of the elision tree, or
 *  by user input.  In either case, we will need to malloc out space
 *  for the new labels.
*/
  if (elision_type == NODE_ELISION) {
    AssignStrVal(&base_name, NULL, node->values->labels[0]);
    AssignStrVal(&long_name, NULL, node->values->labels[1]);
    AssignStrVal(&short_name, NULL, node->values->labels[2]);
    AssignStrVal(&type, NULL, node->values->type);
    xpos = node->x;
    ypos = node->y;
  }
  else {
    AssignStrVal(&base_name, NULL, "SuperNode");
    AssignStrVal(&long_name, NULL, "");
    AssignStrVal(&short_name, NULL, "");
    AssignStrVal(&type, NULL, "SNtype");
    xpos = 0.25 + (drand48() / 2.0);
    ypos = 0.25 + (drand48() / 2.0);
  }
  i_name = (char *)malloc(6);
  sprintf (i_name, "SN%d", SuperCounter++);
  super_node = CreateSuperNode(i_name, base_name, short_name, long_name, type,
			       xpos, ypos, elision_type);
  SuperNode(super_node);

  in_edges = NULL;
  out_edges = NULL;
  u_edges = NULL;
/*
 *  FindExtraEdges will give us a list of all of the incoming edges to
 *  this region, and all of the outgoing edges from this region (although in
 *  this case of a node-selected elision, there will be no outgoing edges
 *  from the region).
*/
  FindExtraEdges(&out_edges, &in_edges, &u_edges, &edge_list);
/*
 *  FilterEdges will remove any nodes from the nodes that have the potential
 *  to be removed from the graph on the basis of the elision policy that is
 *  in effect.  If any changes were made that might create new inbound,
 *  outbound, and undirected edges into or out of the elision region,
 *  then we must filter the edges again to find these edges.
*/
  status = FilterEdges(&out_edges, &in_edges, &u_edges, &edge_list, &node_list,
		  node, elision_policy, elision_type);
  MakeSuperEdges(super_node, out_edges, in_edges, u_edges, &edge_list);
  HideSubgraph(super_node, node_list, edge_list);

  if (status)
    ProcessExtraEdges(super_node);
/*
 *  Attach up all of the edges to the supernode.
*/
}

int
ElideNodeCallback(CallbackInfo, CallbackData)
  struct CallbackInfo *CallbackInfo;
  struct FunctionBindings *CallbackData;
{
  NodeList *node_list;
  EdgeList *edge_list;
  
  node_list = NULL;
  edge_list = NULL;
  
  FillCallbackInfo(CallbackInfo, NODEMASK);
  if (!CallbackInfo->node)
      return;

  if (NodeSuper(CallbackInfo->node))
      return;
/*
 *  We do a depth first search on the node that was selected to determine
 *  region that we will be eliding.  We will immediately mark the nodes
 *  as MARKED meaning that they have to potential to be hidden.
*/
  DepthFirstSearch(CallbackInfo->node, &node_list, &edge_list, MARKED, FALSE);
  Elision(CallbackInfo->node, node_list, edge_list, NODE_ELISION);
  return REDETERMINE | CHANGED_EDGES;
}

int
ElideSweptNodesCallback(CallbackInfo, CallbackData)
  struct CallbackInfo *CallbackInfo;
  struct FunctionBindings *CallbackData;
{
  NodeList *node_list;
  EdgeList *edge_list;

  edge_list = NULL;
  FillCallbackInfo(CallbackInfo, SWEPT_REGION|SWEPT_NODES);
  if (!CallbackInfo->swept_nodes)
      return;

  /* Mark all the nodes as marked */
  for (node_list = CallbackInfo->swept_nodes; node_list;
       node_list = node_list->next)
      MarkNode(node_list->node, MARKED);
    
  DupNodeList(CallbackInfo->swept_nodes, &node_list);
  Elision(NULL, node_list, edge_list, SWEEP_ELISION);
  return REDETERMINE | CHANGED_EDGES;
}

int
ElideSelectedNodesCallback(CallbackInfo, CallbackData)
  struct CallbackInfo *CallbackInfo;
  struct FunctionBindings *CallbackData;
{
  NodeList *node_list;
  EdgeList *edge_list;

  edge_list = NULL;
  FillCallbackInfo(CallbackInfo, SELECTED_NODES);
  if (!CallbackInfo->selected_nodes)
      return;

  /* Mark all the nodes as marked */
  for (node_list = CallbackInfo->selected_nodes; node_list;
       node_list = node_list->next)
      MarkNode(node_list->node, MARKED);
    
  DupNodeList(CallbackInfo->selected_nodes, &node_list);
  Elision(NULL, node_list, edge_list, SWEEP_ELISION);
  return REDETERMINE | CHANGED_EDGES;
}

void
AntiElision(node)
     Vertex *node;
{
  struct SuperNodeList *elision_list, *prev_list;
  EdgeList *edge_list, *prev_edge;
  NodeList *node_list, *prev_node;
  int found;

/*
 *  gotta do that check...
*/
  if (node == NULL)
    return;
/*
 *  Now, try and find the node in the list of vertices that are
 *  elided.  If it is found, then simply mark all of the nodes
 *  and edges as not hidden.
*/
  found = FALSE;
  prev_list = NULL;
  elision_list = graph->super_nodes;
  while ((!found) && (elision_list != NULL)) {
    if (elision_list->super_node == node) {
      edge_list = elision_list->edges;
      node_list = elision_list->nodes;
      found = TRUE;
    }
    else {
      prev_list = elision_list;
      elision_list = elision_list->next;
    }
  }
  if (!found)
    return;				/* we didn't find an elided node */
/*
 *  Okay.  Delete the super node that was created.  This should, if
 *  everything works right, clean up all of the links that were made
 *  when the supernode was first created.
*/
  DeleteNode(node);
/*
 *  Okay. Now, here is what to do.  We don't worry about adding nodes back
 *  because the way elision is set up, then they might get hidden back
 *  because of elision policy.  So we can take all of the nodes are unhide
 *  them.  We will then take all of the edges, and proceed to do operations
 *  on that set to determine if anything else might become hidden as a result
 *  of this unelision.
*/
  prev_node = NULL;
  while (node_list != NULL) {
    UnhideNode(node_list->node);
    prev_node = node_list;
    node_list = node_list->next;
    free(prev_node);
  }
/*
 *  We will call this routine ReAddEdges because we will also do what
 *  is in this routine when adding new edges into the graph.
*/
  ReAddEdges(&edge_list);

/*
 *  Be sure to patch up the elided node list for the graph.  We don't do
 *  this right after finding the super_node list because in ReAddEdges,
 *  the edge_list is destroyed by UnHideSubgraph, and that requires the
 *  graph to still have the super_node list attached.
*/
  if (prev_list == NULL)
    graph->super_nodes = elision_list->next;
  else
    prev_list->next = elision_list->next;

  free(elision_list);
}

int
UnelideNodeCallback(CallbackInfo, CallbackData)
  struct CallbackInfo *CallbackInfo;
  struct FunctionBindings *CallbackData;
{
  FillCallbackInfo(CallbackInfo, NODEMASK);

  AntiElision(CallbackInfo->node);
  return REDETERMINE | CHANGED_EDGES;
}

int
UnelideSweptNodesCallback(CallbackInfo, CallbackData)
  struct CallbackInfo *CallbackInfo;
  struct FunctionBindings *CallbackData;
{
  NodeList *node_list;
  
  FillCallbackInfo(CallbackInfo, SWEPT_REGION | SWEPT_NODES);

  for (node_list = CallbackInfo->swept_nodes; node_list;
       node_list = node_list->next)
      AntiElision(node_list->node);

  return REDETERMINE | CHANGED_EDGES;
}

int
UnelideSelectedNodesCallback(CallbackInfo, CallbackData)
  struct CallbackInfo *CallbackInfo;
  struct FunctionBindings *CallbackData;
{
  NodeList *node_list;
  
  FillCallbackInfo(CallbackInfo, SELECTED_NODES);

  for (node_list = CallbackInfo->selected_nodes; node_list;
       node_list = node_list->next)
      AntiElision(node_list->node);

  return REDETERMINE | CHANGED_EDGES;
}
