/* This file is part of the
 *
 *      Delta Project  (ConversationBuilder)
 *      Human-Computer Interaction Laboratory
 *      University of Illinois at Urbana-Champaign
 *      Department of Computer Science
 *      1304 W. Springfield Avenue
 *      Urbana, Illinois 61801
 *      USA
 *
 *      c 1989,1990,1991 Board of Trustees
 *              University of Illinois
 *              All Rights Reserved
 *
 *      This file is distributed under license and is confidential
 *
 *      File title and purpose
 *      Author:  Doug Bogia (bogia@cs.uiuc.edu)
 *
 *      Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *      Direct enquiries to the project leader please.
 *
 *  mbquery.c was created by Doug Bogia on Oct.  8, 1991
 *  It is designed to attempt to attach to the message bus and
 *  check if a particular program is out there listening to
 *  commands.
 *  Exit codes:
 *   0 - The bus is up and the program is listening.
 *   1 - The bus is not running.
 *   2 - The bus is running, but the program isn't.
 *  10 - bad arguments were sent in.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <mbus/api.h>
#include <mbus/mbus.h>
#include "cb-defs.h"

char *host;
char *prog_name = "Widget Server";
char *cb_base_domain = NULL;
char *cb_server_domain = NULL;
char *cb_ui_domain = NULL;
char *cb_user = NULL;
char *cb_local_domain = NULL;
char *tag = "";
int MBLogLevel = 0;
int port = DEF_MBUS_PORT;
t_mb_connection mbus = NULL;                    /* connection to the MBus */

/* ------------------------------------------------------------------------ */
int
DealWithCommandLineOptions(argc,argv)
     int argc;
     char **argv;
{
  int opt, Usage = 0;
  char *tmp;
  extern int optind;
  extern char opt_err_char, *optarg, *GetUserName();
  extern char *getenv();

  /* environment variable checks */
  host = getenv(ENV_MBUS_HOST);
  if (NULL != (tmp = getenv(ENV_MBUS_PORT))) port = strtol(tmp, NULL, 0);
  cb_base_domain = getenv(ENV_BASE_DOMAIN);
  cb_server_domain = getenv(ENV_SERVER_DOMAIN);
  cb_ui_domain = getenv(ENV_UI_DOMAIN);
  cb_local_domain = getenv(ENV_WS_DOMAIN);

  while ( (opt = getopt(argc,argv,"b:s:i:u:l:h:p:L;n:t:")) != EOF)
    switch (opt)
      {
      case '?' :
	fprintf (stderr, "Unknown option -%c.\n", opt_err_char);
        Usage = 1;
        break;
      case 'b':
	cb_base_domain = optarg; break;
      case 's':
	cb_server_domain = optarg; break;
      case 'i':
	cb_ui_domain = optarg; break;
      case 'u':
	cb_user = optarg; break;
      case 'l':
	cb_local_domain = optarg; break;
      case 'h' : host = optarg; break;
      case 'L' :
        if (NULL == optarg) MBLogLevel = 1;
        else MBLogLevel = strtol(optarg, NULL, 0);
        break;
      case 'n':
	prog_name = optarg;
	break;
      case 'p' :
        {
          int n;
          char *s;
          n = strtol(optarg, &s, 0);
          if (s != optarg) port = n;
        }
        break;
      case 't':
	tag = optarg;
	break;
      }

  if (Usage)
  {
    printf ("Usage:  %s -b b_domain -s s_domain -i i_domain -l l_domain\n",
	    argv[0]);
    printf ("           -u u_domain -h host -p port -L [loglevel] -n name -t tag\n");
    printf ("where:  b_domain is the base domain (default \"cb\")\n");
    printf ("        s_domain is the server domain (default \"server\")\n");
    printf ("        i_domain is the user interface domain (default \"ui\")\n");
    printf ("        u_domain is the user domain (default $USER)\n");
    printf ("        l_domain is the local domain (default \"browser\")\n");
    printf ("        host is the host of the message bus\n");
    printf ("        port is the port of the message bus\n");
    printf ("        loglevel is the logging level to print\n");
    printf ("        name is the name of the program you are interested in.\n");
    printf ("          (default \"Widget Server\")\n");
    printf ("        tag is the tag that you wish to check for responses.\n");
    printf ("          (default \"\")\n");
    printf ("\nEnvironment variables used:\n");
    printf ("CB_BASE_DOMAIN     the base domain\n");
    printf ("CB_SERVER_DOMAIN   the server domain\n");
    printf ("CB_UI_DOMAIN       the user interface domain\n");
    printf ("CB_USER            the user domain\n");
    printf ("CB_WS_DOMAIN       the widget domain\n");
    printf ("MBUS_HOST          the message bus host\n");
    printf ("MBUS_PORT          the message bus port\n");
    exit (10);
  }

  if (NULL == cb_base_domain) cb_base_domain = DEF_BASE_DOMAIN;
  if (NULL == cb_server_domain) cb_server_domain = DEF_SERVER_DOMAIN;
  if (NULL == cb_ui_domain) cb_ui_domain = DEF_UI_DOMAIN;
  if (NULL == cb_user) cb_user = GetUserName();
  if (NULL == cb_local_domain) cb_local_domain = DEF_WS_DOMAIN;

  return optind;
}

/* ------------------------------------------------------------------------ */
void
main(argc,argv)
     int argc;
     char *argv[];
{
  char buff[512];
  int skip, n;
  fd_set           read_fds;    /* Fds to select on */
  fd_set           write_fds;   /* Write Fds */
  struct mb_object * NextMessage; /* This is the next message read */
  t_sexp           info;

  skip = DealWithCommandLineOptions(argc,argv);

  /* Return an error status of 1 if we can't connect to the bus */
  if (NULL == (mbus = MBConnect(host,port))) exit(1);

  /*
   * Ok, we are attached.  See if there is somebody listening to the
   * tag and domain.
   */
  if (!strcmp(cb_user, ""))
  {
    /* If we have a empty string user then we must be quering a server. */
    if (strcmp(cb_local_domain, ""))
    {
      sprintf(buff, "(ping \"%s\" \"%s.%s.%s\")\n", tag, 
	      cb_local_domain,
	      cb_server_domain,
	      cb_base_domain
	      );
    }
    else
    {
      sprintf(buff, "(ping \"%s\" \"%s.%s\")\n", tag,
	      cb_server_domain,
	      cb_base_domain
	      );
    }
  }
  else
  {
    if (strcmp(cb_local_domain, ""))
    {
      sprintf(buff, "(ping \"%s\" \"%s.%s.%s.%s\")\n", tag, 
	      cb_local_domain,
	      cb_user,
	      cb_ui_domain,
	      cb_base_domain
	      );
    }
    else
    {
      sprintf(buff, "(ping \"%s\" \"%s.%s.%s\")\n", tag,
	      cb_user,
	      cb_ui_domain,
	      cb_base_domain
	      );
    }
  }    
  MBtransmit_Cstring(mbus, buff);

  /* Set up for select later */
  FD_ZERO(&read_fds);
  FD_ZERO(&write_fds);
  FD_SET(MBconnection_fd(mbus), &read_fds);

  while (1)
  {
    n = select (16, &read_fds, &write_fds, 0, 0);
    if (n <= 0) continue;

    NextMessage = MBNextMessage (mbus);

    info = MBcar(NextMessage);
  
    if (!MBcompare_Cstring(info, "ping"))
    {
      /* We are only interested in responses to our ping */
      NextMessage = MBcdr(MBcdr(NextMessage));
      while (MBconsp(NextMessage))
      {
	info = MBcar(NextMessage);
	if (!MBcompare_Cstring(info, prog_name))
	{
	  exit (0);
	}
	NextMessage = MBcdr(NextMessage);
      }
      exit (2);
    }
  }
}
