;; INTERMEDIATE FORM:
;; (position font/nil glyph/nil newline/nil text)
;; font==nil - same font,      otherwise select named font
;; glyph==nil - no glyph,      otherwise display named glyph
;; newline==nil - no newline,  otherwise generate a new line
;; text is a lump of text to be displayed in the new font,
;; ** after ** any glyph or newline.

(defvar energize-annotate-print-ps-defns nil
  "*File in which to find PS headers for annotated print.
If nil, use the hardcoded default.")

(defvar energize-annotate-print-ps-whence 'energize-al-print-lpr-buffer
  "*Function to process the buffer once generated.
Returns boolean whether buffer should be deleted")

(defun energize-annotate-print-elisp ()
  "Generate an ELISP data structure which corresponds to the
text & extents from a buffer."
  (interactive)
  (save-excursion
  (let* ((buf (current-buffer))
	 (ex buf)
	 (ex-list (list (list 'default (point-min) (point-max) nil)))
	 (ex-ends (list (point-max)))
	 evs)
    (while (setq ex (next-extent ex))
      (let ((s (prin1-to-string ex)) q)
	(setq q
	       (let* ((bg (extent-property ex 'begin-glyph))
		      (eg (extent-property ex 'end-glyph))
		      (tg (or bg eg))
		      (g (if tg (file-name-nondirectory (pixmap-file-name tg)))))
		 (if bg (list 'begin-glyph g)
		   (if eg (list 'end-glyph g)
		     nil)))
	       )
	(let ((sp (extent-start-position ex))
	      (ep (extent-end-position ex)))
	  (setq q (list (extent-face ex) sp ep q))
	  (if (> ep sp)
	      (progn
	      (setq ex-list (cons q ex-list))
	      (setq ex-ends (cons (extent-end-position ex) ex-ends)))))))
    ;; Make sure these are sorted.
    (let ((exlt '(lambda (x y)
		   (let ((sx (nth 1 x))
			 (sy (nth 1 y)))
		     (if (eq sx sy)
			 (> (nth 2 x) (nth 2 y))
		       (< (nth 1 x) (nth 1 y)))))))
      (setq ex-list (sort ex-list exlt)))
    (setq ex-ends (sort ex-ends '<))

    (let ((exs ex-list)
	  (exe ex-ends)
	  ev stk)
      (while (or exs exe)
	(let ((es (car exs))
	      (ee (car exe))
	      gf)
	  (if (or (null ee) (and es (< (nth 1 es) ee)))
	      (progn ;; process a new starting
		(let* ((gf (nth 3 es))
		       (g (if (eq (car gf) 'begin-glyph) (nth 1 gf) nil)))
		  (setq ev (list (nth 1 es) (car es) g nil nil)))
		;; test that es is worthwhile before stacking it
		(if (null (car es))
		    (if (null (nth 3 es))
			(setq es nil)
		      (if stk ;; if it has a glyph, copy lower stack entry
			  (setcar es (car (car stk))))))
		(if es
		    (setq stk (cons es stk)))
		(setq exs (cdr exs)))
	    ;; ending
	    (let ((stf t)
		  (done nil)
		  (p stk)
		  (q stk))
	      (while (and p (not done))
		(if (eq (nth 2 (car p)) ee)
		    (progn
		      (setq done t)
		      (setq gf (nth 3 (car p)))
		      (if stf
			  (setq stk (cdr p))
			(setcdr q (cdr p))))
		  (setq stf nil))
		(setq q p)
		(setq p (cdr p))))
	    (let ((g (if (eq (car gf) 'end-glyph) (nth 1 gf) nil)))
	      (setq ev (list ee (car (car stk)) g nil nil)))
	    (setq exe (cdr exe))))
	(setq evs (cons ev evs)))
      (setq evs (nreverse evs)))
;; now process EOLs
    (goto-char (point-min))
    (let (eols)
      (while (search-forward "\n" nil 't)
	(setq eols (cons (list (1- (point)) nil nil t nil) eols)))
      (setq eols (nreverse eols))
      (let (ev1)
	(while (or evs eols)
	  (if (or (null evs) (and eols (< (car (car eols)) (car (car evs)))))
	      (progn
		(setq ev1 (cons (car eols) ev1))
		(setq eols (cdr eols)))
	    (setq ev1 (cons (car evs) ev1))
	    (setq evs (cdr evs))))
	(setq evs (nreverse ev1))))
;; finally collect up text segments
    (let ((es evs))
      (while es
	(let ((e (car es))
	      (f 0)
	      s ss p (r 0) q fr to)
	  (if (nth 3 e)
	      (setq f 1))
	  (setq s (buffer-substring (+ (car e) f) (car (car (cdr es)))))
	  (while (setq p (string-match "\t" s r))
	    (goto-char (+ (car e) f p))
	    (setq fr (current-column))
	    (forward-char 1)
	    (setq to (current-column))
	    (setq ss (make-list (- to fr) " "))
	    (setq q (concat q (substring s r p) (mapconcat 'identity ss "")))
	    (setq r (1+ p)))
	  (if (not (eq r 0))
	      (setq s (concat q (substring s r))))
	  (let ((te e) (ti 4))
	    (while (> ti 0)
	      (setq te (cdr te))
	      (setq ti (1- ti)))
	    (setcar te s)))
	(if (null (cdr (cdr es))) (setcdr es nil))
	(setq es (cdr es))))
    evs
)))

(defun energize-annotate-print-ps ()
  "Print buffer taking full note of extents, pixmaps etc."
  (interactive)
  (let ((evs (energize-annotate-print-elisp))
	s tx
	(buf (get-buffer-create "*anno-ps-print*"))
	(def-ps-defns
"%!PS-Adobe-1.0
%%BoundingBox: 0 0 792 612
%%Pages: 1
%%DocumentFonts: Courier
%%+ Times-Bold
%%+ Helvetica-Oblique
%%+ Courier-BoldOblique
%%+ Helvetica
%%+ Courier-Bold
%%+ Courier-Oblique
%%EndComments
100 dict begin
/Dsize 170 def
/colors Dsize dict def
/facecat Dsize dict def
/pixcat Dsize dict def
/newpageflag false def

colors begin
  /aquamarine4   [  69 139 116 ] def
  /black         [   0   0   0 ] def
  /blue3         [   0   0 205 ] def
  /cadetblue4    [  83 134 139 ] def
  /darkseagreen2 [ 180 238 180 ] def
  /gray          [ 190 190 190 ] def
  /gray90        [ 229 229 229 ] def
  /green4        [   0 139   0 ] def
  /lightyellow4  [ 139 139 122 ] def
  /hotpink1      [ 255 110 180 ] def
  /maroon3       [ 205  41 144 ] def
  /paleturquoise [ 175 238 238 ] def
  /plum3         [ 205 150 205 ] def
  /red3          [ 205   0   0 ] def
  /c6920ac       [ 102  32 172 ] def
end

facecat begin % /name font size /textcol /backcol uline
  /default [
    /Courier findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /modeline [
    /Courier findfont 10 scalefont
    12
    /none
    /black
    false
  ] def
  /highlight [
    /Courier findfont 10 scalefont
    12
    /black
    /darkseagreen2
    false
  ] def
  /left-margin [
    /Courier findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /right-margin [
    /Courier findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /bold [
    /Courier-Bold findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /italic [
    /Courier-Oblique findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /bold-italic [
    /Courier-BoldOblique findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /isearch [
    /Courier findfont 10 scalefont
    12
    /black
    /paleturquoise
    false
  ] def
  /primary-selection [
    /Courier findfont 10 scalefont
    12
    /black
    /gray
    false
  ] def
  /secondary-selection [
    /Courier findfont 10 scalefont
    12
    /black
    /paleturquoise
    false
  ] def
  /attributeSmall [
    /Courier findfont 8 scalefont
    10
    /black
    /none
    false
  ] def
  /attributeGlyph [
    /Courier findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /attributeSectionHeader [
    /Courier findfont 10 scalefont
    12
    /cadetblue4
    /none
    false
  ] def
  /attributeToplevelFormGlyph [
    /Courier findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /attributeModifiedToplevelFormGlyph [
    /Courier findfont 10 scalefont
    12
    /black
    /none
    false
  ] def
  /attributeBrowserHeader [
    /Courier findfont 10 scalefont
    12
    /red3
    /none
    false
  ] def
  /attributeWriteProtected [
    /Courier findfont 10 scalefont
    12
    /lightyellow4
    /none
    false
  ] def
  /attributeModifiedText [
    /Courier findfont 10 scalefont
    12
    /black
    /gray90
    false
  ] def
  /attribute50 [
    /Courier findfont 10 scalefont
    12
    /plum3
    /none
    false
  ] def
  /attribute52 [
    /Courier findfont 10 scalefont
    12
    /aquamarine4
    /none
    false
  ] def
  /font-lock-comment-face [
    /Courier findfont 10 scalefont
    12
    /#6920ac
    /none
    false
  ] def
  /font-lock-doc-string-face [
    /Courier findfont 10 scalefont
    12
    /green4
    /none
    false
  ] def
  /font-lock-string-face [
    /Courier findfont 10 scalefont
    12
    /green4
    /none
    false
  ] def
  /font-lock-function-name-face [
    /Courier findfont 10 scalefont
    12
    /red3
    /none
    false
  ] def
  /font-lock-keyword-face [
    /Courier findfont 10 scalefont
    12
    /blue3
    /none
    false
  ] def
  /font-lock-type-face [
    /Courier findfont 10 scalefont
    12
    /blue3
    /none
    false
  ] def
end

pixcat begin
  /archive_tool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\300\\174\\000\\377\\300\\122\\000\\377\\300\\111\\000\\377\\300\\145\\000\\377\\300\\123\\000\\377\\300\\111\\000\\377\\300\\105\\000\\377\\300\\177\\000\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /arrow [14 14 (\\000\\034\\000\\034\\000\\014\\000\\014\\000\\004\\000\\004\\000\\000\\000\\000\\000\\004\\000\\004\\000\\014\\000\\014\\000\\034\\000\\034)] def
  /arrow-hollow [14 14 (\\000\\034\\000\\034\\077\\214\\077\\214\\077\\304\\077\\304\\077\\340\\077\\340\\077\\304\\077\\304\\077\\214\\077\\214\\000\\034\\000\\034)] def
  /blip [3 3 (\\377\\377\\377)] def
  /btool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\300\\034\\140\\377\\300\\077\\340\\377\\301\\377\\340\\377\\301\\377\\340\\377\\303\\077\\340\\377\\302\\034\\140\\377\\302\\034\\000\\377\\300\\034\\000\\377\\300\\034\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /build [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\040\\000\\377\\300\\370\\000\\377\\300\\330\\000\\377\\301\\274\\200\\377\\300\\333\\340\\377\\300\\373\\140\\377\\300\\046\\360\\377\\300\\003\\140\\377\\300\\003\\340\\377\\300\\000\\200\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /collect_tool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\300\\174\\000\\377\\300\\102\\000\\377\\300\\101\\000\\377\\300\\177\\000\\377\\300\\101\\000\\377\\300\\177\\000\\377\\300\\101\\000\\377\\300\\177\\000\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /collection [19 14 (\\377\\377\\377\\374\\017\\377\\374\\007\\377\\374\\003\\377\\374\\001\\377\\375\\376\\377\\375\\376\\377\\374\\000\\377\\374\\000\\377\\375\\376\\377\\375\\376\\377\\374\\000\\377\\374\\000\\377\\377\\377\\377)] def
  /command_tool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\301\\302\\000\\377\\303\\146\\000\\377\\302\\044\\000\\377\\303\\154\\340\\377\\301\\311\\260\\377\\300\\031\\020\\377\\300\\021\\260\\377\\300\\060\\340\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /compile1 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\337\\377\\376\\377\\337\\377\\376\\377\\334\\341\\306\\377\\334\\344\\222\\377\\330\\141\\236\\377\\330\\144\\236\\377\\323\\044\\222\\377\\323\\041\\306\\377\\337\\377\\376\\377\\337\\377\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /compile2 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\201\\377\\377\\177\\201\\377\\377\\177\\230\\341\\307\\177\\232\\344\\223\\177\\232\\141\\237\\177\\232\\144\\237\\177\\233\\044\\223\\177\\231\\041\\307\\177\\201\\377\\377\\177\\201\\377\\377\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /compile3 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\200\\007\\377\\177\\200\\007\\377\\177\\231\\311\\307\\177\\233\\154\\223\\177\\233\\151\\237\\177\\233\\154\\237\\177\\233\\154\\223\\177\\231\\311\\307\\177\\200\\007\\377\\177\\200\\007\\377\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /compile4 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\200\\000\\037\\177\\200\\000\\037\\177\\231\\314\\347\\177\\233\\155\\263\\177\\233\\155\\277\\177\\233\\155\\277\\177\\233\\155\\263\\177\\231\\314\\347\\177\\200\\000\\037\\177\\200\\000\\037\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /compile5 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\200\\000\\000\\177\\200\\000\\000\\177\\231\\314\\346\\177\\233\\155\\266\\177\\233\\155\\266\\177\\233\\155\\266\\177\\233\\155\\266\\177\\231\\314\\346\\177\\200\\000\\000\\177\\200\\000\\000\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /currenterror [19 14 (\\300\\001\\377\\302\\021\\377\\301\\040\\377\\300\\300\\377\\301\\340\\177\\315\\354\\177\\327\\372\\077\\303\\360\\077\\317\\374\\177\\323\\362\\177\\307\\370\\377\\311\\344\\377\\310\\305\\377\\300\\001\\377)] def
  /currentnote [19 14 (\\377\\377\\377\\360\\003\\377\\360\\001\\377\\363\\201\\377\\363\\340\\377\\363\\370\\377\\363\\376\\177\\363\\376\\177\\363\\370\\377\\363\\340\\377\\363\\201\\377\\360\\001\\377\\360\\003\\377\\377\\377\\377)] def
  /currentwarning [19 14 (\\377\\377\\377\\340\\003\\377\\340\\301\\377\\340\\301\\377\\341\\340\\377\\341\\340\\377\\343\\360\\177\\343\\360\\177\\347\\370\\377\\347\\370\\377\\357\\375\\377\\340\\001\\377\\340\\003\\377\\377\\377\\377)] def
  /dbox-error [30 30 (\\377\\377\\377\\377\\377\\340\\077\\377\\377\\000\\007\\377\\376\\037\\303\\377\\370\\177\\360\\377\\361\\377\\374\\177\\360\\377\\376\\177\\344\\177\\377\\077\\306\\077\\377\\037\\317\\037\\377\\237\\317\\217\\377\\237\\237\\307\\377\\317\\237\\343\\377\\317\\237\\361\\377\\317\\237\\370\\377\\317\\237\\374\\177\\317\\237\\376\\077\\317\\237\\377\\037\\317\\317\\377\\217\\237\\317\\377\\307\\237\\307\\377\\343\\037\\347\\377\\361\\077\\363\\377\\370\\177\\361\\377\\370\\177\\370\\177\\360\\377\\376\\037\\303\\377\\377\\000\\007\\377\\377\\340\\077\\377\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /dbox-info [30 30 (\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\374\\177\\377\\377\\370\\077\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\370\\077\\377\\377\\374\\177\\377\\377\\377\\377\\377\\377\\200\\037\\377\\377\\200\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\200\\003\\377\\377\\200\\003\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /dbox-question [30 30 (\\377\\340\\077\\377\\377\\000\\007\\377\\376\\000\\003\\377\\370\\017\\200\\377\\360\\037\\300\\177\\360\\077\\340\\177\\340\\070\\340\\077\\300\\070\\340\\037\\300\\070\\340\\037\\300\\070\\340\\037\\200\\070\\340\\017\\200\\001\\340\\017\\200\\003\\300\\017\\200\\007\\200\\017\\200\\007\\000\\017\\200\\007\\000\\017\\200\\007\\000\\017\\300\\007\\000\\037\\300\\007\\000\\037\\300\\000\\000\\037\\340\\007\\000\\077\\360\\007\\000\\177\\360\\007\\000\\177\\370\\000\\000\\377\\376\\000\\003\\377\\377\\000\\007\\377\\377\\340\\077\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /debugger [26 14 (\\377\\377\\377\\377\\340\\000\\003\\377\\300\\022\\017\\377\\300\\014\\074\\377\\300\\036\\360\\377\\300\\333\\300\\377\\301\\157\\040\\377\\300\\075\\000\\377\\300\\367\\300\\377\\303\\337\\040\\377\\317\\177\\200\\377\\374\\236\\100\\377\\360\\000\\001\\377\\377\\377\\377\\377)] def
  /directories [24 14 (\\377\\377\\377\\377\\377\\377\\376\\037\\377\\374\\000\\077\\374\\000\\037\\377\\377\\037\\360\\000\\237\\360\\000\\237\\370\\000\\137\\370\\000\\137\\374\\000\\037\\374\\000\\037\\377\\377\\377\\377\\377\\377)] def
  /directory [19 14 (\\377\\377\\377\\377\\377\\377\\370\\177\\377\\360\\001\\377\\360\\000\\377\\377\\370\\377\\300\\004\\377\\300\\004\\377\\340\\002\\377\\340\\002\\377\\360\\000\\377\\360\\000\\377\\377\\377\\377\\377\\377\\377)] def
  /dirtycollection [19 14 (\\377\\377\\377\\374\\117\\377\\374\\227\\377\\374\\043\\377\\375\\001\\377\\375\\156\\377\\375\\272\\377\\374\\000\\377\\374\\204\\377\\375\\366\\377\\375\\336\\377\\374\\000\\377\\374\\000\\377\\377\\377\\377)] def
  /dirtyfile [19 14 (\\377\\377\\377\\374\\037\\377\\374\\227\\377\\374\\023\\377\\375\\121\\377\\374\\037\\377\\374\\200\\377\\374\\022\\377\\374\\200\\377\\374\\104\\377\\375\\020\\377\\374\\104\\377\\374\\000\\377\\377\\377\\377)] def
  /dirtylibrary [19 14 (\\377\\377\\377\\374\\017\\377\\375\\227\\377\\374\\303\\377\\375\\151\\377\\374\\262\\377\\375\\030\\377\\375\\214\\377\\374\\326\\377\\374\\142\\377\\375\\060\\377\\374\\132\\377\\375\\000\\377\\377\\377\\377)] def
  /dirtyobjectfile [19 14 (\\377\\377\\377\\374\\017\\377\\375\\127\\377\\375\\203\\377\\375\\321\\377\\374\\344\\377\\374\\160\\377\\375\\072\\377\\374\\034\\377\\375\\116\\377\\374\\046\\377\\374\\212\\377\\374\\000\\377\\377\\377\\377)] def
  /dirtyprogram [19 14 (\\377\\377\\377\\342\\017\\377\\376\\113\\377\\362\\011\\377\\376\\120\\377\\344\\037\\377\\374\\200\\377\\344\\022\\377\\375\\000\\377\\370\\211\\377\\372\\041\\377\\370\\211\\377\\370\\001\\377\\377\\377\\377)] def
  /dirtyproject [19 14 (\\377\\377\\377\\377\\037\\377\\376\\357\\377\\300\\000\\177\\342\\102\\177\\310\\010\\177\\301\\041\\177\\310\\010\\177\\304\\104\\177\\321\\021\\177\\304\\104\\177\\302\\000\\177\\320\\042\\177\\377\\377\\377)] def
  /disablebreakpoint [18 16 (\\377\\377\\377\\374\\017\\377\\370\\007\\377\\360\\303\\377\\340\\301\\377\\300\\300\\377\\300\\300\\377\\300\\300\\377\\300\\300\\377\\300\\300\\377\\300\\300\\377\\340\\301\\377\\360\\303\\377\\370\\007\\377\\374\\017\\377\\377\\377\\377)] def
  /dot [16 16 (\\377\\377\\377\\377\\377\\377\\374\\077\\360\\017\\360\\017\\340\\007\\340\\007\\340\\007\\340\\007\\360\\017\\360\\017\\374\\077\\377\\377\\377\\377\\377\\377)] def
  /dotsx4 [4 4 (\\177\\377\\377\\377)] def
  /dotsx8 [8 8 (\\177\\377\\377\\377\\377\\377\\377\\377)] def
  /editor [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\300\\000\\000\\377\\317\\374\\000\\377\\300\\000\\000\\377\\317\\000\\000\\377\\300\\000\\000\\377\\317\\340\\000\\377\\300\\000\\000\\377\\317\\377\\374\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /enablebreakpoint [18 16 (\\377\\377\\377\\374\\017\\377\\370\\007\\377\\360\\003\\377\\340\\001\\377\\300\\000\\377\\300\\000\\377\\317\\374\\377\\317\\374\\377\\300\\000\\377\\300\\000\\377\\340\\001\\377\\360\\003\\377\\370\\007\\377\\374\\017\\377\\377\\377\\377)] def
  /error [19 14 (\\377\\377\\377\\376\\367\\377\\377\\157\\377\\377\\237\\377\\377\\017\\377\\371\\011\\377\\364\\002\\377\\376\\007\\377\\370\\001\\377\\366\\006\\377\\374\\003\\377\\373\\015\\377\\373\\235\\377\\377\\377\\377)] def
  /fat_lines [8 8 (\\125\\335\\335\\335\\125\\335\\335\\335)] def
  /file [19 14 (\\377\\377\\377\\374\\037\\377\\374\\027\\377\\374\\023\\377\\374\\021\\377\\374\\037\\377\\374\\000\\377\\374\\000\\377\\374\\000\\377\\374\\000\\377\\374\\000\\377\\374\\000\\377\\374\\000\\377\\377\\377\\377)] def
  /frame [16 14 (\\377\\377\\307\\343\\300\\003\\310\\023\\347\\347\\344\\047\\344\\047\\344\\047\\344\\047\\347\\347\\310\\023\\300\\003\\307\\343\\377\\377)] def
  /grapher [26 13 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\301\\200\\060\\377\\300\\300\\140\\377\\300\\140\\300\\377\\301\\377\\360\\377\\300\\140\\300\\377\\300\\300\\140\\377\\301\\200\\060\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /gray [2 2 (\\177\\277)] def
  /gray3 [16 16 (\\167\\167\\377\\377\\335\\335\\377\\377\\167\\167\\377\\377\\335\\335\\377\\377\\167\\167\\377\\377\\335\\335\\377\\377\\167\\167\\377\\377\\335\\335\\377\\377)] def
  /infolinkfrom [22 14 (\\377\\377\\377\\300\\000\\017\\300\\000\\017\\303\\361\\217\\301\\361\\217\\300\\360\\017\\301\\363\\217\\303\\261\\217\\307\\021\\217\\316\\001\\217\\304\\003\\317\\300\\000\\017\\300\\000\\017\\377\\377\\377)] def
  /infolinkto [22 14 (\\377\\377\\377\\300\\000\\017\\300\\000\\017\\304\\001\\217\\316\\001\\217\\307\\020\\017\\303\\263\\217\\301\\361\\217\\300\\361\\217\\301\\361\\217\\303\\363\\317\\300\\000\\017\\300\\000\\017\\377\\377\\377)] def
  /infonote [16 14 (\\377\\377\\300\\003\\300\\003\\301\\203\\301\\203\\300\\003\\303\\203\\301\\203\\301\\203\\301\\203\\303\\303\\300\\003\\300\\003\\377\\377)] def
  /languageelement [19 14 (\\377\\377\\377\\377\\217\\377\\376\\003\\377\\376\\003\\377\\374\\001\\377\\374\\001\\377\\374\\001\\377\\374\\001\\377\\374\\001\\377\\374\\001\\377\\376\\003\\377\\376\\003\\377\\377\\217\\377\\377\\377\\377)] def
  /library [19 14 (\\377\\377\\377\\374\\017\\377\\375\\207\\377\\374\\303\\377\\374\\141\\377\\374\\060\\377\\375\\030\\377\\375\\214\\377\\374\\306\\377\\374\\142\\377\\374\\060\\377\\374\\030\\377\\374\\000\\377\\377\\377\\377)] def
  /library2 [15 15 (\\377\\377\\300\\177\\300\\077\\300\\037\\300\\017\\300\\007\\337\\367\\300\\007\\300\\007\\300\\007\\337\\367\\300\\007\\300\\007\\300\\007\\377\\377)] def
  /light_gray [10 12 (\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377)] def
  /link1 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\337\\377\\376\\377\\320\\036\\002\\377\\300\\014\\000\\377\\307\\314\\370\\377\\307\\314\\370\\377\\307\\314\\370\\377\\307\\314\\370\\377\\300\\014\\000\\377\\320\\036\\002\\377\\337\\377\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /link2 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\377\\376\\377\\317\\036\\002\\377\\337\\014\\000\\377\\330\\314\\370\\377\\376\\314\\370\\377\\376\\314\\370\\377\\330\\314\\370\\377\\337\\014\\000\\377\\317\\036\\002\\377\\300\\377\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /link3 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\007\\376\\377\\317\\346\\002\\377\\337\\364\\000\\377\\330\\064\\370\\377\\376\\174\\370\\377\\376\\174\\370\\377\\330\\064\\370\\377\\337\\364\\000\\377\\317\\346\\002\\377\\300\\007\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /link4 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\000\\076\\377\\317\\341\\302\\377\\337\\363\\300\\377\\330\\063\\070\\377\\376\\177\\270\\377\\376\\177\\270\\377\\330\\063\\070\\377\\337\\363\\300\\377\\317\\341\\302\\377\\300\\000\\076\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /link5 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\000\\000\\377\\317\\341\\374\\377\\337\\363\\376\\377\\330\\063\\006\\377\\376\\377\\337\\377\\376\\377\\337\\377\\330\\063\\006\\377\\337\\363\\376\\377\\317\\341\\374\\377\\300\\000\\000\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /linkfrom [16 14 (\\377\\377\\300\\003\\300\\003\\303\\363\\301\\363\\300\\363\\301\\363\\303\\263\\307\\023\\316\\003\\304\\003\\300\\003\\300\\003\\377\\377)] def
  /linkto [16 14 (\\377\\377\\300\\003\\300\\003\\304\\003\\316\\003\\307\\023\\303\\263\\301\\363\\300\\363\\301\\363\\303\\363\\300\\003\\300\\003\\377\\377)] def
  /makefile [19 14 (\\377\\377\\377\\370\\007\\377\\370\\003\\377\\370\\001\\377\\370\\040\\377\\370\\370\\377\\370\\370\\377\\371\\374\\377\\370\\370\\377\\370\\370\\377\\370\\040\\377\\370\\000\\377\\370\\000\\377\\377\\377\\377)] def
  /memory [13 14 (\\377\\377\\370\\377\\340\\077\\340\\077\\300\\037\\337\\337\\300\\037\\300\\037\\337\\337\\300\\037\\340\\077\\340\\077\\370\\377\\377\\377)] def
  /modifiedtoplevelform [16 14 (\\377\\377\\300\\007\\337\\367\\345\\017\\303\\247\\327\\307\\317\\357\\301\\007\\345\\127\\311\\007\\305\\117\\320\\227\\377\\377\\377\\377)] def
  /monoarchive_tool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\337\\301\\376\\377\\337\\326\\376\\377\\337\\333\\176\\377\\337\\315\\176\\377\\337\\326\\176\\377\\337\\333\\176\\377\\337\\335\\176\\377\\337\\300\\176\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monoarrow [14 14 (\\000\\034\\125\\134\\052\\214\\125\\114\\052\\244\\125\\124\\052\\240\\125\\120\\052\\244\\125\\104\\052\\214\\125\\114\\052\\234\\000\\034)] def
  /monoarrow-hollow [14 14 (\\000\\034\\177\\234\\177\\314\\177\\314\\177\\344\\177\\344\\177\\360\\177\\360\\177\\344\\177\\344\\177\\314\\177\\314\\177\\234\\000\\034)] def
  /monoblip [3 3 (\\377\\377\\377)] def
  /monobtool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\337\\303\\376\\377\\337\\273\\036\\377\\336\\174\\336\\377\\334\\377\\336\\377\\334\\034\\336\\377\\335\\333\\036\\377\\337\\333\\376\\377\\337\\303\\376\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monobuild [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\337\\277\\376\\377\\336\\117\\176\\377\\336\\354\\236\\377\\335\\365\\336\\377\\336\\353\\356\\377\\336\\115\\336\\377\\337\\274\\236\\377\\337\\377\\176\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monocollect_tool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\337\\301\\376\\377\\337\\336\\376\\377\\337\\337\\176\\377\\337\\300\\176\\377\\337\\337\\176\\377\\337\\300\\176\\377\\337\\337\\176\\377\\337\\300\\176\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monocollection [19 14 (\\377\\377\\377\\374\\007\\377\\375\\373\\377\\375\\375\\377\\374\\000\\377\\374\\000\\377\\375\\376\\377\\375\\376\\377\\374\\000\\377\\374\\000\\377\\375\\376\\377\\375\\376\\377\\374\\000\\377\\377\\377\\377)] def
  /monocommand_tool [26 14 (\\377\\377\\377\\300\\340\\000\\001\\300\\337\\377\\376\\300\\336\\075\\376\\300\\334\\231\\376\\300\\335\\333\\376\\300\\334\\223\\036\\300\\336\\066\\116\\300\\337\\346\\356\\300\\337\\356\\116\\300\\337\\317\\036\\300\\337\\377\\376\\300\\340\\000\\001\\300\\377\\377\\377\\300)] def
  /monocompile1 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\337\\377\\376\\377\\337\\377\\376\\377\\334\\341\\306\\377\\334\\344\\222\\377\\330\\141\\236\\377\\330\\144\\236\\377\\323\\044\\222\\377\\323\\041\\306\\377\\337\\377\\376\\377\\337\\377\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /monocompile2 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\201\\377\\377\\177\\201\\377\\377\\177\\230\\341\\307\\177\\232\\344\\223\\177\\232\\141\\237\\177\\232\\144\\237\\177\\233\\044\\223\\177\\231\\041\\307\\177\\201\\377\\377\\177\\201\\377\\377\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /monocompile3 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\200\\007\\377\\177\\200\\007\\377\\177\\231\\311\\307\\177\\233\\154\\223\\177\\233\\151\\237\\177\\233\\154\\237\\177\\233\\154\\223\\177\\231\\311\\307\\177\\200\\007\\377\\177\\200\\007\\377\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /monocompile4 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\200\\000\\037\\177\\200\\000\\037\\177\\231\\314\\347\\177\\233\\155\\263\\177\\233\\155\\277\\177\\233\\155\\277\\177\\233\\155\\263\\177\\231\\314\\347\\177\\200\\000\\037\\177\\200\\000\\037\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /monocompile5 [26 14 (\\377\\377\\377\\377\\200\\000\\000\\177\\200\\000\\000\\177\\200\\000\\000\\177\\231\\314\\346\\177\\233\\155\\266\\177\\233\\155\\266\\177\\233\\155\\266\\177\\233\\155\\266\\177\\231\\314\\346\\177\\200\\000\\000\\177\\200\\000\\000\\177\\200\\000\\000\\177\\377\\377\\377\\377)] def
  /monocurrenterror [19 14 (\\300\\001\\377\\335\\355\\377\\336\\336\\377\\337\\076\\377\\336\\337\\177\\322\\323\\177\\311\\345\\277\\335\\357\\277\\323\\363\\177\\315\\355\\177\\331\\346\\377\\326\\332\\377\\327\\075\\377\\300\\001\\377)] def
  /monocurrentnote [19 14 (\\377\\377\\377\\360\\003\\377\\367\\375\\377\\364\\175\\377\\365\\236\\377\\365\\346\\377\\365\\373\\177\\365\\373\\177\\365\\346\\377\\365\\236\\377\\364\\175\\377\\367\\375\\377\\360\\003\\377\\377\\377\\377)] def
  /monocurrentwarning [19 14 (\\377\\377\\377\\340\\003\\377\\357\\375\\377\\357\\075\\377\\357\\076\\377\\356\\336\\377\\356\\337\\177\\355\\357\\177\\355\\356\\377\\353\\366\\377\\350\\005\\377\\357\\375\\377\\340\\003\\377\\377\\377\\377)] def
  /monodbox-error [30 30 (\\377\\377\\377\\377\\377\\340\\077\\377\\377\\000\\007\\377\\376\\037\\303\\377\\370\\177\\360\\377\\361\\377\\374\\177\\360\\377\\376\\177\\344\\177\\377\\077\\306\\077\\377\\037\\317\\037\\377\\237\\317\\217\\377\\237\\237\\307\\377\\317\\237\\343\\377\\317\\237\\361\\377\\317\\237\\370\\377\\317\\237\\374\\177\\317\\237\\376\\077\\317\\237\\377\\037\\317\\317\\377\\217\\237\\317\\377\\307\\237\\307\\377\\343\\037\\347\\377\\361\\077\\363\\377\\370\\177\\361\\377\\370\\177\\370\\177\\360\\377\\376\\037\\303\\377\\377\\000\\007\\377\\377\\340\\077\\377\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /monodbox-info [30 30 (\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\374\\177\\377\\377\\370\\077\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\370\\077\\377\\377\\374\\177\\377\\377\\377\\377\\377\\377\\200\\037\\377\\377\\200\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\360\\037\\377\\377\\200\\003\\377\\377\\200\\003\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /monodbox-question [30 30 (\\377\\340\\077\\377\\377\\000\\007\\377\\376\\000\\003\\377\\370\\017\\200\\377\\360\\037\\300\\177\\360\\077\\340\\177\\340\\070\\340\\077\\300\\070\\340\\037\\300\\070\\340\\037\\300\\070\\340\\037\\200\\070\\340\\017\\200\\001\\340\\017\\200\\003\\300\\017\\200\\007\\200\\017\\200\\007\\000\\017\\200\\007\\000\\017\\200\\007\\000\\017\\300\\007\\000\\037\\300\\007\\000\\037\\300\\000\\000\\037\\340\\007\\000\\077\\360\\007\\000\\177\\360\\007\\000\\177\\370\\000\\000\\377\\376\\000\\003\\377\\377\\000\\007\\377\\377\\340\\077\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /monodebugger [26 14 (\\377\\377\\377\\377\\340\\000\\003\\377\\337\\355\\361\\377\\337\\363\\302\\377\\337\\355\\016\\377\\337\\050\\076\\377\\336\\200\\336\\377\\337\\300\\376\\377\\337\\006\\076\\377\\334\\076\\336\\377\\320\\236\\176\\377\\343\\155\\276\\377\\360\\000\\001\\377\\377\\377\\377\\377)] def
  /monodirectories [24 14 (\\377\\377\\377\\377\\377\\377\\376\\037\\377\\375\\340\\077\\375\\377\\337\\340\\000\\337\\357\\377\\137\\357\\377\\137\\367\\377\\237\\367\\377\\237\\373\\377\\337\\374\\000\\037\\377\\377\\377\\377\\377\\377)] def
  /monodirectory [19 14 (\\377\\377\\377\\377\\377\\377\\374\\077\\377\\373\\300\\377\\373\\377\\177\\300\\003\\177\\337\\375\\177\\337\\375\\177\\357\\376\\177\\357\\376\\177\\367\\377\\177\\370\\000\\177\\377\\377\\377\\377\\377\\377)] def
  /monodirtycollection [19 14 (\\377\\377\\377\\374\\007\\377\\375\\173\\377\\376\\355\\377\\374\\200\\377\\374\\010\\377\\375\\276\\377\\375\\332\\377\\374\\020\\377\\374\\000\\377\\374\\372\\377\\375\\136\\377\\375\\000\\377\\377\\377\\377)] def
  /monodirtyfile [19 14 (\\377\\377\\377\\376\\007\\377\\376\\343\\377\\376\\265\\377\\376\\346\\377\\376\\260\\177\\376\\327\\177\\376\\155\\177\\376\\377\\177\\376\\273\\177\\376\\356\\177\\376\\267\\177\\376\\000\\177\\377\\377\\377)] def
  /monodirtylibrary [19 14 (\\377\\377\\377\\374\\017\\377\\374\\147\\377\\375\\073\\377\\374\\215\\377\\375\\114\\377\\374\\346\\377\\374\\162\\377\\375\\050\\377\\375\\234\\377\\374\\316\\377\\375\\244\\377\\374\\000\\377\\377\\377\\377)] def
  /monodirtyobjectfile [19 14 (\\377\\377\\377\\376\\007\\377\\376\\253\\377\\376\\335\\377\\376\\344\\377\\376\\167\\177\\376\\072\\177\\376\\335\\177\\376\\156\\177\\376\\327\\177\\376\\373\\177\\376\\151\\177\\376\\000\\177\\377\\377\\377)] def
  /monodirtyprogram [19 14 (\\377\\377\\377\\302\\007\\377\\376\\363\\377\\342\\265\\377\\376\\356\\377\\305\\140\\377\\375\\376\\377\\344\\326\\377\\375\\376\\377\\313\\171\\377\\371\\355\\377\\373\\275\\377\\370\\001\\377\\377\\377\\377)] def
  /monodirtyproject [19 14 (\\377\\377\\377\\377\\037\\377\\376\\357\\377\\300\\000\\177\\335\\275\\177\\327\\367\\177\\336\\336\\177\\327\\367\\177\\333\\273\\177\\316\\356\\177\\333\\273\\177\\335\\377\\177\\300\\000\\177\\377\\377\\377)] def
  /monodisablebreakpoint [18 16 (\\377\\377\\377\\374\\017\\377\\373\\367\\377\\367\\073\\377\\357\\075\\377\\337\\076\\377\\337\\076\\377\\337\\076\\377\\337\\076\\377\\337\\076\\377\\337\\076\\377\\357\\075\\377\\367\\073\\377\\373\\367\\377\\374\\017\\377\\377\\377\\377)] def
  /monodot [16 16 (\\377\\377\\377\\377\\377\\377\\374\\077\\360\\017\\360\\017\\340\\007\\340\\007\\340\\007\\340\\007\\360\\017\\360\\017\\374\\077\\377\\377\\377\\377\\377\\377)] def
  /monodotsx4 [4 4 (\\177\\377\\377\\377)] def
  /monodotsx8 [8 8 (\\177\\377\\377\\377\\377\\377\\377\\377)] def
  /monoeditor [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\337\\377\\376\\377\\320\\003\\376\\377\\337\\377\\376\\377\\320\\377\\376\\377\\337\\377\\376\\377\\320\\037\\376\\377\\337\\377\\376\\377\\320\\000\\002\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monoenablebreakpoint [18 16 (\\377\\377\\377\\374\\017\\377\\373\\367\\377\\367\\373\\377\\357\\375\\377\\337\\376\\377\\337\\376\\377\\320\\002\\377\\320\\002\\377\\337\\376\\377\\337\\376\\377\\357\\375\\377\\367\\373\\377\\373\\367\\377\\374\\017\\377\\377\\377\\377)] def
  /monoerror [19 14 (\\377\\377\\377\\376\\367\\377\\377\\157\\377\\377\\237\\377\\377\\157\\377\\371\\151\\377\\364\\362\\377\\376\\367\\377\\371\\371\\377\\366\\366\\377\\374\\363\\377\\373\\155\\377\\373\\235\\377\\377\\377\\377)] def
  /monofat_lines [8 8 (\\125\\335\\335\\335\\125\\335\\335\\335)] def
  /monofile [19 14 (\\377\\377\\377\\376\\003\\377\\376\\365\\377\\376\\366\\377\\376\\367\\177\\376\\360\\177\\376\\377\\177\\376\\377\\177\\376\\377\\177\\376\\377\\177\\376\\377\\177\\376\\377\\177\\376\\000\\177\\377\\377\\377)] def
  /monoframe [16 14 (\\377\\377\\307\\343\\330\\033\\327\\353\\350\\027\\353\\327\\353\\327\\353\\327\\353\\327\\350\\027\\327\\353\\330\\033\\307\\343\\377\\377)] def
  /monographer [26 13 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\336\\177\\316\\377\\337\\077\\236\\377\\337\\237\\076\\377\\336\\000\\016\\377\\337\\237\\076\\377\\337\\077\\236\\377\\336\\177\\316\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monogray [2 2 (\\177\\277)] def
  /monogray3 [16 16 (\\167\\167\\377\\377\\335\\335\\377\\377\\167\\167\\377\\377\\335\\335\\377\\377\\167\\167\\377\\377\\335\\335\\377\\377\\167\\167\\377\\377\\335\\335\\377\\377)] def
  /monoinfolinkfrom [22 14 (\\377\\377\\377\\300\\000\\017\\337\\377\\357\\334\\016\\157\\336\\356\\157\\337\\157\\357\\336\\254\\157\\335\\116\\157\\332\\356\\157\\325\\376\\157\\333\\374\\057\\337\\377\\357\\300\\000\\017\\377\\377\\377)] def
  /monoinfolinkto [22 14 (\\377\\377\\377\\300\\000\\017\\337\\377\\357\\333\\376\\157\\325\\376\\157\\332\\357\\357\\335\\114\\157\\336\\256\\157\\337\\156\\157\\336\\356\\157\\334\\014\\057\\337\\377\\357\\300\\000\\017\\377\\377\\377)] def
  /monoinfonote [16 14 (\\377\\377\\300\\003\\300\\003\\301\\203\\301\\203\\300\\003\\303\\203\\301\\203\\301\\203\\301\\203\\303\\303\\300\\003\\300\\003\\377\\377)] def
  /monolanguageelement [19 14 (\\377\\377\\377\\377\\217\\377\\376\\163\\377\\376\\373\\377\\375\\375\\377\\375\\375\\377\\375\\375\\377\\375\\375\\377\\375\\375\\377\\375\\375\\377\\376\\373\\377\\376\\163\\377\\377\\217\\377\\377\\377\\377)] def
  /monolibrary [19 14 (\\377\\377\\377\\374\\017\\377\\374\\167\\377\\375\\073\\377\\375\\235\\377\\375\\316\\377\\374\\346\\377\\374\\162\\377\\375\\070\\377\\375\\234\\377\\375\\316\\377\\375\\346\\377\\374\\000\\377\\377\\377\\377)] def
  /monolibrary2 [15 15 (\\377\\377\\300\\177\\337\\277\\337\\337\\337\\357\\337\\367\\300\\007\\337\\367\\337\\367\\337\\367\\300\\007\\337\\367\\337\\367\\300\\007\\377\\377)] def
  /monolight_gray [10 12 (\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377\\125\\177\\377\\377)] def
  /monolink1 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\337\\377\\376\\377\\320\\036\\002\\377\\300\\014\\000\\377\\307\\314\\370\\377\\307\\314\\370\\377\\307\\314\\370\\377\\307\\314\\370\\377\\300\\014\\000\\377\\320\\036\\002\\377\\337\\377\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /monolink2 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\377\\376\\377\\317\\036\\002\\377\\337\\014\\000\\377\\330\\314\\370\\377\\376\\314\\370\\377\\376\\314\\370\\377\\330\\314\\370\\377\\337\\014\\000\\377\\317\\036\\002\\377\\300\\377\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /monolink3 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\007\\376\\377\\317\\346\\002\\377\\337\\364\\000\\377\\330\\064\\370\\377\\376\\174\\370\\377\\376\\174\\370\\377\\330\\064\\370\\377\\337\\364\\000\\377\\317\\346\\002\\377\\300\\007\\376\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /monolink4 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\000\\076\\377\\317\\341\\302\\377\\337\\363\\300\\377\\330\\063\\070\\377\\376\\177\\270\\377\\376\\177\\270\\377\\330\\063\\070\\377\\337\\363\\300\\377\\317\\341\\302\\377\\300\\000\\076\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /monolink5 [26 14 (\\377\\377\\377\\377\\300\\000\\000\\377\\300\\000\\000\\377\\317\\341\\374\\377\\337\\363\\376\\377\\330\\063\\006\\377\\376\\377\\337\\377\\376\\377\\337\\377\\330\\063\\006\\377\\337\\363\\376\\377\\317\\341\\374\\377\\300\\000\\000\\377\\300\\000\\000\\377\\377\\377\\377\\377)] def
  /monolinkfrom [16 14 (\\377\\377\\300\\003\\337\\373\\334\\013\\336\\353\\337\\153\\336\\253\\335\\113\\332\\353\\325\\373\\333\\373\\337\\373\\300\\003\\377\\377)] def
  /monolinkto [16 14 (\\377\\377\\300\\003\\300\\003\\304\\003\\316\\003\\307\\023\\303\\263\\301\\363\\300\\363\\301\\363\\303\\363\\300\\003\\300\\003\\377\\377)] def
  /monomakefile [19 14 (\\377\\377\\377\\374\\007\\377\\375\\373\\377\\375\\375\\377\\375\\356\\377\\375\\223\\177\\375\\273\\177\\375\\175\\177\\375\\273\\177\\375\\223\\177\\375\\357\\177\\375\\377\\177\\374\\000\\177\\377\\377\\377)] def
  /monomemory [13 14 (\\377\\377\\370\\377\\347\\077\\357\\277\\337\\337\\300\\037\\337\\337\\337\\337\\300\\037\\337\\337\\357\\277\\347\\077\\370\\377\\377\\377)] def
  /monomodifiedtoplevelform [16 14 (\\377\\377\\377\\377\\340\\017\\376\\377\\355\\157\\373\\277\\361\\027\\335\\177\\365\\157\\375\\177\\354\\167\\377\\337\\366\\377\\377\\377)] def
  /mononote [19 14 (\\377\\377\\377\\377\\377\\377\\371\\377\\377\\372\\177\\377\\373\\237\\377\\373\\347\\377\\373\\371\\377\\373\\371\\377\\373\\347\\377\\373\\237\\377\\372\\177\\377\\371\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /mononote1 [16 14 (\\377\\377\\300\\003\\337\\373\\337\\373\\337\\373\\337\\373\\336\\173\\336\\173\\337\\373\\337\\373\\337\\373\\337\\373\\300\\003\\377\\377)] def
  /mononote2 [16 14 (\\377\\377\\300\\003\\337\\373\\337\\313\\337\\313\\337\\373\\337\\373\\337\\373\\337\\373\\323\\373\\323\\373\\337\\373\\300\\003\\377\\377)] def
  /mononote3 [16 14 (\\377\\377\\300\\003\\337\\373\\337\\313\\337\\313\\337\\373\\336\\173\\336\\173\\337\\373\\323\\373\\323\\373\\337\\373\\300\\003\\377\\377)] def
  /monoobjectfile [19 14 (\\377\\377\\377\\376\\007\\377\\376\\273\\377\\376\\335\\377\\376\\356\\377\\376\\167\\177\\376\\273\\177\\376\\335\\177\\376\\356\\177\\376\\367\\177\\376\\373\\177\\376\\375\\177\\376\\000\\177\\377\\377\\377)] def
  /monoobjectfile1 [19 14 (\\377\\377\\377\\376\\007\\377\\376\\223\\377\\376\\301\\377\\376\\352\\377\\376\\160\\177\\376\\271\\177\\376\\034\\177\\376\\216\\177\\376\\047\\177\\376\\003\\177\\376\\225\\177\\376\\000\\177\\377\\377\\377)] def
  /monoobjectfile2 [14 14 (\\377\\377\\300\\377\\337\\177\\337\\277\\337\\337\\337\\357\\337\\357\\300\\017\\300\\017\\337\\357\\337\\357\\337\\357\\300\\017\\377\\377)] def
  /monoopeninfonote [16 14 (\\377\\377\\300\\003\\337\\373\\336\\173\\336\\173\\337\\373\\334\\173\\336\\173\\336\\173\\336\\173\\300\\003\\337\\373\\300\\003\\377\\377)] def
  /monoopennote1 [16 14 (\\377\\377\\300\\003\\337\\373\\337\\373\\337\\373\\337\\373\\336\\173\\336\\173\\337\\373\\337\\373\\300\\003\\337\\373\\300\\003\\377\\377)] def
  /monoopennote2 [16 14 (\\377\\377\\300\\003\\337\\373\\337\\313\\337\\313\\337\\373\\337\\373\\337\\373\\337\\373\\323\\373\\300\\003\\337\\373\\300\\003\\377\\377)] def
  /monoopennote3 [16 14 (\\377\\377\\300\\003\\337\\373\\337\\313\\337\\313\\337\\373\\336\\173\\336\\173\\337\\373\\323\\373\\300\\003\\337\\373\\300\\003\\377\\377)] def
  /monopendingtool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\337\\377\\376\\377\\337\\377\\376\\377\\337\\377\\376\\377\\337\\377\\376\\377\\337\\377\\376\\377\\334\\314\\316\\377\\334\\314\\316\\377\\337\\377\\376\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monoprogram [19 14 (\\377\\377\\377\\302\\007\\377\\376\\363\\377\\342\\365\\377\\376\\356\\377\\305\\340\\377\\375\\376\\377\\345\\376\\377\\375\\376\\377\\313\\375\\377\\373\\375\\377\\373\\375\\377\\370\\001\\377\\377\\377\\377)] def
  /monoproject [19 14 (\\377\\377\\377\\377\\037\\377\\376\\357\\377\\300\\000\\177\\337\\377\\177\\337\\377\\177\\337\\377\\177\\337\\377\\177\\337\\377\\177\\337\\377\\177\\337\\377\\177\\337\\377\\177\\300\\000\\177\\377\\377\\377)] def
  /monorandom [16 16 (\\377\\276\\357\\357\\376\\377\\177\\377\\357\\337\\376\\376\\377\\367\\373\\277\\277\\367\\377\\376\\367\\277\\377\\366\\376\\377\\357\\377\\177\\273\\375\\377)] def
  /monoshlib_tool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\334\\301\\316\\377\\333\\326\\366\\377\\333\\333\\166\\377\\333\\315\\166\\377\\333\\326\\166\\377\\333\\333\\166\\377\\333\\335\\166\\377\\334\\300\\116\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monoslant [16 16 (\\357\\357\\337\\337\\277\\277\\177\\177\\376\\376\\375\\375\\373\\373\\367\\367\\357\\357\\337\\337\\277\\277\\177\\177\\376\\376\\375\\375\\373\\373\\367\\367)] def
  /monosquare [7 8 (\\376\\202\\174\\174\\174\\174\\174\\202)] def
  /monotoolstat [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\337\\377\\376\\377\\330\\000\\006\\377\\333\\377\\366\\377\\333\\377\\366\\377\\320\\000\\002\\377\\320\\000\\002\\377\\320\\000\\002\\377\\320\\000\\002\\377\\337\\377\\376\\377\\337\\377\\376\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /monotoplevelform [16 14 (\\377\\377\\377\\377\\340\\017\\376\\377\\375\\177\\373\\277\\361\\037\\375\\177\\375\\177\\375\\177\\374\\177\\377\\377\\377\\377\\377\\377)] def
  /monovisitederror [16 14 (\\377\\377\\373\\337\\375\\277\\376\\177\\375\\277\\345\\247\\323\\313\\372\\137\\346\\147\\333\\333\\363\\317\\355\\267\\356\\167\\377\\377)] def
  /monovisitedwarning [14 14 (\\377\\377\\377\\377\\377\\377\\374\\377\\374\\377\\373\\177\\373\\177\\367\\277\\367\\277\\357\\337\\354\\337\\334\\357\\300\\017\\377\\377)] def
  /monowarning [19 14 (\\377\\377\\377\\377\\377\\377\\377\\237\\377\\377\\237\\377\\377\\157\\377\\377\\157\\377\\376\\367\\377\\376\\367\\377\\375\\373\\377\\375\\373\\377\\373\\375\\377\\370\\001\\377\\377\\377\\377\\377\\377\\377)] def
  /note [19 14 (\\377\\377\\377\\377\\377\\377\\371\\377\\377\\370\\177\\377\\370\\037\\377\\370\\007\\377\\370\\001\\377\\370\\001\\377\\370\\007\\377\\370\\037\\377\\370\\177\\377\\371\\377\\377\\377\\377\\377\\377\\377\\377)] def
  /note1 [16 14 (\\377\\377\\300\\003\\300\\003\\300\\003\\300\\003\\300\\003\\301\\203\\301\\203\\300\\003\\300\\003\\300\\003\\300\\003\\300\\003\\377\\377)] def
  /note2 [16 14 (\\377\\377\\300\\003\\300\\003\\300\\063\\300\\063\\300\\003\\300\\003\\300\\003\\300\\003\\314\\003\\314\\003\\300\\003\\300\\003\\377\\377)] def
  /note3 [16 14 (\\377\\377\\300\\003\\300\\003\\300\\063\\300\\063\\300\\003\\301\\203\\301\\203\\300\\003\\314\\003\\314\\003\\300\\003\\300\\003\\377\\377)] def
  /objectfile [19 14 (\\377\\377\\377\\374\\017\\377\\375\\007\\377\\375\\203\\377\\375\\301\\377\\374\\340\\377\\374\\160\\377\\374\\070\\377\\374\\034\\377\\374\\016\\377\\374\\006\\377\\374\\002\\377\\374\\000\\377\\377\\377\\377)] def
  /objectfile1 [14 14 (\\377\\377\\300\\377\\320\\177\\330\\077\\334\\037\\316\\017\\307\\017\\303\\217\\301\\317\\300\\357\\300\\157\\300\\057\\300\\017\\377\\377)] def
  /objectfile2 [14 14 (\\377\\377\\300\\377\\300\\177\\300\\077\\300\\037\\300\\017\\300\\017\\337\\357\\337\\357\\300\\017\\300\\017\\300\\017\\300\\017\\377\\377)] def
  /openinfonote [16 14 (\\377\\377\\300\\003\\300\\003\\301\\203\\301\\203\\300\\003\\303\\203\\301\\203\\301\\203\\301\\203\\303\\303\\337\\373\\300\\003\\377\\377)] def
  /opennote1 [16 14 (\\377\\377\\300\\003\\300\\003\\300\\003\\300\\003\\300\\003\\301\\203\\301\\203\\300\\003\\300\\003\\300\\003\\337\\373\\300\\003\\377\\377)] def
  /opennote2 [16 14 (\\377\\377\\300\\003\\300\\003\\300\\063\\300\\063\\300\\003\\300\\003\\300\\003\\300\\003\\314\\003\\314\\003\\337\\373\\300\\003\\377\\377)] def
  /opennote3 [16 14 (\\377\\377\\300\\003\\300\\003\\300\\063\\300\\063\\300\\003\\301\\203\\301\\203\\300\\003\\314\\003\\314\\003\\337\\373\\300\\003\\377\\377)] def
  /pendingtool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\300\\000\\000\\377\\300\\000\\000\\377\\300\\000\\000\\377\\300\\000\\000\\377\\300\\000\\000\\377\\303\\063\\060\\377\\303\\063\\060\\377\\300\\000\\000\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /program [19 14 (\\377\\377\\377\\342\\017\\377\\376\\013\\377\\362\\011\\377\\376\\020\\377\\344\\037\\377\\374\\000\\377\\344\\000\\377\\374\\000\\377\\370\\001\\377\\370\\001\\377\\370\\001\\377\\370\\001\\377\\377\\377\\377)] def
  /project [19 14 (\\377\\377\\377\\377\\037\\377\\376\\357\\377\\300\\000\\177\\300\\000\\177\\300\\000\\177\\300\\000\\177\\300\\000\\177\\300\\000\\177\\300\\000\\177\\300\\000\\177\\300\\000\\177\\300\\000\\177\\377\\377\\377)] def
  /random [16 16 (\\377\\276\\357\\357\\376\\377\\177\\377\\357\\337\\376\\376\\377\\367\\373\\277\\277\\367\\377\\376\\367\\277\\377\\366\\376\\377\\357\\377\\177\\273\\375\\377)] def
  /shlib_tool [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\306\\174\\060\\377\\310\\122\\010\\377\\310\\111\\010\\377\\310\\145\\010\\377\\310\\123\\010\\377\\310\\111\\010\\377\\310\\105\\010\\377\\306\\177\\060\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /slant [16 16 (\\357\\357\\337\\337\\277\\277\\177\\177\\376\\376\\375\\375\\373\\373\\367\\367\\357\\357\\337\\337\\277\\277\\177\\177\\376\\376\\375\\375\\373\\373\\367\\367)] def
  /square [7 8 (\\376\\202\\000\\000\\000\\000\\000\\202)] def
  /toolstat [26 14 (\\377\\377\\377\\377\\340\\000\\001\\377\\300\\000\\000\\377\\307\\377\\370\\377\\304\\000\\010\\377\\304\\000\\010\\377\\317\\377\\374\\377\\317\\377\\374\\377\\317\\377\\374\\377\\317\\377\\374\\377\\300\\000\\000\\377\\300\\000\\000\\377\\340\\000\\001\\377\\377\\377\\377\\377)] def
  /toplevelform [16 14 (\\377\\377\\300\\007\\337\\367\\301\\007\\303\\207\\307\\307\\317\\347\\301\\007\\301\\007\\301\\007\\301\\007\\300\\007\\377\\377\\377\\377)] def
  /visitederror [16 14 (\\377\\377\\373\\337\\375\\277\\376\\177\\374\\077\\344\\047\\320\\013\\371\\237\\341\\207\\330\\033\\360\\017\\354\\067\\356\\167\\377\\377)] def
  /visitedwarning [14 14 (\\377\\377\\377\\377\\377\\377\\374\\377\\374\\377\\370\\177\\370\\177\\360\\077\\360\\077\\340\\037\\343\\037\\303\\017\\300\\017\\377\\377)] def
  /warning [19 14 (\\377\\377\\377\\377\\377\\377\\377\\237\\377\\377\\237\\377\\377\\017\\377\\377\\017\\377\\376\\007\\377\\376\\007\\377\\374\\003\\377\\374\\003\\377\\370\\001\\377\\370\\001\\377\\377\\377\\377\\377\\377\\377)] def
end

/tm 10.5 72 mul def
/bm 0.5  72 mul def
/lm 0.35 72 mul def
/rm 7.5 72 mul def
/maxht 0 def
/lastht 0 def
/fntht 0 def
/stkct 0 def
/xpos lm def
/ypos tm def
/ColorP systemdict /colorimage known def
/CFtCol /black def
/bcol /none def
/unl false def

/B {
  bcol /none ne {
    dup gsave
      xpos ypos moveto
      true charpath pathbbox
      /y2 exch def /x2 exch def /y1 exch def /x1 exch def
      x1 y1 moveto x2 y1 lineto x2 y2 lineto x1 y2 lineto
      bcol C fill
    grestore
  } if
} def

/C {
  ColorP {
    dup colors exch known {
      colors exch get
           dup 0 get 255 div
      exch dup 1 get 255 div
      exch     2 get 255 div
      setrgbcolor
    } { pop } ifelse
  } { pop } ifelse
} def

/E {
  newpageflag { L showpage } if
  end
} def

/F {
  dup facecat exch known {
    facecat exch get dup [ /F 3 -1 roll ]
    0 M 
    exch dup 0 get setfont 1 get /fntht exch def
  } { pop } ifelse
} def

/L {
  0 1 stkct 1 sub {
    stkct exch sub -1 roll
    dup 0 get
    dup /F eq {
      exch 1 get
      dup 0 get setfont
      dup 2 get /CFtCol exch def
      dup 3 get /bcol exch def
          4 get /unl exch def
    } if
    dup /S eq {
      exch dup
      /xpos exch 1 get def
      gsave
        2 get B
        unl { U } if
        CFtCol C
        xpos ypos moveto
        show
      grestore
    } if
    dup /P eq {
      exch dup 1 get /xpos exch def
      gsave
	/pix exch 2 get def
	xpos ypos 3 sub translate
	% w h i [ w 0 0 h neg 0 h ] { data } imagemask
	pix 0 get  pix 1 get  false  [ 1.3 0 0 -1.3 0 pix 1 get ]
	pix 2 get
        /maroon3 C
        imagemask
      grestore
    } if
    pop
  } for
  /stkct 0 def
} def

/M {
  dup maxht gt { /maxht exch def } { pop } ifelse
  /stkct stkct 1 add def
} def

/N {
  stkct 0 eq {
    /ypos ypos lastht sub def
  } {
    /ypos ypos maxht sub def
    /lastht maxht def
  } ifelse
  ypos bm lt {
    showpage
    /ypos tm def
  } if
  L
  /xpos lm def
  /maxht 0 def
} def

/P {
  /newpageflag true def
  dup pixcat exch known {
    pixcat exch get
    dup dup
    [ /P xpos 2 add 4 -1 roll ] exch 1 get 1.3 div M
    exch 0 get /xpos xpos 3 -1 roll 1.3 div add 4 add def
  } {
    pop
  } ifelse
} def

/S {
  /newpageflag true def
  dup stringwidth pop xpos add
  dup rm gt
  {
    xpos sub exch /s exch def
    /sp rm xpos sub def
    round cvi s length idiv /ct exch def
    s 0 ct getinterval stringwidth pop sp le
    { /os -1 def /ts {gt} def ct ct 1 s length }
    { /os 0 def /ts {le} def  ct ct -1 0 } ifelse
    { exch pop dup s 0 3 -1 roll getinterval stringwidth pop sp ts
      { exit } if } for
    os add /ct exch def s 0 ct getinterval
    dup stringwidth pop exch
    [ /S xpos 4 -1 roll ] fntht M
    exch xpos add /xpos exch def
    [ /S xpos 1 string dup 0 167 put ] fntht M
    N s ct s length ct sub getinterval S
  } {
    exch [ /S xpos 4 -1 roll ] exch fntht M
    /xpos exch def
  } ifelse
} def

/U {
  gsave
    xpos ypos moveto
    dup gsave true charpath pathbbox grestore
    pop exch pop ypos moveto ypos lineto 0.5 setlinewidth stroke
  grestore
} def

%%EndProlog
%%BeginSetup
/default F
%%EndSetup
%%Page: \"1\" 1
"))
    (save-excursion
      (set-buffer buf)
      (delete-region (point-min) (point-max))
      (if energize-annotate-print-ps-defns
	  (insert-file-contents energize-annotate-print-ps-defns)
	(insert def-ps-defns))
      (goto-char (point-max))
      (while evs
	(let ((e (car evs))
	      f)
	  (if (nth 1 e)
	      (progn
		(if f (insert " "))
		(setq f t)
		(insert (concat "/" (prin1-to-string (nth 1 e)) " F"))))
	  (if (nth 2 e)
	      (progn
		(if f (insert " "))
		(setq f t)
		(insert (concat "/" (nth 2 e) " P"))))
	  (if (nth 3 e)
	      (progn
		(if f (insert " "))
		(setq f t)
		(insert "N")))
	  (if (> (length (nth 4 e)) 0)
	      (progn
		(if f (insert " "))
		(setq f t)
		(insert "(" (energize-al-print-ps-string (nth 4 e)) ")S"))))
	(insert "\n")
	(setq evs (cdr evs)))
      (insert "E" "\n"  "%%EndPage: \"1\" 0" "\n" "%%Trailer" "\n")
      (if (and energize-annotate-print-ps-whence
	       (funcall energize-annotate-print-ps-whence buf))
	  (kill-buffer buf))
      )))

(defun energize-al-print-ps-string (s)
  "Generate a PostScript printable version of a string"
  (let ((st 0))
    (while (setq st (string-match "[()\\\\]" s st))
      (setq s (concat (substring s 0 st) "\\" (substring s st)))
      (setq st (+ st 2))))
  s)

(defun energize-al-print-lpr-buffer (buff)
  "Print buffer. Return true to delete buffer"
  (save-excursion
    (set-buffer buff)
    (lpr-buffer)
    t))
