;; ups.el
;;
;; A modified version of `ps-mode.el', an interface to the Unix ``ps'' command
;;      by Levent N Atasoy <L_Atasoy@MacSch.com>
;; Font-lock, menus, mouse button, ... are defined for XEmacs only.
;; Modified by Bob Weiner, 9/30/94, to work with InfoDock.  Also added
;; ups-show-command-paths variable and made default nil.
;;
;; Modified by Joseph J. Nuspl Jr <nuspl@purdue.edu>, 23-Apr-1996
;;   Added toolbar and menu support.
;;
;; Adapted from the original written by Dirk Grunwald, grunwald@m.cs.uiuc.edu
;; University of Illinois

;; This file is part of InfoDock and is made available under
;; the same conditions as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Installation and Customization:
;;  --------------------
;;
;; To use ups put this in your .emacs file:
;;
;;    (autoload 'ups "ups" "A major-mode for sending signals to processes." t)
;;
;; If you want ups to be loaded from the very beginning, you should have
;;
;;  (require 'ups)
;;
;; (add-hook 'ups-hook 'ups-toolbar)
;;
;; You may want to modify the first set of variables listed below.
;; In addition, you may want to redefine ups-deemphasize-face.

;;; Major Changes from `ps-mode':
;;     Added XEmacs features: menus, hilighting, font-lock, icon, ...
;;     Added Non berkeley-unix definitions for SGI, HPUX, ...?
;;     Changed ps-mode-all to a ups-program-list, which is nicer INMO
;;     Changed some of the key-mappings

;;
;; You may want to set these variables in your .emacs
;;

(defvar ups::toolbar-icon-directory nil
  "Where the toolbar icons for UPS are")

(defun ups-icon-file (sig)
  (expand-file-name (concat "~/lib/images/ups-" sig ".xpm")))

(defvar ups::signals
  '(["Alarm"		"Alrm"	("A" "a")]
    ["Continue"		"Cont"	("C" "c")]
    ["Hangup"		"Hup"	("H" "h")]
    ["Interrupt"	"Int"	("I" "i")]
    ["Kill"		"Kill"	("K" "k")]
    ["Stop"		"Stop"	("S" "s")]
    ["Terminate"	"Term"	("T" "t")]
    )
  "*List of signals*")

(defvar ups::menu
  '("UPS Menu")
  "The UPS menu specification")

(defvar ups::toolbar
  (if (featurep 'toolbar)
      (list (first initial-toolbar-spec)
	    (second initial-toolbar-spec)
	    nil))
  "The UPS toolbar")

(defun ups-init-signal (vec)
  "Initialize a signal"
  (interactive)
  (let (name down sig func icon)
    (setq name (aref vec 0)
	  down (downcase name)
	  sig (aref vec 1)
	  keys (aref vec 2)
	  func (intern (concat "ups-mark-" down))
	  icon (intern (concat "ups::toolbar-" down)))
    ;; Define the actual function
    (eval `(defun ,func ()
	     ,(concat "Toggle the " name " signal")
	     (interactive)
	     (ups-mark-line ,sig ,(upcase sig))))
    ;; Define the key mappings
    (mapcar (function (lambda (key)
			(define-key ups-map key func)))
	    keys)
    ;; Define the toolbar icon
    (eval `(defvar ,icon
	     (if (featurep 'toolbar)
		 (toolbar-make-button-list (ups-icon-file ,(downcase sig))))
	     ,(concat name " signal icon")))
    ;; Add it to the toolbar
    (if (featurep 'toolbar)
	(nconc ups::toolbar
	       (list (vector icon func t
			     (concat "Toggle the " name " signal")))))
    ;; Add it to the pop-up menu
    (nconc ups::menu
	   (list (vector (concat "Toggle the " name " signal") func t)))
    ))

(defvar ups::toolbar-nice-icon
  (if (featurep 'toolbar)
      (toolbar-make-button-list (ups-icon-file "nice")))
  "Renice icon")

(defvar ups::toolbar-update-icon
  (if (featurep 'toolbar)
      (toolbar-make-button-list (ups-icon-file "update")))
  "Update icon")

(defvar ups::toolbar-issue-icon
  (if (featurep 'toolbar)
      (toolbar-make-button-list (ups-icon-file "issue")))
  "Issue icon")

(defun ups-toolbar ()
  "Set the UPS toolbar"
  (interactive)
  (if (not (featurep 'toolbar))
      nil
    (set-specifier default-toolbar (cons (current-buffer) ups::toolbar))))


(defvar ups-show-command-paths nil
  "*If non-nil, show full pathnames for each process command.")

(defvar ups-nice-args "+4"
  "*Default niceness.")

(defvar ups-mouse-mark 'ups-mark-kill
  "*Default mark for the function ups-mouse-mark bound to mouse button2")

(cond
 ((eq system-type 'berkeley-unix)
  (defvar ups-executable "ps"
    "*Executable path of 'ps' process summary program to run.")
  (defvar ups-program-list
    (list
     (concat ups-executable " ugxww")
     (concat ups-executable " auxgww"))
    "*Sequence of arguments passed to ups-program."))
 ;;;
 ((eq system-type 'linux)
  (defvar ups-executable "ps"
    "*Executable path of 'ps' process summary program to run.")
  (defvar ups-program-list
    (list
     (concat ups-executable " -uxww")
     (concat ups-executable " -axvm")
     (concat ups-executable " -auxww"))
    "*Sequence of arguments passed to ups-program."))
 ;; Sys V / Solaris
 (t
  (defvar ups-executable "/bin/ps"
    "*Executable path of 'ps' process summary program to run.")
  (defvar ups-program-list
    (list
     (concat ups-executable " -f -u" (user-login-name) "|sort -n")
     (concat ups-executable " -df|sort -n")
     (concat ups-executable " -ef|sort -n")))))

(defvar ups-program (car ups-program-list)
  "*Program to call to generate process information.
Initially defaults to the first element of ups-program-list.")

(defvar ups-hook nil
  "*Hook to run in the *ps* buffer after starting ups.")


;;;
;;; The following variables are less likely to be of interest to the user.
;;;

(defvar ups-buffer "*Process Status*"
  "*Buffer name of ups information.")

(defvar ups-uptime-program "uptime"
  "*Program to call to generate uptime information.")

(defvar ups-kill-program "kill"
  "*Program to call to kill a process.")

(defvar ups-nice-program "renice"
  "*Program to call to nice a process.")

(defvar ups-bogus-lines 3
  "Number of non-process lines at the top of the display")

(defvar ups-pid-array nil
  "Array of process id's. Array index corresponds to line number in
   current ups-buffer")

(defvar ups-signal-array nil
  "Array of signals to be sent to individual processes. Each signal is
either a signal number or a signal name")

(defvar ups-pid-position nil
  "Position of the PID field in the ups buffer")

(defvar ups-lines nil
  "Number of lines in the current ups buffer")

(defvar ups-map nil
  "Keymap used in ups mode")

;;
;; Definitions for Lemacs/XEmacs only
;;
(if (or (string-match "XEmacs" emacs-version)
	(string-match "Lucid" emacs-version))
    (progn
      ;;   (if (featurep 'frame-icon)
      ;;       (fi-add-mode '(ups . "axe.xbm")))

      ;; (require 'mode-motion)
      ;; (add-hook 'ups-hook 'ups-mode-motion)
      ;; (defun ups-mode-motion ()
      ;;   "Installs mode motion handler for ups mode."
      ;;   (make-local-variable 'mode-motion-hook)
      ;; (setq mode-motion-hook 'mode-motion-highlight-line))

      (require 'font-lock)
      (add-hook 'ups-hook 'turn-on-font-lock)
      (make-face 'ups-deemphasize-face)
      (or (face-differs-from-default-p 'ups-deemphasize-face)
	  (set-face-foreground 'ups-deemphasize-face "Seagreen"))

      (defvar ups-font-lock-keywords
	(list
	 '("^\\S .*" . italic)
	 '("^\\s +root\\s +.*" . ups-deemphasize-face)
	 (cons (concat "^\\s +" (user-login-name) "\\s +.*") 'bold))
	"*Expressions to highlight in ups buffers.")))

(if ups-map
    nil
  (setq ups-map (make-sparse-keymap))
  (define-key ups-map "?" 'ups-build-process-list)
  (define-key ups-map "\C-c\C-c" 'ups-issue-signals)
  (define-key ups-map " " 'next-line)

  (define-key ups-map "B" 'ups-mark-bus)
  (define-key ups-map "b" 'ups-mark-bus)
  
  (define-key ups-map "g" 'ups-build-process-list)
  (define-key ups-map "G" 'ups-build-process-list)

  (define-key ups-map "l" 'ups-next-arg)
  (define-key ups-map "L" 'ups-next-arg)

  (define-key ups-map "N" 'ups-mark-nice)
  (define-key ups-map "n" 'ups-mark-nice)

  (define-key ups-map "p" 'previous-line)
  (define-key ups-map "P" 'previous-line)

  (define-key ups-map "Q" 'ups-mark-quit)
  (define-key ups-map "q" 'ups-quit);; Quit from ups-mode.

  (define-key ups-map "u" 'ups-mark-unmark)
  (define-key ups-map "U" 'ups-mark-unmark)

  (define-key ups-map "V" 'ups-mark-segv)
  (define-key ups-map "v" 'ups-mark-segv)

  (define-key ups-map "x" 'ups-issue-signals)
  (define-key ups-map "X" 'ups-issue-signals)
  )


;; Issue a ups-command to the current buffer, then build the array
;; of process id's and signal numbers.
;;
(defun ups-build-process-list ()
  "Display/update list of active processes."
  (interactive)
  (let ((buffer-read-only nil))
    ;;
    ;; call ups-program
    ;;
    (delete-region (point-min) (point-max))
    (call-process ups-uptime-program nil t nil)
    (insert "\n")
    (shell-command ups-program t)
    ;;
    ;; signals have four spaces for their symbols
    ;;
    (goto-char (point-min))
    (while (not (eobp))
      (beginning-of-line)
      (insert "     ")
      (forward-line 1))
    ;;
    ;; find the word position of the string ``PID'' in the header, since
    ;; this varies with different versions of ps.
    ;;
    (save-excursion
      (let ( eol (i 1) )
	(goto-char (point-min))
	(search-forward "PID")		; get to the ps output
	(end-of-line)
	(setq eol (point))
	(beginning-of-line)
	(setq ups-pid-position nil)
	(while (and (not ups-pid-position) (< (point) eol))
	  (if (looking-at " *PID")
	      (setq ups-pid-position i)
	    (setq i (+ i 1))
	    (forward-word 1))))
      (if ups-show-command-paths
	  nil
	;; Remove leading directories from command names, for brevity.
	(while (re-search-forward " /\\S +/\\| \\\\\\S +\\\\" nil t)
	  (replace-match " ")
	  ;; Only want to replace first occurrence per line.  Don't want to
	  ;; remove directories from arguments sent to the command.
	  (forward-line 1))))
    (setq ups-lines (count-lines (point-min) (point-max)))
    (setq ups-pid-array (make-vector (+ ups-lines 1) nil))
    (setq ups-signal-array (make-vector (+ ups-lines 1) nil))
    (goto-char (point-min))
    (let ((i ups-bogus-lines)
	  (to-skip (- ups-pid-position 1))
	  pid-start pid-end this-pid)
      (forward-line i)
      (while (not (eobp))
	(beginning-of-line)
	(forward-word to-skip)		; skip to beginning of PID field
	(setq pid-start (point))
	(forward-word 1)
	(setq pid-end (point))
	(setq this-pid
	      (string-to-int (buffer-substring pid-start pid-end)))
	(aset ups-pid-array i
	      (if (> this-pid 0) this-pid nil))
	(setq i (+ i 1))
	(forward-line 1))))
  (goto-char (point-min))
  (forward-line ups-bogus-lines))

(fset 'ups 'ups-mode)
(defun ups-mode ()
  "A major-mode for sending signals to processes.
In ups, you indicate signals to send to UNIX processes.
Signals are marked on the left hand side of the display using
an abbreviated name.

The following signals may be sent.  To mark a process, move to the line
corresponding to that process and hit one of the captialized letters
in the list below (the lower case letters work as well).
If the process already has a mark, reissuing it clears the mark

Hup	-- SIGHUP
Int	-- SIGINT
Quit	-- SIGQUIT
Kill	-- SIGKILL
Bus	-- SIGBUS
segV	-- SIGSEGV
Alrm	-- SIGALRM
Term	-- SIGTERM
Stop	-- SIGSTOP
Cont	-- SIGCONT
Nice	-- Nice that sucker

u	-- Clear a previously marked signal
g,?	-- Update the process list
l	-- Use the next member of ups-program-list
x	-- Issue these signals

button2  marks the process under the cursor. See variable ups-mouse-mark."

  (interactive)
  (switch-to-buffer (get-buffer-create ups-buffer))
  (use-local-map ups-map)
  (setq major-mode 'ups-mode
	mode-name "Ups"
	mode-popup-menu ups::menu
	truncate-lines t
	buffer-read-only t)
  (set-syntax-table text-mode-syntax-table)
  (ups-build-process-list)
  (run-hooks 'ups-hook))

(defun ups-quit ()
  (interactive)
  (kill-buffer (current-buffer)))

(defun ups-next-arg ()
  "Use the next member of ups-program-list.
Cycles ups-program through the ups-program-list."
  (interactive)
  (pop-to-buffer ups-buffer)
  (setq ups-program
	(cond
	 ((car (cdr (member ups-program ups-program-list))))
	 (t (car ups-program-list))))
  (ups-build-process-list))


;;
;;	mark line and go forward signal line. Don't allow them to
;;	mark the first line (line 0)
;;
(defun ups-mark-line (signal-symbol this-signal)
  (let ((this-line (count-lines (point-min) (point))))
    (if (not (bolp))
	(setq this-line (- this-line 1)))
    (save-excursion
      (if (> this-line 0)
	  (if (aref ups-pid-array this-line)
	      (let
		  ((buffer-read-only nil))
		(if (equal this-signal (aref ups-signal-array this-line))
		    (progn
		      (setq signal-symbol "     ")
		      (setq this-signal nil )))
		(beginning-of-line)
		(delete-char (length signal-symbol))
		(insert signal-symbol)
		(aset ups-signal-array this-line this-signal)))))
    (next-line 1)))

(defun ups-mark-unmark ()
  (interactive)
  (ups-mark-line "     " nil))

(defun ups-mark-quit ()
  (interactive)
  (ups-mark-line "Quit" "QUIT"))

(defun ups-mark-bus ()
  (interactive)
  (ups-mark-line "Bus" "BUS"))

(defun ups-mark-segv ()
  (interactive)
  (ups-mark-line "segV" "SEGV"))

(defun ups-mark-nice ()
  (interactive)
  (ups-mark-line "Nice" "NICE"))

(defun ups-signal-process (this-pid this-signal)
  (if (string-equal this-signal "NICE")
      (call-process ups-nice-program nil nil nil
		    ups-nice-args (int-to-string this-pid))
    (call-process ups-kill-program nil nil nil
		  (concat "-"
			  (if (numberp this-signal)
			      (int-to-string this-signal)
			    this-signal))
		  (int-to-string this-pid))))

(defun ups-issue-signals ()
  (interactive)
  (let ((i 0))
    (beginning-of-buffer)
    (while (< i ups-lines)
      (if (aref ups-signal-array i)
	  (ups-signal-process (aref ups-pid-array i)
			      (aref ups-signal-array i)))
      (setq i (+ i 1))))
  (ups-build-process-list))


(mapcar 'ups-init-signal ups::signals)
(if (featurep 'toolbar)
    (nconc ups::toolbar
	   (list [:style 2d]
		 [ups::toolbar-nice-icon
		  ups-mark-nice t "Renice a process"]
		 [:style 2d]
		 [ups::toolbar-update-icon
		  ups-build-process-list t "Update process list"]
		 [ups::toolbar-issue-icon
		  ups-issue-signals t "Issues signals"]
		 [:style 2d]
		 )))

(provide 'ups)

