;;; -*- Mode:Emacs-Lisp -*-

;;; This file is part of the Insidious Big Brother Database (aka BBDB),
;;; copyright (c) 1991 Todd Kaufmann <toad@cs.cmu.edu>
;;; Interface to mh-e version 3.7 or later (modeled after bbdb-rmail).
;;; Created  5-Mar-91;  Last modified:  22-apr-92. by jwz.

;;; The Insidious Big Brother Database is free software; you can redistribute
;;; it and/or modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 1, or (at your
;;; option) any later version.
;;;
;;; BBDB is distributed in the hope that it will be useful, but WITHOUT ANY
;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
;;; FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
;;; details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to
;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(require 'bbdb)
(require 'mh-e)    ; Note- we later redefine a function in this file.

(defmacro bbdb/mh-cache-key (message)
  "Return a (numeric) key for MESSAGE"
  ;; assumes message is a buffer-file-name like /usr/celine/Mail/inbox/2323,
  ;;  and gets the 2323 from it.
  (list 'string-to-int (list 'file-name-nondirectory message)))


;;;% Currently assumes msg buffer is the current buffer,
;;;% as usually (always?) is when called from the hook.

(defun bbdb/mh-update-record (&optional offer-to-create)
  "Returns the record corresponding to the current MH message, creating or
modifying it as necessary.  A record will be created if 
bbdb/mail-auto-create-p is non-nil, or if OFFER-TO-CREATE is true and
the user confirms the creation."
  (if bbdb-use-pop-up
      (bbdb/mh-pop-up-bbdb-buffer offer-to-create)
    (if (and (boundp 'mh-show-buffer)
	     (bufferp mh-show-buffer)
	     (buffer-name mh-show-buffer)) ; not killed; gnus messes this up
        (set-buffer mh-show-buffer))
    (let ((msg (bbdb/mh-cache-key buffer-file-name)))
      (if (eq msg 0) (setq msg nil))  ; 0 could mean trouble; be safe.
      (or (bbdb-message-cache-lookup msg nil)  ; nil = current-buffer
	(let ((from (bbdb/mh-get-field "^From[ \t]*:"))
	      name net)
	  (if (or (null from)
		  (string-match (bbdb-user-mail-names)
				(mail-strip-quoted-names from)))
	      ;; if logged-in user sent this, use recipients.
	      (setq from (or (bbdb/mh-get-field "^To[ \t]*:") from)))
	  (if from
	      (bbdb-encache-message msg
	        (bbdb-annotate-message-sender from t
		  (or (bbdb-invoke-hook-for-value bbdb/mail-auto-create-p)
		      offer-to-create)
		  offer-to-create))))))))


(defun bbdb/mh-annotate-sender (string)
  "Add a line to the end of the Notes field of the BBDB record 
corresponding to the sender of this message."
  (interactive (list (if bbdb-readonly-p
			 (error "The Insidious Big Brother Database is read-only.")
			 (read-string "Comments: "))))
  (if (and (boundp 'mh-show-buffer) mh-show-buffer)
      (set-buffer mh-show-buffer))
  (bbdb-annotate-notes (bbdb/mh-update-record t) string))


(defun bbdb/mh-edit-notes ()
  "Edit the notes field of the BBDB record corresponding to the sender of 
this message."
  (interactive)
  (let ((record (or (bbdb/mh-update-record t) (error ""))))
    (bbdb-display-records (list record))
    (bbdb-record-edit-notes record t)))

(defun bbdb/mh-show-sender ()
  "Display the contents of the BBDB for the sender of this message.
This buffer will be in bbdb-mode, with associated keybindings."
  (interactive)
  (if (and (boundp 'mh-show-buffer) mh-show-buffer)
      (set-buffer mh-show-buffer))
  (let ((record (bbdb/mh-update-record t)))
    (if record
	(bbdb-display-records (list record))
	(error "unperson"))))


(defun bbdb/mh-pop-up-bbdb-buffer (&optional offer-to-create)
  "Make the *BBDB* buffer be displayed along with the MH window,
displaying the record corresponding to the sender of the current message."
  (bbdb-pop-up-bbdb-buffer
    (function (lambda (w)
      (let ((b (current-buffer)))
	(set-buffer (window-buffer w))
	;; I don't understand what this is supposed to do - tk
;       (prog1 (eq major-mode 'rmail-mode)       ; no such mode for show buffers... (match "^show" ..) ?
;       (set-buffer b))
	))))
  (let ((bbdb-gag-messages t)
	(bbdb-use-pop-up nil)
	(bbdb-electric-p nil)
	(saved-point (point)))
    (let ((record (bbdb/mh-update-record offer-to-create))
	  (bbdb-elided-display (bbdb-pop-up-elided-display))
	  (b (current-buffer)))
      (bbdb-display-records (if record (list record) nil))
      (set-buffer b)
      (goto-char saved-point)
      record)
  ))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; mh-e modifictions --
;;   this now has a hook that gets called after we suck in the message.

;; this is also called when you reply to a message

(defvar mh-show-message-hook ()
  "Invoked after message is displayed in buffer.")

;;; This has been modified to call mh-show-message-hook after setting up the message
;;;
(defun mh-display-msg (msg-num folder)
  ;; Display message NUMBER of FOLDER.
  (set-buffer folder)
  ;; Bind variables in folder buffer in case they are local
  (let ((formfile mhl-formfile)
	(clean-message-header mh-clean-message-header)
	(invisible-headers mh-invisible-headers)
	(visible-headers mh-visible-headers)
	(msg-filename (mh-msg-filename msg-num))
	(show-buffer mh-show-buffer)
	(folder mh-current-folder))
    (if (not (file-exists-p msg-filename))
	(error "Message %d does not exist" msg-num))
    (switch-to-buffer show-buffer)
    (if mh-bury-show-buffer (bury-buffer (current-buffer)))
    (mh-when (not (equal msg-filename buffer-file-name))
      ;; Buffer does not yet contain message.
      (clear-visited-file-modtime)
      (unlock-buffer)
      (setq buffer-file-name nil)	; no locking during setup
      (erase-buffer)
      (if formfile
	  (if (stringp formfile)
	      (mh-exec-lib-cmd-output "mhl" "-nobell" "-noclear"
				      "-form" formfile msg-filename)
	      (mh-exec-lib-cmd-output "mhl" "-nobell" "-noclear"
				      msg-filename))
	  (insert-file-contents msg-filename))
      (goto-char (point-min))
      (cond (clean-message-header
	     (mh-clean-msg-header (point-min)
				  invisible-headers
				  visible-headers)
	     (goto-char (point-min)))
	    (t
	     (let ((case-fold-search t))
	       (re-search-forward
		"^To:\\|^From:\\|^Subject:\\|^Date:" nil t)
	       (beginning-of-line)
	       (mh-recenter 0))))
      (set-buffer-modified-p nil)
      (setq buffer-file-name msg-filename)
      (set-mark nil)
      (setq mode-line-buffer-identification
	    (list (format mh-show-buffer-mode-line-buffer-id
			  folder msg-num)))))
  (run-hooks 'mh-show-message-hook)
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; this is a more strict version of mh-get-field which takes an regexp

(defun bbdb/mh-get-field (field)
  ;; Find and return the value of field FIELD (regexp) in the current buffer.
  ;; Returns the empty string if the field is not in the message.
  (let ((case-fold-search nil))
    (goto-char (point-min))
    (cond ((not (re-search-forward field nil t)) "")
	  ((looking-at "[\t ]*$") "")
	  (t (re-search-forward "[\t ]*\\([^\t \n].*\\)$" nil t)
	   (let ((field (buffer-substring (match-beginning 1) (match-end 1)))
		 (end-of-match (point)))
	     (forward-line)
	     (while (looking-at "[ \t]") (forward-line 1))
	     (backward-char 1)
	     (if (<= (point) end-of-match)
		 field
		 (format "%s%s" field
			 (buffer-substring end-of-match (point)))))))))

(defun bbdb/mh-execute-commands ()	
  "Process outstanding delete and refile requests."
  (interactive)	
  (save-excursion
    (set-buffer mh-show-buffer)
    (setq bbdb-message-cache nil))
  (bbdb-orig-mh-execute-commands))

(defun mh-send (to cc subject)
  "Compose and send a letter."
  (interactive (list
		(bbdb-read-addresses-with-completion "To: ")
		(bbdb-read-addresses-with-completion "Cc: ")
		(read-string "Subject: ")))
  (let ((config (current-window-configuration)))
    (delete-other-windows)
    (mh-send-sub to cc subject config)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; mail from bbdb-mode using mh

;; these redefine the bbdb-send-mail functions to use mh-send.

;;; Install bbdb into mh-e's show-message function

(defun bbdb-insinuate-mh ()
  "Call this function to hook BBDB into MH-E."
  (define-key mh-folder-mode-map ":" 'bbdb/mh-show-sender)
  (define-key mh-folder-mode-map ";" 'bbdb/mh-edit-notes)
  (bbdb-add-hook 'mh-show-message-hook 'bbdb/mh-update-record)

  ;; We must patch into the "expunge" command to clear the cache, since
  ;; expunging a message invalidates the cache (which is based on msg numbers).
  (or (fboundp 'bbdb-orig-mh-execute-commands)
      (fset 'bbdb-orig-mh-execute-commands 
	    (symbol-function 'mh-execute-commands)))
  (fset 'mh-execute-commands 'bbdb/mh-execute-commands)
  )

(provide 'bbdb-mhe)
