;; Copyright (c) 1992 Jeffrey R. Lewis
;; All rights reserved.
;;
;; Redistribution and use in source and compiled forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 1. Redistributions of source code must retain the above copyright notice,
;;    this list of conditions and the following disclaimer.
;; 2. Redistributions in compiled form must either be accompanied by the
;;    source, or reproduce the above copyright notice, this list of conditions
;;    and the following disclaimer in the documentation and/or other materials
;;    provided with the distribution.
;;
;; THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
;; INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
;; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.

;; Evi 0.98.1 - Emulate Vi
;; LCD Archive Entry:
;; evi|Jeffrey R. Lewis|jlewis@cse.ogi.edu
;; |Emulate Vi - an even better vi emulator
;; |05-01-92|0.98.1|?

;; Here follows Evi 0.9, an even better vi emulator aimed at those who either
;; are well accustomed to vi, or who just simply happen to like its style of
;; editing better than emacs' default.  Evi's first goal is vi compatibility.
;; Its second goal is to be an extension of vi, taking advantage of features
;; supplied by the emacs environment, without simply becoming emacs with vi'ish
;; key bindings.

;; Conventions: in the following, M-x or <M-x> means the escape key followed by
;; the character x (where x is any character), C-x or <C-x> means control-x,
;; <RET> means the return key, <ESC> means the escape key, and *PLEASE NOTE*
;; pairs of single quotes (`') are used for delimiting key sequences (i.e.
;; in examples when you are told to type a key sequence - don't actually type
;; the delimiting quotes).

;; Ideally, you shouldn't need any special manual to start using Evi, other
;; than your vi manual.  However, you should at least glance over the following
;; documentation on extensions and differences.  Useful subjects to search for
;; are:
;;     Starting up Evi
;;     File and buffer management
;;     File and buffer completion
;;     Window management
;;     Accessing emacs commands
;;     Taking advantage of emacs editing modes
;;     Customizing Evi
;;     Arrow keys
;;     Enhancements
;;     New operators
;;     Differences
;;     Supported ex commands and variable settings
;;     Note to vip users
;;     What to do if things go wrong
;;     Credits
;;
;; If you are reading this with Evi, you might try out the following quick-n-
;; dirty menu macro.  Goto the `:' on the next line, and type: `m.+yr@^'
;; :map M "'/0Wm.+"^yr@^| map T 1Gn
;; Then place the cursor on the subject you are interested in, and type `M'.
;; Use `T' to jump back to the menu, and `{' and `}' to move back and forth
;; between subjects (`{', and `}' are just standard vi commands).
;; Unfortunately, I wasn't able to write a proper vi macro to define `M'
;; (I ran into a bug in vi), so I used some Evi extentions to make a better
;; macro.  I'd be curious if anyone can write a proper vi macro for `M'.

;; Starting up Evi
;;
;; To just test Evi out, do:
;;	<M-x>load-file<RET>evi.el<RET>
;;	<M-x>evi<RET>
;; You will now be in the emulator.
;;
;; If you have any trouble at this point due to something in your .exrc or
;; EXINIT, and wish to suppress sourcing of .exrc and EXINIT, place the
;; following in your .emacs:
;;	(setq evi-supress-ex-startup t)
;; Then, send me a note with the contents of your .exrc and EXINIT so we can
;; fix the problem.
;;
;; If you decide to continue using Evi, I would recommend first you byte
;; compile it to speed things up, using the following command (while not in
;; Evi):
;;	<M-x>byte-compile-file<RET>evi.el<RET>
;; Next, if you want to use Evi all the time, put the following at the end
;; of your .emacs file:
;;	(load "~your-login/wherever-evi-is/evi")
;;	(setq term-setup-hook 'evi)
;; this will make emacs go into Evi every time you fire it up.  Of course, you
;; may wish to have Evi properly installed for all to use - consult your local
;; emacs guru.

;; File and buffer management
;;
;; Vi's file management commands have long tried to mimic having multiple
;; buffers, and as such, work well enough to use in an emacs setting.  They
;; of course have to take on slightly different meanings, since it makes
;; little sense to emulate the limitations of the vi/ex model that presumably
;; you are trying to avoid by using evi!
;;
;;	:e	Edit a file in the current window.  With no argument, brings
;;		in a new copy of the file, if it has been subsequently
;;		modified on disk.  `:e!' will override any complaints about
;;		the current buffer being modified, and discard all
;;		modifications.  With a filename argument, edits that file
;;		in the current window (using the copy already in the editor
;;		if it was previously read in).  I'm not sure if `:e! filename'
;;		should have a separate meaning from `:e filename' as in evi we
;;		don't need to worry about the disposition of the current
;;		file before editing the next one.  Opinions are welcome -
;;		currently there's no difference.  `:e#' is shorthand for
;;		edit the most recently accessed buffer not in a window.
;;	:E	Same as `:e', but edits the file in the other window, creating
;;		that window if necessary.  With no filename, splits the
;;		current buffer into two windows.
;;	:n	Switch to the next file in buffer list that's not currently
;;		displayed.  Rotates the current file to the end of the buffer
;;		list, so this will effectively cycle thru all buffers.
;;	:N	Same as `:n', but switches the other window, or creates an
;;		other window and puts the next file in it.
;;	:b	This is a (necessary) extension.  It switches the current
;;		window to the specified buffer, e.g. `:b foo' would switch
;;		to the buffer named `foo' (if it exists).  Default is to
;;		switch to the next buffer not displayed.  `:b!' will create
;;		the buffer if it doesn't exist.
;;	:B	Analogous to :E and :N.
;;	:k	Another (necessary) extension.  Kills the named buffer.  With
;;		no argument, kills current buffer.  `:k!' overrides complaint
;;		about buffer being modified.
;;	:wk	Writes the current buffer, and kills it.
;;	:W	Another extension.  Writes all modified buffers.  `:W!' writes
;;		all with no questions asked
;;	:Wq	As above, then exits emacs.  See `ZZ' below.

;; File and buffer completion
;;
;; Good news for file completion fans: the ex commands which accept
;; filenames as arguments can be file completed using space or tab.
;; Filename completion is turned on after the space that separates the
;; command from the filename.  Similarly, :b and :k will buffer-name
;; complete.  Try it.  One thing you might find handy is `:b  ' to see
;; a list of buffers.  Use C-c or backspace to escape this.

;; Window management
;;
;; First there's the vi command C-^ which in Evi will circulate thru
;; the visible windows, and is equivalent to `:e#' if there's only one
;; window.  Second, there's the `z' command which does limited window
;; management.  However, it's easily extended.  The following are
;; extentions I've thought of so far.  Feel free to suggest others.
;;
;;	z0=	Delete the current window from the screen
;;	z1=	Make the current window fill the screen.
;;	z2=	Split the current window in two vertically.
;;	z0|	Same as z0=
;;	z1|	Same as z1=
;;	z2|	Split the current window in two horizontally.
;;
;;	z<num>+
;;	z<num>-	These let you adjust the size of the current window by <num>.
;;		Use z<num>. to adjust the window size absolutely.
;;
;;	zf	Go to the next window (forward)
;;	zn	same as zf
;;	zb	Go to the previous window (backward)
;;	zp	same as zb
;;
;;	zH
;;	zM
;;	zL	These are aliases for `z<RET>', `z.', and `z-' and correspond
;;		to the arguments to the mark command (this is from vip).

;; Accessing emacs commands
;;
;; By default, no emacs commands are accessible from Evi.  However, the
;; three main classes of emacs commands can easily be made accessible by
;; adding selected elisp commands suggested below to your .evirc file.
;;
;; The C-x prefix commands integrate most easily with Evi since it would
;; only override kill-line in input modes.  To access C-x prefix commands
;; in all modes, use:
;;	(evi-define-key evi-all-keymaps "\C-x" 'evi-emacs-command)
;; The Meta (or ESC prefix) commands are no problem if you just want to
;; access them from the top-level, where in vi ESC only serves the purpose of
;; ringing the bell, which in vi is a handy way of verifying that you're in
;; command mode, but is not vital considering the emacs mode line.  Use the
;; following to access Meta (or ESC prefix) commands only from the top-level:
;;	(evi-define-key '(top-level) "\e" 'evi-top-level-emacs-command)
;; Accessing Meta (or ESC prefix) commands from other modes is problematic
;; because ESC is the command to exit that mode.  However, if you wish, you
;; can redefine the meta prefix to something else - I recommend C-a.  Thus
;; to use Meta commands in all modes use:
;;	(setq evi-meta-prefix-char ?\C-a)
;;	(evi-define-key evi-all-keymaps "\C-a" (cons (current-global-map) ?\e))
;; The C-c prefix commands are also problematic because C-c is vi's interrupt
;; character.  At the top-level, this is again not much of a problem, and
;; you could use:
;;	(evi-define-key '(top-level) "\C-c" 'evi-top-level-emacs-command)
;; C-c will still function as an interrupt character at all other places
;; (including in the middle of a command, where it's most useful).  In the
;; various input modes you could forego C-c as your interrupt character and
;; access the C-c prefix commands by using the following (presumably in
;; addition to the above):
;;	(evi-define-key '(input replace ex) "\C-c" 'evi-emacs-command)

;; Taking advantage of emacs editing modes
;;
;; A number of emacs editing modes have handy local key bindings other than
;; Meta, C-x and C-c prefix bindings.  For example, in C mode, RET does
;; intelligent indenting, and `}' is bound to a command which automatically
;; exdents.  By default, these aren't accessible of course, but you can have
;; Evi enable these local bindings in insert mode by setting:
;;	(setq evi-insert-mode-local-bindings t)
;; As current policy, however, Evi will not allow local mode bindings to
;; override TAB, BS, or DEL, as well as, for obvious reasons, ESC.  ESC
;; prefix commands, however, can be accessed as described in the previous
;; section.

;; Customizing Evi
;;
;; Like vi, Evi will source your .exrc or ~/.exrc file, and/or your EXINIT
;; environment variable.  If your startup runs into problems, let me know -
;; you shouldn't have to change your vi initialization stuff to make Evi
;; happy.
;;
;; If you wish to use some Evi extensions in your startup, but still need to
;; use vi, place these in .exrc.evi, ~/.exrc.evi or EVIINIT so that vi won't
;; gag on them.
;;
;; Emacs lisp startup code for evi, such at that suggested in the previous
;; sections, can be placed in either .evirc or ~/.evirc.
;;
;; And you can, of course, hack away at the the Evi source code if you want
;; something not easily addressed by the above methods.  If you feel what
;; you've done would be generally useful, please email it to me, or post it.
;;
;; One particular customization, not covered elsewhere, is how Evi handles
;; the current directory.  By default it behaves like vi - you have one
;; global current directory, which you change using `:cd' (also see
;; `:pushd', and friends described below).  Alternately, you may like the
;; emacs behaviour better, which is that each buffer has its own idea of
;; the current directory, and by default that directory is the directory
;; that the file for that buffer resides in.  In this mode, you can also
;; change the current directory using `:cd', but that will only affect the
;; current buffer.  To get this behaviour, place the following in your .evirc:
;;	(setq evi-global-directory nil)
;;
;; Another customization you might like to make is to alter the behaviour
;; of `ZZ'.  By default it is bound to `:Wq!', which quietly writes all
;; modified files and exits.  If, however, you would like to be asked about
;; each modified buffer before it is saved in order to avoid accidently
;; saving a file you didn't want saved, map `ZZ' to `:Wq':
;;	map ZZ :Wq\n

;; Arrow keys
;;
;; If you like your arrow keys, there's one incompatibility that's not easily
;; fixed under version 18 of emacs.  Vi recognizes arrow keys that send <ESC>
;; prefixed sequences by using a very short timeout to determine the difference
;; between an <ESC> typed by a person, and an <ESC> sequence sent by the
;; terminal.  A compromise, if you happen to have such a terminal is to
;; `:set notimeout', which makes <ESC><ESC> behave like a single <ESC> and maps
;; your arrow keys to `h', `j', `k', and `l'.  As a side effect in Evi, the
;; normal emacs <ESC> prefix commands will be in effect (except for <ESC><ESC>
;; of course).  This could be undesirable if you're in the habit of hitting
;; the escape key a lot to verify you're in command mode, and follow that
;; immediately with a command.  This should be fixable in version 19 of emacs.

;; Enhancements
;;
;; If you think of any of these as more incompatibilities than enhancements,
;; please let me know.
;;
;; `_' is a new version of the repeat command `.' that prompts you with
;; the keystrokes to repeat, allowing you to edit them before executing.
;; This is particularly useful for the abovementioned complex operators.
;; If you don't wish to re-execute the command, just hit C-c.
;;
;; `C-^', which is supposed to be an alias for `:e#', instead circulates
;; thru the windows on the screen, switching to the most recently accessed
;; other buffer if there's only one window.
;;
;; `:map' accepts the following escapes in the naming and definition of
;; macros: \e for ESC, \n for newline, \r for RET, and \C-x for control-x.
;; \ otherwise works like C-v.  Thus:
;;	map \  \C-f
;; would make the space character be page forward, and
;;	map g \|
;; would make `g' be goto-column.  Note that `|' is normally a command
;; separator and thus must be escaped.
;;
;; You can define exactly what Evi treats as words for the w, b, e, W, B and
;; E commands.  They are defined by setting either or both of the new options
;; `word' (for w, b and e) or `Word' (for W, B and E) to a regular expression
;; describing what words look like.  For example, here's a definition of words
;; that only considers alphanumeric words:
;;	set word=[a-zA-Z0-9]+
;; Contrast this with the default definition:
;;	[a-zA-Z0-9_]+\\\|[^a-zA-Z0-9_\ \t\n]+\\\|^[\ \t]*\n
;; See the emacs documentation on regular expressions for details.
;;
;; There is a command, call it UU for the time being, to continue undoing a
;; previous undo, undoing more changes.  Thus a long enough sequence of UU's
;; will take you back to the unmodified state.  If you went back too far, a
;; `u' will reverse this process and you can progress forward in changes
;; using UU.  Unfortunately, there is no obvious choice of keybinding for
;; this command, as my first choice, `U' is already taken, and we really
;; can't use the vip approach of continuing an undo via `.' because that
;; conflicts with the meaning `u.' in vi of `undo, then do again', which is
;; quite handy for reapplying a change when you initially did it in the
;; wrong place.  I welcome a good idea on this.  My personal solution is
;; to map `[u' to `U' (undo line) and map `U' to UU (continue undo) because
;; I use continue undo more often.  The following is in my .evirc:
;;	(evi-define-key '(vi) "[u" 'evi-undo-line)
;;	(evi-define-key '(vi) "U" 'evi-undo-more)
;;
;; My interpretation of sentence, paragraph, and section motion differs
;; somewhat from vi's in that they behave more analogously to how word
;; motion behaves - e.g. a forward paragraph takes you to the beginning of
;; the next paragraph - not the blank line after the previous paragraph.
;; However, when doing a delete using one of these motions, unless you are
;; at the beginning of the sentence, paragraph or section, the delete will
;; only happen to the end of the sentence, paragraph or section, not to the
;; beginning of the next.  I find this *much* more useful than the vi
;; behaviour - if you disagree, please let me know.
;;
;; `:pushd', `:popd' and `:dirs' commands exist, similar to those found in
;; csh and bash.  Note these only make sense in conjunction with
;; evi-global-directory = t (which is the default).
;;
;; The unnamed register (where deleted text goes) is preserved across
;; excursions into insert mode, etc.  This means you can delete something,
;; insert something, then `put' the deleted text.  In vi, for no apparent
;; reason, you can't do this, even though insert mode doesn't use the
;; unnamed register.
;;
;; `C-d' and `C-u' preserve the goal column (like `j', `k', `C-e' and `C-y' do)
;;
;; Several commands that didn't take counts in vi take counts in Evi.  `p' and
;; `P' take a prefix count and will put the text that many times, regardless
;; of the size of the text - vi will apparently only do the prefix count for
;; less than line sized text.  `/' takes a prefix count to find the nth
;; occurence of a string.  `D' takes a count (I could never figure out why it
;; didn't, since `C' takes a count).
;;
;; Arbitrary regions can be operated on via `m.' (mark current position) and
;; then using the motion `r' or `R' as a motion operand.  The region operated
;; on is bound by the mark and the cursor position at the time the operator
;; was invoked.  `R' extends the region to whole lines.  Thus, for example,
;; the sequence:
;;	m.3j5wdr
;; would delete the text from where the cursor started to 3 lines down and
;; 5 words over.  `R' is often handy for operating on large arbitrary sections
;; of text, for example say you needed to shift some text that ran on for
;; several pages and you weren't sure just how long it was at the start:
;;	m.<C-f><C-f>jjj>R
;; (this idea, of course, comes straight from vip, and emacs users have
;; been using arbitrary regions for years)
;;
;; `:shell' starts up an emacs shell in the current window (instead of
;; suspending emacs, and starting a subshell).  The shell to run comes from
;; the vi variable `ishell', and defaults to the value of the environment
;; variable `SHELL'.  `:gdb program' starts up gdb in the current window on
;; the specified program.  For both of these, you are automatically placed in
;; insert mode, where you should be able to interact as you would expect to,
;; except that <ESC> will take you into command mode.  While in command mode,
;; hitting return will send the current line as input to the shell/gdb,
;; similar to command-line editing vi-style in bash, but will leave you in
;; command mode.
;;
;; The marks used by the mark command `m' are emacs markers, thus they
;; mark a position in a buffer, not necessarily the current one.  This
;; affects the goto mark commands ``' and `''.  For example, if mark
;; `a' is placed in one buffer, then later in another buffer, the command
;; ``a' is typed, evi will first switch to that buffer, then go to the
;; location in that buffer.
;; `'' and ``' also accept `.' and `,' for pop context, and unpop context
;; respectively.  Thus, `'.' will take you to the previous context
;; (defined as in vi by a region of relative motion, with an `absolute'
;; motion pushing a new context.  quotes surround `absolute' because a
;; search is considered an absolute motion for this purpose), and `'.'
;; will take you to the context before that.  There is a ring of 10 contexts
;; so after 10 `'.' commands you'll end up at the original previous
;; context.  `Unpop context' means move forward thru the ring.  `''' and
;; ```' are defined as exchange current location with the location of the
;; previous context.  The context ring is buffer local, so use of it will
;; always keep you in the same buffer.
;;
;; Two changes involving registers.  First, `"'', and `""', are new commands
;; which allow you to insert literal text directly into a register.  `"''
;; inserts a single character, and `""' inserts a string.  E.g.
;; `""hello<ESC>' inserts the string `hello' into the unnamed register,
;; and `"a"'/' inserts a slash into register a.  The choice of command names
;; are intuitive (they suggest quotes around a literal char/string), albeit
;; unfortunate because `"' is supposed to be the prefix for a register
;; specification, not a command.  Second, the register specification `"^'
;; specifies appending to the unnamed register (the one that gets used when
;; no register is specified).  E.g., '"^""ick<ESC>' appends `ick' to the
;; unnamed register.
;;
;; `C-g' gives the column position in addition to the other info
;;
;; `%' exhibits language sensitivity in that it ignores parentheses embedded
;; in quotes.  What defines quotes is based on what minor mode emacs is in
;; (such as c-mode or lisp-mode), or you can roll your own (see emacs
;; command modify-syntax-entry).
;;
;; `=' is no longer specific to :set lisp.  It indents according to the
;; mode.  See emacs command indent-according-to-mode.

;; New operators
;;
;; The `*' operator can be used to send text to emacs processes.  `*' prompts
;; for the name of a process buffer, and the region specified is sent to that
;; process.  Subsequent invocations of `*' will use the same process buffer as
;; last specified as a default.  E.g., to send the current line of text as
;; a command to the emacs shell (see `:shell'), type `***shell*<RET>', or if
;; the shell is already the default, just `**<RET>'.  Paragraph motion is often
;; perfect for sending function definitions to an interpreter, e.g. place the
;; cursor at the beginning of the function, and type `*}<RET>'.  If the
;; function def has blank lines in the middle, you can use `m.' and `yR'
;; described above.  In specifiying the process buffer, you can use buffer
;; completion using space or tab.
;;
;; I'm experimenting with some new complex operators.  I'm particularly
;; interested in your thoughts on these:
;;
;; `[{' operates over lines in a region.  It takes a motion, and a sequence
;; of operations to perform on each line in the region defined by the motion.
;; The sequence of operations is prompted for on the bottom line.  Double the
;; `{' to operate on whole lines.  The point starts in the first column for
;; each line operated on.
;; E.g.
;;	[{}i> <C-v><ESC><RET>
;;	  ^ motion - forward paragraph
;;	   ^ sequence of operations - terminated with a <RET> or <ESC>
;; would prefix every line in the rest of the current paragraph with `> '.
;; The `<C-v><ESC>' sequence inserts an <ESC> into the string you are entering
;; so that it will terminate input when the loop body is executed, not as you
;; are entering the command.
;; E.g.
;;	10[{{i/* <C-v><ESC>A */<C-v><ESC><RET>
;; would place C-style comments around the next 10 lines.
;;
;; `[(' defines a parameterized macro body.  A parameterized macro is diff-
;; erent from standard macro text in that it is parameterized by prefix count
;; and register specification.  `C-_' is a prefix for several keys useful in
;; defining parameterized macros.  `C-_#' is the prefix count applied to this
;; macro, and `C-_"' is the register specification applied to this macro.  All
;; non-macro modification commands can be specified by prefixing them with
;; `C-_'. This is important because the standard command keymap can be
;; changed, and you may want to specify a command macro such that its meaning
;; will be the same regardless of any customizations.  "All non-macro" is
;; specified because several standard modification commands are actually
;; macros themselves.  For example, `i', `o' and `O' are the only non-macro
;; insert commands: `I', `A' etc are all macros.
;; E.g.
;;	"a8[(jC-_#wC-_"dw<RET>
;; would go down one line, move over 8 words, then delete the next word into
;; register `a'.  This is rather contrived, but it gives you the idea.  Param-
;; eterized macro bodies are obviously not very useful typed out each time,
;; and are intended to be the body of a map macro.
;; E.g.
;;	:map M [(jC-_#wC-_"dw<C-v><ESC><RET>
;;	"a8M
;; would be a much more likely scenario for the use of such a macro.

;; Differences
;;
;; These I haven't gotten around to, would be too much bother to implement,
;; would require emacs C source modifications to do sensibly, or are simply
;; unnecessary in an emacs environment.  I'll let you guess which is
;; which ;-)
;;
;; `C-@' in insert mode
;; `C-q'
;; `C-[' doesn't always do the right thing, and it doesn't do the
;;       timeout trick necessary to allow it to recognize keypad sequences
;; `#'
;; `/' search offsets
;; `Q' - quits evi mode instead of going into an ex command loop (which would
;;	 be a bit silly!).
;;
;; digit registers don't work entirely correctly - there are circumstances
;; in which separate lines of a change/deletion are supposed to go into
;; separate registers
;;
;; `:set lisp' has no effect, however, emacs does largely take care of
;; any lisp'ish behaviour you'd want automatically if the file you're
;; editing is suffixed with `.l' or `.el'.  One particular loss, however,
;; is that `)' and `(' don't work on s-expressions like they would in vi
;; with lisp set.

;; Supported ex commands and variable settings
;;
;; Here is a list of the ex commands which are supported:
;;	cd, chdir, copy, delete, edit, file, global, map, move, next, preserve,
;;	print, put, quit, read, recover, set, shell, source, substitute, tag,
;;	unmap, write, wq, yank, !, <, >, &
;; Here are the variable settings which are supported:
;;	autoindent, errorbells, ignorecase, magic, notimeout, shell,
;;	shiftwidth, showmatch, tabstop, wrapmargin, wrapscan

;; Note to vip users:
;;
;; Undo does not continue via `.'.  This is incompatible with vi - the sequence
;; `u.' in vi means `undo, then do again', whereas in vip it means `undo,
;; then undo some more.'  For the vip functionality use evi-undo-more,
;; described in the section on enhancements.
;;
;; The vip commands for editing a file (`v' and `V') and switching buffers
;; (`s', and `S') are not supported.  Use `:e', `:E', `:b', and `:B' instead.
;; See previous section on file and buffer management, or try these cute macros
;; which are (mostly) functional replacements, including doing file and buffer
;; completion:
;; :map v :edit |map V :Edit |map K :kill |map s :buffer |map S :Buffer 
;; (note the space on the end of the line)
;;
;; `:q' exits emacs.  I believe the default behaviour in vip is to simply kill
;; the current buffer (a concept vi doesn't really have) - either that or quit
;; vi emulation.  Any of these choices is reasonable, however, given `:k' for
;; killing buffers (a new command for a new concept), and `Q' for exiting vi
;; emulation, I chose to have `:q' do exactly what it does in vi.

;; What to do if things go wrong
;;
;; There are still some glitches in evi, of course, and if you encounter one,
;; please email me what went wrong, in as much detail as you can.  My address
;; is jlewis@cse.ogi.edu.  In the meantime, if Evi doesn't seem to be respond-
;; ing, or you're having difficulty getting out of some mode, first try `C-c'
;; (this is vi's interrupt character), then if you're still in a pickle, try
;; `C-g' (this is emacs' interrupt character - there are some situations that
;; you might get into where it works instead of `C-c').

;; Credits
;;
;; Masahiko Sato - for having the audacity to write vip in the first place ;-)
;; Eric Bensen - for being the first real advocate
;; and those helpful folks who gave feedback on the first release
;; and the following who've recently given helpful feedback and/or fixes:
;; James Montebello, Steven Dick, Mike Sangrey, Brian Divine, Bill Reynolds,
;; and Roger E. Benz

;; And now, finally, the actual code:

(defmacro defbuffervar (name default-value documentation)
  (list 'progn (list 'defvar name nil documentation)
	       (list 'make-variable-buffer-local (list 'quote name))
	       (list 'set-default (list 'quote name) default-value)))

(defbuffervar evi-enabled nil
  "If t, currently emulating vi in this buffer.")

(defvar evi-debug nil
  "If t, errors generated by emacs are not handled.")

(defvar evi-supress-ex-startup nil
  "If t, don't source .exrc or EXINIT at startup.")

(defbuffervar evi-mode 'vi
  "Current vi mode, one of vi, insert or replace.")

(defvar evi-meta-prefix-char nil
  "Meta-prefix-char to use while in Evi buffers.")

(defvar evi-global-directory t
  "If t, a global current directory is used (this is the default).")

(defvar evi-directory-stack (list default-directory))

(defbuffervar evi-original-local-map nil
  "Original local map.  \(buffer specific\)")

(defbuffervar evi-original-mode-line-buffer-id nil
  "Original mode line buffer identification.  \(buffer specific\)")

(defvar evi-command-keys nil
  "The keystrokes for the current command.")

(defvar evi-prompted nil
  "If t, the current command was prompted for.")

(defbuffervar evi-replace-max nil
  "Maximum excursion of a replace, after which it switches to insert.")

(defbuffervar evi-overstruck-char nil
  "Value of the character overstruck by the `$' marking a partial line change.")

(defbuffervar evi-context nil
  "Current motion context.  One of to-end, to-next, whole-line, or nil.
The value of this variable is passed to evi-motion-command, and is set by
prefix operators like 'd' or '>' to control the type of region defined by
the following motion command.")

(defvar evi-prefix-count nil
  "Current prefix count.")

(defvar evi-prefix-count-multiplier 1
  "Current prefix count multiplier.")

(defvar evi-register-spec nil
  "Current register to use for deletes, yanks, puts, etc.")

(defvar evi-digit-register 0
  "Current delete-ring register cursor.  Points to the register that
will be register 1.")

(defvar evi-last-macro-register nil
  "Last register used to invoke a macro via \\[evi-register-macro].")

(defvar evi-registers (make-vector 72 nil)
  "Vi registers.  0-8 are the delete ring, 9 is the unnamed text register,
10-35 are the alphabetic text registers, and 36-71 are the mark registers.
Each text register is a cons cell with the car being the text in the register
and the cdr being a flag indicating whether or not the text is whole lines.")

(defvar evi-register-unnamed 9
  "Symbolic name for the unnamed register.  Shouldn't change.")

(defbuffervar evi-region-whole-lines nil
  "If t, the current region specified by a motion as an operand encompasses
whole lines.  This is set by evi-motion-command if either the motion is
a vertical one, it is a horizontal motion that covers more than one line
or the operator command requires it ('>' for example only operates on
whole lines).  The value of this variable is stored in the cdr of any
register that gets stored as a result of the current command.  \(buffer
specific\)")

(defbuffervar evi-current-indentation 0
  "The indentation of the most recently auto-indented line.  Used by
evi-newline to determine when to kill auto-indented whitespace.
\(buffer specific\)")

(defvar evi-internal-command nil
  "If t, next command will be executed in internal mode (certain interface
features turned off)")

(defvar evi-scroll-count nil
  "The last specified number of lines to scroll.")

(defbuffervar evi-goal-column 0
  "The column that vertical cursor motion will try to preserve, if possible.")

(defbuffervar evi-reset-goal-column t
  "If t, a horizontal motion has been performed, thus goal column must be reset.")

(defvar evi-search-pattern nil
  "The last pattern specified for searching.")

(defvar evi-search-forward t
  "If t, the last search command was a forward search.")

(defvar evi-find-character nil
  "The last character specified for finding.")

(defvar evi-find-forward t
  "If t, the last find command was a forward search.")

(defvar evi-find-up-to nil
  "If t, the last find command was a find up to command.")

(defbuffervar evi-context-ring (make-vector 10 nil)
  "The last 10 contexts for this buffer.  A context is a location in the buffer
where only relative motions were performed.  A new context is thus saved each
time a non-relative motion is performed.")

(defbuffervar evi-context-ring-cursor 0
  "The cursor pointing to the last context in the context ring.")

(defvar ex-work-space (get-buffer-create " *ex-work-space*")
  "Evi work space for parsing ex commands.")

(defvar ex-tag nil
  "Last tag specified.")

(defvar evi-top-level-map (make-vector 128 'evi-top-level-command))

(defvar evi-vi-map (make-vector 128 nil)
  "The keymap used in vi mode.")

(defvar evi-internal-map (make-vector 128 nil)
  "A subkeymap of vi-map, used to hard-code standard modification operations
for use in defining command macros.")

(defvar evi-motion-map (make-vector 128 nil)
  "The keymap used for operand motions.")

(defvar evi-map-map (make-sparse-keymap)
  "The keymap used for map macros.")

(defvar evi-input-map (make-vector 128 'self-insert-command)
  "The keymap used in input modes.")

(defvar evi-replace-map (make-vector 128 'evi-self-replace)
  "The keymap used in replace mode.")

(defvar evi-insert-map (make-sparse-keymap)
  "The insert mode specific input map.")

(defvar evi-read-string-map (make-sparse-keymap)
  "The evi-read-string specific input map.")

(defvar evi-ex-map (make-sparse-keymap)
  "The keymap used when reading ex commands from the minibuffer")

(defvar evi-input-map-map (make-sparse-keymap)
  "The keymap used for input map macros.")

(defvar evi-completion-map (make-vector 128 'self-insert-command)
  "The keymap used with file/buffer completion.")

(defvar evi-shell-map (make-sparse-keymap)
  "The local keymap used in command mode in a shell buffer.")

(defbuffervar evi-buffer-local-vi-map (make-sparse-keymap)
  "The keymap for buffer specific additions to the vi command map")

(defbuffervar evi-buffer-local-insert-map (make-sparse-keymap)
  "The keymap for buffer specific additions to the insert map")

; it appears to be correct that this not include buffer-local-vi-map
(defvar evi-default-keymap-list (list evi-map-map evi-vi-map))

(defconst evi-all-keymaps
  '(vi insert replace ex)
  "All Evi keymaps.")

(defbuffervar evi-register-parameter nil
  "Register specification to the current parameterized macro.")

(defbuffervar evi-prefix-count-parameter nil
  "Prefix count to the current parameterized macro.")

(defvar evi-last-command-keys nil
  "Command keys for the last complete vi command.")

(defbuffervar evi-insert-point nil
  "The point at which the current insert command began.")

;; Vi option variables
;; ZZ - could/should make some of these buffer local after reading EXINIT

(defconst evi-option-list
  '((("autoindent" "ai") . (bool . evi-auto-indent))
    (("autoprint" "ap") . (bool . nil))
    (("autowrite" "aw") . (bool . nil))
    (("beautify") . (bool . nil))
    (("directory" "dir") . (string . nil))
    (("edcompatible" "ed") . (bool . nil))
    (("errorbells" "eb") . (bool . evi-error-bell))
    (("flash") . (bool . nil))
    (("hardtabs" "ht") . (number . nil))
    (("ignorecase" "ic") . (bool . evi-ignore-case))
    (("ishell" "ish") . (string . explicit-shell-file-name))
    (("lisp") . (bool . nil))
    (("list") . (bool . nil))
    (("magic") . (bool . evi-search-magic))
    (("mesg") . (bool . nil))
    (("modeline") . (bool . nil))
    (("novice") . (bool . nil))
    (("number" "nu") . (bool . nil))
    (("optimize" "opt") . (bool . nil))
    (("paragraphs" "para") . (string . nil))
    (("prompt") . (bool . nil))
    (("readonly" "ro") . (bool . nil))
    (("redraw") . (bool . nil))
    (("remap") . (bool . nil))
    (("report") . (number . nil))
    (("scroll") . (number . nil))
    (("sections" "sect") . (string . nil))
    (("shell" "sh") . (string . shell-file-name))
    (("shiftwidth" "sw") . (number . evi-shift-width))
    (("showmatch" "sm") . (bool . blink-matching-paren))
    (("showmode") . (bool . evi-show-mode))
    (("slowopen" "slow") . (bool . nil))
    (("sourceany") . (bool . nil))
    (("tabstop" "ts") . (number . default-tab-width))
    (("tags") . (string . nil))
    (("taglength" "tl") . (number . nil))
    (("term") . (string . nil))
    (("terse") . (bool . nil))
    (("timeout") . (bool . evi-timeout))
    (("ttytype" "tty") . (string . nil))
    (("warn") . (bool . nil))
    (("word") . (string . evi-word))
    (("Word") . (string . evi-Word))
    (("wrapmargin" "wm") . (number . evi-wrap-margin))
    (("wrapscan" "ws") . (bool . evi-search-wraparound))
    (("writeany" "wa") . (bool . nil))))

(defconst evi-auto-indent nil
  "*If t, automatically indents text inserted on a new line.")

(defconst evi-error-bell nil
  "*If t, ring bell on error.")

(defconst evi-ignore-case nil
  "*If t, ignore case in searches.")

(defconst evi-search-magic t
  "*If t, search patterns are normal regular expressions.  This is the default.
Otherwise, the `magic' characters `.' `[' and `*' are treated as literals and
must be escaped to get their regular expression interpretation.")

(defconst evi-shift-width 8
  "*The number of colums shifted by > and < command, and ^T and ^D
in insert mode.")

(defconst evi-show-mode t
  "*If t, show current vi mode.")

(defconst evi-timeout t
  "*If t, timeout is actually *not* implemented.  If nil, <ESC><ESC> becomes
<ESC>, and arrows keys are mapped to h, j, k and l.")

(defun evi-timeout (value)
  (if value
    (evi-define-key '(vi) "\e" nil)
    (progn (evi-define-key '(vi) "\e" esc-map)
	   (define-key function-keymap "l" 'evi-backward-char)
	   (define-key function-keymap "r" 'evi-forward-char)
	   (define-key function-keymap "u" 'evi-previous-line)
	   (define-key function-keymap "d" 'evi-next-line)
	   ;; ZZ should save \e\e binding and use that in :set timeout
	   (evi-define-key '(vi) "\e\e" nil))))

(defconst evi-word "[a-zA-Z0-9_]+\\|[^a-zA-Z0-9_ \t\n]+\\|^[ \t]*\n"
  "*Regular expression to describe words for w, b and e commands.")

(defconst evi-Word "[^ \t\n]+\\|^[ \t]*\n"
  "*Regular expression to describe words for W, B and E commands.")

(defconst evi-wrap-margin 0
  "*If non-zero, the amount of right margin past which wraparound occurs.")

(defun evi-wrap-margin (margin)
  (if (= margin 0)
    (setq auto-fill-hook nil)
    (progn (setq-default fill-column (- (window-width) margin))
	   (setq auto-fill-hook 'do-auto-fill))))

(defconst evi-search-wraparound t
  "*If t, search wraps around the end of the file.")

(defconst evi-insert-mode-local-bindings nil
  "*If t, emacs buffer-local key bindings will be enabled in insert mode.")

;; Ex commands
;; these are intended to be ordered roughly in order of frequency of use

(defvar ex-commands
  '((("edit" . 1) . ((0 . ((nil . "!") (t . file))) . ex-edit))
    (("buffer" . 1) . ((0 . ((nil . "!") (t . buffer))) . ex-change-buffer))
    (("read" . 1) . ((1 . ((t . "!") (t . file))) . ex-read))
    (("write" . 1) . ((2 . ((nil . "!") (t . ">>") (t . file))) . ex-write))
    (("kill" . 1) . ((0 . ((nil . "!") (t . buffer))) . ex-kill-buffer))
    (("next" . 1) . ((0 . ((nil . "!"))) . ex-next))
    (("Edit" . 1) . ((0 . ((nil . "!") (t . file))) . ex-edit-other-window))
    (("Buffer" . 1) .
     ((0 . ((nil . "!") (t . buffer))) . ex-change-buffer-other-window))
    (("Write" . 1) . ((0 . ((nil . "!"))) . ex-write-all-buffers))
    (("Next" . 1) . ((0 . ((nil . "!"))) . ex-next-other-window))
    (("set" . 2) . ((0 . ((nil . settings))) . ex-set))
    (("substitute" . 1) .
     ((2 . ((t . regular-expression) (backup . regular-expression)
	    (nil . "g") (nil . "c"))) . ex-substitute))
    (("global" . 1) .
     ((2 . ((t . regular-expression) (t . command))) . ex-global))
    (("map" . 3) .
     ((0 . ((nil . "!") (t . word) (t . rest-of-line))) . ex-map))
    (("gdb" . 2) . ((0 . ((t . file))) . ex-gdb))
    (("wk" . 2) . ((0 . nil) . ex-write-kill))
    (("wq" . 2) . ((0 . ((nil . "!"))) . ex-write-quit))
    (("Wq" . 2) . ((0 . ((nil . "!"))) . ex-write-all-and-quit))
    (("append" . 1) . ((1 . nil) . ex-not-implemented))
    (("args" . 2) . ((0 . nil) . ex-not-implemented))
    (("cd" . 2) . ((0 . ((t . file))) . ex-change-directory))
    (("change" . 1) . ((2 . nil) . ex-not-implemented))
    (("chdir" . 3) . ((0 . ((t . file))) . ex-change-directory))
    (("copy" . 2) . ((2 . ((t . address))) . ex-copy))
    (("delete" . 1) . ((2 . ((t . register))) . ex-delete))
    (("dirs" . 2) . ((0 . nil) . ex-directory-stack))
    (("file" . 1) . ((0 . ((t . file))) . ex-file))
    (("insert" . 1) . ((1 . nil) . ex-not-implemented))
    (("join" . 1) . ((2 . nil) . ex-not-implemented))
    (("list" . 1) . ((2 . nil) . ex-not-implemented))
    (("mark" . 2) . ((1 . nil) . ex-not-implemented))
    (("move" . 1) . ((2 . ((t . address))) . ex-move))
    (("number" . 2) . ((2 . nil) . ex-not-implemented))
    (("popd" . 2) . ((0 . nil) . ex-pop-directory))
    (("preserve" . 3) . ((0 . nil) . ex-preserve))
    (("previous" . 4) . ((0 . nil) . ex-not-implemented))
    (("print" . 1) . ((2 . nil) . ex-print))
    (("pushd" . 4) . ((0 . ((t . file))) . ex-push-directory))
    (("put" . 2) . ((1 . ((t . register))) . ex-put))
    (("quit" . 1) . ((0 . ((nil . "!"))) . ex-quit))
    (("recover" . 3) . ((0 . ((nil . "!") (t . file))) . ex-recover))
    (("rewind" . 3) . ((0 . nil) . ex-not-implemented))
    (("shell" . 2) . ((0 . nil) . ex-shell))
    (("source" . 2) . ((0 . ((t . file))) . ex-source-file))
    (("tag" . 1) . ((0 . ((t . word))) . ex-tag))
    (("undo" . 1) . ((0 . nil) . ex-not-implemented))
    (("unmap" . 3) . ((0 . ((nil . "!") (t . word))) . ex-unmap))
    (("version" . 2) . ((0 . nil) . ex-not-implemented))
    (("xit" . 1) . ((0 . nil) . ex-not-implemented))
    (("yank" . 1) . ((2 . ((t . register))) . ex-yank))
    (("!" . 1) . ((2 . ((t . rest-of-line))) . ex-shell-command))
    (("<" . 1) . ((2 . nil) . ex-shift-left))
    (("=" . 1) . ((2 . nil) . ex-not-implemented))
    ((">" . 1) . ((2 . nil) . ex-shift-right))
    (("&" . 1) . ((2 . nil) . ex-substitute-again))
    (("@" . 1) . ((2 . nil) . ex-not-implemented))
    (("" . 0) . ((2 . nil) . ex-null))))

;; Macros

(defmacro defmotion (&rest args)
  (let* ((direction (car args))
	 (function (car (cdr args)))
	 (params (nth 2 args))
	 (documentation (nth 3 args))
	 (body (nthcdr 4 args))
	 (do-function (intern (concat "do-" (symbol-name function)))))
    ; ZZ some rather narly hard-coding here, but does the trick for now
    (cond ((eq (car params) '&char)
	    (` (progn (defun (, function) () (, documentation)
			(interactive)
			(evi-motion-command (quote (, do-function))
					    (quote (, direction))
					    evi-prefix-count evi-context
					    (evi-read-command-char)))
		      (defun (, do-function) (, (cdr params)) (,@ body)))))
	  ((eq (car params) '&string)
	    (` (progn (defun (, function) () (, documentation)
			(interactive)
			(evi-motion-command
			  (quote (, do-function)) (quote (, direction))
			  evi-prefix-count evi-context
			  (evi-read-string (, (car (cdr params))))))
		      (defun (, do-function) (, (cdr (cdr params)))
			(,@ body)))))
	  (t
	    (` (progn (defun (, function) () (, documentation)
			(interactive)
			(evi-motion-command
			  (quote (, do-function)) (quote (, direction))
			  evi-prefix-count evi-context))
		      (defun (, do-function) (, params) (,@ body))))))))

(defmacro evi-iterate (count &rest body)
  (list 'let (list (list 'count count))
	  (append (list 'while (list '> 'count 0)) body
		  (list (list 'setq 'count (list '1- 'count))))
	  (list '= 'count 0)))

(defmacro evi-break ()
  (list 'setq 'count -1))

(defmacro evi-enumerate-condition (item list condition &rest body)
  (list 'let (list (list 'list list) (list item))
    (append
      (list 'while
	(list 'and 'list
	      (list 'progn (list 'setq item '(car list)) condition)))
      (if body
	(append body '((setq list (cdr list))))
	'((setq list (cdr list)))))
    'list))

(defmacro evi-iterate-list (item list &rest body)
  (list 'let (list (list 'list list) (list item) '(found))
    (append
      (list 'while 'list)
      (append (list (list 'setq item '(car list)))
	      body '((setq list (cdr list)))))))

(defmacro evi-find (item list pred)
  (list 'let (list (list 'list list) (list item) '(found))
    (list 'while
      (list 'and 'list
	    (list 'progn (list 'setq item '(car list) 'found pred)
			 '(not found)))
      '(setq list (cdr list)))
    'found))

(defmacro evi-set-goal-column ()
  (` (if evi-reset-goal-column
       (setq evi-goal-column (current-column)
	     evi-reset-goal-column nil))))

(defmacro evi-reset-goal-column ()
  (` (setq evi-reset-goal-column t)))

(defmacro evi-register-text (register)
  (list 'car register))

(defmacro evi-register-whole-lines-p (register)
  (list 'cdr register))

;; Keymaps

(defun evi-define-key (maps key def)
  (evi-enumerate-condition map maps t
    (eval (list 'define-key
		(intern (concat "evi-" (symbol-name map) "-map")) 'key 'def))))

(defun evi-define-macro (maps key macro)
  (evi-enumerate-condition map maps t
    (eval (list 'define-key
		(intern (concat "evi-" (symbol-name map) "-map")) 'key
		(list 'quote (list 'lambda ()
		   '(interactive) (list 'evi-execute-command-macro macro)))))))

(defun evi-make-local-keymap (keydefs)
  (let ((keymap (make-sparse-keymap)))
    (mapcar '(lambda (keydef)
	       (define-key keymap (eval (car keydef)) (car (cdr keydef))))
	    keydefs)
    keymap))

;					"\C-a"
(evi-define-key '(vi)			"\C-b" 'evi-scroll-page-backward)
(evi-define-key '(vi)			"\C-c" 'keyboard-quit)
(evi-define-key '(vi)			"\C-d" 'evi-scroll-text-forward)
(evi-define-key '(vi)			"\C-e" 'evi-scroll-cursor-forward)
(evi-define-key '(vi)			"\C-f" 'evi-scroll-page-forward)
(evi-define-key '(vi)			"\C-g" 'evi-file-info)
(evi-define-key '(vi motion)		"\C-h" 'evi-backward-char)
;					"\C-i"
(evi-define-key '(vi motion)		"\C-j" 'evi-next-line)
;					"\C-k"
(evi-define-key '(vi)			"\C-l" 'redraw-display)
(evi-define-key '(vi motion)		"\C-m" 'evi-beginning-of-next-line)
(evi-define-key '(vi motion)		"\C-n" 'evi-next-line)
;					"\C-o"
(evi-define-key '(vi motion)		"\C-p" 'evi-previous-line)
;					"\C-q" (not implemented)
(evi-define-key '(vi)			"\C-r" 'redraw-display)
;					"\C-s"
;					"\C-t" (not implemented)
(evi-define-key '(vi)			"\C-u" 'evi-scroll-text-backward)
;					"\C-v"
;					"\C-w"
;					"\C-x"
(evi-define-key '(vi)			"\C-y" 'evi-scroll-cursor-backward)
(evi-define-key '(vi)			"\C-z" 'suspend-emacs)
;					"\C-[" (escape)
;					"\C-\"
(evi-define-key '(vi)			"\C-]" 'evi-tag)
(evi-define-key '(vi)			"\C-^" 'evi-other-file)
(evi-define-key '(vi internal motion insert) "\C-_" 'evi-internal-command)
; ZZ should C-_ be for other input maps as well?

(evi-define-key '(vi internal motion)	" " 'evi-forward-char)
(evi-define-key '(vi internal)		"!" 'evi-shell-filter)
(evi-define-key '(vi)			"\"" 'evi-prefix-register)
;					"#"
(evi-define-key '(vi internal motion)	"$" 'evi-end-of-line)
(evi-define-key '(vi internal motion)	"%" 'evi-paren-match)
(evi-define-key '(vi)			"&" 'evi-substitute-again)
(evi-define-key '(vi internal motion)	"'" 'evi-goto-mark-vertical)
(evi-define-key '(vi internal motion)	"(" 'evi-backward-sentence)
(evi-define-key '(vi internal motion)	")" 'evi-forward-sentence)
(evi-define-key '(vi internal)		"*" 'evi-send-to-process)
(evi-define-key '(vi internal motion)	"+" 'evi-beginning-of-next-line)
(evi-define-key '(vi internal motion)	"," 'evi-find-next-character-reverse)
(evi-define-key '(vi internal motion)	"-" 'evi-beginning-of-previous-line)
(evi-define-key '(vi)			"." 'evi-repeat)
(evi-define-key '(vi internal motion)	"/" 'evi-search-forward)
(evi-define-key '(vi internal motion)	"0" 'evi-beginning-of-line)
(evi-define-key '(vi motion)		"1" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"2" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"3" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"4" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"5" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"6" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"7" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"8" 'evi-prefix-digit)
(evi-define-key '(vi motion)		"9" 'evi-prefix-digit)
(evi-define-key '(vi)			":" 'evi-ex-command)
(evi-define-key '(vi internal motion)	";" 'evi-find-next-character)
(evi-define-key '(vi internal)		"<" 'evi-shift-left)
(evi-define-key '(vi internal)		"=" 'evi-indent)
(evi-define-key '(vi internal)		">" 'evi-shift-right)
(evi-define-key '(vi internal motion)	"?" 'evi-search-backward)
(evi-define-key '(vi internal)		"@" 'evi-register-macro)

(evi-define-macro '(vi)			"A" "\C-_i\C-_$")
(evi-define-key '(vi internal motion)	"B" 'evi-backward-Word)
(evi-define-macro '(vi)			"C" "\C-_\"\C-_c\C-_#$")
(evi-define-macro '(vi)			"D" "\C-_\"\C-_d\C-_#$")
(evi-define-key '(vi internal motion)	"E" 'evi-end-of-Word)
(evi-define-key '(vi internal motion)	"F" 'evi-find-char-backwards)
(evi-define-key '(vi internal motion)	"G" 'evi-goto-line)
(evi-define-key '(vi internal motion)	"H" 'evi-goto-top-of-window)
(evi-define-macro '(vi)			"I" "\C-_^\C-_i")
(evi-define-key '(vi internal)		"J" 'evi-join-lines)
;					"K"
(evi-define-key '(vi internal motion)	"L" 'evi-goto-bottom-of-window)
(evi-define-key '(vi internal motion)	"M" 'evi-goto-middle-of-window)
(evi-define-key '(vi internal motion)	"N" 'evi-search-next-reverse)
(evi-define-key '(vi internal)		"O" 'evi-open-before)
(evi-define-key '(vi)			"P" 'evi-put)
(evi-define-key '(vi)			"Q" 'evi-quit-evi)
(evi-define-key '(vi)			"R" 'evi-replace)
(evi-define-macro '(vi)			"S" "\C-_\"\C-_c\C-_#c")
(evi-define-key '(vi internal motion)	"T" 'evi-find-char-backwards-after)
(evi-define-key '(vi)			"U" 'evi-undo-line)
;					"V"
(evi-define-key '(vi internal motion)	"W" 'evi-forward-Word)
(evi-define-macro '(vi)			"X" "\C-_\"\C-_d\C-_#h")
(evi-define-macro '(vi)			"Y" "\C-_\"\C-_y\C-_#y")
(evi-define-macro '(vi)			"ZZ" ":Wq!\n")

(evi-define-key '(vi internal motion)	"[[" 'evi-backward-section)
(evi-define-key '(vi internal motion)	"[(" 'evi-parameterized-macro)
(evi-define-key '(vi internal)		"[{" 'evi-loop-over-lines-in-region)
(evi-define-key '(vi internal motion)	"]]" 'evi-forward-section)
(evi-define-key '(vi internal motion)	"^" 'evi-goto-indentation)
(evi-define-key '(vi)			"_" 'evi-prompt-repeat)
(evi-define-key '(vi internal motion)	"`" 'evi-goto-mark-horizontal)

(evi-define-macro '(vi)			"a" "\C-_l\C-_#\C-_i")
(evi-define-key '(vi internal motion)	"b" 'evi-backward-word)
(evi-define-key '(vi internal)		"c" 'evi-change)
(evi-define-key '(vi internal)		"d" 'evi-delete)
(evi-define-key '(vi internal motion)	"e" 'evi-end-of-word)
(evi-define-key '(vi internal motion)	"f" 'evi-find-character)
;					"g"
(evi-define-key '(vi internal motion)	"h" 'evi-backward-char)
(evi-define-key '(vi internal)		"i" 'evi-insert)
(evi-define-key '(vi internal motion)	"j" 'evi-next-line)
(evi-define-key '(vi internal motion)	"k" 'evi-previous-line)
(evi-define-key '(vi internal motion)	"l" 'evi-forward-char)
(evi-define-key '(vi)			"m" 'evi-mark)
(evi-define-key '(vi internal motion)	"n" 'evi-search-next)
(evi-define-key '(vi internal)		"o" 'evi-open-after)
(evi-define-key '(vi)			"p" 'evi-put-after)
;					"q"
(evi-define-key '(vi)			"r" 'evi-replace-char)
(evi-define-macro '(vi)			"s" "\C-_\"\C-_c\C-_#l")
(evi-define-key '(vi internal motion)	"t" 'evi-find-character-before)
(evi-define-key '(vi)			"u" 'evi-undo)
;					"v"
(evi-define-key '(vi internal motion)	"w" 'evi-forward-word)
(evi-define-macro '(vi)			"x" "\C-_\"\C-_d\C-_#l")
(evi-define-key '(vi internal)		"y" 'evi-yank)
(evi-define-key '(vi)			"z" 'evi-window-control)

(evi-define-key '(vi internal motion)	"{" 'evi-backward-paragraph)
(evi-define-key '(vi internal motion)	"|" 'evi-goto-column)
(evi-define-key '(vi internal motion)	"}" 'evi-forward-paragraph)
(evi-define-key '(vi)			"~" 'evi-toggle-case)

(evi-define-key '(internal) "\"" 'evi-register-parameter)
(evi-define-key '(internal) "#" 'evi-prefix-count-parameter)
(evi-define-key '(internal) "\n" 'self-insert-command)
(evi-define-key '(internal) "\t" 'evi-maybe-indent)
(evi-define-key '(internal motion) "r" 'evi-region)
(evi-define-key '(internal motion) "R" 'evi-region-whole-lines)

; ZZ should define for replace mode also?
(evi-define-key '(input) "\C-v" 'evi-quoted-insert)

(evi-define-key '(input replace) "\C-c" 'evi-input-mode-quit)
(evi-define-key '(input replace) "\e" 'evi-exit-command-loop)

(evi-define-key '(insert) "\C-d" 'evi-backward-indent)
(evi-define-key '(insert) "\C-h" 'evi-insert-mode-delete-backward-char)
(evi-define-key '(insert) "\C-j" 'evi-newline)
(evi-define-key '(insert) "\C-m" 'evi-newline)
(evi-define-key '(insert) "\C-t" 'evi-forward-indent)
(evi-define-key '(insert) "\C-w" "\C-_d\C-_b")
(evi-define-key '(insert) "\C-x" "\C-_d\C-_0")
(evi-define-key '(insert) "\177" 'evi-insert-mode-delete-backward-char)

;(evi-define-key (replace) "\C-d" 'evi-backward-indent)
(evi-define-key '(replace) "\C-h" 'evi-replace-mode-delete-backward-char)
;(evi-define-key (replace) "\C-t" 'evi-forward-indent)
;(evi-define-key (replace) "\C-w" 'evi-delete-backward-word)
(evi-define-key '(replace) "\177" 'evi-replace-mode-delete-backward-char)

(evi-define-key '(read-string ex) "\C-j" 'evi-exit-command-loop)
(evi-define-key '(read-string ex) "\C-m" 'evi-exit-command-loop)

(evi-define-key '(read-string) "\C-h" 'evi-delete-backward-char-maybe-abort)
(evi-define-key '(read-string) "\177" 'evi-delete-backward-char-maybe-abort)

(evi-define-key '(ex) "\C-h" 'ex-delete-backward-char)
(evi-define-key '(ex) "\177" 'ex-delete-backward-char)
(evi-define-key '(ex) "\C-i" 'ex-complete)
;(evi-define-key (ex) "\C-w" 'ex-delete-backward-word)
(evi-define-key '(ex) " " 'ex-space)

(evi-define-key '(completion) "\C-c" 'abort-recursive-edit)
(evi-define-key '(completion) "\C-h" 'evi-completion-delete-backward-char)
(evi-define-key '(completion) "\C-i" 'minibuffer-complete)
(evi-define-key '(completion) "\C-j" 'exit-minibuffer)
(evi-define-key '(completion) "\C-m" 'exit-minibuffer)
(evi-define-key '(completion) "\C-v" 'quoted-insert)
(evi-define-key '(completion) " " 'minibuffer-complete-word)
(evi-define-key '(completion) "?" 'minibuffer-completion-help)
(evi-define-key '(completion) "\177" 'evi-completion-delete-backward-char)

(evi-define-key '(shell) "\C-m" 'shell-send-input)

;; Command macros

(defun evi-execute-command-macro (macro)
  (let ((evi-register-parameter evi-register-spec)
	(evi-register-spec nil)
	(evi-prefix-count-parameter evi-prefix-count)
	(evi-prefix-count nil))
    (evi-execute-macro macro)
    (evi-fixup-cursor 'vertical)))

(defun evi-parameterized-macro ()
  (interactive)
  (let ((macro (evi-read-string "(")))
    (evi-execute-command-macro macro)))

(defun evi-register-macro (char &optional count)
  (interactive (evi-character-arg))
  (let* ((evi-last-command-keys nil)
	 (register-number (evi-register-number char))
	 (macro (evi-register-text (aref evi-registers register-number))))
    (setq evi-last-macro-register register-number)
    (evi-execute-macro macro)))

;; And now we have to do our own keyboard macros...  emacs `keyboard' macros
;; don't cut it as they don't believe in hierarchical commands - the macro
;; has to terminate at the same lisp execution depth as it started.  This
;; is OK for emacs 'cause emacs commands don't build on each other like vi
;; commands do.  If anyone has any idea of how to make emacs `keyboard' macros
;; behave in a manner independent of their execution context, please let me
;; know.
(defvar evi-unread-command-char nil)
(defvar evi-macro-stack nil)
(defvar evi-current-macro nil)
(defvar evi-current-macro-index nil)

(defun evi-execute-macro (macro)
  (evi-push-macro)
  (setq evi-current-macro macro
	evi-current-macro-index 0)
  (while evi-current-macro
    (evi-get-command)))

(defun evi-read-string (prompt &optional initial keymap-list)
  (let ((result
    (save-window-excursion
      ; this seems unduly complicated...
      (set-buffer (window-buffer (minibuffer-window)))
      (select-window (minibuffer-window))
      (delete-region (point-min) (point-max))
      (insert prompt)
      (setq evi-insert-point (point))
      (if initial
	(insert initial))
      (prog1
	(catch 'quit
	  (if (evi-command-loop
		(or keymap-list
		  (list evi-input-map-map evi-read-string-map evi-input-map)))
	    (buffer-substring (1+ (length prompt)) (point-max))))
	(delete-region (point-min) (point-max))))))
    (cond ((eq result t)
	    (keyboard-quit))
	  (result result)
	  (t (throw 'abort t)))))

(defun evi-read-char ()
  (if evi-unread-command-char
    (prog1 evi-unread-command-char
	   (setq evi-unread-command-char nil))
    (if evi-current-macro
      (prog1 (aref evi-current-macro evi-current-macro-index)
	     (setq evi-current-macro-index (1+ evi-current-macro-index))
	     (if (= evi-current-macro-index (length evi-current-macro))
	       (evi-pop-macro)))
      (read-char))))

(defun evi-push-macro ()
  (setq evi-macro-stack (cons (cons evi-current-macro evi-current-macro-index)
			      evi-macro-stack)))

(defun evi-pop-macro ()
  (setq evi-current-macro (car (car evi-macro-stack))
	evi-current-macro-index (cdr (car evi-macro-stack))
	evi-macro-stack (cdr evi-macro-stack)))

(defun evi-internal-command ()
  (interactive)
  (let ((evi-internal-command t))
    (evi-get-command (list evi-internal-map))))

(defun evi-register-parameter ()
  (interactive)
  (let ((evi-register-spec evi-register-parameter))
    (evi-get-command)))

(defun evi-prefix-count-parameter ()
  (interactive)
  (let ((evi-prefix-count evi-prefix-count-parameter))
    (evi-get-command)))

;; Errors

(defun evi-error (&rest args)
  (throw 'abort (apply 'format args)))

;; Get command

(defun evi-command-loop (keymap-list)
  (let ((evi-default-keymap-list keymap-list)
	(loop-command-keys evi-command-keys))
    (prog1
      (catch 'exit
	(while t
	  (setq evi-command-keys "")
	  (let ((message
		  (catch 'abort
		    (evi-get-command keymap-list))))
	    (if message
	      (if (not (eq message t))
		(message message))
	      (setq loop-command-keys
		    (concat loop-command-keys evi-command-keys))))))
      (setq evi-command-keys (concat loop-command-keys evi-command-keys)))))

(defun evi-top-level-command ()
  (interactive)
  (setq evi-unread-command-char last-command-char)
  (let* ((echo-keystrokes 0)
	 (evi-command-keys "")
	 (evi-prompted nil)
	 (message (if evi-debug
		    (catch 'abort
		      (evi-get-command
			(list evi-map-map evi-buffer-local-vi-map
			      evi-vi-map)))
		    (condition-case code
		      (catch 'abort
			(evi-get-command
			  (list evi-map-map evi-buffer-local-vi-map
				evi-vi-map)))
		      (error
			(if (not (eq evi-mode 'vi))
			  (progn
			    (if (or (eq evi-mode 'replace)
				    (eq evi-mode 'change))
			      (evi-exit-replace-mode))
			    (evi-exit-input-mode)))
			(while evi-current-macro
			  (evi-pop-macro))
			(evi-fixup-cursor 'horizontal)
			(signal (car code) (cdr code)))))))
    (if message
      (progn (if (not (eq message t))
	       (progn (if evi-error-bell (beep))
		      (message message)))
	     (evi-fixup-cursor 'horizontal)))))

(defun evi-top-level-emacs-command ()
  (interactive)
  (setq evi-unread-command-char last-command-char)
  (let* ((evi-command-keys ""))
    (evi-get-command (if evi-original-local-map
		       (list evi-original-local-map (current-global-map))
		       (list (current-global-map))))))

(defun evi-emacs-command ()
  (interactive)
  (evi-unread-command-char last-command-char)
  (evi-get-command (if evi-original-local-map
		       (list evi-original-local-map (current-global-map))
		       (list (current-global-map)))))

(defun evi-exit-command-loop ()
  (interactive)
  (throw 'exit t))

(defun evi-get-command (&optional keymap-list)
  (let* ((current-keymap-list (or keymap-list evi-default-keymap-list))
	 (inhibit-quit t)
	 (char (evi-read-command-char))
	 (keys (char-to-string char))
	 (keydef))
    (evi-enumerate-condition keymap current-keymap-list
      (progn
	(setq keydef (lookup-key keymap keys))
	(while
	  (cond ((keymapp keydef)
		  (setq char (evi-read-command-char)
			keys (concat keys (char-to-string char))
			keydef (lookup-key keydef (char-to-string char)))
		  t)
		((stringp keydef)
		  (if evi-prompted (message ""))
		  (setq last-command-char char
			evi-prompted nil)
		  (let ((evi-last-command-keys nil))
		    (setq quit-flag nil
			  inhibit-quit nil)
		    (evi-execute-macro keydef))
		  nil)
		((commandp keydef)
		  (if evi-prompted (message ""))
		  (setq last-command-char char
			evi-prompted nil
			quit-flag nil
			inhibit-quit nil)
		  (call-interactively keydef)
		  nil)
		(t
		  (setq keydef nil))))
	(not keydef)))
    (or keydef (progn (beep)
		      (evi-error "Unknown command `%s'" keys))))
  nil)

(defun evi-read-command-char ()
  (if evi-current-macro
    ; don't add the contents of a macro to evi-command-keys (test this now
    ; because the current command char may be the last char in the macro)
    (evi-read-char)
    (progn
      (and evi-command-keys (> (length evi-command-keys) 0) (evi-sit-for 1)
	   (progn (message "%s -"
		    (mapconcat 'single-key-description evi-command-keys ""))
		  (setq evi-prompted t)))
      (let ((char (evi-read-char)))
	; probably lousy on garbage collection... 
	(if evi-command-keys
	  (setq evi-command-keys
		(concat evi-command-keys (char-to-string char))))
	char))))

(defun evi-unread-command-char (char)
  (setq evi-unread-command-char char
	evi-command-keys
	  (substring evi-command-keys 0 (1- (length evi-command-keys)))))

(defun evi-sit-for (count)
  (if evi-unread-command-char nil (sit-for count)))

(defun evi-save-command-keys ()
  (setq evi-last-command-keys evi-command-keys))

;; Interactive args

(defun evi-count-arg ()
  (list evi-prefix-count))

(defun evi-register-args ()
  (list (car evi-register-spec) (cdr evi-register-spec) evi-prefix-count))

(defun evi-character-arg ()
  (list (evi-read-command-char) evi-prefix-count))

(defun evi-context-arg ()
  (list evi-context))

;; Mode line

(defun evi-change-mode-id (string)
  "Change the mode identification string to STRING."
  (setq mode-line-buffer-identification
    (list
     (concat string ":" (substring "        %16b" (1+ (length string)))))))

(defun evi-refresh-mode-line ()
  "Redraw mode line."
  (set-buffer-modified-p (buffer-modified-p)))

;; Startup & Shutdown

(defun evi ()
  "Start vi emulation in this buffer."
  (interactive)
  (if (not evi-enabled)
    (progn
      (setq evi-original-local-map (current-local-map)
	    evi-original-mode-line-buffer-id mode-line-buffer-identification)
      (if evi-insert-mode-local-bindings
	(setq evi-buffer-local-insert-map
	      (evi-disable-esc-commands (current-local-map))))
      (if evi-meta-prefix-char
	(setq meta-prefix-char evi-meta-prefix-char))))
  (setq evi-enabled t)
  (use-local-map evi-top-level-map)
  (evi-change-mode-id "Vi")
  (evi-refresh-mode-line))

; ZZ currently somewhat crude, needs to be parameterized on something
(defun evi-disable-esc-commands (keymap)
  (if (arrayp keymap)
    (let ((newmap (make-keymap)))
      (evi-iterate 128
	(let ((val (aref keymap (1- count))))
	  (if (= count 28)
	    (if (and val evi-meta-prefix-char)
	      (evi-redefine-keys val newmap
		(char-to-string evi-meta-prefix-char)))
	    (if (or (/= count 9) (/= count 10) (/= count 128))
	      (aset newmap (1- count) val)))))
      newmap)
    (cons 'keymap (evi-disable-esc-commands-sparse (cdr keymap)))))

(defun evi-redefine-keys (keydef keymap keys)
  (if (keymapp keydef)
    (if (arrayp keydef)
      (evi-iterate 128
	(let ((val (aref keydef (1- count))))
	  (if val
	    (evi-redefine-keys val keymap
	      (concat keys (char-to-string (1- count)))))))
      (evi-iterate-list keypair (cdr keydef)
	(evi-redefine-keys (cdr keypair) keymap
	  (concat keys (char-to-string (car keypair))))))
    (define-key keymap keys keydef)))

(defun evi-disable-esc-commands-sparse (bindings)
  (if (null bindings)
    nil
    (let ((val (car (car bindings)))
	  (rest (evi-disable-esc-commands-sparse (cdr bindings))))
      (if (= val 27)
	(if evi-meta-prefix-char
	  (let ((newmap))
	    (fset 'newmap '(keymap))
	    (evi-redefine-keys (cdr (car bindings)) 'newmap
	      (char-to-string evi-meta-prefix-char))
	    (append (cdr (symbol-function 'newmap)) rest))
	  rest)
	(if (or (= val 8) (= val 9) (= val 127))
	  rest
	  (cons (car bindings) rest))))))

(defun evi-quit-evi ()
  "Quit vi emulation in this buffer."
  (interactive)
  (setq evi-enabled nil)
  (use-local-map evi-original-local-map)
  (setq mode-line-buffer-identification evi-original-mode-line-buffer-id)
  ; ZZ
  (if evi-meta-prefix-char
    (setq meta-prefix-char ?\e))
  (evi-refresh-mode-line))

;; Minibuffer

(defun evi-delete-backward-char-maybe-abort ()
  "Backup and delete previous character, aborting command if at
beginning of input."
  (interactive)
  (if (<= (point) evi-insert-point)
    (throw 'exit nil))
  (delete-backward-char 1))

;; Files

(defun evi-other-file ()
  "Switch to other file."
  (interactive)
  (if (one-window-p)
    (let ((buffer (evi-next-file-buffer t)))
      (if buffer
	(switch-to-buffer buffer)
	(message "No other file to display")))
    (other-window 1)))

;; Scrolling

(defun evi-scroll-page-forward (&optional count)
  "Scroll COUNT pages forward."
  (interactive (evi-count-arg))
  (scroll-up (if (eq (or count 1) 1)
	       (- (window-height) 3)
	       (* (1- (window-height)) (or count 1))))
  (evi-reset-goal-column))

(defun evi-scroll-page-backward (&optional count)
  "Scroll COUNT pages backward."
  (interactive (evi-count-arg))
  (scroll-down (if (eq (or count 1) 1)
		 (- (window-height) 3)
		 (* (1- (window-height)) (or count 1))))
  (evi-reset-goal-column))

(defun evi-scroll-text-forward (&optional count)
  "Scroll COUNT lines forward.  Default is one half of a page or the last COUNT
specified to either \\[evi-scroll-text-forward] or \\[evi-scroll-text-backward] if one was previously
given.  The position of the cursor on the screen is maintained."
  (interactive (evi-count-arg))
  (evi-set-goal-column)
  (let ((line-count (if count
		      (setq evi-scroll-count count)
		      (or evi-scroll-count (/ (1- (window-height)) 2))))
	(window-line (count-lines (window-start) (1+ (point)))))
    (scroll-up line-count)
    (forward-line (min (1- window-line) line-count))
    (evi-move-to-column evi-goal-column)))

(defun evi-scroll-text-backward (&optional count)
  "Scroll COUNT lines backward.  Default is one half of a page or the last COUNT
specified to either \\[evi-scroll-up] or \\[evi-scroll-down] if one was previously
given.  The position of the cursor on the screen is maintained."
  (interactive (evi-count-arg))
  (evi-set-goal-column)
  (let ((line-count (if count
		      (setq evi-scroll-count count)
		      (or evi-scroll-count (/ (1- (window-height)) 2))))
	(window-line (count-lines (window-start) (1+ (point)))))
    (scroll-down line-count)
    (forward-line (- (min (- (1- (window-height)) window-line) line-count)))
    (evi-move-to-column evi-goal-column)))

(defun evi-scroll-cursor-forward (&optional count)
  "Scroll COUNT lines forward.  Maintain cursor position in the file
if possible."
  (interactive (evi-count-arg))
  (evi-set-goal-column)
  (scroll-up (or count 1))
  (evi-move-to-column evi-goal-column))

(defun evi-scroll-cursor-backward (&optional count)
  "Scroll COUNT lines backward.  Maintain cursor position in the file
if possible."
  (interactive (evi-count-arg))
  (evi-set-goal-column)
  (scroll-down (or count 1))
  (evi-move-to-column evi-goal-column))

(defun evi-window-control (char &optional linenumber)
  "Position current line on the screen according to the following character.
With a prefix count, position that line."
  (interactive (evi-character-arg))
  (if linenumber
    (do-evi-goto-line linenumber))
  (cond ((and (>= char ?0) (<= char ?9))
	  (let* ((count (evi-read-number (- char ?0)))
		 (char (evi-read-command-char)))
	    (cond ((= char ?.) (enlarge-window (- count (1- (window-height)))))
		  ((= char ?+) (enlarge-window count))
		  ((= char ?-) (shrink-window count))
		  ((= char ?=) (cond ((= count 0) (delete-window))
				     ((= count 1) (delete-other-windows))
				     ((= count 2) (split-window-vertically))
				     (t (evi-error "Invalid window op"))))
		  ((= char ?|) (cond ((= count 0) (delete-window))
				     ((= count 1) (delete-other-windows))
				     ((= count 2)
					(split-window-horizontally)))))))
	((or (= char ?f) (= char ?n)) (select-window (next-window)))
	((or (= char ?b) (= char ?p)) (select-window (previous-window)))
	(t
	  (let ((position
		  (cond ((or (eq char ?\r) (eq char ?H)) 0)
			((or (eq char ?.) (eq char ?M)) (/ (window-height) 2))
			((or (eq char ?-) (eq char ?L)) (- (window-height) 2))
			(t (evi-error "Invalid window op")))))
	    (recenter position))))
  (if evi-prompted (message "")))

;; unlike the motion commands, the scroll commands have no wrapper function
;; to fixup the cursor, soo...
(defun evi-move-to-column (column)
  (move-to-column column)
  (if (and (eolp) (not (bolp)))
    (backward-char)))

;; Insert mode

(defun evi-insert (&optional count)
  "Enter insert mode."
  (interactive (evi-count-arg))
  (setq evi-insert-point (point))
  (evi-enter-insert))

(defun evi-open-after (&optional count)
  "Open a new line below the current one and enter insert mode."
  (interactive (evi-count-arg))
  (end-of-line)
  (insert ?\n)
  (setq evi-insert-point (point))
  (evi-maybe-indent)
  (evi-enter-insert count))

(defun evi-open-before (&optional count)
  "Open a new line above the current one and enter insert mode."
  (interactive (evi-count-arg))
  (beginning-of-line)
  (insert ?\n)
  (backward-char)
  (setq evi-insert-point (point))
  (evi-maybe-indent t)
  (evi-enter-insert count))

(defun evi-enter-insert (&optional count)
  (evi-insert-mode count)
  (if (not (bolp)) (backward-char))
  (evi-reset-goal-column)
  (evi-save-command-keys))

(defun evi-insert-mode (&optional count)
  (setq evi-mode 'insert)
  (if (eobp) (progn (newline 1) (backward-char 1)))
  (evi-change-mode-id "Insert")
  (evi-refresh-mode-line)
  (if (catch 'quit
	(evi-command-loop (list evi-input-map-map evi-buffer-local-insert-map
				evi-insert-map evi-input-map))
	nil)
    (progn (evi-exit-input-mode)
	   (beep)
	   (evi-error "Quit"))
    (progn (evi-maybe-kill-indentation)
	   (evi-exit-input-mode count))))

(defun evi-exit-input-mode (&optional count)
  "Exit from an input mode."
  (interactive)
  (if count
    (let ((input-string (buffer-substring evi-insert-point (point))))
      (evi-insert-n input-string (1- count))))
  (setq evi-mode 'vi)
  (evi-change-mode-id "Vi")
  (evi-refresh-mode-line))

(defun evi-insert-n (string count)
  (evi-iterate count
    (insert string)))

(defun evi-input-mode-quit ()
  "Abort and exit from an input mode."
  (interactive)
  (throw 'quit t))

(defun evi-insert-mode-delete-backward-char ()
  "Backup and delete previous character, but no further than insert point."
  (interactive)
  (if (> (point) evi-insert-point)
    (delete-backward-char 1)
    (message "Beginning of inserted text")))

(defun evi-maybe-indent (&optional forward)
  (interactive)
  (if evi-auto-indent
    (progn
      (let ((start (point)))
	(skip-chars-forward " \t")
	(delete-region start (point)))
      (if (or (not evi-insert-mode-local-bindings)
	      (eq indent-line-function 'indent-to-left-margin))
	(indent-to (save-excursion
		     (if forward (forward-char) (backward-char))
		     (current-indentation)))
	(indent-according-to-mode))
      (setq evi-current-indentation (current-column)))))

(defun evi-maybe-kill-indentation ()
  (and evi-auto-indent (= evi-current-indentation (current-column))
    (let ((region
	   (save-excursion
	     (let ((start (if (progn (skip-chars-backward " \t") (bolp))
			    (point))))
	       (if (and start (progn (skip-chars-forward " \t") (eolp)))
		 (cons start (point)))))))
      (if region
	(delete-region (car region) (cdr region))))))

(defun evi-newline ()
  "Insert a newline, and indent to the current indentation level.
Kills indentation on current line if the line is otherwise empty."
  (interactive)
  (let ((start (point)))
    (insert ?\n)
    (evi-maybe-indent)
    (save-excursion
      (goto-char start)
      (evi-maybe-kill-indentation))))

(defun evi-forward-indent ()
  "Move forward to the next indentation level, defined by shiftwidth."
  (interactive)
  ; eat all preceeding blanks, then fill with tabs, and pad with spaces
  ; to reach the target column
  (let* ((start-column (current-column))
	 (target-column (+ start-column (- evi-shift-width
					   (% start-column evi-shift-width))))
	 (backup-point (save-excursion
			 (skip-chars-backward " ")
			 (point))))
    (delete-backward-char (- (point) backup-point))
    (while (< (setq start-column (current-column)) target-column)
      (insert ?\t))
    (if (> start-column target-column) (delete-backward-char 1))
    (insert-char ?\ (- target-column (current-column)))))

(defun evi-backward-indent ()
  "Move backward to the previous indentation level, defined by shiftwidth."
  (interactive)
  (let* ((start-column (current-column))
	 (offset (let ((toffset (% start-column evi-shift-width)))
		   (if (= toffset 0) evi-shift-width toffset)))
	 (furthest (save-excursion
		     (skip-chars-backward " \t" (max 0 (- (point) offset)))
		     (- start-column (current-column)))))
    (backward-delete-char-untabify (min offset furthest) nil)))

(defun evi-quoted-insert ()
  (interactive)
  (insert (evi-read-char)))

;; Replace mode

(defun evi-replace ()
  "Enter replace mode."
  (interactive)
  (setq evi-mode 'replace)
  (evi-replace-mode (1- (point-max)))
  (if (not (bolp)) (backward-char))
  (if evi-replace-max
    (set-marker evi-replace-max nil))
  (evi-reset-goal-column)
  (evi-save-command-keys))

;(define-key evi-replace-map "\C-d" 'evi-backward-indent)
;(define-key evi-replace-map "\C-t" 'evi-forward-indent)
;(define-key evi-replace-map "\C-w" 'evi-delete-backward-word)

(defvar evi-replaced-string nil)
(defvar evi-replaced-string-index nil)

(defun evi-replace-mode (max-replace-position)
  (or evi-replace-max
      (setq evi-replace-max (make-marker)))
  (set-marker evi-replace-max max-replace-position)
  (setq evi-insert-point (point)
	evi-replaced-string ""
	evi-replaced-string-index 0)
  (evi-change-mode-id "Replce")
  (evi-refresh-mode-line)
  (if (catch 'quit
	(if (catch 'switch-to-insert
	      (evi-command-loop (list evi-input-map-map evi-replace-map))
	      nil)
	  (progn
	    (set-marker evi-replace-max nil)
	    (evi-insert-mode))
	  (progn
	    (evi-exit-replace-mode)
	    (evi-exit-input-mode)))
	nil)
      (progn (evi-exit-replace-mode)
	     (evi-exit-input-mode)
	     (beep)
	     (evi-error "Quit"))))

(defun evi-exit-replace-mode ()
  (if (< evi-replaced-string-index (length evi-replaced-string))
    (save-excursion
      (delete-region (point)
		     (+ (point)
			(- (length evi-replaced-string)
			   evi-replaced-string-index)))
      (insert (substring evi-replaced-string evi-replaced-string-index))))
  (if (eq evi-mode 'change)
    (evi-exit-change-mode))
  (setq evi-overstruck-char nil))

(defun evi-self-replace ()
  "Replace character under cursor with the command character."
  (interactive)
  (if (or (>= (point) evi-replace-max)
	  (= (following-char) ?\n))
    (progn (setq evi-unread-command-char last-command-char)
	   ; ZZ this is gross... should be rewritten properly, if possible
	   (setq evi-command-keys loop-command-keys)
	   (throw 'switch-to-insert t))
    (progn (if (= evi-replaced-string-index (length evi-replaced-string))
	     (setq evi-replaced-string
	       (concat evi-replaced-string
		       (char-to-string (following-char)))))
	   (setq evi-replaced-string-index (1+ evi-replaced-string-index))
	   (let ((start (point)))
	     (evi-replace-one-char last-command-char)
	     ; if auto-indenting happened...
	     (if (> (- (point) start) 1)
	       (setq evi-insert-point (1+ start)
		     evi-replaced-string
		       (buffer-substring (1+ start) (point))
		     evi-replaced-string-index
		       (length evi-replaced-string)))))))

(defun evi-replace-one-char (char)
  (delete-region (point) (1+ (point)))
  (if (and evi-overstruck-char (= (point) evi-replace-max))
    (progn (aset (car (car buffer-undo-list))
		 0 evi-overstruck-char)
	   (setq evi-overstruck-char nil)))
  ; ZZ unpleasantly hardcoded?
  (if (or (= char ?\n) (= char ?\r))
    (evi-newline)
    (insert char)))

(defun evi-replace-mode-delete-backward-char ()
  "Backup to previous character, undoing last replacement, but no further
than insert point."
  (interactive)
  (if (> (point) evi-insert-point)
    (progn (backward-char)
	   (setq evi-replaced-string-index (1- evi-replaced-string-index)))
    (message "Beginning of replaced text")))

(defun evi-replace-char (char &optional count)
  "Replace the following COUNT characters with CHAR."
  (interactive (evi-character-arg))
  (if (catch 'abort
	(evi-motion-command 'do-evi-forward-char 'horizontal count 'to-end))
    (evi-error "Can't replace that many characters")
    (progn (exchange-point-and-mark)
	   (evi-iterate (or count 1)
	     (evi-replace-one-char char))
	   ; ZZ unpleasantly hard-coded?
	   ; should be handled by a general purpose post-auto-indent func
	   (if (or (= char ?\n) (= char ?\r))
	     (evi-maybe-kill-indentation))
	   (if (not (bolp)) (backward-char))))
  (evi-reset-goal-column)
  (evi-save-command-keys))

(defun evi-toggle-case (&optional count)
  "Toggle the case of the following COUNT characters."
  (interactive (evi-count-arg))
  (evi-motion-command 'do-evi-forward-char 'horizontal count 'to-end)
  (save-excursion
    (evi-iterate (- (point) (mark))
      (backward-char)
      (let ((char (following-char)))
	(cond ((and (>= char ?a) (<= char ?z))
		(upcase-region (point) (1+ (point))))
	      ((and (>= char ?A) (<= char ?Z))
		(downcase-region (point) (1+ (point))))))))
  (evi-fixup-cursor 'horizontal)
  (evi-reset-goal-column)
  (evi-save-command-keys))

;; Modification operators

(defun evi-change (&optional count)
  "Change operator."
  (interactive (evi-count-arg))
  (evi-operator-command (or count 1) 'to-end '(evi-change-internal) 1))

(defun evi-change-internal ()
  ; If the region is contained on one line, throw a `$' out to mark the
  ; end of the region, then enter replace mode and delete any un-replaced
  ; text when that is exited, with the replace-max set at the end of the
  ; region so that it will switch to insert mode if necessary.  Otherwise,
  ; delete the region first, and enter insert mode.
  (evi-copy-region-to-registers t)
  ; this makes the undo leave the point at the start of the undone text
  (exchange-point-and-mark)
  (if (save-excursion (end-of-line) (> (mark) (point)))
    (progn (delete-region (point) (mark))
	   (setq evi-insert-point (point))
	   (evi-insert-mode))
    (progn (setq evi-overstruck-char (char-after (1- (mark))))
	   (save-excursion
	     (exchange-point-and-mark)
	     (delete-region (1- (point)) (point))
	     (insert ?$)
	     ;; this is a bit of song and dance to get the cursor to
	     ;; end up in the right place after an undo.  the problem
	     ;; is these two previous statements, which are the first
	     ;; things changed, and thus where the cursor will be left
	     ;; after an undo.  first step: erase the fact that we put
	     ;; the dollar sign there in the first place.
	     (setq buffer-undo-list (cdr (cdr buffer-undo-list))))
	   (setq evi-mode 'change)
	   (evi-replace-mode (1+ (mark)))))
  (if (not (bolp)) (backward-char)))

(defun evi-exit-change-mode ()
  (if (and (marker-position evi-replace-max)
	   (< (point) evi-replace-max))
    (let ((overstrike-offset (1- (- evi-replace-max (point)))))
      (delete-region (point) (marker-position evi-replace-max))
      (set-marker evi-replace-max nil)
      ;; second step: rewrite the undo record with the
      ;; original overstruck character
      (aset (car (car buffer-undo-list))
	    overstrike-offset evi-overstruck-char))))

(defun evi-delete (&optional count)
  "Delete operator."
  (interactive (evi-count-arg))
  (evi-operator-command (or count 1) 'to-next '(evi-delete-internal)))

(defun evi-delete-internal ()
  (evi-copy-region-to-registers t)
  ; this makes the undo leave the point at the start of the undone text
  (exchange-point-and-mark)
  (if (= (point) (mark))
    (message "Nothing deleted")
    (delete-region (point) (mark)))
  (evi-fixup-cursor (if evi-region-whole-lines 'vertical 'horizontal)))

(defun evi-yank (&optional count)
  "Yank operator."
  (interactive (evi-count-arg))
  (save-excursion
    (evi-operator-command (or count 1) 'to-next '(evi-yank-internal))))

(defun evi-yank-internal ()
  (evi-copy-region-to-registers nil)
  (if (= (mark) (point))
    (message "Nothing to yank")))

(defun evi-put-after (&optional register-number register-append count)
  "Put back yanked or deleted text after cursor."
  (interactive (evi-register-args))
  (let ((register
	  (aref evi-registers (or register-number evi-register-unnamed))))
    (if register
      (if (evi-register-whole-lines-p register)
	(progn (end-of-line)
	       (if (not (eobp)) (forward-char))
	       (save-excursion
		 (evi-insert-n (evi-register-text register) (or count 1))))
	(progn (if (not (and (bolp) (eolp)))
		 (forward-char))
	       (evi-insert-n (evi-register-text register) (or count 1))
	       (backward-char)))
      (if register-number
	(message "Nothing in register %c" (evi-register-name register-number))
	(message "No text to put"))))
  (evi-reset-goal-column)
  (evi-save-command-keys))

(defun evi-put (&optional register-number register-append count)
  "Put back yanked or deleted text."
  (interactive (evi-register-args))
  (let ((register
	  (aref evi-registers (or register-number evi-register-unnamed))))
    (if register
      (if (evi-register-whole-lines-p register)
	(progn (beginning-of-line)
	       (save-excursion
		 (evi-insert-n (evi-register-text register) (or count 1))))
	(progn (evi-insert-n (evi-register-text register) (or count 1))
	       (backward-char)))
      (if register-number
	(message "Nothing in register %c" (evi-register-name register-number))
	(message "No text to put"))))
  (evi-reset-goal-column)
  (evi-save-command-keys))

(defun evi-shift-right (&optional count)
  "Shift right operator."
  (interactive (evi-count-arg))
  (evi-operator-command (or count 1) 'whole-lines '(evi-shift-internal 1)))

(defun evi-shift-left (&optional count)
  "Shift left operator."
  (interactive (evi-count-arg))
  (evi-operator-command (or count 1) 'whole-lines '(evi-shift-internal -1)))

(defun evi-shift-internal (direction)
  (if (= (mark) (point))
    (message "Nothing shifted")
    (indent-rigidly (mark) (point) (* evi-shift-width direction)))
  (goto-char (mark))
  (skip-chars-forward " \t"))

(defun evi-indent (&optional count)
  "Indent region."
  (interactive (evi-count-arg))
  (evi-operator-command (or count 1) 'whole-lines '(evi-indent-internal)))

(defun evi-indent-internal ()
  (if (= (mark) (point))
    (message "Nothing indented")
    (indent-region (mark) (point) nil))
  (goto-char (mark))
  (skip-chars-forward " \t"))

(defun evi-shell-filter (&optional count)
  "Filter region thru shell command."
  (interactive (evi-count-arg))
  (save-excursion
    (evi-operator-command (or count 1) 'whole-lines
			  '(evi-filter-internal input-string) t)))

(defun evi-filter-internal (shell-command)
  (shell-command-on-region (mark) (point) shell-command t))

(defun evi-send-to-process (&optional count)
  "Send region to emacs process buffer."
  (interactive (evi-count-arg))
  (save-excursion
    (evi-operator-command (or count 1) 'to-next
			  '(evi-to-process-internal))))

(defvar evi-process-buffer nil)

(defun evi-to-process-internal ()
  (send-region
    (setq evi-process-buffer (read-buffer "* : " evi-process-buffer t))
    (mark) (point)))

(defun evi-loop-over-lines-in-region (&optional count)
  "Execute a sequence of operations on every line in a region."
  (interactive (evi-count-arg))
  (evi-operator-command (or count 1) 'whole-lines
			'(evi-loop-lines-internal input-string) t))

(defun evi-loop-lines-internal (macro)
  (let ((evi-last-command-keys nil)
	(ending-mark (set-marker (make-marker) (point-marker)))
	(evi-prefix-count nil))
    (goto-char (mark))
    (beginning-of-line)
    (while (< (point) (marker-position ending-mark))
      (evi-execute-macro macro)
      (end-of-line)
      (forward-char))
    (set-marker ending-mark nil))
  (evi-fixup-cursor 'vertical))

(defun evi-operator-command (count context operation &optional more-input)
  (let ((evi-context context)
	(evi-prefix-count-multiplier count)
	(evi-default-keymap-list
	  (list (evi-make-local-keymap
		  '(((char-to-string last-command-char) evi-whole-lines)))
		evi-map-map evi-motion-map)))
    (evi-get-command))
  (let ((input-string (if (eq more-input t)
			(evi-read-string (concat evi-command-keys " : ")))))
    (eval operation))
  (evi-reset-goal-column)
  (evi-save-command-keys))

(defun evi-join-lines (&optional count)
  "Join together COUNT + 1 lines, supplying appropriate whitespace."
  (interactive (evi-count-arg))
  (evi-iterate (or count 1)
    (let ((starting-point (point)))
      (end-of-line)
      (if (or (eobp) (progn (forward-char) (eobp)))
	(progn (goto-char starting-point)
	       (evi-break))
	(progn (delete-region (1- (point))
			      (progn (skip-chars-forward " \t") (point)))
	       (if (and (/= (preceding-char) ? )
			(/= (preceding-char) ?\t)
			(/= (following-char) ?))) ; (
		 (insert-char ?  (if (= (preceding-char) ?.) 2 1)))))))
  (evi-reset-goal-column)
  (evi-save-command-keys))

;; Motion command

(defun evi-expand-region-to-lines (context)
  (exchange-point-and-mark)
  (beginning-of-line)
  (exchange-point-and-mark)
  (end-of-line)
  (if (not (or (eobp) (eq context 'to-end))) (forward-char))
  (setq evi-region-whole-lines t))

; 'normalizing' a horizontal region means expanding the region to whole lines
; when 1) the beginning of the region is on the first non-white character
; of a line, and 2) the ending of the region is on the end of the line

(defun evi-normalize-region ()
  (and (eolp)
       (save-excursion
	 (beginning-of-line)
	 (and (> (point) (mark))
	      (progn (goto-char (mark))
		     (skip-chars-backward " \t")
		     (bolp))))
       (progn (exchange-point-and-mark)
	      (beginning-of-line)
	      (exchange-point-and-mark)
	      (if (not (eobp))
		(forward-char))
	      (setq evi-region-whole-lines t))))

(defun evi-fixup-cursor (direction)
  (or evi-internal-command
    (if (eq direction 'horizontal)
      (progn (if (and (eobp) (not (bobp)))
	       (backward-char))
	     (if (and (eolp) (not (bolp)))
	       (backward-char)))
      (if (and (eobp) (not (bobp)))
	(progn (backward-char) (beginning-of-line))
	(if (and (eolp) (not (bolp))) (backward-char))))))

(defun evi-motion-command (move-function direction count context &optional arg)
  (if context
    (set-mark (point))
    ; else, maintain the goal column.  kinda gross this being here, but...
    (if (or (eq move-function 'do-evi-next-line)
	    (eq move-function 'do-evi-previous-line))
      (evi-set-goal-column)
      (evi-reset-goal-column)))
  (if arg
    (funcall move-function arg count context)
    (funcall move-function count context))
  (if context
    (progn
      (if (< (point) (mark)) (exchange-point-and-mark))
      (if (or (eq direction 'vertical) (eq context 'whole-lines))
	(evi-expand-region-to-lines context)
	(progn (setq evi-region-whole-lines nil)
	       (if (eq context 'to-next)
		 (evi-normalize-region)))))
    ; fixup the location of the cursor, if necessary
    (evi-fixup-cursor direction)))

;; Simple motion commands

(defmotion horizontal evi-forward-char (&optional count context)
  "Move right COUNT characters on the current line."
  (forward-char (save-excursion
		  (set-mark (point))
		  (end-of-line)
		  (min (or count 1) (- (point) (mark)))))
  (if (and (eolp) (not context) (not evi-internal-command))
    (evi-error "End of line")))

(defmotion horizontal evi-backward-char (&optional count context)
  "Move left COUNT characters on the current line."
  (backward-char (save-excursion
		   (set-mark (point))
		   (beginning-of-line)
		   (min (1- (or count 1)) (- (mark) (point)))))
  (if (bolp) (evi-error "Beginning of line") (backward-char)))

(defmotion vertical evi-next-line (&optional count context)
  "Go to ARGth next line."
  (evi-next-line-internal (or count 1))
  (if (null context)
    (progn (evi-adjust-scroll-up)
	   (move-to-column evi-goal-column))))

(defmotion vertical evi-beginning-of-next-line (&optional count context)
  "Go to beginning of ARGth next line."
  (evi-next-line-internal (or count 1))
  (if (null context) (evi-adjust-scroll-up))
  (skip-chars-forward " \t"))

;; ZZ maybe can use goal column in fixup-cursor to remove some of this here??
(defun evi-next-line-internal (count)
  (let* ((starting-point (point))
	 (offset (forward-line count)))
    (if (or (/= offset 0) (eobp))
      (progn (goto-char starting-point)
	     (evi-error
		    (if (= count 1) "Last line in buffer"
				    "Not that many lines left in buffer"))))))

(defun evi-adjust-scroll-up ()
  (let ((window-line (count-lines (window-start) (1+ (point))))
	(window-height (1- (window-height))))
    (and (> window-line window-height)
	 (< window-line (+ window-height (/ window-height 3)))
	 (recenter (1- window-height)))))

(defmotion vertical evi-previous-line (&optional count context)
  "Go to ARGth previous line."
  (evi-previous-line-internal (or count 1))
  (if (null context)
    (progn (evi-adjust-scroll-down)
	   (move-to-column evi-goal-column))))

(defmotion vertical evi-beginning-of-previous-line (&optional count context)
  "Go to beginning of ARGth previous line."
  (evi-previous-line-internal (or count 1))
  (if (null context) (evi-adjust-scroll-down))
  (back-to-indentation))

(defun evi-previous-line-internal (count)
  (let* ((starting-point (point))
	 (offset (forward-line (- count))))
    (if (/= offset 0)
      (progn (goto-char starting-point)
	     (evi-error
		    (if (= count 1) "First line in buffer"
				    "Not that many lines left in buffer"))))))

(defun evi-adjust-scroll-down ()
  (if (< (point) (window-start))
    (let ((window-line (count-lines (1+ (point)) (window-start)))
	  (window-height (1- (window-height))))
      (and (< window-line (/ window-height 3))
	   (recenter 0)))))

(defmotion vertical evi-goto-line (&optional count context)
  "Go to line number LINE, or to end of file if no count specified."
  ; ZZ once again... if we know the move won't be far (like on same screen)
  ; perhaps shouldn't push context...
  (evi-push-context)
  (if count
    (let ((starting-point (point)))
      (goto-char (point-min))
      (if (or (> (forward-line (1- count)) 0) (eobp))
	(progn (goto-char starting-point)
	       (evi-error "Past end of buffer"))))
    (progn (goto-char (point-max))
	   (forward-line -1))))

(defmotion vertical evi-goto-top-of-window (&optional offset context)
  "Go to the top line of the window.  With an arg, OFFSET, goes to the
OFFSET'th line of the window."
  (move-to-window-line (1- (or offset 1)))
  (or context
      (skip-chars-forward " \t")))

(defmotion vertical evi-goto-middle-of-window (&optional offset context)
  "Go to the middle line of the window."
  (move-to-window-line (/ (window-height) 2))
  (or context
      (skip-chars-forward " \t")))

(defmotion vertical evi-goto-bottom-of-window (&optional offset context)
  "Go to the bottom line of the window.  With an arg, OFFSET, goes to the
OFFSET'th line from the bottom of the window."
  (move-to-window-line (- (1- (window-height)) (or offset 1)))
  (or context
      (skip-chars-forward " \t")))

(defmotion horizontal evi-goto-column (&optional column context)
  "Go to column COLUMN, or as close to that column as possible."
  (move-to-column (1- column)))

(defmotion vertical evi-whole-lines (&optional count context)
  "Go ARG - 1 lines forward."
  (evi-next-line-internal (1- (or count 1))))

(defmotion horizontal evi-beginning-of-line (&optional count context)
  "Go to beginning of line."
  (beginning-of-line))

; it's not at all clear why this doesn't take a count...
; maybe it should...
(defmotion horizontal evi-goto-indentation (&optional count context)
  "Go to beginning of indented text on current line."
  (beginning-of-line)
  (back-to-indentation))
 
(defmotion horizontal evi-end-of-line (&optional count context)
  "Go to end of line."
  (evi-next-line-internal (1- (or count 1)))
  (end-of-line))

;; Word, sentence, paragraph and section motion commands

(defun evi-eobp ()
  (< (- (point-max) (point)) 3))

(defmotion horizontal evi-forward-word (&optional count context)
  "Move to the beginning of the COUNTth next word."
  (evi-forward-word-internal evi-word (or count 1) context))

(defmotion horizontal evi-forward-Word (&optional count context)
  "Move to the beginning of the COUNTth next white-space delimited word."
  (evi-forward-word-internal evi-Word (or count 1) context))

(defun evi-forward-word-internal (pattern count context)
  (and (not context) (evi-eobp)
       (evi-error "End of buffer"))
  (if context
    (setq count (1- count)))
  (if (looking-at pattern)
    (setq count (1+ count)))
  (if (and (re-search-forward pattern nil 'limit count)
	   (or (not (eq context 'to-next))
	       (re-search-forward pattern
		 (save-excursion (end-of-line) (point)) 'limit)))
    (if (eq context 'to-end)
      (if (or (> count 0) (looking-at pattern))
	(goto-char (match-end 0))
	(forward-char))
      (goto-char (match-beginning 0)))
    (if (eobp)
      (backward-char))))

(defmotion horizontal evi-end-of-word (&optional count context)
  "Move to the end of the COUNTth next word."
  (evi-end-of-word-internal evi-word (or count 1) context))

(defmotion horizontal evi-end-of-Word (&optional count context)
  "Move to the end of the COUNTth next whitespace delimited word."
  (evi-end-of-word-internal evi-Word (or count 1) context))

(defun evi-end-of-word-internal (pattern count context)
  (and (not context) (evi-eobp)
       (evi-error "End of buffer"))
  (forward-char)
  (if (re-search-forward pattern nil 'limit count)
    (goto-char (- (match-end 0) (if context 0 1)))
    (if (eobp)
      (backward-char))))

(defmotion horizontal evi-backward-word (&optional count context)
  "Move to the beginning of the COUNTth previous word."
  (evi-backward-word-internal evi-word (or count 1) context))

(defmotion horizontal evi-backward-Word (&optional count context)
  "Move to the beginning of the COUNTth previous whitespace delimited word."
  (evi-backward-word-internal evi-Word (or count 1) context))

(defun evi-backward-word-internal (pattern count context)
  (if (bobp)
    (evi-error "Beginning of buffer"))
  (evi-iterate count
    (if (re-search-backward pattern nil 'limit)
      (progn
	(looking-at pattern)
	(let ((end (match-end 0))
	      (at-beginning nil))
	  (while (and (looking-at pattern) (= (match-end 0) end)
		      (not (setq at-beginning (bobp))))
	    (backward-char))
	  (if (not at-beginning)
	    (forward-char))))
      (evi-break))))

(defconst evi-sentence-beginning "\\([.?!][]\"')]*\\([\t\n]\\| [ \t\n]\\)\\|^[ \t]*\n\\|\\`\\)[ \t\n]*[^ \t\n]")

(defconst evi-sentence-ending "\\([.?!][]\"')]*\\([\t\n]\\| [ \t\n]\\)\\|^[ \t]*$\\)")

(defun evi-not-at (pattern &optional limit)
  (let ((start (point)))
    (if (re-search-backward pattern limit 'limit)
      (prog1
	(/= (match-end 0) start)
	(goto-char start))
      t)))

(defmotion horizontal evi-forward-sentence (&optional count context)
  "Move to the beginning of the COUNT'th next sentence."
  (and (not context) (evi-eobp)
       (evi-error "End of buffer"))
  (forward-char)
  (and (eq context 'to-next) (evi-not-at evi-sentence-beginning)
       (setq context 'to-end))
  (if (re-search-forward evi-sentence-beginning nil 'limit
			 (- (or count 1) (if context 1 0)))
    (if context
      (if (eq context 'to-end)
	(if (re-search-forward evi-sentence-ending nil 'limit)
	  (skip-chars-backward " \t\n"))
	(if (re-search-forward evi-sentence-beginning
	      (save-excursion
		(re-search-forward evi-paragraph-ending nil 'limit)
		(1- (match-beginning 0)))
	      'limit)
	  (backward-char)))
      (backward-char))))

(defmotion horizontal evi-backward-sentence (&optional count context)
  "Move to the beginning of the COUNT'th previous sentence."
  (if (bobp)
    (evi-error "Beginning of buffer"))
  (skip-chars-backward " \t\n")
  (if (re-search-backward evi-sentence-beginning nil 'limit (or count 1))
    (goto-char (1- (match-end 0)))))

(defconst evi-paragraph-beginning "\\(^[ \t]*\n\\|\\`\\)[ \t\n]*[^ \t\n]")

(defconst evi-paragraph-ending "^[ \t]*$")

(defmotion horizontal evi-forward-paragraph (&optional count context)
  "Move to the beginning of the COUNT'th next paragraph."
  (and (not context) (evi-eobp)
       (evi-error "End of buffer"))
  (forward-char)
  (and (eq context 'to-next) (evi-not-at evi-paragraph-beginning)
       (setq context 'to-end))
  (if (re-search-forward evi-paragraph-beginning nil 'limit
			 (- (or count 1) (if (eq context 'to-end) 1 0)))
    (if (eq context 'to-end)
	(if (re-search-forward evi-paragraph-ending nil 'limit)
	  (goto-char (1- (match-beginning 0))))
      (if context
	(beginning-of-line))
      (backward-char))))

(defmotion horizontal evi-backward-paragraph (&optional count context)
  "Move to the beginning of the COUNT'th previous paragraph."
  (if (bobp)
    (evi-error "Beginning of buffer"))
  (if (re-search-backward evi-paragraph-beginning nil 'limit (or count 1))
    (goto-char (1- (match-end 0)))))

(defconst evi-section-beginning "^\\({\\|\\.\\(NH\\|SH\\|H\\|HU\\|nh\\|sh\\)[ \t\n]\\)")

(defconst evi-section-ending "[ \t\n]*\n\\(}\\|\\.\\(NH\\|SH\\|H\\|HU\\|nh\\|sh\\)[ \t\n]\\)")

(defmotion horizontal evi-forward-section (&optional count context)
  "Move to the beginning of the COUNT'th next section."
  (and (not context) (evi-eobp)
       (evi-error "End of buffer"))
  (or context
      (evi-push-context (point)))
  (let ((start (point)))
    (skip-chars-forward "^ \t\n")
    (or (eobp)
	(forward-char))
    (and (eq context 'to-next) (evi-not-at evi-section-beginning start)
	 (setq context 'to-end)))
  (if (re-search-forward evi-section-beginning nil 'limit
			 (- (or count 1) (if (eq context 'to-end) 1 0)))
    (if (eq context 'to-end)
	(if (re-search-forward evi-section-ending nil 'limit)
	  (or (eq (preceding-char) ?})
	      (goto-char (match-beginning 0))))
      (goto-char (match-beginning 0))
      (if context
	(backward-char)))))

(defmotion horizontal evi-backward-section (&optional count context)
  "Move to the beginning of the COUNT'th previous section."
  (if (bobp)
    (evi-error "Beginning of buffer"))
  (or context
      (evi-push-context (point)))
  (re-search-backward evi-section-beginning nil 'limit (or count 1)))

(defun evi-region ()
  "Define region bounded by mark and point."
  (interactive)
  (if (< (point) (mark)) (exchange-point-and-mark))
  (setq evi-region-whole-lines nil))

(defun evi-region-whole-lines (context)
  "Define whole lines region bounded by mark and point."
  (interactive (evi-context-arg))
  (if (< (point) (mark)) (exchange-point-and-mark))
  (setq evi-region-whole-lines t)
  (evi-expand-region-to-lines evi-context))

;; Searching

(defmotion horizontal evi-search-forward
  (&string "/" string &optional count context)
  "Search forward for the ARGth occurence of a pattern.  A null string will
repeat the previous search."
  (if (not (string= string ""))
    (setq evi-search-pattern string))
  (if evi-search-pattern
    (evi-do-search (setq evi-search-forward t) evi-search-pattern (or count 1))
    (evi-error "No previous search pattern")))

(defmotion horizontal evi-search-backward
  (&string "?" string &optional count context)
  "Search backward for the ARGth occurence of a pattern.  A null string will
repeat the previous search."
  (if (not (string= string ""))
    (setq evi-search-pattern string))
  (if evi-search-pattern
    (evi-do-search
      (setq evi-search-forward nil) evi-search-pattern (or count 1))
    (evi-error "No previous search pattern")))

(defmotion horizontal evi-search-next (&optional count context)
  "Search for the next ARGth occurence of the previous search pattern."
  (if evi-search-pattern
    (evi-do-search evi-search-forward evi-search-pattern (or count 1))
    (evi-error "No previous search pattern")))

(defmotion horizontal evi-search-next-reverse (&optional count context)
  "Search for the next ARGth occurence of the previous search pattern
but look in the opposite direction."
  (let ((evi-search-forward (not evi-search-forward)))
    (do-evi-search-next count context)))

(defun evi-do-search (search-forward string count)
  (let ((case-fold-search evi-ignore-case)
	(search-string (if evi-search-magic string (evi-rework-magic string)))
	(starting-point (point)))
    (if (if search-forward
	  (evi-search-forward-count search-string count)
	  (evi-search-backward-count search-string count))
      (progn
	; ZZ if we know the search didn't take us far, perhaps we shouldn't
	; push a context...
	(evi-push-context starting-point)
        (goto-char (match-beginning 0)))
      (progn
	(goto-char starting-point)
	(evi-error
	  (concat
	    (if (> count 1) "Nth occurrence not found" "Pattern not found")
	    (if evi-search-wraparound ""
	      (if search-forward
		  " before end of file"
		  " before beginning of file"))))))))

;; rework the pattern so that . [ and * become literal, and \. \[ and \*
;; are 'magic' (i.e. behave as . [ and * in a regular expression)
(defun evi-rework-magic (string)
  (let ((offset (string-match "[\\.[*]" string)))
    (if offset
      (if (= (aref string offset) ?\\)
	(let ((next (1+ offset)))
	  (if (> (length string) next)
	    (let ((char (aref string next)))
	      (if (or (= char ?.) (= char ?[) (= char ?*))
		(concat (substring string 0 offset) (char-to-string char)
			(evi-rework-magic (substring string (1+ next))))
		(concat (substring string 0 (1+ next))
			(evi-rework-magic (substring string (1+ next))))))
	    string))
	(concat (substring string 0 offset) "\\"
		(char-to-string (aref string offset))
		(evi-rework-magic (substring string (1+ offset)))))
      string)))

; ZZ use evi-iterate
(defun evi-search-forward-count (string count)
  (if (> count 0)
    (progn (forward-char)
	   (if (re-search-forward string nil t)
	     (evi-search-forward-count string (1- count))
	     (if evi-search-wraparound
	       (progn (goto-char (point-min))
		      (if (re-search-forward string nil t)
			(evi-search-forward-count string (1- count)))))))
    t))

(defun evi-search-backward-count (string count)
  (if (> count 0)
    (if (re-search-backward string nil t)
      (evi-search-backward-count string (1- count))
      (if evi-search-wraparound
	(progn (goto-char (point-max))
	       (if (re-search-backward string nil t)
		 (evi-search-backward-count string (1- count))))))
    t))

(defmotion horizontal evi-find-character (&char char &optional count context)
  "Search for CHAR on the current line.  With COUNT find the COUNT'th occurance."
  (setq evi-find-character char
	evi-find-forward t
	evi-find-up-to nil)
  (evi-find-character-internal (or count 1) context))

(defmotion horizontal evi-find-char-backwards
  (&char char &optional count context)
  "Search backwards for CHAR on the current line.  With COUNT find the
COUNT'th occurance."
  (setq evi-find-character char
	evi-find-forward nil
	evi-find-up-to nil)
  (evi-find-character-backwards-internal (or count 1) context))

(defmotion horizontal evi-find-character-before
  (&char char &optional count context)
  "Search for CHAR on the current line and leave the cursor on the character
before it.  With COUNT find the COUNT'th occurance."
  (setq evi-find-character char
	evi-find-forward t
	evi-find-up-to t)
  (evi-find-character-internal (or count 1) context))

(defmotion horizontal evi-find-char-backwards-after
  (&char char &optional count context)
  "Search backwards for CHAR on the current line and leave the cursor on
the character after it.  With COUNT find the COUNT'th occurance."
  (setq evi-find-character char
	evi-find-forward nil
	evi-find-up-to t)
  (evi-find-character-backwards-internal (or count 1) context))

(defmotion horizontal evi-find-next-character (&optional count context)
  "Search for the next COUNT'th occurence of the previous search character."
  (if evi-find-character
    (if evi-find-forward
      (evi-find-character-internal (or count 1) context)
      (evi-find-character-backwards-internal (or count 1) context))
    (evi-error "No previous search character")))

(defmotion horizontal evi-find-next-character-reverse (&optional count context)
  "Search for the next COUNT'th occurence of the previous search character
in the opposite direction."
  (let ((evi-find-forward (not evi-find-forward)))
    (do-evi-find-next-character count context)))

(defun evi-find-character-internal (count context)
  (forward-char)
  (let ((case-fold-search nil))
    (if (search-forward (char-to-string evi-find-character)
			(save-excursion (end-of-line) (point)) t count)
      (if evi-find-up-to
	(backward-char))
      (progn (backward-char)
	     (evi-error "No more occurences on this line"))))
  (or context
      (backward-char)))

(defun evi-find-character-backwards-internal (count context)
  (let ((case-fold-search nil))
    (or (search-backward (char-to-string evi-find-character)
			 (save-excursion (beginning-of-line) (point)) t count)
	(evi-error "No more occurences on this line")))
  (if evi-find-up-to
    (forward-char)))

(defmotion horizontal evi-paren-match (&optional count context)
  "Move cursor to matching parenthesis, brace or bracket."
  (let ((end-point (save-excursion (end-of-line) (point))))
    (if (re-search-forward "[][(){}]" end-point t)
      (progn (backward-char)
	     (if (looking-at "[({[]") ; )
	       (progn (forward-sexp 1)
		      (or context (backward-char)))
	       (progn (forward-char)
		      (if context (set-mark (1+ (mark))))
		      (backward-sexp 1))))
      (evi-error "Nothing on rest of line to balance"))))

;; Repeating

(defun evi-repeat ()
  "Repeat last modifying command."
  (interactive)
  (let ((command-to-repeat evi-last-command-keys))
    (evi-execute-macro evi-last-command-keys)
    (setq evi-last-command-keys command-to-repeat)))

(defun evi-prompt-repeat ()
  "Print last modifying command."
  (interactive)
  (let ((command (evi-read-string "Repeat: " evi-last-command-keys)))
    (evi-execute-macro command)
    (setq evi-last-command-keys command)))

;; Prefix counts

(defun evi-read-number (prefix-value)
  (let ((char (evi-read-command-char)))
    (if (and (>= char ?0) (<= char ?9))
      (evi-read-number (+ (* prefix-value 10) (- char ?0)))
      (progn (evi-unread-command-char char)
	     prefix-value))))

(defun evi-prefix-digit ()
  "Prefix count."
  (interactive)
  (let ((evi-prefix-count (* evi-prefix-count-multiplier
			     (evi-read-number (- last-command-char ?0)))))
    (evi-get-command)))

;; Registers

(defun evi-prefix-register ()
  "Prefix register."
  (interactive)
  (let ((char (evi-read-command-char)))
    (if (or (eq char ?') (eq char ?")) ;"))
      (evi-copy-string-to-register
        (if (eq char ?') (char-to-string (evi-read-command-char))
			 (evi-read-string "\" "))
	(or evi-register-spec (cons evi-register-unnamed nil)))
      (let ((evi-register-spec (cons (evi-register-number char)
				     (not (and (>= char ?a) (<= char ?z))))))
	(evi-get-command)))))

(defun evi-register-number (register-name)
  (cond ((and (>= register-name ?a) (<= register-name ?z))
	  (+ (- register-name ?a) 10))
	((and (>= register-name ?A) (<= register-name ?Z))
	  (+ (- register-name ?A) 10))
	((and (>= register-name ?0) (<= register-name ?9))
	  (% (+ evi-digit-register (- register-name ?1)) 9))
	((eq register-name ?^)
	  evi-register-unnamed)
	((eq register-name ?@)
	  (or evi-last-macro-register
	      (evi-error "No previous macro register specified")))
	(t (evi-error "Invalid register name"))))

(defun evi-register-name (register-number)
  (if (> register-number 9)
    (+ register-number (- ?a 10))
    (+ register-number ?1)))

(defun evi-copy-region-to-registers (number-register-also)
  (let ((string (buffer-substring (mark) (point))))
    (evi-copy-string-to-register string evi-register-spec)
    (if number-register-also
      (progn (aset evi-registers
		   evi-digit-register (cons string evi-region-whole-lines))
	     (setq evi-digit-register (% (1+ evi-digit-register) 9))))))

(defun evi-copy-string-to-register (string register-spec)
  (let ((register-number (car register-spec)))
    (if (not (eq register-number evi-register-unnamed))
      (aset evi-registers
	    evi-register-unnamed (cons string evi-region-whole-lines)))
    (if register-spec
      (aset evi-registers register-number
	    (if (cdr register-spec)
	      (let ((register (aref evi-registers register-number)))
		(cons (concat (car register) string) (cdr register)))
	      (cons string evi-region-whole-lines))))))

;; Undoing

(defun evi-undo ()
  "Undo previous change."
  (interactive)
  ; ZZ - is this the only place we're concerned with unnecessary output
  ; during a macro?
  (or evi-current-macro
      (message "undo!"))
  (evi-undo-start)
  (evi-undo-one-change)
  (evi-fixup-cursor 'vertical))

(defun evi-undo-line ()
  "Undo all changes to this line."
  (interactive)
  (evi-undo-start)
  (evi-undo-one-line)
  (evi-fixup-cursor 'vertical))

(defun evi-undo-start ()
  (undo-start)
  ; if the first record is a boundary, skip it
  (while (and pending-undo-list (null (car pending-undo-list)))
    (setq pending-undo-list (cdr pending-undo-list))))

(defun evi-undo-more ()
  "Continue undoing previous changes."
  (interactive)
  (if (boundp 'pending-undo-list)
    (progn (message "undo more!")
	   (evi-undo-one-change)
	   (evi-fixup-cursor 'vertical))
    (evi-error "No previous undo to continue")))

(defun evi-undo-one-change ()
  (let ((modified (buffer-modified-p)))
    (undo-more 1)
    (and modified (not (buffer-modified-p))
	 (delete-auto-save-file-if-necessary)))
  (evi-reset-goal-column))

(defvar evi-last-undo-line-mark nil)

; undo records are:
;   (t . ...) which marks a file save
;   ("string" . pos) which undoes a delete
;   (pos . pos) which undoes an insert
(defun evi-undo-one-line ()
  (if (eq evi-last-undo-line-mark (cdr buffer-undo-list))
    (evi-error "No undo for this line"))
  (let* ((begin (save-excursion (beginning-of-line) (point)))
	 (end (save-excursion (end-of-line) (point)))
	 (undo-new nil)
	 (something-to-do nil))
    (evi-enumerate-condition undo-record pending-undo-list
      (cond ((eq (car undo-record) t)
	      (setq undo-new (nconc undo-new list))
	      nil)
	    ((stringp (car undo-record))
	      (if (and (>= (cdr undo-record) begin)
		       (<= (cdr undo-record) end))
		(progn (setq end (+ end (length (car undo-record))))
		       (setq undo-new
			     (nconc undo-new (list undo-record)))
		       (setq something-to-do t)
		       t)
		(progn (setq undo-new (nconc undo-new (list nil) list))
		       nil)))
	    ((integerp (car undo-record))
	      (let* ((first (car undo-record))
		     (second (cdr undo-record))
		     (begin2 (if (< first begin) begin first))
		     (end2 (if (> second end) end second))
		     (diff (- end2 begin2)))
		(if (and (<= first end) (>= second begin) (/= begin2 end2))
		  (progn
		    (setq undo-new
			  (nconc undo-new (list (cons begin2 end2))))
		    (setq something-to-do t)
		    (if (or (< first begin) (> second end))
		      (progn
			(nconc undo-new (list nil))
			(if (< first begin)
			  (nconc undo-new (list (cons first begin))))
			(if (> second end)
			  (nconc undo-new
			    (list (cons (- end diff) (- second diff)))))
			(nconc undo-new (cdr list))
			nil)
		      (progn (setq end (- end diff))
			     t)))
		  (progn
			 (setq undo-new (nconc undo-new (list nil) list))
			 nil))))
	    ((eq undo-record nil)
	      t)))
    (if something-to-do
      (let ((modified (buffer-modified-p)))
	(setq pending-undo-list undo-new)
	(undo-more 1)
	(message "Undo!")
	(setq evi-last-undo-line-mark buffer-undo-list)
	(beginning-of-line)
	(and modified (not (buffer-modified-p))
	     (delete-auto-save-file-if-necessary)))
      (evi-error "No undo for this line")))
  (evi-reset-goal-column))

;; Marks

(defun evi-mark (char &optional count)
  "Mark location."
  (interactive (evi-character-arg))
  (cond ((and (>= char ?a) (<= char ?z))
	  (aset evi-registers (+ (- char ?a) 36) (point-marker)))
	((eq char ?.)
	  (set-mark (point)))))

(defmotion horizontal evi-goto-mark-horizontal (&optional count context)
  "Goto a mark."
  (evi-goto-mark-internal (evi-read-command-char) context))

(defmotion vertical evi-goto-mark-vertical (&optional count context)
  "Goto a mark.  If an operand, define a whole lines region."
  (evi-goto-mark-internal (evi-read-command-char) context)
  (or context
    (back-to-indentation)))

(defun evi-goto-mark-internal (char &optional context)
  (cond ((and (>= char ?a) (<= char ?z))
	  (let ((marker (aref evi-registers (+ (- char ?a) 36))))
	    (if (not (eq (current-buffer) (marker-buffer marker)))
	      (progn (switch-to-buffer (marker-buffer marker))
		     ; unpleasant, but best we can do... (?)
		     (if context (set-mark (point)))))
	    (evi-push-context)
	    (goto-char marker)))
	((or (eq char ?`) (eq char ?'))
	  (goto-char (evi-exchange-context)))
	((eq char ?.)
	  (goto-char (evi-pop-context)))
	((eq char ?,)
	  (goto-char (evi-unpop-context)))))

(defun evi-push-context (&optional offset)
  (let ((marker (if offset (set-marker (make-marker) offset) (point-marker))))
    (aset evi-context-ring evi-context-ring-cursor marker)
    (setq evi-context-ring-cursor
	  (if (= evi-context-ring-cursor 9) 0 (1+ evi-context-ring-cursor)))))

(defun evi-pop-context ()
  (setq evi-context-ring-cursor
    (if (= evi-context-ring-cursor 0) 9 (1- evi-context-ring-cursor)))
  (aref evi-context-ring evi-context-ring-cursor))

(defun evi-unpop-context ()
  (setq evi-context-ring-cursor
    (if (= evi-context-ring-cursor 9) 0 (1+ evi-context-ring-cursor)))
  (aref evi-context-ring evi-context-ring-cursor))

(defun evi-exchange-context ()
  (let ((cursor
	 (if (= evi-context-ring-cursor 0) 9 (1- evi-context-ring-cursor))))
    (prog1 (aref evi-context-ring cursor)
	   (aset evi-context-ring cursor (point-marker)))))

;; Misc

(defun evi-file-info ()
  "Give information on the file associated with the current buffer."
  (interactive)
  (let* ((line-number (count-lines 1 (min (1+ (point)) (point-max))))
	 (total-lines (1- (+ line-number (count-lines (point) (point-max)))))
	 (file-name (buffer-file-name)))
    (message "\"%s\"%s%s line %d of %d, column %d --%d%%--"
	     (if file-name
	       (if evi-global-directory
		 (evi-abbreviate-file-name file-name (car evi-directory-stack))
		 file-name)
	       "")
	     (if (or buffer-read-only
		     (and file-name (not (file-writable-p file-name))))
	       " [Read only]" "")
	     (if (buffer-modified-p) " [Modified]" "")
	     line-number
	     total-lines
	     (1+ (current-column))
	     (/ (* line-number 100) total-lines))))

(defun evi-abbreviate-file-name (file-name directory &optional abbrev)
  (let* ((length (length directory))
	 (ends-in-slash (= (aref directory (1- length)) ?/)))
    (if (and (> length 0)
	     (>= (length file-name) length)
	     (string= (substring file-name 0 length) directory))
      (concat (or abbrev "")
	      (substring file-name
			 (+ length (if (or abbrev ends-in-slash) 0 1))))
      file-name)))

(defun evi-tag ()
  "Go to the tag which is the next word in the buffer."
  (interactive)
  (evi-motion-command 'do-evi-forward-word 'horizontal 1 'to-end)
  (ex-tag (buffer-substring (mark) (point))))

;; Ex

(defun evi-ex-command ()
  "Execute an ex command."
  (interactive)
  (evi-do-ex-command-string (ex-read-command))
  (evi-fixup-cursor 'vertical)
  (evi-save-command-keys))

(defvar ex-complete-enabled nil)
(defvar ex-restart-offset nil)

; there's some rather nasty hoop-jumping going on just to get completion on
; various commands.  hopefully I'm just ignorant and there's really a better
; way of doing this...
(defun ex-read-command ()
  (setq ex-command-initial nil)
  (let ((command nil))
    (while (null command)
      (setq ex-complete-enabled nil
	    ex-restart-offset nil
	    command (evi-read-string ":" ex-command-initial
		      (list evi-input-map-map evi-ex-map evi-input-map)))
      (if ex-restart-offset
	(setq command (evi-read-command-completing
		        (substring command 0 ex-restart-offset)
			(substring command ex-restart-offset)))))
    command))

(defun evi-read-command-completing (initial prefix)
  (setq ex-command-initial nil)
  (let ((minibuffer-local-completion-map evi-completion-map))
   (let ((command
    (concat initial
      (if (eq ex-complete-enabled 'file)
	(completing-read
	  (concat ":" initial) 'read-file-name-internal
	  (if evi-global-directory (car evi-directory-stack) default-directory)
	  nil prefix)
	(completing-read (concat ":" initial)
	  ; it would be nice to be able to just say `buffer-alist' here,
	  ; but NOOooooo...  (see comment about buffer_alist in buffer.c)
	  (mapcar 'list
	    (evi-filter (function (lambda (name) (/= (aref name 0) ? )))
			(mapcar 'buffer-name (buffer-list))))
	  nil nil prefix)))))
    (if ex-command-initial
      (progn (setq ex-command-initial
		   (substring initial 0 (1- (length initial))))
	     nil)
      command))))

(defun evi-completion-delete-backward-char ()
  (interactive)
  (if (bobp)
    (progn (setq ex-command-initial t)
	   (exit-minibuffer))
    (backward-delete-char 1)))

(defun evi-filter (pred list)
  (if list
    (if (funcall pred (car list))
      (cons (car list) (evi-filter pred (cdr list)))
      (evi-filter pred (cdr list)))))

(defun ex-space ()
  (interactive)
  (if ex-complete-enabled
    (progn (skip-chars-backward "^ \t")
	   (setq ex-restart-offset (- (point) evi-insert-point)
		 ; ZZ - note the semi-bogus hardcoded space character....
		 unread-command-char ? )
	   (evi-exit-command-loop)))
  (insert ? )
  (save-excursion
    (goto-char evi-insert-point)
    (ex-scan-addresses)
    (let ((command (ex-scan-command-name)))
      (if command
	(setq ex-complete-enabled (ex-word-arg-type command))))))

(defun ex-word-arg-type (command-struct)
  (let* ((prototype (cdr (car (cdr command-struct))))
	 (rest
	   (evi-enumerate-condition arg prototype
				    (not (or (eq (cdr arg) 'file)
					     (eq (cdr arg) 'buffer))))))
    (if rest (cdr (car rest)))))

(defun ex-delete-backward-char ()
  (interactive)
  (if (<= (point) evi-insert-point)
    (throw 'exit nil)
    (progn
      (if (and ex-complete-enabled (= (preceding-char) ? ))
	(setq ex-complete-enabled nil))
      (delete-backward-char 1))))

(defun ex-complete ()
  (interactive)
  (if ex-complete-enabled
    (progn (skip-chars-backward "^ \t")
	   ; ZZ - note the bogus hardcoded TAB character here and farther down
	   (setq ex-restart-offset (- (point) evi-insert-point)
		 unread-command-char ?\t)
	   (evi-exit-command-loop))
    (insert ?\t)))

(defun evi-do-ex-command-file (filename)
  (if (file-readable-p filename)
    (let ((ex-user-buffer (current-buffer)))
      (set-buffer ex-work-space)
      (delete-region (point-min) (point-max))
      (insert-file-contents filename)
      (goto-char (point-min))
      (evi-do-ex-command)
      (set-buffer ex-user-buffer))))

(defun evi-do-ex-command-string (command-string)
  (let ((ex-user-buffer (current-buffer)))
    (set-buffer ex-work-space)
    (delete-region (point-min) (point-max))
    (insert command-string "\n")
    (goto-char (point-min))
    (evi-do-ex-command)
    (set-buffer ex-user-buffer)))

; ZZ this should be cleaned up
(defvar ex-user-buffer nil)

;; Note - it is expected that the function that calls this one has set
;; ex-user-buffer, and switched to buffer ex-work-space
(defun evi-do-ex-command ()
  (while (not (eobp))
    (let ((command (ex-scan-command)))
      (set-buffer ex-user-buffer)
      (if evi-global-directory
	(let ((default-directory (car evi-directory-stack)))
	  (eval command))
	(eval command))
      (set-buffer ex-work-space)
      (skip-chars-forward "^|\n")
      (forward-char))))

(defun ex-scan-command ()
  (let* ((addresses (ex-scan-addresses))
	 (command-struct (ex-scan-command-name))
	 (number-of-addresses (car (car (cdr command-struct))))
	 (command-name (car (car command-struct)))
	 (command-prototype (cdr (car (cdr command-struct))))
	 (command-function (cdr (cdr command-struct))))
    (if (null command-struct)
      (evi-error "Unknown ex command"))
    (if (> (ex-count-addresses addresses) number-of-addresses)
      (evi-error "The %s command only needs %d addresses"
			    command-name number-of-addresses))
    (let ((parameter-list (ex-scan-parameter-list command-prototype)))
      (cons command-function
	    (cond ((eq number-of-addresses 1)
		    (cons (list 'quote (car addresses)) parameter-list))
		  ((eq number-of-addresses 2)
		    (cons (list 'quote addresses) parameter-list))
		  (t
		    parameter-list))))))

(defun ex-scan-parameter-list (prototype-list)
  (if prototype-list
    (let ((prototype (cdr (car prototype-list)))
	  (skip-white (eq (car (car prototype-list)) t)))
      (if skip-white
	(skip-chars-forward " \t")
	(if (eq (car (car prototype-list)) 'backup)
	  (backward-char)))
      (cons (cond ((null prototype)
		    nil)
		  ((stringp prototype)
		    (ex-scan-string prototype))
		  ((eq prototype 'address)
		    (list 'quote (ex-scan-address)))
		  ((eq prototype 'register)
		    (list 'quote (ex-scan-register)))
		  ((eq prototype 'file)
		    (ex-scan-file))
		  ((eq prototype 'buffer)
		    (ex-scan-buffer))
		  ((eq prototype 'rest-of-line)
		    (ex-scan-rest-of-line))
		  ((eq prototype 'word)
		    (ex-scan-word))
		  ((eq prototype 'regular-expression)
		    (ex-scan-regular-expression))
		  ((eq prototype 'command)
		    (list 'quote (ex-scan-command)))
		  ((eq prototype 'settings)
		    (list 'quote (ex-scan-settings))))
	    (ex-scan-parameter-list (cdr prototype-list))))))

(defun ex-scan-addresses ()
  (skip-chars-forward " \t")
  (if (= (following-char) ?%)
    (cons (cons (cons 'number 1) 0) (cons (cons 'dollar nil) 0))
    (if (looking-at "[-+0-9.$'/?]")
      (cons
	(ex-scan-address)
	(progn (skip-chars-forward " \t")
	       (if (= (following-char) ?,)
		 (progn (forward-char)
			(skip-chars-forward " \t")
			(ex-scan-address))
		 (cons (cons nil nil) 0))))
      (cons (cons (cons nil nil) 0) (cons (cons nil nil) 0)))))

(defun ex-scan-address ()
  (cons (ex-scan-linespec) (ex-scan-line-offset)))

(defun ex-scan-linespec ()
  (let ((char (following-char)))
    (cond
      ((and (>= char ?0) (<= char ?9))
	(let ((start (point)))
	  (skip-chars-forward "0-9")
	  (cons 'number (string-to-int (buffer-substring start (point))))))
      ((eq char ?.)
	(forward-char)
	(cons 'dot nil))
      ((eq char ?$)
	(forward-char)
	(cons 'dollar nil))
      ((eq char ?')
	(forward-char 2)
	(cons 'mark (preceding-char)))
      ((eq char ?/)
	(cons 're-forward (ex-scan-regular-expression)))
      ((eq char ??)
	(cons 're-backward (ex-scan-regular-expression))))))

(defun ex-scan-regular-expression ()
  (forward-char)
  (let ((skip-pattern (concat "^\n\\\\" (char-to-string (preceding-char))))
	(start (point)))
    (skip-chars-forward skip-pattern)
    (while (= (following-char) ?\\)
      (forward-char 2)
      (skip-chars-forward skip-pattern))
    (prog1
      (buffer-substring start (point))
      (if (not (eolp))
	(forward-char)))))

(defun ex-scan-line-offset ()
  (let ((char (following-char)))
    (if (or (eq char ?+) (eq char ?-))
      (let ((start (1+ (point))))
	(forward-char)
	(skip-chars-forward "0-9")
	(let ((offset (string-to-int (buffer-substring start (point)))))
	  (if (eq char ?+)
	    offset
	    (- offset))))
      0)))

;; ZZ maybe recognize here that 0 is invalid?
(defun ex-define-region (addresses whole-lines default-whole-file)
  (let ((start (car addresses))
	(end (cdr addresses)))
    (if (and (null (car (car start))) default-whole-file)
      (progn (set-mark (point-min))
	     (goto-char (point-max)))
      (progn (let ((starting-point (point)))
	       (ex-goto-address start)
	       (set-mark (point))
	       (ex-goto-address end starting-point))
	     (if whole-lines
	       (evi-expand-region-to-lines 'ex))))))

(defvar ex-previous-re nil)

(defun ex-goto-address (address &optional starting-point)
  (let ((token (car (car address)))
	(value (cdr (car address))))
    (cond ((eq token 'number)
	    (goto-line value))
	  ((eq token 'dot)
	    (if starting-point (goto-char starting-point)))
	  ((eq token 'dollar)
	    (goto-char (point-max))
	    (forward-line -1))
	  ((eq token 'mark)
	    (evi-goto-mark-internal value))
	  ((eq token 're-forward)
	    (if (= (length value) 0)
	      (if ex-previous-re
		(setq value ex-previous-re)
		(evi-error "No previous regular expression"))
	      (setq ex-previous-re value))
	    (if starting-point (goto-char starting-point))
	    (end-of-line)
	    (let ((message (catch 'abort
			     (evi-do-search t value 1)
			     nil)))
	      (if message
		(progn (forward-line -1)
		       (evi-error message)))))
	  ((eq token 're-backward)
	    (if starting-point (goto-char starting-point))
	    (evi-do-search nil value 1))))
  (forward-line (cdr address)))

(defun ex-goto-line-after-address (address)
  (if (null (car (car address)))
    (forward-line)
    (if (and (eq (car (car address)) 'number)
	     (= (cdr (car address)) 0))
      (goto-char (point-min))
      (progn (ex-goto-address address)
	     (forward-line)))))

(defun ex-count-addresses (addresses)
  (if (eq (car (car (car addresses))) nil)
    0
    (if (eq (car (car (cdr addresses))) nil)
      1
      2)))

(defun ex-scan-command-name ()
  (skip-chars-forward " \t")
  (let ((start (point)))
    (if (looking-at "[a-zA-Z!<=>&@]")
      (progn (forward-char)
	     (let ((char (preceding-char)))
	       (if (or (and (>= char ?a) (<= char ?z))
		       (and (>= char ?A) (<= char ?Z)))
		 (skip-chars-forward "a-zA-Z")))))
    (ex-lookup-command ex-commands (buffer-substring start (point)))))

(defun ex-lookup-command (command-list command)
  (evi-find cmd-struct command-list
    (if (ex-command-eq command (car cmd-struct))
      cmd-struct)))

(defun ex-command-eq (command command-cell)
  (let ((full-command (car command-cell)))
    (or (string= command full-command)
	(let ((command-length (length command)))
	  (and (>= command-length (cdr command-cell))
	       (< command-length (length full-command))
	       (string= command
			(substring (car command-cell) 0 (length command))))))))

(defun ex-scan-register ()
  (if (= (following-char) ?") ; ")
    (progn
      (forward-char 2)
      (let ((char (preceding-char)))
	(cons (evi-register-number char)
	      (not (and (>= char ?a) (<= char ?z))))))
    ; error checking?
    (cons evi-register-unnamed nil)))

(defun ex-scan-file ()
  (ex-scan-quoted "%#*?$\\\\\C-v" " \t|\n" t))

(defun ex-scan-buffer ()
  (ex-scan-quoted "\\\\\C-v" "|\n"))

(defun ex-scan-rest-of-line ()
  (ex-scan-quoted "\\\\\C-v" "|\n"))

(defun ex-scan-word ()
  (ex-scan-quoted "\\\\\C-v" " \t|\n"))

(defun ex-scan-quoted (stop-chars delim-chars &optional file)
  (let ((start (point))
	(skip-chars (concat "^" stop-chars delim-chars))
	(stop-pat (concat "[" stop-chars "]"))
	(expand-glob nil))
    (skip-chars-forward skip-chars)
    (while (looking-at stop-pat)
      (let ((char (following-char)))
	(cond ((or (= char ?\\) (= char ?\C-v))
		(delete-region (point) (1+ (point)))
		(let ((char (following-char)))
		  (cond ((= char ?e)
			  (delete-region (point) (1+ (point)))
			  (insert ?\e))
			((= char ?n)
			  (delete-region (point) (1+ (point)))
			  (insert ?\n))
			((= char ?r)
			  (delete-region (point) (1+ (point)))
			  (insert ?\r))
			((= char ?t)
			  (delete-region (point) (1+ (point)))
			  (insert ?\t))
			((and (= char ?C) (= (char-after (1+ (point))) ?-))
			  (let ((char (char-after (+ (point) 2))))
			    (insert (- char (if (< char ?a) ?@ ?`)))
			    (delete-region (point) (+ (point) 3))))
			(t (forward-char 1)))))
	      ((= char ?%)
		(let ((file-name (buffer-file-name)))
		  (if file-name
		    (progn
		      (delete-region (point) (1+ (point)))
		      (insert file-name))
		    (evi-error
		      "Buffer has no filename to substitute for %%"))))
	      ((= char ?#)
		(let* ((buffer (evi-next-file-buffer nil))
		       (file-name (and buffer (buffer-file-name buffer))))
		  (if file-name
		    (progn
		      (delete-region (point) (1+ (point)))
		      (insert file-name))
		    (evi-error
		      "No alternate filename to substitute for #"))))
	      (t
		(setq expand-glob t)
		(forward-char))))
      (skip-chars-forward skip-chars))
    (if expand-glob
      (progn (shell-command-on-region start (point)
	       (concat "echo " (buffer-substring start (point))) t)
	     (backward-char)))
    (buffer-substring start (point))))

(defun ex-scan-string (string)
  (let ((string-length (length string)))
    (if (<= string-length
	    (- (save-excursion (skip-chars-forward "^|\n") (point))
	       (point)))
      (let ((buffer-string
	      (buffer-substring (point) (+ (point) string-length))))
	(if (string= string buffer-string)
	  (progn (forward-char string-length)
		 t))))))

(defun ex-not-implemented (&optional arg)
  (message "Command not implemented"))

(defun ex-change-buffer (exclam buffer-name)
  (ex-change-buffer-internal exclam buffer-name nil))

(defun ex-change-buffer-other-window (exclam buffer-name)
  (ex-change-buffer-internal exclam buffer-name t))

(defun ex-change-buffer-internal (exclam buffer-name other-window)
  (if (string= buffer-name "")
    (setq buffer-name (buffer-name (other-buffer (current-buffer)))))
  (let ((found (ex-verify-buffer buffer-name)))
    (if (or exclam found)
      (if other-window
	(switch-to-buffer-other-window buffer-name)
	(switch-to-buffer buffer-name))
      (message "Buffer \"%s\" does not exist" buffer-name))
    (and exclam (not found)
	 (evi))))

(defun ex-verify-buffer (buffer-name)
  (evi-find buf (buffer-list) (string= (buffer-name buf) buffer-name)))

(defun ex-change-directory (directory-name)
  (if (string= directory-name "")
    (setq directory-name "~"))
  (let ((expnd-dir-name (expand-file-name directory-name)))
    (if evi-global-directory
      (setcar evi-directory-stack expnd-dir-name)
      (setq default-directory expnd-dir-name))))

(defun ex-push-directory (directory-name)
  (if (string= directory-name "")
    (if (null (cdr evi-directory-stack))
      (evi-error "Only one directory")
      (setq evi-directory-stack
	    (cons (car (cdr evi-directory-stack))
		  (cons (car evi-directory-stack)
			(cdr (cdr evi-directory-stack))))))
    (setq evi-directory-stack
	  (cons (expand-file-name directory-name) evi-directory-stack))))

(defun ex-pop-directory ()
  (if (null (cdr evi-directory-stack))
    (evi-error "Only one directory left")
    (setq evi-directory-stack (cdr evi-directory-stack))))

(defun ex-directory-stack ()
  (let ((home (getenv "HOME")))
    (message
      (mapconcat (function
		   (lambda (f)
		     (let* ((dir (evi-abbreviate-file-name f home "~"))
			    (end (1- (length dir))))
		       (if (= (aref dir end) ?/)
			 (substring dir 0 end)
			 dir))))
		 evi-directory-stack " "))))

(defun ex-copy (from-addresses to-address)
  (ex-define-region from-addresses t nil)
  (let ((text (buffer-substring (mark) (point))))
    (ex-goto-line-after-address to-address)
    (insert text)))

(defun ex-delete (addresses register-struct)
  (let ((evi-register-spec register-struct))
    (ex-define-region addresses t nil)
    (evi-copy-region-to-registers t)
    ; to make undo's come out right
    (if (< (mark) (point))
      (exchange-point-and-mark))
    (delete-region (point) (mark))))

(defun ex-edit (exclam file-name)
  (ex-edit-internal exclam file-name nil))

(defun ex-edit-other-window (exclam file-name)
  (ex-edit-internal exclam file-name t))

(defun ex-edit-internal (exclam file-name other-window)
  (if (= (length file-name) 0)
    (if (and (not exclam) (not other-window) (buffer-modified-p))
      (message "Buffer modified since last save (use :edit! to override)")
      (if other-window
	(split-window-vertically)
	(if (null (buffer-file-name))
	  (message "Buffer has no file associated with it")
	  (revert-buffer nil t)
	  (evi))))
    (if other-window
      (find-file-other-window file-name)
      (find-file file-name))
    (evi)))

(defun ex-file (file-name)
  (if (= (length file-name) 0)
    (evi-file-info)
    (set-visited-file-name file-name)))

(defun ex-gdb (program-name)
  (let ((shell-mode-hook
	 (function
	   (lambda ()
	     (evi)
	     (setq evi-buffer-local-vi-map evi-shell-map)))))
    (gdb program-name)
    (evi-insert)))

(defun ex-global (addresses pattern command)
  (if (= (length pattern) 0)
    (if ex-previous-re
      (setq pattern ex-previous-re)
      (evi-error "No previous regular expression"))
    (setq ex-previous-re pattern))
  (ex-define-region addresses t t)
  (exchange-point-and-mark)
  (let ((case-fold-search evi-ignore-case)
	(next-line-mark (make-marker))
	(end-line-mark (make-marker))
	(large-region (> (- (mark) (point)) 5000)))
    (if large-region
      (message "running global command... "))
    (set-marker end-line-mark (mark))
    (while (< (point) end-line-mark)
      (if (re-search-forward pattern end-line-mark 1)
	(progn
	  ;; check to make sure ex also does this in case of line wrap
	  (goto-char (match-beginning 0))
	  (save-excursion
	    (forward-line)
	    (set-marker next-line-mark (point)))
	  ; (beginning-of-line)
	  (eval command)
	  (goto-char next-line-mark))))
    (if large-region
      (message "running global command... complete."))
    (set-marker next-line-mark nil)
    (set-marker end-line-mark nil)))

(defun ex-kill-buffer (exclam buffer-name)
  (and (not exclam) (buffer-file-name) (buffer-modified-p)
       (evi-error
	 "No write since last change (use :kill! to override)"))
  (set-buffer-modified-p nil)
  (delete-auto-save-file-if-necessary)
  (kill-buffer (if (string= buffer-name "") (current-buffer) buffer-name))
  (setq ex-user-buffer (current-buffer)))

(defun ex-map (exclam key definition)
  (if (string= definition "")
    (message (mapconcat 'single-key-description
			(lookup-key
			  (if exclam evi-input-map-map evi-map-map) key) ""))
    (if exclam
      (evi-define-key '(input-map) key definition)
      (evi-define-key '(map) key definition))))

(defun ex-move (from-addresses to-address)
  (ex-define-region from-addresses t nil)
  (let ((text (buffer-substring (mark) (point)))
	(to-mark (copy-marker (save-excursion
				(ex-goto-line-after-address to-address)
				(point)))))
    ; to make undo's come out right
    (if (< (mark) (point))
      (exchange-point-and-mark))
    (delete-region (point) (mark))
    (goto-char to-mark)
    (insert text)
    (set-marker to-mark nil)))

(defun ex-preserve ()
  (do-auto-save))

(defun ex-print (addresses)
  (let ((position (save-excursion
		    (ex-define-region addresses t nil) (point))))
    (switch-to-buffer-other-window (current-buffer))
    (goto-char position)
    (select-window (previous-window))))

(defun ex-next (exclam)
  (ex-next-internal exclam nil))

(defun ex-next-other-window (exclam)
  (ex-next-internal exclam t))

(defun ex-next-internal (exclam other-window)
  (let* ((next-buffer (evi-next-file-buffer t)))
    (if next-buffer
      (progn (bury-buffer (current-buffer))
	     (if other-window
	       (switch-to-buffer-other-window next-buffer)
	       (switch-to-buffer next-buffer)))
      (message "All files are displayed"))))

(defun evi-next-file-buffer (not-in-window)
  (let ((rest-of-list
	  (evi-enumerate-condition buffer (cdr (buffer-list))
	    (or (and not-in-window (get-buffer-window buffer))
		(null (buffer-file-name buffer))))))
    (if rest-of-list
      (car rest-of-list))))

(defun ex-put (address register-struct)
  (ex-goto-line-after-address address)
  (let ((register (aref evi-registers (car register-struct))))
    (if register
      (save-excursion
	(insert (evi-register-text register))
	(if (not (evi-register-whole-lines-p register))
	  (insert ?\n)))
      (if evi-register-spec
	(message "Nothing in register %c"
		 (evi-register-name (car evi-register-spec)))
	(message "No text to put")))))

;; ZZ should move to a misc section - actually this shouldn't be here: surely
;; this is defined somewhere else?

(defun list-apply (func l)
  (if l
    (progn (apply func (car l) nil)
	   (list-apply func (cdr l)))))

(defun ex-quit (discard)
  (if discard
    (progn
      (list-apply
	(function (lambda (buf)
	  (if (buffer-file-name buf)
	    (progn (set-buffer buf)
		   (delete-auto-save-file-if-necessary)))))
	(buffer-list))
      (kill-emacs))
    (save-buffers-kill-emacs)))

(defun ex-read (address shell-command arg)
  (ex-goto-line-after-address address)
  (if shell-command
    (shell-command arg t)
    (evi-insert-file arg)))

; there's a bug in insert-file-contents that doesn't record an undo save
; boundary when it's appropriate (ZZ)
(defun evi-insert-file (filename)
  ; the insert will record a save record if appropriate
  (insert ?@)
  (delete-region (1- (point)) (point))
  ; now just erase the existence of the insert and delete
  (setq buffer-undo-list (cdr (cdr buffer-undo-list)))
  (insert-file-contents filename))

(defun ex-recover (exclam file-name)
  (if (= (length file-name) 0)
    (if (setq file-name (buffer-file-name))
      (and (not exclam) (buffer-modified-p)
	   (evi-error
	     "No write since last change (use :recover! to override)"))
      (evi-error "Buffer has no file associated with it")))
  (recover-file file-name)
  (auto-save-mode 1)
  (message "Auto save mode on")
  (evi))

(defun ex-set (settings)
  (if settings
    (ex-set-internal settings)
    (message "Well set!")))

(defun ex-set-internal (settings)
  (if settings
    (let* ((setting (car settings))
	   (name (car setting))
	   (value (cdr setting)))
      (if (integerp value)
	(progn (princ (evi-get-option name))
	       (princ " "))
	(evi-set-option name value))
      (ex-set-internal (cdr settings)))))

(defun ex-scan-settings ()
  (skip-chars-forward " \t")
  (let ((settings nil))
    (while (looking-at "[A-Za-z]")
      (let* ((default-value
	       (if (looking-at "no") (progn (forward-char 2) nil) t))
	     (option (let ((start (point)))
		       (skip-chars-forward "A-Za-z")
		       (buffer-substring start (point)))))
	(cond ((looking-at "=")
		(progn (forward-char 1)
		       (setq settings
			 (cons (cons option (ex-scan-word)) settings))))
	      ((looking-at "?")
		(progn (forward-char 1)
		       (setq settings
			 (cons (cons option ??) settings))))
	      (t
		(setq settings (cons (cons option default-value) settings)))))
      (skip-chars-forward " \t"))
    (if (looking-at "[^|\n]")
      (evi-error "Invalid setting%s"
		 (if settings (format " after `%s'" (car (car settings))) "")))
    settings))

(defun evi-get-option (option)
  (let* ((option-struct (evi-search-option-list evi-option-list option))
	 (type (car (cdr option-struct))))
    (if (eq type nil)
      (evi-error "invalid option `%s'" option)
      (let* ((long-name (car option-struct))
	     (value (eval (cdr (cdr option-struct)))))
	(cond
	  ((eq (cdr (cdr option-struct)) nil)
	    (evi-error "option `%s' not implemented" long-name))
	  ((eq type 'bool)
	    (if (eq value t) long-name (concat "no" long-name)))
	  ((eq type 'number)
	    (concat long-name "=" (int-to-string value)))
	  ((eq type 'string)
	    (concat long-name "=" value))
	  (t
	    (evi-error "invalid type `%s'" (prin1-to-string type))))))))

(defun evi-set-option (option value)
  (let* ((option-struct (evi-search-option-list evi-option-list option))
	 (type (car (cdr option-struct))))
    (cond
      ((eq type nil)
	(evi-error "Invalid option `%s'" option))
      ((eq (cdr (cdr option-struct)) nil)
	;; be gentle for now, just use message
	(message "Option `%s' not implemented" (car option-struct)))
      ((eq type 'bool)
        (if (not (or (eq value t) (eq value nil)))
	    (evi-error "Only %s or no%s allowed" option option)))
      ((eq type 'number)
        (if (or (eq value t) (eq value nil))
	    (evi-error "Use %s=<number> to set, or %s? to query" option option)
	    (setq value (string-to-int value))))
      ((eq type 'string)
        (if (or (eq value t) (eq value nil))
	    (evi-error
	      "Use %s=<string> to set, or %s? to query" option option)))
      (t
	(evi-error "Invalid type `%s'" (prin1-to-string type))))
    (if (cdr (cdr option-struct))
      (set (cdr (cdr option-struct)) value))
    (if (fboundp (cdr (cdr option-struct)))
      (funcall (cdr (cdr option-struct)) value))))

(defun evi-search-option-list (option-list option)
  (let ((option (evi-find option-struct option-list
		  (let ((option-strings (car option-struct)))
		    (if (evi-string-list-match option-strings option)
		      (cons (car option-strings) (cdr option-struct)))))))
    (or option '("" . nil))))

(defun evi-string-list-match (string-list string)
  (if string-list
    (if (string= string (car string-list))
	t
	(evi-string-list-match (cdr string-list) string))))

(defun ex-shell ()
  (let ((shell-mode-hook
	 (function
	   (lambda ()
	     (evi)
	     (setq evi-buffer-local-vi-map evi-shell-map)))))
    (shell)
    (evi-insert)))

(defun ex-source-file (file-name)
  (let ((ex-user-buffer (current-buffer)))
    (set-buffer ex-work-space)
    (let ((work-string (buffer-string))
	  (work-point (point)))
      (set-buffer ex-user-buffer)
      (evi-do-ex-command-file file-name)
      (setq ex-user-buffer (current-buffer))
      (set-buffer ex-work-space)
      (delete-region (point-min) (point-max))
      (insert work-string)
      (goto-char work-point)
      (set-buffer ex-user-buffer))))

(defvar ex-previous-substitute nil)

(defun ex-substitute (addresses pattern replacement global query)
  (ex-define-region addresses t nil)
  (if (= (length pattern) 0)
    (if ex-previous-re
      (setq pattern ex-previous-re)
      (evi-error "No previous regular expression"))
    (setq ex-previous-re pattern))
  (exchange-point-and-mark)
  (let ((case-fold-search evi-ignore-case)
	(next-line-mark (make-marker))
	(end-line-mark (make-marker))
	(large-region (> (- (mark) (point)) 5000)))
    (if large-region
      (message "running substitute command... "))
    (set-marker end-line-mark (mark))
    (while (< (point) end-line-mark)
      (if (re-search-forward pattern end-line-mark 1)
	(progn
	  (goto-char (match-beginning 0))
	  (save-excursion
	    (if global
	      (goto-char (match-end 0))
	      (forward-line))
	    (set-marker next-line-mark (point)))
	  (ex-replace-match query replacement)
	  (goto-char next-line-mark))))
    (if large-region
      (message "running substitute command... complete."))
    (setq ex-previous-substitute
	  (list addresses pattern replacement global query))
    (set-marker next-line-mark nil)
    (set-marker end-line-mark nil)))

(defun ex-substitute-again (addresses)
  (if ex-previous-substitute
    (apply 'ex-substitute addresses (cdr ex-previous-substitute))
    (evi-error "No previous substitution"))
  (setq ex-previous-substitute
	(append (list addresses) (cdr ex-previous-substitute))))

(defun evi-substitute-again ()
  (interactive)
  (if ex-previous-substitute
    (apply 'ex-substitute ex-previous-substitute)
    (evi-error "No previous substitution")))

(defun ex-replace-match (query replacement)
  (if (or (not query)
	  (let ((beginning (match-beginning 0))
		(end (match-end 0)))
	    (save-excursion
	      (goto-char beginning) (insert ?$)
	      (goto-char (1+ end)) (insert ?$))
	    (prog1
	      (y-or-n-p "replace? ")
	      (save-excursion
		(delete-region beginning (1+ beginning))
		(delete-region end (1+ end))
		(setq buffer-undo-list (nthcdr 4 buffer-undo-list))))))
    (progn (delete-region (match-beginning 0) (match-end 0))
	   (insert replacement))))

(defun ex-tag (tag)
  (if (= (length tag) 0)
    (if (null ex-tag)
      (evi-error "No previous tag specified"))
    (setq ex-tag tag))
  (find-tag ex-tag)
  (evi))

(defun ex-unmap (exclam key)
  (if exclam
    (evi-define-key '(input-map) key nil)
    (evi-define-key '(map) key nil)))

; ZZ exclam overrides readonly...
(defun ex-write (addresses exclam append file-arg)
  (let ((file-name (if (= (length file-arg) 0) (buffer-file-name) file-arg)))
    (save-excursion
      (ex-define-region addresses t t)
      (if (and (= (length file-arg) 0)
	       (= (mark) (point-min)) (= (point) (point-max)))
	(progn
	  ; force a write, even if not modified
	  (set-buffer-modified-p t)
	  (basic-save-buffer))
	(write-region (mark) (point) file-name append)))))

(defun ex-write-all-buffers (quietly)
  (save-some-buffers quietly))

(defun ex-write-kill ()
  (set-buffer-modified-p t)
  (basic-save-buffer)
  (ex-kill-buffer nil ""))

(defun ex-write-quit (discard)
  (set-buffer-modified-p t)
  (basic-save-buffer)
  (ex-quit discard))

(defun ex-write-all-and-quit (quietly)
  (save-some-buffers quietly t)
  (kill-emacs))

(defun ex-yank (addresses register-struct)
  (let ((evi-register-spec register-struct))
    (save-excursion
      (ex-define-region addresses t nil)
      (evi-copy-region-to-registers nil))))

(defun ex-shell-command (addresses shell-command)
  (if (null (car (car (car addresses))))
    (shell-command shell-command)
    (progn (ex-define-region addresses t nil)
	   (shell-command-on-region (mark) (point) shell-command t))))

(defun ex-shift-right (addresses)
  (ex-define-region addresses t nil)
  (indent-rigidly (mark) (point) evi-shift-width)
  (forward-line -1)
  (skip-chars-forward " \t"))

(defun ex-shift-left (addresses)
  (ex-define-region addresses t nil)
  (indent-rigidly (mark) (point) (- evi-shift-width))
  (forward-line -1)
  (skip-chars-forward " \t"))

(defun ex-null (addresses)
  (ex-define-region addresses t nil)
  (forward-line -1))

;; Initializing

(let* ((source)
       (message (catch 'abort
		  (or evi-supress-ex-startup
		      (progn
			(setq source "~/.exrc")
			(evi-do-ex-command-file "~/.exrc")
			(setq source "EXINIT")
			(let ((exinit (getenv "EXINIT")))
			  (if exinit
			    (evi-do-ex-command-string exinit)))
			(setq source ".exrc")
			(evi-do-ex-command-file ".exrc")))
		  (setq source "~/.exrc.evi")
		  (evi-do-ex-command-file "~/.exrc.evi")
		  (setq source "EVIINIT")
		  (let ((exinit (getenv "EVIINIT")))
		    (if exinit
		      (evi-do-ex-command-string exinit)))
		  (setq source ".exrc.evi")
		  (evi-do-ex-command-file ".exrc.evi")
		  nil)))
  (if message
    (progn
      (beep)
      (if (not (y-or-n-p (concat "Error in " source
			   (if (eq message t) "" (concat ": " message))
			   ". Continue? ")))
	(kill-emacs)))))

(if (file-readable-p "~/.evirc") (load-file "~/.evirc"))
(if (file-readable-p ".evirc") (load-file ".evirc"))

(if evi-meta-prefix-char
  (make-variable-buffer-local 'meta-prefix-char))
