; Newsgroups: gnu.emacs.sources
; Path: hal.com!decwrl!mips!sdd.hp.com!usc!wupost!m.cs.uiuc.edu!m.cs.uiuc.edu!liberte
; From: liberte@cs.uiuc.edu (Daniel LaLiberte)
; Subject: isearch-mode.el
; Organization: University of Illinois, Urbana-Champaign, Dept CS
; Date: Tue, 17 Mar 1992 16:36:59 GMT
; 
; Here is a new version of isearch-mode that fixes a couple small problems,
; gets rid of the recursive-edit, and lets you more easily replace the
; standard isearch.el package.  It also uses buffer-local variables so
; it might work better with epoch and emacs version 19 - I am not in
; a position to test this, so let me know how it goes.
; 
; Dan LaLiberte
; liberte@cs.uiuc.edu
; (Join the League for Programming Freedom: league@prep.ai.mit.edu)

;; Incremental search minor mode.
;; Copyright (C) 1992 Free Software Foundation, Inc.

;; LCD Archive Entry:
;; isearch-mode|Daniel LaLiberte|liberte@cs.uiuc.edu
;; |A minor mode replacement for isearch.el.
;; |92-03-14|1.0|~/modes/isearch-mode.el.Z|

;; This file is not yet part of GNU Emacs, but it is based almost
;; entirely on isearch.el which is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

;;====================================================================
;; Searching with isearch-mode.el should work just like isearch.el,
;; except it is done in a temporary minor mode that terminates when
;; you finish searching.

;; To use isearch-mode instead of the standard isearch.el, add the
;; following to your .emacs file.  The standard key bindings to
;; isearch-forward, etc, will then use isearch-mode instead of
;; isearch.

;;(fset 'isearch 'isearch-mode)
;;(autoload 'isearch-mode "isearch-mode")

;; The key bindings active within isearch-mode are defined below in
;; `isearch-mode-map' which is given bindings equivalent to
;; the default characters of isearch.el.  With isearch-mode, however,
;; you can bind multi-character keys and it should be easier to add
;; new commands.

;; Note to epoch and emacs version 19 users: isearch-mode should
;; work even if you switch windows with the mouse.  However, if
;; you isearch in a buffer that is also displayed in another window,
;; when you switch to that other window you will still be in
;; isearch mode but not necessarily in the right state for it to work.
;; So ... don't do it unless you are in an experimental mood.
;; You can also experiment with the window-local-variable routines
;; contained in this package but not yet used.
;; Also, I am not sure what happens when you return to an isearching
;; buffer; ideally, the echo area should redisplay the searching status.
;; A select-window-hook might be useful.

;;=========================================================================
;; The following, defined in loaddefs.el, are still used with isearch-mode.

;(defvar search-last-string ""
;  "Last string search for by a search command.
;This does not include direct calls to the primitive search functions,
;and does not include searches that are aborted.")

;(defvar search-last-regexp ""
;  "Last string searched for by a regexp search command.
;This does not include direct calls to the primitive search functions,
;and does not include searches that are aborted.")

;(defconst search-exit-option t
;  "Non-nil means random control characters terminate incremental search.")

;(defvar search-slow-window-lines 1
;  "*Number of lines in slow search display windows.")

;(defconst search-slow-speed 1200
;  "*Highest terminal speed at which to use \"slow\" style incremental search.
;This is the style where a one-line window is created to show the line
;that the search has reached.")


;;====================================================
;; Define isearch-mode keymap.

(defvar isearch-mode-map nil
  "Keymap for isearch-mode.")

(if isearch-mode-map
    nil
  (let ((i 0))
    ;; Printing chars extend the selection by default.
    (setq isearch-mode-map 
	  (make-vector 128 'isearch-other-char))
    ;; Non-printing chars by default suspend isearch mode transparently
    (while (< i ?\ )
      (aset isearch-mode-map i 'isearch-other-control-char)
      (setq i (1+ i)))

    ;; Some non-printing chars change the searching behavior.
    (define-key isearch-mode-map "\C-s" 'isearch-search-forward)
    (define-key isearch-mode-map "\C-r" 'isearch-search-backward)
    (define-key isearch-mode-map "\177" 'isearch-delete-char)
    (define-key isearch-mode-map "\e" 'isearch-exit)
    (define-key isearch-mode-map "\C-g" 'isearch-quit)
    
    (define-key isearch-mode-map "\C-q" 'isearch-quote-char)
    (define-key isearch-mode-map "\r" 'isearch-return-char)

    (define-key isearch-mode-map "\C-w" 'isearch-yank-word)
    (define-key isearch-mode-map "\C-y" 'isearch-yank-line)
    ))

;;========================================================
;; Internal variables declared globally for byte-compiler.
;; These are all made buffer-local during searching.

(defvar isearch-cmds nil
  "Stack of search status sets.")
(defvar isearch-string "")
(defvar isearch-message "")
(defvar isearch-success t)
(defvar isearch-forward nil)
(defvar isearch-other-end nil	
  "Start of last match if forward, end if backward.")
(defvar isearch-invalid-regexp nil)
(defvar isearch-wrapped nil)
(defvar isearch-barrier 0)

(defvar isearch-regexp nil)
(defvar isearch-adjusted nil)
(defvar isearch-slow-terminal-mode nil)
(defvar isearch-small-window nil
  "If t, using a small window.")
(defvar isearch-found-point nil
  "to restore point from a small window.")

(defvar isearch-found-start nil
  "This is the window-start value found by the search.")
(defvar isearch-opoint 0)
(defvar isearch-window-configuration nil
  "The window configuration active at the beginning of the search.")
(defvar isearch-old-local-map [])

;;==============================================================
;; Minor-mode-alist changes - kind of redundant with the
;; echo area, but if isearching in multiple windows, it can be useful.

(or (assq 'isearch-mode minor-mode-alist)
    (nconc minor-mode-alist
	   (list '(isearch-mode isearch-mode))))

(defvar isearch-mode nil)
(make-variable-buffer-local 'isearch-mode)


;;==================================================================
;; isearch-mode only sets up incremental search for the minor mode.
;; All the work is done by the isearch-mode commands.


(defun isearch-mode (forward &optional regexp)
  "Start isearch minor mode.  Called by isearch-forward, etc."
  ;; Make buffer-local variables for isearching.
  ;; We really need window-local variables.
  (mapcar 
   'make-local-variable
   '(isearch-forward 
     isearch-regexp isearch-string isearch-message
     isearch-cmds isearch-success isearch-wrapped
     isearch-barrier isearch-adjusted isearch-invalid-regexp
     isearch-slow-terminal-mode isearch-other-end isearch-small-window
     isearch-found-point isearch-found-start isearch-opoint 
     isearch-window-configuration isearch-old-local-map))

  ;; Initialize global vars.
  (setq isearch-forward forward
	isearch-regexp regexp
	isearch-string ""
	isearch-message ""
	isearch-cmds nil
	isearch-success t
	isearch-wrapped nil
	isearch-barrier (point)
	isearch-adjusted nil
	isearch-invalid-regexp nil
	isearch-slow-terminal-mode (and (<= (baud-rate) search-slow-speed)
					(> (window-height)
					   (* 4 search-slow-window-lines)))
	isearch-other-end nil
	isearch-small-window nil
	isearch-found-point nil

	isearch-found-start nil
	isearch-opoint (point)
	isearch-window-configuration (current-window-configuration)
	isearch-old-local-map (current-local-map)

;;	inhibit-quit t
	)
  (setq	isearch-mode " Isearch")  ;; forward? regexp?
  (set-buffer-modified-p (buffer-modified-p)) ; update modeline

  (isearch-push-state)

  (use-local-map isearch-mode-map)
  (isearch-update)
  )


;;====================================================
;; Some utilities.  Others below.

(defun isearch-update ()
  ;; Called after each command to update the display.  
  (or (>= unread-command-char 0)
      (progn
	(or (input-pending-p)
	    (isearch-message))
	(if (and isearch-slow-terminal-mode
		 (not (or isearch-small-window 
			  (pos-visible-in-window-p))))
	    (progn
	      (setq isearch-small-window t)
	      (setq isearch-found-point (point))
	      (move-to-window-line 0)
	      (let ((window-min-height 1))
		(split-window nil (if (< search-slow-window-lines 0)
				      (1+ (- search-slow-window-lines))
				    (- (window-height)
				       (1+ search-slow-window-lines)))))
	      (if (< search-slow-window-lines 0)
		  (progn (vertical-motion (- 1 search-slow-window-lines))
			 (set-window-start (next-window) (point))
			 (set-window-hscroll (next-window)
					     (window-hscroll))
			 (set-window-hscroll (selected-window) 0))
		(other-window 1))
	      (goto-char isearch-found-point))))))


(defun isearch-done ()
  ;; Called by all commands that terminate isearch-mode.
  (use-local-map isearch-old-local-map)
  (setq isearch-found-start (window-start (selected-window)))
  (setq isearch-found-point (point))
  (set-window-configuration isearch-window-configuration)

  (if (> (length isearch-string) 0)
      (if isearch-regexp
	  (setq search-last-regexp isearch-string)
	(setq search-last-string isearch-string)))
  ;; If there was movement, mark the starting position.
  ;; Maybe should test difference between and set mark iff > threshold.
  (if (/= (point) isearch-opoint)
      (push-mark isearch-opoint)
    (message ""))
  (if isearch-small-window
      (goto-char isearch-found-point)
    ;; Exiting the save-window-excursion clobbers this; restore it.
    (set-window-start (selected-window) isearch-found-start t))

  ;; Kill buffer-local variables for isearching
  (mapcar 
   'kill-local-variable
   '(isearch-forward 
     isearch-regexp isearch-string isearch-message
     isearch-cmds isearch-success isearch-wrapped
     isearch-barrier isearch-adjusted isearch-invalid-regexp
     isearch-slow-terminal-mode isearch-other-end isearch-small-window
     isearch-found-point isearch-found-start isearch-opoint 
     isearch-window-configuration isearch-old-local-map))

  (setq isearch-mode nil)
  (set-buffer-modified-p (buffer-modified-p))
  )


;;====================================================
;; Commands for inside of the isearch minor mode.

(defun isearch-exit ()
  "Exit search normally.
Except, if first thing typed, it means do nonincremental search."
  (interactive)
  (if (= 0 (length isearch-string))
      (nonincremental-search isearch-forward isearch-regexp))
  (isearch-done))


(defun isearch-quit ()
  "The user tried to quit incremental search mode."
  (interactive)
  (ding)
  (discard-input)
  (if isearch-success
      ;; If search is successful, move back to starting point
      ;; and really do quit.
      (progn (goto-char isearch-opoint)
	     (isearch-done))  ; exit and quit
    ;; If search is failing, rub out until it is once more
    ;;  successful.
    (while (not isearch-success) (isearch-pop-state))
    (isearch-update)))


(defun isearch-search-repeat (direction)
  ;; (or (eq char search-repeat-char)
  ;;     (eq char search-reverse-char))
  (if (eq isearch-forward (eq direction 'forward))
      ;; C-s in forward or C-r in reverse.
      (if (equal isearch-string "")
	  ;; If search string is empty, use last one.
	  (setq isearch-string
		(if isearch-regexp
		    search-last-regexp search-last-string)
		isearch-message
		(mapconcat 'text-char-description
			   isearch-string ""))
	;; If already have what to search for, repeat it.
	(or isearch-success
	    (progn 
	      (goto-char (if isearch-forward (point-min) (point-max)))
	      (setq isearch-wrapped t))))
    ;; C-s in reverse or C-r in forward, change direction.
    (setq isearch-forward (not isearch-forward)))

  (setq isearch-barrier (point)) ; For subsequent \| if regexp.
  (setq isearch-success t)
  (or (equal isearch-string "")
      (isearch-search))
  (isearch-push-state)
  (isearch-update))

(defun isearch-search-forward ()
  "Repeat incremental search forwards."
  (interactive)
  (isearch-search-repeat 'forward))

(defun isearch-search-backward ()
  "Repeat incremental search backwards."
  (interactive)
  (isearch-search-repeat 'backward))


(defun isearch-delete-char ()
  "Discard last input item and move point back.  
If no previous match was done, just beep."
  (interactive)
  (if (null (cdr isearch-cmds))
      (ding)
    (isearch-pop-state))
  (isearch-update))



(defun isearch-yank (chunk)
  ;; (or (eq char search-yank-word-char)
  ;;     (eq char search-yank-line-char))
  ;; ^W means gobble next word from buffer.
  ;; ^Y means gobble rest of line from buffer.
  (let ((word (save-excursion
		(and (not isearch-forward) isearch-other-end
		     (goto-char isearch-other-end))
		(buffer-substring
		 (point)
		 (save-excursion
		   (cond
		    ((eq chunk 'word)
		     (forward-word 1))
		    ((eq chunk 'line)
		     (end-of-line)))
		   (point))))))
    (setq isearch-string (concat isearch-string word)
	  isearch-message
	  (concat isearch-message
		  (mapconcat 'text-char-description
			     word ""))))
  (isearch-after-other-char)
  (isearch-update))


(defun isearch-yank-word ()
  "Pull next word from buffer into search string."
  (interactive)
  (isearch-yank 'word))

(defun isearch-yank-line ()
  "Pull rest of line from buffer into search string."
  (interactive)
  (isearch-yank 'line))


(defun isearch-after-other-char ()
  ;; Do the search and update the display.
  (if (and (not isearch-success)
	   ;; unsuccessful regexp search may become
	   ;;  successful by addition of characters which
	   ;;  make isearch-string valid
	   (not isearch-regexp))
      nil
    ;; If a regexp search may have been made more
    ;; liberal, retreat the search start.
    ;; Go back to place last successful search started
    ;; or to the last ^S/^R (barrier), whichever is nearer.
    (and isearch-regexp isearch-success isearch-cmds
	 (cond ((memq char '(?* ??))
		(setq isearch-adjusted t)
		(let ((cs (nth (if isearch-forward
				   5	; isearch-other-end
				 2)	; saved (point)
			       (car (cdr isearch-cmds)))))
		  ;; (car isearch-cmds) is after last search;
		  ;; (car (cdr isearch-cmds)) is from before it.
		  (setq cs (or cs isearch-barrier))
		  (goto-char
		   (if isearch-forward
		       (max cs isearch-barrier)
		     (min cs isearch-barrier)))))
	       ((eq char ?\|)
		(setq isearch-adjusted t)
		(goto-char isearch-barrier))))
    ;; In reverse regexp search, adding a character at
    ;; the end may cause zero or many more chars to be
    ;; matched, in the string following point.
    ;; Allow all those possibiities without moving point as
    ;; long as the match does not extend past search origin.
    (if (and isearch-regexp (not isearch-forward) (not isearch-adjusted)
	     (condition-case ()
		 (looking-at isearch-string)
	       (error nil))
	     (<= (match-end 0) 
		 (min isearch-opoint isearch-barrier)))
	(setq isearch-success t isearch-invalid-regexp nil
	      isearch-other-end (match-end 0))
      ;; Not regexp, not reverse, or no match at point.
      (if (and isearch-other-end (not isearch-adjusted))
	  (goto-char (if isearch-forward isearch-other-end
		       (min isearch-opoint 
			    isearch-barrier 
			    (1+ isearch-other-end)))))
      (isearch-search)
      ))
  (isearch-push-state)
  (isearch-update))


(defun isearch-other-control-char ()
  "Any other control char => unread it and exit the search normally.
But only if search-exit-option is non-nil."
  (interactive)
  (let ((char last-command-char))
    (if search-exit-option
	(progn
	  (setq unread-command-char char)
	  (isearch-done))
      ;; otherwise
      (isearch-after-other-char))))


(defun isearch-quote-char ()
  "Quote special characters for incremental search."
  (interactive)
  (let ((char (read-quoted-char
	       (isearch-message t))))
    (isearch-process-char char)
    ))

(defun isearch-return-char ()
  "Convert return into newline for incremental search."
  (interactive)
  (isearch-process-char ?\n))


(defun isearch-other-char ()
  "Any other character => add it to the search string and search."
  (interactive)
  (isearch-process-char last-command-char))


(defun isearch-process-char (char)
  (setq isearch-string (concat isearch-string
			       (char-to-string char))
	isearch-message 
	(concat isearch-message
		(text-char-description char)))
  (isearch-after-other-char))



;;=============================================================
;; Window-local variables
;; (not used yet)

(defvar window-local-variable-alist nil
  "An alist of windows associated with window local variables and values.
The cdr of each item is another alist of variables and values.")

(defvar last-local-window nil)
(defvar last-window-local-vars nil)

(defun kill-window-local-variables ()
  "Remove the old variable list, if any."
  (setq window-local-variable-alist
	(delq window-local-variable-alist
	      (assq (selected-window)
		    window-local-variable-alist))))

;; Assume that window-local variables are not buffer-local
;; so we can delay storing until absolutely necessary.

(defun store-window-local-variables (&rest vars-and-vals)
  "Store the window local variables for selected window."
  (setq last-local-window (selected-window))
  (setq last-window-local-vars vars-and-vals))


(defun fetch-window-local-variables ()
 "Fetch the window local variables for selected window.
Does nothing if the last store was for the same window."
  (if (not (eq (selected-window) last-local-window))
      (progn
	;; First store the previous values.
	(setq window-local-variable-alist
	      (cons (cons last-local-window
			  last-window-local-vars)
		    (delq window-local-variable-alist
			  (assq last-local-window
				window-local-variable-alist))))
	;; Now fetch the values for the selected-window.
	(setq last-local-window (selected-window))
	(setq last-window-local-vars 
		(cdr (assq last-local-window window-local-variable-alist)))
	(let ((vars-and-vals last-window-local-vars))
	  (while vars-and-vals
	    (set (car vars-and-vals) (car (cdr (vars-and-vals))))
	    (setq vars-and-vals (cdr (cdr vars-and-vals))))))))
		    

;;==============================================================
;; The search status stack, and isearch window-local variables.

(defun isearch-top-state ()
;;  (fetch-window-local-variables)
  (let ((cmd (car isearch-cmds)))
    (setq isearch-string (car cmd)
	  isearch-message (car (cdr cmd))
	  isearch-success (nth 3 cmd)
	  isearch-forward (nth 4 cmd)
	  isearch-other-end (nth 5 cmd)
	  isearch-invalid-regexp (nth 6 cmd)
	  isearch-wrapped (nth 7 cmd)
	  isearch-barrier (nth 8 cmd))
    (goto-char (car (cdr (cdr cmd))))))

(defun isearch-pop-state ()
;;  (fetch-window-local-variables)
  (setq isearch-cmds (cdr isearch-cmds))
  (isearch-top-state)
  )

(defun isearch-push-state ()
  (setq isearch-cmds 
	(cons (list isearch-string isearch-message (point)
		    isearch-success isearch-forward isearch-other-end 
		    isearch-invalid-regexp isearch-wrapped isearch-barrier)
	      isearch-cmds)))

(defun isearch-store-variables ()
  (store-window-local-variables 
   'isearch-cmds isearch-cmds
   'isearch-regexp isearch-regexp
   'isearch-adjusted isearch-adjusted
   'isearch-slow-terminal-mode isearch-slow-terminal-mode
   'isearch-small-window isearch-small-window
   'isearch-found-point isearch-found-point
   'isearch-found-start isearch-found-start
   'isearch-opoint isearch-opoint
   'isearch-window-configuration isearch-window-configuration
   'isearch-old-local-map isearch-old-local-map
   ))

;;=======================================================================
;; More utilities.

(defun isearch-message (&optional c-q-hack ellipsis)
  ;; If about to search, and previous search regexp was invalid,
  ;; check that it still is.  If it is valid now,
  ;; let the message we display while searching say that it is valid.
  (and isearch-invalid-regexp ellipsis
       (condition-case ()
	   (progn (re-search-forward isearch-string (point) t)
		  (setq isearch-invalid-regexp nil))
	 (error nil)))
  ;; If currently failing, display no ellipsis.
  (or isearch-success (setq ellipsis nil))
  (let ((m (concat (if isearch-success "" "failing ")
		   (if isearch-wrapped "wrapped ")
		   (if isearch-regexp "regexp " "")
		   "I-search"
		   (if isearch-forward ": " " backward: ")
		   isearch-message
		   (if c-q-hack "^Q" "")
		   (if isearch-invalid-regexp
		       (concat " [" isearch-invalid-regexp "]")
		     ""))))
    (aset m 0 (upcase (aref m 0)))
    (let ((cursor-in-echo-area ellipsis))
      (if c-q-hack m (message "%s" m)))))


(defun isearch-search ()
  ;; Do the search with the current search string.
  (isearch-message nil t)
  (condition-case lossage
      (let ((inhibit-quit nil))
	(if isearch-regexp (setq isearch-invalid-regexp nil))
	(setq isearch-success
	      (funcall
	       (if isearch-regexp
		   (if isearch-forward 're-search-forward 're-search-backward)
		 (if isearch-forward 'search-forward 'search-backward))
	       isearch-string nil t))
	(if isearch-success
	    (setq isearch-other-end
		  (if isearch-forward (match-beginning 0) (match-end 0)))))

    (quit (setq unread-command-char ?\C-g)
	  (setq isearch-success nil))

    (isearch-invalid-regexp 
     (setq isearch-invalid-regexp (car (cdr lossage)))
     (if (string-match 
	  "\\`Premature \\|\\`Unmatched \\|\\`Invalid "
	  isearch-invalid-regexp)
	 (setq isearch-invalid-regexp "incomplete input"))))

  (if isearch-success
      nil
    ;; Ding if failed this time after succeeding last time.
    (and (nth 3 (car isearch-cmds))
	 (ding))
    (goto-char (nth 2 (car isearch-cmds)))))


;;=================================================
;; This is called from incremental-search
;; if the first input character is the exit character.

;; We store the search string in `isearch-string'
;; which has been bound already by `isearch-search'
;; so that, when we exit, it is copied into `search-last-string'.

(defun nonincremental-search (forward regexp)
  (setq isearch-forward forward
	isearch-regexp regexp)
  (let (message char function string 
	inhibit-quit
	(cursor-in-echo-area t))
    ;; Prompt assuming not word search,
    (setq message (if isearch-regexp 
		      (if isearch-forward "Regexp search: "
			"Regexp search backward: ")
		    (if isearch-forward "Search: " "Search backward: ")))
    (message "%s" message)
    ;; Read 1 char and switch to word search if it is ^W.
    (setq char (read-char))
    (if (eq char search-yank-word-char)
	(setq message (if isearch-forward "Word search: " 
			"Word search backward: "))
      ;; Otherwise let that 1 char be part of the search string.
      (setq unread-command-char char))
    (setq function
	  (if (eq char search-yank-word-char)
	      (if isearch-forward 'word-search-forward 'word-search-backward)
	    (if isearch-regexp
		(if isearch-forward 're-search-forward 're-search-backward)
	      (if isearch-forward 'search-forward 'search-backward))))
    ;; Read the search string with corrected prompt.
    (setq string (read-string message))
    ;; Empty means use default.
    (if (= 0 (length string))
	(setq string search-last-string)
      ;; Set last search string now so it is set even if we fail.
      (setq search-last-string string))
    ;; Since we used the minibuffer, we should be available for redo.
    (setq command-history (cons (list function string) command-history))
    ;; Go ahead and search.
    (funcall function string)))
