/* Generic scrollbar implementation.
   #### Actually, this is not really generic.  The generic and X-specific
   parts need to be split up, similar to what's done for toolbars and
   menubars.
   Copyright (C) 1994, 1995 Board of Trustees, University of Illinois.
   Copyright (C) 1995 Amdahl Corporation.
   Copyright (C) 1995 Sun Microsystems.
   Copyright (C) 1995 Darrell Kindred <dkindred+@cmu.edu>.

This file is part of XEmacs.

XEmacs is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

XEmacs is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with XEmacs; see the file COPYING.  If not, write to the Free
Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Synched up with: Not in FSF. */

/* This file has been Mule-ized. */

#include <config.h>
#include "lisp.h"

#include "device-x.h"
#include "frame-x.h"
#include "glyphs-x.h" /* for kludge in Fx_set_scrollbar_pointer */
#include "EmacsFrame.h"
#include "EmacsManager.h"
#include "lwlib.h"

#include "buffer.h"
#include "commands.h"
#include "scrollbar.h"
#include "window.h"

#define DEFAULT_SCROLLBAR_WIDTH 15
#define DEFAULT_SCROLLBAR_HEIGHT 15

/* Used to prevent changing the size of the thumb while drag scrolling,
   under Motif.  This is necessary because the Motif scrollbar is
   incredibly stupid about updating the thumb and causes lots of flicker
   if it is done too often.
 */
static int inhibit_thumb_size_change;

extern Lisp_Object Qscrollbar_pointer;

Lisp_Object Qx_init_scrollbar_from_resources;

Lisp_Object Qscrollbar_line_up;
Lisp_Object Qscrollbar_line_down;
Lisp_Object Qscrollbar_page_up;
Lisp_Object Qscrollbar_page_down;
Lisp_Object Qscrollbar_to_top;
Lisp_Object Qscrollbar_to_bottom;
Lisp_Object Qscrollbar_vertical_drag;

Lisp_Object Qscrollbar_char_left;
Lisp_Object Qscrollbar_char_right;
Lisp_Object Qscrollbar_page_left;
Lisp_Object Qscrollbar_page_right;
Lisp_Object Qscrollbar_to_left;
Lisp_Object Qscrollbar_to_right;
Lisp_Object Qscrollbar_horizontal_drag;

/* Width of the scrollbar. */
Lisp_Object Vscrollbar_width;

/* Height of the scrollbar. */
Lisp_Object Vscrollbar_height;


/*
 * If w->sb_point is on the top line then return w->sb_point else
 * return w->start.  If flag, then return beginning point of line
 * which w->sb_point lies on.
 */
static Bufpos
scrollbar_point (struct window *w, int flag)
{
  Bufpos start_pos, end_pos, sb_pos;
  Lisp_Object buf;
  struct buffer *b;

  if (NILP (w->buffer)) /* non-leaf window */
    return 0;

  start_pos = marker_position (w->start[CURRENT_DISP]);
  sb_pos = marker_position (w->sb_point);

  if (!flag && sb_pos < start_pos)
    return start_pos;

  buf = get_buffer (w->buffer, 0);
  if (!NILP (buf))
    b = XBUFFER (buf);
  else
    return start_pos;

  if (flag)
    end_pos = find_next_newline_no_quit (b, sb_pos, -1);
  else
    end_pos = find_next_newline_no_quit (b, start_pos, 1);

  if (flag)
    return end_pos;
  else if (sb_pos > end_pos)
    return start_pos;
  else
    return sb_pos;
}


static void
update_one_widget_scrollbar_pointer (struct frame *f, Widget wid)
{
  Lisp_Object cursor = f->scrollbar_pointer;

  if (CURSORP (cursor)) /* #### kludge */
    {
      XDefineCursor (XtDisplay (wid), XtWindow (wid), 
		     XCURSOR (cursor)->cursor);
      XSync (XtDisplay (wid), False);
    }
}

static void
free_scrollbar_instance (struct scrollbar_instance *instance)
{
  if (!instance)
    return;

  if (instance->scrollbar_name)
    xfree (instance->scrollbar_name);

  if (instance->scrollbar_widget)
    {
      if (XtIsManaged (instance->scrollbar_widget))
	XtUnmanageChild (instance->scrollbar_widget);

      lw_destroy_all_widgets (instance->scrollbar_id);
    }
}

static void
free_window_scrollbars (struct window_mirror *mir)
{
  free_scrollbar_instance (mir->scrollbar_vertical_instance);
  mir->scrollbar_vertical_instance = 0;
  free_scrollbar_instance (mir->scrollbar_horizontal_instance);
  mir->scrollbar_horizontal_instance = 0;
}

enum scrollbar_loop
{
  FIND_SCROLLBAR_WINDOW_MIRROR,		/* Arg is id */
  UPDATE_FRAME_SCROLLBARS,		/* No arg */
  FREE_FRAME_SCROLLBARS,		/* No arg */
  X_SET_SCROLLBAR_POINTER,		/* No arg */
  X_WINDOW_IS_SCROLLBAR			/* Arg is x_win */
};

static struct window_mirror *
scrollbar_loop (enum scrollbar_loop type, Lisp_Object window,
		struct window_mirror *mir,
		LWLIB_ID id, Window x_win)
{
  struct window_mirror *retval = NULL;

  while (mir)
    {
      struct scrollbar_instance *vinstance = mir->scrollbar_vertical_instance;
      struct scrollbar_instance *hinstance =
	mir->scrollbar_horizontal_instance;
      struct frame *f;

      assert (!NILP (window));
      f = XFRAME (XWINDOW (window)->frame);

      if (mir->vchild)
	{
	  retval = scrollbar_loop (type, XWINDOW (window)->vchild,
				   mir->vchild, id, x_win);
	}
      else if (mir->hchild)
	{
	  retval = scrollbar_loop (type, XWINDOW (window)->hchild,
				   mir->hchild, id, x_win);
	}

      if (retval != NULL)
	return retval;

      if (hinstance || vinstance)
	{
	  switch (type)
	    {
	    case FIND_SCROLLBAR_WINDOW_MIRROR:
	      if ((vinstance && vinstance->scrollbar_id == id)
		  || (hinstance && hinstance->scrollbar_id == id))
		{
		  return mir;
		}
	      break;
	    case UPDATE_FRAME_SCROLLBARS:
	      if (!mir->vchild && !mir->hchild)
		x_update_window_scrollbars (XWINDOW (window), mir, 1, 0);
	      break;
	    case FREE_FRAME_SCROLLBARS:
	      free_window_scrollbars (mir);
	      break;
	    case X_SET_SCROLLBAR_POINTER:
	      if (!mir->vchild && !mir->hchild)
		{
		  int loop;

		  for (loop = 0; loop < 2; loop++)
		    {
		      Widget widget;

		      if (loop)
			widget = vinstance->scrollbar_widget;
		      else
			widget = hinstance->scrollbar_widget;

		      if (widget && XtIsManaged (widget))
			{
			  update_one_widget_scrollbar_pointer (f, widget);
			}
		    }
		}
	      break;
	    case X_WINDOW_IS_SCROLLBAR:
	      if (!mir->vchild && !mir->hchild)
		{
		  int loop;

		  for (loop = 0; loop < 2; loop++)
		    {
		      Widget widget;

		      if (loop)
			widget = vinstance->scrollbar_widget;
		      else
			widget = hinstance->scrollbar_widget;

		      if (widget && XtIsManaged (widget))
			{
			  if (XtWindow (widget) == x_win)
			    return (struct window_mirror *) 1;
			}
		    }
		}
	      break;
	    default:
	      abort ();
	    }
	}

      mir = mir->next;
      window = XWINDOW (window)->next;
    }

  return NULL;
}
		
static struct window_mirror *
find_scrollbar_window_mirror (struct frame *f, LWLIB_ID id)
{
  if (f->mirror_dirty)
    update_frame_window_mirror (f);
  return scrollbar_loop (FIND_SCROLLBAR_WINDOW_MIRROR, f->root_window,
			 f->root_mirror, id, (Window) NULL);
}

/*
 * This is the only callback provided for vertical scrollbars.  It
 * should be able to handle all of the scrollbar events in
 * scroll_action (see lwlib.h).  The client data will be of type
 * scroll_event (see lwlib.h). */
static void
update_vertical_scrollbar_callback (Widget widget, LWLIB_ID id,
				    XtPointer client_data)
{
  /* This function can GC */
  scroll_event *data = (scroll_event *) client_data;
  struct device *d = get_device_from_display (XtDisplay (widget));
  struct frame *f = x_any_window_to_frame (d, XtWindow (widget));
  Lisp_Object win;
  struct scrollbar_instance *instance;
  struct window_mirror *mirror;

  if (!f)
    return;

  mirror = find_scrollbar_window_mirror (f, id);
  win = real_window (mirror, 1);

  if (NILP (win))
    return;
  instance = mirror->scrollbar_vertical_instance;

  /* It seems that this is necessary whenever signal_special_Xt_user_event()
     is called.  #### Why??? */
  DEVICE_X_MOUSE_TIMESTAMP (d) = DEVICE_X_GLOBAL_MOUSE_TIMESTAMP (d);

  switch (data->action)
    {
    case SCROLLBAR_LINE_UP:
      signal_special_Xt_user_event (Qscrollbar_line_up, win);
      break;

    case SCROLLBAR_LINE_DOWN:
      signal_special_Xt_user_event (Qscrollbar_line_down, win);
      break;

      /* The Athena scrollbar paging behavior is that of xterms.
         Depending on where you click the size of the page varies.
         Motif always does a standard Emacs page. */
    case SCROLLBAR_PAGE_UP:
#if !defined (LWLIB_SCROLLBARS_MOTIF) && !defined (LWLIB_SCROLLBARS_LUCID)
      {
	double tmp = ((double) data->slider_value /
		      (double) instance->data.scrollbar_height);
	double line = tmp *
	  (double) window_displayed_height (XWINDOW (win));
    
	if (line > -1.0)
	  line = -1.0;
	signal_special_Xt_user_event (Qscrollbar_page_up,
				      Fcons (win, make_number ((int) line)));
      }
#else
      signal_special_Xt_user_event (Qscrollbar_page_up, Fcons (win, Qnil));
#endif
      break;

    case SCROLLBAR_PAGE_DOWN:
#if !defined (LWLIB_SCROLLBARS_MOTIF) && !defined (LWLIB_SCROLLBARS_LUCID)
      {
	double tmp = ((double) data->slider_value /
		      (double) instance->data.scrollbar_height);
	double line = tmp *
	  (double) window_displayed_height (XWINDOW (win));

	if (instance->data.maximum >
	    (instance->data.slider_size + instance->data.slider_position))
	  {
	    if (line < 1.0)
	      line = 1.0;
	    signal_special_Xt_user_event (Qscrollbar_page_down,
					  Fcons (win,
						 make_number ((int) line)));
	  }
      }
#else
      signal_special_Xt_user_event (Qscrollbar_page_down, Fcons (win, Qnil));
#endif
      break;

    case SCROLLBAR_TOP:
      signal_special_Xt_user_event (Qscrollbar_to_top, win);
      break;

    case SCROLLBAR_BOTTOM:
      signal_special_Xt_user_event (Qscrollbar_to_bottom, win);
      break;


    case SCROLLBAR_CHANGE:
      inhibit_thumb_size_change = 0;
#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
      instance->vdrag_orig_value = data->slider_value;
      instance->vdrag_orig_window_start = XINT (Fwindow_start (win));
#endif
      break;

    case SCROLLBAR_DRAG:
      {
	int value;

	inhibit_thumb_size_change = 1;

#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
	/* Doing drags with Motif-like scrollbars is a mess, since we
	   want to avoid having the window position jump when you
	   first grab the scrollbar, but we also want to ensure that
	   you can scroll all the way to the top or bottom of the
	   buffer.  This can all be replaced with something sane when
	   we get line-based scrolling. */
	
	if (instance->vdrag_orig_value < 0) 
	  {
	    instance->vdrag_orig_value = data->slider_value;
	    instance->vdrag_orig_window_start = XINT (Fwindow_start (win));
	  }

	/* Could replace this piecewise linear scrolling with a
	   quadratic through the three points, but I'm not sure that
	   would feel any nicer in practice. */
	if (data->slider_value < instance->vdrag_orig_value) 
	  {
	    /* We've dragged up; slide linearly from original position to
	       window-start=data.minimum, slider-value=data.minimum. */

	    if (instance->vdrag_orig_value <= instance->data.minimum) 
	      {
		/* shouldn't get here, but just in case */
		value = instance->data.minimum;
	      }
	    else
	      {
		value = (instance->data.minimum
			 + (((double) (instance->vdrag_orig_window_start 
				       - instance->data.minimum)
			     * (data->slider_value - instance->data.minimum)) 
			    / (instance->vdrag_orig_value
			       - instance->data.minimum)));
	      }
	  }
	else 
	  {
	    /* We've dragged down; slide linearly from original position to
	       window-start=data.maximum, slider-value=data.maximum. */

	    if (instance->vdrag_orig_value 
		>= instance->data.maximum - instance->data.slider_size)
	      {
		/* avoid divide by zero */
		value = instance->vdrag_orig_window_start;
	      } 
	    else 
	      {
		value = (instance->vdrag_orig_window_start
			 + (((double) (instance->data.maximum
				       - instance->vdrag_orig_window_start)
			     * (data->slider_value 
				- instance->vdrag_orig_value)) 
			    / (instance->data.maximum
			       - instance->data.slider_size
			       - instance->vdrag_orig_value)));
	      }
	  }
#else
	value = data->slider_value;
#endif

	if (value >= instance->data.maximum)
	  value = instance->data.maximum - 1;
	if (value < instance->data.minimum)
	  value = instance->data.minimum;

	signal_special_Xt_user_event (Qscrollbar_vertical_drag,
				      Fcons (win, make_number (value)));
      }
      break;

    }
}

/*
 * This is the only callback provided for horizontal scrollbars.  It
 * should be able to handle all of the scrollbar events in
 * scroll_action (see lwlib.h).  The client data will be of type
 * scroll_event (see lwlib.h). */
static void
update_horizontal_scrollbar_callback (Widget widget, LWLIB_ID id,
				      XtPointer client_data)
{
  scroll_event *data = (scroll_event *) client_data;
  struct device *d = get_device_from_display (XtDisplay (widget));
  struct frame *f = x_any_window_to_frame (d, XtWindow (widget));
  Lisp_Object win;
  struct window_mirror *mirror;

  if (!f)
    return;

  mirror = find_scrollbar_window_mirror (f, id);
  win = real_window (mirror, 1);

  if (NILP (win))
    return;

  /* It seems that this is necessary whenever signal_special_Xt_user_event()
     is called.  #### Why??? */
  DEVICE_X_MOUSE_TIMESTAMP (d) = DEVICE_X_GLOBAL_MOUSE_TIMESTAMP (d);

  switch (data->action)
    {
    case SCROLLBAR_LINE_UP:
      signal_special_Xt_user_event (Qscrollbar_char_left, win);
      break;
    case SCROLLBAR_LINE_DOWN:
      signal_special_Xt_user_event (Qscrollbar_char_right, win);
      break;
    case SCROLLBAR_PAGE_UP:
      signal_special_Xt_user_event (Qscrollbar_page_left, win);
      break;
    case SCROLLBAR_PAGE_DOWN:
      signal_special_Xt_user_event (Qscrollbar_page_right, win);
      break;
    case SCROLLBAR_TOP:
      signal_special_Xt_user_event (Qscrollbar_to_left, win);
      break;
    case SCROLLBAR_BOTTOM:
      signal_special_Xt_user_event (Qscrollbar_to_right, win);
      break;
    case SCROLLBAR_CHANGE:
      inhibit_thumb_size_change = 0;
      break;
    case SCROLLBAR_DRAG:
      inhibit_thumb_size_change = 1;
      /* #### Fix the damn toolkit code so they all work the same way.
         Lucid is the one mostly wrong.*/
#if defined (LWLIB_SCROLLBARS_LUCID)
      signal_special_Xt_user_event (Qscrollbar_horizontal_drag,
				    (Fcons
				     (win, make_number (data->slider_value))));
#else
      signal_special_Xt_user_event (Qscrollbar_horizontal_drag,
				    (Fcons
				     (win,
				      make_number (data->slider_value - 1))));
#endif
      break;
    default:
      break;
    }
}

extern LWLIB_ID new_lwlib_id (void);

/*
 * Create a new scrollbar for FRAME.  Scrollbars are only destroyed
 * when a frame is deleted.
 */
static struct scrollbar_instance *
create_scrollbar_instance (struct frame *f, int vertical)
{
  char buffer[32];
  struct scrollbar_instance *instance =
    (struct scrollbar_instance *) xmalloc (sizeof (*instance));
  memset (instance, 0, sizeof (*instance));

  instance->scrollbar_id = new_lwlib_id ();
  sprintf (buffer, "scrollbar_%d", instance->scrollbar_id);
  instance->scrollbar_name = xstrdup (buffer);
#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
  instance->vdrag_orig_value = -1;
#endif

  if (vertical)
    {
      instance->scrollbar_widget =
	lw_create_widget ("vertical-scrollbar", instance->scrollbar_name,
			  instance->scrollbar_id,
			  NULL, FRAME_X_CONTAINER_WIDGET (f), 0,
			  update_vertical_scrollbar_callback, NULL, NULL);
    }
  else
    {
      instance->scrollbar_widget =
	lw_create_widget ("horizontal-scrollbar", instance->scrollbar_name,
			  instance->scrollbar_id,
			  NULL, FRAME_X_CONTAINER_WIDGET (f), 0,
			  update_horizontal_scrollbar_callback, NULL, NULL);
    }

  return instance;
}

#define GET_SCROLLBAR_INSTANCE_INTERNAL(cache)				\
  do {									\
    if (FRAME_X_SB_##cache (f))						\
      {									\
        struct scrollbar_instance *retval = FRAME_X_SB_##cache (f);	\
        FRAME_X_SB_##cache (f) = FRAME_X_SB_##cache (f)->next;		\
        retval->next = NULL;						\
        return retval;							\
      }									\
  } while (0)

static struct scrollbar_instance *
get_scrollbar_instance (struct frame *f, int vertical)
{
  /* Check if there are any available scrollbars already in existence. */
  if (vertical)
    GET_SCROLLBAR_INSTANCE_INTERNAL (VCACHE);
  else
    GET_SCROLLBAR_INSTANCE_INTERNAL (HCACHE);

  return create_scrollbar_instance (f, vertical);
}
#undef GET_SCROLLBAR_INSTANCE_INTERNAL

#define RELEASE_SCROLLBAR_INSTANCE_INTERNAL(cache)			\
  do {									\
    if (!FRAME_X_SB_##cache (f))					\
      {									\
	instance->next = NULL;						\
	FRAME_X_SB_##cache (f) = instance;				\
      }									\
    else								\
      {									\
	instance->next = FRAME_X_SB_##cache (f);			\
	FRAME_X_SB_##cache (f) = instance;				\
      }									\
  } while (0)

static void
release_scrollbar_instance (struct frame *f, int vertical,
			    struct scrollbar_instance *instance)
{
  if (vertical)
    RELEASE_SCROLLBAR_INSTANCE_INTERNAL (VCACHE);
  else
    RELEASE_SCROLLBAR_INSTANCE_INTERNAL (HCACHE);
}
#undef RELEASE_SCROLLBAR_INSTANCE_INTERNAL

/* We had to unmanage the instance here since this is called outside
   of the normal patch which would lead to that happening. */
void
x_release_window_scrollbars (struct window_mirror *mir)
{
  if (mir->scrollbar_vertical_instance)
    {
      release_scrollbar_instance (mir->frame, 1,
				  mir->scrollbar_vertical_instance);
      if (XtIsManaged (mir->scrollbar_vertical_instance->scrollbar_widget))
 	XtUnmanageChild (mir->scrollbar_vertical_instance->scrollbar_widget);
    }
  mir->scrollbar_vertical_instance = 0;
 
  if (mir->scrollbar_horizontal_instance)
    {
      release_scrollbar_instance (mir->frame, 0,
				  mir->scrollbar_horizontal_instance);
      if (XtIsManaged (mir->scrollbar_horizontal_instance->scrollbar_widget))
 	XtUnmanageChild (mir->scrollbar_horizontal_instance->scrollbar_widget);
    }
  mir->scrollbar_horizontal_instance = 0;
}

/*
 * Create a widget value structure for passing down to lwlib so that
 * it can update the scrollbar widgets.
 */
static widget_value *
scrollbar_instance_to_widget_value (struct scrollbar_instance *instance)
{
  widget_value *wv;

  wv = xmalloc_widget_value ();
  /* #### maybe should add malloc_scrollbar_values to resource these? */
  wv->scrollbar_data = (scrollbar_values *)
    xmalloc (sizeof (scrollbar_values));

  wv->name = instance->scrollbar_name;
  wv->value = 0;
  wv->key = 0;
  wv->enabled = instance->scrollbar_is_active;
  wv->selected = 0;
  wv->call_data = NULL;

  *wv->scrollbar_data = instance->data;

  wv->next = NULL;

  return wv;
}

#define UPDATE_DATA_FIELD(field, value) \
  if (instance->field != value) {\
    instance->field = value;\
    instance->scrollbar_instance_changed = 1;\
  }\

/*
 * Update a window's horizontal or vertical scrollbar.
 */
static void
update_scrollbar_instance (struct window *w, int vertical,
			   struct scrollbar_instance *instance)
{
  struct frame *f = XFRAME (w->frame);
  struct buffer *b = XBUFFER (w->buffer);
  Bufpos start_pos, end_pos, sb_pos;
  int new_width, new_height;
  int scrollbar_width = window_scrollbar_width (w);
  int scrollbar_height = window_scrollbar_height (w);

  end_pos = BUF_Z (b) - w->window_end_pos[CURRENT_DISP];
  sb_pos = scrollbar_point (w, 0);
  start_pos = sb_pos;

  /*
   * The end position must be strictly greater than the start position,
   * at least for the Motify scrollbar.  It shouldn't hurt anything for
   * other scrollbar implementations.
   */
  if (end_pos <= start_pos)
    end_pos = start_pos + 1;

  if (vertical)
    {
      new_height = WINDOW_TEXT_HEIGHT (w);
      new_width = scrollbar_width;
    }
  else
    {
      new_height = scrollbar_height;
      new_width = WINDOW_TEXT_WIDTH (w);
    }

  /* If the height and width are not greater than 0, then later on the
     Motif widgets will bitch and moan. */
  if (new_height <= 0)
    new_height = 1;
  if (new_width <= 0)
    new_width = 1;

  /*
   * The UPDATE_DATA_FIELD macro won't work for this because it is
   * a Lisp object.
   */
  if (!instance->mirror || XWINDOW (real_window (instance->mirror, 0)) != w)
    {
      abort ();
#ifdef THE_OLD_WAY
      instance->mirror = find_window_mirror (w);
      instance->scrollbar_instance_changed = 1;
#endif /* THE_OLD_WAY */
    }
      
  /*
   * Only character-based scrollbars are implemented at the moment.
   * Line-based will be implemented in the future.
   */

  /* don't use UPDATE_DATA_FIELD because we want to set this without
     tripping the changed field. */
  instance->scrollbar_is_active = 1;
  UPDATE_DATA_FIELD (data.line_increment, 1);
  UPDATE_DATA_FIELD (data.page_increment, 1);
#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
  if (!inhibit_thumb_size_change)
#endif
    {
      int x_offset, y_offset;

      /* Scrollbars are always the farthest from the text area. */
      if (vertical)
	{
	  x_offset = (f->scrollbar_on_left
		      ? WINDOW_LEFT (w)
		      : WINDOW_RIGHT (w) - scrollbar_width);
	  y_offset = WINDOW_TEXT_TOP (w) + f->scrollbar_y_offset;
	}
      else
	{
	  x_offset = WINDOW_TEXT_LEFT (w);
	  y_offset = f->scrollbar_y_offset +
	    (f->scrollbar_on_top
	     ? WINDOW_TOP (w)
	     : WINDOW_TEXT_BOTTOM (w) + window_bottom_toolbar_height (w));
	}

      UPDATE_DATA_FIELD (data.scrollbar_width, new_width);
      UPDATE_DATA_FIELD (data.scrollbar_height, new_height);
      UPDATE_DATA_FIELD (data.scrollbar_x, x_offset);
      UPDATE_DATA_FIELD (data.scrollbar_y, y_offset);
    }

  /*
   * A disabled scrollbar has its slider sized to the entire height of
   * the scrollbar.  Currently the minibuffer scrollbar is disabled.
   */
  if (!MINI_WINDOW_P (w) && vertical)
    {
      int new_slider_size;

#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
      if (!inhibit_thumb_size_change)
#endif
	{
	  UPDATE_DATA_FIELD (data.minimum, BUF_BEGV (b));
	  UPDATE_DATA_FIELD (data.maximum,
			     max (BUF_ZV (b),
				  instance->data.minimum + 1));
	  new_slider_size = min ((end_pos - start_pos),
				 (instance->data.maximum -
				  instance->data.minimum));
	  UPDATE_DATA_FIELD (data.slider_size, new_slider_size);
	  UPDATE_DATA_FIELD (data.slider_position, sb_pos);
#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
	  instance->vdrag_orig_value = instance->data.slider_position;
	  instance->vdrag_orig_window_start = 
	    marker_position (w->start[CURRENT_DISP]);
#endif
	}
    }
  else if (!MINI_WINDOW_P (w))
    {
      /* The minus one is to account for the truncation glyph. */
      int wcw = window_char_width (w, 0) - 1;
      int max_width, max_slide;

      if (w->max_line_len < wcw)
	{
	  max_width = 1;
	  max_slide = 1;
	  wcw = 1;
	}
      else
	{
	  max_width = w->max_line_len + 2;
	  max_slide = max_width - wcw;
	}

      UPDATE_DATA_FIELD (data.minimum, 0);
      UPDATE_DATA_FIELD (data.maximum, max_width);
      UPDATE_DATA_FIELD (data.slider_size, wcw);
      UPDATE_DATA_FIELD (data.slider_position,
			 min (w->hscroll, max_slide));
    }
  else
    {
      UPDATE_DATA_FIELD (data.minimum, 1);
      UPDATE_DATA_FIELD (data.maximum, 2);
      UPDATE_DATA_FIELD (data.slider_size, 1);
      UPDATE_DATA_FIELD (data.slider_position, 1);
      instance->scrollbar_is_active = 0;
    }
}

static void
update_one_scrollbar_bs (struct frame *f, Widget sb_widget)
{
  Boolean use_backing_store;

  XtVaGetValues (FRAME_X_TEXT_WIDGET (f),
		 XtNuseBackingStore, &use_backing_store, 0);

  if (use_backing_store && sb_widget)
    {
      unsigned long mask = CWBackingStore;
      XSetWindowAttributes attrs;
	      
      attrs.backing_store = Always;
      XChangeWindowAttributes (XtDisplay (sb_widget),
			       XtWindow (sb_widget),
			       mask,
			       &attrs);
    }
}

static void
update_instance_status (struct window *w, int active, int size,
			struct scrollbar_instance *instance)
{
  struct frame *f = XFRAME (w->frame);
  char managed = XtIsManaged (instance->scrollbar_widget);

  if (active && size)
    {
      widget_value *wv = scrollbar_instance_to_widget_value (instance);

      if (instance->scrollbar_instance_changed)
	{
	  lw_modify_all_widgets (instance->scrollbar_id, wv, 0);
	  instance->scrollbar_instance_changed = 0;
	}

      if (!managed)
	{
	  XtManageChild (instance->scrollbar_widget);
	  if (XtWindow (instance->scrollbar_widget))
	    {
	      /* Raise this window so that it's visible on top of the
                 text window below it. */
	      XRaiseWindow (XtDisplay (instance->scrollbar_widget),
			    XtWindow (instance->scrollbar_widget));
	      update_one_widget_scrollbar_pointer (f,
						   instance->scrollbar_widget);

	      if (!instance->backing_store_initialized)
		{
		  update_one_scrollbar_bs (f, instance->scrollbar_widget);
		  instance->backing_store_initialized = 1;
		}
	    }
	}

      if (!wv->scrollbar_data) abort ();
      xfree (wv->scrollbar_data);
      wv->scrollbar_data = 0;
      free_widget_value (wv);
    }
  else if (managed)
    {
      XtUnmanageChild (instance->scrollbar_widget);
    }
}

void
x_update_window_scrollbars (struct window *w, struct window_mirror *mirror,
			    int active, int horiz_only)
{
  struct frame *f = XFRAME (w->frame);
  struct scrollbar_instance *instance;

  in_display++;

  /* It is possible for this to get called from the mirror update
     routines.  In that case the structure is in an indeterminate
     state but we know exactly what struct we are working with.  So we
     pass it in in that case.  We also take advantage of it at some
     other points where we know what the mirror struct is. */
  if (!mirror)
    mirror = find_window_mirror (w);

  if (!mirror->scrollbar_vertical_instance && active)
    {
      mirror->scrollbar_vertical_instance = get_scrollbar_instance (f, 1);
    }
  if (!mirror->scrollbar_horizontal_instance && active)
    {
      mirror->scrollbar_horizontal_instance = get_scrollbar_instance (f, 0);
    }

  if (!horiz_only && mirror->scrollbar_vertical_instance)
    {
      int size = (active ? window_scrollbar_width (w) : 0);

      instance = mirror->scrollbar_vertical_instance;
      instance->scrollbar_is_active = active;
      instance->mirror = mirror;
      if (active && size)
	update_scrollbar_instance (w, 1, instance);
      update_instance_status (w, active, size, instance);
      if (!active)
 	{
 	  release_scrollbar_instance (f, 1, instance);
 	  mirror->scrollbar_vertical_instance = NULL;
 	}
    }

  if (mirror->scrollbar_horizontal_instance)
    {
      int size = (active ? window_scrollbar_height (w) : 0);

      instance = mirror->scrollbar_horizontal_instance;
      instance->scrollbar_is_active = active;
      instance->mirror = mirror;
      if (active && size)
	update_scrollbar_instance (w, 0, instance);
      update_instance_status (w, active, size, instance);
      if (!active)
 	{
 	  release_scrollbar_instance (f, 0, instance);
 	  mirror->scrollbar_horizontal_instance = NULL;
 	}
    }

  in_display--;
}

/* Make sure that all scrollbars on frame are up-to-date. */
void
x_update_frame_scrollbars (struct frame *f)
{
  /* Consider this code to be "in_display" so that we abort() if Fsignal()
     gets called. */
  in_display++;
  scrollbar_loop (UPDATE_FRAME_SCROLLBARS, f->root_window, f->root_mirror, 0,
		  (Window) NULL);
  in_display--;
  if (in_display < 0) abort ();
}

/* Do the resourcing for this frame's scrollbars. */
void
init_frame_scrollbars (struct frame *f)
{
  Lisp_Object frame = Qnil;

  XSETFRAME (frame, f);
  if (FRAME_IS_X (f))
    call_critical_lisp_code (XDEVICE (FRAME_DEVICE (f)),
			     Qx_init_scrollbar_from_resources,
			     frame);
}

void
init_device_scrollbars (struct device *d)
{
  Lisp_Object device = Qnil;

  XSETDEVICE (device, d);
  if (DEVICE_IS_X (d))
    call_critical_lisp_code (d,
			     Qx_init_scrollbar_from_resources,
			     device);
}

void
init_global_scrollbars (struct device *d)
{
  call_critical_lisp_code (d,
			   Qx_init_scrollbar_from_resources,
			   Qglobal);
}

/* Destroy all scrollbars associated with FRAME.  Only called from
   delete_frame_internal.
 */
#define FREE_FRAME_SCROLLBARS_INTERNAL(cache)				\
  do {									\
    while (FRAME_X_SB_##cache (f))					\
      {									\
	struct scrollbar_instance *tofree = FRAME_X_SB_##cache (f);	\
	FRAME_X_SB_##cache (f) = FRAME_X_SB_##cache (f)->next;		\
	tofree->next = NULL;						\
	free_scrollbar_instance (tofree);				\
      }									\
  } while (0)

void
free_frame_scrollbars (struct frame *f)
{
  if (!FRAME_IS_X (f))
    return;

  if (f->mirror_dirty)
    update_frame_window_mirror (f);
  scrollbar_loop (FREE_FRAME_SCROLLBARS, f->root_window, f->root_mirror, 0,
		  (Window) NULL);

  FREE_FRAME_SCROLLBARS_INTERNAL (VCACHE);
  FREE_FRAME_SCROLLBARS_INTERNAL (HCACHE);
}
#undef FREE_FRAME_SCROLLBARS_INTERNAL

/* This function is called as a result of a change to the
   `scrollbar-width' specifier. */

static void
scrollbar_width_changed_in_frame (Lisp_Object specifier, struct frame *f,
				  Lisp_Object oldval)
{
  XtWidgetGeometry req, repl;
  Lisp_Object newval = f->scrollbar_width;

  if (!FRAME_IS_X (f))
    return;

  /* We want the text area to stay the same size.  So, we query the
     current size and then adjust it for the change in the scrollbar
     width. */

  in_specifier_change_function++;
  if (!in_resource_setting)
    /* mirror the value in the frame resources, unless it was already
       done. */
    XtVaSetValues (FRAME_X_TEXT_WIDGET (f), XtNscrollBarWidth,
		   XINT (newval), 0);
  if (XtIsRealized (FRAME_X_CONTAINER_WIDGET (f)))
    {
      req.request_mode = 0;
      /* the query-geometry method looks at the current value
	 of f->scrollbar_width, so temporarily set it back to the
	 old one. */
      f->scrollbar_width = oldval;
      XtQueryGeometry (FRAME_X_CONTAINER_WIDGET (f), &req, &repl);
      f->scrollbar_width = newval;
      
      repl.width += XINT (newval) - XINT (oldval);
      
      EmacsManagerChangeSize (FRAME_X_CONTAINER_WIDGET (f), repl.width,
			      repl.height);
    }
  in_specifier_change_function--;
}

/* This function is called as a result of a change to the
   `scrollbar-height' specifier.  */

static void
scrollbar_height_changed_in_frame (Lisp_Object specifier, struct frame *f,
				   Lisp_Object oldval)
{
  XtWidgetGeometry req, repl;
  Lisp_Object newval = f->scrollbar_height;

  if (!FRAME_IS_X (f))
    return;

  /* We want the text area to stay the same size.  So, we query the
     current size and then adjust it for the change in the scrollbar
     height. */

  in_specifier_change_function++;
  if (!in_resource_setting)
    /* mirror the value in the frame resources, unless it was already
       done.  Also don't do it if this is the when the frame is
       being created -- the widgets don't even exist yet, and even
       if they did, we wouldn't want to overwrite the resource
       information (which might specify a user preference). */
    XtVaSetValues (FRAME_X_TEXT_WIDGET (f), XtNscrollBarHeight,
		   XINT (newval), 0);
  if (XtIsRealized (FRAME_X_CONTAINER_WIDGET (f)))
    {
      req.request_mode = 0;
      /* the query-geometry method looks at the current value
	 of f->scrollbar_height, so temporarily set it back to the
	 old one. */
      f->scrollbar_height = oldval;
      XtQueryGeometry (FRAME_X_CONTAINER_WIDGET (f), &req, &repl);
      f->scrollbar_height = newval;
      
      repl.height += XINT (newval) - XINT (oldval);
      
      EmacsManagerChangeSize (FRAME_X_CONTAINER_WIDGET (f), repl.width,
			      repl.height);
    }
  in_specifier_change_function--;
}

void
x_set_scrollbar_pointer (struct frame *f, Lisp_Object cursor)
{
  if (!FRAME_IS_X (f))
    return;

  if (!EQ (f->scrollbar_pointer, cursor))
    {
      /* #### If the user cuts this pointer, we'll get X errors.
         This needs to be rethunk. */
      f->scrollbar_pointer = cursor;

      if (f->mirror_dirty)
	update_frame_window_mirror (f);
      scrollbar_loop (X_SET_SCROLLBAR_POINTER, f->root_window, f->root_mirror,
		      0, (Window) NULL);
    }
}

int
x_window_is_scrollbar (struct frame *f, Window win)
{
  if (!FRAME_IS_X (f))
    return 0;

  if (f->mirror_dirty)
    update_frame_window_mirror (f);
  return (int) scrollbar_loop (X_WINDOW_IS_SCROLLBAR, f->root_window,
			       f->root_mirror, 0, win);
}

/* ####

   All of the following stuff is functions that handle scrollbar
   actions.  All of it should be moved into Lisp.  This may require
   adding some badly-needed primitives. */

/********** vertical scrollbar stuff **********/

/*
 * If the original point is still visible, put the cursor back there.
 * Otherwise, when scrolling down stick it at the beginning of the
 * first visible line and when scrolling up stick it at the beginning
 * of the last visible line.
 */

/* #### This function should be moved into Lisp */
static void
scrollbar_reset_cursor (Lisp_Object win, Lisp_Object orig_pt)
{
  /* When this function is called we know that start is already
     accurate.  We know this because either set-window-start or
     recenter was called immediately prior to it being called. */
  Lisp_Object buf;
  Bufpos start_pos = XINT (Fwindow_start (win));
  Bufpos ptint = XINT (orig_pt);
  struct window *w = XWINDOW (win);
  int selected = ((w == XWINDOW (Fselected_window (XFRAME (w->frame)->device)))
		  ? 1
		  : 0);

  buf = Fwindow_buffer (win);
  if (NILP (buf))
    return;	/* the window was deleted out from under us */

  if (ptint < XINT (Fwindow_start (win)))
    {
      if (selected)
	Fgoto_char (make_number (start_pos), buf);
      else
	Fset_window_point (win, make_number (start_pos));
    }
  else if (!point_would_be_visible (XWINDOW (win), start_pos, ptint))
    {
      Fmove_to_window_line (make_number (-1), win);

      if (selected)
	Fbeginning_of_line (Qnil, buf);
      else
	{
	  /* #### Taken from forward-line. */
	  Bufpos pos;

	  pos = find_next_newline (XBUFFER (buf),
				   marker_position (w->pointm[CURRENT_DISP]),
				   -1);
	  Fset_window_point (win, make_number (pos));
	}
    }
  else
    {
      if (selected)
	Fgoto_char (orig_pt, buf);
      else
	Fset_window_point (win, orig_pt);
    }
}

DEFUN ("scrollbar-line-up", Fscrollbar_line_up, Sscrollbar_line_up, 1, 1, 0,
  "Function called when the line-up arrow on the scrollbar is clicked.\n\
This is the little arrow at the top of the scrollbar.  One argument, the\n\
scrollbar's window.  You can advise this function to change the scrollbar\n\
behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  window_scroll (window, make_number (1), -1, 1);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-line-down", Fscrollbar_line_down, Sscrollbar_line_down,
       1, 1, 0,
  "Function called when the line-down arrow on the scrollbar is clicked.\n\
This is the little arrow at the bottom of the scrollbar.  One argument, the\n\
scrollbar's window.  You can advise this function to change the scrollbar\n\
behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  window_scroll (window, make_number (1), 1, 1);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-page-up", Fscrollbar_page_up, Sscrollbar_page_up,
       1, 1, 0,
  "Function called when the user gives the \"page-up\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.) One argument,\n\
a cons containing the scrollbar's window and a value (#### document me!\n\
This value is nil for Motif/Lucid scrollbars and a number for Athena\n\
scrollbars).  You can advise this function to change the scrollbar\n\
behavior.")
     (object)
     Lisp_Object object;
{
  Lisp_Object window = Fcar (object);

  CHECK_LIVE_WINDOW (window, 0);
  /* Motif and Athena scrollbars behave differently, but in accordance
     with their standard behaviors.  It is not possible to hide the
     differences down in lwlib because knowledge of XEmacs buffer and
     cursor motion routines is necessary. */
#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
  window_scroll (window, Qnil, -1, 1);
#else /* Athena */
  {
    Bufpos bufpos;
    Lisp_Object value = Fcdr (object);

    CHECK_INT (value, 0);
    Fmove_to_window_line (Qzero, window);
    /* can't use Fvertical_motion() because it moves the buffer point
       rather than the window's point.

       #### It does?  Why does it take a window argument then? */
    bufpos = vmotion (XWINDOW (window), XINT (Fwindow_point (window)),
		      XINT (value), 0);
    Fset_window_point (window, make_number (bufpos));
    Frecenter (Qzero, window);
  }
#endif /* Athena */
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-page-down", Fscrollbar_page_down, Sscrollbar_page_down,
       1, 1, 0,
  "Function called when the user gives the \"page-down\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.) One argument,\n\
a cons containing the scrollbar's window and a value (#### document me!\n\
This value is nil for Motif/Lucid scrollbars and a number for Athena\n\
scrollbars).  You can advise this function to change the scrollbar\n\
behavior.")
     (object)
     Lisp_Object object;
{
  Lisp_Object window = Fcar (object);

  CHECK_LIVE_WINDOW (window, 0);
  /* Motif and Athena scrollbars behave differently, but in accordance
     with their standard behaviors.  It is not possible to hide the
     differences down in lwlib because knowledge of XEmacs buffer and
     cursor motion routines is necessary. */
#if defined (LWLIB_SCROLLBARS_MOTIF) || defined (LWLIB_SCROLLBARS_LUCID)
  window_scroll (window, Qnil, 1, 1);
#else /* Athena */
  {
    Lisp_Object value = Fcdr (object);
    CHECK_INT (value, 0);
    Fmove_to_window_line (value, window);
    Frecenter (Qzero, window);
  }
#endif /* Athena */
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-to-top", Fscrollbar_to_top, Sscrollbar_to_top,
       1, 1, 0,
  "Function called when the user gives the \"to-top\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.). One argument,\n\
the scrollbar's window.  You can advise this function to change the\n\
scrollbar behavior.")
     (window)
     Lisp_Object window;
{
  Lisp_Object orig_pt;

  orig_pt = Fwindow_point (window);
  Fset_window_point (window, Fpoint_min (Fwindow_buffer (window)));
  Frecenter (Qzero, window);
  scrollbar_reset_cursor (window, orig_pt);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-to-bottom", Fscrollbar_to_bottom, Sscrollbar_to_bottom,
       1, 1, 0,
  "Function called when the user gives the \"to-bottom\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.). One argument,\n\
the scrollbar's window.  You can advise this function to change the\n\
scrollbar behavior.")
     (window)
     Lisp_Object window;
{
  Lisp_Object orig_pt;

  orig_pt = Fwindow_point (window);
  Fset_window_point (window, Fpoint_max (Fwindow_buffer (window)));
  Frecenter (Qzero, window);
  scrollbar_reset_cursor (window, orig_pt);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-vertical-drag", Fscrollbar_vertical_drag,
       Sscrollbar_vertical_drag, 1, 1, 0,
  "Function called when the user drags the vertical scrollbar thumb.\n\
One argument, a cons containing the scrollbar's window and a value\n\
(#### document me!).  You can advise this function to change the\n\
scrollbar behavior.")
     (object)
     Lisp_Object object;
{
  Bufpos start_pos;
  Lisp_Object orig_pt;
  Lisp_Object window = Fcar (object);
  Lisp_Object value = Fcdr (object);

  orig_pt = Fwindow_point (window);
  Fset_marker (XWINDOW (window)->sb_point, value, Fwindow_buffer (window));
  start_pos = scrollbar_point (XWINDOW (window), 1);
  Fset_window_start (window, make_number (start_pos), Qnil);
  scrollbar_reset_cursor (window, orig_pt);
  zmacs_region_stays = 1;
  return Qnil;
}


/********** horizontal scrollbar stuff. **********/

static void
scrollbar_set_hscroll (Lisp_Object window, int hscroll)
{
  struct window *w = XWINDOW (window);
  int wcw = window_char_width (w, 0) - 1;
  int max_len = w->max_line_len + 1;

  if (hscroll > (max_len - wcw))
    hscroll = (max_len - wcw);

  /* Can't allow this out of set-window-hscroll's acceptable range. */
  if (hscroll < 0)
    hscroll = 0;
  else if (hscroll >= (1 << (SHORTBITS - 1)))
    hscroll = (1 << (SHORTBITS - 1)) - 1;

  if (hscroll != w->hscroll)
    Fset_window_hscroll (window, make_number (hscroll));
}

DEFUN ("scrollbar-char-left", Fscrollbar_char_left, Sscrollbar_char_left,
       1, 1, 0,
  "Function called when the char-left arrow on the scrollbar is clicked.\n\
This is the little arrow to the left of the scrollbar.  One argument, the\n\
scrollbar's window.  You can advise this function to change the scrollbar\n\
behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  scrollbar_set_hscroll (window, XWINDOW (window)->hscroll - 1);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-char-right", Fscrollbar_char_right, Sscrollbar_char_right,
       1, 1, 0,
  "Function called when the char-right arrow on the scrollbar is clicked.\n\
This is the little arrow to the right of the scrollbar.  One argument, the\n\
scrollbar's window.  You can advise this function to change the scrollbar\n\
behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  scrollbar_set_hscroll (window, XWINDOW (window)->hscroll + 1);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-page-left", Fscrollbar_page_left, Sscrollbar_page_left,
       1, 1, 0,
  "Function called when the user gives the \"page-left\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.) One argument,\n\
the scrollbar's window.  You can advise this function to change the\n\
scrollbar behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  scrollbar_set_hscroll (window, XWINDOW (window)->hscroll -
			 (window_char_width (XWINDOW (window), 0) - 2));
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-page-right", Fscrollbar_page_right, Sscrollbar_page_right,
       1, 1, 0,
  "Function called when the user gives the \"page-right\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.) One argument,\n\
the scrollbar's window.  You can advise this function to change the\n\
scrollbar behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  scrollbar_set_hscroll (window, XWINDOW (window)->hscroll +
			 (window_char_width (XWINDOW (window), 0) - 2));
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-to-left", Fscrollbar_to_left, Sscrollbar_to_left,
       1, 1, 0,
  "Function called when the user gives the \"to-left\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.). One argument,\n\
the scrollbar's window.  You can advise this function to change the\n\
scrollbar behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  scrollbar_set_hscroll (window, 0);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-to-right", Fscrollbar_to_right, Sscrollbar_to_right,
       1, 1, 0,
  "Function called when the user gives the \"to-right\" scrollbar action.\n\
(The way this is done can vary from scrollbar to scrollbar.). One argument,\n\
the scrollbar's window.  You can advise this function to change the\n\
scrollbar behavior.")
     (window)
     Lisp_Object window;
{
  CHECK_LIVE_WINDOW (window, 0);
  /* We just pass in a huge number here and scrollbar_set_hscroll will
     take care of putting it into the correct range which will give us
     the desired behavior. */
  scrollbar_set_hscroll (window, (1 << (SHORTBITS - 1)));
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-horizontal-drag", Fscrollbar_horizontal_drag,
       Sscrollbar_horizontal_drag, 1, 1, 0,
  "Function called when the user drags the horizontal scrollbar thumb.\n\
One argument, a cons containing the scrollbar's window and a value\n\
representing how many columns the thumb is slid over.  You can advise\n\
this function to change the scrollbar behavior.")
     (object)
     Lisp_Object object;
{
  Lisp_Object window = Fcar (object);
  Lisp_Object value = Fcdr (object);

  CHECK_LIVE_WINDOW (window, 0);
  CHECK_INT (value, 0);
  scrollbar_set_hscroll (window, XINT (value));
  zmacs_region_stays = 1;
  return Qnil;
}

void
init_scrollbar_once (void)
{
}

void
syms_of_scrollbar (void)
{
  defsymbol (&Qscrollbar_line_up, "scrollbar-line-up");
  defsymbol (&Qscrollbar_line_down, "scrollbar-line-down");
  defsymbol (&Qscrollbar_page_up, "scrollbar-page-up");
  defsymbol (&Qscrollbar_page_down, "scrollbar-page-down");
  defsymbol (&Qscrollbar_to_top, "scrollbar-to-top");
  defsymbol (&Qscrollbar_to_bottom, "scrollbar-to-bottom");
  defsymbol (&Qscrollbar_vertical_drag, "scrollbar-vertical-drag");
  
  defsymbol (&Qscrollbar_char_left, "scrollbar-char-left");
  defsymbol (&Qscrollbar_char_right, "scrollbar-char-right");
  defsymbol (&Qscrollbar_page_left, "scrollbar-page-left");
  defsymbol (&Qscrollbar_page_right, "scrollbar-page-right");
  defsymbol (&Qscrollbar_to_left, "scrollbar-to-left");
  defsymbol (&Qscrollbar_to_right, "scrollbar-to-right");
  defsymbol (&Qscrollbar_horizontal_drag, "scrollbar-horizontal-drag");

#if defined (LWLIB_SCROLLBARS_LUCID)
  Fprovide (intern ("lucid-scrollbars"));
#elif defined (LWLIB_SCROLLBARS_MOTIF)
  Fprovide (intern ("motif-scrollbars"));
#elif defined (LWLIB_SCROLLBARS_ATHENA)
  Fprovide (intern ("athena-scrollbars"));
#endif

  DEFVAR_SPECIFIER ("scrollbar-width", &Vscrollbar_width,
    "*Width of vertical scrollbars.\n\
This is a specifier; use `set-specifier' to change it.");
  Vscrollbar_width = Fmake_specifier (Qnatnum);
  set_specifier_fallback
    (Vscrollbar_width,
     list1 (Fcons (Qnil, make_number (DEFAULT_SCROLLBAR_WIDTH))));
  set_specifier_caching (Vscrollbar_width,
			 slot_offset (struct window,
				      scrollbar_width),
			 some_window_value_changed,
			 slot_offset (struct frame,
				      scrollbar_width),
			 scrollbar_width_changed_in_frame);

  DEFVAR_SPECIFIER ("scrollbar-height", &Vscrollbar_height,
    "*Width of vertical scrollbars.\n\
This is a specifier; use `set-specifier' to change it.");
  Vscrollbar_height = Fmake_specifier (Qnatnum);
  set_specifier_fallback
    (Vscrollbar_height,
     list1 (Fcons (Qnil, make_number (DEFAULT_SCROLLBAR_HEIGHT))));
  set_specifier_caching (Vscrollbar_height,
			 slot_offset (struct window,
				      scrollbar_height),
			 some_window_value_changed,
			 slot_offset (struct frame,
				      scrollbar_height),
			 scrollbar_height_changed_in_frame);

  defsymbol (&Qx_init_scrollbar_from_resources,
	     "x-init-scrollbar-from-resources");

  /* #### All these functions should be moved into Lisp.
     See comment above. */
  defsubr (&Sscrollbar_line_up);
  defsubr (&Sscrollbar_line_down);
  defsubr (&Sscrollbar_page_up);
  defsubr (&Sscrollbar_page_down);
  defsubr (&Sscrollbar_to_top);
  defsubr (&Sscrollbar_to_bottom);
  defsubr (&Sscrollbar_vertical_drag);

  /* #### These too. */
  defsubr (&Sscrollbar_char_left);
  defsubr (&Sscrollbar_char_right);
  defsubr (&Sscrollbar_page_left);
  defsubr (&Sscrollbar_page_right);
  defsubr (&Sscrollbar_to_left);
  defsubr (&Sscrollbar_to_right);
  defsubr (&Sscrollbar_horizontal_drag);
}  
