/* ----------------------------------------------------------------------
 * FILE: rh.c
 * VERSION: 2
 * Written by: Ken Stauffer
 * VERSION 2.8
 * Enhancements and bug fixes by Rick Ohnemus
 * 
 * printhelp(), execute(), exam1(), exam2(), exam3(), exam4(), main()
 *
 *
 * ---------------------------------------------------------------------- */

#include "rh.h"
#include <pwd.h>
#include <varargs.h>

#define VERSION "Version 2.8"

extern void	ftrw();
extern char *	graphic();
extern void	printentry();
extern void	printformat();
extern void	program();
extern void	rhfinish();
extern void	rhinit();

extern char *sys_errlist[];
extern int errno;
extern int sys_nerr;

static char *prog_name;

#ifdef S_IFLNK

#define SWITCHES "DFLVbcde:f:hilp:qrstvx:"

static char *usage = {
    "Usage: %s [-DFLVbcdhilqrsv] [ -f filename ] [-] [ -e expression ]\n\t[ -p format ] [-x command ] file ...\n"
};

#else

#define SWITCHES "Vbcde:f:hilp:qrstvx:"

static char *usage = {
    "Usage: %s [-Vbcdhilqrsv] [ -f filename ] [-] [ -e expression ]\n\t[ -p format ] [-x command ] file ...\n"
};
#endif

void error(s)
char *s;
{
    (void) fprintf(stderr, "%s: %s\n", prog_name, s);
    exit(1);
}

/*VARARGS*/
void warning(va_alist)
va_dcl
{
    char *m_ptr;
    char *msg_format;
    char *nfp;
    char *p;
    char new_format[BUFSIZ];
    int save_errno;
    va_list args;
    
    
    save_errno = errno;
    
    va_start(args);
    msg_format = va_arg(args, char *);
    
    (void) fprintf(stderr, "%s: ", prog_name);
    
    if ((m_ptr = strstr(msg_format, "%m")) != (char *) NULL) {
	(void) strncpy(new_format, msg_format, m_ptr - msg_format);
	nfp = new_format + (m_ptr - msg_format);
	if (save_errno < sys_nerr) {
	    for (p = sys_errlist[save_errno]; *p != '\0'; *nfp++ = *p++)	/*SUPPRESS 530*/
		/* void */ ;
	}
	else {
	    (void) sprintf(nfp, "errno = %d", save_errno);
	    nfp += strlen(nfp);
	}
	(void) strcpy(nfp, m_ptr + 2);
	msg_format = new_format;
    }
    
    (void) vfprintf(stderr, msg_format, args);
    
    for (p = msg_format; *p != '\0'; p++)	/*SUPPRESS 530*/
	/* void */ ;
    
    if (*(p - 1) != '\n')
	(void) putc('\n', stderr);
    
    va_end(args);
    
    return;
}

/* ----------------------------------------------------------------------
 * printhelp:
 *	Print out the help screen. The string 's' is argv[0].
 *	Called when the -h option is used.
 *
 */

static void printhelp(s)
char *s;
{
    int i;
    struct symbol *p;

    (void) printf(usage, s);
    (void) printf("\n%s\n", VERSION);
    (void) printf("\noptions:\n");
#ifdef S_IFLNK
    (void) printf("\t-D\tfollow symbolic links that point to directories\n");
    (void) printf("\t-F\tfollow symbolic links that point to non-directories\n");
    (void) printf("\t-L\tfollow all symbolic links\n");
#endif
    (void) printf("\t-V\twrite version number to stderr and exit\n");
    (void) printf("\t-b\tprint leading \"./\" if no directory\n");
    (void) printf("\t-c\tdo not read .rhrc or system rh files\n");
    (void) printf("\t-d\tleading '?' or '*' matches leading '.' in file name\n");
    (void) printf("\t-e\tget expression from the command line\n");
    (void) printf("\t-f\tget expression from file\n");
    (void) printf("\t-i\tignore case during file name matching\n");
    (void) printf("\t-h\tshow this message\n");
    (void) printf("\t-l\tlong filename output\n");
    (void) printf("\t-p\tprint file information using format\n");
    (void) printf("\t-q\tdisplay non-graphic characters in filenames as ?\n");
    (void) printf("\t-r\tmakes %s non-recursive\n", s);
    (void) printf("\t-s\tprint users/groups as strings instead of numbers\n");
    (void) printf("\t-t\tprint full date/time instead of ls(1) type date/time\n");
    (void) printf("\t-v\tverbose output\n");
    (void) printf("\t-x\texecute a unix command for matching files\n");
    
    (void) printf("\nvalid symbols:\n");
    for (i = 1, p = symbols; p != (struct symbol *) NULL; p = p->next, i++)
	(void) printf("%12s%s", p->name,
		      (((i - 1) % 5) == 4
		       || p->next == (struct symbol *) NULL) ? "\n" : " ");
    
    (void) printf("\nC operators:\n");
    (void) printf("\t! ~ - * / %% + < <= > >= == != & ^ | << >> && || ?:\n");
    (void) printf("\nspecial operators:\n");
    (void) printf("\t$username , @groupname , \"*.c\" , [date/time spec]\n\n");
    
    exit(1);
}

#ifdef RLIMIT_NOFILE
static int max_descriptors()
{
    struct rlimit newrlp;
    struct rlimit rlp;
    
    if (getrlimit(RLIMIT_NOFILE, &rlp) == 0) {
	if (rlp.rlim_cur < NOFILE) {
	    newrlp.rlim_max = newrlp.rlim_cur = NOFILE;
	    if (setrlimit(RLIMIT_NOFILE, &newrlp) == 0)
		return NOFILE;
	}
	return rlp.rlim_cur;
    }
    else
	return NOFILE;
}
#endif

static int fmt_req_stat()
{
    char *	fp;
    int		stat_all = FALSE;

    for (fp = attr.format; *fp != '\0' && !stat_all; fp++) {
	if (*fp != '%')
	    continue;

	fp++;

	switch (*fp) {
	case '\0':
	    error("format character missing after '%%'");
	    exit(1);
	    /*NOTREACHED*/
	    
	case 'a':
#if BSD
	case 'b':
	case 'B':
#endif
	case 'c':
	case 'g':
	case 'G':
	case 'i':
	case 'l':
	case 'm':
	case 'p':
	case 'P':
	case 'r':
	case 'R':
	case 'u':
	case 'U':
	case 'w':
	case 'W':
	case 'y':
	case 'z':
	    stat_all = TRUE;
	    break;
	}
    }

    return stat_all;
}

/* ----------------------------------------------------------------------
 * execute:
 *	Execute the program contained in the StackProgram[]
 *	array. Each element of the StackProgram[] array contains
 *	a pointer to a function.
 *	Programs are NULL terminated.
 *	Returns the result of the expression.
 *
 */

static long execute()
{
    register long eval;
    register void (*efunc)();
    
    SP=0;
#ifndef OLD_START
    if (startPC == -1)
	return 1;
#endif
    for (PC = startPC;
	 (efunc = StackProgram[PC].func) != (void (*)()) NULL;
	 PC++) {
	eval = StackProgram[PC].value;
	(*efunc)(eval);
	if (SP >= MEM) {
	    (void) fprintf(stderr, "stack overflow\n");
	    exit(1);
	}
    }
    return (Stack[0]);
}

/* ----------------------------------------------------------------------
 * exam1: exam2: exam3, exam4:
 *	One of these functions is called for every file that 'rh' examines.
 *	exam{1,2,3,4}() first calls execute to see if the
 *	expression is true, it then prints the file if the expression
 *	evaluated to true (non-zero).
 *
 */

/* print file out by itself */
static void exam1()
{
    if (execute())
	(void) printf("%s\n", attr.graphic ? graphic(attr.fname) : attr.fname);
    return;
}

/* long output of file */
static void exam2()
{
    if (execute())
	printentry(attr.verbose, attr.str_owner, attr.full_date,
		   attr.buf, attr.fname);
    return;
}

/* formatted output of file information */
static void exam3()
{
    if (execute())
	printformat(attr.buf, attr.fname, attr.depth);
    return;
}

/* do a system(3) call to desired command */
static void exam4()
{
    char *end;
    char *n;
    char *p;
    char *q;
    char *r;
    char command[NCARGS];
    int f_l;
    int n_l;
    int rv;
    
    if (execute()) {
	p = command;
	end = command + sizeof(command);
	n = (char *) NULL;
	q = attr.command;
	f_l = strlen(attr.fname);
	while (*q != '\0') {
	    if (*q != '%')
		*p++ = *q++;
	    else {
		q++;
		if (*q == 's') {
		    if ((p + f_l) >= end) {
			warning("%s: maximum command line length exceeded",
				attr.fname);
			return;
		    }
		    r = attr.fname;
		    while ((*p++ = *r++) != '\0')	/*SUPPRESS 530*/
			/* void */ ;
		    p--;
		}
		else if (*q == 'S') {
		    if (n == (char *) NULL) {
			if ((n = strrchr(attr.fname, '/')) == (char *) NULL)
			    n = attr.fname;
			else
			    n++;
			n_l = strlen(n);
		    }
		    if ((p + n_l) >= end) {
			warning("%s: maximum command line length exceeded",
				attr.fname);
			return;
		    }
		    r = n;
		    while ((*p++ = *r++) != '\0')	/*SUPPRESS 530*/
			/* void */ ;
		    p--;
		}
		else {
		    if ((p + 2) >= end) {
			warning("%s: maximum command line length exceeded",
				attr.fname);
			return;
		    }
		    *p++ = '%';
		    *p++ = *q;
		}
		q++;
	    }
	    if (p == end) {
		warning("%s: maximum command line length exceeded", attr.fname);
		return;
	    }
	}
	*p = '\0';
	rv = system(command);
	if (attr.verbose)
	    (void) printf("%s exit(%d)\n", command, rv);
    }
    
    return;
}

/* ----------------------------------------------------------------------
 * main:
 *	parse arguments.
 *	-d, -F, -i, -l, -r, -u, and -v options can occur as often as desired.
 *      -h and -V will cause rh to exit after they print their messages.
 *	-f, -x, -p, and -e can only occur once and MUST have an argument.
 *
 *      Read and "compile" any .rhrc file found in path given by $RHPATH
 *	 or read and "compile" the $HOME/.rhrc file, if it exists.
 *	Read and "compile" any -f filename, if present.
 *	Read and "compile" any -e expression, if present.
 *	If after all that no start expression is found then read from
 *	stdin if '-' switch is given on command line otherwise all files
 *	found will match.
 *	Perform the recursive hunt on remaining arguments.
 *
 */

main(argc, argv)
int argc;
char *argv[];
{
    extern char *	optarg;
    extern int		optind;
    extern char *	resolve_name();

    int			dashb = FALSE;
    int			dashc = TRUE;		/* -c option */
    char *		dashe = (char *) NULL;	/* -e option */
    char *		dashf = (char *) NULL;	/* -f option */
    int			dashl = FALSE;
    int			dashr = TRUE;
    int			depth;
    void		(*examptr)() = exam1;	/* default output function */
    int			i;
    struct passwd *	info;
    char		initfile[MAXPATHLEN + 1];
    char *		rhpath;			/* RHPATH env. variable */


    if ((prog_name = strrchr(argv[0], '/')) == (char *) NULL)
	prog_name = argv[0];
    else
	prog_name++;

    attr.command = (char *) NULL;	/* -x option */
    attr.format = (char *) NULL;	/* -p option */
    attr.dot_special = TRUE;		/* -d option */
#ifdef S_IFLNK
    attr.follow_dir = FALSE;		/* -D & -L options */
    attr.follow_file = FALSE;		/* -F & -L options */
#endif
    attr.graphic = FALSE;		/* -q option */
    attr.ignore_case = FALSE;		/* -i option */
    attr.prune = FALSE;			/* prune (obviously :-) */
    attr.stat_all = FALSE;		/* don't need to stat everything */
    attr.str_owner = FALSE;		/* -u option */
    attr.verbose = FALSE;		/* -v option */
    attr.full_date = FALSE;		/* -t option */


    rhinit();

    while ((i = getopt(argc, argv, SWITCHES)) != EOF) {

	switch (i) {
	    
	case 'b':
	    dashb = TRUE;
	    break;
	    
	case 'c':
	    dashc = FALSE;
	    break;

#ifdef S_IFLNK
	case 'D':
	    attr.follow_dir = TRUE;
	    attr.stat_all = TRUE;
	    break;
#endif
	    
	case 'd':
	    attr.dot_special = FALSE;
	    break;
	    
	case 'e':
	    if (dashe != (char *) NULL) {
		(void) fprintf(stderr, "%s: too many -e options\n", argv[0]);
		exit(1);
	    }
	    dashe = optarg;
	    break;
	    
#ifdef S_IFLNK
	case 'F':
	    attr.follow_file = TRUE;
	    attr.stat_all = TRUE;
	    break;
#endif
	    
	case 'f':
	    if (dashf != (char *) NULL) {
		(void) fprintf(stderr, "%s: too many -f options\n", argv[0]);
		exit(1);
	    }
	    dashf = optarg;
	    break;
	    
	case 'h':
	    printhelp(argv[0]);
	    /*NOTREACHED*/
	    
	case 'i':
	    attr.ignore_case = TRUE;
	    break;
	    
#ifdef S_IFLNK
	case 'L':
	    attr.follow_dir = TRUE;
	    attr.follow_file = TRUE;
	    attr.stat_all = TRUE;
	    break;
#endif
	    
	case 'l':
	    examptr = exam2;
	    dashl = TRUE;
	    attr.stat_all = TRUE;
	    break;
	    
	case 'p':
	    if (attr.format != (char *) NULL) {
		(void) fprintf(stderr, "%s: too many -p options\n", argv[0]);
		exit(1);
	    }
	    attr.format = optarg;
	    examptr = exam3;
	    break;
	    
	case 'q':
	    attr.graphic = TRUE;
	    break;
	    
	case 'r':
	    dashr = FALSE;
	    break;
	    
	case 's':
	    attr.str_owner = TRUE;
	    break;
	    
	case 't':
	    attr.full_date = TRUE;
	    break;

	case 'V':
	    error(VERSION);
	    /*NOTREACHED*/
	    
	case 'v':
	    attr.verbose = TRUE;
	    break;
	    
	case 'x': 
	    if (attr.command != (char *) NULL) {
		(void) fprintf(stderr, "%s: too many -x options\n", argv[0]);
		exit(1);
	    }
	    examptr = exam4;
	    attr.command = optarg;
	    break;
	    
	default:
	    (void) fprintf(stderr, "%s: invalid option, use -h for help\n",
			   argv[0]);
	    (void) fprintf(stderr, usage, argv[0]);
	    exit(1);
	}
	
    }
    
    if (attr.command != (char *) NULL) {
	if (dashl)
	    error("cannot have both -x and -l options");
	else if (attr.format != (char *) NULL) {
	    warning("-x overrides -p, -p ignored");
	    attr.format = (char *) NULL;
	}
    }
    else if (attr.format != (char *) NULL) {
	if (dashl) {
	    warning("-p overrides -l, -l ignored");
	    dashl = FALSE;
	}
	if (attr.verbose) {
	    warning("-p overrides -v, -v ignored");
	    attr.verbose = FALSE;
	}
	if (!attr.stat_all)
	    attr.stat_all = fmt_req_stat();
    }

    if (isatty(fileno(stdout)))
	attr.graphic = TRUE;
    
    PC = 0;
    startPC = -1;
    
    if (dashc && (dashf != (char *) NULL || dashe != (char *) NULL)) {
	if ((rhpath = getenv(RHPATHENV)) == (char *) NULL) {
home_rhrc:
	    if ((expfname = getenv(HOMEENV)) == (char *) NULL) {
		info = getpwuid(getuid());
		if (info != (struct passwd *) NULL) {
		    (void) sprintf(initfile, "%s/%s", info->pw_dir, RHRC);
		    expfname = initfile;
		    if ((expfile = fopen(expfname, "r")) != (FILE *) NULL) {
			expstr = (char *) NULL;
			program();
			(void) fclose(expfile);
		    }
		}
	    }
	    else {
		(void) sprintf(initfile, "%s/%s", expfname, RHRC);
		expfname = initfile;
		if ((expfile = fopen(expfname, "r")) != (FILE *) NULL) {
		    expstr = (char *) NULL;
		    program();
		    (void) fclose(expfile);
		}
	    }
	}
	else {
	    expfname = resolve_name(rhpath, RHRC);
	    if (expfname == (char *) NULL)
		goto home_rhrc;
	    if ((expfile = fopen(expfname, "r")) != (FILE *) NULL) {
		expstr = (char *) NULL;
		program();
		(void) fclose(expfile);
	    }
	    free(expfname);
	}
    }
    
    if (dashf != (char *) NULL) {
	expstr = (char *) NULL;
	expfname = resolve_name(rhpath, dashf);
	if (expfname == (char *) NULL)
	    error("unable to resolve path");
	if ((expfile = fopen(expfname, "r")) == (FILE *) NULL) {
	    (void) fprintf(stderr, "%s: ", argv[0]);
	    perror(expfname);
	    exit(1);
	}
	program();
	(void) fclose(expfile);
	free(expfname);
    }
    
    if (dashe != (char *) NULL) {
	expfile = (FILE *) NULL;
	expstr = dashe;
	program();
    }
    
#ifndef OLD_START
    if (startPC == -1
	&& optind < argc
	&& argv[optind][0] == '-'
	&& argv[optind][1] == '\0') {
	optind++;
	expstr = (char *) NULL;
	expfname = "stdin";
	expfile = stdin;
	program();
    }
#else
    if (startPC == -1) {
	expstr = (char *) NULL;
	expfname = "stdin";
	expfile = stdin;
	program();
    }
    
    if (startPC == -1) {
	(void) fprintf(stderr, "%s: no start expression specified\n", argv[0]);
	exit(1);
    }
#endif
    
    rhfinish();
    
    depth = dashr ? DEPTH : 1;
    
    if (optind >= argc)
	ftrw(dashb ? "." : "", examptr, depth, dashr);
    else {
	for (/* void */; optind < argc; optind++)
	    ftrw(argv[optind], examptr, depth, dashr);
    }
    
    exit(0);
    /*NOTREACHED*/
}
