/* ----------------------------------------------------------------------
 * FILE: rhdir.c
 * VERSION: 2
 * Written by: Ken Stauffer
 * VERSION 2.8
 * Enhancements and bug fixes by Rick Ohnemus
 * 
 * This file contains the "non portable" stuff dealing with
 * directories.
 *
 * printentry(), printformat(), ftrw(), fwt1(), ls_date(), graphic()
 *
 *
 * ---------------------------------------------------------------------- */

#include <ctype.h>
#include "rh.h"

#define user_index(b)	((000777 & (b)) >> 6) + ((b) & S_ISUID ? 8 : 0) 
#define group_index(b)	((000077 & b) >> 3) + ((b) & S_ISGID ? 8 : 0)
#define all_index(b)	((000007 & (b)) + (((b) & S_ISVTX) ? 8 : 0))
#define ftype_index(b)	((b) >> 13)

#define hassubdir(b)	((b)->st_nlink > 2)

#define isdot(s)	((s)[1] == '\0' && (s)[0] == '.')
#define isdotdot(s)	((s)[2] == '\0' && (s)[1] == '.' && (s)[0] == '.')

#ifndef S_IFLNK
#define    lstat        stat
#endif

#define isproper(m)    (!attr.prune && S_ISDIR(m))

#ifdef    POSIX_DIRECTORY_LIBRARY
#include <dirent.h>
#else
#include <sys/dir.h>
#endif

extern char *ctime();
extern void warning();


static char *ftype[] = { "p", "c" , "d" , "b" , "-" , "l" , "s" , "t" };

static char *perm[] = {
    "---", "--x", "-w-", "-wx" , "r--", "r-x", "rw-", "rwx" ,
    "--S", "--s", "-wS", "-ws" , "r-S", "r-s", "rwS", "rws"
};

static char *perm2[] = {
    "---", "--x", "-w-", "-wx" , "r--", "r-x", "rw-", "rwx" ,
    "--T", "--t", "-wT", "-wt" , "r-T", "r-t", "rwT", "rwt"
};

char *graphic(name)
char *name;
{
    static char new_name[MAXPATHLEN + 1];
    
    char *p = new_name;
    
    while (*name != '\0') {
	if (isascii(*name) && isgraph(*name))
	    *p++ = *name++;
	else {
	    *p++ = '?';
	    name++;
	}
    }
    *p = '\0';
    return new_name;
}

static char *ls_date(ftime, full_date)
time_t ftime;
int full_date;
{
#if SUNOS_4
    char *ts = ctime(&ftime);
#else
    char *ts = ctime((long *) &ftime);
#endif
    static char date[32];
    
    if (full_date) {
	char *to = date;
	while (*ts != '\n' && *ts != '\0')
	    *to++ = *ts++;
	*to = '\0';
    }
    else if (ftime < past_time || ftime > (now->value + 5L * 60L))
	(void) sprintf(date, "%6.6s  %4.4s", ts + 4, ts + 20);
    else
	(void) sprintf(date, "%12.12s", ts + 4);
    
    return date;
}

void printformat(buf, name, depth)
struct stat *buf;
char *name;
int depth;
{
    char *fp;
    char *p;
    int i;
    struct group_info *gi;
    struct user_info *ui;
    unsigned char c;
    
    
    for (fp = attr.format; *fp != '\0'; fp++) {
	if (*fp != '%') {
	    if (*fp != '\\') {
		(void) putc(*fp, stdout);
		continue;
	    }
	    fp++;
	    
	    switch (*fp) {
		
	    case '\0':
		(void) putc('\\', stdout);
		break;
		
	    case '0':
		if (*(fp + 1) == 'x' || *(fp + 1) == 'X') {
		    if (isxdigit(*(fp + 2))) {
			fp += 2;
			for (i = c = 0; i < 2 && isxdigit(*fp); i++, fp++) {
			    c <<= 4;
			    if (isupper(*fp))
				*fp = tolower(*fp);
			    c += strchr(hex_digits, *fp) - hex_digits;
			}
			(void) putc((char) c, stdout);
			fp--;
		    }
		    else
			(void) putc(*fp, stdout);
		}
		else if (*(fp + 1) >= '0' && *(fp + 1) <= '7') {
		    fp++;
		    for (i = c = 0; i < 3 && *fp >= '0' && *fp <= '7'; i++, fp++) {
			c <<= 3;
			c += *fp - '0';
		    }
		    (void) putc((char) c, stdout);
		    fp--;
		}
		else
		    (void) putc(*fp, stdout);
		break;
		
	    case 'a':
		(void) putc('\007', stdout);
		break;
		
	    case 'b':
		(void) putc('\b', stdout);
		break;
		
	    case 'f':
		(void) putc('\f', stdout);
		break;
		
	    case 'n':
		(void) putc('\n', stdout);
		break;
		
	    case 'r':
		(void) putc('\r', stdout);
		break;
		
	    case 't':
		(void) putc('\t', stdout);
		break;
		
	    case 'v':
		(void) putc('\v', stdout);
		break;
		
	    default:
		(void) putc(*fp, stdout);
		break;
	    }
	    
	    continue;
	}
	
	fp++;
	
	switch (*fp) {
	    
	case '\0':
	    (void) fprintf(stderr, "format character missing after '%%'\n");
	    exit(1);
	    
	case 'a':
	    (void) printf(ls_date(buf->st_atime, attr.full_date));
	    break;
	    
	case 'A':
	    (void) printf("%10u", buf->st_atime);
	    break;

#if BSD
	case 'b':
	    (void) printf("%10d", buf->st_blocks);
	    break;
	    
	case 'B':
	    (void) printf("%6d", buf->st_blksize);
	    break;
#endif
	    
	case 'c':
	    (void) printf(ls_date(buf->st_ctime, attr.full_date));
	    break;
	    
	case 'C':
	    (void) printf("%10u", buf->st_ctime);
	    break;

	case 'd':
	    if ((p = rindex(name, '/')) == (char *) NULL)
		(void) putc('.', stdout);
	    else if (p == name)
		(void) putc('/', stdout);
	    else {
		*p = '\0';
		(void) printf(attr.graphic ? graphic(name) : name);
		*p = '/';
	    }
	    break;
	    
	case 'D':
	    (void) printf("%4d", depth);
	    break;
	    
	case 'g':
	    (void) printf("%8d", buf->st_gid);
	    break;
	    
	case 'G':
	    if ((gi = getgigid(buf->st_gid)) == (struct group_info *) NULL)
		(void) printf("%8d", buf->st_gid);
	    else
		(void) printf("%-8.8s", gi->name);
	    break;
	    
	case 'i':
	    (void) printf("%6d", buf->st_ino);
	    break;
	    
	case 'l':
	    (void) printf("%3d", buf->st_nlink);
	    break;
	    
	case 'm':
	    (void) printf(ls_date(buf->st_mtime, attr.full_date));
	    break;
	    
	case 'M':
	    (void) printf("%10u", buf->st_mtime);
	    break;

	case 'p':
	    (void) printf("%s%s%s%s", 
			  ftype[ftype_index(buf->st_mode)],
			  perm[user_index(buf->st_mode)],
			  perm[group_index(buf->st_mode)],
			  perm2[all_index(buf->st_mode)]);
	    break;
	    
	case 'P':
	    (void) printf("%#6o", buf->st_mode);
	    break;
	    
	case 'r':
	    if (S_ISCHR(buf->st_mode) || S_ISBLK(buf->st_mode))
		(void) printf("%4u", minor(buf->st_rdev));
	    else
		(void) printf("xxxx");
	    break;
	    
	case 'R':
	    if (S_ISCHR(buf->st_mode) || S_ISBLK(buf->st_mode))
		(void) printf("%4u", major(buf->st_rdev));
	    else
		(void) printf("xxxx");
	    break;
	    
	case 'n':
	case 's':
	    (void) printf(attr.graphic ? graphic(name) : name);
	    break;
	    
	case 'N':
	case 'S':
	    if ((p = rindex(name, '/')) == (char *) NULL)
		(void) printf(attr.graphic ? graphic(name) : name);
	    else
		(void) printf(attr.graphic ? graphic(p + 1) : (p + 1));
	    break;
	    
	case 'u':
	    (void) printf("%8d", buf->st_uid);
	    break;
	    
	case 'U':
	    if ((ui = getuiuid(buf->st_uid)) == (struct user_info *) NULL)
		(void) printf("%8d", buf->st_uid);
	    else
		(void) printf("%-8.8s", ui->name);
	    break;
	    
	case 'w':
	    (void) printf("%02x", (unsigned char) minor(buf->st_dev));
	    break;
	    
	case 'W':
	    (void) printf("%02x", (unsigned char) major(buf->st_dev));
	    break;
	    
	case 'y':
	    (void) printf("%04x", (unsigned short) buf->st_dev);
	    break;
	    
	case 'z':
	    (void) printf("%9d", buf->st_size);
	    break;
	    
	default:
	    (void) putc(*fp, stdout);
	    break;
	}
    }
    
    (void) putc('\n', stdout);
    return;
}

/* ----------------------------------------------------------------------
 * printentry:
 *	Display filename,permissions and size in a '/bin/ls' like
 *	format. If verbose is non-zero then more information is
 *	displayed.
 * uses the macros:
 *	user_index(b)
 *	group_index(b)
 *	all_index(b)
 *	ftype_index(b)
 *
 */

void printentry(verbose, str_owner, full_date, buf, name)
int verbose;
int str_owner;
int full_date;
struct stat *buf;
char *name;
{
    char gid_buf[20];
    char uid_buf[20];
    struct group_info *gi;
    struct user_info *ui;
    
    
    if (verbose) {
	if (str_owner) {
	    if ((ui = getuiuid(buf->st_uid)) == (struct user_info *) NULL)
		(void) sprintf(uid_buf, "%8d", buf->st_uid);
	    else
		(void) sprintf(uid_buf, "%-8.8s", ui->name);
	    if ((gi = getgigid(buf->st_gid)) == (struct group_info *) NULL)
		(void) sprintf(gid_buf, "%8d", buf->st_gid);
	    else
		(void) sprintf(gid_buf, "%-8.8s", gi->name);
	}
	else {
	    (void) sprintf(uid_buf, "%8d", buf->st_uid);
	    (void) sprintf(gid_buf, "%8d", buf->st_gid);
	}
	if (S_ISCHR(buf->st_mode) || S_ISBLK(buf->st_mode))
	    (void) printf("%s%s%s%s %s %s %3d,%3d %s %-s\n",
			  ftype[ftype_index(buf->st_mode)],
			  perm[user_index(buf->st_mode)],
			  perm[group_index(buf->st_mode)],
			  perm2[all_index(buf->st_mode)],
			  uid_buf,
			  gid_buf,
			  major(buf->st_rdev),
			  minor(buf->st_rdev),
			  ls_date(buf->st_mtime, full_date),
			  attr.graphic ? graphic(name) : name);
	else
	    (void) printf("%s%s%s%s %s %s %6d %s %-s\n",
			  ftype[ftype_index(buf->st_mode)],
			  perm[user_index(buf->st_mode)],
			  perm[group_index(buf->st_mode)],
			  perm2[all_index(buf->st_mode)],
			  uid_buf,
			  gid_buf,
			  buf->st_size,
			  ls_date(buf->st_mtime, full_date),
			  attr.graphic ? graphic(name) : name);
	
    }
    else {
	if (S_ISCHR(buf->st_mode) || S_ISBLK(buf->st_mode))
	    (void) printf("%s%s%s%s %3d,%3d %-s\n",
			  ftype[ftype_index(buf->st_mode)],
			  perm[user_index(buf->st_mode)],
			  perm[group_index(buf->st_mode)],
			  perm2[all_index(buf->st_mode)],
			  major(buf->st_rdev),
			  minor(buf->st_rdev),
			  attr.graphic ? graphic(name) : name);
	else
	    (void) printf("%s%s%s%s %9d %-s\n",
			  ftype[ftype_index(buf->st_mode)],
			  perm[user_index(buf->st_mode)],
			  perm[group_index(buf->st_mode)],
			  perm2[all_index(buf->st_mode)],
			  buf->st_size,
			  attr.graphic ? graphic(name) : name);
    }
    
    return;
}

static int handle_file()
{
    struct stat	statbuf;

#ifdef S_IFLNK
    if (lstat(attr.fname, attr.buf) < 0) {
	warning("%s: %m", attr.fname);
	return 0;
    }
    if ((attr.follow_dir || attr.follow_file) && S_ISLNK(attr.buf->st_mode)) {
	if (stat(attr.fname, &statbuf) == 0) {
	    if (S_ISDIR(statbuf.st_mode)) {
		if (attr.follow_dir) {
		    int   len;
		    char  linkbuf[MAXPATHLEN];

		    len = readlink(attr.fname, linkbuf, sizeof(linkbuf));
		    if (len != 1 || *linkbuf != '.')
			*attr.buf = statbuf;
		}
	    }
	    else if (attr.follow_file)
		*attr.buf = statbuf;
	}
    }
#else
    if (stat(attr.fname, attr.buf) < 0) {
	warning("%s: %m", attr.fname);
	return 0;
    }
#endif

    (*(attr.func))();
    return 1;
}

/* ----------------------------------------------------------------------
 * fwt1:
 *	'p' points to the end of the string in attr.fname
 *
 *	2 versions of this routine currently live here:
 *	"new-style", for systems with a BSD or POSIX-style
 *	directory library, and systems without such a
 *	directory library. They both differ in
 *	the manner in which they access directories.
 *	Any changes needed to work on another system
 *	should only have to made for this routine.
 *
 *	Below is the "directory library" version of fwt1()
 *
 */

#if defined(POSIX_DIRECTORY_LIBRARY) || defined(BSD)

static void fwt1(depth, dashr, p, stat_all, nlink)
int depth;
int dashr;
char *p;
int stat_all;
int nlink;
{
    char *q;
    char *s;
    DIR *dirp;
#ifdef POSIX_DIRECTORY_LIBRARY
    struct dirent *dp;
#define DIRTYPE	struct dirent
#else
    struct direct *dp;
#define DIRTYPE struct direct
#endif
    
    
    if (depth == 0) {
	if (dashr)
	    warning("%s: file tree depth exceeded", attr.fname);
	return;
    }
    attr.depth++;
    
    if ((dirp = opendir(attr.fname)) == (DIR *) NULL) {
	warning("%s: %m", attr.fname);
	return;
    }
    
    nlink -= 2;

    if (stat_all) {
	if (attr.stat_all) {
	    for (dp = readdir(dirp);
		 dp != (DIRTYPE *) NULL;
		 dp = readdir(dirp)) {
		if (isdot(dp->d_name) || isdotdot(dp->d_name))
		    continue;
		s = p;
		q = dp->d_name;
		while ((*s++ = *q++) != '\0')		/*SUPPRESS 530*/
		    /* void */ ;
		s--;

		if (!handle_file())
		    continue;
		if (isproper(attr.buf->st_mode)) {
		    *s++ = '/';
		    *s = '\0';
		    fwt1(depth - 1, dashr, s,
			 attr.stat_all | hassubdir(attr.buf),
			 attr.buf->st_nlink);
		}
	    }
	}
	else {
	    for (dp = readdir(dirp);
		 dp != (DIRTYPE *) NULL;
		 dp = readdir(dirp)) {
		if (isdot(dp->d_name) || isdotdot(dp->d_name))
		    continue;
		s = p;
		q = dp->d_name;
		while ((*s++ = *q++) != '\0')		/*SUPPRESS 530*/
		    /* void */ ;
		s--;
		
		if (!handle_file())
		    continue;
		if (isproper(attr.buf->st_mode)) {
		    *s++ = '/';
		    *s = '\0';
		    fwt1(depth - 1, dashr, s,
			 attr.stat_all | hassubdir(attr.buf),
			 attr.buf->st_nlink);
		    nlink--;
		    if (nlink == 0)
			break;
		}
	    }
	    if (dp != (DIRTYPE *) NULL) {
		for (dp = readdir(dirp);
		     dp != (DIRTYPE *) NULL;
		     dp = readdir(dirp)) {
		    if (isdot(dp->d_name) || isdotdot(dp->d_name))
			continue;
		    s = p;
		    q = dp->d_name;
		    while ((*s++ = *q++) != '\0')	/*SUPPRESS 530*/
			/* void */ ;
		    (*(attr.func))();
		}
	    }
	}
    }
    else {
	for (dp = readdir(dirp); dp != (DIRTYPE *) NULL; dp = readdir(dirp)) {
	    if (isdot(dp->d_name) || isdotdot(dp->d_name))
		continue;
	    s = p;
	    q = dp->d_name;
	    while ((*s++ = *q++) != '\0')		/*SUPPRESS 530*/
		/* void */ ;
	    (*(attr.func))();
	}
    }
    
    (void) closedir(dirp);
    attr.depth--;
    attr.prune = FALSE;
    *p = '\0';
    
    return;
}
#else

/* ----------------------------------------------------------------------
 * fwt1:
 *	This function does the same thing as fwt1() above, but is
 *	meant for systems without a directory library, that does
 *	directory reading "by hand".
 *
 *    Below is the "no directory library" version of fwt1()
 *
 */

static void fwt1(depth, dashr, p)
int depth;
int dashr;
char *p;
{
    char *q;
    char *s;
    FILE *dirp;
    struct direct dp;
    
    if (depth == 0) {
	if (dashr)
	    warning("%s: file tree depth exceeded", attr.fname);
	return;
    }
    attr.depth++;
    
    if ((dirp = fopen(attr.fname, "r")) == (FILE *) NULL) {
	warning("%s: %m", attr.fname);
	return;
    }
    
    while (fread(&dp, sizeof(struct direct), 1, dirp) == 1) {
	if (isdot(dp.d_name) || isdotdot(dp.d_name) || dp.d_ino==0)
	    continue;
	s = p;
	q = dp.d_name;
	while ((*s++ = *q++) != '\0')
	    /* void */; 
	s--;
	
	if (!handle_file())
	    continue;
	if (isproper(attr.buf->st_mode)) {
	    *s++ = '/';
	    *s = '\0';
	    fwt1(depth - 1, dashr, s);
	}
    }
    (void) fclose(dirp);
    attr.depth--;
    attr.prune = FALSE;
    *p = '\0';
    return;
}

#endif

/* ----------------------------------------------------------------------
 * ftrw:
 *	Entry point to do the search, ftrw is a front end
 *	to the recursive fwt1.
 *	ftrw() initializes some global variables and
 *	builds the initial filename string which is passed to
 *	fwt1().
 */

void ftrw(file, fn, depth, dashr)
char *file;
void (*fn)();
int depth;
int dashr;
{
    char *name_end;
    char filebuf[MAXPATHLEN + 1];
    struct stat statbuf;
    
    
    attr.prune = FALSE;
    attr.depth = 0;
    attr.func = fn;
    name_end = attr.fname = filebuf;
    attr.buf = &statbuf;

    if (*file == '\0') {
	*name_end = '\0';
	if (stat(".", attr.buf) == 0)
	    fwt1(depth, dashr, name_end,
		 attr.stat_all | hassubdir(attr.buf), attr.buf->st_nlink);
	else
	    warning(".: %m");
    }
    else {
	while ((*name_end++ = *file++) != '\0')		/*SUPPRESS 530*/
	    /* void */ ;
	name_end--;
	
	if (!handle_file())
	    return;
	
	if (*(name_end - 1) != '/') {
	    *name_end++ = '/';
	    *name_end = '\0';
	}
	if (isproper(attr.buf->st_mode))
	    fwt1(depth, dashr, name_end,
		 attr.stat_all | hassubdir(attr.buf), attr.buf->st_nlink);
    }
    
    return;
}
