/* ----------------------------------------------------------------------
 * FILE: rhparse.c
 * VERSION: 2
 * Written by: Ken Stauffer
 * VERSION 2.8
 * Enhancements and bug fixes by Rick Ohnemus
 * 
 * This contains the parser for the C expressions,
 * gettoken(), getit() and ungetit() routines.
 * expression(), expr(), exp0(), ... , factor()
 * locatename(), push(), find_macro()
 *
 *
 * ---------------------------------------------------------------------- */

#include "rh.h"
#include <sys/timeb.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#if defined(sun)
#include <mntent.h>
#endif

#if !defined(SUNOS_4)
extern char *strdup();
#endif

static int cpos = 0;		/* current character position */
static int lineno = 0;		/* current line number */

static void expr0();

/* ----------------------------------------------------------------------
 * getit:
 *	Return the next character, input is obtained from a file or
 *	a string.
 *	If expstr == NULL then input is from the file called 'expfname'
 *	with file pointer 'expfile'.
 *
 *	If expstr != NULL then input is from the string 'expstr'
 *
 */

static int getit()
{
    int c;
   

    if (expstr == (char *) NULL)
	c = getc(expfile);
    else if (*expstr == '\0')
	c = EOF;
    else
	c = *expstr++;
    if (c == '\n') {
	lineno++;
	cpos = 0;
    }
    cpos++;
    return(c);
}

/* ----------------------------------------------------------------------
 * ungetit:
 *	Unget a char.
 *	A character is ungotten using the same scheme as stated for
 *	getit() for determining where input is comming from.
 *
 */

static void ungetit(c)
int c;
{
    if (c == '\n') {
	lineno--;
	cpos = 1;
    }
    else
	cpos--;
    if (expstr == (char *) NULL)
	(void) ungetc(c, expfile);
    else if (c > 0)
	expstr--;
    return;
}

/* ----------------------------------------------------------------------
 * parse_error:
 *	Print an error message and quit.
 */
 
static void parse_error(s)
char *s;
{
    if (expstr == (char *) NULL)
	(void) fprintf(stderr, "%s: ", expfname);
    else
	(void) fprintf(stderr, "Command line: ");
    
    (void) fprintf(stderr, "line: %d, char: %d, %s.\n", lineno, cpos, s);
    exit(1);
}

/* ----------------------------------------------------------------------
 * insertname:
 *	Inserts the symbol named 's' with type 't' and value 'val'
 *	into the symbol table. Return the a pointer to the symbol
 *	table entry. The symbol is inserted into the head of the
 *	linked list. This behavior is relied upon elswhere.
 *
 */

struct symbol *insertname(s, t, val)
char *s;
int t;
long val;
{
    struct symbol *sym;


    sym = (struct symbol *) malloc(sizeof(struct symbol));
    if (sym == (struct symbol *) NULL)
	error("no more memory");
    if ((sym->name = strdup(s)) == (char *) NULL)
	error("no more memory");
    sym->type = t;
    sym->value = val;
    sym->req_stat = FALSE;
    sym->next = symbols;
    symbols = sym;
    
    return (sym);
}

/* ----------------------------------------------------------------------
 * locatename:
 *	Do a linear search for 's' in the linked list symbols.
 *
 */

struct symbol *locatename(s)
char *s;
{
    struct symbol *p;


    for (p = symbols; p != (struct symbol *) NULL; p = p->next)
	if (strcmp(s, p->name) == 0)
	    return (p);
    return ((struct symbol *) NULL);
}

/* ----------------------------------------------------------------------
 * push:
 *	"assemble" the instruction into the StackProgram[] array.
 *
 */

static void push(func, val)
void (*func)();
long val;
{
    if (PC >= LENGTH)
	parse_error("program to big");
    StackProgram[PC].func = func;
    StackProgram[PC++].value = val;
    return;
}

/* ----------------------------------------------------------------------
 * gettoken:
 *	Return the next token.
 *	global variable: tokenval will contain any extra
 *	attribute associated with the returned token, ie
 *	the VALUE of a number, the index of the string etc...
 *	tokensym will be a pointer to the symbol table entry for
 *	any symbol encountered.
 *
 */
 
static long gettoken()
{
    char buf[IDLENGTH + 1];
    char *bufp = buf;
    int c = getit();
    int incomment = FALSE;
    

    while (c == ' ' || c == '\t' || c == '\n' || c == '/' || incomment) {
	if (c == '/' && !incomment) {
	    c = getit();
	    if (c == '*') {
		incomment = TRUE;
		c = getit();
	    }
	    else {
		ungetit(c);
		c = '/';
		break;
	    }
	}
	else if (c == '*') {
	    c = getit();
	    if (c == '/') {
		incomment = FALSE;
		c = getit();
	    }
	}
	else
	    c = getit();
    }
    
    if (c == '0') {
	tokenval = 0;
	c = getit();
	if (c == 'x' || c == 'X') {
	    int flag = 1;
	    while (isxdigit(c = getit())) {
		flag = 0;
		tokenval <<= 4;
		if (isupper(c))
		    c = tolower(c);
		tokenval += strchr(hex_digits, c) - hex_digits;
	    }
	    if (flag)
		parse_error("bad hex constant");
	}
	else {
	    for (/* void */; c >= '0' && c <= '7'; c = getit()) {
		tokenval <<= 3;
		tokenval += c - '0';
	    }
	    if (isdigit(c))
		parse_error("bad octal constant");
	}
	ungetit(c);
	return (NUMBER);
    }
    
    if (isdigit(c)) {
	tokenval = c - '0';
	while (isdigit((c = getit()))) {
	    tokenval *= 10;
	    tokenval += c - '0';
	}
	if (c == 'K')			/* kilo- */
	    tokenval *= 1024;
	else if (c == 'M')		/* meg- */
	    tokenval *= 1024 * 1024;
	else if (c == 'G')		/* giga- */
	    tokenval *= 1024 * 1024 * 1024;
	else if (c == 'm')		/* minutes */
	    tokenval *= 60;
	else if (c == 'h')		/* hours */
	    tokenval *= 60 * 60;
	else if (c == 'd')		/* days */
	    tokenval *= 60 * 60 * 24;
	else if (c == 'w')		/* weeks */
	    tokenval *= 60 * 60 * 24 * 7;
	else
	    ungetit(c);
	return (NUMBER);
    }
	
    if (isalpha(c)) {
	int count = 0;
	do {
	    if (count++ < IDLENGTH)
		*bufp++ = c;
	    else
		parse_error("identifier too long");
	    c = getit();
	} while (isalnum(c));
	ungetit(c);
	*bufp = '\0';
	if ((tokensym = locatename(buf)) == (struct symbol *) NULL)
	    tokensym = insertname(buf, IDENTIFIER, 0L);
	/* ==================== */
	/* actually check program after it is built in rh.c */
	/* look for anything from table that requires stat() */
	if (!attr.stat_all)
	    attr.stat_all = tokensym->req_stat;
	tokenval = tokensym->value;
	return (tokensym->type);
    }

    if (c == '"') {
	char *end;
	char *p;
	char strbuf[MAXNAMLEN + 1];
	
	p = strbuf;
	end = strbuf + (sizeof(strbuf) - 1);
	
	while ((c = getit()) != '"' && c != EOF) {
	    if (p == end)
		parse_error("file name pattern too long");
	    *p++ = c;
	}
	if (c == EOF)
	    parse_error("terminating '\"' missing in file name pattern");
	*p = '\0';
	if (strbuf[0] == '\0')
	    parse_error("empty file name pattern");
	if ((p = strdup(strbuf)) == (char *) NULL)
	    error("no more memory");
	tokenval = (long) p;
	return (STR);
    }
    
    if (c == '`') {
#if defined(sun)
	char *end;
	char *p;
	char strbuf[MNTMAXSTR + 1];
	
	p = strbuf;
	end = strbuf + (sizeof(strbuf) - 1);
	
	while ((c = getit()) != EOF && isalnum(c) || c == '.') {
	    if (p == end)
		parse_error("filesystem type too long");
	    *p++ = c;
	}
	ungetit(c);
	*p = '\0';
	if (strbuf[0] == '\0')
	    parse_error("empty filesystem type");
	if (!mounted_fstype(strbuf))
	    p = (char *) NULL;
	else if ((p = strdup(strbuf)) == (char *) NULL)
	    error("no more memory");
	tokenval = (long) p;
	return (FSTYPE);
#else
	parse_error("filesystem types not supported");
#endif
    }
    
    if (c == '=') {
	c = getit();
	if (c == '=')
	    return (EQ);
	else {
	    ungetit(c);
	    return ('=');
	}
    }
    
    if (c == '$') {
	int count=0;
	struct user_info *info;
	
	c = getit();
	if (c == EOF)
	    parse_error("user name missing after $");
	if (c == '$') {
	    tokenval = getuid();
	    return (NUMBER);
	}
	do {
	    if (count++ < IDLENGTH)
		*bufp++ = c;
	    else
		parse_error("user name too long");
	    c = getit();
	} while (isalnum(c));
	ungetit(c);
	*bufp='\0';
	if ((info = getuinam(buf)) == (struct user_info *) NULL) 
	    parse_error("no such user");
	tokenval = info->uid;
	return (NUMBER);
    }
    
    if (c == '@') {
	int count=0;
	struct group_info *info;
	
	c = getit();
	if (c == EOF)
	    parse_error("group name missing after @");
	if (c == '@') {
	    tokenval = getgid();
	    return (NUMBER);
	}
	do {
	    if (count++ < IDLENGTH)
		*bufp++ = c;
	    else
		parse_error("group name too long");
	    c = getit();
	} while (isalnum(c));
	ungetit(c);
	*bufp = '\0';
	if ((info = getginam(buf)) == (struct group_info *) NULL) 
	    parse_error("no such group");
	tokenval = info->gid;
	return (NUMBER);
    }
    
    if (c == '!') {
	c = getit();
	if (c == '=')
	    return (NE);
	ungetit(c);
	return ('!');
    }
    
    if (c == '>') {
	c = getit();
	if (c == '=')
	    return (GE);
	if (c == '>')
	    return (SHIFTR);
	ungetit(c);
	return ('>');
    }
    
    if (c == '<') {
	c = getit();
	if (c == '=')
	    return (LE);
	if (c == '<')
	    return (SHIFTL);
	ungetit(c);
	return ('<');
    }
    
    if (c == '&') {
	c = getit();
	if (c == '&')
	    return (AND);
	ungetit(c);
	return ('&');
    }
    
    if (c == '|') {
	c = getit();
	if (c == '|')
	    return (OR);
	ungetit(c);
	return ('|');
    }
    
    if (c == '[') {
	char *end;
	char *p;
	char datebuf[MAXNAMLEN + 1];
	
	p = datebuf;
	end = datebuf + (sizeof(datebuf) - 1);
	while ((c = getit()) != ']' && c != EOF) {
	    if (p == end)
		parse_error("date specification too long");
	    *p++ = c;
	}
	if (c == EOF)
	    parse_error("terminating ']' missing in date specification");
	*p = '\0';
	if (datebuf[0] == '\0')
	    parse_error("empty file name pattern");
	if ((p = strdup(datebuf)) == (char *) NULL)
	    error("no more memory");
	tokenval = (long) p;
	return (DATESPEC);
    }

    return (c);
}

/* ----------------------------------------------------------------------
 * expression:
 *	Parse an expression. (top-level routine)
 *	OPERATOR ?:
 *
 */

static void expression()
{
    int qm;
    int colon;
    
    
    expr0();
    if (token == '?') {
	token = gettoken();
	qm = PC;
	push(c_qm, 0L);
	expression();
	if (token != ':')
	    parse_error("missing ':'");
	token = gettoken();
	colon = PC;
	push(c_colon, 0L);
	expression();
	
	StackProgram[qm].value = colon;
	StackProgram[colon].value = PC - 1;
    }
    return;
}

/* ----------------------------------------------------------------------
 * explist:
 *	argc is the number of arguments expected.
 *	Parse an expression list of the form:
 *		<explist> ==> ( <exps> )
 *			| ( )
 *			| empty
 *
 *		<exps> ==> <exps> , <expression>
 *			| <expression>
 *
 */

static void explist(argc)
long argc;
{
    if (token != '(' && argc == 0)
	return;
    
    if (token != '(')
	parse_error("missing '('");
    token = gettoken();
    
    if (argc == 0 && token == ')') {
	token = gettoken();
	return;
    }
    
    for (;;) {
	expression();
	argc--;
	if (token == ')')
	    break;
	if (token != ',')
	    parse_error("missing ','");
	token = gettoken();
    }
    
    token = gettoken();
    if (argc)
	parse_error("wrong number of arguments");
    return;
}	

/* ----------------------------------------------------------------------
 * factor:
 *	Parse a factor. Could be a number, variable, date, function call or
 *	regular expression string.
 */
 
static void factor()
{
    extern time_t	getdate();
    long		pc;
    long		seconds;
    
    
    switch (token) {
	
    case '(':
	token = gettoken();
	expression();
	if (token != ')')
	    parse_error("missing ')'");
	token = gettoken();
	break;
	
    case NUMBER:
	push(c_number, tokenval);
	token = gettoken();
	break;
	
    case FUNCTION:
	pc = tokensym->value;
	token = gettoken();
	explist(StackProgram[pc].value);
	push(c_func, pc);
	break;
	
    case PARAM:
	push(c_param, tokensym->value);
	token = gettoken();
	break;
	
    case BLTIN:
    case FIELD:
	push(tokensym->func, tokenval);
	token = gettoken();
	break;
	
    case DATESPEC:
	seconds = (long) getdate((char *) tokenval, (struct timeb *) NULL);
	if (seconds == -1) {
	    cpos -= strlen((char *) tokenval) + 1;
	    free((char *) tokenval);
	    parse_error("invalid date specification");
	}
	push(c_number, seconds);
	free((char *) tokenval);
	token = gettoken();
	break;
	
    case STR:
	push(c_str, tokenval);
	token = gettoken();
	break;
	
    case FSTYPE:
	push(c_fstype, tokenval);
	token = gettoken();
	break;
	
    case IDENTIFIER:
	parse_error("undefined identifier");
	
    default:
	parse_error("syntax error");
    }
    
    return;
}

/* OPERATOR ~ ! - */ 
static void expr10()
{
    if (token == '!') {
	token = gettoken();
	expr10();
	push(c_not, 0L);
    }
    else if (token == '~') {
	token = gettoken();
	expr10();
	push(c_bnot, 0L);
    }
    else if (token == '-') {
	token = gettoken();
	expr10();
	push(c_uniminus, 0L);
    }
    else
	factor();
    return;
}

/* OPERATOR * / % */
static void expr9()
{
    expr10();
    while (token) {
	if (token == '*') {
	    token = gettoken();
	    expr10();
	    push(c_mul, 0L);
	}
	else if (token == '/') {
	    token = gettoken();
	    expr10();
	    push(c_div, 0L);
	}
	else if (token == '%') {
	    token = gettoken();
	    expr10();
	    push(c_mod, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR + - */
static void expr8()
{
    expr9();
    while (token) {
	if (token == '+') {
	    token = gettoken();
	    expr9();
	    push(c_plus, 0L);
	}
	else if (token == '-') {
	    token = gettoken();
	    expr9();
	    push(c_minus, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR << >> */
static void expr7()
{
    expr8();
    while (token) {
	if (token == SHIFTL) {
	    token = gettoken();
	    expr8();
	    push(c_lshift, 0L);
	}
	else if (token == SHIFTR) {
	    token = gettoken();
	    expr8();
	    push(c_rshift, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR < <= > >= */
static void expr6()
{
    expr7();
    while (token) {
	if (token == LE) {
	    token = gettoken();
	    expr7();
	    push(c_le, 0L);
	}
	else if (token == GE) {
	    token = gettoken();
	    expr7();
	    push(c_ge, 0L);
	}
	else if (token == '>') {
	    token = gettoken();
	    expr7();
	    push(c_gt, 0L);
	}
	else if (token == '<') {
	    token = gettoken();
	    expr7();
	    push(c_lt, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR == != */
static void expr5()
{
    expr6();
    while (token) {
	if (token == EQ) {
	    token = gettoken();
	    expr6();
	    push(c_eq, 0L);
	}
	else if (token == NE) {
	    token = gettoken();
	    expr6();
	    push(c_ne, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR & */
static void expr4()
{
    expr5();
    for (;;) {
	if (token == '&') {
	    token = gettoken();
	    expr5();
	    push(c_band, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR ^ */
static void expr3()
{
    expr4();
    for (;;) {
	if (token == '^') {
	    token = gettoken();
	    expr4();
	    push(c_bxor, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR | */
static void expr2()
{
    expr3();
    for (;;) {
	if (token == '|') {
	    token = gettoken();
	    expr3();
	    push(c_bor, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR && */ 
static void expr1()
{
    expr2();
    for (;;) {
	if (token == AND) {
	    token = gettoken();
	    expr2();
	    push(c_and, 0L);
	}
	else
	    break;
    }
    return;
}

/* OPERATOR || */ 
static void expr0()
{
    expr1();
    for (;;) {
	if (token == OR) {
	    token = gettoken();
	    expr1();
	    push(c_or, 0L);
	}
	else
	    break;
    }
    return;
}

/* ----------------------------------------------------------------------
 * idlist:
 *	Return the maximum offset obtained in parsing the parameter list.
 *	<id-list> ==> ( <ids> )
 *		| ()
 *		| empty
 *
 *	<ids> ==> IDENTIFIER <idtail>
 *	<idtail> ==> <ids> , <idtail>
 *		| empty
 */

static long idlist()
{
    long offset = 0;
    
    
    if (token == '(')
	token = gettoken();
    else if (token == '{')
	return (0L);
    else
	parse_error("expected '(' or '{'");
    
    if (token == ')') {
	token = gettoken();
	return (0L);
    }
    
    for (;;) {
	if (token != IDENTIFIER)
	    parse_error("identifier expected");
	tokensym->type = PARAM;
	tokensym->func = c_param;
	tokensym->value = offset++;
	token = gettoken();
	if (token == ')')
	    break;
	if (token != ',')
	    parse_error("expected ')'");
	token = gettoken();
    }
    
    token = gettoken();
    return (offset);
}

/* ----------------------------------------------------------------------
 * function:
 *	parse a function definition. Grammer for a function is:
 *	<function> ==> IDENTIFIER <id-list> { RETURN <expression> ; }
 *
 *	<id-list> ==> ( <ids> )
 *			| ( )
 *			| empty
 *
 *	<ids> ==> IDENTIFIER <idtail>
 *
 *	<idtail> ==> , <ids>
 *		| empty
 *
 */

static void function()
{
    struct symbol *s;
    
    
    s = tokensym;
    tokensym->value = PC;
    tokensym->type = FUNCTION;
    tokensym->req_stat = FALSE;
    tokensym->func = c_func;
    
    token = gettoken();
    
    push((void (*)()) NULL, idlist());	/* save number of args for function */
    
    if (token != '{')
	parse_error("expected '{'");
    token = gettoken();
    
    if (token != RETURN)
	parse_error("expected keyword: return");
    token = gettoken();
    
    expression();
    
    if (token != ';')
	parse_error("expected ';'");
    token = gettoken();
    
    push(c_return, StackProgram[s->value].value);
    
    /* free up the parameter symbols */
    while (symbols->type == PARAM) {
	s = symbols;
	symbols = symbols->next;
	free(s->name);
	free((char *) s);
    }
    
    if (token != '}')
	parse_error("expected '}'");
    token = gettoken();
    return;
}

/* ----------------------------------------------------------------------
 * program:
 *	Parse a program of the form:
 *		<program> ==> <function-list> <expression> EOF
 *			| <function-list> EOF
 *			| <function-list> <expression> ;
 *
 *		<function-list> ==> <function> <function-list>
 *				| empty
 */

void program()
{
    cpos = 0;
    lineno = 1;
    
    token = gettoken();
    
    for (;;) {
	if (token != IDENTIFIER)
	    break;
	function();
    }
    
    if (token != EOF) {
	startPC = PC;
	expression();
	push((void (*)()) NULL, 0L);
    }
    
    if (token != EOF && token != ';')
	parse_error("EOF expected");
    
    return;
}
