/*
**  Do shell-style pattern matching for ?, \, [], and * characters.
**  Might not be robust in face of malformed patterns; e.g., "foo[a-"
**  could cause a segmentation violation.  It is 8bit clean.
**
**  Written by Rich $alz, mirror!rs, Wed Nov 26 19:03:17 EST 1986.
**  Rich $alz is now <rsalz@bbn.com>.
**  Special thanks to Lars Mathiesen <thorinn@diku.dk> for the ABORT code.
**  This can greatly speed up failing wildcard patterns.  For example:
**	pattern: -*-*-*-*-*-*-12-*-*-*-m-*-*-*
**	text 1:	 -adobe-courier-bold-o-normal--12-120-75-75-m-70-iso8859-1
**	text 2:	 -adobe-courier-bold-o-normal--12-120-75-75-X-70-iso8859-1
**  Text 1 matches with 51 calls, while text 2 fails with 54 calls.  Without
**  the ABORT, then it takes 22310 calls to fail.  Ugh.
**
**  bernie    613-01 91/01/04 19:34 
**	Fixed problem with terminating * not matching with (null)
**
**  bernie    597-00 91/01/08 11:24 
**	Fixed shell glob negate from '^' to '!'
**
**  bernie    597-02 91/01/21 13:43 
**	Fixed . matching * or ? on first char.
**
**  bernie      1-00 91/02/14 10:28 
**	Fixed Star return on ABORT
**
**  rick@sparky.IMD.Sterling.COM, Tue Mar 19 10:22:05 CST 1991
**	Changed negate class character back to '^'.
**	Fixed character class handling so it will work with malformed patterns.
**	Added code for '.' being special at beginning of file name.
**	Added code for ignoring case of characters.
*/

#define TRUE		1
#define FALSE		0
#define ABORT		-1

#define NEGATE_CLASS	'^'

static unsigned char charmap[] = {
    '\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
    '\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
    '\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
    '\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
    '\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
    '\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
    '\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
    '\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
    '\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
    '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
    '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
    '\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
    '\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
    '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
    '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
    '\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
    '\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
    '\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
    '\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
    '\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
    '\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
    '\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
    '\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
    '\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
    '\300', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
    '\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
    '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
    '\370', '\371', '\372', '\333', '\334', '\335', '\336', '\337',
    '\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
    '\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
    '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
    '\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};

static int Star(text, pattern, igncase)
register unsigned char	*text;
register unsigned char	*pattern;
int			igncase;
{
    register int stat;

    /* gobble up * match */
    while ((stat = DoMatch(text, pattern, igncase)) == FALSE)
	if (*++text == '\0')
	    return ABORT;
    return stat;
}


/* match string "text" to pattern "pattern" */

static int DoMatch(text, pattern, igncase)
register unsigned char	*text;
register unsigned char	*pattern;
int			igncase;
{
    register int		last;
    register int 	 	matched;
    register int 	 	reverse;
    register unsigned char	*tp;
    register unsigned char	*class_end;
    register unsigned char	*cmap = charmap;
    
    /* parse the string to end */
    for (/* void */; *pattern != '\0'; text++, pattern++) {
	
	if (*text == '\0')
	    return (*pattern == '*' && *++pattern == '\0') ? TRUE : ABORT;

	switch (*pattern) { /* parse pattern */
	    
	case '\\':
	    /* Literal match with following character. */
	    if (*(pattern + 1) != '\0')
		pattern++;
	    /* FALLTHROUGH */
	    
	default: /*literal match*/
	    if (igncase) {
		if (cmap[*text] != cmap[*pattern])
		    return FALSE;
	    }
	    else if (*text != *pattern)
		return FALSE;
	    continue;
	    
	case '?':
	    /* Match anything. */
	    continue;
	    
	case '*':
	    /* Trailing star matches everything. */
	    return (*++pattern == '\0') ? TRUE : Star(text, pattern, igncase);
	    
	case '[':
	    for (tp = ++pattern; *tp != '\0' && *tp != ']'; tp++) {
		if (*tp == '\\') {
		    if (*++tp == '\0')
			break;
		}
	    }
	    if (*tp == '\0' || tp == pattern) {
		/* Invalid character class so just do match against '[' */
		if (*text != '[')
		    return FALSE;
		pattern--;
		continue;
	    }
	    class_end = tp;
	    
	    /* inverse character class. */
	    if (*pattern == NEGATE_CLASS && *(pattern + 1) != ']') {
		reverse = TRUE;
		pattern++;
	    }
	    else
		reverse = FALSE;
	    
	    tp = pattern;
            last = *pattern;
	    for (matched = FALSE;
		 *pattern != '\0' && *pattern != ']';
		 last = *pattern, pattern++){
		if (*pattern == '-'
		    && pattern != tp
		    && *(pattern + 1) != ']') {
		    if (*++pattern == '\\')
			pattern++;
		    if (igncase) {
			if (cmap[*text] <= cmap[*pattern]
			    && cmap[*text] >= cmap[last]) {
			    pattern = class_end;
			    matched = TRUE;
			    break;
			}
		    }
		    else if (*text <= *pattern && *text >= last) {
			pattern = class_end;
			matched = TRUE;
			break;
		    }
		}
		else {
		    if (*pattern == '\\')
			pattern++;
		    if (igncase) {
			if (cmap[*text] == cmap[*pattern]) {
			    pattern = class_end;
			    matched = TRUE;
			    break;
			}
		    }
		    else if (*text == *pattern) {
			pattern = class_end;
			matched = TRUE;
			break;
		    }
		}
	    }
	    
	    if (matched == reverse)
		return FALSE;
	    continue;
	}
    }
    
    return *text == '\0';
}


int wildmat(text, pattern, dotspec, igncase)
register unsigned char	*text;
register unsigned char	*pattern;
int			dotspec;
int			igncase;
{
    if (dotspec && *text == '.' && *pattern != '.')
	return FALSE;
    else
	return DoMatch(text, pattern, igncase) == TRUE;
}
