/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1996, 1997, 1998, 1999
 *	Sleepycat Software.  All rights reserved.
 */

#include "db_config.h"

#ifndef lint
static const char revid[] = "$Id: bt_curadj.c,v 1.1.1.5.2.2 2000/02/08 00:43:09 noriko Exp $";
#endif /* not lint */

#ifndef NO_SYSTEM_INCLUDES
#include <sys/types.h>
#endif

#include "db_int.h"
#include "db_page.h"
#include "btree.h"

#ifdef DEBUG
/*
 * __bam_cprint --
 *	Display the current internal cursor.
 *
 * PUBLIC: void __bam_cprint __P((DBC *));
 */
void
__bam_cprint(dbc)
	DBC *dbc;
{
	BTREE_CURSOR *cp;

	cp = dbc->internal;

	fprintf(stderr, "\tinternal: ovflsize: %lu", (u_long)cp->ovflsize);
	if (dbc->dbtype == DB_RECNO)
		fprintf(stderr, " recno: %lu", (u_long)cp->recno);
	if (F_ISSET(cp, C_DELETED))
		fprintf(stderr, " (deleted)");
	fprintf(stderr, "\n");
}
#endif

/*
 * __bam_ca_delete --
 *	Update the cursors when items are deleted and when already deleted
 *	items are overwritten.  Return the number of relevant cursors found.
 *
 * PUBLIC: int __bam_ca_delete __P((DB *, db_pgno_t, u_int32_t, int));
 */
int
__bam_ca_delete(dbp, pgno, indx, delete)
	DB *dbp;
	db_pgno_t pgno;
	u_int32_t indx;
	int delete;
{
	BTREE_CURSOR *cp;
	DBC *dbc;
	int count;		/* !!!: Has to contain max number of cursors. */

	/*
	 * Adjust the cursors.  We don't have to review the cursors for any
	 * thread of control other than the current one, because we have the
	 * page write locked at this point, and any other thread of control
	 * had better be using a different locker ID, meaning only cursors in
	 * our thread of control can be on the page.
	 *
	 * It's possible for multiple cursors within the thread to have write
	 * locks on the same page, but, cursors within a thread must be single
	 * threaded, so all we're locking here is the cursor linked list.
	 */
	MUTEX_THREAD_LOCK(dbp->mutexp);
	for (count = 0, dbc = TAILQ_FIRST(&dbp->active_queue);
	    dbc != NULL; dbc = TAILQ_NEXT(dbc, links)) {
		if (dbc->pgno == pgno && dbc->indx == indx) {
			cp = dbc->internal;
			if (delete)
				F_SET(cp, C_DELETED);
			else
				F_CLR(cp, C_DELETED);
			++count;
		}
	}
	MUTEX_THREAD_UNLOCK(dbp->mutexp);

	return (count);
}

/*
 * __ram_ca_delete --
 *	Return the number of relevant cursors.
 *
 * PUBLIC: int __ram_ca_delete __P((DB *, db_pgno_t));
 */
int
__ram_ca_delete(dbp, root_pgno)
	DB *dbp;
	db_pgno_t root_pgno;
{
	DBC *dbc;

	/*
	 * Review the cursors.  See the comment in __bam_ca_delete().
	 */
	MUTEX_THREAD_LOCK(dbp->mutexp);
	for (dbc = TAILQ_FIRST(&dbp->active_queue);
	    dbc != NULL; dbc = TAILQ_NEXT(dbc, links))
		if (dbc->root == root_pgno)
			break;
	MUTEX_THREAD_UNLOCK(dbp->mutexp);
	return (dbc == NULL ? 0 : 1);
}

/*
 * __bam_ca_di --
 *	Adjust the cursors during a delete or insert.
 *
 * PUBLIC: void __bam_ca_di __P((DB *, db_pgno_t, u_int32_t, int));
 */
void
__bam_ca_di(dbp, pgno, indx, adjust)
	DB *dbp;
	db_pgno_t pgno;
	u_int32_t indx;
	int adjust;
{
	DBC *dbc;

	/*
	 * Adjust the cursors.  See the comment in __bam_ca_delete().
	 */
	MUTEX_THREAD_LOCK(dbp->mutexp);
	for (dbc = TAILQ_FIRST(&dbp->active_queue);
	    dbc != NULL; dbc = TAILQ_NEXT(dbc, links)) {
		if (dbc->dbtype == DB_RECNO)
			continue;
		if (dbc->pgno == pgno && dbc->indx >= indx) {
			/* Cursor indices should never be negative. */
			DB_ASSERT(dbc->indx != 0 || adjust > 0);

			dbc->indx += adjust;
		}
	}
	MUTEX_THREAD_UNLOCK(dbp->mutexp);
}

/*
 * __bam_ca_dup --
 *	Adjust the cursors when moving items from a leaf page to a duplicates
 *	page.
 *
 * PUBLIC: int __bam_ca_dup __P((DB *,
 * PUBLIC:    db_pgno_t, u_int32_t, u_int32_t, db_pgno_t, u_int32_t));
 */
int
__bam_ca_dup(dbp, first, fpgno, fi, tpgno, ti)
	DB *dbp;
	db_pgno_t fpgno, tpgno;
	u_int32_t first, fi, ti;
{
	BTREE_CURSOR *cp, *orig_cp;
	DBC *dbc, *dbc_nopd;
	int ret;

	/*
	 * Adjust the cursors.  See the comment in __bam_ca_delete().
	 */
loop:
	MUTEX_THREAD_LOCK(dbp->mutexp);
	for (dbc = TAILQ_FIRST(&dbp->active_queue);
	    dbc != NULL; dbc = TAILQ_NEXT(dbc, links)) {
		/*
		 * Ignore matching entries that have already been moved,
		 * we move from the same location on the leaf page more
		 * than once.
		 */
		if (dbc->opd != NULL || dbc->pgno != fpgno || dbc->indx != fi)
			continue;

		/*
		 * Allocate a new cursor and create the stack.  If duplicates
		 * are sorted, we've just created an off-page duplicate Btree.
		 * If duplicates aren't sorted, we've just created a Recno tree.
		 */
		MUTEX_THREAD_UNLOCK(dbp->mutexp);
		if ((ret = __db_icursor(dbp, dbc->txn,
		    dbp->dup_compare == NULL ? DB_RECNO : DB_BTREE,
		    tpgno, 1, &dbc_nopd)) != 0)
			return (ret);

		dbc_nopd->pgno = tpgno;
		dbc_nopd->indx = ti;

		cp = dbc_nopd->internal;
		if (dbp->dup_compare == NULL) {
			/*
			 * Converting to off-page Recno trees is tricky.  The
			 * record number for the cursor is the index + 1 (to
			 * convert to 1-based record numbers).
			 */
			cp->recno = ti + 1;
		}

		/*
		 * Transfer the deleted flag from the top-level cursor to the
		 * created one.
		 */
		orig_cp = dbc->internal;
		if (F_ISSET(orig_cp, C_DELETED)) {
			F_SET(cp, C_DELETED);
			F_CLR(orig_cp, C_DELETED);
		}

		/* Stack the cursors and reset the initial cursor's index. */
		dbc->opd = dbc_nopd;
		dbc->indx = first;
		/* We released the MUTEX to get a cursor, start over. */
		goto loop;
	}
	MUTEX_THREAD_UNLOCK(dbp->mutexp);

	return (0);
}

/*
 * __bam_ca_rsplit --
 *	Adjust the cursors when doing reverse splits.
 *
 * PUBLIC: void __bam_ca_rsplit __P((DB *, db_pgno_t, db_pgno_t));
 */
void
__bam_ca_rsplit(dbp, fpgno, tpgno)
	DB *dbp;
	db_pgno_t fpgno, tpgno;
{
	DBC *dbc;

	/*
	 * Adjust the cursors.  See the comment in __bam_ca_delete().
	 */
	MUTEX_THREAD_LOCK(dbp->mutexp);
	for (dbc = TAILQ_FIRST(&dbp->active_queue);
	    dbc != NULL; dbc = TAILQ_NEXT(dbc, links)) {
		if (dbc->dbtype == DB_RECNO)
			continue;
		if (dbc->pgno == fpgno)
			dbc->pgno = tpgno;
	}
	MUTEX_THREAD_UNLOCK(dbp->mutexp);
}

/*
 * __bam_ca_split --
 *	Adjust the cursors when splitting a page.
 *
 * PUBLIC: void __bam_ca_split __P((DB *,
 * PUBLIC:    db_pgno_t, db_pgno_t, db_pgno_t, u_int32_t, int));
 */
void
__bam_ca_split(dbp, ppgno, lpgno, rpgno, split_indx, cleft)
	DB *dbp;
	db_pgno_t ppgno, lpgno, rpgno;
	u_int32_t split_indx;
	int cleft;
{
	DBC *dbc;

	/*
	 * Adjust the cursors.  See the comment in __bam_ca_delete().
	 *
	 * If splitting the page that a cursor was on, the cursor has to be
	 * adjusted to point to the same record as before the split.  Most
	 * of the time we don't adjust pointers to the left page, because
	 * we're going to copy its contents back over the original page.  If
	 * the cursor is on the right page, it is decremented by the number of
	 * records split to the left page.
	 */
	MUTEX_THREAD_LOCK(dbp->mutexp);
	for (dbc = TAILQ_FIRST(&dbp->active_queue);
	    dbc != NULL; dbc = TAILQ_NEXT(dbc, links)) {
		if (dbc->dbtype == DB_RECNO)
			continue;
		if (dbc->pgno == ppgno) {
			if (dbc->indx < split_indx) {
				if (cleft)
					dbc->pgno = lpgno;
			} else {
				dbc->pgno = rpgno;
				dbc->indx -= split_indx;
			}
		}
	}
	MUTEX_THREAD_UNLOCK(dbp->mutexp);
}
