/**************************************************************************
 * 
 * Class:  InfoConnection implementation
 * Author: Mark Roseman
 * 
 * Connection that maintains a special callback, typically used for passing
 * information needed to work with the connection (e.g. ID number).
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 05/28/92 MR        initial version
 * 09/20/92 MR        added error check for createReaderAndWriter
 *
 **************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk/connaction.h>
#include <gk/straction.h>
#include <Dispatch/rpcstream.h>
#include <gk/reader.h>
#include <gk/writer.h>
#include <gk/groupkit.h>
#include <gk/infoconn.h>


implementPtrList(InfoConnList, InfoConnection);

declareStrActionCallback(InfoConnection);
implementStrActionCallback(InfoConnection);


/**************************************************************************
 * 
 * Constructors for creating connection via host/port, file descriptor,
 * or rpcstream.
 *
 **************************************************************************/

InfoConnection::InfoConnection(char* host, int port, int id, 
			       ReaderCallbackTable *tbl, int info_msg, 
			       ConnAction* act) : 
	  Connection(host,port,id,tbl), infomsg_(info_msg), infoCB_(act) 
{
  callbacks()->insert( info_msg, new StrActionCallback(InfoConnection)
		      (this, &InfoConnection::infoMsg));
}

InfoConnection::InfoConnection(int fd, int id, ReaderCallbackTable *tbl, 
			       int info_msg, ConnAction* act) : 
          Connection(fd,id,tbl), infomsg_(info_msg), infoCB_(act) 
{
  callbacks()->insert( info_msg, new StrActionCallback(InfoConnection)
		      (this, &InfoConnection::infoMsg));
}

InfoConnection::InfoConnection(rpcstream* server, int id, 
			       ReaderCallbackTable *tbl, int info_msg, 
			       ConnAction* act) : 
          Connection(server,id,tbl), infomsg_(info_msg), infoCB_(act) 
{ 
  callbacks()->insert( info_msg, new StrActionCallback(InfoConnection)
		      (this, &InfoConnection::infoMsg));
}


/**************************************************************************
 * 
 * This routine gets called when the particular message we're watching for
 * is found.  Call the specified callback, passing along both the message
 * string as well as the connection object itself (so people can assign
 * values to it if necessary).
 *
 **************************************************************************/

void InfoConnection::infoMsg(char *s) {
  if(infoCB_ != nil)
    infoCB_->execute(s,this);
}


/**************************************************************************
 * 
 * Constructor for list of InfoConnections.
 *
 **************************************************************************/

InfoConnectionList::InfoConnectionList(int msg, ConnAction* act) : 
           infomsg_(msg), infoCB_(act) 
{
  callbacks_ = new ReaderCallbackTable(FUNCTBLSIZE);
  list_ = (ConnList*) new InfoConnList();
}


/**************************************************************************
 * 
 * Routines to add a connection to the list.
 *
 **************************************************************************/

Connection* InfoConnectionList::add(char *host, int port, int id) {
  InfoConnection* c = new InfoConnection(host,port,id,callbacks(), 
					 infomsg_, infoCB_);
  if( c->writer()->server()) {
    list()->append( c );
    return c;
  } else
    return nil;
}

Connection* InfoConnectionList::add(int fd, int id) {
  InfoConnection* c = new InfoConnection(fd,id,callbacks(), infomsg_, infoCB_);
  list()->append( c );
  return c;
}

Connection* InfoConnectionList::add(rpcstream* server, int id) {
  InfoConnection* c = new InfoConnection(server,id,callbacks(), infomsg_, 
					 infoCB_);
  list()->append( c );
  return c;
}

