/*
 * ParticleAnimation
 */

#include <InterViews/canvas.h>
#include <InterViews/hit.h>
#include <InterViews/patch.h>
#include <InterViews/printer.h>
#include <OS/list.h>

#include <stdio.h>

#include "gems.h"
#include "particle.h"
#include "figure.h"
#include "particle_animation.h"
#include "color_scale.h"

static const int ParticleAllocated = 0x01;
static const int ParticleExtended = 0x02;
static const int ParticlePositioned = 0x04;

static const float epsilon = 0.1;

ParticleAnimation::ParticleAnimation(Glyph* background,
       RealCoord top, RealCoord bottom,
       RealCoord left, RealCoord right,
       RealCoord floor, RealCoord ceiling
   ) : Patch(background)
{
  _color_scale = Gems::instance()->color_scale();

  _canvas = nil;
  _allocated = false;
  _head = nil;

  _real_top = top;
  _real_bottom = bottom;
  _real_left = left;
  _real_right = right;
  _real_floor = floor;
  _real_ceiling = ceiling;

  _xy_screen_to_real = nil;
  _dxy_screen_to_real = nil;
}

void ParticleAnimation::update_screen_to_real()
{
//  printf("ParticleAnimation::update_screen_to_real()\n");
//  printf("\t Screen top=%f bottom=%f left=%f right=%f\n", top(), bottom(), left(), right());
//  printf("\t Real top=%f bottom=%f left=%f right=%f\n", _real_top, _real_bottom, _real_left, _real_right);

  if (_xy_screen_to_real) Resource::unref(_xy_screen_to_real);
  if (_dxy_screen_to_real) Resource::unref(_dxy_screen_to_real);

  _xy_screen_to_real = new Transformer();
  _dxy_screen_to_real = new Transformer();

  Coord real_dx = _real_right - _real_left;
  Coord real_dy = _real_top - _real_bottom;

  Coord screen_dx = right() - left();
  Coord screen_dy = top() - bottom();

  Coord x_scale = real_dx / screen_dx;
  Coord y_scale = real_dy / screen_dy;

  _dxy_screen_to_real->scale(x_scale, y_scale);
  
  _xy_screen_to_real->scale(x_scale, y_scale);

  Coord x_offset = _real_left - left();
  Coord y_offset = _real_bottom - bottom();

  _xy_screen_to_real->translate(x_offset, y_offset);
}

ParticleAnimation::~ParticleAnimation()
{
  if (_xy_screen_to_real) {
    Resource::unref(_xy_screen_to_real);
    Resource::unref(_dxy_screen_to_real);
  }
  
  remove(); // remove all particles
  _canvas = nil;
}

ScreenCoord ParticleAnimation::left() const { return _allocation.left(); }
ScreenCoord ParticleAnimation::right() const { return _allocation.right(); }
ScreenCoord ParticleAnimation::bottom() const { return _allocation.bottom(); }
ScreenCoord ParticleAnimation::top() const { return _allocation.top(); }
ScreenCoord ParticleAnimation::x() const { return _allocation.x(); }
ScreenCoord ParticleAnimation::y() const { return _allocation.y(); }

void ParticleAnimation::move(Particle* particle)
{
//  printf("ParticleAnimation::move()\n");

  if (!(particle->_status & ParticleAllocated))
    return;

  /*
   * Remove the particle if it went off the screen.
   */
  ScreenCoord x = particle->_screen_x;
  ScreenCoord y = particle->_screen_y;
  RealCoord z = particle->_real_z;

//  printf("\t Screen x=%f Screen y=%f \t Real z=%f\n", x, y, z);
//  float epsilon = 5.0 * *_time_step;
  float epsilon = 5.0 * 1000.0;

  if (x < left() || x > right() || y < bottom() || y > top() ||
      particle->_real_y > _real_top - epsilon ||
      particle->_real_y < _real_bottom + epsilon ||
      particle->_real_x > _real_right - epsilon ||
      particle->_real_x < _real_left + epsilon) {

//    printf("\tremoving particle\n");
    remove(particle);
    return;
  }

  if (z > _real_ceiling || z < _real_floor) {
//    printf("\tremoving particle\n");
    remove(particle);
    return;
  }

  /*
   * Move the particle if necessary.
   */

  if (particle->_xy_move || particle->_z_move) {

    if (particle->_z_move) {
//      printf("\tMoving to z=%f _real_floor=%f _real_ceiling=%f\n", z, _real_floor, _real_ceiling);

      particle->_figure->stroke(
       	   _color_scale->color(
       	       _color_scale->linear_index(z, _real_floor, _real_ceiling)
       	   )
       );
    }

    Extension& b = particle->_extension;
    Allocation& a = particle->_allocation;
    Allotment& ax = a.allotment(Dimension_X);
    Allotment& ay = a.allotment(Dimension_Y);
    Coord newx = _allocation.x() + particle->_screen_x;
    Coord newy = _allocation.y() + particle->_screen_y;
    Allotment n_ax(newx, ax.span(), ax.alignment());
    Allotment n_ay(newy, ay.span(), ay.alignment());
    a.allot(Dimension_X, n_ax);
    a.allot(Dimension_Y, n_ay);
    if (_canvas != nil)
      _canvas->damage(b.left(), b.bottom(), b.right(), b.top());
    b.xy_extents(fil, -fil, fil, -fil);
    particle->allocate(_canvas, a, b);
    if (_canvas != nil)
      _canvas->damage(b.left(), b.bottom(), b.right(), b.top());
  }
}

void ParticleAnimation::append(Particle* particle)
{
//  printf("ParticleAnimation::append()\n");

  if (_head) {
    _head->_previous = particle;
    particle->_next = _head;
  }
  _head = particle;

 Resource::ref(particle);
}

void ParticleAnimation::remove(Particle* particle)
{
//  printf("ParticleAnimation::remove()\n");

  if (_canvas != nil && (particle->_status & ParticleAllocated)) {
    Extension b = particle->_extension;
    _canvas->damage(b.left(), b.bottom(), b.right(), b.top());
  }

  if (particle == _head) {
    if (particle->_next)
      _head = particle->_next;
    else
      _head = nil;
  } else {
    particle->_previous->_next = particle->_next;
    if (particle->_next)
      particle->_next->_previous = particle->_previous;
  }

  Resource::unref(particle);
}

void ParticleAnimation::allocate(
   Canvas* c, const Allocation& allocation, Extension& ext)
{
//  printf("ParticleAnimation::allocate()\n");

  boolean allocation_changed = false;
  boolean update_transformation_matrix = true;

  /*
   * If an old allocation exists see if it is equal to the new allocation
   */
  if (_allocated) {
    Allotment& old_x_allotment = _allocation.allotment(Dimension_X);
    Allotment& old_y_allotment = _allocation.allotment(Dimension_Y);
    
    allocation_changed = !(
       	   allocation.allotment(Dimension_X).equals(old_x_allotment, epsilon) &&
       	   allocation.allotment(Dimension_Y).equals(old_y_allotment, epsilon)
       );
  }

  update_transformation_matrix = allocation_changed || _real_size_changed;

  _canvas = c;
  _allocation = allocation;
  _allocated = true;
  Patch::allocate(c, allocation, ext);

  if (update_transformation_matrix)
    update_screen_to_real();

  for (Particle* particle = head() ; particle ; particle = particle->next() ) {

    // if the transformation matrix changed the screen coordinates do too

    if (update_transformation_matrix) {
      xy_real_to_screen(
       	   particle->_real_x, particle->_real_y,
       	   particle->_screen_x, particle->_screen_y
       );

      dxy_real_to_screen(
       	   particle->_real_dx, particle->_real_dy,
       	   particle->_screen_dx, particle->_screen_dy
       );
    }
    allocate(particle);
    ext.extend(particle->_extension);
  }

  _real_size_changed = false;
}

void ParticleAnimation::allocate(Particle* particle)
{
  Allocation& a = particle->_allocation;
  Extension& b = particle->_extension;
  Requisition s;
  particle->request(s);
  Allotment ax = Allotment(
       _allocation.x() + particle->_screen_x,
       s.requirement(Dimension_X).natural(),
       s.requirement(Dimension_X).alignment()
    );
  Allotment ay = Allotment(
       _allocation.y() + particle->_screen_y,
       s.requirement(Dimension_Y).natural(),
       s.requirement(Dimension_Y).alignment()
    );
  if (
      !(particle->_status & ParticleAllocated)
      || !ax.equals(a.allotment(Dimension_X), epsilon)
      || !ay.equals(a.allotment(Dimension_Y), epsilon)
      ) {
    if (_canvas != nil && (particle->_status & ParticleExtended))
      _canvas->damage(b.left(), b.bottom(), b.right(), b.top());
    a.allot(Dimension_X, ax);
    a.allot(Dimension_Y, ay);
    b.xy_extents(fil, -fil, fil, -fil);
    particle->allocate(_canvas, a, b);
    if (_canvas != nil)
      _canvas->damage(b.left(), b.bottom(), b.right(), b.top());
  }
  particle->_status |= ParticleAllocated | ParticleExtended;
}

void ParticleAnimation::draw(Canvas* canvas, const Allocation& a) const
{
  Patch::draw(canvas, a);
  
  for (Particle* particle = head() ; particle ; particle = particle->next() ) {
     Allocation& a = particle->_allocation;
     Extension& b = particle->_extension;
     if (canvas->damaged(b.left(), b.bottom(), b.right(), b.top()))
       particle->draw(canvas, a);
   }
}

void ParticleAnimation::print(Printer* p, const Allocation& a) const
{
  Patch::print(p, a);

  for (Particle* particle = head() ; particle ; particle = particle->next() ) {
    Allocation& a = particle->_allocation;
    Extension& b = particle->_extension;
    if (p->damaged(b.left(), b.bottom(), b.right(), b.top()))
      particle->print(p, a);
  }
}

void ParticleAnimation::pick(Canvas* c, const Allocation& a, int depth, Hit& h)
{
  Patch::pick(c, a, depth, h);

  Coord x = h.left();
  Coord y = h.bottom();

  GlyphIndex index = 0;

  for (Particle* particle = head() ; particle ; particle = particle->next() ) {
    Extension& b = particle->_extension;
    Allocation& a = particle->_allocation;
    if (
       x >= b.left() && x < b.right() &&
       y >= b.bottom() && y < b.top()
    ) {
      h.begin(depth, this, index);
      particle->pick(c, a, depth + 1, h);
      h.end();
    }
    ++index;
  }
}

void ParticleAnimation::remove()
{
  Particle* next;

  for (Particle* current = head() ; current ; current = next ) {
    next = current->next();
    remove(current);
  }
}

void ParticleAnimation::update_real_size(
       Coord top, Coord bottom, Coord left, Coord right, 
       Coord floor, Coord ceiling)
{
  printf("ParticleAnimation::update_real_size()\n");
  printf("\t Real top=%f bottom=%f right=%f left=%f floor=%f ceiling=%f\n", top, bottom, right, left, floor, ceiling);

  if (top > 0.0) _real_top = top;
  if (bottom > 0.0) _real_bottom = bottom;
  if (left > 0.0) _real_left = left;
  if (right > 0.0) _real_right = right;
  if (floor > 0.0) _real_floor = floor;
  if (ceiling > 0.0) _real_ceiling = ceiling;

  _real_size_changed = true;

  reallocate();
}

void ParticleAnimation::append(Glyph*)
{
  printf("Particleanimation::append(Glyph*)\n");
  printf("\tMust be called with a Particle\n");
  exit(1);
};
