/* File: drawing_selector.c
 * Created: 5/28/92 by John Butare(jb7p@andrew.cmu.edu)
 * Description:
 *	
 * Modifications:
 */

#include <InterViews/patch.h>
#include <InterViews/event.h>
#include <InterViews/page.h>
#include <InterViews/brush.h>
#include <InterViews/listener.h>
#include <InterViews/hit.h>
#include <InterViews/color.h>
#include <InterViews/cursor.h>
#include <InterViews/window.h>
#include <InterViews/style.h>
#include <InterViews/action.h>

#include <stdio.h>

#include "gems.h"
#include "figure.h"
#include "clip.h"
#include "selector.h"

/*
 * SelectorTool
 */

static const int click_inside = 0x00;
static const int click_top = 0x01;
static const int click_bottom = 0x02;
static const int click_left = 0x04;
static const int click_right = 0x08;

class SelectorTool : public Rectangle {
 public:
  SelectorTool(Patch* page_patch, Page* page, Window* window, Action* update =nil);
  ~SelectorTool();

  virtual void press(Event&);
  virtual void sense(Event&);
  virtual void drag(Event&);
  virtual void release(Event&);

  void move(Coord x, Coord y, Coord width =-1.0, Coord height =-1.0);
  void size(Coord& width, Coord& height);
  void location(Coord& x, Coord& y, Coord& width, Coord& height);
  void screen_to_page(Event& e, Coord& x, Coord& y);
  void initialize(int click_type, Coord x, Coord y);
  void initialize(int click_type);

 private:
  void change_cursor(int click);

  Coord pointer_offset_x;
  Coord pointer_offset_y;

  Page* _page;
  Patch* _page_patch;
  Window* _window;

  int _click_type;
  int clip(Coord x, Coord y);

  Action* _update;
};


SelectorTool::SelectorTool(Patch* page_patch, Page* page, Window* window, Action* update)
	: Rectangle(new Brush(2.0), Gems::instance()->style()->foreground(), nil,
       	   0.0, 0.0, 50.0, 50.0)
{
  _page = page;
  _page_patch = page_patch;
  _window = window;
  _update = update;
  motion(true);
  button(true, Event::left);
  
  _page->append(this);
  _page->move(0, 50.0, 50.0);

//  printf("SelectorTool::SelectorTool\n");
//  printf(" left=%f right=%f top=%f bottom=%f\n", _page->left(), _page->right(), _page->top(),  _page->bottom());
}

SelectorTool::~SelectorTool() { }
void SelectorTool::sense(Event& e)
{
//  printf("SelectorTool::sense()\n");
  Coord px, py;
  screen_to_page(e, px, py);
  change_cursor(clip(px, py));
}

void SelectorTool::change_cursor(int click)
{
  Cursor* cursor = _window->cursor();

  if (click == 0) { // inside
    Cursor* hand = Gems::instance()->cursor(Gems::instance()->hand);
    if (cursor != hand)
      _window->cursor(hand);
  } else if ((click == click_left) || (click == click_right)){
    if (cursor != crosshairs) 
      _window->cursor(crosshairs);
  } else if ((click == click_top) || (click == click_bottom)) { 
    if (cursor != crosshairs)
      _window->cursor(crosshairs);
  } else if ((click & click_left) && (click  & click_top)) {
    if (cursor != upperleft) 
      _window->cursor(upperleft);
  } else if ((click & click_right) && (click  & click_top)) {
    if (cursor != upperright)
      _window->cursor(upperright);
  } else if ((click & click_bottom) && (click  & click_right)) {
    if (cursor != lowerright)
      _window->cursor(lowerright);
  } else if ((click & click_left) && (click  & click_bottom))
    if (cursor != lowerleft)
      _window->cursor(lowerleft);
}

void SelectorTool::screen_to_page(Event& e, Coord& x, Coord& y)
{
  x = e.pointer_x() - _page->left();
  y = e.pointer_y() - _page->bottom();
}

void SelectorTool::release(Event& e)
{
 if (_update)
   _update->execute();
 PointerHandler::release(e);
}

void SelectorTool::press(Event& e)
{
//  printf("SelectorTool::press()\n");

  Coord px, py;
  screen_to_page(e, px, py);
  int clip_type = clip(px, py);
  initialize(clip_type, px, py);

//  printf("click type is %d\n", _click_type);

  PointerHandler::press(e);
}

void SelectorTool::initialize(int click_type)
{
  _click_type = click_type;

  if (_click_type == click_inside) {
    Coord x1, y1;
    _page->location(0, x1, y1);
    pointer_offset_x = x1;
    pointer_offset_y = y1;
  }
}

void SelectorTool::initialize(int click_type, Coord px, Coord py)
{
  _click_type = click_type;

  if (_click_type == click_inside) {
    Coord x1, y1;
    _page->location(0, x1, y1);
    pointer_offset_x = px - x1;
    pointer_offset_y = py - y1;
  }
}

int SelectorTool::clip(Coord x, Coord y)
{
  int clip = 0;

//  printf("SelectorTool::clip()\n");

  Coord x1, y1, width, height;
  location(x1, y1, width, height);

  Coord x2 = x1 + width;
  Coord y2 = y1 + height;

  if (line_clip(x, y, x, y, x1, y1, x1, y2)) {
    printf("\tleft\n");
    clip |= click_left;
  }

//  printf("\tx=%f y=%f x1=%f y1=%f x2=%f y2=%f\n", x, y, x1, y1, x2, y2);

  if (line_clip(x, y, x, y, x1, y1, x2, y1)) {
//    printf("\tbottom\n");
    clip |= click_bottom;
  }

  if (line_clip(x, y, x, y, x1, y2, x2, y2)) {
//    printf("\ttop\n");
    clip |= click_top;
  }

  if (line_clip(x, y, x, y, x2, y1, x2, y2)) {
//    printf("\tright\n");
    clip |= click_right;
  }

  return(clip);
}

void SelectorTool::drag(Event& e)
{
//  printf("SelectorTool::drag()\n");

  Coord px, py;
  screen_to_page(e, px, py);

//  printf(" px=%f py=%f\n", px, py);
//  printf(" left=%f right=%f top=%f bottom=%f\n", _page->left(), _page->right(), _page->top(),  _page->bottom());
//  printf(" _click_type=%d\n", _click_type);

  Coord left = _page->left();
  Coord right = _page->right();
  Coord top = _page->top();
  Coord bottom = _page->bottom();

  if (px < left)
    px = left;
  else if (px > right)
    px = right;

  if (py < bottom)
    py = bottom;
  else if (py > top)
    py = top;

  Coord x1, y1, width, height;
  location(x1, y1, width, height);

  Coord x2 = x1 + width;
  Coord y2 = y1 + height;

  int new_click_type = _click_type;
  Coord new_x1 = x1, new_y1 = y1;
  Coord new_x2 = x2, new_y2 = y2;

  if (_click_type == click_inside) {
    new_x1 = px - pointer_offset_x;
    new_y1 = py - pointer_offset_y;
    
    new_x2 = new_x1 + width;
    new_y2 = new_y1 + height;
    
    if (new_x1 < left)
      new_x1 = left;
    else if (new_x2 > right)
      new_x1 = right - width;

    if (new_y1 < bottom)
      new_y1  = bottom;
    else if (new_y2 > top)
      new_y1 = top - height;

    new_x2 = new_x1 + width;
    new_y2 = new_y1 + height;
  } else {
    
    if (_click_type & click_top) {
      if (py > y1)
	new_y2 = py;
      else {
	new_click_type = (_click_type | click_bottom) & ~click_top;
	new_y1 = py;
	new_y2 = y1;
      }
    }

    if (_click_type & click_left) {
      if (px < x2)
	new_x1 = px;
      else {
	new_click_type = (_click_type | click_right) & ~click_left;
	new_x1 = x2;
	new_x2 = px;
      }
    }

    if (_click_type & click_bottom) {
      if (py < y2)
	new_y1 = py;
      else {
	new_click_type = (_click_type | click_top) & ~click_bottom;	
	new_y1 = y2;
	new_y2 = py;
      }
    }

    if (_click_type & click_right) {
      if (px > x1)
	new_x2 = px;
      else {
	new_click_type = (_click_type | click_left) & ~click_right;	
	new_x1 = px;
	new_x2 = x1;
      }
    }
  }

  if ((new_x1 != x1) || (new_y1 != y1) || 
      (new_x2 != x2) || (new_y2 != y2)) {
    x1 = new_x1;
    y1 = new_y1;

    width = new_x2  - new_x1;
    height = new_y2  - new_y1;

    _click_type = new_click_type;

    if (_click_type == click_inside) {
      _page->move(0, x1, y1);
    } else {
      move(x1, y1, width, height);
      change_cursor(_click_type);
    }
  }
}

void SelectorTool::move(Coord x, Coord y, Coord width, Coord height)
{
  Coord old_width, old_height;
  size(old_width, old_height);

//  printf("SelectorTool::move()\n");
//  printf("\told_width=%d old_height=%d\n");

  if (width < 0.0)
    width = old_width;

  if (height < 0.0)
    height = old_height;

  clear_points();
  add_point(0,0);
  add_point(0, height);
  add_point(width, height);
  add_point(width, 0);
  
  _page->change(0);
  _page_patch->reallocate();
  _page->move(0, x, y);
}

void SelectorTool::size(Coord& width, Coord& height)
{
  point(2, width, height);
}

void SelectorTool::location(Coord& x, Coord&y, Coord& width, Coord& height)
{
  _page->location(0, x, y);
  point(2, width, height);
}

/*
 * Selector
 */

Selector::Selector(DrawingBox* box, Window* window, Action* update) : Drawing(box)
{
  _page = new Page(box->structure());
  _patch = new Patch(_page);
  _window = window;

  append(_patch);
  motion(true);
  button(true, Event::middle);

  _tool = new SelectorTool(_patch, _page, window, update);
}

Selector::~Selector()
{
  Resource::unref(_tool);
}

void Selector::sense(Event& e)	
{
  fix_cursor();
  PointerHandler::sense(e);
}

void Selector::press(Event& e)
{
//  printf("Selector::press()\n");
  visible(true);
  Coord px, py;
  _tool->screen_to_page(e, px, py);
  move(px, py, 0.0, 0.0);
  _tool->initialize(click_bottom | click_right);
  PointerHandler::press(e);
}

void Selector::release(Event& e)
{
  fix_cursor();
  PointerHandler::release(e);
}

void Selector::drag(Event& e)
{
  _tool->drag(e);
  PointerHandler::drag(e);
}

void Selector::fix_cursor()
{
  if (_window->cursor() != arrow) _window->cursor(defaultCursor);
}

void Selector::move(Coord x, Coord y, Coord width, Coord height)
{
  _tool->move(x, y, width, height);
}

void Selector::size(Coord width, Coord height)
{
  _tool->size(width, height);
}

void Selector::location(Coord& x, Coord&y, Coord& width, Coord& height)
{
  _tool->location(x, y, width, height);
}

void Selector::zoom_info(float& px, float& py, float& pwidth, float& pheight)
{
  Coord x, y, width, height;
  location(x, y, width, height);
  
  Coord page_width = (_page->right() - _page->left());
  Coord page_height = (_page->top() - _page->bottom());

  px = x / page_width;
  py = y / page_height;

  pwidth = width / page_width;
  pheight = height / page_height;
  
  printf("Selector::zoom_info()\n");
  printf("\tpx=%f py=%f pwidth=%f pheight=%f\n", px, py, pwidth, pheight);
}
