/* File: gems_kit.h
 * Created: 4/4/92 by John Butare(jb7p@andrew.cmu.edu)
 * Description:
 *	
 * Modifications:
 */

/*
 * GemsKit -- object for creating common UI components
 */

#ifndef gems_kit_h
#define gems_kit_h

#include <InterViews/enter-scope.h>
#include <InterViews/coord.h>
#include <IV-look/kit.h>
#include <OS/string.h>
#include <InterViews/boolean.h>

class Action;
class Glyph;
class String;
class Style;
class Telltale;
class GenericButton;
class GemsRadioButton;
class ToggleButton;
class GemsLabel;
class GemsTextEditor;

class GemsKit  {
protected:
  GemsKit();
public:
  static GemsKit* instance();
  virtual ~GemsKit();
  
  /* beveling for shading */
  virtual Glyph* inset_diamond(Glyph* g, Style* s) const = 0;
  virtual Glyph* outset_diamond(Glyph* g, Style* s) const = 0;
  virtual Glyph* flat_diamond(Glyph* g, Style* s) const = 0;

  /* 
   * Label
   */
  virtual GemsLabel* label(char*, Style*) const = 0;

  /* 
   * TextEditor
   */
  virtual GemsTextEditor* text_editor(
       Style*, char* default_text ="", Action* ok =nil, Action* cancel =nil, 
       Action* keypress =nil
   ) const = 0;
  
  /*
   * Buttons
   *
   * Button constructors have the format:
   *
   * (body, style, select, deselect, label_left, label_right, value)
   * 
   * body: appears in body of the button.  The body can be a glyph, a String,
   *       or a character pointer.  If this parameter is not give(i.e. the
   *       the style is the first parameter), then the body is a check box.
   *       This in only true for radio and toggle buttons.
   *
   * Style* style: style to use to create the button.
   *
   * Action* select,
   * Action* deselect: actions to call when the button is selected or 
   *                   deselected.  Only ToggleButtons have the deselect 
   *                   parameter.  If select or deselect are not
   *       	       specified they default to nil.  
   *
   * Glyph* label_left,
   * Glyph* label_right: Glyphs to place to the left or the right of the
   *                     button.  The labels are placed in a LRBox.  They
   *                     default to nil.
   * 
   * boolean* value: this variable is set to the state of the button.
   *                 GenericButtons do not have this parameter.  It
   *                 defaults to nil if not specified.
   *
   * Coord size: size of the diamond for radio buttons with no body parameter
   *             only. 
   */
  
  /* 
   * Generic Buttons
   */
  GenericButton* push_button(const char*, Style*, Action* =nil,
	Glyph* =nil, Glyph* =nil) const;

  GenericButton* push_button(const String&, Style*, Action* =nil,
	Glyph* =nil, Glyph* =nil) const;

  virtual GenericButton* push_button(Glyph*, Style*, Action* =nil,
	Glyph* =nil, Glyph* =nil) const = 0;

  /*
   * Toggle Buttons
   */
  ToggleButton* toggle_button(const char*, Style*, Action* =nil,
	Action* = nil, Glyph* =nil, Glyph* =nil, boolean* =nil) const;

  ToggleButton* toggle_button(const String&, Style*, Action* =nil,
	Action* =nil, Glyph* =nil, Glyph* =nil, boolean* =nil) const;

  virtual ToggleButton* toggle_button(Style*, Action* =nil,
	Action* =nil, Glyph* =nil, Glyph* =nil, boolean* =nil) const = 0;

  virtual ToggleButton* toggle_button(Glyph*, Style*, Action* =nil,
    	Action* =nil, Glyph* =nil, Glyph* =nil, boolean* =nil) const = 0;

  /*
   * Radio Buttons
   */
  GemsRadioButton* radio_button(const char*, Style*, Action* =nil, Action* =nil,
        Glyph* =nil, Glyph* =nil, boolean* =nil) const;

  GemsRadioButton* radio_button(const String&, Style*, Action* =nil, Action* =nil,
	Glyph* =nil, Glyph* =nil, boolean* =nil) const;

  virtual GemsRadioButton* radio_button(Style*, Action* =nil, Action* =nil,
 	Glyph* =nil, Glyph* =nil, boolean* =nil, Coord =20.0) const = 0;

  virtual GemsRadioButton* radio_button(Glyph*, Style*, Action* =nil, Action* =nil,
	Glyph* =nil, Glyph* =nil, boolean* =nil) const = 0;
};

/*
 * Generic Buttons
 */

inline GenericButton* GemsKit::push_button(const char* str, Style* s,
	Action*sa, Glyph* l, Glyph* r) const
{ return push_button(Kit::instance()->label(str, s), s, sa, l, r); }

inline GenericButton* GemsKit::push_button(const String& str, Style* s,
	Action*sa, Glyph* l, Glyph* r) const
{ return push_button(Kit::instance()->label(str, s), s, sa, l, r); }

/*
 * Toggle Buttons
 */
inline ToggleButton* GemsKit::toggle_button(const char* str, Style* s,
	Action* sa, Action* da, Glyph* l, Glyph* r, boolean* v) const
{ return toggle_button(Kit::instance()->label(str, s), s, sa, da, l, r, v); }

inline ToggleButton* GemsKit::toggle_button(const String& str, Style* s,
	Action* sa, Action* da, Glyph* l, Glyph* r, boolean* v) const
{ return toggle_button(Kit::instance()->label(str, s), s, sa, da, l, r, v); }

/*
 * Radio Buttons
 */
inline GemsRadioButton* GemsKit::radio_button(const char* str, Style* s,
	Action* sa, Action* da, Glyph* l, Glyph* r, boolean* v) const
{ return radio_button(Kit::instance()->label(str, s), s, sa, da, l, r, v); }

inline GemsRadioButton* GemsKit::radio_button(const String& str, Style* s,
	Action* sa, Action* da, Glyph* l, Glyph* r, boolean* v) const
{ return radio_button(Kit::instance()->label(str, s), s, sa, da, l, r, v); }

#endif
