/*
 * Label - ASCII text glyph with font
 */

#include <InterViews/canvas.h>
#include <InterViews/color.h>
#include <InterViews/font.h>
#include <InterViews/hit.h>
#include <OS/string.h>

#include "label.h"
#include <stream.h>
#include <strings.h>

GemsLabel::GemsLabel(char* text, const Font* f, const Color* c) : Glyph() {
  text_ = new char[strlen(text)];
  strcpy(text_, text);
  font_ = f;
  Resource::ref(font_);
  color_ = c;
  Resource::ref(color_);
}

GemsLabel::~GemsLabel() {
    Resource::unref(font_);
    Resource::unref(color_);
}

void GemsLabel::request(Requisition& requisition) const {
    const Font* f = font_;
    Coord width = f->width(text_, strlen(text_));
    Coord ascent = f->ascent();
    Coord descent = f->descent();
    Coord height = ascent + descent;
    float alignment = (height == 0) ? 0 : descent / height;
    Requirement rx(width, 0, 0, 0);
    Requirement ry(height, 0, 0, alignment);
    requisition.require(Dimension_X, rx);
    requisition.require(Dimension_Y, ry);
}

void GemsLabel::allocate(Canvas* c, const Allocation& a, Extension& ext) {
    const Font* f = font_;
    int len = strlen(text_);
    allocation_ = a;
    extension_ = ext;
    canvas_ = c;
    Coord x = a.x();
    Coord y = a.y();
    ext.xy_extents(
	x - f->left_bearing(text_, len), x + f->right_bearing(text_, len),
	y - f->descent(text_, len), y + f->ascent(text_, len)
    );
}

void GemsLabel::draw(Canvas* canvas, const Allocation& a) const {
    if (canvas != nil) {
        Coord x = a.x();
        Coord y = a.y();
        const Font* f = font_;
        const Color* c = color_;
	const char* p = text_;
        const char* q = &p[strlen(text_)];
        for (; p < q; p++) {
            Coord width = f->width(*p);
            canvas->character(f, *p, width, c, x, y);
            x += width;
        }
    }
}

void GemsLabel::pick(Canvas*, const Allocation& a, int depth, Hit& h) {
    Coord x = h.left();
    Coord y = h.bottom();
    if (x >= a.left() && x < a.right() && y >= a.bottom() && y < a.top()) {
        int index = font_->index(
	    text_, strlen(text_), x - a.x(), true
	);
        h.target(depth, this, index);
    }
}

void GemsLabel::redraw() const
{
    if (canvas_ != nil) {
        Coord l = extension_.left();
        Coord b = extension_.bottom();
        Coord r = extension_.right();
        Coord t = extension_.top();
        if (l < r && b < t) {
            canvas_->damage(l, b, r, t);
        }
    }
}

void GemsLabel::reallocate()
{
    Requisition s;
    request(s);
    extension_.xy_extents(fil, -fil, fil, -fil);
    allocate(canvas_, allocation_, extension_);
}

void GemsLabel::string(char* word)
{
  delete text_;
  text_ = new char[strlen(word)];
  strcpy(text_, word);
  redraw();
  reallocate();
  redraw();
}

char* GemsLabel::string()
{
  return (text_);
}
