;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
;;
;; EHTS was designed and implemented by:
;;
;;	Uffe Kock Wiil 		(kock@iesd.auc.dk)
;;	Claus Bo Nielsen 	(cbn@cci.dk)
;;
;; at The University of Aalborg in Denmark spring 1990, and is provided
;; for unrestricted use provided that this legend is included on all
;; tape media and as a part of the software program in whole or part.
;; Users may copy or modify EHTS without charge, but are not authorized
;; to license or distribute it to anyone else except as part of a
;; product or program developed by the user.
;;   
;; EHTS IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
;; WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE
;; PRACTICE.
;; 
;; EHTS is provided with no support and without any obligation on the
;; part of the authors, to assist in its use, correction, modification
;; or enhancement.
;; 
;; THE AUTHORS SHALL HAVE NO LIABILITY WITH RESPECT TO THE INFRINGEMENT
;; OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY EHTS OR ANY PART
;; THEREOF.
;; 
;; In no event will the authors and/or The University of Aalborg be
;; liable for any lost revenue or profits or other special, indirect and
;; consequential damages, even if the authors and/or The University of
;; Aalborg has been advised of the possibility of such damages.
;; 
;; Please address all correspondence to:
;; 
;; Uffe Kock Wiil
;; Department of Computer Science,
;; The University of Aalborg,      Email:  kock@iesd.auc.dk
;; Fredrik Bajers Vej 7E,          Phone:  + 45 98 15 42 11 (Ext 5051)
;; DK-9220 Aalborg, Denmark.       Fax:    + 45 98 15 81 29
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global variables (user changeable)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar ehts-user-alist '(("kock" . 0)("kasper" . 1)("assnah" . 2)
			  ("jep" . 3)("mt" . 4)("markus" . 5)
			  ("mac" . 6)("kp" . 7)("htk" . 8)
			  ("henrik" . 9)("levi" . 10)("tolboel" . 11))
  "*An alist holding the user-login-names of users having access to
EHTS - used when people login and when invoking the talk command")

(defvar ehts-user-message-alist '(("kock" . 0)("kasper" . 1)("assnah" . 2)
				  ("jep" . 3)("mt" . 4)("markus" . 5)
				  ("mac" . 6)("kp" . 7)("htk" . 8)
				  ("henrik" . 9)("levi" . 10)("tolboel" . 11)
				  ("global" . 15))
  "*As ehts-user-alist, but used when invoking the message command
- global must be present to send global messages")

(defvar ehts-on-machine ""
  "*Variable to hold the machine name where the EhtsBase is running.")
(setq ehts-on-machine "")

(defvar ehts-browser-on-machine ""
  "*Variable to hold the machine name where the EHTS Browser is running.")
(setq ehts-browser-on-machine "")

(defvar ehts-userID ""
  "*Variable to hold the userID, e.i. the name that EhtsBase knows of.")
(setq ehts-userID (user-login-name))

(defconst ehts-uniqe-socket 10008
  "*Variable to hold the number of the uniqe-socket (used while connecting).")

(defconst ehts-browser-socket 10009
  "*Variable to hold the number of the browser-socket.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global variables (users should not change these (!))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar ehts-socket nil
  "Variable to store a socket number (used while connecting).")

(defvar ehts-connected-to-a-base nil
  "Variable telling if we are connected to an EhtsBase.")

(defvar ehts-connected-to-a-browser nil
  "Variable telling if we are connected to a browser.")

(defvar ehts-node-name-alist '()
  "Alist holding information about node names and node numbers.")

;;; Make the alist local for each buffer. By this way we can have an
;;; alist containing link information for each buffer, that is each
;;; data node.

(defvar ehts-node-link-alist '()
  "Alist for each buffer, holding link information.")
(make-variable-buffer-local 'ehts-node-link-alist)

(defvar ehts-edit-alist '()
  "Alist holding information about locked nodes.")

;;; Epoch can display different fonts on each screen, so we built an alist
;;; holding the available X11 fonts

(defvar ehts-font-types '(("5x8" . 0) ("6x9" . 1) ("6x10" . 2)
			  ("6x12" . 3) ("6x13" . 4) ("7x13" . 5)
			  ("7x14" . 6) ("8x13" . 7) ("8x16" . 8)
			  ("9x15" . 9) ("10x20" . 10) ("12x24" . 11)
			  ("fixed" . 12))
  "An alist holding different font types for nodes.")

;;; 3 alists holding information about the HyperBase specific commands, it's
;;; easyer to remember a name than a number !!

(defvar node-link-list '(("node" . 0)("link" . 1))
  "An alist to choose between node and link.")

(defvar ehts-hb-operations '(("all" . 0) ("read" . 1) ("write" . 2)
		   ("create node" . 5) ("delete" . 6) ("link" . 7)
		   ("move link" . 8) ("remove link" . 9)
		   ("event" . 10) ("unevent" . 11) ("show event" . 12)
		   ("lock" . 13) ("unlock" . 14) ("show lock" . 15)
		   ("connect" . 16) ("disconnect" . 17) ("browser" . 18))
  "An alist holding the EhtsBase operations.")

(defvar ehts-hb-keys '(("all" . 0) ("datanode no" . 513) ("data size" . 514)
		       ("links to me" . 515) ("link num" . 516) ("data" . 517)
		       ("link no" . 1025) ("use count" . 1026)
		       ("to data node no" . 1027) ("geometry" . 1)
		       ("font" . 2) ("n last modified date" . 3)
		       ("n last modified by" . 4) ("n created date" . 5)
		       ("n created by" . 6) ("n name" . 7)
		       ("l last modified date" . 256)
		       ("l last modified by" . 257)
		       ("l created date" . 258) ("l created by" . 259)
		       ("l name" . 260))
  "An alist holding the EhtsBase keys.")

;;; The following variables a used by the browser commands.

(defvar ehts-browser-command-string nil
  "A variable to hold the pending browser command.")

(defvar ehts-pending-browser-command nil
  "Is there any pending browser commands?")

;;; The next 3 variable are used by the event mecanism. If the variable
;;; "ehts-in-command" is nil, it means that handling events can be carried
;;; out, but if it's t then we are in a critical region (talking to the base),
;;; and handling events would fuck things up !!! The "ehts-pending-events"
;;; tells us if there are any events that should be carried out. And the list
;;; in the variable "ehts-event-list" keeps the event strings to be proceded.

(defvar ehts-in-command nil
  "An event stick.")
(defvar ehts-pending-event nil
  "Is there pending events?")
(defvar ehts-event-list '()
  "List to hold event strings.")
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The EHTS-MODE
;;; =============
;;; ehts-kill-emacs
;;; ehts-mode
;;; ehts (used for commandline startup (e.g. emacs -f ehts)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq minor-mode-alist (cons '(ehts-mode " Ehts")
			     minor-mode-alist))
(require 'minor-map)
(make-variable-buffer-local 'ehts-mode)
(setq-default ehts-mode nil)

;;; Make sure to disconnect when killing Emacs
(defun ehts-kill-emacs ()
  "Make sure that Emacs are disconnected from any EhtsBase."
  (interactive)
  (if (y-or-n-p "Really wan't to exit from EHTS and kill Epoch? ")
      (progn
	(if ehts-connected-to-a-base		; if connected to HB
	    (ehts-disconnect))		;   then disconnect
	(if ehts-connected-to-a-browser	; if connected to browser
	    (ehts-disconnect-from-browser))	;   then disconnect
	(save-buffers-kill-emacs))
    (message "(OK)")))		; and kill Emacs
(global-set-key "\C-x\C-c" 'ehts-kill-emacs) ; setup the exit key

(defun ehts-mode (arg) 
  "Ehts-mode (Emacs HyperText System). 

See the \"dir ehts help\" node  for help."
  (interactive "P")
  (if (or (and (null arg) ehts-mode)
          (<= (prefix-numeric-value arg) 0))
      (if ehts-mode
          (progn                        ;Turn it off
            (unbind-minor-mode 'ehts-mode)
	    (message "ehts-mode is off")
            (setq ehts-mode nil)))
    (if ehts-mode				;Turn it on
        ()
      (minor-set-key "\C-xk"    'ehts-kill-buffer 'ehts-mode)
      (minor-set-key "\C-x\C-f" 'ehts-find-node 'ehts-mode)
      (minor-set-key "\C-x\C-s" 'ehts-write-buffer 'ehts-mode)
      (minor-set-key "\C-cb"    'ehts-disconnect-from-browser 'ehts-mode)
      (minor-set-key "\C-cc"    'ehts-center-region 'ehts-mode)
      (minor-set-key "\C-cd"    'ehts-disconnect 'ehts-mode)
      (minor-set-key "\C-c\C-a" 'ehts-attributes 'ehts-mode)
      (minor-set-key "\C-c\C-b" 'ehts-connect-to-browser 'ehts-mode)
      (minor-set-key "\C-c\C-c" 'ehts-change-link-name 'ehts-mode)
      (minor-set-key "\C-c\C-d" 'ehts-delete 'ehts-mode)
      (minor-set-key "\C-c\C-e" 'ehts-do-edit 'ehts-mode)
      (minor-set-key "\C-c\C-f" 'ehts-set-fill-column 'ehts-mode)
      (minor-set-key "\C-c\C-k" 'ehts-kill-screen-and-buffer 'ehts-mode)
      (minor-set-key "\C-c\C-l" 'ehts-follow-link 'ehts-mode)
      (minor-set-key "\C-c\C-m" 'ehts-move-link 'ehts-mode)
      (minor-set-key "\C-c\C-s" 'ehts-send-message 'ehts-mode)
      (minor-set-key "\C-c\C-t" 'ehts-talk 'ehts-mode)
      (minor-set-key "\C-c\C-u" 'ehts-unlock-node 'ehts-mode)
      (setq ehts-mode t)
      (if (not (assoc (user-login-name) ehts-user-alist))
	  (error "You are not registered as user of EHTS - Good-bye")
	(if ehts-connected-to-a-base
	    ()
	  (ehts-welcome)		 ;; else say hello
	  (sit-for 0)
	  (ehts-connect)		 ;; and connect
	  (ehts-mode 1))		 ;; set current buffer in ehts-mode
	(message "Ehts-mode is on.")))
    (set-buffer-modified-p (buffer-modified-p))))


(defun ehts ()
  "Function to call EHTS from commandline (e.g. emacs -f ehts)."
  (interactive)
  (ehts-mode 1))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The CONNECT function and the EHTS-HB-SYS-CALL
;;; =============================================
;;; ehts-connect
;;; "process filters"
;;; "get machine and userID"
;;; ehts-hb-sys-call
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-connect ()
  "Connect to the EhtsBase placed on \"ehts-on-machine\"."
  (if (string-equal ehts-on-machine "")
      (setq ehts-on-machine (call-interactively 'ehts-get-machine)))
  (if (string-equal ehts-userID  "")
      (setq ehts-userID (call-interactively 'ehts-get-userID)))
  (message (concat "Connecting to: \"" ehts-on-machine "\", with userID: \""
		   ehts-userID "\" "))
  ;;; Get the 3 socket ports from the uniqe-socket
  (setq ehts-uniqe-object (open-network-stream
		   "ehts-uniqe-socket" nil ehts-on-machine ehts-uniqe-socket))
  (set-process-filter ehts-uniqe-object 'ehts-uniqe-process-filter)
  (accept-process-output ehts-uniqe-object)
  (setq ehts-read-object (open-network-stream
		    "ehts-read-socket" nil ehts-on-machine ehts-socket))
  (accept-process-output ehts-uniqe-object)
  (setq ehts-write-object (open-network-stream
		    "ehts-write-socket" nil ehts-on-machine ehts-socket))
  (accept-process-output ehts-uniqe-object)
  (setq ehts-event-object (open-network-stream
		    "ehts-event-socket" nil ehts-on-machine ehts-socket))
  (delete-process "ehts-uniqe-socket")	; we don't need it any more!
  ;;; Send the userID
  (process-send-string ehts-write-object (concat ehts-userID "\000"))
  ;;; And setup the processfilters for the 3 ports
  (set-process-filter ehts-write-object 'ehts-write-process-filter)
  (set-process-filter ehts-read-object 'ehts-read-process-filter)
  (set-process-filter ehts-event-object 'ehts-event-process-filter)
  (message (concat "Connected to EhtsBase on \"" ehts-on-machine "\""))
  (setq ehts-connected-to-a-base t)
  (ehts-setup-events 0 "lock" "data")	   ; event on update intentions
  (ehts-setup-events 0 "write" "n name")   ; event on creating new or renaming
					   ; old nodes.
  (ehts-setup-events 0 "write" "data")	   ; event on updated data areas 
  (ehts-setup-events 0 "write" "font")     ; event on updated fonts
  (ehts-setup-events 0 "write" "geometry") ; event on updated geometry
  (ehts-setup-events 0 "delete" 0)	   ; event on all delete calls.
  
  ;;  (ehts-insert-in-node-table (ehts-browse "data"))
  (ehts-read-all-names)
  ;; lookup if there is a node called "user-full-name", if there is
  ;; show it, if not see if there exists a node called "dir ehts
  ;; help", if show it, if not don't do anything.
  (if (assoc (user-full-name) ehts-node-name-alist)
      (progn
	(ehts-get-node (user-full-name))
	(let (buffer screen)
	  (setq buffer "*Ehts Welcome*")
	  (setq screen (ehts-find-buffer-screen buffer))
	  (kill-buffer buffer)
	  (switch-to-buffer (user-full-name))
	  (remove-screen screen)))
    (if (assoc "dir ehts help" ehts-node-name-alist)
	(progn
	  (ehts-get-node "dir ehts help")
	  (let (buffer screen)
	    (setq buffer "*Ehts Welcome*")
	    (setq screen (ehts-find-buffer-screen buffer))
	    (kill-buffer buffer)
	    (switch-to-buffer "dir ehts help")
	    (remove-screen screen))))))
    
;;; Process filters

(defun ehts-write-process-filter (proc str)
  "Process filter for the write-port."
  (error "This should not happen! Input on the write socket."))

(defvar ehts-return-value nil
  "Variable to hold the read string.")

(defun ehts-read-process-filter (proc str)
  "Process filter for the read-port."
  (setq ehts-return-value (concat ehts-return-value str)))

(defun ehts-event-process-filter (proc str)
  "Process filter for the event-port."
  (setq ehts-event-list (cons str ehts-event-list)) ; insert in list
  (if ehts-in-command			; if not ready for event-handling
      (setq ehts-pending-event t)	; set the flag that there is event 
    (ehts-parse-event)))		; else parse the event-list

(defun ehts-uniqe-process-filter (proc str)
  "Process filter for the uniqe-socket-port, just return the port number"
  (setq ehts-socket (4chars-to-int str)))

(defun ehts-browser-process-filter (proc str)
  "Process filter for the browser-port."
  (setq ehts-browser-command-string str)
  (if ehts-in-command
      (setq ehts-pending-browser-command t)
    (ehts-parse-browser-command)))

;;; Get the machine and user name

(defun ehts-get-machine (arg)
  "Get the machine name of where the EhtsBase is placed."
  (interactive "sConnect to the EhtsBase on machine: ")
  arg)

(defun ehts-get-userID (arg)
  "Get the user ID, that is the name the EhtsBase server should know about."
  (interactive "sUserID to the EhtsBase server: ")
  arg)

(defun ehts-hb-sys-call (function entnum attribute value-string check)
  "Function to handle all communication between EhtsBase and Emacs."
  (let (hb-string)
    (setq hb-string "")
    (if (not function)			; are function nil
	()
      (if (integer-or-marker-p function)    ; is function an int or string?
	  ()
	(setq function (cdr (assoc function ehts-hb-operations)))) ; string
      (setq hb-string (concat hb-string (ehts-int-to-4bytes function))))
    (if (not entnum)			
	()
      (if (integer-or-marker-p entnum)
	  ()
	(setq entnum (cdr (assoc entnum ehts-node-name-alist))))
      (setq hb-string (concat hb-string (ehts-int-to-4bytes entnum))))
    (if (not attribute)			
	()
      (if (integer-or-marker-p attribute)
	  ()
	(setq attribute (cdr (assoc attribute ehts-hb-keys))))
      (setq hb-string (concat hb-string (ehts-int-to-4bytes attribute))))
    (if (not value-string)			
	()
      (setq string-length (length value-string))
      (setq hb-string (concat hb-string (ehts-int-to-4bytes string-length)))
      (setq hb-string (concat hb-string value-string)))
    (process-send-string ehts-write-object hb-string)
    (if check
	(ehts-read-4bytes)
      t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Direct HyperBase functions
;;; ==========================
;;; ehts-disconnect (C-c d)
;;; ehts-createnode
;;; ehts-browse
;;; ehts-lock
;;; ehts-unlock
;;; ehts-setup-events
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-disconnect ()
  "Disconnect from the EhtsBase."
  (interactive)
  (ehts-command t)
  (ehts-hb-sys-call "disconnect" nil nil nil nil)
  (ehts-command nil)
  (delete-process ehts-write-object)
  (delete-process ehts-read-object)
  (delete-process ehts-event-object)
  (setq ehts-connected-to-a-base nil)
  (message "Disconnected from EhtsBase on \"%s\"." ehts-on-machine)
  (sit-for 2))

(defun ehts-createnode ()
  "Create a new node in the EhtsBase."
  (let (ret)
    (ehts-command t)
    (if (= (ehts-hb-sys-call "create node" nil nil nil t) 0)
	(progn
	  (setq ret (ehts-read-4bytes))
	  (ehts-command nil)
	  ret)
      (ehts-command nil)
      (error "Creating a new node failed")))) 

(defun ehts-browse (type)
  "Call the EhtsBase and return a list of data/link numbers."
  (if (equal type "data")			
      (setq type 0)
    (setq type 1))
  (ehts-command t)
  (if (< (ehts-hb-sys-call "browser" type nil nil t) 0)
      (progn
	(ehts-command nil)
	(error "Browsing failed, panic !!!"))
    (let (number
	  (number-list '()))
      (setq number (ehts-read-4bytes))	; get number of bytes
      (while (> number 4)		; read all except one (the NULL)
	(setq number-list (cons (ehts-read-4bytes) number-list)) ;cons the list
	(setq number (- number 4)))	; have read one int
      (ehts-read-4bytes)		; read the NULL
      (ehts-command nil)
      number-list)))			; return the list

(defun ehts-lock (node)
  "Lock an entity."
  (ehts-command t)
  (ehts-hb-sys-call "lock" node "data" nil t)
  (ehts-command nil))

(defun ehts-unlock (node)
  "Unlock an entity."
  (ehts-command t)
  (ehts-hb-sys-call "unlock" node "data" nil t)
  (ehts-command nil))

(defun ehts-setup-events (node operation key)
  "Setup events."
  (ehts-command t)
  (ehts-hb-sys-call "event" node nil nil nil)
  (if (< (ehts-hb-sys-call operation nil key nil t) 0)
      (progn
	(ehts-command nil)
	(message "Setting-up event failed!")))
  (ehts-command nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions working on NODES
;;; ==========================
;;; ehts-write-buffer (C-x C-s)
;;; ehts-find-node (C-x C-f)
;;; ehts-do-edit (C-c C-e)
;;; ehts-unlock-node (C-c C-u)
;;; ehts-get-node
;;; ehts-show-screen
;;; ehts-prepare-new-node
;;; ehts-insert-in-node-table
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-write-buffer ()
  "Write the current buffer to the EhtsBase."
  (interactive)
    (let (buffername ehts-ent-no)
      (setq buffername  (buffer-name))
      (if (not (buffer-modified-p))
	  (message "(No changes need to be saved)")
	(if (assoc buffername ehts-node-name-alist)
	    (progn
	      (message "Writing \"%s\" ... " buffername)
	      (ehts-command t)
	      (if (< (ehts-hb-sys-call "write" buffername "data"
				       (buffer-string) t) 0)
		  (progn
		    (ehts-command nil)
		    (error "Writing \"%s\" failed!" buffername))
		(ehts-hb-sys-call "write" buffername "n last modified by"
				  (concat (user-full-name) "\000") t)
		(ehts-hb-sys-call "write" buffername "n last modified date"
				  (concat (current-time-string) "\000") t)
		(ehts-hb-sys-call "write" buffername "geometry"
				  (concat (screen-width) "x"
					  (screen-height) "\000") t )
		(ehts-command nil)
		(set-buffer-modified-p nil)
		(message "Writing \"%s\" ... done!" buffername)))
	  (ehts-command nil)
	  (pop-to-buffer "*Ehts-Error*")
	  (erase-buffer)
	  (insert-string
	   "You have tried to save the buffer in the EhtsBase.\n")
	  (insert-string
	   "The current buffer is not known by Emacs as a EhtsBase buffer,\n")
	  (insert-string "that is there is no node for this buffer.\n")
	  (insert-string "You have to create a new node with \\C-x\\C-f.")
	  (switch-to-buffer buffername)))))

(defun ehts-find-node ()
  "Same as \"find-file\" but this one is for the EhtsBase."
  (interactive)
  (let (ret)
    (setq ret (completing-read "Find node named: "
			       ehts-node-name-alist nil nil nil))
    (if (string-equal "" ret)
	(error "Can't create node without name...")
    (ehts-get-node ret))))

(defun ehts-do-edit ()
  "Make buffer writeable."
  (interactive)
  (let (ret lockuser name num)
    (setq name (buffer-name))
    (ehts-command t)
    (setq ret (ehts-hb-sys-call "lock" name "data" nil t))
    (ehts-command nil)
    (cond ((= ret 0)
	   (setq buffer-read-only nil)
	   (set-buffer-modified-p (buffer-modified-p)) ; force update
					; of mode line
	   (setq num (cdr (assoc name ehts-node-name-alist)))
	   (setq ehts-edit-alist (cons (cons name num) ehts-edit-alist))
	   (message "Node is locked."))
	  ((= ret 350)
	   (ehts-command t)
	   (if (< (ehts-hb-sys-call "show lock" (buffer-name) "data" nil t) 0)
	       (progn
		 (ehts-command nil)
		 (error "Can't use \"show lock\", panic !!!")))
	   (setq lockuser (ehts-read-null-string))
	   (ehts-command nil)
	   (if (string= lockuser (user-login-name))
	       ()
	     (beep)
	     (message "Node locked by: \"%s\"" lockuser)
	     (sit-for 3)
	     (if (y-or-n-p "Do you wish to know when unlocked? ")
		 (progn
		   (ehts-setup-events (buffer-name) "unlock" "data")
		   (message "(OK)"))
	       (message "(OK)"))))
	  (t (error "Return value: %d" ret)))))

(defun ehts-unlock-node ()
  "Unlock current buffers node."
  (interactive)
  (let (name num)
    (if (buffer-modified-p)
	(progn
	  (beep)
	  (if (y-or-n-p "Save buffer before unlocking node? ")
	      (ehts-write-buffer)
	    (set-buffer-modified-p nil))))
    (setq name (buffer-name))
    (setq num (cdr (assoc name ehts-node-name-alist)))
    (ehts-unlock name)
    (message "Node is unlocked.")
    (setq buffer-read-only t)
    (set-buffer-modified-p (buffer-modified-p))
    (setq ehts-edit-alist (ehts-remove-from-table num ehts-edit-alist))))

(defun ehts-get-node (name)
  "Find a node by it's name, if not found create one."
  (let (num ret flag screenid)
      (if (assoc name ehts-node-name-alist)
	  (progn
	    (setq screenid (ehts-find-buffer-screen name))
	    (setq num (cdr (assoc name ehts-node-name-alist)))
	    (if screenid
		(progn
		  (switch-screen screenid)
		  (error "Node is allready displayed."))
	      (message "Reading \"%s\" ... " name)
	      (ehts-command t)
	      (setq ret (ehts-hb-sys-call "read" name "data" nil t)) 
	      (cond ((< ret 0)		; Fatal error !!!
		     (error "Can't read node, panic !!!"))
		    ((= ret 350)		; Locked
		     (setq flag t))	; Show the message "(Node ...)"
		    ((> ret 0)		; Default
		     (error "Return code: %d" ret)))
	      (setq ret (ehts-read-string))
	      (ehts-command nil)
	      (get-buffer-create name)
	      (ehts-show-screen name nil)
	      (set-buffer name)
	      (ehts-set-fill-column)
	      (auto-fill-mode 1)
	      (ehts-mode 1)
	      (setq buffer-read-only nil) ; if buffer exist 
	      (erase-buffer)		; make sure that the buffer is clean
	      (insert-string ret)
	      (goto-char (point-min))
	      (message "Reading \"%s\" ... done!" name)))
	(setq num (ehts-createnode))		; else find a new node
	(get-buffer-create name)
	(ehts-show-screen name t)
	(set-buffer name)
	(ehts-set-fill-column)
	(auto-fill-mode 1)
	(ehts-mode 1)
	(ehts-prepare-new-node name num))
      (setq ehts-node-link-alist (ehts-read-all-link-names num))
					; read all the links 
      (save-excursion
	(set-buffer name)
	(setq buffer-read-only t)
	(set-buffer-modified-p nil)
	(set-buffer-modified-p (buffer-modified-p)))
      (if flag				; the node is locked
	  (cond ((string-match ".talk" name)
		 nil)
		((string-match ".message" name)
		 nil)
		(t
		 (ehts-command t)
		 (if (< (ehts-hb-sys-call "show lock" (buffer-name)
					  "data" nil t) 0) 
		     (progn
		       (ehts-command nil)
		       (error "Can't \"show lock\", panic !!!"))
		   (beep)
		   (message (concat "Node locked by: \""
				    (ehts-read-null-string)
				    "\""))
		   (ehts-command nil))))
	t)))

(defun ehts-show-screen (name new)
  "Popup a screen in the right size and font. Returns the screenid."
  (let (geo-str the-font)
    (if new
	;; if name is a new buffer, create a new screen 
	(create-screen name (list (cons 'font "fixed")
				  (cons 'title name)
				  (cons 'geometry "60x15")))
      (ehts-command t)
      (if (< (ehts-hb-sys-call "read" name "font" nil t) 0)
	  (progn
	    (ehts-command nil)
	    (error "Can't read the font key, panic !!!")))
      (setq the-font (ehts-read-null-string))
      (if (< (ehts-hb-sys-call "read" name "geometry" nil t) 0)
	  (progn
	    (ehts-command nil)
	    (error "Can't read the geometry key, panic !!!")))
      (setq geo-str (ehts-read-null-string))
      (ehts-command nil)
      (create-screen name (list (cons 'font the-font)
				(cons 'title name)
				(cons 'geometry geo-str))))))

(defun ehts-prepare-new-node (name num)
  "Prepare a new buffer, that is make room for links."
  (let (hb-string)
    (setq hb-string
	  (concat (ehts-int-to-4bytes 100)
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes num)
		  (ehts-int-to-4bytes 7)
		  (ehts-int-to-4bytes (length (concat name "\000")))
		  (concat name "\000")
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes num)
		  (ehts-int-to-4bytes 6)
		  (ehts-int-to-4bytes (length (concat (user-full-name)
						      "\000"))) 
		  (concat (user-full-name) "\000")
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes num)
		  (ehts-int-to-4bytes 5)
		  (ehts-int-to-4bytes (length (concat
					       (current-time-string) "\000")))
		  (concat (current-time-string) "\000")
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes num)
		  (ehts-int-to-4bytes 4)
		  (ehts-int-to-4bytes (length (concat (user-full-name)
						      "\000"))) 
		  (concat (user-full-name) "\000")
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes num)
		  (ehts-int-to-4bytes 3)
		  (ehts-int-to-4bytes (length (concat
					       (current-time-string) "\000")))
		  (concat (current-time-string) "\000")
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes num)
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes (length (concat "fixed" "\000")))
		  (concat "fixed" "\000")
		  (ehts-int-to-4bytes 2)
		  (ehts-int-to-4bytes num)
		  (ehts-int-to-4bytes 1)
		  (ehts-int-to-4bytes (length (concat "60x15" "\000")))
		  (concat "60x15" "\000")
		  (ehts-int-to-4bytes 100)))
    (ehts-command t)
    (process-send-string ehts-write-object hb-string)
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (ehts-command nil)))

(defun ehts-insert-in-node-table (list-of-numbers)
  "Make a alist of node names and numbers."
  (while (not (equal list-of-numbers nil))
    (let (num ret)
      (setq num (car list-of-numbers))
      (message "Building list of node name (%d)" num)
      (ehts-command t)
      (if (< (ehts-hb-sys-call "read" num "n name" nil t) 0)
	  (progn
	    (ehts-command nil)
	    (error "Reading node name failed, panic !!!"))
	(setq ehts-node-name-alist
	      (cons (cons (ehts-read-null-string) num) ehts-node-name-alist))
	(ehts-command nil)
	(setq list-of-numbers (cdr list-of-numbers)))))
  (message "(OK)"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions working on LINKS
;;; ==========================
;;; ehts-follow-link (C-c C-l)
;;; ehts-move-link (C-c C-m)
;;; ehts-change-link-name (C-c C-c)
;;; ehts-make-link
;;; ehts-move-link-number
;;; ehts-replace-link-string
;;; ehts-remove-link-string
;;; ehts-delete-all-links
;;; ehts-delete-link
;;; ehts-insert-in-link-table
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-follow-link ()
  "Follow a link. If it does not exist make a new one."
  (interactive)
  (let (tonode linknum curbuf completion-ignore-case)
    (setq curbuf (buffer-name))
    (setq completion-ignore-case t)
    (setq tonode (completing-read "Follow link named: "
				  ehts-node-link-alist nil nil nil))
    (if (string-equal "" tonode)
	(error "Can't create link without name..."))
    (if (assoc tonode ehts-node-link-alist)
	(progn
	  (if (not (buffer-modified-p))
	      ()
	    (beep)
	    (if (y-or-n-p "Save buffer before follow link? ")
		(ehts-write-buffer)))
	  (setq linknum (cdr (assoc tonode ehts-node-link-alist)))
	  (ehts-command t)
	  (if (< (ehts-hb-sys-call "read" linknum "to data node no" nil t) 0)
	      (progn
		(ehts-command nil)
		(error "Can't read \"to data node no\" in link, panic !!!")))
	  (ehts-read-4bytes)		; I know it's a string,
					; but the string is
					; allways 4 bytes.
	  (setq tonode (ehts-read-4bytes))
	  (if (< (ehts-hb-sys-call "read" tonode "n name" nil t) 0)
	      (progn
		(ehts-command nil)
		(error "Can't read \"name\" in data node, panic !!!")))
	  (ehts-get-node (ehts-read-null-string))
	  (ehts-command nil))
      (ehts-make-link tonode curbuf))))		; else make a new link

(defun ehts-move-link ()
  "Move a link in the current node to another node."
  (interactive)
  (let (linknum)
    (if (equal ehts-node-link-alist '())
	(error "No links in this node!"))
    (if buffer-read-only
	(error "Can't move link in read-only node.")
      (setq linknum (cdr (assoc (completing-read "Move link named: "
		 ehts-node-link-alist nil t nil) ehts-node-link-alist)))
      (ehts-move-link-number linknum))))

(defun ehts-change-link-name ()
  "Change the link name string in a node."
  (interactive)
  (let (link newname)
    (if (equal ehts-node-link-alist '())
	(error "No links in this node!"))
    (if buffer-read-only
	(error "Can't change link name in a read-only node.")
      (setq link (cdr (assoc (completing-read "Change link named: "
		  ehts-node-link-alist nil t nil) ehts-node-link-alist)))
      (setq newname (completing-read "To new link name: "
				     ehts-node-link-alist nil nil nil))
      (if (assoc newname ehts-node-link-alist)
	  (error "Allready link with that name !"))
      (setq ehts-node-link-alist (ehts-replace-link-string
				  link newname ehts-node-link-alist)))))

(defun ehts-make-link (value curbuf)
  "Making a link in the current buffer at the current point."
  (if (equal t buffer-read-only)
      (error "Can't make a link in a read-only buffer."))
  (let (toname linknum)
    (message "Making new link with name \"%s\"" value)
    (sit-for 1)			; just wait for 1 sec.
    (setq toname (completing-read "Link to node named: "
				  ehts-node-name-alist nil nil nil))
    (insert-string (concat "[-> " value "]"))
    (if (not (assoc toname ehts-node-name-alist))
	  (ehts-get-node toname))	; make a new data node
    (ehts-command t)
    (ehts-hb-sys-call "link" toname nil nil nil)
    (if (< (ehts-hb-sys-call nil toname nil nil t) 0)
	(progn
	  (ehts-command nil)
	  (error "Can't create a link.")))
    (setq linknum (ehts-read-4bytes))
    (if (< (ehts-hb-sys-call "link" curbuf linknum nil t) 0)
	(progn
	  (ehts-command nil)
	  (error "Can't link link.")))
    (ehts-read-4bytes)
    (if (< (ehts-hb-sys-call
	    "write" linknum "l name" (concat value "\000") t) 0)
	(progn
	  (ehts-command nil)
	  (error "Can't write name into link.")))
    (ehts-hb-sys-call "write" linknum "l created by"
		      (concat (user-full-name) "\000") t)
    (ehts-hb-sys-call "write" linknum "l created date"
		      (concat (current-time-string) "\000") t)
    (ehts-hb-sys-call "write" linknum "l last modified by"
		      (concat (user-full-name) "\000") t)
    (ehts-hb-sys-call "write" linknum "l last modified date"
		      (concat (current-time-string) "\000") t)
    (ehts-command nil)
    (save-excursion
      (set-buffer curbuf)
      (ehts-insert-in-link-table (cons linknum '())))))

(defun ehts-move-link-number (linknum)
  "Move link \"linknum\" to another node"
  (let (nodename nodenum)
    (setq nodename (completing-read "Move to node named: "
				    ehts-node-name-alist nil nil nil))
    (if (not (assoc nodename ehts-node-name-alist))
	(ehts-get-node nodename))
    (setq nodenum (cdr (assoc nodename ehts-node-name-alist)))
    (ehts-command t)
    (ehts-hb-sys-call "move link" linknum nil nil nil) 
    (if (not (eq (ehts-hb-sys-call nil nodenum nil nil t) 0))
	(progn
	  (ehts-command nil)
	  (error "Can't move link, panic !!! ")))
    (ehts-hb-sys-call "write" linknum "l last modified by"
		      (concat (user-full-name) "\000") t)
    (ehts-hb-sys-call "write" linknum "l last modified date"
		      (concat (current-time-string) "\000") t)
    (ehts-command nil)
    nodenum))

(defun ehts-replace-link-string (link newname alist)
  "Replace link string in node."
  (save-excursion
    (let (ret)
      (goto-char (point-min))
      (replace-string (concat "[-> " (car (rassq link alist)) "]")
		      (concat "[-> " newname "]"))
      (setq alist (ehts-remove-from-table link alist))
      (setq alist (cons (cons newname link) alist))
      (ehts-command t)
      (ehts-hb-sys-call "write" link "l name" (concat newname "\000") t)
      (ehts-hb-sys-call "write" link "l last modified by"
			(concat (user-full-name) "\000") t)
      (ehts-hb-sys-call "write" link "l last modified date"
			(concat (current-time-string) "\000") t)
      (ehts-command nil)))
  alist)

(defun ehts-remove-link-string (linknum)
  "Remove the link string from the node ([-> link])."
  (save-excursion
    (goto-char (point-min))
    (replace-string (concat "[-> " (car (rassq linknum ehts-node-link-alist))
			    "]") "")))

(defun ehts-delete-all-links (alist)
  "Delete all links in a node, when this has to be deleted."
  (let (linknum)
    (while (not (equal alist nil))
      (setq linknum (cdr (car alist)))
      (ehts-command t)
      (if (= (ehts-hb-sys-call "delete" linknum nil nil t) 0)
	  (ehts-command nil)
	(ehts-command nil)
	(error "Can't delete link, panic !!!"))
      (setq alist (cdr alist)))))

(defun ehts-delete-link (node link)
  "Deletes a link in a datanode."
  (ehts-command t)
  (ehts-hb-sys-call "remove link" nodenum nil nil nil)
  (if (= (ehts-hb-sys-call nil linknum nil nil t) 0)
      (message "(OK)")
    (ehts-command nil)
    (error "Can't remove link from node, panic !!!"))
  (if (= (ehts-hb-sys-call "delete" linknum nil nil t) 0)
      (message "(OK)")
    (ehts-command nil)
    (error "Can't delete link, panic !!!"))
  (ehts-command nil))

(defun ehts-insert-in-link-table (list-of-numbers)
  "Make a alist of link names and numbers."
  (while (not (equal list-of-numbers nil))
    (let (num)
      (setq num (car list-of-numbers))
      (message "Building list of link name (%d)" num)
      (ehts-command t)
      (if (< (ehts-hb-sys-call "read" num "l name" nil t) 0)
	  (progn
	    (ehts-command nil)
	    (error "Reading link name failed, panic !!!"))
	(setq ehts-node-link-alist
	      (cons (cons (ehts-read-null-string) num) ehts-node-link-alist))
	(ehts-command nil)
	(setq list-of-numbers (cdr list-of-numbers)))))
  (message "(OK)"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions working on LINKS and NODES (ATTRIBUTES)
;;; =================================================
;;; ehts-delete (C-c C-d)
;;; ehts-attributes (C-c C-a)
;;; ehts-setup-attribute-screen
;;; ehts-create-attribute-alist
;;; ehts-change-attributes
;;; ehts-update
;;; ehts-create-node-attribute-string
;;; ehts-create-link-attribute-string
;;; ehts-kill-buffer
;;; ehts-kill-attribute-screen-and-buffer
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar ehts-attribute-buffer "")
(make-variable-buffer-local 'ehts-attribute-buffer)

(defvar ehts-attribute-entity 0)
(make-variable-buffer-local 'ehts-attribute-entity)

(defvar ehts-attribute-alist '())
(make-variable-buffer-local 'ehts-attribute-alist)

(defun ehts-delete ()
  "Delete an entity (node/link)."
  (interactive)
  (let (entity nodenum linknum ret buffer screen string)
    (setq entity (cdr (assoc (completing-read "Delete (node/link): "
			       node-link-list nil t nil) node-link-list)))
    (if (= entity 0)
	;; delete node
	(progn				
	  (setq nodenum (cdr (assoc (completing-read "Delete node named: "
		     ehts-node-name-alist nil t nil) ehts-node-name-alist)))
	  (setq string (concat "Really delete node named \""
		       (car (rassq nodenum ehts-node-name-alist)) "\" ? "))
	  (if (y-or-n-p string)
	      (progn
		(let (alist)
		  (setq alist (ehts-read-all-link-names nodenum))
		  (setq buffer (car (rassq nodenum ehts-node-name-alist)))
		  (ehts-command t)
		  (setq ret (ehts-hb-sys-call "delete" nodenum nil nil t))
		  (ehts-command nil)
		  (cond ((= ret -204)
			 (error
			  "Can't delete node when still referenced !!!"))
			((= ret 350)
			 (error "Can't delete locked node !!!"))
			((= ret 0)
			 (if (get-buffer buffer)
			     (progn
			       (setq screen (ehts-find-buffer-screen buffer))
			       (kill-buffer buffer)
			       (if screen
				   (remove-screen screen))))
			 (ehts-delete-all-links alist)
			 (ehts-unlock nodenum)
			 (message "(OK)"))
			(t (error "Unknown error")))))
	    (message " ")))
      ;; delete link
      (if (equal ehts-node-link-alist '())
	  (error "No links in this node!"))
      (if buffer-read-only
	  (error "Can't delete link from read-only node."))
      (setq linknum (cdr (assoc (completing-read "Delete link named: "
		 ehts-node-link-alist nil t nil) ehts-node-link-alist)))
      (setq string (concat "Really delete link named \""
		   (car (rassq linknum ehts-node-link-alist)) "\" ? "))
      (if (y-or-n-p string)
	  (progn      
	    (ehts-command t)
	    (setq nodenum (cdr (assoc (buffer-name) ehts-node-name-alist)))
	    (ehts-delete-link nodenum linknum)
	    (ehts-remove-link-string linknum)
	    (setq ehts-node-link-alist
		  (ehts-remove-from-table linknum ehts-node-link-alist))
	    (ehts-command nil))
	(message " ")))))

(defun ehts-attributes ()
  "Show the attributes of a node or a link."
  (interactive)
  (let (entity name string number buffer screenid)
    (setq buffer (buffer-name))
    (setq entity (cdr (assoc
		       (completing-read "Show attributes in (node/link): "
					node-link-list nil t nil)
		       node-link-list)))
    (if (= entity 0)
	(progn
	  (setq name (concat "Attributes - " buffer))
	  (if (not (ehts-allready-displayed name))
	      (progn
		(setq number (cdr (assoc buffer ehts-node-name-alist)))
		(setq string (ehts-create-node-attribute-string number))
		(ehts-setup-attribute-screen name string entity buffer))))
      (if (eq ehts-node-link-alist '())
	  (error "No links in this node."))
      (setq name (concat "Attributes - " (car (assoc
	       (completing-read "Show attributes in link named: "
		ehts-node-link-alist nil t nil) ehts-node-link-alist))))
      (if (not (ehts-allready-displayed name))
	  (progn
	    (setq number (cdr (assoc (substring name 13)
				     ehts-node-link-alist)))
	    (setq string (ehts-create-link-attribute-string number))
	    (ehts-setup-attribute-screen name string entity buffer))))))

(defun ehts-setup-attribute-screen (name string entity buffer)
  "Setup a screen to show attributes in a node/link."
  (get-buffer-create name)
  (create-screen name (list (cons 'font "fixed")
			    (cons 'title name)
			    (cons 'geometry "75x20")))
  (set-buffer name)
  (ehts-mode 1)
  (setq buffer-read-only nil)
  (erase-buffer)
  (goto-char (point-min))
  (insert-string string)
  (setq buffer-read-only t)
  (set-buffer-modified-p nil)
  (setq ehts-attribute-entity entity)
  (setq ehts-attribute-buffer buffer)
  (ehts-create-attribute-alist entity)
  (local-unset-key "\C-x\C-f")
  (local-unset-key "\C-x\C-s")
  (local-unset-key "\C-cb")
  (local-unset-key "\C-cc")
  (local-unset-key "\C-cd")
  (local-unset-key "\C-c\C-a")
  (local-unset-key "\C-c\C-b")
  (local-unset-key "\C-c\C-d")
  (local-unset-key "\C-c\C-e")
  (local-unset-key "\C-c\C-f")
  (local-unset-key "\C-c\C-l")
  (local-unset-key "\C-c\C-m")
  (local-unset-key "\C-c\C-u")
  (local-set-key "\C-c\C-c" 'ehts-change-attributes)
  (local-set-key "\C-c\C-k" 'ehts-kill-attribute-screen-and-buffer))
    
(defun ehts-create-attribute-alist (entity)
  "Create alist to keep changeable attributes in attribute-buffer."
  (if (= entity 0)
      (progn
	(let (p1 p2)
	  (goto-line 4)
	  (setq p1 (point))
	  (end-of-line)
	  (setq p2 (1- (point)))
	  (setq ehts-attribute-alist
		(cons (cons "Name" (substring string p1 p2))
		      ehts-attribute-alist))
	  (goto-line 9)
	  (setq p1 (point))
	  (end-of-line)
	  (setq p2 (1- (point)))
	  (setq ehts-attribute-alist
		(cons (cons "Font" (substring string p1 p2))
		      ehts-attribute-alist))
	  (goto-line 10)
	  (setq p1 (point))
	  (end-of-line)
	  (setq p2 (1- (point)))
	  (setq ehts-attribute-alist
		(cons (cons "Geometry" (substring string p1 p2))
		      ehts-attribute-alist))))
    (let (p1 p2)
      (goto-line 4)
      (setq p1 (point))
      (end-of-line)
      (setq p2 (1- (point)))
      (setq ehts-attribute-alist
	    (cons (cons "Name" (substring string p1 p2))
		  ehts-attribute-alist))
      (goto-line 14)
      (setq p1 (point))
      (end-of-line)
      (setq p2 (1- (point)))
      (setq ehts-attribute-alist
	    (cons (cons "To node number" (substring string p1 p2))
		  ehts-attribute-alist)))))

(defun ehts-change-attributes ()
  "Change attributes in a node or a link."
  (interactive)
  (if (not (assoc ehts-attribute-buffer ehts-edit-alist))
      (error "Node \"%s\" must be locked." ehts-attribute-buffer)
    (let (attribute string newstring curbuf oldscreen screen
		    completion-ignore-case)
      (setq completion-ignore-case t)
      (setq curbuf (buffer-name))
      (setq attribute (car (assoc
			    (completing-read "Change attribute: "
					     ehts-attribute-alist nil t nil)
			    ehts-attribute-alist)))
      (setq string (cdr (assoc attribute ehts-attribute-alist)))
      (cond ((equal attribute "Name")
	     (if (= ehts-attribute-entity 0) ; entity = node
		 (progn
		   (setq newstring (completing-read "New node name: "
					    ehts-node-name-alist nil nil nil))
		   (if (assoc newstring ehts-node-name-alist)
		       (error "Allready node with that name !"))
		   (setq oldscreen (current-screen))
		   (save-excursion
		     (setq screen (ehts-find-buffer-screen
				   ehts-attribute-buffer))
		     (set-buffer ehts-attribute-buffer)
		     (rename-buffer newstring)
		     (select-screen screen)
		     (title newstring screen)
		     (select-screen oldscreen))
		   (setq newstring (concat "(1) *Name..............."
					   newstring))
		   (setq buffer-read-only nil)
		   (save-excursion
		     (goto-char (point-min))
		     (replace-string string newstring))
		   (rename-buffer (concat "Attributes - "
					  (substring newstring 24)))
		   (title (concat "Attributes - "
				  (substring newstring 24))
			  oldscreen)
		   (set-buffer-modified-p nil)
		   (setq buffer-read-only t)
		   (setq ehts-attribute-alist (ehts-remove-from-table
					       string ehts-attribute-alist))
		   (setq ehts-attribute-alist (cons
					       (cons "Name" newstring)
					       ehts-attribute-alist))
		   (setq number (cdr (assoc ehts-attribute-buffer
					    ehts-node-name-alist)))
		   (setq ehts-attribute-buffer (substring newstring 24))
		   (ehts-command t)
		   (if (not (eq (ehts-hb-sys-call "write" number 7
					  (concat (substring newstring 24)
							  "\000") t) 0))
		       (progn
			 (ehts-command nil)
			 (error "Can't write attribute in node, panic !!!")))
		   (ehts-command nil)
		   (ehts-update ehts-attribute-entity number))
	       (let (link)
		 (save-excursion		; entity = link
		   (set-buffer ehts-attribute-buffer) 
		   (setq newstring (completing-read "New link name: "
					    ehts-node-link-alist nil nil nil))
		   (if (assoc newstring ehts-node-link-alist)
		       (error "Allready link with that name !"))
		   (setq link (cdr (assoc (substring string 24)
					  ehts-node-link-alist)))
		   (setq ehts-node-link-alist
			 (ehts-replace-link-string link newstring
						   ehts-node-link-alist)))
		 (setq newstring (concat "(1) *Name..............."
					 newstring))
		 (setq buffer-read-only nil)
		 (save-excursion
		   (goto-char (point-min))
		   (replace-string string newstring))
		 (rename-buffer (concat "Attributes - "
					(substring newstring 24)))
		 (title (concat "Attributes - "
				(substring newstring 24))
			oldscreen)
		 (set-buffer-modified-p nil)
		 (setq buffer-read-only t)
		 (setq ehts-attribute-alist (ehts-remove-from-table
					     string ehts-attribute-alist))
		 (setq ehts-attribute-alist (cons
					     (cons "Name" newstring)
					     ehts-attribute-alist)))))
	    ((equal attribute "Font")
	     (setq newstring (completing-read "New font: "
					      ehts-font-types nil t nil))
	     (setq oldscreen (current-screen))
	     (save-excursion
	       (setq screen (ehts-find-buffer-screen ehts-attribute-buffer))
	       (select-screen screen)
	       (font newstring)
	       (select-screen oldscreen))
	     (setq newstring (concat "(6) *Font..............."
				     newstring))
	     (setq buffer-read-only nil)
	     (save-excursion
	       (goto-char (point-min))
	       (replace-string string newstring))
	     (set-buffer-modified-p nil)
	     (setq buffer-read-only t)
	     (setq ehts-attribute-alist (ehts-remove-from-table
					 string ehts-attribute-alist))
	     (setq ehts-attribute-alist (cons (cons "Font" newstring)
					      ehts-attribute-alist))
	     (setq number (cdr (assoc ehts-attribute-buffer
				      ehts-node-name-alist)))
	     (ehts-command t)
	     (if (not (eq (ehts-hb-sys-call "write" number 2
					    (concat (substring newstring 24)
						    "\000") t) 0))
		 (progn
		   (ehts-command nil)
		   (error "Can't write attribute in node, panic !!!")))
	     (ehts-command nil)
	     (ehts-update ehts-attribute-entity number))
	    ((equal attribute "Geometry")
	     (let (x y)
	       (setq x (call-interactively 'ehts-get-window-width))
	       (setq y (call-interactively 'ehts-get-window-heigth))
	       (setq newstring (concat x "x" y))
	       (setq oldscreen (current-screen))
	       (save-excursion
		 (setq screen (ehts-find-buffer-screen
			       ehts-attribute-buffer))
		 (select-screen screen)
		 (change-screen-size (string-to-int x)
					    (string-to-int y))
		 (redisplay-screen)
		 (setq fill-column (window-width))
		 (select-screen oldscreen)))
	     (setq newstring (concat "(7) *Geometry..........." newstring))
	     (setq buffer-read-only nil)
	     (save-excursion
	       (goto-char (point-min))
	       (replace-string string newstring))
	     (set-buffer-modified-p nil)
	     (setq buffer-read-only t)
	     (setq ehts-attribute-alist (ehts-remove-from-table
					 string ehts-attribute-alist))
	     (setq ehts-attribute-alist (cons
					 (cons "Geometry" newstring)
					 ehts-attribute-alist))
	     (setq number (cdr (assoc ehts-attribute-buffer
				      ehts-node-name-alist)))
	     (ehts-command t)
	     (if (not (eq (ehts-hb-sys-call "write" number 1
					    (concat (substring newstring 24)
						    "\000") t) 0))
		 (progn
		   (ehts-command nil)
		   (error "Can't write attribute in node, panic !!!")))
	     (ehts-command nil)
	     (ehts-update ehts-attribute-entity number))
	    ((equal attribute "To node number")
	     (let (linkname linknum nodename)
	       (setq linkname (substring (cdr (assoc "Name"
						     ehts-attribute-alist))
					 24))
	       (save-excursion
		 (set-buffer ehts-attribute-buffer)
		 (setq linknum (cdr (assoc linkname ehts-node-link-alist))))
	       (setq newstring (ehts-move-link-number linknum))
	       (setq nodename (car (rassq newstring ehts-node-name-alist)))
	       (setq newstring (concat "(8) *To node number....."
				       newstring
				       " = "
				       nodename))
	       (setq buffer-read-only nil)
	       (save-excursion
		 (goto-char (point-min))
		 (replace-string string newstring))
	       (set-buffer-modified-p nil)
	       (setq buffer-read-only t)
	       (setq ehts-attribute-alist (ehts-remove-from-table
					   string ehts-attribute-alist))
	       (setq ehts-attribute-alist (cons
					   (cons "To node number" newstring)
					   ehts-attribute-alist))))))))
  
(defun ehts-update (entity number)
  "Write the modified attributes in an entity."
  (if (= entity 0)
      (progn
	(ehts-command t)
	(if (not (eq (ehts-hb-sys-call "write" number 4
				       (concat (user-full-name) "\000") t) 0))
	    (progn
	      (ehts-command nil)
	      (error "Can't write attribute in node, panic !!!")))
	(if (not (eq (ehts-hb-sys-call "write" number 3
				       (concat (current-time-string)
					       "\000") t) 0))
	    (progn
	      (ehts-command nil)
	      (error "Can't write attribute in node, panic !!!")))
	(ehts-command nil))
    (ehts-command t)
    (if (not (eq (ehts-hb-sys-call "write" number 257
				   (concat (user-full-name) "\000") t) 0))
	(progn
	  (ehts-command nil)
	  (error "Can't write attribute in link, panic !!!")))
    (if (not (eq (ehts-hb-sys-call "write" number 256
				   (concat (current-time-string) "\000") t) 0))
	(progn
	  (ehts-command nil)
	  (error "Can't write attribute in link, panic !!!")))
    (ehts-command nil)))

(defun ehts-create-node-attribute-string (number)
  "Create and return a string of all attributes in a node."
  (let (hb-string string)
    (message "Reading attributes....")
    (setq hb-string (concat (ehts-int-to-4bytes 100)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 7)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 6)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 5)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 4)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 3)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 2)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 513)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 514)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 515)
			    (ehts-int-to-4bytes 100)))
    (ehts-command t)
    (process-send-string ehts-write-object hb-string)
    (setq string "\nEmacs HyperText System attributes:\n\n")
    (ehts-read-4bytes)
    (setq string (concat string " (1) *Name..............."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (2) Created by.........."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (3) Created date........"
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (4) Last modified by...."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (5) Last modified date.."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (6) *Font..............."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (7) *Geometry..........."
			 (ehts-read-null-string) "\n"))
    (setq string (concat string "\nHyberBase attributes:\n\n"))
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (setq string (concat string " (8) Data node number...."
			 (ehts-read-4bytes) "\n"))
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (setq string (concat string " (9) Data size..........."
			 (ehts-read-4bytes) "\n"))
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (setq string (concat string "(10) Links to me........."
			 (ehts-read-4bytes)
			 "\n\n\nAttributes marked with an * can be changed"
			 " - use C-c C-c."))
    (ehts-command nil)
    (message "Reading attributes....done")
    string))

(defun ehts-create-link-attribute-string (number)
  "Create and return a string of all attributes in a link."
  (let (hb-string string to-node)
    (message "Reading attributes....")
    (setq hb-string (concat (ehts-int-to-4bytes 100)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 260)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 259)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 258)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 257)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 256)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 1025)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 1026)
			    (ehts-int-to-4bytes 1)
			    (ehts-int-to-4bytes number)
			    (ehts-int-to-4bytes 1027)
			    (ehts-int-to-4bytes 100)))
    (ehts-command t)
    (process-send-string ehts-write-object hb-string)
    (setq string "\nEmacs HyperText System attributes:\n\n")
    (ehts-read-4bytes)
    (setq string (concat string " (1) *Name..............."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (2) Created by.........."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (3) Created date........"
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (4) Last modified by...."
			 (ehts-read-null-string) "\n"))
    (ehts-read-4bytes)
    (setq string (concat string " (5) Last modified date.."
			 (ehts-read-null-string) "\n"))
    (setq string (concat string "\nHyberBase attributes:\n\n"))
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (setq string (concat string " (6) Link number........."
			 (ehts-read-4bytes) "\n"))
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (setq string (concat string " (7) Use count..........."
			 (ehts-read-4bytes) "\n"))
    (ehts-read-4bytes)
    (ehts-read-4bytes)
    (setq to-node (ehts-read-4bytes))
    (ehts-command nil)
    (setq string (concat string " (8) *To node number....."
			 to-node
			 " = "
			 (car (rassq to-node ehts-node-name-alist))
			 "\n\n\nAttributes marked with an * can be changed"
			 "- use C-c C-c."))
    (message "Reading attributes....done")
    string))

(defun ehts-kill-buffer ()
  "To shadow the real function."
  (interactive)
  (error "\"kill-buffer\" is switched off - use C-c C-k."))

(defun ehts-kill-attribute-screen-and-buffer ()
  "Remove attribute buffer and it's screen."
  (interactive)
  (kill-buffer (buffer-name))
  (remove-screen))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions working on EVENTS
;;; ===========================
;;; ehts-user-setup-event
;;; ehts-command
;;; ehts-parse-event
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-user-setup-event ()
  "Setup events, ask user."
  (let (node ehts-node-name-alist operation key)
    (setq ehts-node-name-alist (cons (cons "all" 0) ehts-node-name-alist))
    (setq node (completing-read "Event on node named: " ehts-node-name-alist
				nil t nil))
    (if (string= node "all")
	(setq node 0))
    (setq operation (completing-read "Event on operation: "
				     ehts-hb-operations nil t nil))
    (setq key (completing-read "Event on key: " ehts-hb-keys nil t nil))
    (ehts-setup-events node operation key)))

(defun ehts-command (t-or-nil)
  "Set the global flag \"ehts-in-command\"."
  (if t-or-nil
      (progn
	(setq ehts-in-command t)
	(setq inhibit-quit t))
    (setq ehts-in-command nil)
    (setq quit-flag nil)
    (setq inhibit-quit nil)
    (if ehts-pending-browser-command
	(ehts-parse-browser-command))
    (if ehts-pending-event
	(ehts-parse-event))))

(defun ehts-parse-event ()
  "Parse the incomming event."
  (let (event-str ent-no operation key userid opname keyname new)
;    (message "Handling events...")
    (while (car ehts-event-list)	; while elements in list
      (setq event-str (car ehts-event-list))        ; list string
      (setq ehts-event-list (cdr ehts-event-list))  ; delete first element
      (setq userid "")			            ; erase old
      (while (not (= (string-to-char event-str) 0)) ; read the username
	(setq userid (concat userid (substring event-str 0 1)))
	(setq event-str (substring event-str 1)))
      (setq ent-no (4chars-to-int (substring event-str 1 5 )))
      (setq operation (4chars-to-int (substring event-str 5 9)))
      (setq opname (car (rassq operation ehts-hb-operations)))
      (setq key (4chars-to-int (substring event-str 9 13)))
      (setq keyname (car (rassq key ehts-hb-keys)))
      ;; Then we can parse the incomming event, by looking at the
      ;; operation and key (attribute).
      (if (rassq ent-no ehts-node-name-alist)
	  (setq new nil)
	(setq new t))
      (cond ((string= opname "write")
	     (let (buffer screen oldscreen rename newbuffer)
	       (setq oldscreen (current-screen))
	       (cond ((string= keyname "n name") ; new node or a rename
		      (setq buffer (car (rassq ent-no ehts-node-name-alist)))
		      (if (rassq ent-no ehts-node-name-alist)
			    (setq ehts-node-name-alist 
				  (ehts-remove-from-table
				   ent-no ehts-node-name-alist)))
		      (ehts-insert-in-node-table (cons ent-no '()))
		      (setq newbuffer (car (rassq ent-no
						  ehts-node-name-alist)))
		      (if new
			  ()
			(if (assoc buffer ehts-edit-alist)
			    (progn
			      (setq ehts-edit-alist (ehts-remove-from-table
						     ent-no ehts-edit-alist))
			      (setq ehts-edit-alist (cons
						     (cons newbuffer ent-no)
						     ehts-edit-alist)))
			  (setq screen (ehts-find-buffer-screen buffer))
			  (save-excursion
			    (if screen
				(progn
				  (set-buffer buffer)
				  (rename-buffer newbuffer)
				  (select-screen screen)
				  (title newbuffer screen)
				  (select-screen oldscreen)))))))
		     ((string= keyname "data")
		      (setq buffer (car (rassq ent-no ehts-node-name-alist)))
		      (setq screen (ehts-find-buffer-screen buffer))
		      (if (assoc buffer ehts-edit-alist)
			  ()
			(let (data-string)
			  (save-excursion
			    (if screen
				(progn
				  (set-buffer buffer)
				  (ehts-command t)
				  (ehts-hb-sys-call "read" ent-no "data" nil t)
				  (setq data-string (ehts-read-string))
				  (ehts-command nil)
				  (setq buffer-read-only nil)
				  (erase-buffer)
				  (insert-string data-string)
				  (setq ehts-node-link-alist
					(ehts-read-all-link-names buffer))
				  (set-buffer-modified-p nil)
				  (setq buffer-read-only t)
				  (goto-char (point-min))))))
			(if (string= buffer "global.message")
			    (if (not (ehts-screen-already-displayed buffer))
				(ehts-get-node buffer)))
			(if (string= buffer (concat (user-login-name)
						    ".message"))
			    (if (not (ehts-screen-already-displayed buffer))
				(ehts-get-node buffer)))))
		     ((string= keyname "font")
		      (if new
			  ()
			(setq buffer (car (rassq ent-no
						 ehts-node-name-alist)))
			(setq screen (ehts-find-buffer-screen buffer))
			(if (assoc buffer ehts-edit-alist)
			    ()
			  (save-excursion
			    (if screen
				(progn
				  (select-screen screen)
				  (ehts-command t)
				  (ehts-hb-sys-call "read" ent-no "font" nil t)
				  (font (ehts-read-null-string))
				  (ehts-command nil)
				  (select-screen oldscreen)))))))
		     ((string= keyname "geometry")
		      (if new
			  ()
			(setq buffer (car (rassq ent-no
						 ehts-node-name-alist)))
			(setq screen (ehts-find-buffer-screen buffer))
			(if (assoc buffer ehts-edit-alist)
			    ()
			  (save-excursion
			    (if screen
				(progn
				  (let (size x y temp)
				    (select-screen screen)
				    (ehts-command t)
				    (ehts-hb-sys-call "read" ent-no
						      "geometry" nil t)
				    (setq size (ehts-read-null-string))
				    (ehts-command nil)
				    (setq x (string-to-int size))
				    (setq temp (int-to-string x))
				    (setq size (substring
						size (1+ (length temp))))
				    (setq y (string-to-int size))
				    (change-screen-size x y)
				    (redisplay-screen)
				    (setq fill-column (- (window-width) 3))
				    (select-screen oldscreen)))))))))))
	    ((string= opname "lock")
	     (setq buffer (car (rassq ent-no ehts-node-name-alist)))
	     (setq screen (ehts-find-buffer-screen buffer))
	     (let (user notlocked othernode ownnode notopened)
	       (ehts-command t)
	       (if (< (ehts-hb-sys-call "show lock" buffer "data" nil t) 0)  
		   (progn
		     (ehts-command nil)
		     (error "Can't use \"show lock\", panic !!!")))
	       (setq user (ehts-read-null-string))
	       (ehts-command nil)
	       (setq othernode (concat user ".talk"))
	       (setq ownnode (concat (user-login-name) ".talk"))
	       (if screen
		   (progn
		     (if (string= user (user-login-name))
			 ()
		       (if (string= buffer ownnode)
			   ()
			 (beep)
			 (message "\"%s\" has locked node: \"%s\"" user
				  buffer)))))
	       (if (string= buffer ownnode)
		   (if (string= user (user-login-name))
		       ()
		     (setq notlocked t)
		     (if (not (ehts-screen-already-displayed othernode))
			 (setq notlocked (ehts-get-node othernode)))
		     (if (and notlocked)
			 (progn
			   (ehts-do-edit)
			   (if (not
				(ehts-screen-already-displayed ownnode))
			       (progn
				 (ehts-get-node ownnode)
				 (beep)
				 (message "\"%s\" wants to talk to you." user)
				 ))))))))
	    ((string= opname "delete")
	     (if (= (% ent-no 2) 0)
		 (progn
		   (setq buffer (car (rassq ent-no ehts-node-name-alist)))
		   (setq screen (ehts-find-buffer-screen buffer))
		   (if screen
		       (progn
			 (kill-buffer buffer)
			 (remove-screen screen)))
		   (setq ehts-node-name-alist (ehts-remove-from-table
					       ent-no ehts-node-name-alist)))))
	    ((string= opname "unlock")
	     (beep)
	     (pop-to-buffer "*** EHTS EVENT ***")
	     (erase-buffer)
	     (insert-string (concat "\"" userid "\"\n"))
	     (insert-string "--- Has Unlocked Node ---\n")
	     (insert-string
	      (concat "\"" (car (rassq ent-no
				       ehts-node-name-alist)) "\"\n"))
	     (insert-string "(type \"C-x 0\" to remove this)")
	     (message "\"%s\" has unlocked node \"%s\"." userid
		      (car (rassq ent-no ehts-node-name-alist)))
	     (sit-for 2)
	     (ehts-command t)
	     (ehts-hb-sys-call "unevent" ent-no nil nil nil)
	     (ehts-hb-sys-call operation nil key nil t)
	     (ehts-command nil))))
;    (message "Handling events...done!")
    (setq ehts-pending-event nil)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; General Ehts functions
;;; =======================
;;; ehts-kill-screen-and-buffer (C-c C-k)
;;; ehts-set-fill-column (C-c C-f)
;;; ehts-center-region (C-c c)
;;; ehts-welcome
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-kill-screen-and-buffer ()
  "Remove current buffer and it's screen."
  (interactive)
  (if (string-match ".talk" (buffer-name))
      (if buffer-read-only
	  ()
	(erase-buffer)
	(ehts-write-buffer)))
  (if (not buffer-read-only)
      (ehts-unlock-node))
  (kill-buffer (buffer-name))
  (remove-screen))

(defun ehts-set-fill-column ()
  "Set the fill-column in current screen to screen-width minus 3."
  (interactive)
  (setq fill-column (- (screen-width) 3))
  (message "Fill-column set to: %s." fill-column))

(defun ehts-center-region ()
  "Center region in current screen with fill-column = (screen-width)."
  (interactive)
  (setq fill-column (- (screen-width) 3))
  (call-interactively 'center-region))

(defun ehts-welcome ()
  "Say welcome to the ehts."
  (switch-to-buffer "*Ehts Welcome*")
  (erase-buffer)
  (let (beg fill-column)
    (change-screen-size 45 12)
    (redisplay-screen)
    (font "7x14")
    (setq fill-column (window-width))
    (setq beg (point))
    (insert-string "\nE H T S\n=========\nEmacs HyperText System\n")
    (insert-string "Version 2 (900321)\n\n")
    (insert-string "(C) copyright 1990 by\n\n")
    (insert-string "Uffe Kock Wiil          Claus Bo Nielsen \n")
    (insert-string "(kock@iesd.auc.dk)     (cbn@iesd.auc.dk)")
    (center-region beg (point))
    (goto-char (point-min))
    (set-buffer-modified-p nil)
    (title "*Ehts Welcome*" (current-screen))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions doing "small" things
;;; ==============================
;;; ehts-remove-from-table
;;; ehts-find-buffer-screen
;;; ehts-allready-displayed
;;; ehts-get-window-width
;;; ehts-get-window-heigth
;;; toggle-read-only
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-remove-from-table (number alist)
  "Remove an item from an alist."
  (let (alist-copy)
    (setq alist-copy (copy-alist alist))
    (setq alist '())
    (while (not (equal alist-copy nil))
      (if (not (equal number (cdr (car alist-copy))))
	  (setq alist (cons (car alist-copy) alist)))
      (setq alist-copy (cdr alist-copy)))) alist)

(defun ehts-find-buffer-screen (name)
  "Find the screen of buffer name, if not found return nil."
  (let (window)
    (setq window (epoch::get-buffer-window name))
    (if window
	(epoch::screen-of-window window)
      nil)))

(defun ehts-allready-displayed (name)
  "Test if a buffer allready is displayed"
  (let (screenid)
    (setq screenid (ehts-find-buffer-screen name))
    (if screenid
	(progn
	  (switch-screen screenid)
	  (error "Attributes is allready displayed.")
	  t)
      nil)))

(defun ehts-get-window-width (arg)
  "Get the new window width."
  (interactive "sNew window width: ")
  arg)

(defun ehts-get-window-heigth (arg)
  "Get the new window heigth."
  (interactive "sNew window height: ")
  arg)

(defun toggle-read-only ()
  "To shadow the real function."
  (interactive)
  (error "\"toggle-read-only\" is switched off - use C-c C-e."))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Functions converting from char to int and vica versa
;;; ====================================================
;;; 4chars-to-int
;;; 3chars-to-int
;;; ehts-int-to-4bytes
;;; ehts-int-to-3bytes
;;; in-string
;;; ehts-read-null-string
;;; ehts-read-4bytes
;;; ehts-read-string
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun 4chars-to-int (str)
  "Convert 4 chars (32bit) to 3 chars, a Emacs int."
  (cond ((= (string-to-char (substring str 0 1)) 0)
	 (3chars-to-int (substring str 1 4)))
    (t (+ -8388608 (3chars-to-int (substring str 1 4))))
    ))
      
(defun 3chars-to-int (str)
  "Convert 3 chars to a Emacs int."
  (+ (* 65536 (mod (string-to-char (substring str 0 1)) 128))
     (* 256 (string-to-char (substring str 1 2)))
     (string-to-char (substring str 2 3))))
  
(defun ehts-int-to-4bytes (num)
  "convert a Emacs int to 4bytes."
  (cond ((< num 0)
	 (concat "\377" (ehts-int-to-3bytes (- num -8388608))))
	(t (concat "\0" (ehts-int-to-3bytes num)))))

(defun ehts-int-to-3bytes (num)
  "convert a Emacs int to 3bytes."
  (concat (char-to-string (mod (/ num 65536) 128))
	  (char-to-string (mod (/ num 256) 256))
	  (char-to-string (mod num 256))))

(defun in-string (thechar thestr)
  "Find if CHAR is in STRING, return non-nil if found."
  (let (len flag)
    (setq len (length thestr))
    (setq flag nil)
    (while (and (> len 0) (not flag))
      (if (string= (substring thestr 0 1) thechar)
	  (setq flag t)
	(setq thestr (substring thestr 1))
	(setq len (1- len))))
    flag))

(defun ehts-read-null-string ()
  "Return a null-string."
  (let (string nullstring)
    (setq string (ehts-read-string))
    (while (not (= (string-to-char string) 0))
      (setq nullstring (concat nullstring (substring string 0 1)))
      (setq string (substring string 1)))
    nullstring))

(defun ehts-read-4bytes ()
  "Read 4 bytes from the read-port."
  (while (< (length ehts-return-value) 4)
    (accept-process-output ehts-read-object))
  (setq tmp (substring ehts-return-value 0 4))
  (setq ehts-return-value
	(substring ehts-return-value 4 (length ehts-return-value)))
  (4chars-to-int tmp))

(defun ehts-read-string ()
  "Read a string, send from the HyperBase."
  (let (size return-value)
    (setq size (ehts-read-4bytes))	; read the length (in bytes)
    (while (> size (length ehts-return-value))
      (accept-process-output ehts-read-object))
    (setq return-value (substring ehts-return-value 0 size))
    (setq ehts-return-value
	  (substring ehts-return-value size (length ehts-return-value)))
    return-value))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SPECIEL TEST FUNCTIONS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-read-all-names ()
  "A fast browser for names, putting into the alist."
  (let (number-list tmp-list hb-string)
    (setq number-list (ehts-browse "data"))
    (setq tmp-list number-list)
    (setq hb-string (ehts-int-to-4bytes 100))
    (message "Building list of node names...")
    (while (not (equal tmp-list nil))
      (let (num)
	(setq num (car tmp-list))
	(setq hb-string (concat hb-string (concat (ehts-int-to-4bytes 1)
						  (ehts-int-to-4bytes num)
						  (ehts-int-to-4bytes 7))))
	(setq tmp-list (cdr tmp-list))))
    (message "Building list of node names...done!")
    (setq hb-string (concat hb-string (ehts-int-to-4bytes 100)))
    (ehts-command t)
    (process-send-string ehts-write-object hb-string)
    (message "Extracting names...")
    (while (not (equal number-list nil))
      (let (num)
	(setq num (car number-list))
	(ehts-read-4bytes)		; the return value of READ
	(setq ehts-node-name-alist
	      (cons (cons (ehts-read-null-string) num) ehts-node-name-alist))
	(setq number-list (cdr number-list))))
    (ehts-command nil)
    (message "Extracting names...done!")))

(defun ehts-read-all-link-names (nodenum)
  "A fast browser for link names, putting into the alist."
  (ehts-command t)
  (if (< (ehts-hb-sys-call "read" nodenum "link num" nil t) 0)
      (progn
	(ehts-command nil)
	(error "Can't read linknum key, panic!!!"))
    (let (size hb-string number-list flag link-alist)
      (setq size (ehts-read-4bytes))
      (if (> size 0)
	  (progn
	    (setq hb-string (ehts-int-to-4bytes 100))
	    (setq number-list '())
	    (message "Building list of link names...")
	    (while (> size 0)
	      (let (num)
		(setq num (ehts-read-4bytes))
		(setq number-list (cons num number-list))
		(setq hb-string (concat hb-string
					(concat (ehts-int-to-4bytes 1)
						(ehts-int-to-4bytes num)
						(ehts-int-to-4bytes 260))))
		(setq size (- size 4))))
	    (message "Building list of link names...done!")
	    (setq hb-string (concat hb-string (ehts-int-to-4bytes 100)))
	    (process-send-string ehts-write-object hb-string)
	    (message "Extracting link names...")
	    (setq number-list (reverse number-list))
	    (while (not (equal number-list nil))
	      (let (num)
		(setq num (car number-list))
		(ehts-read-4bytes)		; the return value of READ
		(setq link-alist
		      (cons (cons (ehts-read-null-string) num)
			    link-alist))
		(setq number-list (cdr number-list))))
	    (message "Extracting link names...done!")))
      (ehts-command nil) link-alist )))


 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; BROWSER functions
;;; =================
;;; ehts-connect-to-browser
;;; ehts-disconnect-from-browser
;;; ehts-parse-browser-command
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-connect-to-browser ()
  "Connect to a browser, if any on the EHTS client machine."
  (interactive)
  (let (ehts-browser-machine)
    (setq ehts-browser-machine "")
    (if (string-equal ehts-browser-on-machine "")
	(setq ehts-browser-on-machine (call-interactively
				       'ehts-get-browser-machine))
      (setq ehts-browser-machine (call-interactively
				  'ehts-get-new-browser-machine)))
    (if (not (string-equal ehts-browser-machine ""))
	(setq ehts-browser-on-machine ehts-browser-machine))
    (message "Connecting to browser on \"%s\"." ehts-browser-on-machine)
    (setq ehts-browser-object (open-network-stream
			       "ehts-browser-socket" nil ehts-browser-on-machine
			       ehts-browser-socket))
    (set-process-filter ehts-browser-object 'ehts-browser-process-filter)
    (message "Connected to browser on \"%s\"." ehts-browser-on-machine)
    (setq ehts-connected-to-a-browser t)))

(defun ehts-get-browser-machine (arg)
  "Get the machine name of where the EHTS Browser is placed."
  (interactive "sConnect to EHTS Browser on machine: ")
  arg)

(defun ehts-get-new-browser-machine (arg)
  "Get the machine name of where the EHTS Browser is placed."
  (interactive "sConnect to EHTS Browser (default previous browser) : ")
  arg)

(defun ehts-disconnect-from-browser ()
  "Disconnect from the browser."
  (interactive)
  (if ehts-connected-to-a-browser
      (progn
	(delete-process ehts-browser-object)
	(setq ehts-connected-to-a-browser nil)
	(message "Disconnected from browser on \"%s\"." ehts-browser-on-machine)
	(sit-for 2))))

(defun ehts-parse-browser-command ()
  "Parse the incoming browser command."
  (let (op ent num name buffer string entity)
    (if (> (length ehts-browser-command-string) 7)
	(progn
	  (setq op (4chars-to-int (substring
				   ehts-browser-command-string 0 4)))
	  (setq ent (4chars-to-int (substring
				    ehts-browser-command-string 4 8)))
	  (if (= (length ehts-browser-command-string) 12)
	      (progn
		(setq num (4chars-to-int (substring
					  ehts-browser-command-string
					  8 12)))
		(setq buffer (car (rassq num ehts-node-name-alist))))
	    (setq buffer (car (rassq ent ehts-node-name-alist))))
	  (cond ((= op 0)		; data
		 (if (= 0 (% ent 2))
		     (progn
		       (ehts-get-node buffer))))
		((= op 1)		; attributes
		 (setq entity (% ent 2))
		 (if (= entity 0)
		     (progn
		       (setq name (concat "Attributes - " buffer))
		       (if (not (ehts-allready-displayed name))
			   (progn
			     (setq string (ehts-create-node-attribute-string
					   ent))
			     (ehts-setup-attribute-screen name string
							  entity buffer))))
		   (ehts-command t)
		   (if (not (= (ehts-hb-sys-call "read" ent 260 nil t) 0))
		       (error "Can't read link name, panic !!!"))
		   (setq name (concat "Attributes - "
				      (ehts-read-null-string)))
		   (ehts-command nil)
		   (if (not (ehts-allready-displayed name))
		       (progn
			 (setq string (ehts-create-link-attribute-string ent))
			 (ehts-setup-attribute-screen name string
						      entity buffer))))))
	  (setq ehts-browser-command-string nil)))))
 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; MOUSE functions
;;; =================
;;; ehts-mouse-follow-link
;;; ehts-mouse-kill-screen-and-buffer
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(global-set-mouse mouse-left mouse-shift-up 'ehts-mouse-follow-link)
(global-set-mouse mouse-right mouse-shift-up
		  'ehts-mouse-kill-screen-and-buffer)   

(defun ehts-mouse-follow-link (arg)
  "Follow link where mouse is pointed when shift left mouse button is pushed." 
  (let (found beg end linkname linknum)
    (mouse::set-point arg)
    (setq found (search-backward "[-> " nil t nil))
    (if found
	(progn
	  (forward-char 4)
	  (setq beg (point))
	  (search-forward "]" nil t nil)
	  (backward-char 1)
	  (setq end (point))
	  (setq linkname (buffer-substring beg end))
	  (search-backward "[-> " nil t nil)
	  (if (not (buffer-modified-p))
	      ()
	    (beep)
	    (if (y-or-n-p "Save buffer before follow link? ")
		(ehts-write-buffer)))
	  (setq linknum (cdr (assoc linkname ehts-node-link-alist)))
	  (ehts-command t)
	  (if (< (ehts-hb-sys-call "read" linknum "to data node no" nil t) 0)
	      (progn
		(ehts-command nil)
		(error "Can't read \"to data node no\" in link, panic !!!")))
	  (ehts-read-4bytes)
	  (setq tonode (ehts-read-4bytes))
	  (if (< (ehts-hb-sys-call "read" tonode "n name" nil t) 0)
	      (progn
		(ehts-command nil)
		(error "Can't read \"name\" in data node, panic !!!")))
	  (ehts-get-node (ehts-read-null-string))
	  (ehts-command nil))
      (message "Link not found"))))

(defun ehts-mouse-kill-screen-and-buffer (arg)
  "Close screen where mouse is placed when shift right mouse button is pushed."
  (if (string-match ".talk" (buffer-name))
      (if buffer-read-only
	  ()
	(erase-buffer)
	(ehts-write-buffer)))
  (if (> (length (buffer-name)) 13)
      (if (not (equal "Attributes - " (substring (buffer-name) 0 13)))
	  (if (not buffer-read-only)
	      (ehts-unlock-node)))
    (if (not buffer-read-only)
	(ehts-unlock-node)))
  (kill-buffer (buffer-name))
  (remove-screen))
 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; COMMUNICATION functions
;;; =======================
;;; ehts-talk (C-c C-t)
;;; ehts-send-message (C-c C-s)
;;; ehts-screen-already-displayed
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ehts-talk ()
  "Talk to other user inside EHTS environment."
  (interactive)
  (let (user othernode ownnode notlocked)
    (setq user (completing-read "Talk to user: " ehts-user-alist nil t nil))
    (setq othernode (concat user ".talk"))
    (setq ownnode (concat (user-login-name) ".talk"))
    (setq notlocked t)
    (if (not (ehts-screen-already-displayed othernode))
	(setq notlocked (ehts-get-node othernode)))
    (if notlocked
	(progn
	  (ehts-do-edit)
	  (if (not (ehts-screen-already-displayed ownnode))
	      (ehts-get-node ownnode)))
      ;; kunne lave show lock her og svare hvem vedkommende allerede 
      ;; snakker med
      (beep)
      (message "User \"%s\" is busy talking" user))))

(defun ehts-send-message ()
  "Send a global message or a message to a single user inside EHTS
environment"
  (interactive)
  (let (user messagenode notlocked)
    (setq user (completing-read "Send message to (global = all users): "
				ehts-user-message-alist nil t nil))
    (setq messagenode (concat user ".message"))
    (setq notlocked t)
    (if (not (ehts-screen-already-displayed messagenode))
      (setq notlocked (ehts-get-node messagenode)))
    (if notlocked
	(ehts-do-edit)
      ;; kunne lave show lock her og svare hvem der er ved at
      ;; kreere en message
      (beep)
      (message "Messagenode: \"%s\" is already locked" messagenode))))
    

(defun ehts-screen-already-displayed (name)
  "Test if a buffer already is displayed"
  (let (screenid)
    (setq screenid (ehts-find-buffer-screen name))
    (if screenid
	(progn
	  (switch-screen screenid)
	  t)
      nil)))
