/*******************************************************************
**
** HyperBase was designed and implemented by:
**
**	Uffe Kock Wiil 		(kock@iesd.auc.dk)
**	Claus Bo Nielsen 	(cbn@cci.dk)
**	Carsten Ruseng Jakobsen (ruseng@sun.com)
**	Finn Soelvsten
**	Per Magnus Petersen
**	Poul Larsen
**	Hans Mejdahl Jeppesen
**
** at The University of Aalborg in Denmark autumn 1989, and is provided
** for unrestricted use provided that this legend is included on all
** tape media and as a part of the software program in whole or part.
** Users may copy or modify HyperBase without charge, but are not
** authorized to license or distribute it to anyone else except as part
** of a product or program developed by the user.
**  
** HyperBase IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING
** THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A
** PARTICULAR PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR
** TRADE PRACTICE.
**  
** HyperBase is provided with no support and without any obligation on
** the part of the authors, to assist in its use, correction,
** modification or enhancement.
** 
** THE AUTHORS SHALL HAVE NO LIABILITY WITH RESPECT TO THE INFRINGEMENT
** OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY HyperBase OR ANY PART
** THEREOF.
** 
** In no event will the authors and/or The University of Aalborg be
** liable for any lost revenue or profits or other special, indirect and
** consequential damages, even if the authors and/or The University of
** Aalborg has been advised of the possibility of such damages.
** 
** Please address all correspondence to:
** 
** Uffe Kock Wiil
** Department of Computer Science,
** The University of Aalborg,      Email:  kock@iesd.auc.dk
** Fredrik Bajers Vej 7E,          Phone:  + 45 98 15 42 11 (Ext 5051)
** DK-9220 Aalborg, Denmark.       Fax:    + 45 98 15 81 29
**
*******************************************************************/

#include <stream.h>
#include <String.h>
#include <std.h>
#include <stdio.h>
#include "Operations.hh"
#include "../config/hb_config.cc"  // hb_config.cc includes hb_config.hh
#include "../block1/Filehandler.hh"
#include "error_codes.cc"

#define DEBUG 0
#define LINKOFFSET 256
#define DATAOFFSET 1

enum { START,SIZE } table_last_entry; 
Filehandler NAME_OF_BLOCK1;

/*******************************************************************/
/*******************************************************************
** Private procedures.                                            **
** These procedures are used to access the different keys in the  **
** nodes.                                                         **
********************************************************************/
/*******************************************************************/


/*******************************************************************
** GetCharDataField.
** Private procedure used to extract a user defined from a datanode.
** The field is extracted by putting a pointer at the start of the
** field and the reading it out char by char.
** The next 3 procedures works in a very similar way but are doing
** slightly different things.
*******************************************************************/

char* Operations::GetCharDataField(char* ptr,int key)
{
  if (DEBUG)
    cerr << "GetCharDataField : " << key << "\n";
  
  if ((no_of_user_data_keys == 0) || (key < 1) 
      || (key > no_of_user_data_keys))
    return NULL;

  int size = user_data_keys[key - 1].length;
  char* Value = new(char[size+1]);
  char* Pointer = (char*) (dataoffset[key - 1] + (int) ptr);

  for (int i = 0; (i < size); i++)
    Value[i] = Pointer[i];
  return Value;
}

/*******************************************************************
** GetCharLinkField.
** Reads a user defined field in a linknode.
*******************************************************************/

char* Operations::GetCharLinkField(char* ptr,int key)
{
  if (DEBUG)
    cerr << "GetCharLinkField : " << key << "\n";

  key -= LINKOFFSET; // remove key offset
  if ((no_of_user_link_keys == 0) || (key < 0) 
      || (key + 1 > no_of_user_link_keys))
    return NULL;

  int size = user_link_keys[key].length;
  char* Value = new(char[size+1]);
  char* Pointer = (char*) (linkoffset[key] + (int) ptr);
  
  for (int i = 0; i < size; i++)
    Value[i] = Pointer[i];
  return Value;
}

/*******************************************************************
** PutCharDataField.
** Writes a user defined field in a datanode.
** The len field is suposed to have been checked already !!
*******************************************************************/

int Operations::PutCharDataField(char* ptr,int key,char* Value,long len)
{
  if (DEBUG)
    cerr << "PutCharDataField : " << key << ":" << Value << "\n";
  
  if ((no_of_user_data_keys == 0) || (key < 1) 
      || (key > no_of_user_data_keys))
    return KEY_NOT_RECOGNIZED;
  char* Pointer = (char*) (dataoffset[key - 1] + (int) ptr);

  for (int i = 0;i < len; i++)
    Pointer[i] = Value[i];
  return OK;
}

/*******************************************************************
** PutCharLinkField.
** Writes a user defined field in a linknode.
** The len field is suposed to have been checked already !!
*******************************************************************/

int Operations::PutCharLinkField(char* ptr,int key,char* Value,long len)
{
  if (DEBUG)
    cerr << "PutCharLinkField : " << key << ":" << Value << "\n";
  
  key -= LINKOFFSET; // remove key offset
  if ((no_of_user_link_keys == 0) || (key < 0) 
      || (key + 1 > no_of_user_link_keys))
    return KEY_NOT_RECOGNIZED;
  char* Pointer = (char*) (linkoffset[key] + (int) ptr);

  for (int i = 0; i < len; i++)
    Pointer[i] = Value[i];
  return OK;
}


/*******************************************************************/
/*******************************************************************
** Public procedures.                                             **
** These procedures are almost entirely written in the language   **
** given by the private procedures and the Filehandler block.     **
********************************************************************/
/*******************************************************************/

/*******************************************************************
** Browse.
** Primitive browser returning all links or nodes (numbers). Just
** hand the problem down.
*******************************************************************/

int Operations::Browse(int Type, long* Result[], long* Number)
{
  return NAME_OF_BLOCK1.Browse(Type, Result, Number);
}

/*******************************************************************
** Delete.
** This procedure deletes an entity in the hyperbase. 
** If the deleted entity is a linknode, the "links_to_me" in the 
** datanode is decremented.
** If the deleted entity is a datanode, all the pointed links from
** the datanode will have there usecount decremented.
*******************************************************************/

int Operations::Delete(long ent_no)
{
  if (DEBUG)
    cerr << "Delete : " << ent_no << "\n";
  
  char *ptr, *lkptr;
  long Pointed_datanode, OldLinksToMe, *longptr;
  linknode* node;
  datanode* dtnode;
  
  if (odd(ent_no) == LINKNODE)
  {    
    if (NAME_OF_BLOCK1.OpenE(ent_no,&ptr) != OK)
      return ENTITY_NOT_FOUND;
    node = (linknode*) ptr;
    if (node->usecount != 0)
      return POINTERS_TO_ENTITY;
    Pointed_datanode = node->to_datanode_no;
    if (NAME_OF_BLOCK1.OpenE(Pointed_datanode,&ptr) != OK)
      return POINTED_ENTITY_NOT_FOUND;
    dtnode = (datanode*) ptr;
    dtnode->links_to_me -= 1;
    if (NAME_OF_BLOCK1.CloseE(ptr) == OK)
      return NAME_OF_BLOCK1.DeleteE(ent_no);
    return WRONG;
  }
  else // datanode
  {
    if (NAME_OF_BLOCK1.OpenE(ent_no,&ptr) != OK)
      return ENTITY_NOT_FOUND;
    dtnode = (datanode*) ptr;
    if (dtnode->links_to_me != 0)
      return POINTERS_TO_ENTITY;
    if (NAME_OF_BLOCK1.OpenLink(&lkptr) != OK)
      return WRONG;
    int EntityNotFound = 0;
    for (longptr = (long*) lkptr;*longptr; *longptr++)
    {
      if (NAME_OF_BLOCK1.OpenE(*longptr,&ptr) == OK)
      {
	// We should now have the link in the address given by ptr
	node = (linknode*) ptr;
	node->usecount -= 1;
	if (NAME_OF_BLOCK1.CloseE(ptr) != OK)
	  return WRONG;
      }
      else
	EntityNotFound = 1;
    }
    if (EntityNotFound == 0)
      return NAME_OF_BLOCK1.DeleteE(ent_no);
    else
      if (NAME_OF_BLOCK1.DeleteE(ent_no) != OK)
	return WRONG;
      else
	return POINTED_ENTITY_NOT_FOUND;
  }
}

/*******************************************************************
** CreateNode.
** Creates a new datanode, and returns the entity number in ent_no.
*******************************************************************/

int Operations::CreateNode(long* ent_no)
{
  if (DEBUG)
    cerr << "CreateNode\n";

  char* ptr;
  long test;
  
  if (NAME_OF_BLOCK1.OpenNewE(DATANODE,&ptr) != OK)
    return UNABLE_TO_OPEN;
  datanode* node = (datanode*) ptr;
  *ent_no =  node->datanode_no;
  return NAME_OF_BLOCK1.CloseE(ptr); // do I need this ?
}

/*******************************************************************
** Link.
** Makes the different pointers in the hyperbase.
** If "ToEnt" is a datanode, there will be made a new linknode.
** This link will be made pointing to the datanode, and the 
** datanode's "links_to_me" will be incremented.
** On the contrary - if "ToEnt" is a linknode the first parameter
** must be a datanode. Then there is made a pointer from the 
** datanode to the link, and the links "usecount" is incremented.
*******************************************************************/

int Operations::Link(long* EntNo, long ToEnt)
{
  if (DEBUG)
    cerr << "Link : " << ToEnt << "\n";
  char *ptr, *links;
  long *newlinks, *movelinks;
  linknode* node;
  datanode* dtnode;
  
  if (odd(ToEnt) == DATANODE)
  {    
    // argument 2 a datanode and a link returned in argument 1
    // create new link with necessary key-values
    if (NAME_OF_BLOCK1.OpenNewE(LINKNODE,&ptr) != OK)
      return ENTITY_NOT_FOUND;
    node = (linknode*) ptr;
    node->to_datanode_no = ToEnt;
    if (NAME_OF_BLOCK1.CloseE(ptr) != OK)
      return WRONG;
    *EntNo = node->linknode_no;
    // update links_to_me in pointed datanode
    if (NAME_OF_BLOCK1.OpenE(ToEnt,&ptr) != OK)
      return POINTED_ENTITY_NOT_FOUND;
    dtnode = (datanode*) ptr;
    dtnode->links_to_me += 1;
    return NAME_OF_BLOCK1.CloseE(ptr);
  }
  else
  {
    // argument 1 a node and argument 2 a link
    if (odd(*EntNo) != DATANODE)
      return WRONG_ARGUMENTS;
    // update links in datanode
    if (NAME_OF_BLOCK1.OpenE(*EntNo,&ptr) != OK)
      return WRONG;
    if (NAME_OF_BLOCK1.OpenLink(&links) != OK)
      return UNABLE_TO_OPEN;
    // copies the linkpointers to a new location in memory
    char* hide = links;
    movelinks = (long*) links;
    for (int i = 1; *movelinks++ ; i++);
    newlinks = new(long[i+1]);
    movelinks = (long*) links;
    links = (char*) newlinks;
    for (i = 0; *movelinks;newlinks[i++] = *movelinks++);
    newlinks[i++] = ToEnt;
    newlinks[i] = 0;
    delete(hide);
    if (NAME_OF_BLOCK1.CloseLink(links) != OK)
      return UNABLE_TO_WRITE_LINKS;
    // update usecount in Link
    if (NAME_OF_BLOCK1.OpenE(ToEnt,&ptr) != OK)
      return WRONG;
    node = (linknode*) ptr;
    node->usecount += 1;
    return NAME_OF_BLOCK1.CloseE(ptr);
  }
}


/*******************************************************************
** MoveLink.
** Changes a Links "to_datanode_no" to NewNode.
*******************************************************************/

int Operations::MoveLink(long LinkNo, long NewNode)
{
  if (DEBUG)
    cerr << "MoveLink : " << LinkNo << ":" << NewNode << "\n";

  char *ptr,*dptr;
  
  if (NAME_OF_BLOCK1.OpenE(LinkNo,&ptr) != OK)
    return ENTITY_NOT_FOUND;
  linknode* node = (linknode*) ptr;
  if (NAME_OF_BLOCK1.OpenE(node->to_datanode_no,&dptr) == OK)
  {
    datanode* dnode = (datanode*) dptr;
    dnode->links_to_me -= 1;
    if (NAME_OF_BLOCK1.CloseE(dptr) != OK)
    {
      delete(ptr);
      delete(dptr);
    }
  }
  if (NAME_OF_BLOCK1.OpenE(NewNode,&dptr) == OK)
  {
    datanode* dnode = (datanode*) dptr;
    dnode->links_to_me += 1;
    if (NAME_OF_BLOCK1.CloseE(dptr) != OK)
    {
      delete(ptr);
      delete(dptr);
    }
  }
  delete(ptr); // I only need one version of the link
  if (NAME_OF_BLOCK1.OpenE(LinkNo,&ptr) != OK)
    return WRONG; // I have just read it once
  node->to_datanode_no = NewNode;
  return NAME_OF_BLOCK1.CloseE(ptr);
}

/*******************************************************************
** RemoveLink.
** Can only be used on a datanode. The pointer to the specified
** link is removed from the datanodes linklist, and the other
** pointers are pulled together.
*******************************************************************/

int Operations::RemoveLink(long ent_no,long link)
{
  char *lks, *node;
  long *movelinks;
  
  if (DEBUG)
    cerr << "RemoveLink : " << link << "\n";
  if (odd(ent_no) != DATANODE)
    return WRONG_ARGUMENTS;
  if (NAME_OF_BLOCK1.OpenE(ent_no, &node) != OK)
    return UNABLE_TO_OPEN;
  if (NAME_OF_BLOCK1.OpenLink(&lks) != OK)
    return UNABLE_TO_OPEN;
  for (movelinks = (long*) lks;
       *movelinks != link && *movelinks;movelinks++);
  if (*movelinks)
  {
    long* helplinks = movelinks;
    helplinks++;
    while (*movelinks++ = *helplinks++);
  }
  else
    return LINK_NOT_FOUND;
  if (NAME_OF_BLOCK1.CloseLink(lks) != OK)
    return UNABLE_TO_WRITE_LINKS;
  if (NAME_OF_BLOCK1.OpenE(link, &node) != OK)
    return UNABLE_TO_OPEN;
  linknode* temp = (linknode*) node;
  temp->usecount -= 1;
  return   NAME_OF_BLOCK1.CloseE(node);
}

/*******************************************************************
** Read.
** Reads the different keys in the nodes. As arguments can be used
** both linknodes and datanodes.
*******************************************************************/
int Operations::Read(long ent_no,int key,char** Value,long* len)
{
  if (DEBUG)
    cerr << "Read : " <<  ent_no << ":" << key  << "\n";
  char* ptr;
  
  if (NAME_OF_BLOCK1.OpenE(ent_no,&ptr) != OK)
    return UNABLE_TO_OPEN;
  
//  len = new(long);  magnus & uffe 

  if (key >= LINK_NO) // Standard linknode key
  {
    linknode* node = (linknode*) ptr;
    if (odd(ent_no) == DATANODE)
      return CANT_USE_LINK_KEYS_IN_DATANODE;
    long* Result = new(long);
    *Value = (char*) Result; 
    switch (key)
    {
    case LINK_NO:
      *Result = node->linknode_no;
      *len = sizeof(long); 
      break;
    case USECOUNT:
      *Result = node->usecount;
      *len = sizeof(long); 
      break;
    case TODATANODE_NO:
      *Result = node->to_datanode_no;
      *len = sizeof(long); 
      break;
        default:
      return KEY_NOT_RECOGNIZED;
    }
    return OK;
  }
  
  if (key > ALLDATA) // Standard datanode key
  {
    datanode* node = (datanode*) ptr;
    if (odd(ent_no) == LINKNODE)
      return CANT_USE_DATA_KEYS_IN_LINKNODE;
    long* Result = new(long);
    *Value = (char*) Result;
    switch (key)
    {
    case LINKNUM:
      char *help;
      int i;

      delete(Result); 
      if (NAME_OF_BLOCK1.OpenLink(&help) != OK)
	return UNABLE_TO_OPEN;
      long* links = (long*) help;
      for (i = 0; *links++ ; i++); // counts the number of links
      *len = i * sizeof(long);
      *Value = help;
      break;
    case DATA:
      delete(Result); 
      *len = node->size; 
      return NAME_OF_BLOCK1.OpenData(Value);
    case DATANODE_NO:
      *Result = node->datanode_no;
      *len = sizeof(long); 
      break;
    case DATASIZE:
      *Result = node->size;
      *len = sizeof(long); 
      break;
    case LINKSTOME:
      *Result = node->links_to_me;
      *len = sizeof(long); 
      break;
    default:
      return KEY_NOT_RECOGNIZED;
    }
    return OK;
  }
  
  if (key > 255) // User defined link key
  {
    if (odd(ent_no) == DATANODE)
      return CANT_USE_LINK_KEYS_IN_DATANODE;

    *len = user_link_keys[key - 256].length;
    *Value = GetCharLinkField(ptr,key);
    if (*Value == NULL) 
      return KEY_NOT_RECOGNIZED;
    else
      return OK;
  }
   
  // User defined data key
  if (odd(ent_no) == LINKNODE)
    return CANT_USE_DATA_KEYS_IN_LINKNODE;

  *len = user_data_keys[key - 1].length;
  *Value = GetCharDataField(ptr,key);
  if (*Value == NULL) 
    return KEY_NOT_RECOGNIZED;
  else
    return OK;
}


/*******************************************************************
** Write.
** Writes into the different entities in the hyperbase. As arguments
** can be used both link- and datanodes.
*******************************************************************/

int Operations::Write(long ent_no,int key,char* Value,long len)
{
  if (DEBUG)
    cerr << "Write : " <<  ent_no << ":" << key << "\n";
  
  char* ptr;
  
  if (NAME_OF_BLOCK1.OpenE(ent_no,&ptr) != OK)
    return UNABLE_TO_OPEN;
  
  if (key >= LINK_NO) // Standard LinkKey
  {
    return DIRECT_WRITE_NOT_ALLOWED; // user is not allowed to write 
    // This return is intended because we not will allow direct acces 
    // to the system keys.
    // If somebody wants to allow direct writes just remove the above
    // return

    linknode* node = (linknode*) ptr;
    if (odd(ent_no) == DATANODE)
      return CANT_USE_LINK_KEYS_IN_DATANODE;
    switch (key)
    {
    case LINK_NO:
      node->linknode_no = *(long*)Value;
      break;
    case USECOUNT:
      node->usecount = *(long*)Value;
      break;
    case TODATANODE_NO:
      node->to_datanode_no = *(long*)Value;
      break;
    default: 
      return KEY_NOT_RECOGNIZED;
    }
    return NAME_OF_BLOCK1.CloseE(ptr);
  }
  
  if (key > ALLDATA) // Standard datanode key
  {
    datanode* node = (datanode*) ptr;
    if (odd(ent_no) == LINKNODE)
      return CANT_USE_DATA_KEYS_IN_LINKNODE;
    switch (key)
    {
     case DATA:
      char* Data;
      if (NAME_OF_BLOCK1.OpenData(&Data) == OK)
	return NAME_OF_BLOCK1.CloseData(Value,len);
      break;
    default:
      return DIRECT_WRITE_NOT_ALLOWED;
    /* This default return is intended because we not will allow direct 
       access to the following keys */
/*    case DATANODE_NO:
      node->datanode_no = Value;
      break;
    case DATASIZE:
      node->size = Value;
      break;
    case LINKSTOME:
      node->links_to_me = Value;
      break;
    case LINKNUM:
      char* Link;
      if (NAME_OF_BLOCK1.OpenLink(&Link) == OK)
	return NAME_OF_BLOCK1.CloseLink(Value);
      break;	
    default:
      return KEY_NOT_RECOGNIZED; */
    }
    return NAME_OF_BLOCK1.CloseE(ptr);
  }
  
  if (key > 255) // User defined link key
  {
    if (odd(ent_no) == DATANODE)
      return CANT_USE_DATA_KEYS_IN_LINKNODE;
    if ((user_link_keys[key - 256].length < len) || (len == 0))
      return WRONG_LENGTH; 
    (void) PutCharLinkField(ptr,key,Value,len);
    return NAME_OF_BLOCK1.CloseE(ptr);
  }

  // User defined data key
  if (odd(ent_no) == LINKNODE)
    return  CANT_USE_DATA_KEYS_IN_LINKNODE;
  if ((user_data_keys[key - 1].length < len) || (len == 0))
    return WRONG_LENGTH; 
  (void) PutCharDataField(ptr,key,Value,len);
  return NAME_OF_BLOCK1.CloseE(ptr);
}

/*******************************************************************
** ERead.
** Reads in an entire entity, making it possible for the user to
** obtain all the data in the entity.
*******************************************************************/

int Operations::ERead(long ent_no, char** Value)
{
  if (DEBUG)
    cerr << "ERead : " << ent_no << "\n";
  
  char* help;
  
  if (NAME_OF_BLOCK1.OpenE(ent_no,&help) == OK)
  {
    *Value = help; 
    return OK;
  }
  else
    return WRONG;
}

/*******************************************************************
** EWrite.
** Writes all the user defined attributes into the specified
** entity in the hyperbase.
*******************************************************************/

int Operations::EWrite(long ent_no, char* Value)
{
  if (DEBUG)
    cerr << "EWrite : " << ent_no << "\n";

  char* ptr;
  
  if (NAME_OF_BLOCK1.OpenE(ent_no,&Value) != OK)
    return UNABLE_TO_OPEN;
  if (odd(ent_no) == DATANODE)
  {
    char* help;
    for (int i = DATAOFFSET; i < DATAOFFSET + no_of_user_data_keys; i++)
    {
      help = GetCharDataField(Value,i);
      if (Write(ent_no,i,help,user_data_keys[i].length ) != OK)
	return WRONG;
      delete (help);
    }
    return NAME_OF_BLOCK1.CloseE(ptr);
  }  
  else
  {
    char* help;
    for (int i = LINKOFFSET; i < LINKOFFSET + no_of_user_link_keys; i++)
    {
      help = GetCharLinkField(Value, i + LINKOFFSET);
      if (Write(ent_no, i + LINKOFFSET, help, user_data_keys[i].length) != OK)
	return WRONG;
      delete (help);
    }
    return NAME_OF_BLOCK1.CloseE(ptr);
  }  
}
