/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024-2025 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */
package tech.libeufin.nexus.iso20022

import tech.libeufin.common.*
import tech.libeufin.nexus.*
import tech.libeufin.nexus.ebics.Dialect
import java.io.InputStream
import java.time.Instant
import java.time.ZoneOffset
import java.util.UUID


sealed interface TxNotification {
    val executionTime: Instant
}

/** ID for incoming transactions */
data class IncomingId(
    /** ISO20022 UETR */
    val uetr: UUID? = null,
    /** ISO20022 TxID */
    val txId: String? = null,
    /** ISO20022 AcctSvcrRef */
    val acctSvcrRef: String? = null,
) {
    constructor(uetr: String, txId: String?, acctSvcrRef: String?) : this(UUID.fromString(uetr), txId, acctSvcrRef);

    fun ref(): String = uetr?.toString() ?: txId ?: acctSvcrRef!!

    override fun toString(): String = buildString {
        append('(')
        if (uetr != null) {
            append("uetr=")
            append(uetr.toString())
        }
        if (txId != null) {
            if (length != 1) append(" ")
            append("tx=")
            append(txId)
        }
        if (acctSvcrRef != null) {
            if (length != 1) append(" ")
            append("ref=")
            append(acctSvcrRef)
        }
        append(')')
    }
}

sealed interface OutId {}

/** ID for outgoing transactions */
data class OutgoingId(
    /** 
     * Unique msg ID generated by libeufin-nexus
     * ISO20022 MessageId
     **/
    val msgId: String? = null,
    /** 
     * Unique end-to-end ID generated by libeufin-nexus
     * ISO20022 EndToEndId or MessageId (retrocompatibility)
     **/
    val endToEndId: String? = null,
    /** 
     * Unique end-to-end ID generated by the bank
     * ISO20022 AcctSvcrRef
     **/
    val acctSvcrRef: String? = null,
): OutId {
    fun ref(): String = endToEndId ?: acctSvcrRef ?: msgId!!
    override fun toString(): String = buildString {
        append('(')
        if (msgId != null && msgId != endToEndId) {
            append("msg=")
            append(msgId.toString())
        }
        if (endToEndId != null) {
            if (length != 1) append(" ")
            append("e2e=")
            append(endToEndId)
        }
        if (acctSvcrRef != null) {
            if (length != 1) append(" ")
            append("ref=")
            append(acctSvcrRef)
        }
        append(')')
    }
}

/** ID for outgoing batches */
data class BatchId(
    /** 
     * Unique msg ID generated by libeufin-nexus
     * ISO20022 MessageId
     **/
    val msgId: String,
    /** 
     * Unique end-to-end ID generated by the bank
     * ISO20022 AcctSvcrRef
     **/
    val acctSvcrRef: String? = null,
): OutId {
    fun ref(): String = msgId
    override fun toString(): String = buildString {
        append('(')
        if (msgId != null) {
            append("msg=")
            append(msgId.toString())
        }
        if (acctSvcrRef != null) {
            if (length != 1) append(" ")
            append("ref=")
            append(acctSvcrRef)
        }
        append(')')
    }
}


/** ISO20022 incoming payment */
data class IncomingPayment(
    val id: IncomingId,
    val amount: TalerAmount,
    val creditFee: TalerAmount? = null,
    val subject: String?,
    override val executionTime: Instant,
    val debtor: IbanPayto?
): TxNotification {
    override fun toString(): String = buildString {
        append("IN ")
        append(executionTime.fmtDate())
        append(" ")
        append(amount)
        if (creditFee != null) {
            append("-")
            append(creditFee)
        }
        append(" ")
        append(id)
        if (debtor != null) {
            append(" debtor=")
            append(debtor)
        }
        if (subject != null) {
            append(" subject='")
            append(subject)
            append("'")
        }
    }
}

/** ISO20022 outgoing payment */
data class OutgoingPayment(
    val id: OutgoingId,
    val amount: TalerAmount,
    val subject: String?,
    override val executionTime: Instant,
    val creditor: IbanPayto?
): TxNotification {
    override fun toString(): String = buildString {
        append("OUT ")
        append(executionTime.fmtDate())
        append(" ")
        append(amount)
        append(" ")
        append(id)
        if (creditor != null) {
            append(" creditor=")
            append(creditor)
        }
        if (subject != null) {
            append(" subject='")
            append(subject)
            append("'")
        }
    }
}

/** ISO20022 outgoing batch */
data class OutgoingBatch(
    /** ISO20022 MessageId */
    val msgId: String,
    override val executionTime: Instant,
): TxNotification {
    override fun toString(): String {
        return "BATCH ${executionTime.fmtDate()} $msgId"
    }
}

/** ISO20022 outgoing reversal */
data class OutgoingReversal(
    /** ISO20022 EndToEndId */
    val endToEndId: String,
    /** ISO20022 MessageId */
    val msgId: String? = null,
    val reason: String?,
    override val executionTime: Instant
): TxNotification {
    override fun toString(): String {
        val msgIdFmt = if (msgId == null) "" else "$msgId."
        return "BOUNCE ${executionTime.fmtDate()} $msgIdFmt$endToEndId: $reason"
    }
}

private class IncompleteTx(val msg: String): Exception(msg)

private enum class Kind {
    CRDT,
    DBIT
}

/** Parse a payto */
private fun XmlDestructor.payto(prefix: String): IbanPayto? {
    return opt("RltdPties") { 
        val iban = opt("${prefix}Acct")?.one("Id")?.opt("IBAN")?.text()
        if (iban != null) {
            val name = opt(prefix) { opt("Nm")?.text() ?: opt("Pty")?.one("Nm")?.text() }
            // TODO more performant option
            ibanPayto(iban, name)
        } else {
            null
        }
    }
}

/** Check if an entry status is BOOK */
private fun XmlDestructor.isBooked(): Boolean {
    // We check at the Sts or Sts/Cd level for retrocompatibility
    return one("Sts") {
        val status = opt("Cd")?.text() ?: text()
        status == "BOOK"
    }
}

/** Parse the instruction execution date */
private fun XmlDestructor.executionDate(): Instant {
    // Value date if present else booking date
    val date = opt("ValDt") ?: one("BookgDt")
    val parsed = date.opt("Dt") {
        date().atStartOfDay()
    } ?: date.one("DtTm") {
        dateTime()
    }
    return parsed.toInstant(ZoneOffset.UTC)
}

/** Parse batch message ID and transaction end-to-end ID as generated by libeufin-nexus */
private fun XmlDestructor.outgoingId(ref: String?): OutId = 
    opt("Refs") {
        val endToEndId = opt("EndToEndId")?.text()
        val msgId = opt("MsgId")?.text()
        val ref = if (ref != "NOTPROVIDED") ref else null
        if (msgId != null && endToEndId == null) {
            // This is a batch representation
            BatchId(msgId, ref)
        } else if (endToEndId == "NOTPROVIDED") {
            // If not set use MsgId as end-to-end ID for retrocompatibility
            OutgoingId(msgId, msgId, ref)
        } else {
            OutgoingId(msgId, endToEndId, ref)
        }
    } ?: OutgoingId(null, null, ref)

/** Parse transaction ids as provided by bank*/
private fun XmlDestructor.incomingId(ref: String?): IncomingId =
    opt("Refs") {
        val uetr = opt("UETR")?.uuid()
        val txId = opt("TxId")?.text()
        IncomingId(uetr, txId, ref)
    } ?: IncomingId(null, null, ref)


/** Parse and format transaction return reasons */
private fun XmlDestructor.returnReason(): String = opt("RtrInf") {
    val code = one("Rsn").one("Cd").enum<ExternalReturnReasonCode>()
    val info = map("AddtlInf") { text() }.joinToString("")
    buildString {
        append("${code.isoCode} '${code.description}'")
        if (info.isNotEmpty()) {
            append(" - '$info'")
        }
    }
} ?: opt("RmtInf") {
    map("Ustrd") { text() }.joinToString("")
} ?: ""

/** Parse amount */
private fun XmlDestructor.amount() = one("Amt") {
    val currency = attr("Ccy")
    val amount = text()
    val concat = if (amount.startsWith('.')) {
        "$currency:0$amount"
    } else {
        "$currency:$amount"
    }
    TalerAmount(concat)
}

sealed interface ComplexAmount {
    data class Simple(
        val amount: TalerAmount
    ): ComplexAmount
    data class Converted(
        val sent: TalerAmount,
        val received: TalerAmount
    ): ComplexAmount
    data class Charged(
        val received: TalerAmount,
        val creditFee: TalerAmount
    ): ComplexAmount

    /// The amount to register in database
    fun amount(): TalerAmount {
        return when (this) {
            is Simple -> amount
            is Converted -> received
            is Charged -> received
        }
    }

    /// The credit fee to register in database
    fun creditFee(): TalerAmount? {
        return when (this) {
            is Simple, is Converted -> null
            is Charged -> creditFee
        }
    }

    /// Check that entry and tx amount are compatible and return the result
    fun resolve(other: ComplexAmount): ComplexAmount {
        when (this) {
            is Simple -> {
                when (other) {
                    is Simple -> {}
                    is Converted -> require(other.sent == amount || other.received == amount) { "bad currency conversion $other != $this" }
                    is Charged -> require(other.received == amount + other.creditFee) { "bad tx charge $other != $amount" }
                }
                return other
            }
            is Converted -> {
                require(other is Simple)
                require(other.amount == sent)
                return this
            }
            is Charged -> {
                require(this == other) { "$this != $other" }
                return this
            }
        }
    }
} 

private fun XmlDestructor.complexAmount(): ComplexAmount? {
    var overflow = false;
    val received = opt("Amt") {
        val currency = attr("Ccy")
        var amount = text()
        overflow = amount.startsWith('-')
        amount = amount.trimStart('-')
        val concat = if (amount.startsWith('.')) {
            "$currency:0$amount"
        } else {
            "$currency:$amount"
        }
        TalerAmount(concat)
    }
    if (received == null) return null
    
    val sent = opt("AmtDtls")?.opt("TxAmt") {
        amount()
    } ?: received

    var creditFee: TalerAmount? = null
    opt("Chrgs")?.each("Rcrd") {
        if (one("ChrgInclInd").bool() && one("CdtDbtInd").text() == "DBIT" ) {
            val amount = amount()
            creditFee = creditFee?.let { it + amount } ?: amount
        }
    }

    if (received.currency != sent.currency) {
        require(creditFee == null) { "Do not support fee on currency conversion" }
        require(!overflow)
        return ComplexAmount.Converted(sent, received)
    } else if (creditFee == null && received == sent && !overflow) {
        return ComplexAmount.Simple(received)
    } else {
        if (received != sent || overflow) {
            val diff = if (overflow) {
                sent + received
            } else {
                sent - received
            }
            require(creditFee == null || creditFee == diff)
            return ComplexAmount.Charged(if (overflow) received else sent, diff)
        } else {
            val diff = requireNotNull(creditFee)
            return ComplexAmount.Charged(received, diff)
        }
    }
}

/** Parse bank transaction code */
private fun XmlDestructor.bankTransactionCode(): BankTransactionCode {
    return one("BkTxCd").one("Domn") {
        val domain = one("Cd").enum<ExternalBankTransactionDomainCode>()
        one("Fmly") {
            val family = one("Cd").enum<ExternalBankTransactionFamilyCode>()
            val subFamily = one("SubFmlyCd").enum<ExternalBankTransactionSubFamilyCode>()
            
            BankTransactionCode(domain, family, subFamily)
        }
    }
}

/** Parse optional bank transaction code */
private fun XmlDestructor.optBankTransactionCode(): BankTransactionCode? {
    return opt("BkTxCd")?.one("Domn") {
        val domain = one("Cd").enum<ExternalBankTransactionDomainCode>()
        one("Fmly") {
            val family = one("Cd").enum<ExternalBankTransactionFamilyCode>()
            val subFamily = one("SubFmlyCd").enum<ExternalBankTransactionSubFamilyCode>()
            
            BankTransactionCode(domain, family, subFamily)
        }
    }
}

/** Parse transaction wire transfer subject */
private fun XmlDestructor.wireTransferSubject(): String? = opt("RmtInf") {
    map("Ustrd") { text() }?.joinToString("")?.trim()
}

/** Parse account information */
private fun XmlDestructor.account(): Pair<String, String?> = one("Acct") {
    Pair(
        one("Id") {
            (opt("IBAN") ?: one("Othr").one("Id")).text()
        },
        opt("Ccy")?.text()
    )
}

data class AccountTransactions(
    val iban: String?,
    val currency: String?,
    val txs: List<TxNotification>
) {
    companion object {
        internal fun fromParts(iban: String?, currency: String?, txsInfos: List<TxInfo>): AccountTransactions {
            val txs = txsInfos.mapNotNull {
                try {
                    it.parse()
                } catch (e: IncompleteTx) {
                    // TODO: add more info in doc or in log message?
                    logger.warn("skip incomplete tx: ${e.msg}")
                    null
                }    
            }
            return AccountTransactions(iban, currency, txs)
        }
    }
}

/** Parse camt.054 or camt.053 file */
fun parseTx(notifXml: InputStream, dialect: Dialect): List<AccountTransactions> {
    /*
        In ISO 20022 specifications, most fields are optional and the same information 
        can be written several times in different places. For libeufin, we're only 
        interested in a subset of the available values that can be found in both camt.052,
        camt.053 and camt.054. This function should not fail on legitimate files and should 
        simply warn when available information are insufficient.

        EBICS and ISO20022 do not provide a perfect transaction identifier. The best is the 
        UETR (unique end-to-end transaction reference), which is a universally unique 
        identifier (UUID). However, it is not supplied by all banks. TxId (TransactionIdentification) 
        is a unique identification as assigned by the first instructing agent. As its format 
        is ambiguous, its uniqueness is not guaranteed by the standard, and it is only 
        supposed to be unique for a “pre-agreed period”, whatever that means. These two 
        identifiers are optional in the standard, but have the advantage of being unique 
        and can be used to track a transaction between banks so we use them when available.

        It is also possible to use AccountServicerReference, which is a unique reference 
        assigned by the account servicing institution. They can be present at several levels
        (batch level, transaction level, etc.) and are often optional. They also have the 
        disadvantage of being known only by the account servicing institution. They should 
        therefore only be used as a last resort.
    */
    logger.trace("Parse transactions camt file for $dialect")
    val accountTxs = mutableListOf<AccountTransactions>()

    /** Common parsing logic for camt.052, camt.053 and camt.054 */
    fun XmlDestructor.parseInner() {
        val (iban, currency) = account()
        val txInfos = mutableListOf<TxInfo>()
        val batches = each("Ntry") {
            if (!isBooked()) return@each
            val entryCode = bankTransactionCode()
            val reversal = opt("RvslInd")?.text() == "true"
            val entryKind = opt("CdtDbtInd")?.enum<Kind>();
            val tmp = opt("NtryDtls")?.map("TxDtls") { this } ?: return@each 
            val unique = tmp.size == 1
            val entryRef = opt("AcctSvcrRef")?.text()
            val bookDate = executionDate()
            val entryAmount = complexAmount()!!
            var totalAmount: TalerAmount? = null
            for (it in tmp) {it.run {
                // Transaction direction
                val txKind = opt("CdtDbtInd")?.enum<Kind>()
                val kind: Kind = requireNotNull(entryKind ?: txKind) { "Missing entry kind" }
                
                // Transaction code
                val code = optBankTransactionCode() ?: entryCode

                // Amount
                val txAmount = complexAmount()
                val amount = if (unique) {
                    if (entryAmount != null && txAmount != null) {
                        entryAmount.resolve(txAmount)
                    } else {
                        requireNotNull(entryAmount ?: txAmount) { "Missing unique tx amount" }
                    }
                } else {
                    requireNotNull(txAmount) { "Missing batch tx amount" }
                }
                if (entryAmount != null) {
                    totalAmount = totalAmount?.let { it + amount.amount() } ?: amount.amount()
                }

                // Ref
                val txRef = opt("Refs")?.opt("AcctSvcrRef")?.text()
                val ref = if (txRef == null) {
                    // We can only use the entry ref as the transaction ref if there is a single transaction in the batch
                    if (entryRef != null && unique) {
                        entryRef
                    } else {
                        null
                    }
                } else {
                    txRef
                }
                
                if (code.isReversal() || reversal) {
                    val outgoingId = outgoingId(ref)
                    when (kind) {
                        Kind.CRDT -> {
                            val reason = returnReason()
                            txInfos.add(TxInfo.CreditReversal(
                                bookDate = bookDate,
                                id = outgoingId,
                                reason = reason,
                                code = code
                            ))
                        }
                        Kind.DBIT -> {
                            val id = incomingId(ref)
                            val subject = wireTransferSubject()
                            val debtor = payto("Dbtr")
                            val creditFee = amount.creditFee()
                            requireNotNull(creditFee) { "Do not support failed debit without credit fee" }
                            require(creditFee > amount.amount())
                            txInfos.add(TxInfo.Credit(
                                bookDate = bookDate,
                                id = id,
                                amount = amount.amount(),
                                subject = subject,
                                debtor = debtor,
                                code = code,
                                creditFee = creditFee
                            ))
                        }
                    }
                } else {
                    val subject = wireTransferSubject()
                    when (kind) {
                        Kind.CRDT -> {
                            val id = incomingId(ref)
                            val debtor = payto("Dbtr")
                            txInfos.add(TxInfo.Credit(
                                bookDate = bookDate,
                                id = id,
                                amount = amount.amount(),
                                subject = subject,
                                debtor = debtor,
                                code = code,
                                creditFee = amount.creditFee()
                            ))
                        }
                        Kind.DBIT -> {
                            val outgoingId = outgoingId(ref)
                            val creditor = payto("Cdtr")
                            require(amount.creditFee() == null) { "Do not support debit with credit fees" }
                            txInfos.add(TxInfo.Debit(
                                bookDate = bookDate,
                                id = outgoingId,
                                amount = amount.amount(),
                                subject = subject,
                                creditor = creditor,
                                code = code
                            ))
                        }
                    }
                }
            }}
            if (totalAmount != null) {
                //require(totalAmount == entryAmount.amount()) { "Entry amount doesn't match batch amount sum $entryAmount != $totalAmount" }
            }
        }
        accountTxs.add(AccountTransactions.fromParts(iban, currency, txInfos))
    }
    XmlDestructor.fromStream(notifXml, "Document") {
        // Camt.053
        opt("BkToCstmrStmt")?.each("Stmt") { parseInner() }
        // Camt.052
        opt("BkToCstmrAcctRpt")?.each("Rpt") { parseInner() }
        // Camt.054
        opt("BkToCstmrDbtCdtNtfctn")?.each("Ntfctn") { parseInner() }
    }
    return accountTxs
}

sealed interface TxInfo {
    data class CreditReversal(
        val bookDate: Instant,
        val code: BankTransactionCode,
        val id: OutId,
        val reason: String
    ): TxInfo
    data class Credit(
        val bookDate: Instant,
        val code: BankTransactionCode,
        val id: IncomingId,
        val amount: TalerAmount,
        val creditFee: TalerAmount?,
        val subject: String?,
        val debtor: IbanPayto?
    ): TxInfo
    data class Debit(
        val bookDate: Instant,
        val code: BankTransactionCode,
        val id: OutId,
        val amount: TalerAmount,
        val subject: String?,
        val creditor: IbanPayto?
    ): TxInfo

    fun parse(): TxNotification {
        return when (this) {
            is TxInfo.CreditReversal -> {
                if (id !is OutgoingId || id.endToEndId == null) 
                    throw IncompleteTx("missing unique ID for Credit reversal $id")
                OutgoingReversal(
                    endToEndId = id.endToEndId!!,
                    msgId = id.msgId,
                    reason = reason,
                    executionTime = bookDate
                )
            }
            is TxInfo.Credit -> {
                if (id.uetr == null && id.txId == null && id.acctSvcrRef == null)
                    throw IncompleteTx("missing unique ID for Credit $id")
                IncomingPayment(
                    amount = amount,
                    creditFee = creditFee,
                    id = id,
                    debtor = debtor,
                    executionTime = bookDate,
                    subject = subject,
                )
            }
            is TxInfo.Debit -> {
                when (id) {
                    is OutgoingId -> {
                        if (id.endToEndId == null && id.msgId == null && id.acctSvcrRef == null) {
                            throw IncompleteTx("missing unique ID for Debit $id")
                        } else {
                            OutgoingPayment(
                                id = OutgoingId(
                                    endToEndId = id.endToEndId,
                                    acctSvcrRef = id.acctSvcrRef,
                                    msgId = id.msgId,
                                ),
                                amount = amount,
                                executionTime = bookDate,
                                creditor = creditor,
                                subject = subject
                            )
                        }
                    }
                    is BatchId -> {
                        OutgoingBatch(
                            msgId = id.msgId,
                            executionTime = bookDate,
                        )
                    }
                }
            }
        }
    }
}

data class BankTransactionCode(
    val domain: ExternalBankTransactionDomainCode,
    val family: ExternalBankTransactionFamilyCode,
    val subFamily: ExternalBankTransactionSubFamilyCode
) {
    fun isReversal(): Boolean = REVERSAL_CODE.contains(subFamily)
    fun isPayment(): Boolean = domain == ExternalBankTransactionDomainCode.PMNT || subFamily == ExternalBankTransactionSubFamilyCode.PSTE

    override fun toString(): String = 
        "${domain.name} ${family.name} ${subFamily.name} - '${domain.description}' '${family.description}' '${subFamily.description}'"

    companion object {
        private val REVERSAL_CODE = setOf(
            ExternalBankTransactionSubFamilyCode.RPCR,
            ExternalBankTransactionSubFamilyCode.RRTN,
            ExternalBankTransactionSubFamilyCode.PSTE,
        )
    }
}