/*
%Z%  FILE: %M%  RELEASE: %I%  DATE: %G%, %U%
*/
/**************************************************************************

Process:

    nicl

    Network Interface Command Language Processor.


Author:    Alex Measday, ISI


Purpose:

    Program NICL ...


    Invocation:

        % nicl [-debug] [-tty] [-vperror]
               [<script_file>] [<arguments> ...]

    where

        "-debug"
            enables debug output (written to STDOUT).
        "-tty"
            enables the monitoring of standard input.  Tcl commands entered
            on standard input are read and executed.  By default, NICL does
            NOT monitor standard input.
        "-vperror"
            turns VPERROR() message output on.  VPERROR() messages are
            low-level error messages generated by TPOCC library functions;
            normally, they are disabled.  If enabled, the messages are
            output to STDERR.
        "<script_file>"
            is the name of an initial Tcl script file that NICL reads and
            executes.  If no file is specified, NICL reads from standard
            input.
        "<arguments> ..."
            are passed to the script file through a global Tcl variable.
            All the command line arguments following the script file name
            are concatenated in a Tcl list and assigned to global Tcl
            variable, ARGV (all capital letters).  The UNIX command
            line-style options in the list can then be parsed using
            the NICL command, "ncl_opt".

**************************************************************************/

#ifdef sccs
    static char sccsid[] = "File: %M%  Release: %I%  Date: %G%, %U%" ;
#endif

#include  <errno.h>			/* System error definitions. */
#include  <stdio.h>			/* Standard I/O definitions. */
#include  <stdlib.h>			/* Standard C library definitions. */
#include  "tcl.h"			/* Tcl definitions. */
#include  "fparse.h"			/* Definition of pathname component types. */
#include  "libncl.h"			/* NICL library definitions. */
#include  "libutilgen.h"		/* LIBUTILGEN definitions. */
#ifdef vms
#    include  "libutilvms.h"		/* LIBUTILVMS definitions. */
#endif
#include  "nix_util.h"			/* Network I/O Handler definitions. */
#include  "opt_util.h"			/* Option scanning definitions. */
#define  MAIN
#include  "nicl.h"			/* NICL definitions. */
#undef  MAIN

extern  int  vperror_print ;		/* Global print switch (1/0 = yes/no). */


main (

#    if __STDC__ || defined(vaxc)
        int  argc,
        char  *argv[],
        char  *envp[])
#    else
        argc, argv, envp)

        int  argc ;
        char  *argv[] ;
        char  *envp[] ;
#    endif

{  /* Local variables. */
    char  *argument, buffer[16], *script_file ;
    int  errflg, i, option, use_tty, vperror_save ;





    vperror_save = vperror_print ;	/* Save state of VPERROR_PRINT flag. */
    vperror_print = 1 ;			/* Enable VPERROR output during initialization. */


/*******************************************************************************
  Scan the command line options.
*******************************************************************************/

    debug = 0 ;  script_file = NULL ;  use_tty = 0 ;
    opt_init (argc, argv, 0, "{debug}{tty}{vperror}", NULL) ;
    errflg = 0 ;

    while (option = opt_get (NULL, &argument)) {

        switch (option) {
        case 1:			/* "-debug" */
            debug = 1 ;  libncl_debug = 1 ;  nix_util_debug = 1 ;
            break ;
        case 2:			/* "-tty" */
            use_tty = 1 ;
            break ;
        case 3:			/* "-vperror" */
            vperror_save = 1 ;
            break ;
        case NONOPT:		/* "<script_file>" */
            script_file = argument ;
            break ;
        case OPTERR:
            errflg++ ;  break ;
        default :  break ;
        }
				/* Pass the remaining command line arguments
				   to the script file. */
        if (script_file != NULL)  break ;

    }

    if (errflg) {
        fprintf (stderr,
                 "Usage:  nicl [-debug] [-tty] [-vperror] [<script_file>] [<arguments> ...]\n") ;
        exit (EINVAL) ;
    }

/*******************************************************************************
    Initialize the Tcl command interpreter and store the remaining command
    line arguments in a global Tcl variable, where they can be referenced
    by the script file.
*******************************************************************************/

    ncl_tcl_interpreter = Tcl_CreateInterp () ;

    if (ncl_init (ncl_tcl_interpreter, NULL)) {
        vperror ("[%s] Error initializing the Tcl command interpreter.\nncl_init: ",
                 argv[0]) ;
        exit (errno) ;
    }

    i = opt_index (NULL) ;
    if ((argc - 1) > i)
        argument = Tcl_Merge (argc - 1 - i, &argv[i+1]) ;
    else
        argument = str_dupl ("", -1) ;
    sprintf (buffer, "%d", argc -1 - i) ;
    Tcl_SetVar (ncl_tcl_interpreter, "ARGC", buffer, TCL_GLOBAL_ONLY) ;
    Tcl_SetVar (ncl_tcl_interpreter, "ARGV", argument, TCL_GLOBAL_ONLY) ;
    if (debug)  printf ("(nicl) ARGC = %s, ARGV = \"%s\"\n", buffer, argument) ;
    free (argument) ;


/*******************************************************************************
    Read and execute the script file.
*******************************************************************************/

    if (script_file != NULL) {
        script_file = fparse (script_file, ".ncl", NULL, NULL) ;
        if (Tcl_EvalFile (ncl_tcl_interpreter, script_file) != TCL_OK) {
            vperror ("[%s] Error executing script file, %s: %s\nTcl_EvalFile: ",
                     argv[0], script_file, ncl_tcl_interpreter->result) ;
            exit (errno) ;
        }
    }


/*******************************************************************************
    If standard input is to be monitored for user-entered Tcl commands, then
    register standard input as an I/O source with the NIX I/O handler. (Under
    VMS, register a timeout timer so that standard input is periodically
    polled.)
*******************************************************************************/

    if (use_tty) {
#ifdef vms
        int  channel ;

        if (tty_create (NULL, &channel)) {
            vperror ("[NICL] Error assigning a channel to standard input.\n") ;
            exit (errno) ;
        }
        NxAddTimeOut (NULL, TTY_POLLING_INTERVAL,
                      ncl_tty_input, (void *) channel) ;
#else
        NxAddInput (NULL, fileno (stdin), NxInputReadMask,
                    ncl_tty_input, NULL) ;
#endif
    }


/*******************************************************************************
    Loop forever, processing input events as they occur.
*******************************************************************************/

    vperror_print = vperror_save ;

#ifdef vms
    NxMainLoopEF (NULL, 0) ;
#else
    NxMainLoop (NULL) ;
#endif


}
