/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

/*
 * ot.c  -- Main for Open Track bugs database  (Enter/Update/Validate/...)
 *
 */

#define MAINPROGRAM

#include <string.h>
#include <ctype.h>
#include <stdio.h>
/*
#include <stdlib.h>
*/
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <pwd.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

#ifndef LINT
static char RCSid_ot[] =
    "$RCSfile: ot.c,v $ $Revision: 1.1.8.1 $ $Date: 1994/01/19 17:56:02 $";
#endif


/*
 *			m a i n
 *
 */
int
main(argc, argv, envp)
int argc;
char ** argv;
char ** envp;
{
    char *cp1;
    OTProject proj;
    int exitStatus;
    OTErr tmpCode = OT_SUCCESS;
    OTErr errCode = OT_SUCCESS;

    /*
     * Initialize TCL, control blocks, read environment variables and
     * preparse command line for project and template file names 
     */

    if ( errCode = otInitialInit(&argc, argv) ) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "%s\n", cp1);
	exit(1);
    }

    if (!otCB->cb_project) {
	otPutPostedMessage(OT_NEED_PROJECT);
	cp1 = otGetPostedMessage();
	fprintf(stderr, "%s\n", cp1);
	exit(1);
    }

    if ( errCode = otReadProject(otCB->cb_project) ) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "%s\n", cp1);
	exit(1);
    }

    /* Indicate command line interface */
    otCB->cb_pcb->pcb_ui = CLI;

    /* parse command line args and set control block values */

    if (errCode = otParseArgsCLI(argc,argv)) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "%s\n", cp1);
	exit(1);
    }


    switch ( otCB->cb_operation) {
	case ENTER:
	    errCode = otDoEnter();
	    otCatchSignals(2);
	    otCleanAfterEnter(errCode);
	    break;

	case UPDATE:
	    errCode = otDoUpdate();
            otCatchSignals(2);
	    if ( errCode ) {
		(void)otRemoteTcl("unlock");
	    }
	    tmpCode = otCleanAfterUpdate(errCode);
	    if (!errCode && tmpCode)
		errCode = tmpCode;
	    break;

        case VALIDATE:
	    errCode = otDoValidate();
	    otCleanAfterValidate();
	    break;

	case DELETE:
	    errCode = otDoDelete();
	    otCleanAfterDelete(errCode);
	    break;

	case TEMPLATE:
	    errCode = otDoTemplate();
	    otCleanAfterTemplate();
	    break;

	case INITIAL:
	    otPutPostedMessage( OT_NO_OPERATION);
	    errCode = OT_NO_OPERATION;
	    break;

	default:
	    otPutPostedMessage( OT_INTERNAL_ERROR, 
				"main (illegal otCB->cb_operation)" );
	    errCode = OT_INTERNAL_ERROR;
	    break;
    }
    
    if (errCode) {
	if ( cp1 = otGetPostedMessage() )
	    fprintf(stderr, "%s\n", cp1);
    }

    /* Free up memory used by the control block structures and their members */

    exitStatus = errCode ? 1 : 0;

    if ( errCode = otBringDownServer() ) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "%s\n", cp1 ? cp1 : "error in server" );
    }

    otFreeControlBlock(); 

    exit(exitStatus);
}



OTErr
otDoTemplate()
{

    OTErr errCode = OT_SUCCESS;

    if (errCode = otInitTemplate()) 
	return errCode;

    if (otCB->cb_templFile)
        errCode = otWriteTemplateToFilename(otCB->cb_ecb->ecb_blankStruct,
		otCB->cb_templFile);
    else 
	errCode = otWriteTemplateToFilename(otCB->cb_ecb->ecb_blankStruct,
		"Template");

    return errCode;
}



OTErr
otDoEnter()
{
    char tmpfile[PATHLEN];
    OTTemplate * tp;
    OTProject *pjp;
    char *subject_line;
    char command[LONGVALUE];
    char name[NAMELEN];
    OTFilen fname;
    OTPrivateCB *pcp;
    Tcl_Interp *interp;
    char *cp;

    OTErr errCode = OT_SUCCESS;

/* Initialize OTEnterCB and blank template structure. Process
   cb_append and cb_templFile. */

    if (errCode = otInitEnter(tmpfile)) 
	return errCode;

    /*
     * TODO: otInitEnter() copies the tmpfile name to rmTfile sometimes, but
     * not always.  Let's review this later.
     */ 
    strcpy(rmTfile, tmpfile);
    pcp = otCB->cb_pcb;
    interp = pcp->pcb_interp;
    pjp = pcp->pcb_project;

/* If neither template file nor tcl string are passed, invoke an editor. 
   Otherwise copy passed template file to a temporary file */

    if (!otCB->cb_templFile && !otCB->cb_tclString) {
	/* ot -e	*/
        /* Write template structure to a file and invoke an editor */
        if (errCode = otEditForCLI(tmpfile))
            return errCode;
    } 

/* Loop in validate/edit cycle until user has corrected errors for an
   interractive interface; one validation cycle otherwise */

    if (errCode = otValidateEditCLI(tmpfile))
	return errCode;

    /* We're going to ignore most all signals from now on...  */
    otCatchSignals(1);

    if (!otConformEdit())
	return errCode;

    if (otCB->cb_ecb->ecb_tStruct)
	otFreeTemplate(otCB->cb_ecb->ecb_tStruct);

    if ( errCode = otReadTemplateFromFilename( &(otCB->cb_ecb->ecb_tStruct),
	tmpfile, TRUE, TRUE ) )
	return errCode;

    tp = otCB->cb_ecb->ecb_tStruct;
    /*
     * There was a validation error on the server - a rare occurance, since
     * normally validation is done successfully on the client.  This may mean
     * that the server and the client are out of sync.
     */
    if ( errCode = otRequestEnter() ) {
	return errCode;
    }

    sprintf(name, "new %s", pjp->object_name);

    strcpy(fname.dir, tmpfile);
    cp = strrchr(fname.dir, '/');
    cp++;
    *cp = 0;

    cp = strrchr(tmpfile, '/');
    cp++;
    strcpy(fname.name, cp);

    /*
     * Write again, so CR number is recorded in file.
     */
    if ( errCode = otWriteTemplateToFilename( tp, tmpfile ) )
	return errCode;

    subject_line = otGetSubjectLine(NULL, tp, fname, pcp->pcb_operation, pjp);

    otNotify(pjp, tp, fname, fname, otCB->cb_operation,	pcp->pcb_notify,
	name, subject_line);

    /*
     * TODO Let's review this - should rmTfile be removed on interrupt? then
     * rm rmTfile in signal handler (it doesn't seem to do it now).
     */
    otCleanup();

    if ( otCB->cb_pcb->pcb_ui == CLI ) {
	fprintf(stderr, "Created %s number %ld\n", pjp->object_name,
	    otGetIDNumValue(tp));
    }

    return errCode;

}


OTErr
otDoUpdate()
{
    OTTemplate * tp;
    long lnum;
    char tmpfile[PATHLEN];
    char command[LONGVALUE];
    OTErr errCode = OT_SUCCESS;
    char name[LONGVALUE];
    OTFilen fname;
    OTProject *pjp;
    char *cp, *subject_line;
    OTPrivateCB *pcp;

    /* Initialize Enter control block, check input for conflicts, 
     * get an Object for an update
     */
    pcp = otCB->cb_pcb;
    pjp = pcp->pcb_project;

    /*
     * Client-side, we create a filename to hold 1) the edited version of the
     * template, OR 2) a copy of the user-specified template.
     */
    tmpfile[0] = 0;
    tmpnam(tmpfile);
    strcpy(rmTfile, tmpfile);


    if ( errCode = otInitTemplate() ) {
	return errCode;
    } else {
        otCB->cb_ecb->ecb_origStruct = otCB->cb_ecb->ecb_blankStruct;
	otCB->cb_ecb->ecb_blankStruct = 0;

	/*
	 * Is this really necessary, since otDupTemplate() will be called
	 * later by otRequestInitiateUpdate()?
	 */
	if ( errCode = otDupTemplate( otCB->cb_ecb->ecb_origStruct,
	    &(otCB->cb_ecb->ecb_tStruct) ) ) {
	    return errCode;
	}
    }

    if ( otCB->cb_templFile ) {

	if ( errCode = otReadTemplateFromFilename(&tp, otCB->cb_templFile,
	    TRUE, TRUE) ) {
	    return errCode;
	}
	lnum = otGetIDNumValue(tp);
	otFreeTemplate(tp);
	if ( lnum ) {
	    if ( !otCB->cb_CRNumber ) {
		otCB->cb_CRNumber = lnum;
	    } else if ( lnum != otCB->cb_CRNumber ) {
		otPutPostedMessage(OT_DIFFERENT_NUMBER, 
		    otCB->cb_pcb->pcb_project->object_name, lnum,
		    otCB->cb_pcb->pcb_project->object_name, otCB->cb_CRNumber);
		return OT_DIFFERENT_NUMBER;
	    }
	}
    }

    /*
     * Required by otValidateEditCLI() checking.
     */
    otCB->cb_pcb->pcb_CRNumber = otCB->cb_CRNumber;

    /* Catch SIGINT, so if the CR is locked by the "update" command (called
       from the otRequestInitiateUpdate), the server will unlock it */
      
    otCatchSignals(2);

    if ( errCode = otRequestInitiateUpdate() ) {
	return errCode;
    }

    if ( otCB->cb_templFile || otCB->cb_tclString ) {
	if ( errCode = otWriteTemplateToFilename( otCB->cb_ecb->ecb_tStruct, 
	    tmpfile ) ) {
	    return errCode;
	}
    }

    if ( otCB->cb_templFile ) {

	if (!otCB->cb_append )
	    sprintf(command, "cp %s %s", otCB->cb_templFile, tmpfile);
	else
	    sprintf(command, "cat %s >> %s", otCB->cb_templFile, tmpfile);

	if (system(command)) {
	    otPutPostedMessage(OT_TEMPLATE_MOVE_DB, otCB->cb_templFile);
	    errCode = OT_TEMPLATE_MOVE_DB;
	}
	if (otCB->cb_ecb->ecb_tStruct)
	    otFreeTemplate(otCB->cb_ecb->ecb_tStruct);

	if ( errCode = otReadTemplateFromFilename(&otCB->cb_ecb->ecb_tStruct,
	    tmpfile, TRUE, TRUE) ) {
	    return errCode;
	}
    }

/* If neither template file nor tcl string are passed, invoke an editor */

    if (!otCB->cb_templFile && !otCB->cb_tclString) {

        /* Write template structure to a file and invoke an editor */
	if (errCode = otEditForCLI(tmpfile))
	    return errCode;
    }
	
/* Loop in validate/edit cycle until user has corrected errors for an
   interractive interface; one validation cycle otherwise */

    if (errCode = otValidateEditCLI(tmpfile))
        return errCode;

    /* We're going to ignore most all signals from now on...  */
    otCatchSignals(1);

    if (!otConformEdit())
	return errCode;

/* Store an updated object under RCS, call otNotify and otMakeSum */

    if ( errCode = otWriteTemplateToFilename( otCB->cb_ecb->ecb_tStruct, 
	tmpfile ) ) {
	return errCode;
    }

    /*
     * There was a validation error on the server - a rare occurance, since
     * normally validation is done successfully on the client.  This may mean
     * that the server and the client are out of sync.
     */
    if ( errCode = otRequestCompleteUpdate() ) {
	return errCode;
    }

    strcpy(fname.dir, tmpfile);
    cp = strrchr(fname.dir, '/');
    cp++;
    *cp = 0;

    cp = strrchr(tmpfile, '/');
    cp++;
    strcpy(fname.name, cp);

    /*
     * Write again, so CR number, perhaps values assigned by postProcess
     * are written.
     */
    tp = otCB->cb_ecb->ecb_tStruct;
    if ( errCode = otWriteTemplateToFilename( tp, tmpfile ) )
	return errCode;

    if (errCode = otFormDeltamsg(tmpfile, name, FALSE)) {
	return errCode;	
    }

    tp = otCB->cb_ecb->ecb_tStruct;
    subject_line = otGetSubjectLine(otCB->cb_ecb->ecb_origStruct, tp, fname, pcp->pcb_operation, pjp);

    otNotify(pjp, tp, fname, fname, otCB->cb_operation,	pcp->pcb_notify,
	name, subject_line);

    /*
     * TODO Let's review this - should rmTfile be removed on interrupt? then
     * rm rmTfile in signal handler (it doesn't seem to do it now).
     */
    otCleanup();

    if ( otCB->cb_pcb->pcb_ui == CLI ) {
	fprintf(stderr, "Updated %s number %ld\n", pjp->object_name,
	    otGetIDNumValue(tp));
    }

    return errCode;

}



OTErr
otDoValidate()
{
    char *cp;
    bool valid = TRUE;
    OTErr errCode = SUCCESS;

    if (errCode = otInitValidate() )
	return errCode;

    /*
     * Check if the template file contains only values for the default
     * template.
     */
    if (errCode = otTemplateChanged()) {
	if ( OT_WARN(errCode) || OT_SYSTEM(errCode) || 
				    (errCode != OT_TEMPLATE_UNCHANGED))
	    return errCode;
    }

    /*
     * Call validation functions.
     */
    if ( errCode = otValidateLhs() ) {
	valid = FALSE;
	if ( OT_WARN(errCode) || OT_SYSTEM(errCode) ) 
	    return errCode;
    }

    if (errCode = otValidateData()) {
	valid = FALSE;
	if ( OT_WARN(errCode) || OT_SYSTEM(errCode) ) 
	    return errCode;
    }

    if (errCode = otValidateDepend()) {
	valid = FALSE;
	if ( OT_WARN(errCode) || OT_SYSTEM(errCode) )
	    return errCode;
    }

    if (valid)  {
        return errCode;
    }
    else {
	cp = otGetPostedMessage();
	fprintf(stderr, "%s\n", cp);
    }

    if (!valid) {
	errCode = OT_INVALID;
    }

    return errCode;
}


OTErr
otDoDelete()
{
    OTErr errCode = OT_SUCCESS;
    OTFilen fname;
    OTPrivateCB *pcp;
    OTProject *pjp;
    OTTemplate *tp;
    char tmpfile[PATHLEN];
    char *cp, *subject_line;
    char name[NAMELEN];

    pcp = otCB->cb_pcb;
    pjp = pcp->pcb_project;
    /*
     * Client-side, we create a filename to hold 1) the edited version of the
     * template, OR 2) a copy of the user-specified template.
     */
    tmpfile[0] = 0;
    tmpnam(tmpfile);
    strcpy(rmTfile, tmpfile);

    /*
     * Pull CR number in from command line.
     */
    otCB->cb_pcb->pcb_CRNumber = otCB->cb_CRNumber;

    /*
     * We need to initialize both orig and current template structures here
     * so the TCL string returned by tclString command has some storage.
     */
    if ( errCode = otInitTemplate() ) {
	return errCode;
    } else {
        otCB->cb_ecb->ecb_origStruct = otCB->cb_ecb->ecb_blankStruct;
	otCB->cb_ecb->ecb_blankStruct = 0;

	/*
	 * Is this really necessary, since otDupTemplate() will be called
	 * later by otRequestInitiateUpdate()?
	 */
	if ( errCode = otDupTemplate( otCB->cb_ecb->ecb_origStruct,
	    &(otCB->cb_ecb->ecb_tStruct) ) ) {
	    return errCode;
	}
    }
    
    if ( errCode = otRequestDelete() )
	return errCode;

    tp = otCB->cb_ecb->ecb_tStruct;
    /*
     * Write again, so CR is written for notification.
     */
    if ( errCode = otWriteTemplateToFilename( tp, tmpfile ) )
	return errCode;

    strcpy(fname.dir, tmpfile);
    cp = strrchr(fname.dir, '/');
    cp++;
    *cp = 0;

    cp = strrchr(tmpfile, '/');
    cp++;
    strcpy(fname.name, cp);

    sprintf(name, "deleted %s", pjp->object_name);
    subject_line = otGetSubjectLine(NULL, tp, fname, pcp->pcb_operation, pjp);

    otNotify(pjp, tp, fname, fname, otCB->cb_operation, pcp->pcb_notify, name,
	subject_line);

    otCleanup();

    if ( otCB->cb_pcb->pcb_ui == CLI ) {
	fprintf(stderr, "otDeleteObject: Deleted %s number %ld\n",
	    pjp->object_name, otGetIDNumValue(tp));
    }

    return errCode;

}
