/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

#ifndef LINT
static char RCSid_otProject[] =
    "$RCSfile: otNetwork.c,v $ $Revision: 1.1.8.2 $ $Date: 1994/01/19 18:09:49 $";
#endif

/*
 * The following copyright is included to acknowledge the use of code
 * from the Tcl/DP package.
 */

/*
 * Copyright 1992 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

/*
 * otNetwork.c - network interfaces for remote TCL use.
 */

#include <errno.h>
#include <stdio.h>
#ifdef _INC_SYS_TYPES
#include <sys/types.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>

#include <tcl.h>
#include <tclInt.h>

#ifdef _TK
#include <tk.h>
#endif

#include "ot.h"
#include "otInt.h"

#include "dputil.h"
#include "dpnetwork.h"

#define TOK_RPC		'e'
#define TOK_RDO		'd'
#define TOK_RET		'r'
#define TOK_ERR		'x'

/* ---------------------------------------------------- */

static int rpcWaiting[256];		/* indexed by fd;	 */
static int  rpcResult[256];		/* indexed by fd;	 */
static char *rpcValue[256];		/* indexed by fd;	 */

static double         rpcTime[256];	/* indexed by fd;	 */
static int         rpcTimeout[256];	/* indexed by fd;	 */
static char *rpcTimeoutReturn[256];	/* indexed by fd;	 */


int ot_TkError();
int otCloseRPC();

OTErr otNetInit()
{
    Tcl_Interp *interp = otCB->cb_pcb->pcb_interp;

    Tdp_Init(interp);
    /*
     * For security, we may replace dpInit() w/ the line(s) that create
     * an instance of rpcFile variable in libdp/util.c.
     */
/*
    Tcl_CreateCommand(interp, "tkerror", ot_TkError, (ClientData) NULL,
	(void (*) ()) NULL);
    Tcl_CreateCommand(interp, "closeRPC", otCloseRPC, (ClientData) NULL,
	(void (*) ()) NULL);
 */

}


OTErr otConn()
{
    ClientData notUsed;
    OTErr err;
    OTPrivateCB *pcp = otCB->cb_pcb;
    OTProject *prjp = pcp->pcb_project;
    Tcl_Interp *interp = pcp->pcb_interp;
    char portNum[SHORTSTR], connFile[SHORTSTR], command[SHORTSTR];
    char tclOpenCommand[LONGVALUE];
    int sargc, res;
    char *sargv[5];
    OpenFile *filePtr;
    char *cp, *pn, *otDbpath;
    char *rc_cp, *str;
    char fn[PATHLEN];

    Tcl_ResetResult(interp);
    notUsed = (ClientData)0;
    if ( !prjp->server_host ) {
        otPutPostedMessage(OT_HOST_UNSPECIFIED);
	return OT_HOST_UNSPECIFIED;
    }

    if ( prjp->odexmcli[0] ) {
	sargc = 3;
	sargv[0] = "open";
	sargv[1] = tclOpenCommand;

	sprintf(tclOpenCommand, "|%s", prjp->odexmcli);
	sargv[2] = "r+";
	sargv[3] = 0;
	if ( Tcl_OpenCmd(notUsed, interp, sargc, (char **)sargv) ) {
	    return OT_GENERAL_ERROR;
	} else if ( Tdp_GetOpenFile(interp, interp->result, 1, 0, &filePtr) ) {
	    return OT_GENERAL_ERROR;
	}

	strcpy(pcp->pcb_fileInName, interp->result);
	strcpy(pcp->pcb_fileOutName, interp->result);

    } else {

	portNum[0] = 0;
	if ( pn = getenv("OT_PORT") )
	    strcpy(portNum, pn);
	else if (prjp->server_port)
	    sprintf(portNum, "%d", prjp->server_port);

	if ( !portNum[0] ) {
	    otPutPostedMessage(OT_PORT_UNSPECIFIED);
	    return OT_PORT_UNSPECIFIED;
	}

	sargc = 3;
	sargv[0] = "connect";
	sargv[1] = prjp->server_host;
	sargv[2] = portNum;
	sargv[3] = 0;

	if ( Tdp_ConnectCmd(notUsed, interp, sargc, sargv) != TCL_OK ) {
	    return OT_GENERAL_ERROR;
	}
	strcpy(connFile, interp->result);
    
	/*
	 * The return value from Tdp_ConnectCmd() has an appended status field 
	 * which we must remove.
	 */
	for (cp = connFile; *cp; cp++)
	    if (*cp == ' ') {
		*cp = 0;
		break;
	    }
	if (Tdp_GetOpenFile(interp, connFile, 1, 0, &filePtr) != TCL_OK) {
	    otPutPostedMessage(OT_INTERNAL_ERROR, "Tdp_GetOpenFile()");
	    return OT_INTERNAL_ERROR;
	}
	strcpy(pcp->pcb_fileInName, connFile);
	strcpy(pcp->pcb_fileOutName, connFile);
    }

    pcp->pcb_fileIn = pcp->pcb_fileOut = filePtr;

    /*
     * Always initiate session w/ setting username.  For the moment 
     * the status return is dropped.
     */
    err = otGetUserName();

    if ( otDbpath = getenv("OT_DBPATH") )
        sprintf(command, "userName set %s; set env(OT_DBPATH) %s\n",
	    otCB->cb_pcb->pcb_uName, otDbpath);
    else 
	sprintf(command, "userName set %s\n", otCB->cb_pcb->pcb_uName);

    if ( (res = Tdp_RPC(interp, pcp->pcb_fileIn, pcp->pcb_fileOut,
	command, -1, -1, 0)) != TCL_OK ) {
	otPutPostedMessage(OT_TCL_CALLER, "otConn()", interp->result);
	return OT_TCL_CALLER;	
    }

    if ( err )
	return err;

    return OT_SUCCESS;
    
}


int
otRemoteTcl(command)
char *command;
{
    OTErr err;
    OTPrivateCB *pcp = otCB->cb_pcb;
    OTProject *prjp = pcp->pcb_project;
    Tcl_Interp *interp = pcp->pcb_interp;
    int result;

    if ( pcp->pcb_fileOut == 0 ) {
	if ( err = otConn() )
	    return err;
    }
    result = Tdp_RPC(interp, pcp->pcb_fileOut, pcp->pcb_fileIn, command,
	-1, -1, 0);

    return result;

}

OTErr
otBringDownServer()
{
    OTErr err = OT_SUCCESS;
    OTPrivateCB *pcp = otCB->cb_pcb;
    ClientData notUsed;
    Tcl_Interp *interp = pcp->pcb_interp;
    int res, sargc;
    static char cmd[SHORTSTR];
    char *sargv[4];

    if ( !pcp->pcb_fileIn || !pcp->pcb_fileOut )
	return err;

    res = otRemoteTcl("serverExit");

    if ( pcp->pcb_fileIn->numPids ) {
	sargc = 2;
	sargv[0] = "close";
	sargv[1] = pcp->pcb_fileInName;
	sargv[2] = 0;

	if ( Tcl_CloseCmd(notUsed, interp, sargc, (char **)sargv) ) {
	    otPutPostedMessage(OT_INTERNAL_ERROR, interp->result);
	    err = OT_INTERNAL_ERROR;
	}
    } else {
	if ( fclose(pcp->pcb_fileIn->f) ) {
	    otPutPostedMessage(OT_GENERAL_ERROR,
		"error in closing connection to server");
	    err = OT_GENERAL_ERROR;
	}
    }

    pcp->pcb_fileIn = pcp->pcb_fileOut = 0;
    pcp->pcb_fileInName[0] = pcp->pcb_fileOutName[0] = 0;

    return err;

}

OTErr
otMakeServer()
{
    OTPrivateCB *pcp = otCB->cb_pcb;
    Tcl_Interp *interp = pcp->pcb_interp;
    OpenFile *filePtrIn, *filePtrOut;
    int result, sargc, sock, fd, srclen;
    char *sargv[4];
    char rpcFileBuf[NAMELEN];
    static char cmd[SHORTSTR];
    char *pn;
    ClientData notUsed;
    struct sockaddr_in addr, src;
    u_short tmp;

    /*
     * Return these fprintf's as OTErr statuses.
     */
    if ( pn = getenv("OT_PORT") ) {

	sscanf(pn, "%hd", &tmp);
	addr.sin_port = htons(tmp);

	if ( (sock = socket(AF_INET, SOCK_STREAM, 0)) < 0 ) {
	    otPutPostedMessage( OT_SOCKET );
	    return OT_SOCKET;
	}

	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(sock, (struct sockaddr *)&addr,
	    sizeof(struct sockaddr_in)) < 0) {
	    /*
	     * Later, change this to OT_BIND.
	     */
	    otPutPostedMessage( OT_SOCKET );
	    return OT_SOCKET;
	}

	if ( listen(sock, SOMAXCONN) < 0 ) {
	    /*
	     * Later, change this to OT_LISTEN.
	     */
	    otPutPostedMessage( OT_SOCKET );
	    return OT_SOCKET;
	}

	srclen = sizeof(struct sockaddr);
	do {
	    errno = 0;
	    fd = accept(sock, (struct sockaddr *)&src, &srclen);
	} while ( fd < 0 || errno == EINTR );

	dup2(fd, 0);
	close(fd);
	dup2(0,1);
	dup2(0,2);
    }

    Tdp_Init(interp);

    if (Tdp_GetOpenFile(interp, "stdin", 0, 0, &filePtrIn) != TCL_OK) {
	otPutPostedMessage(OT_INTERNAL_ERROR, "Tdp_GetOpenFile() failed");
	return OT_INTERNAL_ERROR;
    }
    if (Tdp_GetOpenFile(interp, "stdout", 1, 0, &filePtrOut) != TCL_OK) {
	otPutPostedMessage(OT_INTERNAL_ERROR, "Tdp_GetOpenFile() failed");
	return OT_INTERNAL_ERROR;
    }

    /*
     * NB these are pointers to storage on the stack.
     */
    pcp->pcb_fileIn = filePtrIn;
    pcp->pcb_fileOut = filePtrOut;

    sprintf(rpcFileBuf, "file%d", fileno(filePtrIn->f));

    Tcl_SetVar(interp, "rpcFile", rpcFileBuf, TCL_GLOBAL_ONLY);

    /*
     * Main loop.
     */
    result = 0;
    while ( !result && pcp->pcb_fileIn )
	result = ot_ProcessRPCMessages(interp, pcp->pcb_fileIn, 
	    pcp->pcb_fileOut, 1);

    sprintf(cmd, "close $rpcFile");
    result = Tcl_Eval(interp, cmd);
    logWarn("ot: closing rpc connection: %s",
	result ? interp->result : "exit");

    if (otCB->cb_pcb->pcb_efp) {
	fclose(otCB->cb_pcb->pcb_efp);
	otCB->cb_pcb->pcb_efp = 0;
    }

    return OT_SUCCESS;

}


int
otServerExit(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    otCB->cb_pcb->pcb_fileIn = 0;

    /*
     * I don't use otCleanAfterUpdate() because it takes a status argument stat
     * which causes otCleanAfterEnter() to call otCleanup() only on an error
     * condition signaled by stat - we want it to clean up all the time.
     */
    if ( gotCRfile[0] ) {
	logWarn("ot: unlocking file %s", gotCRfile);
	if ( !otUnlockObject() )
	    logWarn("ot: error unlocking file");
    }
    if ( otCB->cb_pcb->pcb_kwikPixSock )
	close ( otCB->cb_pcb->pcb_kwikPixSock ) ;

    otCleanup();
    return TCL_OK;


}


int
ot_TkError(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    /*
     * TODO - generalize this for appropriate errors.
     */
    if (argc > 1) {
        if (!strncmp(argv[1], "Tdp_PacketReceive -- error reading", 34))
	  ;
      else
	fprintf(stderr, "net error: %s\n", argv[1]);	
    }
    return TCL_OK;

}

/*
 *--------------------------------------------------------------
 *
 * ot_ProcessRPCMessages --
 *
 *	This procedure is the C interface to the "ProcessRPCMessages" 
 *	command.  This command reads one or more messages off of a given
 *	socket, and processes these messages as RPC messages.
 *	An RPC message is headed by one of the following:
 *
 *	TOK_RET	: message is the return value 
 *		  of an RPC which this process requested.
 *	TOK_ERR	: message is the error return value 
 *		  of an RPC which this process requested.
 *	TOK_RDO	: message is an incoming request for this
 *		  process to evaluate an RDO.
 *	TOK_RPC	: message is an incoming request for this
 *		  process to evaluate an RPC.
 *
 *	This procedure will block as it reads the socket.
 *	If wait is specified (non-zero), then this procedure will
 *	not return until it either a TOK_RET or TOK_ERR message
 *	is received.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
ot_ProcessRPCMessages(interp, filePtrIn, filePtrOut, wait)
    Tcl_Interp *interp;
    OpenFile *filePtrIn;  /* Socket connected to RPC peer or other input  */
    OpenFile *filePtrOut; /* Socket connected to RPC peer or other output */
    int wait;		/* 1 if wait for an incoming TOK_RET
			   or TK_ERR message. */
{
    char *buffer;
    char token;

    int i;
    int fd;

    int result;

    int    argc;
    char **argv;

    char temp[20];

    if ( filePtrIn->f )
	fd = fileno (filePtrIn->f);
    else
	return TCL_ERROR;

    while (1) 
      {
	if (Tdp_PacketReceive (interp, filePtrIn, 1) != TCL_OK) 
	  {
	    rpcWaiting[fd] = 0;
	    rpcResult[fd]  = TCL_ERROR;
	    rpcTimeout[fd] = 0;
	    rpcTime[fd]    = 0.0;
	    
	    if (rpcValue[fd])
	      ckfree ((char *) rpcValue[fd]);
	    rpcValue[fd] = NULL;

	    if (rpcTimeoutReturn[fd])
	      ckfree ((char *) rpcTimeoutReturn[fd]);
	    rpcTimeoutReturn[fd] = NULL;

	    Tcl_AppendResult (interp, 
			      "RPC error : Tdp_ProcessRPCMessages",
			      (char *) NULL);
/* OT CHANGE */
#ifdef _TK
	    sprintf (temp, "file%d", fd);
	    Tcl_VarEval (interp, "filehandler ", temp,
			 (char *) NULL);
	    Tcl_VarEval (interp, "close ", temp,
			 (char *) NULL);
#endif

	    return TCL_ERROR;
	  }

	if (Tcl_SplitList (interp, interp->result, &argc, &argv) != TCL_OK) 
	  return TCL_ERROR;

	for (i = 0; i < argc; i++) 
	  {
	    buffer = argv[i];

	    /* Skip whitespace;
	     */
	    while (buffer && (buffer[0] != '\0') && (buffer[0] == ' '))
	      buffer++;

	    /* Grab incoming message token;
	     */
	    token = buffer[0];	
	    buffer++;

	    /* Skip whitespace;
	     */
	    while (buffer && (buffer[0] != '\0') && (buffer[0] == ' '))
		buffer++;

	    /* Handle received message depending on token;
	     */
	    switch (token) 
	      {
	      case TOK_RET:		/* received message is the return 
					 * value of an RPC which this process
					 * originated. */
		rpcWaiting[fd] = 0;
		rpcResult[fd]  = TCL_OK;
		rpcTimeout[fd] = 0;
		rpcTime[fd]    = 0.0;

		if (rpcValue[fd])
		  ckfree ((char *) rpcValue[fd]);
		rpcValue[fd] = NULL;

		rpcValue[fd] = (char *) ckalloc (strlen (buffer) + 1);
		strcpy (rpcValue[fd], buffer);

		if (rpcTimeoutReturn[fd])
		  ckfree ((char *) rpcTimeoutReturn[fd]);
		rpcTimeoutReturn[fd] = NULL;

		ckfree ((char *) argv);
		return TCL_OK;

	      case TOK_ERR:		/* received message is the error 
					 * return value of an RPC which 
					 * this process originated. */
		rpcWaiting[fd] = 0;
		rpcResult[fd]  = TCL_ERROR;
		rpcTimeout[fd] = 0;
		rpcTime[fd]    = 0.0;

		if (rpcValue[fd])
		  ckfree ((char *) rpcValue[fd]);
		rpcValue[fd] = NULL;

		rpcValue[fd] = (char *) ckalloc (strlen (buffer) + 1);
		strcpy (rpcValue[fd], buffer);

		if (rpcTimeoutReturn[fd])
		  ckfree ((char *) rpcTimeoutReturn[fd]);
		rpcTimeoutReturn[fd] = NULL;

		result = Tcl_VarEval (interp, "error {", buffer, "}",
				      (char *) NULL);

		ckfree ((char *) argv);
		return (result);

	      case TOK_RDO:		/* evaluate the received message 
					 * as an RDO request */
		{
		  char *command;

		  command = ckalloc (strlen (buffer) + 1);
		  strcpy (command, buffer);

		  Tdp_ReceiveRPC (interp, filePtrIn, filePtrOut, command, 0);

		  ckfree ((char *) command);
		}
		break;
		
	      case TOK_RPC:		/* evaluate the received message 
					 * as an RPC request */
		{
		  char *command;

		  command = ckalloc (strlen (buffer) + 1);
		  strcpy (command, buffer);

		  /*
		   * LOGGING:
		   *   the variable 'command' contains the string to log.
		   */

		  logWarn("%s: %s", 
		      otCB->cb_pcb->pcb_uName ? otCB->cb_pcb->pcb_uName : "ot",
		      command);
		  Tdp_ReceiveRPC (interp, filePtrIn, filePtrOut, command, 1);
		  ckfree ((char *) command);
		}
		break;

	      default:
		break;
	      }
	  }

	ckfree ((char *) argv);

	if (!wait)
	  return TCL_OK;
      }
}
