/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */


#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <sys/time.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

#ifndef LINT
static char RCSid_otUpdate[] =
    "$RCSfile: otUpdate.c,v $ $Revision: 1.1.6.3 $ $Date: 1994/01/25 19:27:52 $";

#endif


/*
 *              List of functions defined in the otUpdate.c
 *
 *	otPrepareForUpdate()
 *	otInitUpdate()
 *	otGetObjectForUpdate()
 *	otUpdateObject()
 */


OTErr otChkTemplateForUpdate();
OTErr otChkInputForUpdate();
OTErr otMoveFileToDB();
OTErr otFormDeltamsg();


OTErr
otPrepareForUpdate(tmpfile)
char * tmpfile;
{

    OTErr errCode = OT_SUCCESS;

    if (errCode = otInitUpdate())
	return errCode;

    if (errCode = otChkInputForUpdate())
	return errCode;

    if (errCode = otGetObjectForUpdate(tmpfile))
	 return errCode;

    return errCode;
}



OTErr
otInitUpdate()
{
    OTTemplate * tStruct;

    OTErr errCode = OT_SUCCESS;

    otClearEnterCB(otCB->cb_ecb);

    otCB->cb_pcb->pcb_CRNumber = otCB->cb_CRNumber;

    /*
     * If template file is passed -- read it into structure and check it
     * (i.e., project matches project in template, CR number in template 
     * matches CR number on cmd line).  
     */
    if (otCB->cb_templFile && !otCB->cb_append) {
	if (errCode = otChkTemplateForUpdate(&tStruct))
	    return errCode;

        /* store template structure pointer */
        otCB->cb_ecb->ecb_tStruct = tStruct;
    }

    return errCode;
}


OTErr
otChkTemplateForUpdate(tStruct)
OTTemplate **tStruct;
{
    long id;
    OTErr errCode = OT_SUCCESS;

    /* read file into template structure */
    if (errCode = otReadTemplateFromFilename(tStruct, otCB->cb_templFile, 
	TRUE, TRUE))
        return errCode;

    /* check that project name in the template maches the project */
    if (errCode = otChkProjectInTemplate(*tStruct, otCB->cb_project))
        return errCode;

    /* make sure there is no CR number conflict */
    if (id = otGetIDNumValue(*tStruct)) {
        if (otCB->cb_CRNumber && (otCB->cb_CRNumber != id)) {
	    otPutPostedMessage(OT_DIFFERENT_NUMBER, 
		    otCB->cb_pcb->pcb_project->object_name, id, 
		    otCB->cb_pcb->pcb_project->object_name, otCB->cb_CRNumber);
	    return OT_DIFFERENT_NUMBER;
        }
        otCB->cb_pcb->pcb_CRNumber = id;
    }
    return errCode;
}



OTErr
otChkInputForUpdate()
{
    long  topNumber;
    OTErr errCode = OT_SUCCESS;

    OTProject * proj = otCB->cb_pcb->pcb_project;

    /* CR number is required for an update */
    if (!otCB->cb_pcb->pcb_CRNumber) {
	otPutPostedMessage(OT_UPDATE_NUMBER, proj->object_name);
	return OT_UPDATE_NUMBER;
    }

    /* make sure CR number is within the number range for the project */
    if (errCode = otReadTopNumber(proj, &topNumber)) 
	return errCode;

    if (otCB->cb_pcb->pcb_CRNumber >= topNumber) {
        otPutPostedMessage(OT_OBJECT_NOT_FOUND, proj->object_name,
	    otCB->cb_pcb->pcb_CRNumber);
	return OT_OBJECT_NOT_FOUND;
    }

    /* append function requires template file */
    if (otCB->cb_append && (!otCB->cb_templFile)) {
	otPutPostedMessage(OT_APPEND_USAGE);
	return OT_APPEND_USAGE;
    }

    return errCode;
}



OTErr
otGetObjectForUpdate(tmpfile)
char * tmpfile;
{
    OTTemplate * origStruct;
    OTFilen * fname;

    long CRNumber = otCB->cb_pcb->pcb_CRNumber;		/* object number */
    OTProject * proj = otCB->cb_pcb->pcb_project; 	
    OTErr errCode = OT_SUCCESS;

    /* form CR filename */
    fname = 0;
    if (errCode = otFormName(proj, CRNumber, &fname)) {
	if (fname)
	    free(fname);
	return errCode;
    }

    otCB->cb_ecb->ecb_fname = fname;

    if (errCode = otLockCRFile(otCB, proj, *fname))
	return errCode;

    strcat(fname->name, ".edit");    /* name for the local version */

    sprintf(tmpfile, "%s%s", fname->dir, fname->name); 
    strcpy(rmTfile, tmpfile); 

    if (errCode = otReadTemplateFromFilename(&origStruct, tmpfile, TRUE, TRUE))
        return errCode;

    otCB->cb_ecb->ecb_origStruct = origStruct;

    /* if template file is passed, move it to the database area */
    /* this code should never get hit in the v3 server implementation */
    if (otCB->cb_templFile) {
	if (errCode = otMoveFileToDB(otCB))
	    return errCode;
    }

    return errCode;
}



OTErr
otMoveFileToDB(cb)
OTControlBlock * cb;		/* control block structure */
{

    OTTemplate * tStruct;
    char command[COMMAND];
    char tmpFname[NAMELEN];

    OTErr errCode = OT_SUCCESS;

    sprintf(tmpFname, "%s%s", cb->cb_ecb->ecb_fname->dir, 
			    cb->cb_ecb->ecb_fname->name);

    if (!cb->cb_append)
	sprintf(command, "(cp %s %s) >2 /dev/null", cb->cb_templFile,
	    tmpFname);
    else
	sprintf(command, "(cat %s >> %s) >2 /dev/null", cb->cb_templFile,
	    tmpFname);

    if (system(command)) {
	otPutPostedMessage(OT_TEMPLATE_MOVE_DB, cb->cb_templFile);
	errCode = OT_TEMPLATE_MOVE_DB;
    }

    /* need to build tStruct fro an append operation */
    if (cb->cb_append) {

	if (errCode = otReadTemplateFromFilename(&tStruct, tmpFname, TRUE,
	    TRUE))
	    return errCode;

	cb->cb_ecb->ecb_tStruct = tStruct;
    }

    return errCode;
}



/*
 * Do change analysis, notification, delta.
 */
OTErr
otUpdateObject(tmpfile)
char    *tmpfile;		/* template file */
{
    char deltamsg[LONGVALUE];
    char *cp1;
    char *subject_line;
    OTFilen newFile;
    OTFilen *pFilen;

    OTProject * proj = otCB->cb_pcb->pcb_project;
    OTTemplate * origtStruct = otCB->cb_ecb->ecb_origStruct;
    OTTemplate * tStruct = otCB->cb_ecb->ecb_tStruct;
    OTFilen * fname = otCB->cb_ecb->ecb_fname;
    OTOperation operation = otCB->cb_operation;
    bool notify = otCB->cb_pcb->pcb_notify;

    OTErr errCode = OT_SUCCESS;

    /*
     * When user's template file is specified, it is already moved/appended
     * to the object's file in the RCS area.  However, the -k flag can be
     * used in combination with the -x flag, in which case the template may
     * have been changed in the template.  Therefore we must write.
     */
    if ( (!otCB->cb_templFile) || (otCB->cb_tclString) ) {
	if (errCode = otWriteTemplateToFilename(tStruct, tmpfile))
	    return errCode;

	otFormFilen(&newFile, tmpfile);
    }

    if (errCode = otFormDeltamsg(tmpfile, deltamsg, TRUE))
	return errCode;

    if (otCB->cb_templFile) 
	pFilen = fname;
    else
	pFilen = &newFile;

    if ((errCode = otDeltaCRFile(fname, *pFilen, deltamsg, FALSE, proj)) 
		    == OT_RCS_CI) {
	otPutPostedMessage(OT_RCS_ADMIN, proj->object_name, fname->name);
	errCode = OT_RCS_ADMIN;
    }

    if (errCode == OT_OBJECT_UPDATED) {
	errCode = OT_SUCCESS;
    }

    return errCode;

}


/* Form delta message and history lines for an update */
OTErr
otFormDeltamsg(tmpfile, deltamsg, appendHistory)
char    *tmpfile;		/* template file */
char	*deltamsg;
bool    appendHistory;
{
    FILE *tfp;
    long timer;
    struct tm *timeptr;
    char times[NAMELEN];
    char *kwd, *cp, *cp1;
    OTNote *newnotep;
    int i, j;

    OTProject * proj = otCB->cb_pcb->pcb_project;
    OTTemplate * otp = otCB->cb_ecb->ecb_origStruct;
    OTTemplate * tp = otCB->cb_ecb->ecb_tStruct;

    OTErr errCode = OT_SUCCESS;

    if ( !otp || !tp ) {
	otPutPostedMessage(OT_INTERNAL_ERROR, "template: otFormDeltamsg()");
	return OT_INTERNAL_ERROR;
    }

    if ( !otp->tr || !tp->tr ) {
	otPutPostedMessage(OT_INTERNAL_ERROR, "headers: otFormDeltamsg()");
	return OT_INTERNAL_ERROR;
    }

    if ((tfp = fopen(tmpfile, "a+")) == NULL ) {
	otPutPostedMessage(OT_TEMPLATE_UPDATE, tmpfile, proj->object_name);
	return OT_TEMPLATE_UPDATE;
    }

    timer = time(0L);
    timeptr = localtime(&timer);
    strcpy(times, asctime(timeptr));
    times[strlen(times) - 1] = 0;

    /* What fields have changed? --> a source control message */
	strcpy(deltamsg, "changed fields: ");

    DBUG_MED((stderr, "Test original fields\n"));

    for (i=0; otp->tr[i].field && otp->tr[i].field[0];  i++) {

	if ( !(kwd = otGetFieldNameAbbr(otp->tr[i].field, proj)) )
	    kwd = otp->tr[i].field;

	if (!(cp = otGetHeaderFieldValue(otp->tr[i].field, tp)) &&
	    !otp->tr[i].value)
	    continue;
	else {
	    DBUG_MED((stderr, "otp %d == %s and tp's is %s\n", i, otp->tr[i].field, otGetHeaderFieldValue(otp->tr[i].field, tp)));
	    if ((!otp->tr[i].value && cp) || (otp->tr[i].value && !cp) ||
		strcmp(otp->tr[i].value, cp)) {
	        
	        int k, l;
		
	        DBUG_MED((stderr, "values not identical (%s != %s)\n", otp->tr[i].value, cp));
		for (k=0, l=strlen(deltamsg); otp->tr[i].field && 
		    otp->tr[i].field[k]; k++)
		    if ( (otp->tr[i].field[k] != '"') && 
			 (otp->tr[i].field[k] != '\'') )
			deltamsg[l++] = otp->tr[i].field[k];

		deltamsg[l] = 0;
		strcat(deltamsg, "  ");
		     
		/*
		 * kwd could be null if original field has been deleted...
		 */
		otRemoveNewline(otp->tr[i].value);
		otRemoveNewline(cp);
		if (appendHistory)
		    fprintf(tfp, "+0HISTORY %s %s\t%s: %s -> %s\n", times, 
		        otCB->cb_pcb->pcb_uName, kwd, 
		        otp->tr[i].value?otp->tr[i].value : "NULL",
			cp ? cp : "NULL");
	    }
	}
    }

    DBUG_MED((stderr, "test new fields...\n"));

    for (i=0; tp->tr[i].field && tp->tr[i].field[0];  i++) {

        DBUG_MED((stderr, "Look for match for %s\n", tp->tr[i].field));

	for (j=0; otp->tr[j].field && otp->tr[j].field[0]; j++)
	    if ((!strcmp(tp->tr[i].field, otp->tr[j].field)) 
		|| (!tp->tr[i].value) )
		break;

	DBUG_MED((stderr, "Found %s value in original structure\n", otp->tr[j].field));
	    /*
	     * if tp->tr[i].field was NOT found in the original tStruct,
	     * then it is a field "invented" by the user and therefore
	     * should go in the delta message and history line.
	     */

	if (otp->tr[j].field && !otp->tr[j].field[0]) {
	    int k, l;

	    for (k=0, l=strlen(deltamsg); 
		tp->tr[i].field && tp->tr[i].field[k]; k++)
	        if ( (tp->tr[i].field[k] != '"') &&
		     (tp->tr[i].field[k] != '\'') )
		    deltamsg[l++] = tp->tr[i].field[k];
	    deltamsg[l] = 0;
	    strcat(deltamsg, "  ");

	    otRemoveNewline(tp->tr[i].value);
	    if (appendHistory)
	        fprintf(tfp, "+0HISTORY %s %s\tadded %s (%s)\n", times, 
	            otCB->cb_pcb->pcb_uName, tp->tr[i].field,
		    tp->tr[i].value ? tp->tr[i].value : "NULL");
	}
    }

    if (errCode = otNotesCmp(otp->notep, tp->notep, &newnotep)) 
	return errCode;
    
    if (newnotep) {
	strcat(deltamsg, " new/changed/deleted note(s)");

	for (i=0; (newnotep[i].who[0]) && i < MAXTEMPLATE; i++) {
	    strcat(deltamsg, " [");			    
	    strcat(deltamsg, newnotep[i].who);
	    strcat(deltamsg, " ");
	    strcat(deltamsg, newnotep[i].date);
	    strcat(deltamsg, " ");
	    strcat(deltamsg, newnotep[i].sens);
	    strcat(deltamsg, "]");
	    if (appendHistory)
	        fprintf(tfp, "+0HISTORY %s %s\tadded/changed [%s %s %s]\n", 
		    times, otCB->cb_pcb->pcb_uName,
		    newnotep[i].who ? newnotep[i].who : "",
		    newnotep[i].date ? newnotep[i].date : "",
		    newnotep[i].sens ? newnotep[i].sens : "");
	}
	freeNotes(newnotep);
    }

    fclose(tfp);

    /*
     * Put the new information in the OTNotedata structure.  (This routine
     * could be rewritten to deposit the history info directly in ecb_tStruct's
     * notedata area.
     */
    if ( otCB->cb_ecb->ecb_tStruct )
	otFreeTemplate(otCB->cb_ecb->ecb_tStruct);

    errCode = otReadTemplateFromFilename( &(otCB->cb_ecb->ecb_tStruct),
	tmpfile, TRUE, TRUE );

    return errCode;
}


/*
 *                o t R e q u e s t I n i t i a t e U p d a t e
 *
 * This is the call of the client.
 */
OTErr
otRequestInitiateUpdate()
{
    int status;
    char prjCmd[LONGVALUE], fldCmd[LONGVALUE];
    char updateCmd[LONGVALUE];
    char name[NAMELEN];
    char *tclString;
    OTPrivateCB *pcp;
    OTProject *pjp;
    OTTemplate *ott_template;
    OTFilen fname;
    OTErr err;
    Tcl_Interp *interp;
    char *newCR = 0;

    /*
     * via otRemoteTcl...
     *
     * . if OT_DBPATH is set, send it
     * project set <projName>
     * update
     * send 'tclString orig' so we refresh the CR in ecb_{orig,}tStruct...
     */

    pcp = otCB->cb_pcb;
    interp = pcp->pcb_interp;

    if ( pjp = pcp->pcb_project ) {
	sprintf(prjCmd, "project set %s\n", pjp->name);

	if ( status = otRemoteTcl(prjCmd) ) {
	    otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	    return OT_TCL_CALLER;
	}
	DBUG_MIN((stderr, "sent >> %s\n", prjCmd));

	sprintf(fldCmd, "project fields\n");
	if ( status = otRemoteTcl(fldCmd) ) {
	    otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	    return OT_GENERAL_ERROR;
	}
	DBUG_MIN((stderr, "sent >> %s\n", fldCmd));
	if ( otCopyString( interp->result, &(pjp->server_fields) ) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "otRequestEnter()");
	    return OT_MALLOC_LOCATION;
	}
	DBUG_MIN((stderr, "received >> %s\n", pjp->server_fields));
    } else {
	otPutPostedMessage(OT_NEED_PROJECT);
	return OT_NEED_PROJECT;
    }

    sprintf(updateCmd, "update %ld\n", otCB->cb_CRNumber);
    if (status = otRemoteTcl(updateCmd)) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_TCL_CALLER;
    }
    DBUG_MIN((stderr, "sent >> %s", updateCmd));

    if (status = otRemoteTcl("tclString orig\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_TCL_CALLER;
    } else {
        if ( otCopyString(interp->result, &newCR) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION,
		"otRequestInitiateUpdate()");
	    return OT_MALLOC_LOCATION;
	}

	DBUG_MIN((stderr, "received >> %s\n", newCR));
	if ( newCR || *newCR ) {
	    err = otTclEval(newCR);
	    free(newCR);
	}
	if (err)
	    return err;
    }

    /*
     * The orig and current templates are both required if validation is to
     * be done on the client-side.  No need to free ecb_tStruct, since it is
     * still blank.
     */
    if ( otCB->cb_ecb->ecb_tStruct )
	otFreeTemplate(otCB->cb_ecb->ecb_tStruct);
    if ( err = otDupTemplate( otCB->cb_ecb->ecb_origStruct,
	&(otCB->cb_ecb->ecb_tStruct) ) ) {
	return err;
    }

    return OT_SUCCESS;

}

/*
 *                o t R e q u e s t C o m p l e t e U p d a t e
 *
 * This is the call of the client.
 */
/* how will we know where the existing temp file is? */
OTErr
otRequestCompleteUpdate()
{
    int status;
    char *otDbpath;
    char pathCmd[LONGVALUE], prjCmd[LONGVALUE];
    char name[NAMELEN];
    char *tclString;
    OTPrivateCB *pcp;
    OTProject *pjp;
    OTTemplate *ott_template;
    OTFilen fname;
    OTErr err;
    bool orig;
    Tcl_Interp *interp;
    char *xString, *newCR;

    /*
     * via otRemoteTcl...
     *
     * send template in TclString format
     * send <tclString> (user's -x argument)
     * send commit
     * send tclString 'the command' so we refresh the CR in ecb_tStruct...
     */

    newCR = 0;
    orig = FALSE;
    pcp = otCB->cb_pcb;
    ott_template = otCB->cb_ecb->ecb_tStruct;
    interp = pcp->pcb_interp;

    xString = otCB->cb_tclString;

    if ( !(pjp = pcp->pcb_project) ) {
	otPutPostedMessage(OT_NEED_PROJECT);
	return OT_NEED_PROJECT;
    }

    if ( err = otWriteTemplateToTcl(pjp, ott_template, &tclString, orig) ) {
	return err;
    }

    /* TODO.
     * what error processing required here - we need to send exit to the server
     * if we can't build any tcl strings.
     */

    status = otRemoteTcl(tclString);
    DBUG_MIN((stderr, "sent >> %s\n", tclString));

    free(tclString);
    if (status) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    }

    if ( xString ) {
	if (status = otRemoteTcl(xString)) {
	    otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	    return OT_GENERAL_ERROR;
	}
	DBUG_MIN((stderr, "sent >> %s\n", xString));
    }

    if (status = otRemoteTcl("commit\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    } else if ( *interp->result ) {
	otPutPostedMessage(OT_INVALID, interp->result);
	fprintf(stderr, "%s\n", otGetPostedMessage() );
/*	return OT_INVALID; */
    }
    DBUG_MIN((stderr, "sent >> commit\n"));

    if (status = otRemoteTcl("tclString\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_TCL_CALLER;
    } else {
        if ( otCopyString(interp->result, &newCR) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION,
		"otRequestCompleteUpdate()");
	    return OT_MALLOC_LOCATION;
	}

	DBUG_MIN((stderr, "received >> %s\n", newCR));
	err = otTclEval(newCR);
	free(newCR);
	if (err)
	    return err;
    }

    return OT_SUCCESS;

}
