#
# Drawing Conference Application
# ------------------------------
#
# This is a *really* simple drawing program, allowing users to create and
# manipulate rectangles in a canvas.  There is absolutely no concurrency
# control, locking or anything else, so you can have great games of "cursor
# wars" with this one.  This will form the basis of a drawing program based
# on the GroupDraw object model.  
#
# Right now, every object has an id number associated with it, which is used
# by the applications to refer to the objects.  These id numbers map onto
# canvas id numbers at each site; hence while the same canvas ids may refer
# to different objects at different sites, the object ids will refer to the
# same object.
#

#source "conf.tcl"
initConf $argv


#
# Set up the basic window, some globals, canvas level bindings, etc.
#

gkDefaultMenu .menu
canvas .c 
pack append . .menu {top fillx} .c top
set startX 0
set startY 0
set nextID [expr {[keylget ouropts usernum] * 100}]

set cursors(0) ""
bind .c <1> { start %x %y }
bind .c <B1-Motion> {doit %x %y; movecursor %x %y }
bind .c <Motion> {movecursor %x %y }


#
# This is called on a mouse down on the canvas, signifying the start
# of a new rectangle.  Assign it an id number, create it locally and 
# tell the other users to create it
#

proc start {x y} {global startX startY others theid id global nextID
    set startX $x;    set startY $y
    set theid $nextID; incr nextID
    setrect $theid $x $y $x $y
    foreach i $others {
	RDO [keylget i filedesc] setrect $theid $x $y $x $y
    }
}


#
# This does the work of "resizing" the rectangle on the local display.
# It also creates the object if necessary.  The id() array holds the mappings
# between object ids and canvas ids.
#

proc setrect {objid x0 y0 x1 y1} { global id
    if {[info exists id($objid)] != 1} {
	keylset id($objid) canvasid [.c create rectangle $x0 $y0 $x1 $y1]
	.c bind [keylget id($objid) canvasid] <3> \
	    "global lastX lastY; set lastX %x; set lastY %y"
	.c bind [keylget id($objid) canvasid] <B3-Motion> \
	    " domove $objid %x %y; movecursor %x %y "
    } else {
	eval ".c coords [keylget id($objid) canvasid] $x0 $y0 $x1 $y1"
    }
}


#
# This is called when the mouse is moved while pressing button 1.  Essentially
# we're resizing the rectangle when we first create it.  Display locally
# and tell everyone else.
#

proc doit {x y} { global startX startY others theid
    setrect $theid $startX $startY $x $y
    foreach i $others { 
	RDO [keylget i filedesc] setrect $theid $startX $startY $x $y
    }
}

set lastX 0
set lastY 0


#
# Called when the rectangle is grabbed with the right mouse button and
# moved.  Move the rectangle on the local display, and broadcast its new
# coordinates to the other users
#

proc domove {theid x y} { global lastX lastY id others
    .c move [keylget id($theid) canvasid] [expr $x-$lastX] [expr $y-$lastY]
    foreach i $others {
	RDO [keylget i filedesc] eval \
	    "setrect $theid [.c coords [keylget id($theid) canvasid]]"
    }
    set lastX $x
    set lastY $y
}


#
# Called when the local cursor moves; tell everyone else about this
#

proc movecursor {x y} { global others ouropts
    foreach i $others {
	RDO [keylget i filedesc] moveit [keylget ouropts usernum] $x $y
    }
}


#
# Called when a remote cursor is moved.  Move the bitmap for the remote
# cursor.
#

proc moveit {who x y} { global cursors gk_library
    if {[info exists cursors($who)] != 1} {
	set cursors($who) [.c create bitmap 2c 2c -bitmap @$gk_library/bitmaps/cursor.bit]
    }
    .c coords $cursors($who) $x $y
}


#
# When a user leaves, get rid of their cursor bitmap on the canvas.
#

trace variable deleted_user w nuked_user

proc nuked_user args { global deleted_user cursors
    if {[info exists cursors([keylget deleted_user usernum])]} {
        .c delete $cursors([keylget deleted_user usernum])
        unset cursors([keylget deleted_user usernum])
    }
}

