/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define INKERNEL 1
#define _KERNEL 1

#include <sys/sysmacros.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/signal.h>
#include <sys/user.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/open.h>
#include <sys/uio.h>
#include <sys/cred.h>
#include <sys/ddi.h>
#include <sys/poll.h>
#include <sys/kmem.h>
#include <sys/filio.h>
#include <sys/termios.h>

#include "midi.h"

extern struct midi_softc midi_sc[];
extern int midi_addrs[];
extern int midi_intrs[];
extern int NumMidi;

#if defined(__STDC__) || defined(__cplusplus)
#define __P(protos)	protos
#else
#define __P(protos)	()
#endif

/* I've been burned too many times by different byte ordering */
#define OUTB(x, y) outb((x), (y))
#define INB(x) inb((x))

void midiinit __P(());
int midiopen __P((dev_t *dev, int flag, int otyp, cred_t *cred_p));
int midiclose __P((dev_t dev, int flag, int otyp, cred_t *cred_p));
int midiread __P((dev_t dev, uio_t *uio_p, cred_t *cred_p));
int midiwrite __P((dev_t dev, uio_t *uio_p, cred_t *cred_p));
int midiioctl __P((dev_t dev, int cmd, int arg, int mode, cred_t *cred_p,
    int *rval_p));
int midipoll __P((dev_t dev, int /* short */ events, int anyyet,
    short *reventsp, struct pollhead **phpp));
int midiintr __P((int ivec));

static void midi_copy_event __P((struct event *, struct event *));
static void midi_initq __P((struct event_queue *));
static int midi_deq __P((struct event_queue *, struct event **));
static int midi_peekq __P((struct event_queue *, struct event **));
static int midi_enq __P((struct event_queue *, struct event *));
static int midi_reset __P((struct midi_softc *));
static int midi_wait_rdy_rcv __P((struct midi_softc *softc));
static int midi_send_command __P((struct midi_softc *softc,
    /* u_char */ int comm));
static int midi_uart __P((struct midi_softc *));
static int midi_event2smf __P((struct midi_softc *, struct event *,
    struct stynamic *));
static int midi_next_byte __P((struct midi_softc *, struct uio *));
static int midi_uio2event __P((struct midi_softc *, struct uio *,
    struct event *));
static int midi_fix2var __P((u_long, u_char *));
static int midi_var2fix __P((u_char *, u_long *));
static u_long midi_smf2kernel_tick __P((struct midi_softc *, long));
static u_long midi_kernel2smf_tick __P((struct midi_softc *, long, long));
static void midi_timeout __P((caddr_t arg));
static void midi_add_complete_event __P((struct midi_softc *));
static void midi_schedule_timeout __P((struct midi_softc *));
static void midi_write_event __P((struct midi_softc *, struct event *));
static void midi_reset_devices __P((struct midi_softc *));
static void stynamic_add_byte __P((struct stynamic *, int /* u_char */));
static void stynamic_add_bytes __P((struct stynamic *, u_char *, int));
static u_char stynamic_get_byte __P((struct stynamic *, int));
static void stynamic_copy __P((struct stynamic *, void *, int));
static void stynamic_copy_from __P((struct stynamic *, int, void *, int));
static void stynamic_append __P((struct stynamic *, struct stynamic *));
static void stynamic_release __P((struct stynamic *));
static void stynamic_shift __P((struct stynamic *, int));
static void stynamic_init __P((struct stynamic *));
static void stynamic_print __P((struct stynamic *));
static int midi_killable __P((proc_t *owner, int pid));

void
midiinit()
{
	struct midi_softc *softc;
	int i;

	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		softc->addr = midi_addrs[i];
		softc->intr = midi_intrs[i];
		softc->status = MIDI_RD_BLOCK;
		if (!midi_reset(softc))
			if (!midi_reset(softc)) {
				printf("Couldn't reset MPU401\n");
				softc->status |= MIDI_UNAVAILABLE;
				continue;
			}
		if (!midi_uart(softc)) {
			printf("Couldn't put MPU401 into UART mode!\n");
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}

		/* allocate memory for event queues */
		if ((softc->rqueue = kmem_alloc(sizeof(struct event_queue),
		    KM_NOSLEEP)) == NULL) {
			printf("No memory for rqueue\n");
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}
		if ((softc->wqueue = kmem_alloc(sizeof(struct event_queue),
		    KM_NOSLEEP)) == NULL) {
			printf("No memory for wqueue\n");
			softc->status |= MIDI_UNAVAILABLE;
			kmem_free(softc->rqueue, sizeof(struct event_queue));
		}
		/* zero read/write queue to clear stynamic structures */
		bzero(softc->rqueue, sizeof(struct event_queue));
		bzero(softc->wqueue, sizeof(struct event_queue));
		stynamic_init(&softc->rpartial);
		stynamic_init(&softc->wpartial);
		softc->wpartialpos = 0;
	}
}

int
midiopen(dev, flag, otyp, cred_p)
	dev_t *dev;
	int flag, otyp;
	cred_t *cred_p;
{
	register struct midi_softc *softc;
	register int unit;
 
	unit = minor(*dev);
	if (unit >= NumMidi)
		return (ENXIO);
	softc = &midi_sc[unit];
	if (softc->status & MIDI_UNAVAILABLE)
		return (EIO);
	
	if (softc->status & MIDI_OPEN)
		return (EBUSY);
	else
		softc->status = MIDI_OPEN;

	/*
	 * reads will block until something appears
	 */
	softc->status |= MIDI_RD_BLOCK;

	/* initialize the queues */
	midi_initq(softc->rqueue);
	midi_initq(softc->wqueue);

	softc->partial_event.event.len = 0;
	softc->partial_event.event.datad = NULL;

	/* make sure we are in UART mode */
	if (!midi_fullreset(softc)) {
		printf("Couldn't put MPU401 into UART mode!\n");
		softc->status |= MIDI_UNAVAILABLE;
		kmem_free(softc->rqueue, sizeof(struct event_queue));
		kmem_free(softc->wqueue, sizeof(struct event_queue));
		return (EIO);
	}

	softc->pgid = u.u_procp->p_pid;
	softc->owner = u.u_procp;

	/* are we going to read, write or both? */
	if (flag & FREAD)
		softc->status |= MIDI_READING;
	else
		softc->status &= ~MIDI_READING;
	if (flag & FWRITE)
		softc->status |= MIDI_WRITING;
	else
		softc->status &= ~MIDI_WRITING;
	if (flag & FNONBLOCK)
		softc->status |= MIDI_NONBLOCK;
	else
		softc->status &= ~MIDI_NONBLOCK;

	return (0);
}

int
midiclose(dev, flag, otyp, cred_p)
	dev_t dev;
	int flag, otyp;
	cred_t *cred_p;
{
	register struct midi_softc *softc;
	register int error, unit;

	unit = minor(dev);
	softc = &midi_sc[unit];

	/*
	 * we're not going to finish closing until everything has
	 * been played.
	 */
	error = 0;
	if (softc->status & MIDI_WRITING) {
		/* same as MFLUSHQ ioctl */
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			do {
				if (error = sleep((caddr_t)&softc->status,
				    PWAIT | PCATCH))
					softc->status &= ~MIDI_FLUSH_SLEEP;
			} while (softc->status & MIDI_FLUSH_SLEEP);
		}
	}

	/* turn off any notes that might be stuck on */
	midi_reset_devices(softc);
	midi_fullreset(softc);

	softc->status &= ~MIDI_OPEN;
	return (error);
}

int
midiread(dev, uio, cred_p)
	dev_t dev;
	uio_t *uio;
	cred_t *cred_p;
{
	struct event *event;
	register struct midi_softc *softc;
	int error, in, num_to_move, unit;

	unit = minor(dev);
	softc = &midi_sc[unit];

	if (softc->rqueue->count == 0 && softc->rpartial.len == 0) {
		if (softc->status & MIDI_NONBLOCK)
			return (EWOULDBLOCK);
		else {
			softc->status |= MIDI_RD_SLEEP;
			do {
				if (error = sleep((caddr_t)softc->rqueue,
				    PWAIT | PCATCH))
					return (error);
			} while (softc->status & MIDI_RD_SLEEP);
			/* XXX maybe check for abort here */
		}
	}
	while (uio->uio_resid) {
		/*
		 * dequeue an event if partial is empty
		 */
		if (softc->rpartial.len == 0) {
			if (!midi_deq(softc->rqueue, &event)) {
				softc->status |= MIDI_RD_BLOCK;
				return (0);
			}
			midi_event2smf(softc, event, &softc->rpartial);
			styanmic_release(&event->data);
		}
		/* read out as much of rpartial as possible */
		num_to_move = MIN(softc->rpartial.len, uio->uio_resid);
		if (softc->rpartial.len <= STYNAMIC_SIZE) {
			if (uiomove(softc->rpartial.datas, num_to_move,
			    UIO_READ, uio) == -1)
			return (-1);
		} else {
			if (uiomove(softc->rpartial.datad, num_to_move,
			    UIO_READ, uio) == -1)
			return (-1);
		}
		stynamic_shift(&softc->rpartial, num_to_move);
	}
	return (0);
}

int
midiwrite(dev, uio, cred_p)
	dev_t dev;
	uio_t *uio;
	cred_t *cred_p;
{
	register struct midi_softc *softc;
	struct event event;
	int convert, error, unit, iovec, empty_queue;

	unit = minor(dev);
	softc = &midi_sc[unit];

	if (softc->wqueue->count == 0)
		empty_queue = 1;
	else
		empty_queue = 0;

	/* check to see if we'll block */
	if (softc->status & MIDI_WR_BLOCK) {
		if (softc->status & MIDI_NONBLOCK)
			return (EWOULDBLOCK);
		else {
			softc->status |= MIDI_WR_SLEEP;
			do {
				if (error = sleep((caddr_t)softc->wqueue,
				    PWAIT | PCATCH))
					return (error);
			} while (softc->status & MIDI_WR_SLEEP);
		}
	}
	/* if returns from sleep and should abort because of DRAIN */
	if (softc->status & MIDI_WR_ABORT) {
		softc->status &= ~MIDI_WR_ABORT;
		return (0);
	}

	stynamic_init(&event.data);
	while (uio->uio_resid) {
		/* block if queue is full */
		if (softc->wqueue->count == MIDI_Q_SIZE) {
			softc->status |= MIDI_WR_BLOCK;
			if (empty_queue) {
				midi_schedule_timeout(softc);
				empty_queue = 0;
			}
			if (softc->status & MIDI_NONBLOCK)
				return (0);
			else {
				softc->status |= MIDI_WR_SLEEP;
				do {
					if (error = sleep(softc->wqueue,
					    PWAIT | PCATCH))
						return (error);
				} while (softc->status & MIDI_WR_SLEEP);
			}
		}

		/*
		 * 1) get a complete event off queue
		 * 2) convert it from SMF to board format
		 * 3) deal with it
		 */
		convert = midi_uio2event(softc, uio, &event);
		switch (convert) {
		case 0:
			break;
		case -1:
			/* not a complete event - we'll get it next time */
			if (empty_queue)
				midi_schedule_timeout(softc);
			return (0);
		case -2:
			/* it was a noop event */
			break;
		default:
			return (convert);
		}
		if (convert == -2)
			continue;

		if (midi_enq(softc->wqueue, &event) == -1)
			return (EIO);
		stynamic_release(&event.data);
		/* set flag so next time we cross LOW water we will SIGIO */
		if (softc->wqueue->count >= MIDI_LOW_WATER)
			softc->status |= MIDI_SENDIO;
	}

	if (empty_queue)
		midi_schedule_timeout(softc);

	return (0);
}

/*
 * XXXINTR
 * I don't know what ivec is.  I'm really hoping it is something
 * useful like the unit number.
 * If it is the interrupt number, then we can loop through all the
 * midi_sc comparing ivec against softc->intr until we find the right
 * one.  No biggie, I just don't know which way it works.
 * For for I'll assume by default that it is the interrupt number.
 */
int
midiintr(ivec)
	int ivec;
{
	register struct midi_softc *softc;
	register u_char code;
	register struct partial_event *pe;

#if 1
	int i;

	for (i = 0; i < NumMidi && midi_sc[i].intr != ivec; i++);
	if (i == NumMidi) {
		printf("midi: ivec(%d) is not a valid MPU401 intr number\n");
		return;
	}
	softc = &midi_sc[i];
#else
	softc = &midi_sc[ivec];
#endif

	code = INB(softc->addr + MIDI_DATA);
	if (code == 0xfe && (softc->status & MIDI_NEED_ACK)) {
		softc->status &= ~MIDI_NEED_ACK;
		return (1);
	}
	/* throw away data if no one has the device open */
	if (!(softc->status & MIDI_READING))
		return (1);

	/* pass input data to out port if necessary */
	if (softc->status & MIDI_THRU) {
		if (!midi_wait_rdy_rcv(softc))
			printf("Couldn't pass data thru\n");
		else
			OUTB(softc->addr + MIDI_DATA, code);
	}
	pe = &softc->partial_event;

	/* check for realtime events */
	if ((code & 0xf8) == 0xf8) {
		switch (code) {
		case 0xfa:	/* start */
			/* reset prev_incoming */
			softc->prev_incoming = lbolt;
			break;
		case 0xff:	/* reset */
			/*
			 * I don't really want to do a full reset
			 * I'll just clear any event in progress
			 */
			stynamic_release(&pe->event);
			pe->state = START;
			break;
		case 0xf8:	/* timing clock */
		case 0xf9:	/* undefined */
		case 0xfb:	/* continue */
		case 0xfc:	/* stop */
		case 0xfd:	/* undefined */
		case 0xfe:	/* active sensing */
			break;
		}
		return (1);
	}
INTR_SWITCH:
	switch (pe->state) {
	case START:
		/* record when the time when the event started */
		pe->time = lbolt;
		pe->tempo = softc->rtempo;
		/* start collecting the input */
		stynamic_release(&pe->event);
		/* decide what state is next */
		if (!(code & 0x80)) {
			stynamic_add_byte(&pe->event, code);
			switch (pe->rs) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				/*
				 * code is the first data byte, but
				 * we still need to get the second
				 */
				pe->state = NEEDDATA2;
				break;
			case 0xc0:
			case 0xd0:
				/* code is the only data byte */
				pe->state = START;
				midi_add_complete_event(softc);
				break;
			default:
				break;
			}
		} else {
			switch (code & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA1;
				break;
			case 0xc0:
			case 0xd0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA2;
				break;
			default:
				switch (code) {
				case 0xf0: /* sysex */
					stynamic_add_byte(&pe->event, code);
					pe->state = SYSEX;
					break;
				case 0xf1: /* undefined */
				case 0xf4: /* undefined */
				case 0xf5: /* undefined */
				case 0xf6: /* tune request */
				case 0xf7: /* EOX (terminator) */
					/* ignore these */
					break;
				case 0xf2: /* song position */
					pe->state = SYSTEM2;
					break;
				case 0xf3: /* song select */
					pe->state = SYSTEM1;
					break;
				}
				break;
			}
		}
		break;
	case NEEDDATA1:
		stynamic_add_byte(&pe->event, code);
		pe->state = NEEDDATA2;
		break;
	case NEEDDATA2:
		stynamic_add_byte(&pe->event, code);
		pe->state = START;
		midi_add_complete_event(softc);
		break;
	case SYSEX:
		/* any non-data byte ends sysex */
		if (!(code & 0x80))
			stynamic_add_byte(&pe->event, code);
		else {
			stynamic_add_byte(&pe->event, 0xf7);
			midi_add_complete_event(softc);
			pe->state = START;
			if (code != 0xf7)
				goto INTR_SWITCH;
		}
		break;
	case SYSTEM1:
		/* throw away one data byte of a system message */
		pe->state = START;
		break;
	case SYSTEM2:
		/* throw away two data bytes of a system message */
		pe->state = SYSTEM1;
		break;
	}
	return (1);
}

int
midiioctl(dev, cmd, arg, mode, cred_p, rval_p)
	dev_t dev;
	int cmd, arg, mode;
	cred_t *cred_p;
	int *rval_p;
{
	struct event *event;
	struct midi_softc *softc;
	register int error, unit;
	int i, ret, val;

	unit = minor(dev);
	softc = &midi_sc[unit];

	switch (cmd) {
	case FIOASYNC:
	case MASYNC:
		/*
		 * Linux doesn't properly process the FIOASYNC
		 * ioctl entry point, thus we have two.  For
		 * compatibility purposes, I'm duplicating it
		 * here too.
		 */

		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
		if (!val)
			softc->status &= ~MIDI_ASYNC;
		else {
			struct proc *p;

			softc->status |= MIDI_ASYNC;
			if (softc->wqueue->count < MIDI_LOW_WATER
			    && softc->status & MIDI_WRITING) {
				if (softc->pgid < 0)
					signal(-softc->pgid, SIGIO);
				else if ((p = prfind(softc->pgid)) != 0)
					psignal(p, SIGIO);
			}
		}
		break;
	case TIOCSPGRP:
		/*
		 * XXXSIGNAL we need to check to be sure we can kill the
		 * the process we are being told to kill
		 */
		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
		if ((ret = midi_killable(softc->owner, val) != 0)) != 0)
			return (ret);
		softc->pgid = val;
		break;
	case TIOCGPGRP:
		copyout((caddr_t)&softc->pgid, (caddr_t)arg, sizeof(int));
		break;
	case MRESET:
		if (!midi_fullreset(softc))
			return (EIO);
		break;
	case MSDIVISION:
		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
  		/* must recalculate play remainder */
  		softc->premainder = softc->premainder * val / softc->division;
  		softc->division = val;
		break;
	case MGDIVISION:
		copyout((caddr_t)&softc->division, (caddr_t)arg, sizeof(int));
		break;
	case MDRAIN:
		/* dequeue all everything */
		while (midi_deq(softc->rqueue, &event))
			stynamic_release(&event->data);
		while (midi_deq(softc->wqueue, &event))
			stynamic_release(&event->data);
		/* remove any events already being timed */
		if (softc->timeout_id != -1)
			untimeout(softc->timeout_id);
		softc->status &= ~MIDI_WR_BLOCK;
		softc->status |= MIDI_RD_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			softc->status |= MIDI_WR_ABORT;
			wakeup((caddr_t)softc->wqueue);
		}
		midi_reset_devices(softc);
		break;
	case MFLUSH:
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			do {
				if (error = sleep((caddr_t)&softc->status,
				    PWAIT | PCATCH))
					return (error);
			} while (softc->status & MIDI_FLUSH_SLEEP);
		}
		break;
	case MGPLAYQ:
		copyout((caddr_t)&softc->wqueue->count, (caddr_t)arg,
		    sizeof(int));
		break;
	case MGRECQ:
		copyout((caddr_t)&softc->rqueue->count, (caddr_t)arg,
		    sizeof(int));
		break;
	case MGQAVAIL:
		val = MIDI_Q_SIZE - softc->wqueue->count;
		copyout((caddr_t)&val, (caddr_t)arg, sizeof(int));
		break;
	case MTHRU:
		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
		if (val)
			softc->status |= MIDI_THRU;
		else
			softc->status &= ~MIDI_THRU;
		break;
	case MRECONPLAY:
		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
		if (val)
			softc->status |= MIDI_RECONPLAY;
		else
			softc->status &= ~MIDI_RECONPLAY;
		break;
	default:
		return (ENOTTY);
	}
	return (0);
}

/*
 * XXXPOLL
 * I don't know how this is supposed to work.  The SVR4/386
 * DDI/DKI reference manual doesn't explain it well enough for me.
 */
int
midipoll(dev, events, anyyet, reventsp, phpp)
	dev_t dev;
	short events;
	int anyyet;
	short *reventsp;
	struct pollhead **phpp;
{
	register struct midi_softc *softc;
	int unit;

	unit = minor(dev);
	softc = &midi_sc[unit];

	*reventsp = 0;
	if (events & POLLIN && !(softc->status & MIDI_RD_BLOCK))
		*reventsp |= POLLIN;
	if (events & POLLOUT && !(softc->status & MIDI_WR_BLOCK))
		*reventsp |= POLLOUT;
	if (*reventsp == 0)
		if (!anyyet)
			*phpp = &softc->pollhead;
	return (0);
}

int
midi_fullreset(softc)
	struct midi_softc *softc;
{
	u_char pitch;
	struct event *event;

	/* dequeue all everything */
	while (midi_deq(softc->rqueue, &event))
		stynamic_release(&event->data);
	while (midi_deq(softc->wqueue, &event))
		stynamic_release(&event->data);
	/* remove any events already being timed */
	if (softc->timeout_id != -1)
		untimeout(softc->timeout_id);
	softc->timeout_id = -1;
	/* mark start time */
	softc->prev_incoming = lbolt;
	/* reset fractional count */
	softc->premainder = 0;
	softc->rremainder = 0;
	/* initialize some variables */
	/* this is 120 bpm when a quarter note == 1 beat */
	softc->ptempo = 500000;
	softc->rtempo = 500000;
	softc->prev_rtempo = 500000;

	/* clear noteon */
	for (pitch = 0; pitch <= 0x7f; pitch++)
		softc->noteon[pitch] = 0;
	softc->noteonrs = 0;

	/* clear running play state */
	softc->writers = 0;

	/* reset partial event stuff */
	stynamic_release(&softc->partial_event.event);
	softc->partial_event.state = START;
	stynamic_release(&softc->rpartial);
	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;

	/* defaults to 120 clocks per beat */
	softc->division = 120;
	/* reset and enter uart mode */
	return (midi_uart(softc));
}

int
midi_wait_rdy_rcv(softc)
	struct midi_softc *softc;
{
	int flag;
	register int i;

	for (i = 0; i < MIDI_TRIES; i++) {
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_RDY_RCV;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);
	return(1);
}

int
midi_send_command(softc, comm)
	struct midi_softc *softc;
	u_char comm;
{
	int flag;
	register int i, s;
	unsigned char ack;

	/* writing command and setting flag must be atomic */
	s = spltty();
	OUTB(softc->addr + MIDI_COMMAND, comm);
	softc->status |= MIDI_NEED_ACK;
	splx(s);

	/* time out after MIDI_TRIES times */
	for (i = 0; i < MIDI_TRIES; i++) {
		/* did we pick up the ack via midiintr? */
		if (!(softc->status & MIDI_NEED_ACK))
			break;
		/* can we read a data byte? */
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);

	if (softc->status & MIDI_NEED_ACK) {
		ack = INB(softc->addr + MIDI_DATA);
		if (ack != MIDI_ACK)
			return (0);
	}
	softc->status &= ~MIDI_NEED_ACK;
	return (1);
}

int
midi_reset(softc)
	struct midi_softc *softc;
{

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_RESET))
		return (0);

	return (1);
}

int
midi_uart(softc)
	struct midi_softc *softc;
{

	/*
	 * first reset.  We can't issue the UART command when already
	 * in UART mode.
	 */
	(void)midi_reset(softc);

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_UART))
		return (0);
	return (1);
}

int
midi_event2smf(softc, event, smf)
	struct midi_softc *softc;
	struct event *event;
	struct stynamic *smf;
{
	long tempo, smf_ticks;
	int tmp_len;
	u_char tmp_buf[4];

	/* convert from kernel ticks to SMF ticks */
	smf_ticks = midi_kernel2smf_tick(softc, event->tempo, event->time);
	tmp_len = midi_fix2var(smf_ticks, tmp_buf);
	stynamic_add_bytes(smf, tmp_buf, tmp_len);

	switch (event->type) {
	case NORMAL:
		stynamic_append(smf, &event->data);
		break;
	case TEMPO:
		/* this case just won't occur */
		stynamic_copy(&event->data, (void *)&tempo, sizeof(tempo));
		stynamic_add_byte(smf, 0xff);
		stynamic_add_byte(smf, 0x51);
		stynamic_add_byte(smf, 0x03);
		stynamic_add_byte(smf, (tempo & 0xff0000) >> 16);
		stynamic_add_byte(smf, (tempo & 0xff00) >> 8);
		stynamic_add_byte(smf, tempo & 0xff00);
		break;
	case SYSX:
		stynamic_add_byte(smf, 0xf0);
		/* skip over the leading 0xf0 */
		stynamic_shift(&event->data, 1);
		tmp_len = midi_fix2var(event->data.len, tmp_buf);
		stynamic_add_bytes(smf, tmp_buf, tmp_len);
		styanmic_append(smf, &event->data);
		break;
	}
	return (1);
}

int
midi_next_byte(softc, uio)
	struct midi_softc *softc;
	struct uio *uio;
{
	int byte;

	/* if we're not at the end of a partial event, read from it */
	if (softc->wpartialpos < softc->wpartial.len) {
		byte = stynamic_get_byte(&softc->wpartial,
		    softc->wpartialpos++);
		return (byte);
	} else {
		/* read from uio and copy uio onto partial event */
		if ((byte = uwritec(uio)) == -1) {
			/*
			 * reset partialpos so next time through
			 * we'll read from the partial event if
			 * it is non-zero in length
			 */
			softc->wpartialpos = 0;
			return (-1);
		}
		stynamic_add_byte(&softc->wpartial, byte);
		softc->wpartialpos = softc->wpartial.len;
		return (byte);
	}
}

int
midi_uio2event(softc, uio, event)
	struct midi_softc *softc;
	struct uio *uio;
	struct event *event;
{
	u_long smf_ticks, ulen;
	long tempo;
	int byte, error, extra_byte, i, len, num_data_bytes;
	int rs_change;
	u_char meta_type, tmp_buf[256];

	/* copy in timing portion */
	len = 0;
	do {
		if ((byte = midi_next_byte(softc, uio)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		tmp_buf[len++] = byte;
	} while (byte & 0x80);

	/* compute time in smf ticks */
	midi_var2fix(tmp_buf, &smf_ticks);
	smf_ticks += softc->write_noop_time;
	softc->write_noop_time = 0;

	/* now convert from smf to kernel */
	event->time = midi_smf2kernel_tick(softc, smf_ticks);

	/* get first byte of event data */
	if ((byte = midi_next_byte(softc, uio)) == -1) {
		stynamic_release(&event->data);
		return (-1);
	}
	switch (byte) {
	case 0xf0:
		/* basic sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		stynamic_add_byte(&event->data, 0xf0);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		break;
	case 0xf7:
		/* continued sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(smf, &ulen);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		if (error = uiomove(event->data, event->length, UIO_WRITE,
		    uio))
			return (error);
		break;
	case 0xff:
		/* meta events */
		if ((byte = midi_next_byte(softc, uio)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		meta_type = byte;
		/* get length of meta data */
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		/* read it in  - meta events are not over 256 in size */
		for (i = 0; i < ulen; i++) {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
		}
		switch (meta_type) {
		default:
			/*
			 * we'll skip these events, but we need to
			 * save the timing info
			 */
			softc->write_noop_time = smf_ticks;
			stynamic_release(&softc->wpartial);
			softc->wpartialpos = 0;
			return (-2);
		case 0x51:
			/* tempo event */
			event->type = TEMPO;
			tempo = 0;
			for (i = 0; i < 3; i++) {
				tempo = tempo << 8;
				tempo |= tmp_buf[i];
			}
			stynamic_add_bytes(&event->data, (u_char *)&tempo,
			    sizeof(tempo));
			/*
			 * change ptempo now, rtempo will change when
			 * the event's time comes up
			 */
			softc->ptempo = tempo;
			break;
		}
		break;
	default:
		if ((byte & 0xf0) == 0x80) {
			u_char rs_chan, rs_type;

			/* check note off events separately */
			tmp_buf[0] = byte;
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[1] = byte;
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[2] = byte;
			len = 3;
			/*
			 * check to see if we can collapse and use
			 * running state
			 */
			rs_type = softc->writers & 0xf0;
			rs_chan = softc->writers & 0x0f;
			/*
			 * collapse to running state if time is 0
			 * and the running state is the same
			 * or the running state is note on for the
			 * same channel and the note off velocity is
			 * zero.
			 */
			if (event->time == 0 && (softc->writers == tmp_buf[0]
			    || (tmp_buf[2] == 0 && rs_type == 0x90
			    && rs_chan == (tmp_buf[0] & 0x0f)))) {
				tmp_buf[0] = tmp_buf[1];
				tmp_buf[1] = tmp_buf[2];
				len = 2;
			} else {
				softc->writers = tmp_buf[0];
			}
		} else {
			extra_byte = 0;
			rs_change = 0;
			if ((byte & 0x80) && (byte != softc->writers)) {
				softc->writers = byte;
				rs_change = 1;
			}
			len = 0;
			if (event->time != 0 || rs_change) {
				/*
				 * stick in a mode byte if time is non-zero
				 * This is so we don't confuse hardware that
				 * is turned on while we're playing
				 * also add it if the running state changes
				 */
				tmp_buf[0] = softc->writers;
				len = 1;
			}
			if (byte & 0x80)
				extra_byte = 1;
			else {
				tmp_buf[len] = byte;
				len++;
			}

			switch (softc->writers & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				num_data_bytes = 1;
				break;
			default:
				num_data_bytes = 0;
			}
			for (i = 0; i < num_data_bytes + extra_byte; i++) {
				if ((byte = midi_next_byte(softc, uio)) == -1) {
					stynamic_release(&event->data);
					return (-1);
				}
				tmp_buf[len++] = byte;
			}
		}
		event->type = NORMAL;
		stynamic_add_bytes(&event->data, tmp_buf, len);
	}
	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;
	return (0);
}

int
midi_fix2var(fix, var)
	u_long fix;
	u_char *var;
{
	int i;
	unsigned char buf[4], *bptr;

	buf[0] = buf[1] = buf[2] = buf[3] = 0;
	bptr = buf;
	*bptr++ = fix & 0x7f;
	while ((fix >>= 7) > 0) {
		*bptr |= 0x80;
		*bptr++ += (fix & 0x7f);
        }

	i = 0;
	do {
		*var++ = *--bptr;
		i++;
	} while (bptr != buf);

	return (i);
}

int
midi_var2fix(var, fix)
	u_char *var;
	u_long *fix;
{
	int delta;

	*fix = 0;
	delta = 0;
	do {
		*fix = (*fix << 7) + (*var & 0x7f);
		delta++;
	} while (*var++ & 0x80);

	return (delta);
}

u_long
midi_smf2kernel_tick(softc, smf)
	struct midi_softc *softc;
	long smf;
{
  	long long denominator, numerator;
  	u_long kernel;
   
 	numerator = (long long)softc->ptempo * Hz * smf + softc->premainder;
  	denominator = 1000000 * (long long)softc->division;
  	kernel = numerator / denominator;
  	softc->premainder = numerator % denominator;
   	return (kernel);
}

u_long
midi_kernel2smf_tick(softc, tempo, kernel)
	struct midi_softc *softc;
	long tempo, kernel;
{
  	long long numerator, denominator;
  	u_long smf;
   
  	if (softc->prev_rtempo != tempo) {
  		/*
  		 * also update the rremainder to reflect tempo
  		 * change
  		 */
  		softc->rremainder = softc->rremainder * tempo
  		    / softc->prev_rtempo;
  		softc->prev_rtempo = tempo;
  	}
 	numerator = (long long)softc->division * 1000000 * kernel
 	    + softc->rremainder;
  	denominator = (long long)tempo * Hz;
  	smf = numerator / denominator;
  	softc->rremainder = numerator % denominator;
   	return (smf);
}

void
midi_add_complete_event(softc)
	struct midi_softc *softc;
{
	struct timeval t;
	struct timeval curr;
	struct event event;
	struct partial_event *pe;
	long ticks;

	stynamic_init(&event.data);
	pe = &softc->partial_event;
	/* calculate delta time */
	ticks = softc->prev_incoming - pe->time;
	if (ticks < 0) {
		ticks = 0;
		pe->time = softc->prev_incoming;
	}
	event.time = ticks;
	event.tempo = pe->tempo;
	switch (stynamic_get_byte(&pe->event, 0)) {
	case 0xf0:
		/* sysex */
		event.type = SYSX;
		break;
	default:
		event.type = NORMAL;
		break;
	}
	stynamic_append(&event.data, &softc->partial_event.event);
	/* enqueue new event */
	midi_enq(softc->rqueue, &event);
	stynamic_release(&event.data);
	/* readjust previous event time */
	softc->prev_incoming = pe->time;

	softc->status &= ~MIDI_RD_BLOCK;
	if (softc->status & MIDI_RD_SLEEP) {
		softc->status &= ~MIDI_RD_SLEEP;
		wakeup((caddr_t)softc->rqueue);
	}
	if (softc->status & MIDI_ASYNC) {
		struct proc *p;

		if (softc->pgid < 0)
			signal(-softc->pgid, SIGIO);
		else if ((p = prfind(softc->pgid)) != 0)
			psignal(p, SIGIO);
	}

	/* notify poll that there is data to be read */
	/*
	 * XXXPOLL - do I only want to do this is a poll is in
	 * progress and anyyet is 0?  Then I'll need to set
	 * a status flag, and how do I know when to unset it after
	 * the poll times out (ala select).
	 */
	pollwakeup(&softc->pollhead, POLLIN);

	return;
}

void
midi_schedule_timeout(softc)
	struct midi_softc *softc;
{
	struct event *event;

	if (!midi_peekq(softc->wqueue, &event))
		return;
	softc->timeout_id = timeout(midi_timeout, (caddr_t)softc,
	    event->time);

	/* clear record timer? */
	if (softc->status & MIDI_RECONPLAY) {
		softc->prev_incoming = lbolt;
		/* clear flag */
		softc->status &= ~MIDI_RECONPLAY;
	}
}

void
midi_timeout(arg)
	caddr_t arg;
{
	struct event *event;
	struct midi_softc *softc = (struct midi_softc *)arg;
	int i;

	/* send first event since we know it is ready */
	midi_deq(softc->wqueue, &event);
	midi_write_event(softc, event);
	stynamic_release(&event->data);

	/* process all events that also occur at this time (event->time == 0) */
	for(;;) {
		if (!midi_peekq(softc->wqueue, &event))
			return;
		if (event->time != 0)
			break;	
		midi_deq(softc->wqueue, NULL);
		midi_write_event(softc, event);
		stynamic_release(&event->data);
	}

	/* setup timeout for next event */
	midi_schedule_timeout(softc);
}

void
midi_write_event(softc, event)
	struct midi_softc *softc;
	struct event *event;
{
	int i, j;
	u_char bytes[4], channel, command, *dataptr;

	switch (event->type) {
	case TEMPO:
		stynamic_copy(&event->data, &softc->rtempo,
		    sizeof(softc->rtempo));
		break;
	case NORMAL:
		/*
		 * fourth byte might not be valid, but who cares,
		 * we're only reading and in the kernel.  We'll
		 * ignore it if it isn't.
		 */
		stynamic_copy(&event->data, &bytes, 4);
		if (!(bytes[0] & 0x80))
			dataptr = &bytes[0];
		else {
			softc->noteonrs = bytes[0];
			dataptr = &bytes[1];
		}
		command = softc->noteonrs & 0xf0;
		channel = softc->noteonrs & 0x0f;
		if (command == 0x90) {
			/*
			 * set of clear appropriate bit in noteon
			 * array depending on velocity value
			 */
			if (dataptr[1] != 0)
				softc->noteon[dataptr[0]] |= 1 << channel;
			else
				softc->noteon[dataptr[0]] &= ~(1 << channel);
		}
		if (command == 0x80)
			/* clear bit */
			softc->noteon[dataptr[0]] &= ~(1 << channel);
		/* FALLTHRU */
	default:
		for (i = 0; i < event->data.len; i++) {
			if (!midi_wait_rdy_rcv(softc))
				break;
			OUTB(softc->addr + MIDI_DATA,
			    stynamic_get_byte(&event->data, i));
		}
		break;
	}
	if (softc->wqueue->count < MIDI_LOW_WATER) {
		softc->status &= ~MIDI_WR_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			wakeup((caddr_t)softc->wqueue);
		}
		if (softc->status & MIDI_ASYNC &&
		    (softc->status & MIDI_SENDIO || softc->wqueue->count
		    == 0)) {
			struct proc *p;

			if (softc->pgid < 0)
				signal(-softc->pgid, SIGIO);
			else if ((p = prfind(softc->pgid)) != 0)
				psignal(p, SIGIO);
		}
		softc->status &= ~MIDI_SENDIO;
		/* notify poll that writes will succeed */
		/*
		 * XXXPOLL - do I only want to do this is a poll is in
		 * progress and anyyet is 0?  Then I'll need to set
		 * a status flag, and how do I know when to unset it after
		 * the poll times out (ala select).
		 */
		pollwakeup(&softc->pollhead, POLLOUT);
	}
	if (softc->status & MIDI_FLUSH_SLEEP && softc->wqueue->count == 0) {
		softc->status &= ~MIDI_FLUSH_SLEEP;
		wakeup((caddr_t)&softc->status);
	}
}


/*
 * try to reset the midi devices as best we can
 */
void
midi_reset_devices(softc)
	struct midi_softc *softc;
{
	int i;
	u_char channel, pitch;

	/* manual note off calls - turn off any note that is on */
	for (pitch = 0; pitch <= 0x7f; pitch++) {
		for (channel = 0; channel <= 0x0f; channel++) {
			if ((softc->noteon[pitch] >> (int)channel) & 0x01) {
				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, channel | 0x90);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, pitch);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, 0);
			}
		}
		softc->noteon[pitch] = 0;
	}
	for (channel = 0; channel <= 0x0f; channel++) {
		/*
		 * send paramter event for all notes off for redundancy
		 * some older synths don't support this
		 */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, channel | 0xb0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x7b);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* modulation controller to zero */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x01);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* reset all controllers */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x79);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* lift sustain pedal */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* center pitch wheel */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0xe0 | channel);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
	}
	softc->noteonrs = 0;
}

void
midi_initq(eq)
	struct event_queue *eq;
{

	eq->count = 0;
	eq->end = &eq->events[MIDI_Q_SIZE - 1];
	eq->head = eq->events;
	eq->tail = eq->events;
	/* zero events to clear stynamic stuctures */
	bzero(eq->events, MIDI_Q_SIZE * sizeof(struct event));
}

void
midi_copy_event(e1, e2)
	struct event *e1, *e2;
{
	short i;

	e1->time = e2->time;
	e1->type = e2->type;
	e1->tempo = e2->tempo;
	stynamic_release(&e1->data);
	stynamic_append(&e1->data, &e2->data);
}

int
midi_deq(eq, event)
	struct event_queue *eq;
	struct event **event;
{
	int s;

	s = spltty();
	if (eq->count == 0) {
		splx(s);
		return (0);
	}
	if (event == NULL)
		eq->head++;
	else
		*event = eq->head++;
	if (eq->head > eq->end)
		eq->head = eq->events;
	eq->count--;
	splx(s);
	return (1);
}

int
midi_peekq(eq, event)
	struct event_queue *eq;
	struct event **event;
{
	int s;

	s = spltty();
	if (eq->count == 0) {
		splx(s);
		return (0);
	}
	*event = eq->head;
	splx(s);
	return (1);
}

int
midi_enq(eq, event)
	struct event_queue *eq;
	struct event *event;
{
	int s;

	s = spltty();
	if (eq->count == MIDI_Q_SIZE) {
		splx(s);
		return (0);
	}
	midi_copy_event(eq->tail++, event);
	if (eq->tail > eq->end)
		eq->tail = eq->events;
	eq->count++;
	splx(s);
	return (1);
}

void
stynamic_add_byte(sty, byte)
	struct stynamic *sty;
	u_char byte;
{
	int s;
	u_char *new_ptr;

	s = spltty();
	if (sty->len < STYNAMIC_SIZE)
		sty->datas[sty->len++] = byte;
	else {
		if (sty->len < sty->allocated) {
			sty->datad[sty->len++] = byte;
		} else {
			new_ptr = kmem_alloc(sty->allocated + STYNAMIC_ALLOC,
			    KM_NOSLEEP);
			if (new_ptr == NULL) {
				printf("midi: out of memory\n");
				return;
			}
			if (sty->allocated == 0)
				bcopy(sty->datas, new_ptr, sty->len);
			else {
				bcopy(sty->datad, new_ptr, sty->len);
				kmem_free(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			sty->datad[sty->len++] = byte;
			sty->allocated += STYNAMIC_ALLOC;
		}
	}
	splx(s);
}

void
stynamic_add_bytes(sty, bytes, num)
	struct stynamic *sty;
	u_char *bytes;
	int num;
{
	int s, size_inc;
	u_char *new_ptr;

	s = spltty();
	if (sty->len + num <= STYNAMIC_SIZE) {
		bcopy(bytes, &sty->datas[sty->len], num);
		sty->len += num;
	} else {
		if (sty->len + num <= sty->allocated) {
			bcopy(bytes, &sty->datad[sty->len], num);
			sty->len += num;
		} else {
			size_inc = (num / STYNAMIC_ALLOC + 1) * STYNAMIC_ALLOC;
			new_ptr = kmem_alloc(sty->allocated + size_inc,
			    KM_NOSLEEP);
			if (sty->allocated == 0)
				bcopy(sty->datas, new_ptr, sty->len);
			else {
				bcopy(sty->datad, new_ptr, sty->len);
				kmem_free(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			bcopy(bytes, &sty->datad[sty->len], num);
			sty->allocated += size_inc;
			sty->len += num;
		}
	}
	splx(s);
}

u_char
stynamic_get_byte(sty, index)
	struct stynamic *sty;
	int index;
{
	int s;

	s = spltty();
	if (sty->len <= 4)
		return (sty->datas[index]);
	else
		return (sty->datad[index]);
	splx(s);
}

void
stynamic_copy(sty, dest, len)
	struct stynamic *sty;
	void *dest;
	int len;
{
	int s;

	s = spltty();
	if (sty->len <= 4)
		bcopy(sty->datas, dest, len);
	else
		bcopy(sty->datad, dest, len);
	splx(s);
}

void
stynamic_append(dest, src)
	struct stynamic *dest;
	struct stynamic *src;
{

	if (src->len <= 4)
		stynamic_add_bytes(dest, src->datas, src->len);
	else
		stynamic_add_bytes(dest, src->datad, src->len);
}

void
stynamic_copy_from(sty, index, dest, len)
	struct stynamic *sty;
	int index, len;
	void *dest;
{
	int s;

	s = spltty();
	if (sty->len <= 4)
		bcopy(&sty->datas[index], dest, len);
	else
		bcopy(&sty->datad[index], dest, len);
	splx(s);
}


void
stynamic_release(sty)
	struct stynamic *sty;
{
	int s;

	s = spltty();
	if (sty->len > STYNAMIC_SIZE)
		kmem_free(sty->datad, sty->allocated);
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	bzero(sty->datas, STYNAMIC_SIZE);
	splx(s);
}

void
stynamic_shift(sty, num)
	struct stynamic *sty;
	int num;
{
	int rem, s;
	u_char *ptr;

	if (sty->len <= num) {
		stynamic_release(sty);
		return;
	}
	s = spltty();
	if (sty->len > STYNAMIC_SIZE)
		ptr = &sty->datad[num];
	else
		ptr = &sty->datas[num];
	rem = sty->len - num;
	if (rem > STYNAMIC_SIZE)
		bcopy(ptr, sty->datad, rem);
	else {
		bcopy(ptr, sty->datas, rem);
		if (sty->datad != NULL) {
			kmem_free(sty->datad, sty->allocated);
			sty->datad = NULL;
			sty->allocated = 0;
		}
	}
	splx(s);
	sty->len = rem;
}

void
stynamic_init(sty)
	struct stynamic *sty;
{
	int s;

	s = spltty();
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	bzero(sty->datas, STYNAMIC_SIZE);
	splx(s);
}

void
stynamic_print(sty)
	struct stynamic *sty;
{

	printf("\t\tlen = %d\n", sty->len);
	printf("\t\tallocated = %d\n", sty->allocated);
	printf("\t\tdatad = 0x%x\n", sty->datad);
}

/*
 * XXXSIGNAL
 * I have no idea how to verify that a process is killable under
 * SVR4.  We need to check both individual processes and process
 * groups.  My drivers don't even cope with broadcast kills, so
 * don't worry about that one.
 * See the BSD and LINUX versions for ideas on how to verify this.
 */
int
midi_killable(owner, pid)
	proc_t *owner;
	int pid;
{

	if (pid == 0)
		return (0);
	if (pid == -1)
		return (EPERM);
	return (0);
}
