/*-
 tkGraphic.c --

    This module implements "graphic" widgets for the Tk toolkit.  The
    underlying vogle software graphics device associated with this
    widget is also contained in this module.

    Graphics are much like Frames except that they can have redraw
    procs bound to them built from The VOGLE graphics library. This
    widget is based on the standard Tk frame widget. As a result the
    notice below applies to this file.

*/


/*-
 * Copyright 1990 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */


#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include <tclHash.h>
#include "default.h"
#include "tk.h"


#define TKWIN_COLORMAP(tkwin) DefaultColormapOfScreen(Tk_Screen(tkwin))
#define DEF_GRAPHIC_FONT "helvetica"
#define strEQ(a,b) ((*(a)==*(b))&&!(strcmp(a,b)))

static char *
ckstrdup(src) 
    char *src;
{
    char *ret;
    int len = strlen(src);
    if ((ret = ckalloc(len+1))) {
	memcpy(ret, src, len);
	ret[len] = '\0';
    }
    return ret;
}

/*
 * A data structure of the following type is kept for each
 * graphic that currently exists for this process:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the graphic.  NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Tcl_Interp *interp;		/* Interpreter associated with
				 * widget.  Used to delete widget
				 * command.  */
    Tk_Uid screenName;		/* If this window isn't a toplevel window
				 * then this is NULL;  otherwise it gives
				 * the name of the screen on which window
				 * is displayed. */
    Tk_3DBorder border;		/* Structure used to draw 3-D border and
				 * background. */

    int borderWidth;		/* Width of 3-D border (if any). */
    int relief;			/* 3-d effect: TK_RELIEF_RAISED etc. */

    Tk_Uid geometry;		/* Geometry that user requested. */
    int flags;			/* Various flags;  see below for
				 * definitions. */

    char *self;			/* our path name */
    char *command;		/* drawing command (if any) */
    GC gc;			/* drawing GC */
    XFontStruct *fontPtr;	/* current hardware font */

    unsigned long colour;	/* current drawing color */
    Colormap colormap;		/* X's colormap. right now always the 
    				 * associated screen's default colormap */
#   define CMAPSIZE 128
#   define IS_SPECIALCOLOR(x) ((x)>CMAPSIZE)	
#   define SPECIALCOLOR_BACKGROUND CMAPSIZE+1

    XColor *carray[CMAPSIZE];   /* vogle's color index */

    Drawable drawable;		/* thing to draw into, pm if doubleBuffered
    				 * otherwise Tk_WindowId(tkwin);
    				 * member */
    Pixmap pm;			/* pixmap used if doubleBuffered */
    int pmWidth;		/* width of pm */
    int pmHeight;		/* height of pm */
    Display *pmDisplay;		/* needed for creating/Freeing pm
    			  	 * (tkwin may be null at times) */
    int doubleBuffered;		/* true if double buffering is supported.
    				 * it does _not_ indicate whether the front
    				 * or back buffer is being drawn into  */
    int squareAspect;		/* keep dimensions square ?
    				 * or scale to aspect ratio of widget
    				 */
    float xTrans, yTrans;
    float xScale,yScale,zScale;
} Graphic;

/* 
 * CGP is the pointer to the current vogle tk graphic device, it is NULL 
 * if the current vogle device is not a tk graphic 
 */
Graphic *CGP = (Graphic *)NULL; 

/*
 * Flag bits for graphics:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler
 *				has already been queued to redraw
 *				this window.
 * CLEAR_NEEDED;		Need to clear the window when redrawing.
 */

#define REDRAW_PENDING		1
#define CLEAR_NEEDED		2

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_FRAME_BG_COLOR, Tk_Offset(Graphic, border), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_FRAME_BG_MONO, Tk_Offset(Graphic, border), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_INT, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_FRAME_BORDER_WIDTH, Tk_Offset(Graphic, borderWidth), 0},
    {TK_CONFIG_STRING, "-geometry", "geometry", "Geometry",
	DEF_FRAME_GEOMETRY, Tk_Offset(Graphic, geometry), 0},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_FRAME_RELIEF, Tk_Offset(Graphic, relief), 0},

    /* * */
    {TK_CONFIG_FONT, "-font", "font", "Font",
	DEF_GRAPHIC_FONT, Tk_Offset(Graphic, fontPtr), 0},
    {TK_CONFIG_STRING, "-command", "command", "Command",
	(char *)0, Tk_Offset(Graphic, command), 0},
    {TK_CONFIG_BOOLEAN, "-doublebuffered", "doubleBuffered", "DoubleBuffered",
	"0", Tk_Offset(Graphic, doubleBuffered), 0},
    {TK_CONFIG_BOOLEAN, "-squareaspect", "squareAspect", "SquareAspect",
	"1", Tk_Offset(Graphic, squareAspect), 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/* 
 * tk and vogle have clashing BLACK and WHITE defines
 * that's why this stuff is where it is 
 */
#undef BLACK
#undef WHITE

/* define VOGLE if this driver is reallt for the VOGLE library */

#define VOGLE 1

#ifdef VOGLE
#include <vogle.h>
#include <vogle_inlines.h>
#else
#include <vogl.h>
#include <vogl_inlines.h>
#endif

/*
 * Forward declarations for procedures defined later in this file:
 */
/*
 * -- graphic widget functions
 */
static int	ColorInitForGraphic _ANSI_ARGS_((Graphic *graphicPtr)); 
static int	ResetFontForGraphic _ANSI_ARGS_((Graphic *graphicPtr)); 
static int	ConfigureGraphic _ANSI_ARGS_((Tcl_Interp *interp, Graphic *graphicPtr,
		    int argc, char **argv, int flags));
static void	DestroyGraphic _ANSI_ARGS_((ClientData clientData));
static int	DisplayGraphic _ANSI_ARGS_((ClientData clientData));
static void	GraphicEventProc _ANSI_ARGS_((ClientData clientData,
		    XEvent *eventPtr));
static int	GraphicWidgetCmd _ANSI_ARGS_((ClientData clientData,
		    Tcl_Interp *interp, int argc, char **argv));
static void	MapGraphic _ANSI_ARGS_((ClientData clientData));
static void	InitializeGraphic _ANSI_ARGS_(());
static int	Tk_MakeGraphicExist _ANSI_ARGS_((Graphic *graphicPtr));
int 		GraphicCmd _ANSI_ARGS_((ClientData clientData, 
			Tcl_Interp *interp, int argc, char **argv));

/*
 * -- TG_xxx() functions, (driver layer of vogle)
 */
static int      TG_Kill_Back_Buffer _ANSI_ARGS_((Graphic *graphicPtr));
int		TG_init _ANSI_ARGS_(());


Tcl_HashTable TG_Registry; /* mapping of vogle device name to Tk widget */

/*
 *
 */
static int initialized = 0;
static Tk_Uid GraphicConst_background;

static void
InitializeGraphic()
{
    char buf[256];
    if (!initialized) {
	initialized = 1;
	Tcl_InitHashTable(&TG_Registry, TCL_STRING_KEYS);
	/*
	 * Construct some Tk_Uid constants that may be used frequently
	 */
	GraphicConst_background = Tk_GetUid("background");
    }
    if (!vgetdev(buf)) {
	voutput("/dev/null");
	vinit("postscript"); /* init vogle package to the postscript device */
    }
}

/*
 * Make sure the widgets drawable is of the right configuration for 
 * graphicPtr. This should always be called before any drawing takes
 */
static int
Tk_MakeGraphicExist(graphicPtr)
    Graphic *graphicPtr;
{
    /* 
     * application could potentially start drawing stuff before window is
     * mapped. Not too likely but it could happen 
     */
    if (Tk_WindowId(graphicPtr->tkwin) == None) {
	Tk_MakeWindowExist(graphicPtr->tkwin);
    }

    if (!graphicPtr->doubleBuffered) {
	graphicPtr->drawable = Tk_WindowId(graphicPtr->tkwin);
    } else {
	int width, height;
	width = Tk_Width(graphicPtr->tkwin);
	height = Tk_Height(graphicPtr->tkwin);
	/* 
	 * TO BE DONE ??
	 * we may want to use current one if it's bigger but near 
	 * the desired size.
	 */
	if ((graphicPtr->pmWidth != width)||(graphicPtr->pmHeight != height)) {
	    if (graphicPtr->pm != None) {
		XFreePixmap(graphicPtr->pmDisplay, graphicPtr->pm);
	    }
	    graphicPtr->pmWidth = width;
	    graphicPtr->pmHeight = height;
	    graphicPtr->drawable = graphicPtr->pm = 
		XCreatePixmap(graphicPtr->pmDisplay, 
		    Tk_WindowId(graphicPtr->tkwin),
		    width, 
		    height,
		    DefaultDepthOfScreen(Tk_Screen(graphicPtr->tkwin)));
	}
    }
    return 0;
}

static int
TG_Kill_Back_Buffer(graphicPtr)
    Graphic *graphicPtr;
{
    graphicPtr->doubleBuffered = 0;
    /*
     * Free the back buffer
     */
    if (graphicPtr->pm != None) {
	XFreePixmap(graphicPtr->pmDisplay, graphicPtr->pm);
	graphicPtr->pm = None;
    }
    /*
     * set the widget drawable to be the window
     */
    graphicPtr->pmWidth = graphicPtr->pmHeight = -1;
    if (graphicPtr->tkwin && (Tk_WindowId(graphicPtr->tkwin) != NULL)) { 
	graphicPtr->drawable = Tk_WindowId(graphicPtr->tkwin);
    } else {
	graphicPtr->drawable =  None;
    }
    return 0;
}


/*
 *--------------------------------------------------------------
 *
 * GraphicCmd --
 *
 *	This procedure is invoked to process the "graphic"
 *	Tcl command.  See the user documentation for details on 
 *	what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
GraphicCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Tk_Window tkwin = (Tk_Window) clientData;
    Tk_Window new;
    register Graphic *graphicPtr;
    Tk_Uid screenUid;
    char *class, *screen;
    int src, dst;
    int isnewentry;
    Tcl_HashEntry *entryPtr;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " pathName [options]\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (!initialized) {
	InitializeGraphic();
    }

    /*
     * The code below is a special hack that extracts a few key
     * options from the argument list now, rather than letting
     * ConfigureGraphic do it.  This is necessary because we have
     * to know the window's screen (if it's top-level) and its
     * class before creating the window.
     */

    screen = NULL;
    class = (argv[0][0] == 't') ? "Toplevel" : "Graphic";
    for (src = 2, dst = 2; src < argc;  src += 2) {
	char c;

	c = argv[src][1];
	if ((c == 'c')
		&& (strncmp(argv[src], "-class", strlen(argv[src])) == 0)) {
	    class = argv[src+1];
	} else if ((argv[0][0] == 't') && (c == 's')
		&& (strncmp(argv[src], "-screen", strlen(argv[src])) == 0)) {
	    screen = argv[src+1];
	} else {
	    argv[dst] = argv[src];
	    argv[dst+1] = argv[src+1];
	    dst += 2;
	}
    }
    argc -= src-dst;

    /*
     * Provide a default screen for top-level windows (same as screen
     * of parent window).
     */

    if ((argv[0][0] == 't') && (screen == NULL)) {
	screen = "";
    }
    if (screen != NULL) {
	screenUid = Tk_GetUid(screen);
    } else {
	screenUid = NULL;
    }

    /* enter the  widget in the registry used by vogle to access widgets */

    entryPtr = Tcl_CreateHashEntry(&TG_Registry, argv[1], &isnewentry);
    if (!isnewentry) {
	/* 
	 * yow ! Already an entry ?
	 *
	 * This could potentially happen if there is more than one
	 * interpreter running in this process 
	 */
	Tcl_AppendResult(interp,
	    "graphic '",
	    argv[1], 
	    "' already exists in the TG_Registry",
	    (char *)NULL);
	return TCL_ERROR;
    }


    /*
     * Create the window.
     */

    new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], screenUid);
    if (new == NULL) {
	return TCL_ERROR;
    }
    Tk_SetClass(new, class);
    graphicPtr = (Graphic *) ckalloc(sizeof(Graphic));
    if (graphicPtr == NULL) {
	Tcl_AppendResult(interp, argv[1], " : ckalloc error", (char *)NULL);
	Tk_DestroyWindow(new);
	return TCL_ERROR;
    }
    graphicPtr->tkwin = new;
    graphicPtr->interp = interp;
    graphicPtr->screenName = screenUid;
    graphicPtr->border = NULL;
    graphicPtr->geometry = NULL;
    graphicPtr->flags = 0;
    graphicPtr->self = ckstrdup(argv[1]);
    graphicPtr->command = (char *) NULL;
    graphicPtr->pmDisplay = Tk_Display(tkwin);
    graphicPtr->pm = None;
    graphicPtr->drawable = None;
    graphicPtr->doubleBuffered = 0;
    graphicPtr->squareAspect= 0;
    graphicPtr->xTrans = graphicPtr->yTrans = 0.0;
    graphicPtr->xScale = graphicPtr->yScale = graphicPtr->zScale = 0.0;
    graphicPtr->pmWidth = -100;
    graphicPtr->pmHeight = -100;
    {
	XGCValues values;
	graphicPtr->gc = XCreateGC(Tk_Display(tkwin), 
				DefaultRootWindow(Tk_Display(tkwin)),
				(unsigned long)0,
				&values);
    }
    graphicPtr->fontPtr = (XFontStruct *)NULL;
    graphicPtr->colormap = TKWIN_COLORMAP(graphicPtr->tkwin);


    Tk_CreateEventHandler(graphicPtr->tkwin, 
	    ExposureMask|StructureNotifyMask,
	    GraphicEventProc, (ClientData) graphicPtr);
    Tcl_CreateCommand(interp, Tk_PathName(graphicPtr->tkwin),
	    GraphicWidgetCmd, (ClientData) graphicPtr, (void (*)()) NULL);
    Tcl_SetHashValue(entryPtr, (ClientData) graphicPtr);
    ColorInitForGraphic(graphicPtr);
    if (ConfigureGraphic(interp, graphicPtr, argc-2, argv+2, 0) != TCL_OK) {
	Tk_DestroyWindow(graphicPtr->tkwin);
	return TCL_ERROR;
    }
    if (screenUid != NULL) {
	Tk_DoWhenIdle(MapGraphic, (ClientData) graphicPtr);
    }
    Tcl_SetResult(interp, Tk_PathName(graphicPtr->tkwin), TCL_STATIC);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GraphicWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a graphic widget.  See the user
 *	documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
GraphicWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Information about graphic widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    register Graphic *graphicPtr = (Graphic *) clientData;
    int result = TCL_OK;
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option [arg arg ...]\"", (char *) NULL);
	return TCL_ERROR;
    }

    Tk_Preserve((ClientData) graphicPtr);
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)) {
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, graphicPtr->tkwin, configSpecs,
		    (char *) graphicPtr, (char *) NULL, 0);
	} else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, graphicPtr->tkwin, configSpecs,
		    (char *) graphicPtr, argv[2], 0);
	} else {
	    result = ConfigureGraphic(interp, graphicPtr, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
	if (!graphicPtr->doubleBuffered) {
	    TG_Kill_Back_Buffer(graphicPtr);
	}
    } else if ((c == 'p') && (strncmp(argv[1], "paint", length) == 0)) {
	if (argc == 2) {
	    result = DisplayGraphic(graphicPtr);
	} else if (argc == 3) {
	    char *savedCommand = graphicPtr->command;
	    graphicPtr->command = argv[2];
	    result = DisplayGraphic(graphicPtr);
	    graphicPtr->command = savedCommand;
	}
    } else {
	Tcl_AppendResult(interp,
	    " ", argv[0], ": bad option for ", 
	    "\"", argv[1], "\":  must be config or paint", 
	    (char *) NULL);
	result = TCL_ERROR;
    }
    Tk_Release((ClientData) graphicPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyGraphic --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 *	to clean up the internal structure of a graphic at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the graphic is Freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyGraphic(clientData)
    ClientData clientData;	/* Info about graphic widget. */
{
    register Graphic *graphicPtr = (Graphic *) clientData;
    int isnewentry;
    Tcl_HashEntry *entryPtr;
    int i;

    entryPtr = Tcl_FindHashEntry(&TG_Registry, graphicPtr->self);
    if (!entryPtr) {
	/*
	 * this should never happen
	 */
	fprintf(stderr, "DestroyGraphic: %s not registered!", graphicPtr->self);
	abort();
    } else { 
	Tcl_DeleteHashEntry(entryPtr);
    }

    /*
     * Free up the dynamically Allocated resources for this graphic
     */

    /* FREE_COLOR_MAP(graphicPtr->colormap) */ /* right now this is static */

    if (graphicPtr->carray) {
	for (i = 0; i < CMAPSIZE; i++) {
	    if (graphicPtr->carray[i]) {
		Tk_FreeColor(graphicPtr->carray[i]);
	    }
	}
    }

    if (graphicPtr->self != NULL) {
	ckfree(graphicPtr->self);
    }
    if (graphicPtr->command != NULL) {
	ckfree(graphicPtr->command);
    }
    if (graphicPtr->gc != NULL) {
	XFreeGC(graphicPtr->pmDisplay, graphicPtr->gc);
    }
    if (graphicPtr->fontPtr != NULL) {
	Tk_FreeFontStruct(graphicPtr->fontPtr);
    }
    if (graphicPtr->pm != None) {
	XFreePixmap(graphicPtr->pmDisplay, graphicPtr->pm);
    }
    if (graphicPtr->border != NULL) {
	Tk_Free3DBorder(graphicPtr->border);
    }
    if (graphicPtr->geometry != NULL) {
	ckfree(graphicPtr->geometry);
    }
    ckfree((char *) graphicPtr);
}

/*
 * ResetFontForGraphic(graphicPtr) 
 * Reset the font related information for the graphic
 *
 * Usually called once a Tk_ConfigureWidget of a graphic has taken place.
 */
static int
ResetFontForGraphic(graphicPtr) 
    Graphic *graphicPtr;
{
    if (graphicPtr->fontPtr) {
	XGCValues gcValues;
	gcValues.font = graphicPtr->fontPtr->fid;
	XChangeGC(graphicPtr->pmDisplay, graphicPtr->gc, GCFont, &gcValues);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ConfigureGraphic --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a graphic widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for graphicPtr;  old resources get Freed, if there
 *	were any.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureGraphic(interp, graphicPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    register Graphic *graphicPtr;	/* Information about widget;  may or may
				 * not already have values for some fields. */
    int argc;			/* Number of valid entries in argv. */
    char **argv;		/* Arguments. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    if (Tk_ConfigureWidget(interp, graphicPtr->tkwin, configSpecs,
	    argc, argv, (char *) graphicPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    Tk_SetBackgroundFromBorder(graphicPtr->tkwin, graphicPtr->border);
    ResetFontForGraphic(graphicPtr);

    if (graphicPtr->geometry != NULL) {
	int height, width;

	if (sscanf(graphicPtr->geometry, "%dx%d", &width, &height) != 2) {
	    Tcl_AppendResult(interp, "bad geometry \"", graphicPtr->geometry,
		    "\": expected widthxheight", (char *) NULL);
	    return TCL_ERROR;
	}
	Tk_GeometryRequest(graphicPtr->tkwin, width, height);
    }
    Tk_SetInternalBorder(graphicPtr->tkwin, graphicPtr->borderWidth);

    if (Tk_IsMapped(graphicPtr->tkwin)
	    && !(graphicPtr->flags & REDRAW_PENDING)) {
	Tk_DoWhenIdle(DisplayGraphic, (ClientData) graphicPtr);
	graphicPtr->flags |= REDRAW_PENDING|CLEAR_NEEDED;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayGraphic --
 *
 *	This procedure is invoked to display a graphic widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to display the graphic in its
 *	current mode.
 *
 *----------------------------------------------------------------------
 */

static int
DisplayGraphic(clientData, cmds)
    ClientData clientData;	/* Information about widget. */
{
    register Graphic *graphicPtr = (Graphic *) clientData;
    register Tk_Window tkwin = graphicPtr->tkwin;
    int result;

    graphicPtr->flags &= ~(REDRAW_PENDING|CLEAR_NEEDED);
    if ((tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	return TCL_OK;
    } else {
	Tk_MakeGraphicExist(graphicPtr);
    }

    /* 
     * If clearing of the window is needed then do so.
     * Clearing is never needed if we are doubleBuffer-ed 
     * since the whole window is going to be copied over 
     * anyway.
     */
    if ((graphicPtr->flags & CLEAR_NEEDED) && !graphicPtr->doubleBuffered) {
	XClearWindow(Tk_Display(tkwin), Tk_WindowId(tkwin));
    }

    /*
     * do user defined drawing ( if any )
     * a matrix and viewport is automatically pushed and popped from the stack
     */
    if (graphicPtr->command) {
	/*
	 * set the current VOGLE device to ourself
	 */
	/* extern */ int tclVoglePushMatrixCount;
	tclVoglePushMatrixCount = 0;

	vinit(graphicPtr->self);

	if ((graphicPtr->xTrans != .0) || (graphicPtr->yTrans != .0)) {
	    translate(graphicPtr->xTrans, graphicPtr->yTrans, .0);
	}
	if ((graphicPtr->xScale != .0) || (graphicPtr->yScale != .0) ||
	    (graphicPtr->zScale != .0)) {
	    scale(graphicPtr->xScale, graphicPtr->yScale, graphicPtr->zScale);
	}

	/*
	 * do the desired commands
	 */
	result = Tcl_Eval(graphicPtr->interp, graphicPtr->command, 0, (char **)0);
	switch(result) {
	case TCL_OK:
	    break;
	case TCL_BREAK:
	case TCL_CONTINUE:
	    result = TCL_OK;
	default:
	    result = TCL_ERROR;
	    break;
	}

	if (tclVoglePushMatrixCount) {
	    if (1) {
		/* XXX */
		/*- 
		 * -- make a debug option to turn this on/off ??
		 * -- does not catch unpopped matrices in objects
		 * -- should also check attribute stack too ??
		 */
		Tcl_AddErrorInfo(graphicPtr->interp,
		    "not all matrices pushed in -command were popped");
		    result = TCL_ERROR;
	    }
	    while(tclVoglePushMatrixCount--) {
		popmatrix();
	    }
	}

	if (tkwin == NULL) {
	    /* 
	     * The widget is being destroyed out from underneath us 
	     * in the -command. See GraphicEventProc() for more explantions.
	     */
	    Tcl_AddErrorInfo(graphicPtr->interp,
		    "graphic widget destroyed while drawing (via update?)");
	    result = TCL_ERROR;
	}

	if (result != TCL_OK) {
	    TkBindError(graphicPtr->interp);
	    return(result);
	}

	/*
	 * draw the border
	 */
	if ((graphicPtr->border != NULL) && 
	    (graphicPtr->relief != TK_RELIEF_FLAT)) {
	    Tk_Draw3DRectangle(
		Tk_Display(tkwin), graphicPtr->drawable,
		graphicPtr->border, 0, 0, Tk_Width(tkwin), Tk_Height(tkwin),
		graphicPtr->borderWidth, graphicPtr->relief
	    );
	}
    } else {
	/*
	 * no user drawing specified, paint it like a regular frame
	 */
	if ((graphicPtr->border != NULL) && 
	    (graphicPtr->relief != TK_RELIEF_FLAT)) {
	    Tk_Fill3DRectangle(
		Tk_Display(tkwin), graphicPtr->drawable,
		graphicPtr->border, 0, 0, Tk_Width(tkwin), Tk_Height(tkwin),
		graphicPtr->borderWidth, graphicPtr->relief
	    );
	}
	result = TCL_OK;
    }


    /*
     * if doubleBuffered, flush the buffer
     */
    if (graphicPtr->doubleBuffered) {
	XCopyArea(Tk_Display(tkwin), 
	    graphicPtr->pm,
	    Tk_WindowId(graphicPtr->tkwin),
	    graphicPtr->gc,
	    0, 0,
	    graphicPtr->pmWidth,
	    graphicPtr->pmHeight,
	    0, 0);
    }

    if (result != TCL_OK) {
	fprintf(stderr, "In %s -command, line %d: %s\n", 
	    graphicPtr->self,
	    graphicPtr->interp->errorLine,
	    graphicPtr->interp->result ?
		graphicPtr->interp->result : "unknown error"
	);
    }
    XFlush(Tk_Display(tkwin));
    return(result);
}


/*
 *--------------------------------------------------------------
 *
 * GraphicEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher on
 *	structure changes to a graphic.  For graphics with 3D
 *	borders, this procedure is also invoked for exposures.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
GraphicEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    register XEvent *eventPtr;	/* Information about event. */
{
    register Graphic *graphicPtr = (Graphic *) clientData;

    if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
	/*
	 * IF 
	 *     there is something to paint &&
	 *     there is a window to paint in &&
	 *     we have not already marked this window as damaged
	 * THEN
	 *     -- make an when-idle callback to repaint this widget.
	 *     -- mark the window as damaged
	 */
	if ((graphicPtr->command || (graphicPtr->relief != TK_RELIEF_FLAT)) &&
	    (graphicPtr->tkwin != NULL) && 
	   !(graphicPtr->flags & REDRAW_PENDING)) {
		Tk_DoWhenIdle(DisplayGraphic, (ClientData) graphicPtr);
		graphicPtr->flags |= REDRAW_PENDING;
	}
    } else if (eventPtr->type == DestroyNotify) {
	/*
	 * Make real sure that the current vogle device is not the widget
	 * we are about to destroy. Yes, this could be the case ! 
	 *
	 * E.g. , suppose this widget has an infinitely looping -command
	 * which periodically calls 'update'. 
	 *
	 * If at some point we are told to destroy by some external event
	 * processed by update (like the window mgr sending us a WM_DELETE),
	 * once the update returns the -command would continue one 
	 * and start drawing on a widget which is at least partially 
	 * destroyed.
	 *
	 * We could sprinkle the vogle driver calls with checks all over 
	 * the place to make sure it's device is still valid. Instead
	 * we do a bit of a kludge here and just switch the current vogle
	 * device to "postscript" with an output stream of /dev/null. 
	 * if the current vogle device is this widget. a better solution
	 * would probably be switch over to a "null" vogle device.
	 */
	char *currentDev = CurrentVogleDevice();
	if (currentDev && !strcmp(graphicPtr->self, currentDev)) {
	    vexit();
	    voutput("/dev/null");
	    vinit("postscript");
	}
	Tcl_DeleteCommand(graphicPtr->interp, Tk_PathName(graphicPtr->tkwin));
	graphicPtr->tkwin = NULL;
	if (graphicPtr->flags & REDRAW_PENDING) {
	    Tk_CancelIdleCall(DisplayGraphic, (ClientData) graphicPtr);
	}
	Tk_EventuallyFree((ClientData) graphicPtr, DestroyGraphic);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MapGraphic --
 *
 *	This procedure is invoked as a when-idle handler to map a
 *	newly-created graphic.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The graphic given by the clientData argument is mapped.
 *
 *----------------------------------------------------------------------
 */

static void
MapGraphic(clientData)
    ClientData clientData;		/* Pointer to graphic structure. */
{
    Graphic *graphicPtr = (Graphic *) clientData;

    if (graphicPtr->tkwin == NULL) {
	return;
    }

    /*
     * Wait for all other background events to be processed before
     * mapping window.  This ensures that the window's correct geometry
     * will have been determined before it is first mapped, so that the
     * window manager doesn't get a false idea of its desired geometry.
     */

    do {
	if (Tk_DoOneEvent(1) == 0) {
	    break;
	}

	/*
	 * After each event, make sure that the window still exists,
	 * and quit if the window has been destroyed.
	 */

	if (graphicPtr->tkwin == NULL) {
	    return;
	}
    } while (1);
    Tk_MapWindow(graphicPtr->tkwin);
}


/*-

    TkGraphic_AllocColor --

    Colors are allocated as they are needed and then kept around.

    This is required to preserve the vogle attribute stack pushing and
    popping functionality.

    The Color Look up method relies on the fact that Tk_GetColor caches
    colors.

    - Get the color via the TK approved way. If we later discover the
    color already exists we can just destroy what we have allocated
    since we are really just incrementing/decrementing reference
    counters in this case.

    - Check our new color against the user entries already present in
    the widgets color list to see if the new color matches one of them

    - If there is a match then Free the color we just allocated and call
    the the vogle color function with the index of the matched color.

    - If there is no match add the new color to an open slot in the
    user settable area of the color list. if there is no user slots
    left, then free the newcolor and return a TCL_ERROR.

-*/

int
TkGraphic_AllocColor(graphicPtr, colorname)
    register Graphic *graphicPtr; /* this will get called a lot */
    char *colorname;
{
    register int i;
    int open_slot = -1;
    XColor *newcolor;
    Tk_Uid colorId = Tk_GetUid(colorname);
    if (colorId == GraphicConst_background) {
	return SPECIALCOLOR_BACKGROUND;
    }

    /*
	If the supplied graphicPtr is NULL use the whatever the current
	graphics device is.
    */
    if (!graphicPtr && !(graphicPtr = CGP)) {
	return -2;
    }
    if (!(newcolor = Tk_GetColor(graphicPtr->interp, graphicPtr->tkwin, 
			graphicPtr->colormap, colorId))) {
	return -1;
    }

    /*
     * XXX - CMAPSIZE is not all that huge, but still it might be better
     * XXX - to use something other than scanning through the array
     * XXX - for color lookup.
     */
    for (i=0; i<CMAPSIZE; i++) {
	if (graphicPtr->carray[i] == newcolor) {
	    /* Color is already present */
	    Tk_FreeColor(newcolor);
	    return i;
	}
	if ((open_slot<0) && !(graphicPtr->carray[i])) {
	    /* An open slot. Record this. (we may need it later) */
	    open_slot = i;
	}
    }

    /* 
	A new color , find a slot for it (if we can)
    */
    if (open_slot < 0) {
	Tk_FreeColor(newcolor);
	return -3;
    } else {
	graphicPtr->carray[open_slot] = newcolor;
	return open_slot;
    }
}


/*
 * ColorInitForGraphic --
 * 	Initialize the colors for this graphic
 */
static int
ColorInitForGraphic(graphicPtr)
    Graphic *graphicPtr;
{
#define set_color(G, I, CNAME) \
    G->carray[I] = Tk_GetColor(G->interp, G->tkwin, G->colormap, Tk_GetUid(CNAME))
#define null_color(G, I) \
    G->carray[I] = (XColor *)0

    int i=0;
    int depth = DefaultDepth(graphicPtr->pmDisplay, 
    			Tk_ScreenNumber(graphicPtr->tkwin));

    switch (depth) {
    case 1:
    case 2:
    case 3:
	set_color(graphicPtr, i++, "black");
	for (; i < 8; i++) {
	    /* vogle c functions may rely on these colors other wise I
	    would not even bother allocating them statically */
	    set_color(graphicPtr, i, "white");
	}
	for (; i < CMAPSIZE; i++) {
	    null_color(graphicPtr, i);
	}
	break;

    case 4:
    case 5:
    case 6:
    case 7:
	/* not sure what to do here  it would be nice to define the
	standard vogle colors but that may be using up too many color
	slots for peoples liking. (Tk is kind of a color hog) */

	/* for now just set up default indexes like B&W */

	set_color(graphicPtr, i++, "black");
	for (; i < 8; i++) {
	    /* vogle c functions may rely on these colors other wise I
	    would not even bother allocating them statically */
	    set_color(graphicPtr, i, "white");
	}
	for (; i < CMAPSIZE; i++) {
	    null_color(graphicPtr, i);
	}
	break;

    default:
	assert(depth > 0);

	/* vogle c functions may rely on these colors other wise I
	would not even bother allocating them statically */

	set_color(graphicPtr, i++, "black");
	set_color(graphicPtr, i++, "red");
	set_color(graphicPtr, i++, "green");
	set_color(graphicPtr, i++, "yellow");
	set_color(graphicPtr, i++, "blue");
	set_color(graphicPtr, i++, "magenta");
	set_color(graphicPtr, i++, "cyan");
	set_color(graphicPtr, i++, "white");
	for (; i < CMAPSIZE; i++) {
	    null_color(graphicPtr, i);
	}
	break;
    }
    return 0;

#undef set_color
#undef null_color
}

int
FlushGraphic(name)
char *name;
{
    Graphic *graphicPtr;
    Tcl_HashEntry *entryPtr;

    if (name && (entryPtr = Tcl_FindHashEntry(&TG_Registry, name))) {
	graphicPtr = (Graphic *)Tcl_GetHashValue(entryPtr);
	XFlush(Tk_Display(graphicPtr->tkwin));
    }
    return TCL_OK;
}



/*
 VOGLE driver for Tk X toolkit.

 Issue --
    how to integrate the tk and Vogle package without changing either
    package radically.

 Approach --

    From vogle's point of view each tk widget is a different vogle device.

    The vogle device name is the same as the Tk widget name.

    When a widget is created, most of the vogle device initialization
    is done at widget create time. The vogle initialization function for
    a Tk vogle device really does not do any initialization at all, it's
    present mainly to allow device switching to occur via vnewdev().

    The same situation is present for device shutdown. most of the work
    is just built in to the widget Tk code. The vogle Tk driver code
    just saves off whatever state it needs from the current vogle device 
    to the associated Tk widget.

    This approach allows the integration of the VOGLE graphics
    capabilites into a Tk widget with VOGLE having to have no code
    added to it other than the conventional vogel driver interface.

*/


/* XXX */

#ifdef VOGLE

static char *PackageName = "vogle";
#define LARGEFONT	"-adobe-courier-medium-r-normal--24-240-75-75-m-150-iso8859-1"
#define SMALLFONT	"-adobe-courier-medium-r-normal--10-100-75-75-m-60-iso8859-1"

#else

static char *PackageName = "vogl";
#define LARGEFONT	"9x15bold"
#define SMALLFONT	"6x13bold"

#endif


#define TG_reset_hardware_font(G) \
    if (G->fontPtr) { \
	vdevice.hheight = G->fontPtr->max_bounds.ascent + \
			  G->fontPtr->max_bounds.descent; \
	vdevice.hwidth =  G->fontPtr->max_bounds.width; \
    }

#define MIN(x,y)	((x) < (y) ? (x) : (y))
#define	EV_MASK		KeyPressMask|ButtonReleaseMask|ExposureMask|ButtonPressMask


#if SYNCRONIZE_TG
#define FLUSH(D) if (vdevice.sync) XFlush(D)
#define SYNC(D,A) XSync(D,A)
/* do the normal vogle flushing */
#else
/* totally turn off flushing */
#define FLUSH(D)
#define SYNC(D,A)
#endif

/*
 * no initialization gets done here really.
 * it's more for switching between devices
 *
 * it's used mainly for when switching devices.
 */
TG_init()
{
    /*
     * Before anything else, we've got to get the handle to the 
     * tkGraphic widget which is where all the real device particulars
     * are
     */
    {
	Tcl_HashEntry *entryPtr;

	entryPtr = Tcl_FindHashEntry(&TG_Registry, vdevice.dev.devname);
	if (!entryPtr) {
	    fprintf(stderr, 
		"tk graphic %s does not exist", 
		vdevice.dev.devname);
	    return 0;
	}
	CGP = (Graphic *)Tcl_GetHashValue(entryPtr);
    }

    vdevice.depth = DefaultDepth(CGP->pmDisplay, Tk_ScreenNumber(CGP->tkwin));
    TG_reset_hardware_font(CGP);
    /*
     * reset the default color
     */
    {
	char *colorname = Tk_NameOf3DBorder(CGP->border);
	int c = TkGraphic_AllocColor(CGP, colorname);
	if (c > -1) {
	    /* 
	     * Cannot use color. It will complain that we are not
	     * initialized all the way yet so do it manually.  
	     */
	    vdevice.attr->a.color = c;
	    TG_color(c);
	} else {
	    perror("TG: error allocating default color");
	}
    }
    Tk_MakeGraphicExist(CGP);

    /* 
     * set device size
     */
    {
	int w,h;
	w = Tk_Width(CGP->tkwin);
	h = Tk_Height(CGP->tkwin);

	if (CGP->squareAspect) {
	    vdevice.sizeX = vdevice.sizeY = MIN(h, w) - 1;
	} else {
	    vdevice.sizeX = w-1;
	    vdevice.sizeY = h-1;
	}
	vdevice.sizeSx = w - 1;
	vdevice.sizeSy = h - 1;
    }
    return(1);
}

/*
 * TG_exit
 *
 * does nothing. the Tk widget handles cleaning up everything
 *
 */
TG_exit()
{
    CGP = (Graphic *)NULL;
    return(1);
}

/*
 * TG_draw
 *
 *	draws a line from the current graphics position to (x, y).
 *
 * Note: (0, 0) is defined as the top left of the window in X (easy
 * to forget).
 */
TG_draw(x, y)
    int	x, y;
{
    XDrawLine(CGP->pmDisplay,
	    CGP->drawable,
	    CGP->gc,
	    vdevice.cpVx, vdevice.sizeSy - vdevice.cpVy,
	    x, vdevice.sizeSy - y
    );

    FLUSH(CGP->pmDisplay);
}

/*
 * TG_getkey
 *
 *	grab a character from the keyboard - blocks until one is there.
 */
int
TG_getkey()
{
    perror("TG_getkey not implemented, use tk for input"); 
    return 0;
}

/*
 * TG_checkkey
 *
 *	Check if there has been a keyboard key pressed.
 *	and return it if there is.
 */
int
TG_checkkey()
{
    perror("TG_checkkey not implemented, use tk for input"); 
    return 0;
}

/*
 * TG_locator
 *
 *	return the window location of the cursor, plus which mouse button,
 * if any, is been pressed.
 */
int
TG_locator(wx, wy)
    int	*wx, *wy;
{
    perror("TG_locator not implemented, use tk for input"); 
    return 0;
}

#ifdef VOGLE
/*
 * TG_clear
 *
 * Clear the screen (or current buffer )to current colour
 */
TG_clear()
{
    XSetBackground(CGP->pmDisplay, CGP->gc, CGP->colour);
    XFillRectangle(CGP->pmDisplay,
	CGP->drawable,
	CGP->gc,
	0,
	0, 
	(unsigned int)vdevice.sizeSx + 1,
	(unsigned int)vdevice.sizeSy + 1
    );

    FLUSH(CGP->pmDisplay);
}
#else
/*
 * TG_clear
 *
 * Clear the screen (or current buffer )to current colour
 */
TG_clear()
{
    unsigned int    w = vdevice.maxVx - vdevice.minVx;
    unsigned int    h = vdevice.maxVy - vdevice.minVy;

    XSetBackground(CGP->pmDisplay, CGP->gc, CGP->colour);
    XFillRectangle(CGP->pmDisplay,
	CGP->drawable,
	CGP->gc,
	vdevice.minVx,
	vdevice.sizeSy - vdevice.maxVy,
	w,
	h
    );

    FLUSH(CGP->pmDisplay);
}
#endif

/*
 * TG_color
 *
 *	set the current drawing color index.
 */
TG_color(ind)
    int	ind;
{
    if (!CGP->carray[ind]) {
	ind = 0;
	if (!CGP->carray[ind])
	    return -1;
	fprintf(stderr, "TG: no color for slot %d using slot 0\n", ind);
    }
    CGP->colour = (CGP->carray[ind])->pixel;
    XSetForeground(CGP->pmDisplay, CGP->gc, CGP->colour);
}

/*
 * TG_mapcolor -- not done
 *
 *	change index i in the color map to the appropriate r, g, b, value.
 *	it is not a particularly good idea to set colors using rgb values
 *	directly. It is better to use the Standard X color names via the 
 *	TkGraphic_SetDrawColor() function or the tclvogle g_setcolor command.
 */
TG_mapcolor(i, r, g, b)
    int	i;
    int	r, g, b;
{
    if (i >= CMAPSIZE || i < 0)
	return(-1);
    assert(0);

    /*
     * For Black and White.
     * If the index is 0 and r,g,b != 0 then we are remapping black.
     * If the index != 0 and r,g,b == 0 then we make it black.
     */
    Tk_FreeColor(CGP->carray[i]);

#if LATER
    if (vdevice.depth == 1) {
#else
    if (1) {
#endif
	/* B&W */
	char *color_name;
	if (i == 0 && (r != 0 || g != 0 || b != 0)) 
	    color_name = "white";
	else if (i != 0 && r == 0 && g == 0 && b == 0)
	    color_name = "black";
	CGP->carray[i] = Tk_GetColor(CGP->interp, CGP->tkwin, 
				CGP->colormap, Tk_GetUid(color_name));
    } else {
#if LATER
	/* Color */
	int	stat;
	XColor	tmp;
	tmp.red = (unsigned short)(r / 255.0 * 65535);
	tmp.green = (unsigned short)(g / 255.0 * 65535);
	tmp.blue = (unsigned short)(b / 255.0 * 65535);
	tmp.flags = 0;
	tmp.pixel = (unsigned long)i;
	CGP->carray[i] = Tk_GetColorByValue(CGP->interp, CGP->tkwin, 
			    CGP->colormap, &tmp);
#endif
    }

    FLUSH(CGP->pmDisplay);
    return(0);
}
	
/*
 * TG_font
 *
 *   Set up a hardware font. Return 1 on success 0 otherwise.
 *
 */
static char *TG_font_argv[3];

TG_font(fontfile)
    char	*fontfile;
{
    int result = TCL_OK;
    /*
     * is small is specified, use the font defined in the widget
     */
    if (!strcmp(fontfile, "small")) {
	ResetFontForGraphic(CGP);
	TG_reset_hardware_font(CGP);
    } else if (!strcmp(fontfile, "large")) {
	fontfile = LARGEFONT;
    } else {
	TG_font_argv[0] = "-font";
	TG_font_argv[1] = fontfile;
	TG_font_argv[2] = (char *)0;
	result = Tk_ConfigureWidget(CGP->interp, CGP->tkwin, configSpecs, 
		2, TG_font_argv, (char *)CGP, TK_CONFIG_ARGV_ONLY);
	ResetFontForGraphic(CGP);
	TG_reset_hardware_font(CGP);
    }
    return (result == TCL_OK) ? 1 : 0;
}

/* 
 * TG_char
 *
 *	 outputs one char - is more complicated for other devices
 */
static char TG_char_buf[2] = { '\0', '\0' };
TG_char(c)
    char	c;
{
    *TG_char_buf = c;
    XDrawString(CGP->pmDisplay, CGP->drawable, CGP->gc, 
	vdevice.cpVx, (int)(vdevice.sizeSy - vdevice.cpVy), TG_char_buf, 1);
    FLUSH(CGP->pmDisplay);
}

/*
 * TG_string
 *
 *	Display a string at the current drawing position.
 */
TG_string(s)
    register char *s;
{
    XDrawString(CGP->pmDisplay, CGP->drawable, CGP->gc, 
	vdevice.cpVx, (int)(vdevice.sizeSy - vdevice.cpVy), s, strlen(s));
    FLUSH(CGP->pmDisplay);
}

/*
 * TG_fill
 *
 *	fill a polygon
 */
#define TG_FILL_MAXPTS 1024
TG_fill(n, x, y)
	int	n, x[], y[];
{
    XPoint	plist[TG_FILL_MAXPTS];
    register int	i;

    if (n > TG_FILL_MAXPTS) {
	char buf[128];
	sprintf(buf, 
	    "%s: more than TG_FILL_MAXPTS points in a polygon", PackageName);
	verror(buf);
    }

    for (i = 0; i < n; i++) {
	plist[i].x = x[i];
	plist[i].y = vdevice.sizeSy - y[i];
    }

    XFillPolygon(CGP->pmDisplay, CGP->drawable, CGP->gc, 
	plist, n, Nonconvex, CoordModeOrigin);

    vdevice.cpVx = x[n-1];
    vdevice.cpVy = y[n-1];

    FLUSH(CGP->pmDisplay);
}

/*
 * TG_backbuf
 *
 *	Enable double buffering 
 */
TG_backbuf()
{
    /* memory management of the back buffer is done within the widget */
    if (!CGP->doubleBuffered) {
	return -1;
    } 
    Tk_MakeGraphicExist(CGP);
    CGP->drawable = CGP->pm;
    return 1;
}

/*
 * TG_swapbuf
 *
 *	Swap the back and front buffers. (Really, just copy the
 *	back buffer to the screen).
 */
TG_swapbuf()
{
    if (!CGP->doubleBuffered) {
	return -1;
    }
    XCopyArea(CGP->pmDisplay,
	CGP->drawable,
	Tk_WindowId(CGP->tkwin),
	CGP->gc,
	0, 0,
	(unsigned int)vdevice.sizeSx + 1,
	(unsigned int)vdevice.sizeSy + 1,
	0, 0
    );
    SYNC(CGP->pmDisplay, 0);	/* Not FLUSH */
    return 1;
}


/*
 * TG_frontbuf
 *
 *	Make sure we draw to the front buffer (screen).
 */
TG_frontbuf()
{
    if (!CGP->doubleBuffered) {
	return -1;
    } 
    CGP->drawable = Tk_WindowId(CGP->tkwin);
    return 1;
}

/*
 * Syncronise the display with what we think has been sent to it...
 */
TG_sync()
{
    SYNC(CGP->pmDisplay, 0);
}
