// Copyright (c) 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_WEBUI_NEW_TAB_PAGE_NEW_TAB_PAGE_HANDLER_H_
#define CHROME_BROWSER_UI_WEBUI_NEW_TAB_PAGE_NEW_TAB_PAGE_HANDLER_H_

#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "chrome/browser/search/background/ntp_background_service_observer.h"
#include "chrome/browser/search/instant_service_observer.h"
#include "chrome/browser/ui/omnibox/omnibox_tab_helper.h"
#include "chrome/browser/ui/webui/new_tab_page/new_tab_page.mojom.h"
#include "chrome/common/search/instant_types.h"
#include "components/search_provider_logos/logo_common.h"
#include "content/public/browser/web_contents_observer.h"
#include "mojo/public/cpp/bindings/pending_receiver.h"
#include "mojo/public/cpp/bindings/pending_remote.h"
#include "mojo/public/cpp/bindings/receiver.h"
#include "mojo/public/cpp/bindings/remote.h"

class GURL;
class InstantService;
class NtpBackgroundService;
class Profile;

namespace chrome_colors {
class ChromeColorsService;
}  // namespace chrome_colors

namespace content {
class WebContents;
}  // namespace content

namespace search_provider_logos {
class LogoService;
}  // namespace search_provider_logos

class NewTabPageHandler : public new_tab_page::mojom::PageHandler,
                          public InstantServiceObserver,
                          public NtpBackgroundServiceObserver,
                          public OmniboxTabHelper::Observer {
 public:
  NewTabPageHandler(mojo::PendingReceiver<new_tab_page::mojom::PageHandler>
                        pending_page_handler,
                    mojo::PendingRemote<new_tab_page::mojom::Page> pending_page,
                    Profile* profile,
                    content::WebContents* web_contents);
  ~NewTabPageHandler() override;

  // new_tab_page::mojom::PageHandler:
  void AddMostVisitedTile(const GURL& url,
                          const std::string& title,
                          AddMostVisitedTileCallback callback) override;
  void DeleteMostVisitedTile(const GURL& url) override;
  void RestoreMostVisitedDefaults() override;
  void ReorderMostVisitedTile(const GURL& url, uint8_t new_pos) override;
  void SetMostVisitedSettings(bool custom_links_enabled, bool visible) override;
  void UndoMostVisitedTileAction() override;
  void UpdateMostVisitedInfo() override;
  void UpdateMostVisitedTile(const GURL& url,
                             const GURL& new_url,
                             const std::string& new_title,
                             UpdateMostVisitedTileCallback callback) override;
  void ApplyDefaultTheme() override;
  void ApplyAutogeneratedTheme(const SkColor& frame_color) override;
  void ApplyChromeTheme(int32_t id) override;
  void ConfirmThemeChanges() override;
  void GetChromeThemes(GetChromeThemesCallback callback) override;
  void RevertThemeChanges() override;
  void GetBackgroundCollections(
      GetBackgroundCollectionsCallback callback) override;
  void GetBackgroundImages(const std::string& collection_id,
                           GetBackgroundImagesCallback callback) override;
  void FocusOmnibox() override;
  void PasteIntoOmnibox(const std::string& text) override;
  void GetDoodle(GetDoodleCallback callback) override;

 private:
  // InstantServiceObserver:
  void NtpThemeChanged(const NtpTheme& theme) override;
  void MostVisitedInfoChanged(const InstantMostVisitedInfo& info) override;

  // NtpBackgroundServiceObserver:
  void OnCollectionInfoAvailable() override;
  void OnCollectionImagesAvailable() override;
  void OnNextCollectionImageAvailable() override;
  void OnNtpBackgroundServiceShuttingDown() override;

  // OmniboxTabHelper::Observer:
  void OnOmniboxInputStateChanged() override;
  void OnOmniboxFocusChanged(OmniboxFocusState state,
                             OmniboxFocusChangeReason reason) override;

  void OnLogoAvailable(
      GetDoodleCallback callback,
      search_provider_logos::LogoCallbackReason type,
      const base::Optional<search_provider_logos::EncodedLogo>& logo);

  chrome_colors::ChromeColorsService* chrome_colors_service_;
  InstantService* instant_service_;
  NtpBackgroundService* ntp_background_service_;
  search_provider_logos::LogoService* logo_service_;
  GURL last_blacklisted_;
  GetBackgroundCollectionsCallback background_collections_callback_;
  std::string images_request_collection_id_;
  GetBackgroundImagesCallback background_images_callback_;
  mojo::Remote<new_tab_page::mojom::Page> page_;
  mojo::Receiver<new_tab_page::mojom::PageHandler> receiver_;
  content::WebContents* web_contents_;
  base::WeakPtrFactory<NewTabPageHandler> weak_ptr_factory_{this};

  DISALLOW_COPY_AND_ASSIGN(NewTabPageHandler);
};

#endif  // CHROME_BROWSER_UI_WEBUI_NEW_TAB_PAGE_NEW_TAB_PAGE_HANDLER_H_
