# Copyright (C) 2009 Aaron Bentley <aaron@aaronbentley.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


from cStringIO import StringIO
import os

from bzrlib.tests import TestCaseWithTransport
from bzrlib.plugins.bzrtools import errors
from bzrlib.plugins.bzrtools.conflict_diff import (
    ConflictDiffer
)


class TestConflictDiff(TestCaseWithTransport):

    def test_conflict_diff_this(self):
        self.build_tree_contents(
            [('foo.THIS', 'this\n'), ('foo.BASE', 'base\n')])
        s = StringIO()
        ConflictDiffer().conflict_diff(s, 'foo', 'this')
        self.assertEqual('--- foo.BASE\n+++ foo.THIS\n'
                         '@@ -1,1 +1,1 @@\n'
                         '-base\n+this\n\n', s.getvalue())

    def test_no_conflict(self):
        tree = self.make_branch_and_tree('.')
        self.build_tree_contents([('foo', 'base\n')])
        tree.add('foo')
        e = self.assertRaises(errors.NoConflictFiles,
                              ConflictDiffer().get_old_lines,
                              'foo',
                              'foo.BASE')
        self.assertEqual('foo.BASE does not exist and there are no pending'
                         ' merges.', str(e))

    def test_get_old_lines(self):
        self.build_tree_contents([('foo.BASE', 'base\n')])
        old_lines = ConflictDiffer().get_old_lines('foo', 'foo.BASE')
        self.assertEqual(['base\n'], old_lines)

    def test_get_old_lines_no_base(self):
        tree = self.make_branch_and_tree('tree')
        self.build_tree_contents([('tree/foo', 'base\n')])
        tree.add('foo')
        tree.commit('added foo')
        other = tree.bzrdir.sprout('other').open_workingtree()
        self.build_tree_contents([('other/foo', 'other\n')])
        other.commit('Changed foo text')
        self.build_tree_contents([('tree/foo', 'this\n')])
        tree.commit('Changed foo text')
        tree.merge_from_branch(other.branch)
        os.unlink('tree/foo.BASE')
        old_lines = ConflictDiffer().get_old_lines('tree/foo', 'tree/foo.BASE')
        self.assertEqual(['base\n'], old_lines)
