"""Pytest conftest.py."""

from __future__ import annotations

from pathlib import Path

import numpy as np
import pytest

from symfc.utils.utils import SymfcAtoms

cwd = Path(__file__).parent


def pytest_addoption(parser):
    """Add command option to pytest."""
    parser.addoption(
        "--runbig", action="store_true", default=False, help="run big tests"
    )


def pytest_configure(config):
    """Set up marker big."""
    config.addinivalue_line("markers", "big: mark test as big to run")


def pytest_collection_modifyitems(config, items):
    """Add mechanism to run with --runbig."""
    if config.getoption("--runbig"):
        # --runbig given in cli: do not skip slow tests
        return
    skip_big = pytest.mark.skip(reason="need --runbig option to run")
    for item in items:
        if "big" in item.keywords:
            item.add_marker(skip_big)


@pytest.fixture(scope="session")
def cell_nacl_111() -> SymfcAtoms:
    """Return unitcell of NaCl."""
    lattice = [
        [5.690301476175671, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 5.690301476175671, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 5.690301476175671],
    ]
    points = [
        [0.000000000000000, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 0.500000000000000, 0.500000000000000],
        [0.500000000000000, 0.000000000000000, 0.500000000000000],
        [0.500000000000000, 0.500000000000000, 0.000000000000000],
        [0.500000000000000, 0.500000000000000, 0.500000000000000],
        [0.500000000000000, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 0.500000000000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 0.500000000000000],
    ]
    numbers = [11, 11, 11, 11, 17, 17, 17, 17]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    return cell


@pytest.fixture(scope="session")
def cell_gan_111() -> SymfcAtoms:
    """Return unitcell of GaN."""
    lattice = [
        [3.180765520000000, 0.000000000000000, 0.000000000000000],
        [-1.590382760000000, 2.754623740000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 5.182605840000000],
    ]
    points = [
        [0.333333329999992, 0.666666669999983, 0.124191919999999],
        [0.666666669999991, 0.333333329999998, 0.624191919999998],
        [0.333333329999992, 0.666666669999983, 0.500808080000002],
        [0.666666669999991, 0.333333329999998, 0.000808080000000],
    ]
    numbers = [7, 7, 31, 31]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    return cell


@pytest.fixture(scope="session")
def cell_wurtzite_332() -> SymfcAtoms:
    """Return 3x3x2 supercell of wurtzite."""
    lattice = [
        [11.35855848088148, 0.0, 0.0],
        [-5.679279240440739, 9.836800194814545, 0.0],
        [0.0, 0.0, 12.425356538818003],
    ]
    positions = [
        [0.111111111111111, 0.222222222222222, 0.001063232855807],
        [0.444444444444444, 0.222222222222222, 0.001063232855807],
        [0.777777777777778, 0.222222222222222, 0.001063232855807],
        [0.111111111111111, 0.555555555555556, 0.001063232855807],
        [0.444444444444444, 0.555555555555556, 0.001063232855807],
        [0.777777777777778, 0.555555555555556, 0.001063232855807],
        [0.111111111111111, 0.888888888888889, 0.001063232855807],
        [0.444444444444444, 0.888888888888889, 0.001063232855807],
        [0.777777777777778, 0.888888888888889, 0.001063232855807],
        [0.111111111111111, 0.222222222222222, 0.501063232855807],
        [0.444444444444444, 0.222222222222222, 0.501063232855807],
        [0.777777777777778, 0.222222222222222, 0.501063232855807],
        [0.111111111111111, 0.555555555555556, 0.501063232855807],
        [0.444444444444444, 0.555555555555556, 0.501063232855807],
        [0.777777777777778, 0.555555555555556, 0.501063232855807],
        [0.111111111111111, 0.888888888888889, 0.501063232855807],
        [0.444444444444444, 0.888888888888889, 0.501063232855807],
        [0.777777777777778, 0.888888888888889, 0.501063232855807],
        [0.222222222222222, 0.111111111111111, 0.251063232855807],
        [0.555555555555556, 0.111111111111111, 0.251063232855807],
        [0.888888888888889, 0.111111111111111, 0.251063232855807],
        [0.222222222222222, 0.444444444444444, 0.251063232855807],
        [0.555555555555556, 0.444444444444444, 0.251063232855807],
        [0.888888888888889, 0.444444444444444, 0.251063232855807],
        [0.222222222222222, 0.777777777777778, 0.251063232855807],
        [0.555555555555556, 0.777777777777778, 0.251063232855807],
        [0.888888888888889, 0.777777777777778, 0.251063232855807],
        [0.222222222222222, 0.111111111111111, 0.751063232855807],
        [0.555555555555556, 0.111111111111111, 0.751063232855807],
        [0.888888888888889, 0.111111111111111, 0.751063232855807],
        [0.222222222222222, 0.444444444444444, 0.751063232855807],
        [0.555555555555556, 0.444444444444444, 0.751063232855807],
        [0.888888888888889, 0.444444444444444, 0.751063232855807],
        [0.222222222222222, 0.777777777777778, 0.751063232855807],
        [0.555555555555556, 0.777777777777778, 0.751063232855807],
        [0.888888888888889, 0.777777777777778, 0.751063232855807],
        [0.111111111111111, 0.222222222222222, 0.188158257144195],
        [0.444444444444444, 0.222222222222222, 0.188158257144195],
        [0.777777777777778, 0.222222222222222, 0.188158257144195],
        [0.111111111111111, 0.555555555555556, 0.188158257144195],
        [0.444444444444444, 0.555555555555556, 0.188158257144195],
        [0.777777777777778, 0.555555555555556, 0.188158257144195],
        [0.111111111111111, 0.888888888888889, 0.188158257144195],
        [0.444444444444444, 0.888888888888889, 0.188158257144195],
        [0.777777777777778, 0.888888888888889, 0.188158257144195],
        [0.111111111111111, 0.222222222222222, 0.688158257144195],
        [0.444444444444444, 0.222222222222222, 0.688158257144195],
        [0.777777777777778, 0.222222222222222, 0.688158257144195],
        [0.111111111111111, 0.555555555555556, 0.688158257144195],
        [0.444444444444444, 0.555555555555556, 0.688158257144195],
        [0.777777777777778, 0.555555555555556, 0.688158257144195],
        [0.111111111111111, 0.888888888888889, 0.688158257144195],
        [0.444444444444444, 0.888888888888889, 0.688158257144195],
        [0.777777777777778, 0.888888888888889, 0.688158257144195],
        [0.222222222222222, 0.111111111111111, 0.438158257144194],
        [0.555555555555556, 0.111111111111111, 0.438158257144194],
        [0.888888888888889, 0.111111111111111, 0.438158257144194],
        [0.222222222222222, 0.444444444444444, 0.438158257144194],
        [0.555555555555556, 0.444444444444444, 0.438158257144194],
        [0.888888888888889, 0.444444444444444, 0.438158257144194],
        [0.222222222222222, 0.777777777777778, 0.438158257144194],
        [0.555555555555556, 0.777777777777778, 0.438158257144194],
        [0.888888888888889, 0.777777777777778, 0.438158257144194],
        [0.222222222222222, 0.111111111111111, 0.938158257144194],
        [0.555555555555556, 0.111111111111111, 0.938158257144194],
        [0.888888888888889, 0.111111111111111, 0.938158257144194],
        [0.222222222222222, 0.444444444444444, 0.938158257144194],
        [0.555555555555556, 0.444444444444444, 0.938158257144194],
        [0.888888888888889, 0.444444444444444, 0.938158257144194],
        [0.222222222222222, 0.777777777777778, 0.938158257144194],
        [0.555555555555556, 0.777777777777778, 0.938158257144194],
        [0.888888888888889, 0.777777777777778, 0.938158257144194],
    ]
    numbers = [1 for i in range(36)] + [2 for i in range(36)]
    supercell = SymfcAtoms(cell=lattice, scaled_positions=positions, numbers=numbers)
    return supercell


@pytest.fixture(scope="session")
def cell_nacl_222() -> SymfcAtoms:
    """Return 2x2x2 supercell of NaCl."""
    lattice = [
        [11.281120000000000, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 11.281120000000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 11.281120000000000],
    ]
    points = [
        [0.000000000000000, 0.000000000000000, 0.000000000000000],
        [0.500000000000000, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 0.500000000000000, 0.000000000000000],
        [0.500000000000000, 0.500000000000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 0.500000000000000],
        [0.500000000000000, 0.000000000000000, 0.500000000000000],
        [0.000000000000000, 0.500000000000000, 0.500000000000000],
        [0.500000000000000, 0.500000000000000, 0.500000000000000],
        [0.000000000000000, 0.250000000000000, 0.250000000000000],
        [0.500000000000000, 0.250000000000000, 0.250000000000000],
        [0.000000000000000, 0.750000000000000, 0.250000000000000],
        [0.500000000000000, 0.750000000000000, 0.250000000000000],
        [0.000000000000000, 0.250000000000000, 0.750000000000000],
        [0.500000000000000, 0.250000000000000, 0.750000000000000],
        [0.000000000000000, 0.750000000000000, 0.750000000000000],
        [0.500000000000000, 0.750000000000000, 0.750000000000000],
        [0.250000000000000, 0.000000000000000, 0.250000000000000],
        [0.750000000000000, 0.000000000000000, 0.250000000000000],
        [0.250000000000000, 0.500000000000000, 0.250000000000000],
        [0.750000000000000, 0.500000000000000, 0.250000000000000],
        [0.250000000000000, 0.000000000000000, 0.750000000000000],
        [0.750000000000000, 0.000000000000000, 0.750000000000000],
        [0.250000000000000, 0.500000000000000, 0.750000000000000],
        [0.750000000000000, 0.500000000000000, 0.750000000000000],
        [0.250000000000000, 0.250000000000000, 0.000000000000000],
        [0.750000000000000, 0.250000000000000, 0.000000000000000],
        [0.250000000000000, 0.750000000000000, 0.000000000000000],
        [0.750000000000000, 0.750000000000000, 0.000000000000000],
        [0.250000000000000, 0.250000000000000, 0.500000000000000],
        [0.750000000000000, 0.250000000000000, 0.500000000000000],
        [0.250000000000000, 0.750000000000000, 0.500000000000000],
        [0.750000000000000, 0.750000000000000, 0.500000000000000],
        [0.250000000000000, 0.250000000000000, 0.250000000000000],
        [0.750000000000000, 0.250000000000000, 0.250000000000000],
        [0.250000000000000, 0.750000000000000, 0.250000000000000],
        [0.750000000000000, 0.750000000000000, 0.250000000000000],
        [0.250000000000000, 0.250000000000000, 0.750000000000000],
        [0.750000000000000, 0.250000000000000, 0.750000000000000],
        [0.250000000000000, 0.750000000000000, 0.750000000000000],
        [0.750000000000000, 0.750000000000000, 0.750000000000000],
        [0.250000000000000, 0.000000000000000, 0.000000000000000],
        [0.750000000000000, 0.000000000000000, 0.000000000000000],
        [0.250000000000000, 0.500000000000000, 0.000000000000000],
        [0.750000000000000, 0.500000000000000, 0.000000000000000],
        [0.250000000000000, 0.000000000000000, 0.500000000000000],
        [0.750000000000000, 0.000000000000000, 0.500000000000000],
        [0.250000000000000, 0.500000000000000, 0.500000000000000],
        [0.750000000000000, 0.500000000000000, 0.500000000000000],
        [0.000000000000000, 0.250000000000000, 0.000000000000000],
        [0.500000000000000, 0.250000000000000, 0.000000000000000],
        [0.000000000000000, 0.750000000000000, 0.000000000000000],
        [0.500000000000000, 0.750000000000000, 0.000000000000000],
        [0.000000000000000, 0.250000000000000, 0.500000000000000],
        [0.500000000000000, 0.250000000000000, 0.500000000000000],
        [0.000000000000000, 0.750000000000000, 0.500000000000000],
        [0.500000000000000, 0.750000000000000, 0.500000000000000],
        [0.000000000000000, 0.000000000000000, 0.250000000000000],
        [0.500000000000000, 0.000000000000000, 0.250000000000000],
        [0.000000000000000, 0.500000000000000, 0.250000000000000],
        [0.500000000000000, 0.500000000000000, 0.250000000000000],
        [0.000000000000000, 0.000000000000000, 0.750000000000000],
        [0.500000000000000, 0.000000000000000, 0.750000000000000],
        [0.000000000000000, 0.500000000000000, 0.750000000000000],
        [0.500000000000000, 0.500000000000000, 0.750000000000000],
    ]
    numbers = [
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        11,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
        17,
    ]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    return cell


@pytest.fixture(scope="session")
def ph_nacl_222(cell_nacl_222) -> tuple[SymfcAtoms, np.ndarray, np.ndarray]:
    """Return NaCl-222 data."""
    N = 20
    dfset = np.loadtxt(cwd / "dfset_NaCl_222_rd.xz")
    d = dfset[:, :3].reshape(N, -1, 3)
    f = dfset[:, 3:].reshape(N, -1, 3)
    return cell_nacl_222, d, f


@pytest.fixture(scope="session")
def ph_sno2_223() -> tuple[SymfcAtoms, np.ndarray, np.ndarray]:
    """Return SnO2-223 data."""
    lattice = [
        [9.545162500000000, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 9.545162500000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 9.649138170000001],
    ]
    points = [
        [0.096865350000001, 0.403134649999997, 0.166666666666667],
        [0.596865350000001, 0.403134649999997, 0.166666666666667],
        [0.096865350000001, 0.903134649999997, 0.166666666666667],
        [0.596865350000001, 0.903134649999997, 0.166666666666667],
        [0.096865350000001, 0.403134649999997, 0.500000000000000],
        [0.596865350000001, 0.403134649999997, 0.500000000000000],
        [0.096865350000001, 0.903134649999997, 0.500000000000000],
        [0.596865350000001, 0.903134649999997, 0.500000000000000],
        [0.096865350000001, 0.403134649999997, 0.833333333333333],
        [0.596865350000001, 0.403134649999997, 0.833333333333333],
        [0.096865350000001, 0.903134649999997, 0.833333333333333],
        [0.596865350000001, 0.903134649999997, 0.833333333333333],
        [0.403134649999997, 0.096865350000001, 0.166666666666667],
        [0.903134649999997, 0.096865350000001, 0.166666666666667],
        [0.403134649999997, 0.596865350000001, 0.166666666666667],
        [0.903134649999997, 0.596865350000001, 0.166666666666667],
        [0.403134649999997, 0.096865350000001, 0.500000000000000],
        [0.903134649999997, 0.096865350000001, 0.500000000000000],
        [0.403134649999997, 0.596865350000001, 0.500000000000000],
        [0.903134649999997, 0.596865350000001, 0.500000000000000],
        [0.403134649999997, 0.096865350000001, 0.833333333333333],
        [0.903134649999997, 0.096865350000001, 0.833333333333333],
        [0.403134649999997, 0.596865350000001, 0.833333333333333],
        [0.903134649999997, 0.596865350000001, 0.833333333333333],
        [0.153134649999998, 0.153134649999998, 0.000000000000000],
        [0.653134649999997, 0.153134649999998, 0.000000000000000],
        [0.153134649999998, 0.653134649999997, 0.000000000000000],
        [0.653134649999997, 0.653134649999997, 0.000000000000000],
        [0.153134649999998, 0.153134649999998, 0.333333333333333],
        [0.653134649999997, 0.153134649999998, 0.333333333333333],
        [0.153134649999998, 0.653134649999997, 0.333333333333333],
        [0.653134649999997, 0.653134649999997, 0.333333333333333],
        [0.153134649999998, 0.153134649999998, 0.666666666666667],
        [0.653134649999997, 0.153134649999998, 0.666666666666667],
        [0.153134649999998, 0.653134649999997, 0.666666666666667],
        [0.653134649999997, 0.653134649999997, 0.666666666666667],
        [0.346865350000003, 0.346865350000003, 0.000000000000000],
        [0.846865350000003, 0.346865350000003, 0.000000000000000],
        [0.346865350000003, 0.846865350000003, 0.000000000000000],
        [0.846865350000003, 0.846865350000003, 0.000000000000000],
        [0.346865350000003, 0.346865350000003, 0.333333333333333],
        [0.846865350000003, 0.346865350000003, 0.333333333333333],
        [0.346865350000003, 0.846865350000003, 0.333333333333333],
        [0.846865350000003, 0.846865350000003, 0.333333333333333],
        [0.346865350000003, 0.346865350000003, 0.666666666666667],
        [0.846865350000003, 0.346865350000003, 0.666666666666667],
        [0.346865350000003, 0.846865350000003, 0.666666666666667],
        [0.846865350000003, 0.846865350000003, 0.666666666666667],
        [0.250000000000000, 0.250000000000000, 0.166666666666667],
        [0.750000000000000, 0.250000000000000, 0.166666666666667],
        [0.250000000000000, 0.750000000000000, 0.166666666666667],
        [0.750000000000000, 0.750000000000000, 0.166666666666667],
        [0.250000000000000, 0.250000000000000, 0.500000000000000],
        [0.750000000000000, 0.250000000000000, 0.500000000000000],
        [0.250000000000000, 0.750000000000000, 0.500000000000000],
        [0.750000000000000, 0.750000000000000, 0.500000000000000],
        [0.250000000000000, 0.250000000000000, 0.833333333333333],
        [0.750000000000000, 0.250000000000000, 0.833333333333333],
        [0.250000000000000, 0.750000000000000, 0.833333333333333],
        [0.750000000000000, 0.750000000000000, 0.833333333333333],
        [0.000000000000000, 0.000000000000000, 0.000000000000000],
        [0.500000000000000, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 0.500000000000000, 0.000000000000000],
        [0.500000000000000, 0.500000000000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 0.333333333333333],
        [0.500000000000000, 0.000000000000000, 0.333333333333333],
        [0.000000000000000, 0.500000000000000, 0.333333333333333],
        [0.500000000000000, 0.500000000000000, 0.333333333333333],
        [0.000000000000000, 0.000000000000000, 0.666666666666667],
        [0.500000000000000, 0.000000000000000, 0.666666666666667],
        [0.000000000000000, 0.500000000000000, 0.666666666666667],
        [0.500000000000000, 0.500000000000000, 0.666666666666667],
    ]
    numbers = [
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
    ]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    N = 40
    dfset = np.loadtxt(cwd / "dfset_SnO2_223_rd.xz")
    d = dfset[:, :3].reshape(N, -1, 3)
    f = dfset[:, 3:].reshape(N, -1, 3)
    return cell, d, f


@pytest.fixture(scope="session")
def ph_sio2_221() -> tuple[SymfcAtoms, np.ndarray, np.ndarray]:
    """Return SiO2-221 data."""
    lattice = [
        [9.919749440000000, 0.000000000000000, 0.000000000000000],
        [-4.959874720000000, 8.590755020000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 5.453559970000000],
    ]
    points = [
        [0.293442579999996, 0.367112664999997, 0.213280300000002],
        [0.793442579999996, 0.367112664999997, 0.213280300000002],
        [0.293442579999996, 0.867112664999997, 0.213280300000002],
        [0.793442579999996, 0.867112664999997, 0.213280300000002],
        [0.073670084999997, 0.206557419999994, 0.879946960000002],
        [0.573670084999997, 0.206557419999994, 0.879946960000002],
        [0.073670084999997, 0.706557419999994, 0.879946960000002],
        [0.573670084999997, 0.706557419999994, 0.879946960000002],
        [0.132887334999998, 0.426329914999998, 0.546613630000002],
        [0.632887334999998, 0.426329914999998, 0.546613630000002],
        [0.132887334999998, 0.926329914999998, 0.546613630000002],
        [0.632887334999998, 0.926329914999998, 0.546613630000002],
        [0.367112665000005, 0.293442580000006, 0.786719699999998],
        [0.867112665000005, 0.293442580000006, 0.786719699999998],
        [0.367112665000005, 0.793442580000006, 0.786719699999998],
        [0.867112665000005, 0.793442580000006, 0.786719699999998],
        [0.426329914999999, 0.132887335000003, 0.453386369999998],
        [0.926329914999999, 0.132887335000003, 0.453386369999998],
        [0.426329914999999, 0.632887335000003, 0.453386369999998],
        [0.926329914999999, 0.632887335000003, 0.453386369999998],
        [0.206557420000001, 0.073670085000000, 0.120053040000000],
        [0.706557420000001, 0.073670085000000, 0.120053040000000],
        [0.206557420000001, 0.573670085000000, 0.120053040000000],
        [0.706557420000001, 0.573670085000000, 0.120053040000000],
        [0.235454159999997, 0.235454160000002, 0.000000000000000],
        [0.735454159999998, 0.235454160000002, 0.000000000000000],
        [0.235454159999997, 0.735454160000002, 0.000000000000000],
        [0.735454159999998, 0.735454160000002, 0.000000000000000],
        [0.264545839999997, 0.000000000000000, 0.333333330000000],
        [0.764545839999997, 0.000000000000000, 0.333333330000000],
        [0.264545839999997, 0.500000000000000, 0.333333330000000],
        [0.764545839999997, 0.500000000000000, 0.333333330000000],
        [0.999999999999995, 0.264545839999998, 0.666666670000000],
        [0.499999999999996, 0.264545839999998, 0.666666670000000],
        [0.999999999999995, 0.764545839999998, 0.666666670000000],
        [0.499999999999996, 0.764545839999998, 0.666666670000000],
    ]
    numbers = [
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        8,
        14,
        14,
        14,
        14,
        14,
        14,
        14,
        14,
        14,
        14,
        14,
        14,
    ]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    N = 40
    dfset = np.loadtxt(cwd / "dfset_SiO2_221_rd.xz")
    d = dfset[:, :3].reshape(N, -1, 3)
    f = dfset[:, 3:].reshape(N, -1, 3)
    return cell, d, f


@pytest.fixture(scope="session")
def ph_gan_222() -> tuple[SymfcAtoms, np.ndarray, np.ndarray]:
    """Return GaN-222 data."""
    lattice = [
        [6.361531040000000, 0.000000000000000, 0.000000000000000],
        [-3.180765520000000, 5.509247480000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 10.365211680000000],
    ]
    points = [
        [0.166666664999996, 0.333333334999992, 0.062095959999999],
        [0.666666664999996, 0.333333334999992, 0.062095959999999],
        [0.166666664999996, 0.833333334999992, 0.062095959999999],
        [0.666666664999996, 0.833333334999992, 0.062095959999999],
        [0.166666664999996, 0.333333334999992, 0.562095960000000],
        [0.666666664999996, 0.333333334999992, 0.562095960000000],
        [0.166666664999996, 0.833333334999992, 0.562095960000000],
        [0.666666664999996, 0.833333334999992, 0.562095960000000],
        [0.333333334999996, 0.166666664999999, 0.312095959999999],
        [0.833333334999996, 0.166666664999999, 0.312095959999999],
        [0.333333334999996, 0.666666664999999, 0.312095959999999],
        [0.833333334999996, 0.666666664999999, 0.312095959999999],
        [0.333333334999996, 0.166666664999999, 0.812095959999999],
        [0.833333334999996, 0.166666664999999, 0.812095959999999],
        [0.333333334999996, 0.666666664999999, 0.812095959999999],
        [0.833333334999996, 0.666666664999999, 0.812095959999999],
        [0.166666664999996, 0.333333334999992, 0.250404040000001],
        [0.666666664999996, 0.333333334999992, 0.250404040000001],
        [0.166666664999996, 0.833333334999992, 0.250404040000001],
        [0.666666664999996, 0.833333334999992, 0.250404040000001],
        [0.166666664999996, 0.333333334999992, 0.750404040000001],
        [0.666666664999996, 0.333333334999992, 0.750404040000001],
        [0.166666664999996, 0.833333334999992, 0.750404040000001],
        [0.666666664999996, 0.833333334999992, 0.750404040000001],
        [0.333333334999996, 0.166666664999999, 0.000404040000000],
        [0.833333334999996, 0.166666664999999, 0.000404040000000],
        [0.333333334999996, 0.666666664999999, 0.000404040000000],
        [0.833333334999996, 0.666666664999999, 0.000404040000000],
        [0.333333334999996, 0.166666664999999, 0.500404040000000],
        [0.833333334999996, 0.166666664999999, 0.500404040000000],
        [0.333333334999996, 0.666666664999999, 0.500404040000000],
        [0.833333334999996, 0.666666664999999, 0.500404040000000],
    ]
    numbers = [
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
    ]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    N = 40
    dfset = np.loadtxt(cwd / "dfset_GaN_222_rd.xz")
    d = dfset[:, :3].reshape(N, -1, 3)
    f = dfset[:, 3:].reshape(N, -1, 3)
    return cell, d, f


@pytest.fixture(scope="session")
def ph_gan_442() -> tuple[SymfcAtoms, np.ndarray, np.ndarray]:
    """Return GaN-442 data."""
    lattice = [
        [12.723062080000000, 0.000000000000000, 0.000000000000000],
        [-6.361531040000000, 11.018494960000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 10.365211680000000],
    ]
    points = [
        [0.083333332499998, 0.166666667499996, 0.062095959999999],
        [0.333333332499998, 0.166666667499996, 0.062095959999999],
        [0.583333332499998, 0.166666667499996, 0.062095959999999],
        [0.833333332499998, 0.166666667499996, 0.062095959999999],
        [0.083333332499998, 0.416666667499996, 0.062095959999999],
        [0.333333332499998, 0.416666667499996, 0.062095959999999],
        [0.583333332499998, 0.416666667499996, 0.062095959999999],
        [0.833333332499998, 0.416666667499996, 0.062095959999999],
        [0.083333332499998, 0.666666667499996, 0.062095959999999],
        [0.333333332499998, 0.666666667499996, 0.062095959999999],
        [0.583333332499998, 0.666666667499996, 0.062095959999999],
        [0.833333332499998, 0.666666667499996, 0.062095959999999],
        [0.083333332499998, 0.916666667499996, 0.062095959999999],
        [0.333333332499998, 0.916666667499996, 0.062095959999999],
        [0.583333332499998, 0.916666667499996, 0.062095959999999],
        [0.833333332499998, 0.916666667499996, 0.062095959999999],
        [0.083333332499998, 0.166666667499996, 0.562095960000000],
        [0.333333332499998, 0.166666667499996, 0.562095960000000],
        [0.583333332499998, 0.166666667499996, 0.562095960000000],
        [0.833333332499998, 0.166666667499996, 0.562095960000000],
        [0.083333332499998, 0.416666667499996, 0.562095960000000],
        [0.333333332499998, 0.416666667499996, 0.562095960000000],
        [0.583333332499998, 0.416666667499996, 0.562095960000000],
        [0.833333332499998, 0.416666667499996, 0.562095960000000],
        [0.083333332499998, 0.666666667499996, 0.562095960000000],
        [0.333333332499998, 0.666666667499996, 0.562095960000000],
        [0.583333332499998, 0.666666667499996, 0.562095960000000],
        [0.833333332499998, 0.666666667499996, 0.562095960000000],
        [0.083333332499998, 0.916666667499996, 0.562095960000000],
        [0.333333332499998, 0.916666667499996, 0.562095960000000],
        [0.583333332499998, 0.916666667499996, 0.562095960000000],
        [0.833333332499998, 0.916666667499996, 0.562095960000000],
        [0.166666667499998, 0.083333332499999, 0.312095959999999],
        [0.416666667499998, 0.083333332499999, 0.312095959999999],
        [0.666666667499998, 0.083333332499999, 0.312095959999999],
        [0.916666667499998, 0.083333332499999, 0.312095959999999],
        [0.166666667499998, 0.333333332500000, 0.312095959999999],
        [0.416666667499998, 0.333333332500000, 0.312095959999999],
        [0.666666667499998, 0.333333332500000, 0.312095959999999],
        [0.916666667499998, 0.333333332500000, 0.312095959999999],
        [0.166666667499998, 0.583333332500000, 0.312095959999999],
        [0.416666667499998, 0.583333332500000, 0.312095959999999],
        [0.666666667499998, 0.583333332500000, 0.312095959999999],
        [0.916666667499998, 0.583333332500000, 0.312095959999999],
        [0.166666667499998, 0.833333332500000, 0.312095959999999],
        [0.416666667499998, 0.833333332500000, 0.312095959999999],
        [0.666666667499998, 0.833333332500000, 0.312095959999999],
        [0.916666667499998, 0.833333332500000, 0.312095959999999],
        [0.166666667499998, 0.083333332499999, 0.812095959999999],
        [0.416666667499998, 0.083333332499999, 0.812095959999999],
        [0.666666667499998, 0.083333332499999, 0.812095959999999],
        [0.916666667499998, 0.083333332499999, 0.812095959999999],
        [0.166666667499998, 0.333333332500000, 0.812095959999999],
        [0.416666667499998, 0.333333332500000, 0.812095959999999],
        [0.666666667499998, 0.333333332500000, 0.812095959999999],
        [0.916666667499998, 0.333333332500000, 0.812095959999999],
        [0.166666667499998, 0.583333332500000, 0.812095959999999],
        [0.416666667499998, 0.583333332500000, 0.812095959999999],
        [0.666666667499998, 0.583333332500000, 0.812095959999999],
        [0.916666667499998, 0.583333332500000, 0.812095959999999],
        [0.166666667499998, 0.833333332500000, 0.812095959999999],
        [0.416666667499998, 0.833333332500000, 0.812095959999999],
        [0.666666667499998, 0.833333332500000, 0.812095959999999],
        [0.916666667499998, 0.833333332500000, 0.812095959999999],
        [0.083333332499998, 0.166666667499996, 0.250404040000001],
        [0.333333332499998, 0.166666667499996, 0.250404040000001],
        [0.583333332499998, 0.166666667499996, 0.250404040000001],
        [0.833333332499998, 0.166666667499996, 0.250404040000001],
        [0.083333332499998, 0.416666667499996, 0.250404040000001],
        [0.333333332499998, 0.416666667499996, 0.250404040000001],
        [0.583333332499998, 0.416666667499996, 0.250404040000001],
        [0.833333332499998, 0.416666667499996, 0.250404040000001],
        [0.083333332499998, 0.666666667499996, 0.250404040000001],
        [0.333333332499998, 0.666666667499996, 0.250404040000001],
        [0.583333332499998, 0.666666667499996, 0.250404040000001],
        [0.833333332499998, 0.666666667499996, 0.250404040000001],
        [0.083333332499998, 0.916666667499996, 0.250404040000001],
        [0.333333332499998, 0.916666667499996, 0.250404040000001],
        [0.583333332499998, 0.916666667499996, 0.250404040000001],
        [0.833333332499998, 0.916666667499996, 0.250404040000001],
        [0.083333332499998, 0.166666667499996, 0.750404040000001],
        [0.333333332499998, 0.166666667499996, 0.750404040000001],
        [0.583333332499998, 0.166666667499996, 0.750404040000001],
        [0.833333332499998, 0.166666667499996, 0.750404040000001],
        [0.083333332499998, 0.416666667499996, 0.750404040000001],
        [0.333333332499998, 0.416666667499996, 0.750404040000001],
        [0.583333332499998, 0.416666667499996, 0.750404040000001],
        [0.833333332499998, 0.416666667499996, 0.750404040000001],
        [0.083333332499998, 0.666666667499996, 0.750404040000001],
        [0.333333332499998, 0.666666667499996, 0.750404040000001],
        [0.583333332499998, 0.666666667499996, 0.750404040000001],
        [0.833333332499998, 0.666666667499996, 0.750404040000001],
        [0.083333332499998, 0.916666667499996, 0.750404040000001],
        [0.333333332499998, 0.916666667499996, 0.750404040000001],
        [0.583333332499998, 0.916666667499996, 0.750404040000001],
        [0.833333332499998, 0.916666667499996, 0.750404040000001],
        [0.166666667499998, 0.083333332499999, 0.000404040000000],
        [0.416666667499998, 0.083333332499999, 0.000404040000000],
        [0.666666667499998, 0.083333332499999, 0.000404040000000],
        [0.916666667499998, 0.083333332499999, 0.000404040000000],
        [0.166666667499998, 0.333333332500000, 0.000404040000000],
        [0.416666667499998, 0.333333332500000, 0.000404040000000],
        [0.666666667499998, 0.333333332500000, 0.000404040000000],
        [0.916666667499998, 0.333333332500000, 0.000404040000000],
        [0.166666667499998, 0.583333332500000, 0.000404040000000],
        [0.416666667499998, 0.583333332500000, 0.000404040000000],
        [0.666666667499998, 0.583333332500000, 0.000404040000000],
        [0.916666667499998, 0.583333332500000, 0.000404040000000],
        [0.166666667499998, 0.833333332500000, 0.000404040000000],
        [0.416666667499998, 0.833333332500000, 0.000404040000000],
        [0.666666667499998, 0.833333332500000, 0.000404040000000],
        [0.916666667499998, 0.833333332500000, 0.000404040000000],
        [0.166666667499998, 0.083333332499999, 0.500404040000000],
        [0.416666667499998, 0.083333332499999, 0.500404040000000],
        [0.666666667499998, 0.083333332499999, 0.500404040000000],
        [0.916666667499998, 0.083333332499999, 0.500404040000000],
        [0.166666667499998, 0.333333332500000, 0.500404040000000],
        [0.416666667499998, 0.333333332500000, 0.500404040000000],
        [0.666666667499998, 0.333333332500000, 0.500404040000000],
        [0.916666667499998, 0.333333332500000, 0.500404040000000],
        [0.166666667499998, 0.583333332500000, 0.500404040000000],
        [0.416666667499998, 0.583333332500000, 0.500404040000000],
        [0.666666667499998, 0.583333332500000, 0.500404040000000],
        [0.916666667499998, 0.583333332500000, 0.500404040000000],
        [0.166666667499998, 0.833333332500000, 0.500404040000000],
        [0.416666667499998, 0.833333332500000, 0.500404040000000],
        [0.666666667499998, 0.833333332500000, 0.500404040000000],
        [0.916666667499998, 0.833333332500000, 0.500404040000000],
    ]
    numbers = [
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        7,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
        31,
    ]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    N = 40
    dfset = np.loadtxt(cwd / "dfset_GaN_442_rd.xz")
    d = dfset[:, :3].reshape(N, -1, 3)
    f = dfset[:, 3:].reshape(N, -1, 3)
    return cell, d, f


@pytest.fixture(scope="session")
def ph3_si_111_fc3() -> tuple[SymfcAtoms, np.ndarray, np.ndarray]:
    """Return Si-111 data for fc3."""
    lattice = [
        [5.433560030000000, 0.000000000000000, 0.000000000000000],
        [0.000000000000000, 5.433560030000000, 0.000000000000000],
        [0.000000000000000, 0.000000000000000, 5.433560030000000],
    ]
    points = [
        [0.875000000000000, 0.875000000000000, 0.875000000000000],
        [0.875000000000000, 0.375000000000000, 0.375000000000000],
        [0.375000000000000, 0.875000000000000, 0.375000000000000],
        [0.375000000000000, 0.375000000000000, 0.875000000000000],
        [0.125000000000000, 0.125000000000000, 0.125000000000000],
        [0.125000000000000, 0.625000000000000, 0.625000000000000],
        [0.625000000000000, 0.125000000000000, 0.625000000000000],
        [0.625000000000000, 0.625000000000000, 0.125000000000000],
    ]
    numbers = [
        14,
        14,
        14,
        14,
        14,
        14,
        14,
        14,
    ]
    cell = SymfcAtoms(cell=lattice, scaled_positions=points, numbers=numbers)
    N = 20
    dfset = np.loadtxt(cwd / "dfset_Si_111_fc3_rd.xz")
    d = dfset[:, :3].reshape(N, -1, 3)
    f = dfset[:, 3:].reshape(N, -1, 3)
    return cell, d, f
