package org.jboss.cache.commands;

import org.jboss.cache.InvocationContext;

/**
 * The core of the command-based cache framework.  Commands correspond to specific areas of functionality in the cache,
 * and can be replicated using the {@link org.jboss.cache.marshall.Marshaller} framework.
 *
 * @author Mircea.Markus@jboss.com
 * @author Manik Surtani
 * @since 2.2.0
 */
public interface ReplicableCommand extends Cloneable
{
   /**
    * Performs the primary function of the command.  Please see specific implementation classes for details on what is
    * performed as well as return types.
    * <b>Important</b>: this method will be invoked at the end of interceptors chain. It should never be called directly from
    * a custom interceptor.
    *
    * @param ctx invocation context
    * @return arbitrary return value generated by performing this command
    * @throws Throwable in the event of problems.
    */
   Object perform(InvocationContext ctx) throws Throwable;

   /**
    * Used by marshallers to convert this command into an id for streaming.
    *
    * @return the method id of this command.  This is compatible with pre-2.2.0 MethodCall ids.
    */
   int getCommandId();

   /**
    * Used by marshallers to stream this command across a network
    *
    * @return an object array of arguments, compatible with pre-2.2.0 MethodCall args.
    */
   Object[] getParameters();

   /**
    * Used by the {@link org.jboss.cache.factories.CommandsFactory} to create a command from raw data read off a stream.
    *
    * @param commandId  command id to set.  This is usually unused but *could* be used in the event of a command having multiple IDs, such as {@link org.jboss.cache.commands.write.PutKeyValueCommand}.
    * @param parameters object array of args
    */
   void setParameters(int commandId, Object[] parameters);
}
