/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.eviction;

import org.jboss.cache.config.ConfigurationException;
import org.jboss.cache.config.Dynamic;

/**
 * Configuration implementation for {@link LRUPolicy}.
 * <p/>
 * If configured via XML, expects the following:
 * <p/>
 * <pre>
 * <region name="/maxAgeTest/">
 *    <attribute name="maxNodes">10000</attribute>
 *    <attribute name="timeToLiveSeconds">8</attribute>
 *    <attribute name="maxAgeSeconds">10</attribute>
 * </region>
 * </pre>
 *
 * @author Daniel Huang (dhuang@jboss.org)
 * @version $Revision: 6120 $
 */
public class LRUConfiguration extends EvictionPolicyConfigBase
{
   /**
    * The serialVersionUID
    */
   private static final long serialVersionUID = -3426716488271559729L;

   @Dynamic
   private int timeToLiveSeconds;
   @Dynamic
   private int maxAgeSeconds;

   public LRUConfiguration()
   {
      super();
      // Force config of ttls
      setTimeToLiveSeconds(-1);
   }

   @Override
   protected void setEvictionPolicyClassName()
   {
      setEvictionPolicyClass(LRUPolicy.class.getName());
   }


   public int getTimeToLiveSeconds()
   {
      return timeToLiveSeconds;
   }

   public void setTimeToLiveSeconds(int timeToLiveSeconds)
   {
      testImmutability("timeToLiveSeconds");
      this.timeToLiveSeconds = timeToLiveSeconds;
   }

   public int getMaxAgeSeconds()
   {
      return maxAgeSeconds;
   }

   public void setMaxAgeSeconds(int maxAgeSeconds)
   {
      testImmutability("maxAgeSeconds");
      this.maxAgeSeconds = maxAgeSeconds;
   }

   /**
    * Requires a positive timeToLiveSeconds value or ConfigurationException
    * is thrown.
    */
   @Override
   public void validate() throws ConfigurationException
   {
      if (timeToLiveSeconds < 0)
      {
         throw new ConfigurationException("timeToLiveSeconds must be " +
               "configured to a value greater than or equal to 0");
      }
   }

   @Override
   public String toString()
   {
      StringBuilder str = new StringBuilder();
      str.append("LRUConfiguration: timeToLiveSeconds = ").append(getTimeToLiveSeconds()).append(" maxAgeSeconds =");
      str.append(getMaxAgeSeconds()).append(" maxNodes =").append(getMaxNodes());
      return str.toString();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (super.equals(obj))
      {
         LRUConfiguration other = (LRUConfiguration) obj;
         return maxAgeSeconds == other.maxAgeSeconds
               && timeToLiveSeconds == other.timeToLiveSeconds;
      }
      return false;
   }

   @Override
   public int hashCode()
   {
      int result = super.hashCode();
      result = 31 * result + maxAgeSeconds;
      result = 31 * result + timeToLiveSeconds;
      return result;
   }

   @Override
   public void reset()
   {
      super.reset();
      setTimeToLiveSeconds(-1);
   }

   @Override
   public LRUConfiguration clone() throws CloneNotSupportedException
   {
      return (LRUConfiguration) super.clone();
   }

}
