/*-------------------------------------------------------------------------
 * Copyright (c) 2000-2002 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * prop_dlg
 *
 * Synopsis:
 *   property dlg
 *
 * $Id: prop_dlg.c,v 1.8 2002/05/30 01:02:01 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "draw.h"
#include "ini_defs.h"

/*
 * Definitions of important widget IDs
 */
#define NUM_ROUNDS_SPIN      "num_rounds"
#define SEC_PER_ROUND_SPIN   "secs_per_round"
#define SEC_BTWN_ROUND_SPIN  "secs_btwn_round"
#define RESP_DELAY_SPIN      "resp_delay"
#define ADD_PCT_SPIN         "add_pct"
#define SUB_PCT_SPIN         "sub_pct" 
#define MULT_PCT_SPIN        "mult_pct" 
#define MAX_ADD_SPIN         "max_add_spin"
#define MAX_SUB_SPIN         "max_sub_spin"
#define MAX_MULT_SPIN        "max_mult_spin"
#define GUESS_PCT_SPIN       "guess_pct_spin"
#define CORRECT_PCT_SPIN     "correct_pct_spin"
#define GUESS_DEV_SPIN       "guess_dev_spin"
#define GUESS_TIME_SPIN      "guess_time_spin"
#define GUESS_TIME_DEV_SPIN  "guess_time_dev_spin"
#define TITLE_FONT_PICKER    "title_font_picker"
#define TEXT_FONT_PICKER     "text_font_picker"
#define TEXT_COLOR_PICKER    "text_color_picker"

/*
 * Any widget that is associated with a configuration element will store
 * object data using these labels so we can determine what configuration
 * data needs to be changed.
 */
#define CONFIG_SECTION   "config_section"
#define CONFIG_DATA_EL   "config_data_elemnt"

/*
 * The percent spin buttons for the operators all need to add up to
 * 100%.  For that reason, when one is adjusted, the others will need
 * to be adjusted also.  Since there are three percent values in
 * the equation, each percent spin button will need a first and second
 * choice to adjust when changing the values.
 */
#define FIRST_ADJUST    "first_adjust_choice"
#define SECOND_ADJUST   "second_adjust_choice"

static void
on_prop_dlg_destroy (GtkWidget   *widget,
                     gpointer     user_data)
{
   /*
    * Just drop any un-committed changes.
    */
   gnome_config_drop_all();
}


/*
 * on_int_spin_button_changed
 *
 * This is the generic handler for value changes.  It simply saves the 
 * new value for the item changed.  Other items may require special
 * processing besides this.
 */
static void
on_int_spin_button_changed (GtkWidget *widget,
                            gpointer   user_data)
{
   gchar *section;
   gchar *label;
   gint   value;

   section = gtk_object_get_data (GTK_OBJECT (widget), CONFIG_SECTION);
   label = gtk_object_get_data (GTK_OBJECT (widget), CONFIG_DATA_EL);
   g_assert (section != NULL);
   g_assert (label != NULL);

   value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));

   ini_set_int (PACKAGE, section, label, value);

   gnome_property_box_changed (GNOME_PROPERTY_BOX (user_data));
}


static void
on_pct_value_changed (GtkWidget *widget,
                      gpointer   user_data)
{
   GtkWidget   *first_pct_sb;
   GtkWidget   *second_pct_sb;
   gint         primary_pct;
   gint         first_adjust_pct;
   gint         second_adjust_pct;

   first_pct_sb = gtk_object_get_data (GTK_OBJECT (widget), FIRST_ADJUST);
   second_pct_sb = gtk_object_get_data (GTK_OBJECT (widget), SECOND_ADJUST);
   g_assert (first_pct_sb != NULL);
   g_assert (second_pct_sb != NULL);
   
   /*
    * Disable the change handlers for the scroll buttons we may end of
    * adjusting.
    */
   gtk_signal_handler_block_by_func (GTK_OBJECT (first_pct_sb),
                                     GTK_SIGNAL_FUNC (on_pct_value_changed),
                                     user_data);
   gtk_signal_handler_block_by_func (GTK_OBJECT (second_pct_sb),
                                     GTK_SIGNAL_FUNC (on_pct_value_changed),
                                     user_data);
   primary_pct = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
   first_adjust_pct = gtk_spin_button_get_value_as_int
      (GTK_SPIN_BUTTON (first_pct_sb));
   second_adjust_pct = gtk_spin_button_get_value_as_int
      (GTK_SPIN_BUTTON (second_pct_sb));

   /*
    * These items all need to add up to 100.  To get them back in line,
    * start with "first adjustment pct".  If it gets down to nothing, start
    * adjusting the "second adjustment pct".
    *
    * Note that since these values are stored in the config file, and
    * can be changed by the user, no assumptions should be made about
    * the three values adding up to 100 to begin with.
    */
   first_adjust_pct = 100 - (primary_pct + second_adjust_pct);
   if (first_adjust_pct < 0)
      {
      first_adjust_pct = 0;
      second_adjust_pct = 100 - primary_pct;
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (second_pct_sb), 
                                 (gfloat)second_adjust_pct);      
      on_int_spin_button_changed (second_pct_sb, user_data);
      }
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (first_pct_sb), 
                              (gfloat)first_adjust_pct);      
   on_int_spin_button_changed (first_pct_sb, user_data);
   g_assert ((primary_pct + first_adjust_pct + second_adjust_pct) == 100);
   
   gtk_signal_handler_unblock_by_func
      (GTK_OBJECT (first_pct_sb),
       GTK_SIGNAL_FUNC (on_pct_value_changed),
       user_data);
   gtk_signal_handler_unblock_by_func
      (GTK_OBJECT (second_pct_sb),
       GTK_SIGNAL_FUNC (on_pct_value_changed),
       user_data);

   on_int_spin_button_changed (widget, user_data);
}


static void
on_font_set (GnomeFontPicker *fontpicker,
             gchar *font_str,
             gpointer user_data)
{
   gchar *section;
   gchar *label;

   section = gtk_object_get_data (GTK_OBJECT (fontpicker), CONFIG_SECTION);
   label = gtk_object_get_data (GTK_OBJECT (fontpicker), CONFIG_DATA_EL);
   g_assert (section != NULL);
   g_assert (label != NULL);

   ini_set_string (PACKAGE, section, label, font_str);
   gnome_property_box_changed (GNOME_PROPERTY_BOX (user_data));
}


static void
on_text_color_set (GnomeColorPicker *colorpicker,
                   guint             red, 
                   guint             green, 
                   guint             blue, 
                   guint             alpha, 
                   gpointer          user_data)
{
   ini_set_int (PACKAGE, FONTS, FONT_COLOR_R, red);
   ini_set_int (PACKAGE, FONTS, FONT_COLOR_G, green);
   ini_set_int (PACKAGE, FONTS, FONT_COLOR_B, blue);
   gnome_property_box_changed (GNOME_PROPERTY_BOX (user_data));
}


static void
on_apply (GtkWidget *propertybox, 
          gint       page_num,
          gpointer   user_data)
{
   if (page_num == -1)
      {
      gnome_config_sync();
      setup_text_gc();
      }
}


static void
on_help (GtkWidget *propertybox,
         gint       page_num,
         gpointer   user_data)
{
   static GnomeHelpMenuEntry help_ref0 = { "MathWar", 
                                           "config.html#GENL-CONFIG" };
   static GnomeHelpMenuEntry help_ref1 = { "MathWar", "cp-config.html" };
   static GnomeHelpMenuEntry help_ref2 = { "MathWar", "oper-config.html" };

   switch (page_num)
      {
      case 0:
         gnome_help_display (NULL, &help_ref0);
         break;

      case 1:
         gnome_help_display (NULL, &help_ref1);
         break;

      case 2:
         gnome_help_display (NULL, &help_ref2);
         break;

      default:
         g_assert_not_reached ();
      }
}


static void 
create_general_tab (GtkWidget *prop_box, GtkWidget *parent)
{
   GtkWidget   *tab_lbl;
   GtkWidget   *table;
   GtkWidget   *label;
   GtkObject   *num_rounds_adj;
   GtkWidget   *num_rounds;
   GtkObject   *sec_per_round_adj;
   GtkWidget   *sec_per_round;
   GtkObject   *sec_btwn_round_adj;
   GtkWidget   *sec_btwn_round;
   GtkObject   *resp_delay_adj;
   GtkWidget   *resp_delay;
   GtkWidget   *title_fp;
   GtkWidget   *text_fp;
   GtkWidget   *text_cp;
   gchar       *strval;
   gint         value;
   gint         red;
   gint         green;
   gint         blue;
   GtkTooltips *tooltips;

   tooltips = gtk_object_get_data (GTK_OBJECT (parent), "tooltips");
   g_assert (tooltips != NULL);

   /*
    * Labels
    */
   tab_lbl = gtk_label_new (_("General"));
   gtk_widget_ref (tab_lbl);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             "general_tab_lbl",
                             tab_lbl,
                            (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (tab_lbl);

   table = gtk_table_new (5, 4, FALSE);
   gtk_widget_ref (table);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             "general_table",
                             table,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (table);
   gtk_container_set_border_width (GTK_CONTAINER (table), 5);
   gtk_table_set_row_spacings (GTK_TABLE (table), 3);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);

   label = gtk_label_new (_("Number of Rounds:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "num_round_lbl", 
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Seconds per Round:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "sec_round_lbl", 
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Title Text Font:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "title_text_lbl", 
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Text Font:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "text_lbl", 
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Text Color:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "text_color_lbl", 
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Seconds Between Rounds:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "btw_round_lbl", 
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Response Delay:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "btw_round_lbl", 
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 1, 2,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   /*
    * Spin buttons and associated adjustments.
    */
   value = ini_get_int (PACKAGE, BASIC_PARMS, NUM_ROUNDS, DEF_NUM_ROUNDS);
   num_rounds_adj = gtk_adjustment_new (value, 
                                        MIN_ROUNDS,
                                        MAX_ROUNDS,
                                        1, 10, 10);
   num_rounds = gtk_spin_button_new (GTK_ADJUSTMENT (num_rounds_adj), 1, 0);
   gtk_widget_ref (num_rounds);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             NUM_ROUNDS_SPIN,
                             num_rounds,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (num_rounds),
                        CONFIG_SECTION,
                        BASIC_PARMS);
   gtk_object_set_data (GTK_OBJECT (num_rounds),
                        CONFIG_DATA_EL,
                        NUM_ROUNDS);
   gtk_signal_connect (GTK_OBJECT (num_rounds), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), num_rounds, 1, 2, 0, 1,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (num_rounds);
   gtk_tooltips_set_tip (tooltips, num_rounds, 
                         _("The number of rounds in a game.  Changes will not take effect until the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, BASIC_PARMS, SECONDS, DEF_SECONDS);
   sec_per_round_adj = gtk_adjustment_new (value,
                                           MIN_SECS,
                                           MAX_SECS,
                                           1, 10, 10);
   sec_per_round = gtk_spin_button_new
      (GTK_ADJUSTMENT (sec_per_round_adj), 1, 0);
   gtk_widget_ref (sec_per_round);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             SEC_PER_ROUND_SPIN,
                             sec_per_round,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (sec_per_round),
                        CONFIG_SECTION,
                        BASIC_PARMS);
   gtk_object_set_data (GTK_OBJECT (sec_per_round),
                        CONFIG_DATA_EL,
                        SECONDS);
   gtk_signal_connect (GTK_OBJECT (sec_per_round), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), sec_per_round, 1, 2, 1, 2,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (sec_per_round);
   gtk_tooltips_set_tip (tooltips, sec_per_round, 
                         _("The number of seconds in a round.  Changes will not take effect until the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, BASIC_PARMS, SECONDS_BTWN, DEF_SECS_BTWN);
   sec_btwn_round_adj = gtk_adjustment_new (value,
                                            MIN_DEL_SECS,
                                            MAX_DEL_SECS,
                                            1, 10, 10);
   sec_btwn_round = gtk_spin_button_new
      (GTK_ADJUSTMENT (sec_btwn_round_adj), 1, 0);
   gtk_widget_ref (sec_btwn_round);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             SEC_BTWN_ROUND_SPIN,
                             sec_btwn_round,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (sec_btwn_round),
                        CONFIG_SECTION,
                        BASIC_PARMS);
   gtk_object_set_data (GTK_OBJECT (sec_btwn_round),
                        CONFIG_DATA_EL,
                        SECONDS_BTWN);
   gtk_signal_connect (GTK_OBJECT (sec_btwn_round), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), sec_btwn_round, 3, 4, 0, 1,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (sec_btwn_round);
   gtk_tooltips_set_tip (tooltips, sec_btwn_round, 
                         _("The number of seconds that the cards are shown face down before each round."),
                         NULL);

   value = ini_get_int (PACKAGE, BASIC_PARMS, RESP_DELAY, DEF_RESP_DELAY);
   resp_delay_adj = gtk_adjustment_new (value, MIN_DEL_SECS,
                                        MAX_DEL_SECS, 1, 10, 10);
   resp_delay = gtk_spin_button_new (GTK_ADJUSTMENT (resp_delay_adj), 1, 0);
   gtk_widget_ref (resp_delay);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             RESP_DELAY_SPIN,
                             resp_delay,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (resp_delay),
                        CONFIG_SECTION,
                        BASIC_PARMS);
   gtk_object_set_data (GTK_OBJECT (resp_delay),
                        CONFIG_DATA_EL,
                        RESP_DELAY);
   gtk_signal_connect (GTK_OBJECT (resp_delay), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), resp_delay, 3, 4, 1, 2,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (resp_delay);
   gtk_tooltips_set_tip (tooltips, resp_delay, 
                         _("The number of seconds that most messages are displayed.  This only applies to messages that do not require any input from the player."),
                         NULL);

   /*
    * Font pickers
    */
   title_fp = gnome_font_picker_new();
   gtk_widget_ref (title_fp);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             TITLE_FONT_PICKER,
                             title_fp,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_signal_connect (GTK_OBJECT (title_fp), "font-set",
                       GTK_SIGNAL_FUNC (on_font_set),
                       prop_box);
   gtk_object_set_data (GTK_OBJECT (title_fp), CONFIG_SECTION, FONTS);
   gtk_object_set_data (GTK_OBJECT (title_fp), CONFIG_DATA_EL, TITLE_FONT);
   gtk_table_attach (GTK_TABLE (table), title_fp, 1, 4, 2, 3,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gnome_font_picker_set_mode (GNOME_FONT_PICKER (title_fp), 
                               GNOME_FONT_PICKER_MODE_FONT_INFO);
   strval = ini_get_string (PACKAGE, FONTS, TITLE_FONT, DEF_TITLE_FONT);
   gnome_font_picker_set_font_name (GNOME_FONT_PICKER (title_fp), strval);
   g_free (strval);
   gtk_widget_show (title_fp);
   gtk_tooltips_set_tip (tooltips, title_fp, 
                         _("Choose the font used to dispaly title text.  Changes take effect the next time text is drawn."),
                         NULL);

   text_fp = gnome_font_picker_new();
   gtk_widget_ref (text_fp);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             TEXT_FONT_PICKER,
                             text_fp,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_signal_connect (GTK_OBJECT (text_fp), "font-set",
                       GTK_SIGNAL_FUNC (on_font_set),
                       prop_box);
   gtk_object_set_data (GTK_OBJECT (text_fp), CONFIG_SECTION, FONTS);
   gtk_object_set_data (GTK_OBJECT (text_fp), CONFIG_DATA_EL, TEXT_FONT);
   gtk_table_attach (GTK_TABLE (table), text_fp, 1, 4, 3, 4,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gnome_font_picker_set_mode (GNOME_FONT_PICKER (text_fp), 
                               GNOME_FONT_PICKER_MODE_FONT_INFO);
   strval = ini_get_string (PACKAGE, FONTS, TEXT_FONT, DEF_TEXT_FONT);
   gnome_font_picker_set_font_name (GNOME_FONT_PICKER (text_fp), strval);
   g_free (strval);
   gtk_widget_show (text_fp);
   gtk_tooltips_set_tip (tooltips, text_fp, 
                         _("Choose the font used to dispaly normal text.  Changes take effect the next time text is drawn."),
                         NULL);

   /*
    * Color Picker
    */
   text_cp = gnome_color_picker_new ();
   gtk_widget_ref (text_cp);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             TEXT_COLOR_PICKER,
                             text_cp,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_table_attach (GTK_TABLE (table), text_cp, 1, 4, 4, 5,
                     (GtkAttachOptions) (GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   red = ini_get_int (PACKAGE, FONTS, FONT_COLOR_R, DEF_RED);
   green = ini_get_int (PACKAGE, FONTS, FONT_COLOR_G, DEF_GREEN);
   blue = ini_get_int (PACKAGE, FONTS, FONT_COLOR_B, DEF_BLUE);
   gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (text_cp),
                               red, green, blue, DEF_ALPHA);
   gtk_signal_connect (GTK_OBJECT (text_cp), "color-set",
                       GTK_SIGNAL_FUNC (on_text_color_set),
                       prop_box);
   gtk_widget_show (text_cp);
   gtk_tooltips_set_tip (tooltips, text_cp, 
                         _("Choose the color used for drawing text.  Changes take effect the next time text is drawn."),
                         NULL);

   gnome_property_box_append_page (GNOME_PROPERTY_BOX (prop_box),
                                   table,
                                   tab_lbl);
}


static void 
create_comp_player_tab (GtkWidget *prop_box, GtkWidget *parent)
{
   GtkWidget   *tab_lbl;
   GtkWidget   *table;
   GtkWidget   *label;
   GtkObject   *adjustment;
   GtkWidget   *spinner;
   gint         value;
   GtkTooltips *tooltips;

   tooltips = gtk_object_get_data (GTK_OBJECT (parent), "tooltips");
   g_assert (tooltips != NULL);

   tab_lbl = gtk_label_new (_("Computer Player"));
   gtk_widget_ref (tab_lbl);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             "comp_player_tab_lbl", 
                             tab_lbl,
                            (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (tab_lbl);

   table = gtk_table_new (3, 4, FALSE);
   gtk_widget_ref (table);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "comp_player_table",
                             table,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_container_set_border_width (GTK_CONTAINER (table), 5);
   gtk_table_set_row_spacings (GTK_TABLE (table), 3);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_widget_show (table);

   label = gtk_label_new (_("Computer Guess Percentage:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "comp_guess_pct_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Percent Guessed Correctly:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "guess_correct_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Maximum Guess Deviation:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "max_guess_dev_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Guess Time:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "guess_time_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Guess Time Deviation:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "guess_time_dev_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 1, 2,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   value = ini_get_int (PACKAGE, COMP_PLAYER, GUESS_PCT, DEF_GUESS_PCT);
   adjustment = gtk_adjustment_new (value,
                                    MIN_PCT,
                                    MAX_PCT,
                                    1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             GUESS_PCT_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        COMP_PLAYER);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        GUESS_PCT);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 1, 2, 0, 1,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("The average percent of rounds that the computer will make a guess (assuming the human player does not guess first).  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, COMP_PLAYER, 
                        RIGHT_PCT, DEF_RIGHT_PCT);
   adjustment = gtk_adjustment_new (value,
                                    MIN_PCT,
                                    MAX_PCT,
                                    1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             CORRECT_PCT_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        COMP_PLAYER);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        RIGHT_PCT);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 1, 2, 1, 2,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("The average percent of guesses that the computer will be correct.  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, COMP_PLAYER, GUESS_DEV, DEF_GUESS_DEV);
   adjustment = gtk_adjustment_new (value,
                                    MIN_DEV_VALUE,
                                    MAX_DEV_VALUE,
                                    1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             GUESS_DEV_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        COMP_PLAYER);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        GUESS_DEV);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 1, 2, 2, 3,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("Creates a window that the computer might be incorrect by when making an incorrect guess.  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, COMP_PLAYER, GUESS_TIME, DEF_TIME);
   adjustment = gtk_adjustment_new (value, MIN_PCT, MAX_PCT, 1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             GUESS_TIME_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        COMP_PLAYER);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        GUESS_TIME);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 3, 4, 0, 1,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("Average time that the computer player will wait before making a guess, expressed as a percent of the round time.  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, COMP_PLAYER, MAX_DEVIATION, DEF_MAX_DEV);
   adjustment = gtk_adjustment_new (value,
                                    MIN_DEV_VALUE,
                                    MAX_DEV_VALUE,
                                    1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             GUESS_TIME_DEV_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        COMP_PLAYER);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        MAX_DEVIATION);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 3, 4, 1, 2,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("Creates a window within which the computer player will guess.  Changes take effect in the next new game."),
                         NULL);

   gnome_property_box_append_page (GNOME_PROPERTY_BOX (prop_box), 
                                   table, 
                                   tab_lbl);
}


static void 
create_operator_tab (GtkWidget *prop_box, GtkWidget *parent)
{
   GtkWidget   *tab_lbl;
   GtkWidget   *table;
   GtkWidget   *label;
   GtkObject   *adjustment;
   GtkWidget   *spinner;
   GtkWidget   *add_spinner;
   GtkWidget   *sub_spinner;
   GtkWidget   *mult_spinner;
   gint         value;
   GtkTooltips *tooltips;

   tooltips = gtk_object_get_data (GTK_OBJECT (parent), "tooltips");
   g_assert (tooltips != NULL);

   tab_lbl = gtk_label_new (_("Operators"));
   gtk_widget_ref (tab_lbl);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             "operator_tab_lbl", 
                             tab_lbl,
                            (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (tab_lbl);

   table = gtk_table_new (3, 4, FALSE);
   gtk_widget_ref (table);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "operator_table",
                             table,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_container_set_border_width (GTK_CONTAINER (table), 5);
   gtk_table_set_row_spacings (GTK_TABLE (table), 3);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_widget_show (table);

   label = gtk_label_new (_("Addition Percent:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "add_pct_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Subtraction Percent:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "sub_pct_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Multiplication Percent:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "mult_pct_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Max Addition Number:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "add_num_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Max Subtraction Number:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "sub_num_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 1, 2,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   label = gtk_label_new (_("Max Multiplication Number:"));
   gtk_widget_ref (label);
   gtk_object_set_data_full (GTK_OBJECT (prop_box),
                             "mult_num_lbl",
                             label,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_widget_show (label);
   gtk_table_attach (GTK_TABLE (table), label, 2, 3, 2, 3,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
   gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

   value = ini_get_int (PACKAGE, OPERATORS, ADD_PCT, DEF_ADD_PCT);
   adjustment = gtk_adjustment_new (value, MIN_PCT, MAX_PCT, 1, 10, 10);
   add_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (add_spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             ADD_PCT_SPIN,
                             add_spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (add_spinner),
                        CONFIG_SECTION,
                        OPERATORS);
   gtk_object_set_data (GTK_OBJECT (add_spinner),
                        CONFIG_DATA_EL,
                        ADD_PCT);
   gtk_signal_connect (GTK_OBJECT (add_spinner), "changed",
                       GTK_SIGNAL_FUNC (on_pct_value_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), add_spinner, 1, 2, 0, 1,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (add_spinner);
   gtk_tooltips_set_tip (tooltips, add_spinner, 
                         _("The percent of rounds that will be addition problems.  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, OPERATORS, SUB_PCT, DEF_SUB_PCT);
   adjustment = gtk_adjustment_new (value, MIN_PCT, MAX_PCT, 1, 10, 10);
   sub_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (sub_spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             SUB_PCT_SPIN,
                             sub_spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (sub_spinner),
                        CONFIG_SECTION,
                        OPERATORS);
   gtk_object_set_data (GTK_OBJECT (sub_spinner),
                        CONFIG_DATA_EL,
                        SUB_PCT);
   gtk_signal_connect (GTK_OBJECT (sub_spinner), "changed",
                       GTK_SIGNAL_FUNC (on_pct_value_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), sub_spinner, 1, 2, 1, 2,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (sub_spinner);
   gtk_tooltips_set_tip (tooltips, sub_spinner, 
                         _("The percent of rounds that will be subtraction problems.  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, OPERATORS, MULT_PCT, DEF_MULT_PCT);
   adjustment = gtk_adjustment_new (value, MIN_PCT, MAX_PCT, 1, 10, 10);
   mult_spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (mult_spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             MULT_PCT_SPIN,
                             mult_spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (mult_spinner),
                        CONFIG_SECTION,
                        OPERATORS);
   gtk_object_set_data (GTK_OBJECT (mult_spinner),
                        CONFIG_DATA_EL,
                        MULT_PCT);
   gtk_signal_connect (GTK_OBJECT (mult_spinner), "changed",
                       GTK_SIGNAL_FUNC (on_pct_value_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), mult_spinner, 1, 2, 2, 3,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (mult_spinner);
   gtk_tooltips_set_tip (tooltips, mult_spinner, 
                         _("The percent of rounds that will be multiplication problems.  Changes take effect in the next new game."),
                         NULL);

   /*
    * Now that all the pct spinners have been created, we need to link 
    * them so that each one knows its first and second choices to 
    * adjust when their own values change.
    *
    * The scheme should be cyclic, top to bottom, as on the screen.
    */
   gtk_object_set_data (GTK_OBJECT (add_spinner), 
                        FIRST_ADJUST, sub_spinner);
   gtk_object_set_data (GTK_OBJECT (add_spinner),
                        SECOND_ADJUST, mult_spinner);
   gtk_object_set_data (GTK_OBJECT (sub_spinner), 
                        FIRST_ADJUST, mult_spinner);
   gtk_object_set_data (GTK_OBJECT (sub_spinner),
                        SECOND_ADJUST, add_spinner);
   gtk_object_set_data (GTK_OBJECT (mult_spinner), 
                        FIRST_ADJUST, add_spinner);
   gtk_object_set_data (GTK_OBJECT (mult_spinner),
                        SECOND_ADJUST, sub_spinner);

   value = ini_get_int (PACKAGE, OPERATORS, ADD_MAX, DEF_ADD_MAX);
   adjustment = gtk_adjustment_new (value, MIN_NUMBER, MAX_NUMBER, 1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             MAX_ADD_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        OPERATORS);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        ADD_MAX);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 3, 4, 0, 1,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("The largest card value that will be used for addition problems.  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, OPERATORS, SUB_MAX, DEF_SUB_MAX);
   adjustment = gtk_adjustment_new (value, MIN_NUMBER, MAX_NUMBER, 1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             MAX_SUB_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        OPERATORS);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        SUB_MAX);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 3, 4, 1, 2,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("The largest card value that will be used for subtraction problems.  Changes take effect in the next new game."),
                         NULL);

   value = ini_get_int (PACKAGE, OPERATORS, MULT_MAX, DEF_MULT_MAX);
   adjustment = gtk_adjustment_new (value, MIN_NUMBER, MAX_NUMBER, 1, 10, 10);
   spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
   gtk_widget_ref (spinner);
   gtk_object_set_data_full (GTK_OBJECT (prop_box), 
                             MAX_MULT_SPIN,
                             spinner,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_SECTION,
                        OPERATORS);
   gtk_object_set_data (GTK_OBJECT (spinner),
                        CONFIG_DATA_EL,
                        MULT_MAX);
   gtk_signal_connect (GTK_OBJECT (spinner), "changed",
                       GTK_SIGNAL_FUNC (on_int_spin_button_changed),
                       prop_box);
   gtk_table_attach (GTK_TABLE (table), spinner, 3, 4, 2, 3,
                     (GtkAttachOptions) (0),
                     (GtkAttachOptions) (0), 0, 0);
   gtk_widget_show (spinner);
   gtk_tooltips_set_tip (tooltips, spinner, 
                         _("The largest card value that will be used for multiplication problems.  Changes take effect in the next new game."),
                         NULL);

   gnome_property_box_append_page (GNOME_PROPERTY_BOX (prop_box), 
                                   table, 
                                   tab_lbl);
}


static GtkWidget *
create_prop_dlg (GtkWidget *parent)
{
   GtkWidget  *prop_dlg;

   prop_dlg = gnome_property_box_new();

   create_general_tab (prop_dlg, parent);
   create_comp_player_tab (prop_dlg, parent);
   create_operator_tab (prop_dlg, parent);
   gnome_dialog_set_parent (GNOME_DIALOG (prop_dlg), GTK_WINDOW (parent));

   return prop_dlg;
}


void
show_prop_dlg (GtkWidget *parent)
{
   static GtkWidget  *prop_dlg = NULL;

   /*
    * If the dialog box exists, just make sure it is visable, otherwise
    * create it.
    */
   if (prop_dlg)
      {
      gdk_window_show (prop_dlg->window);
      gdk_window_raise (prop_dlg->window);
      }
   else
      {
      prop_dlg = create_prop_dlg (parent);

      /*
       * When the dialog is destroyed, call gtk_widget_destroyed(), which
       * will set the "user_data" (&prop_dlg in this case), to NULL.
       */
      gtk_signal_connect (GTK_OBJECT (prop_dlg),
                          "destroy",
                          GTK_SIGNAL_FUNC (gtk_widget_destroyed),
                          &prop_dlg);
      gtk_signal_connect (GTK_OBJECT (prop_dlg),
                          "destroy",
                          GTK_SIGNAL_FUNC (on_prop_dlg_destroy),
                          NULL);
      gtk_signal_connect (GTK_OBJECT (prop_dlg),
                          "apply",
                          GTK_SIGNAL_FUNC (on_apply),
                          NULL);
      gtk_signal_connect (GTK_OBJECT (prop_dlg),
                          "help",
                          GTK_SIGNAL_FUNC (on_help),
                          NULL);

      gtk_widget_show (prop_dlg);
      }
}
