/* Ada language support definitions for GDB, the GNU debugger.

   Copyright (C) 1992, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005,
   2007, 2008, 2009, 2010 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#if !defined (ADA_LANG_H)
#define ADA_LANG_H 1

struct partial_symbol;
struct frame_info;

#include "value.h"
#include "gdbtypes.h"
#include "breakpoint.h"

/* Names of specific files known to be part of the runtime
   system and that might consider (confusing) debugging information.
   Each name (a basic regular expression string) is followed by a
   comma.  FIXME: Should be part of a configuration file. */
#if defined(__alpha__) && defined(__osf__)
#define ADA_KNOWN_RUNTIME_FILE_NAME_PATTERNS \
   "^[agis]-.*\\.ad[bs]$", \
   "/usr/shlib/libpthread\\.so",
#elif defined (__linux__)
#define ADA_KNOWN_RUNTIME_FILE_NAME_PATTERNS \
   "^[agis]-.*\\.ad[bs]$", \
   "/lib.*/libpthread\\.so[.0-9]*$", "/lib.*/libpthread\\.a$", \
   "/lib.*/libc\\.so[.0-9]*$", "/lib.*/libc\\.a$",
#endif

#if !defined (ADA_KNOWN_RUNTIME_FILE_NAME_PATTERNS)
#define ADA_KNOWN_RUNTIME_FILE_NAME_PATTERNS \
   "^[agis]-.*\\.ad[bs]$",
#endif

/* Names of compiler-generated auxiliary functions probably of no
   interest to users. Each name (a basic regular expression string)
   is followed by a comma. */
#define ADA_KNOWN_AUXILIARY_FUNCTION_NAME_PATTERNS \
   "___clean[.$a-zA-Z0-9_]*$",

/* The maximum number of frame levels searched for non-local,
 * non-global symbols.  This limit exists as a precaution to prevent
 * infinite search loops when the stack is screwed up. */
#define MAX_ENCLOSING_FRAME_LEVELS 7

/* Maximum number of steps followed in looking for the ultimate
   referent of a renaming.  This prevents certain infinite loops that
   can otherwise result. */
#define MAX_RENAMING_CHAIN_LENGTH 10

struct block;

/* Corresponding encoded/decoded names and opcodes for Ada user-definable
   operators. */
struct ada_opname_map
{
  const char *encoded;
  const char *decoded;
  enum exp_opcode op;
};

/* Table of Ada operators in encoded and decoded forms. */
/* Defined in ada-lang.c */
extern const struct ada_opname_map ada_opname_table[];

/* A tuple, (symbol, block), representing one instance of a 
 * symbol-lookup operation. */
struct ada_symbol_info {
  struct symbol* sym;
  struct block* block;
};

/* Denotes a type of renaming symbol (see ada_parse_renaming).  */
enum ada_renaming_category
  {
    /* Indicates a symbol that does not encode a renaming.  */
    ADA_NOT_RENAMING,

    /* For symbols declared
         Foo : TYPE renamed OBJECT;  */
    ADA_OBJECT_RENAMING,

    /* For symbols declared
         Foo : exception renames EXCEPTION;  */
    ADA_EXCEPTION_RENAMING,
    /* For packages declared
          package Foo renames PACKAGE; */
    ADA_PACKAGE_RENAMING,
    /* For subprograms declared
          SUBPROGRAM_SPEC renames SUBPROGRAM;
       (Currently not used).  */
    ADA_SUBPROGRAM_RENAMING
  };

/* Ada task structures.  */

struct ada_task_info
{
  /* The PTID of the thread that this task runs on.  This ptid is computed
     in a target-dependent way from the associated Task Control Block.  */
  ptid_t ptid;

  /* The ID of the task.  */
  CORE_ADDR task_id;

  /* The name of the task.  */
  char name[257];

  /* The current state of the task.  */
  int state;

  /* The priority associated to the task.  */
  int priority;

  /* If non-zero, the task ID of the parent task.  */
  CORE_ADDR parent;

  /* If the task is waiting on a task entry, this field contains
     the ID of the other task.  Zero otherwise.  */
  CORE_ADDR called_task;

  /* If the task is accepting a rendezvous with another task, this field
     contains the ID of the calling task.  Zero otherwise.  */
  CORE_ADDR caller_task;
};

/* Assuming V points to an array of S objects,  make sure that it contains at
   least M objects, updating V and S as necessary. */

#define GROW_VECT(v, s, m)                                              \
   if ((s) < (m)) (v) = grow_vect (v, &(s), m, sizeof *(v));

extern void *grow_vect (void *, size_t *, size_t, int);

extern int ada_get_field_index (const struct type *type,
                                const char *field_name,
                                int maybe_missing);

extern int ada_parse (void);    /* Defined in ada-exp.y */

extern void ada_error (char *); /* Defined in ada-exp.y */

                        /* Defined in ada-typeprint.c */
extern void ada_print_type (struct type *, char *, struct ui_file *, int,
                            int);

extern int ada_val_print (struct type *, const gdb_byte *, int, CORE_ADDR,
                          struct ui_file *, int,
			  const struct value_print_options *);

extern int ada_value_print (struct value *, struct ui_file *,
			    const struct value_print_options *);

                                /* Defined in ada-lang.c */

extern void ada_emit_char (int, struct type *, struct ui_file *, int, int);

extern void ada_printchar (int, struct type *, struct ui_file *);

extern void ada_printstr (struct ui_file *, struct type *, const gdb_byte *,
			  unsigned int, const char *, int,
			  const struct value_print_options *);

struct value *ada_convert_actual (struct value *actual,
                                  struct type *formal_type0,
				  struct gdbarch *gdbarch,
                                  CORE_ADDR *sp);

extern struct value *ada_value_subscript (struct value *, int,
                                          struct value **);

extern struct type *ada_array_element_type (struct type *, int);

extern int ada_array_arity (struct type *);

struct type *ada_type_of_array (struct value *, int);

extern struct value *ada_coerce_to_simple_array_ptr (struct value *);

extern int ada_is_simple_array_type (struct type *);

extern int ada_is_array_descriptor_type (struct type *);

extern int ada_is_bogus_array_descriptor (struct type *);

extern LONGEST ada_discrete_type_low_bound (struct type *);

extern LONGEST ada_discrete_type_high_bound (struct type *);

extern char *ada_decode_symbol (const struct general_symbol_info*);

extern const char *ada_decode (const char*);

extern enum language ada_update_initial_language (enum language, 
						  struct partial_symtab*);

extern void clear_ada_sym_cache (void);

extern int ada_lookup_symbol_list (const char *, const struct block *,
                                   domain_enum, struct ada_symbol_info**);

extern char *ada_fold_name (const char *);

extern struct symbol *ada_lookup_symbol (const char *, const struct block *,
                                         domain_enum, int *);

extern struct symbol *
ada_lookup_encoded_symbol (const char *, const struct block *,
			   domain_enum namespace, struct block **);

extern struct minimal_symbol *ada_lookup_simple_minsym (const char *);

extern void ada_fill_in_ada_prototype (struct symbol *);

extern int user_select_syms (struct ada_symbol_info *, int, int);

extern int get_selections (int *, int, int, int, char *);

extern char *ada_start_decode_line_1 (char *);

extern struct symtabs_and_lines ada_finish_decode_line_1 (char **,
                                                          struct symtab *,
                                                          int, char ***);

extern struct symtabs_and_lines ada_sals_for_line (const char*, int,
						   int, char***, int);

extern int ada_scan_number (const char *, int, LONGEST *, int *);

extern struct type *ada_parent_type (struct type *);

extern int ada_is_ignored_field (struct type *, int);

extern int ada_is_constrained_packed_array_type (struct type *);

extern struct value *ada_value_primitive_packed_val (struct value *,
						     const gdb_byte *,
                                                     long, int, int,
                                                     struct type *);

extern struct type *ada_coerce_to_simple_array_type (struct type *);

extern int ada_is_character_type (struct type *);

extern int ada_is_string_type (struct type *);

extern int ada_is_tagged_type (struct type *, int);

extern int ada_is_tag_type (struct type *);

extern struct type *ada_tag_type (struct value *);

extern struct value *ada_value_tag (struct value *);

extern const char *ada_tag_name (struct value *);

extern int ada_is_parent_field (struct type *, int);

extern int ada_is_wrapper_field (struct type *, int);

extern int ada_is_variant_part (struct type *, int);

extern struct type *ada_variant_discrim_type (struct type *, struct type *);

extern int ada_is_others_clause (struct type *, int);

extern int ada_in_variant (LONGEST, struct type *, int);

extern char *ada_variant_discrim_name (struct type *);

extern struct value *ada_value_struct_elt (struct value *, char *, int);

extern int ada_is_aligner_type (struct type *);

extern struct type *ada_aligned_type (struct type *);

extern const gdb_byte *ada_aligned_value_addr (struct type *,
					       const gdb_byte *);

extern const char *ada_attribute_name (enum exp_opcode);

extern int ada_is_fixed_point_type (struct type *);

extern int ada_is_system_address_type (struct type *);

extern DOUBLEST ada_delta (struct type *);

extern DOUBLEST ada_fixed_to_float (struct type *, LONGEST);

extern LONGEST ada_float_to_fixed (struct type *, DOUBLEST);

extern struct type *ada_system_address_type (void);

extern int ada_which_variant_applies (struct type *, struct type *,
				      const gdb_byte *);

extern struct type *ada_to_fixed_type (struct type *, const gdb_byte *,
				       CORE_ADDR, struct value *,
                                       int check_tag);

extern struct type *ada_template_to_fixed_record_type_1 (struct type *type,
							 const gdb_byte *valaddr,
							 CORE_ADDR address,
							 struct value *dval0,
							 int keep_dynamic_fields);

extern int ada_name_prefix_len (const char *);

extern char *ada_type_name (struct type *);

extern struct type *ada_find_parallel_type (struct type *,
                                            const char *suffix);

extern LONGEST get_int_var_value (char *, int *);

extern struct symbol *ada_find_any_symbol (const char *name);

extern struct type *ada_find_any_type (const char *name);

extern struct symbol *ada_find_renaming_symbol (const char *name,
                                                struct block *block);

extern int ada_prefer_type (struct type *, struct type *);

extern struct type *ada_get_base_type (struct type *);

extern struct type *ada_check_typedef (struct type *);

extern char *ada_encode (const char *);

extern const char *ada_enum_name (const char *);

extern int ada_is_modular_type (struct type *);

extern ULONGEST ada_modulus (struct type *);

extern struct value *ada_value_ind (struct value *);

extern void ada_print_scalar (struct type *, LONGEST, struct ui_file *);

extern int ada_is_range_type_name (const char *);

extern enum ada_renaming_category ada_parse_renaming (struct symbol *,
						      const char **,
						      int *, const char **);

extern void ada_find_printable_frame (struct frame_info *fi);

extern char *ada_breakpoint_rewrite (char *, int *);

extern char *ada_main_name (void);

/* Tasking-related: ada-tasks.c */

extern int valid_task_id (int);

extern int ada_get_task_number (ptid_t);

extern int ada_build_task_list (int warn_if_null);

extern int ada_exception_catchpoint_p (struct breakpoint *b);
  
extern struct symtab_and_line
  ada_decode_exception_location (char *args, char **addr_string,
                                 char **exp_string, char **cond_string,
                                 struct expression **cond,
                                 struct breakpoint_ops **ops);

extern struct symtab_and_line
  ada_decode_assert_location (char *args, char **addr_string,
                              struct breakpoint_ops **ops);


#endif
