/*
** Copyright 2001 Double Precision, Inc.
** See COPYING for distribution information.
**
*/
#include "config.h"
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>
#include <libintl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/hdreg.h>
#include <linux/cdrom.h>
#include "superfloppy.h"

static const char rcsid[]="$Id: idefloppy.c,v 1.1 2001/02/13 00:15:38 mrsam Exp $";

#define IDEFLOPPY_IOCTL_FORMAT_SUPPORTED	0x4600
#define IDEFLOPPY_IOCTL_FORMAT_GET_CAPACITY	0x4601
#define IDEFLOPPY_IOCTL_FORMAT_START		0x4602
#define IDEFLOPPY_IOCTL_FORMAT_GET_PROGRESS	0x4603

struct idefloppy_format_capacities {
	int nformats;
	struct {
		int nblocks;
		int blocksize;
	} formats[16];
} ;

struct idefloppy_format_cmd {
	int nblocks;
	int blocksize;
	int flags;
	} ;

static int do_format(const char *, int, int,
		     int (*fmt_func)(const char *, int), int);

int idefloppy_capacity(const char *dev, int (*fmt_func)(const char *, int),
		       int flags)
{
	static struct idefloppy_format_capacities caps;
	int fd=open(dev, O_RDONLY|O_NDELAY);
	int i;

	if (fd < 0)
	{
		perror(dev);
		return (1);
	}

	caps.nformats=sizeof(caps.formats)/sizeof(caps.formats[0]);

	if (ioctl(fd, IDEFLOPPY_IOCTL_FORMAT_GET_CAPACITY, &caps) < 0)
	{
		perror(dev);
		return (1);
	}

	if (flags & FLAGS_FORMAT)
	{
		/* Take the first capacity, and format */

		close(fd);
		if (caps.nformats <= 0)
			return (1);
		return (do_format(dev, caps.formats[0].nblocks,
				  caps.formats[0].blocksize,
				  fmt_func, flags));
	}

	for (i=0; i<caps.nformats; i++)
	{
		int nblocks=caps.formats[i].nblocks;
		int blocksize=caps.formats[i].blocksize;
		char fmtbuf[200];

		/*
		** Try to factor nblocks into tracks x sectors value
		** simply by trying a couple of well known sectors per track
		** values, until it fits.  If we fail, just use 1.
		*/

		static const int try_secs[]={36, 30, 22, 64};

		int j;
		int sec_size=1;

		for (j=0; j<sizeof(try_secs)/sizeof(try_secs[0]); j++)
			if ((nblocks % try_secs[j]) == 0)
			{
				sec_size=try_secs[j];
				break;
			}

		sprintf(fmtbuf, "%dx%dx%d", nblocks / sec_size, sec_size,
			blocksize);

		if (flags & FLAGS_BRIEF)
			printf("%s\n", fmtbuf);
		else
			printf("  %-15s (%s)\n", fmtbuf,
			       fmtsize(nblocks, blocksize));
	}
	return (0);
}

static void print_format_progress(int n, int flags)
{
	long nn = (long)n * 100 / 65536;

	if (flags & FLAGS_BRIEF)
		printf("%d\n", (int)nn);
	else
		printf("\b\b\b\b%3d%%", (int)nn);
	fflush(stdout);
}

int idefloppy_probe(const char *dev, int *foundflag)
{
	int fd=open(dev, O_RDONLY|O_NDELAY);
	struct hd_driveid hdid;

	if (foundflag)
		*foundflag=0;

	if (fd < 0)
		return (0);

	if (ioctl(fd, IDEFLOPPY_IOCTL_FORMAT_SUPPORTED, 0) < 0)
	{
		close(fd);
		return (0);
	}

	if (ioctl(fd, HDIO_GET_IDENTITY, &hdid) < 0)
	{
		perror(dev);
		close(fd);
		return (-1);
	}

	if (foundflag)
	{
		*foundflag=1;
	}
	else
	{
		printf(_("idefloppy %s: "), dev);

#define GET_STRING(x) char c[sizeof(x)+1]; memcpy(c, (x), sizeof(c)); \
			c[sizeof(x)]=0;
		
		{
			GET_STRING(hdid.model);

			printf("%s\n", c[0] ? c:_("Unknown model"));
		}

		{
			GET_STRING(hdid.fw_rev);
			if (c[0])
				printf(_("  Revision: %s\n"), c);
		}
		
		{
			GET_STRING(hdid.serial_no);
			if (c[0])
				printf(_("  Serial number: %s\n"), c);
		}
	}
	close(fd);
	return (0);
}

int idefloppy_format(const char *dev, const char *fmt,
		     int (*fmt_func)(const char *, int), int flags)
{
	const char *p=strchr(fmt, 'x');
	int blocks;

	if (p)
	{
		blocks=atoi(fmt) * atoi(p+1);
		p=strchr(p+1, 'x');
		if (p)
			return (do_format(dev, blocks, atoi(p+1),
					  fmt_func, flags));
	}

	fprintf(stderr, _("%s: invalid format."), fmt);
	return (1);
}

static int do_format(const char *dev, int b, int bs,
		     int (*fmt_func)(const char *, int),
		     int flags)
{
	static struct idefloppy_format_cmd ifc;

	int fd;
	int fudge;

	ifc.nblocks=b;
	ifc.blocksize=bs;
	ifc.flags= flags & FLAGS_FMTVERIFY ? 1:0;

	fd=open(dev, O_RDWR|O_NDELAY);

	if (fd < 0)
	{
		perror(dev);
		return (1);
	}

	if (!(flags & FLAGS_BRIEF))
	{
		if (kilobytes(ifc.nblocks, ifc.blocksize) >= 4096)
		{
			int yn, c;
			char *y=_("Yy");

			printf(_("WARNING: formatting high-density media may damage it.  Check the manual\nfor your floppy drive for more information.  Do you want to proceed? (N): "));

			c=getchar();
			for (yn=c; c != EOF && c != '\n'; c=getchar())
				;

			if (strchr(y, yn) == NULL)
			{
				return (0);
			}
		}
		printf(_("Formatting %s...     "), fmtsize(ifc.nblocks,
							   ifc.blocksize));
		fflush(stdout);
	}

	if (ioctl(fd, IDEFLOPPY_IOCTL_FORMAT_START, &ifc) < 0)
	{
		perror("ioctl");
		return (1);
	}

	fudge=100;
	if (fmt_func)
		fudge=95;

	for (;;)
	{
		int progress;

		if (ioctl(fd, IDEFLOPPY_IOCTL_FORMAT_GET_PROGRESS,
			  &progress) < 0)
			break;

		if (progress >= 65536)
			break;

		progress= progress * fudge / 100;

		if (flags & FLAGS_FMTIDEVERIFY)
			progress /= 2;

		print_format_progress(progress, flags);
		sleep(3);
	}

	close(fd);

	if (flags & FLAGS_FMTIDEVERIFY)
	{
		char *buf;
		int todo;

		fd=open(dev, O_RDONLY);
		if (fd < 0)
		{
			perror(dev);
			return (1);
		}

		buf=malloc(512 * 36);
		if (!buf)
		{
			close(fd);
			perror("malloc");
		}

		todo=b;
		while (todo > 0)
		{
			int n=36;

			if (n > todo)
				n=todo;

			errno=EIO;
			n=read(fd, buf, n * 512);
			if (n <= 0 || (n % 512) != 0)
			{
				free(buf);
				perror("read");
				close(fd);
				return (1);
			}

			todo -= n / 512;

			n = fudge - todo * fudge / b / 2;

			if (flags & FLAGS_BRIEF)
				printf("%d\n", n);
			else
				printf("\b\b\b\b%3d%%", n);
			fflush(stdout);
		}
		free(buf);
		close(fd);
	}
	if (fmt_func)
	{
		int i;

		i=(*fmt_func)(dev, flags);
		if (i)
		{
			printf("\n");
			return (i);
		}
	}
	print_format_progress(65536, flags);
	printf("\n");
	if (flags & FLAGS_EJECT)
		return (idefloppy_eject(dev, flags));
	return (0);
}

int idefloppy_eject(const char *dev, int flags)
{
	int fd=open(dev, O_RDONLY);

	if (fd < 0)
	{
		if (!(flags & FLAGS_BRIEF))
		{
			perror(dev);
			return (1);
		}
		return (0);
	}

	ioctl(fd, CDROMEJECT, NULL);
	close(fd);
	return (0);
}
