/*
 * @(#)ClassRecordIOUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the 
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.StringReader;
import java.io.StringWriter;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.compiler.ModifiedMethod;


/**
 * Tests the ClassRecordIO class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 22, 2003
 */
public class ClassRecordIOUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = ClassRecordIOUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public ClassRecordIOUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testWriteClass1() throws Exception
    {
        StringWriter sw = new StringWriter();
        ModifiedMethod mm = createModifiedMethod( THIS_CLASS, 0 );
        ClassRecord cr = createClassRecord( THIS_CLASS, mm );
        ClassRecordIO crio = new ClassRecordIO();
        
        crio.writeClass( cr, sw );
        String out = sw.toString();
        
        assertTrue(
            "Did not write any data.",
            out.length() > 0 );
        assertEquals(
            "Did not write the correct data.",
            THIS_CLASS.getName().length()+":"+THIS_CLASS.getName()+
            ","+cr.getClassCRC()+","+cr.getSourceFileName().length()+
            ":"+cr.getSourceFileName()+",1[("+mm.getMethodName().length()+":"+
            mm.getMethodName()+")]",
            out );
    }
    
    
    public void testWriteClass2() throws Exception
    {
        ClassRecord cr = new ClassRecord( "a.MyClass", 100, "MyClass.java",
            new String[ 0 ], createAnalysisModuleSet() );
        StringWriter sw = new StringWriter();
        ClassRecordIO crio = new ClassRecordIO();
        
        crio.writeClass( cr, sw );
        String out = sw.toString();
        
        assertEquals(
            "Did not write the correct data.",
            "9:a.MyClass,100,12:MyClass.java,0[]",
            out );
    }
    
    
    public void testReadClass1() throws Exception
    {
        StringReader sr = new StringReader(
            "9:a.MyClass,100,12:MyClass.java,2[(4:a()V)(6:b([S)V)]"
            );
        ClassRecordIO crio = new ClassRecordIO();
        
        ClassRecord cr = crio.readClass( createAnalysisModuleSet(), sr );
        assertNotNull(
            "Returned null class record.",
            cr );
        assertEquals(
            "Did not return correct class name.",
            "a.MyClass",
            cr.getClassName() );
        assertEquals(
            "Did not return correct class check-sum.",
            100L,
            cr.getClassCRC() );
        assertEquals(
            "Did not return correct source file name.",
            "MyClass.java",
            cr.getSourceFileName() );
        String methods[] = cr.getMethods();
        assertEquals(
            "Did not find correct number of methods.",
            2,
            methods.length );
        assertEquals(
            "First method isn't correct signature.",
            "a()V",
            methods[0] );
        assertEquals(
            "Second method isn't correct signature.",
            "b([S)V",
            methods[1] );
    }
    
    
    public void testReadClass2() throws Exception
    {
        StringReader sr = new StringReader(
            "9:b.MyClass,100,13:MyClass2.java,0[]"
            );
        ClassRecordIO crio = new ClassRecordIO();
        
        ClassRecord cr = crio.readClass( createAnalysisModuleSet(), sr );
        assertNotNull(
            "Returned null class record.",
            cr );
        assertEquals(
            "Did not return correct class name.",
            "b.MyClass",
            cr.getClassName() );
        assertEquals(
            "Did not return correct class check-sum.",
            100L,
            cr.getClassCRC() );
        assertEquals(
            "Did not return correct source file name.",
            "MyClass2.java",
            cr.getSourceFileName() );
        assertEquals(
            "Created methods out of nothing.",
            0,
            cr.getMethods().length );
    }
    
    
    public void testReadWriteClass1() throws Exception
    {
        StringWriter sw = new StringWriter();
        ModifiedMethod mm = createModifiedMethod( THIS_CLASS, 0 );
        ClassRecord crIn = createClassRecord( THIS_CLASS, mm );
        ClassRecordIO crio = new ClassRecordIO();
        
        crio.writeClass( crIn, sw );
        
        String out = sw.toString();
        StringReader sr = new StringReader( out );
        ClassRecord crOut = crio.readClass( createAnalysisModuleSet(), sr );

        assertNotNull(
            "Returned null class record.",
            crOut );
        assertEquals(
            "Did not return correct class name.",
            crIn.getClassName(),
            crOut.getClassName() );
        assertEquals(
            "Did not return correct class check-sum.",
            crIn.getClassCRC(),
            crOut.getClassCRC() );
        assertEquals(
            "Did not return correct method count.",
            crIn.getMethodCount(),
            crOut.getMethodCount() );
        for (int i = 0; i < crIn.getMethodCount(); ++i)
        {
            assertEquals(
                "Did not return same method for index "+i+".",
                crIn.getMethodAt( (short)i ),
                crOut.getMethodAt( (short)i ) );
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected ModifiedMethod createModifiedMethod( Class c, int index )
            throws Exception
    {
        return CCCreatorUtil.createModifiedMethod( c, 0 );
    }
    
    
    protected AnalysisModuleSet createAnalysisModuleSet()
    {
        return CCCreatorUtil.createAnalysisModuleSet( 3 );
    }
    
    
    protected ClassRecord createClassRecord( Class c, ModifiedMethod mm )
            throws Exception
    {
        return CCCreatorUtil.createClassRecord( c, mm,
            createAnalysisModuleSet() );
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();

       
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

