/*
 * Logserver
 * Copyright (C) 2017-2025 Joel Reardon
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef __RENDER_QUEUE__H__
#define __RENDER_QUEUE__H__

#include <condition_variable>
#include <list>
#include <mutex>
#include <queue>

#include "constants.h"
#include "format_string.h"

/* Render queue stores the lines of logserver to draw. Each loglines is
 * associated with its own render queue. The filter runner creates FormatStrings
 * at a position and passes it to the render queue. The Renderer itself pulls
 * FormatStrings from the queue and actually draws them. The current epoch for
 * the rendered values is passed alongside when added and the requested epoch is
 * provided when removing. RenderQueue takes ownership of the FormatStrings
 */
class RenderQueue {
public:
	RenderQueue()  {}

	/* Goes through the queue and deletes any format strings that weren't
	 * rendered already */
	virtual ~RenderQueue() {
		unique_lock<mutex> ul(_m);
		while (_queue.size()) {
			auto ret = _queue.front();
			delete get<1>(ret);
			_queue.pop_front();
		}
		_cond.notify_all();
	}

	/* add a format string at position pos for epoch render_epoch. This
	 * function return falses if we aren't interested in this epoch */
	bool add(FormatString* data,
		 size_t pos,
		 size_t render_epoch) {
		unique_lock<mutex> ul(_m);
		assert(render_epoch != G::NO_POS);
		if (_highest_epoch && render_epoch < *_highest_epoch) {
			delete data;
			return false;
		}
		_queue.push_back(make_tuple(pos, data, render_epoch));
		_cond.notify_all();
		return true;
	}

	// removes a FormatString from the queue with a target epoch.
	// pos and fs are set if one is found, and true is returned.
	// any FormatStrings for earlier epochs are deleted
	bool remove(size_t render_epoch, size_t* pos, FormatString** fs) {
		unique_lock<mutex> ul(_m);
		if (!_highest_epoch || *_highest_epoch < render_epoch)
			_highest_epoch = render_epoch;
		// we can pop and push through this iteration so we note the
		// queue size to make sure we only pass through it once
		size_t n = _queue.size();
		while (n) {
			assert(_queue.size());
			auto ret = _queue.front();
			_queue.pop_front();
			if (get<2>(ret) < render_epoch) {
				delete get<1>(ret);
			} else if (get<2>(ret) == render_epoch) {
				*pos = get<0>(ret);
				*fs = get<1>(ret);
				return true;
			} else {
				_queue.push_back(ret);
			}
			--n;
		}
		return false;
	}

	// waits briefly for new lines to appear in queue.
	void wait_for_work() {
		unique_lock<mutex> ul(_m);
		if (!_queue.empty()) return;
		_cond.wait_for(ul, chrono::milliseconds(100));
	}

	// returns true if the queue is empty
	bool empty() {
		unique_lock<mutex> ul(_m);
		return _queue.empty();
	}

protected:
	// list of (pos, string, epoch) tuples
	list<tuple<size_t, FormatString*, size_t>> _queue;
	// mutex for thread safety
	mutex _m;
	// signaled when new lines are added to queue
	condition_variable _cond;
	// highest requested epoch
	optional<size_t> _highest_epoch;
};

#endif  // __RENDER_QUEUE__H__
