/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( '@stdlib/math/base/assert/is-nan' );
var gamma = require( '@stdlib/math/base/special/gamma' );
var SQRT_TWO = require( '@stdlib/constants/float64/sqrt-two' );


// MAIN //

/**
* Returns the expected value of a chi distribution.
*
* @param {NonNegativeNumber} k - degrees of freedom
* @returns {NonNegativeNumber} expected value
*
* @example
* var v = mean( 9.0 );
* // returns ~2.918
*
* @example
* var v = mean( 1.0 );
* // returns ~0.798
*
* @example
* var v = mean( -0.2 );
* // returns NaN
*
* @example
* var v = mean( NaN );
* // returns NaN
*/
function mean( k ) {
	if ( isnan( k ) || k < 0.0 ) {
		return NaN;
	}
	return SQRT_TWO * gamma( ( k+1.0 ) / 2.0 ) / gamma( k / 2.0 );
}


// EXPORTS //

module.exports = mean;
