use super::Isotope;
use crate::Percent;

#[cfg(feature = "ranges")]
/// Range from the minimum to the maximum natural abundance across all isotopes
///
/// Only counts the isotopes that occur naturally in some amount, so the minimum is not 0.
///
/// Convenience constant to avoid writing the code below when this range is needed:
///
/// ```
/// use mendeleev::{Isotope, ISOTOPE_NATURAL_ABUNDANCE_RANGE};
/// let all_values = Isotope::iter().flat_map(|e| e.natural_abundance());
/// let min = all_values.clone().min_by(|a, b| a.total_cmp(&b)).unwrap();
/// let max = all_values.max_by(|a, b| a.total_cmp(&b)).unwrap();
/// assert_eq!(min..=max, ISOTOPE_NATURAL_ABUNDANCE_RANGE);
/// ```
pub const ISOTOPE_NATURAL_ABUNDANCE_RANGE: core::ops::RangeInclusive<Percent> =
    Percent(0.0002)..=Percent(100.0);

macro_rules! pct {
    ($value:literal) => {
        Some(Percent($value))
    };
}

impl Isotope {
    /// The isotopes relative natural abundance on Earth, as a percentage,
    /// or None if the isotope does not occur naturally or only in trace amounts
    ///
    /// ```
    /// use mendeleev::{Isotope, Percent};
    /// assert_eq!(Isotope::H1.natural_abundance(), Some(Percent(99.9855)));
    /// assert_eq!(Isotope::H3.natural_abundance(), None);
    /// assert_eq!(Isotope::Og295.natural_abundance(), None);
    /// ```
    pub const fn natural_abundance(&self) -> Option<Percent> {
        use Isotope as I;
        match self {
            I::H1 => pct!(99.9855),
            I::H2 => pct!(0.0145),
            I::H3 => None,
            I::H4 => None,
            I::H5 => None,
            I::H6 => None,
            I::H7 => None,
            I::He3 => pct!(0.0002),
            I::He4 => pct!(99.9998),
            I::He5 => None,
            I::He6 => None,
            I::He7 => None,
            I::He8 => None,
            I::He9 => None,
            I::He10 => None,
            I::Li3 => None,
            I::Li4 => None,
            I::Li5 => None,
            I::Li6 => pct!(4.85),
            I::Li7 => pct!(95.15),
            I::Li8 => None,
            I::Li9 => None,
            I::Li10 => None,
            I::Li11 => None,
            I::Li12 => None,
            I::Li13 => None,
            I::Be5 => None,
            I::Be6 => None,
            I::Be7 => None,
            I::Be8 => None,
            I::Be9 => pct!(100.0),
            I::Be10 => None,
            I::Be11 => None,
            I::Be12 => None,
            I::Be13 => None,
            I::Be14 => None,
            I::Be15 => None,
            I::Be16 => None,
            I::B6 => None,
            I::B7 => None,
            I::B8 => None,
            I::B9 => None,
            I::B10 => pct!(19.65),
            I::B11 => pct!(80.35),
            I::B12 => None,
            I::B13 => None,
            I::B14 => None,
            I::B15 => None,
            I::B16 => None,
            I::B17 => None,
            I::B18 => None,
            I::B19 => None,
            I::B20 => None,
            I::B21 => None,
            I::C8 => None,
            I::C9 => None,
            I::C10 => None,
            I::C11 => None,
            I::C12 => pct!(98.94),
            I::C13 => pct!(1.06),
            I::C14 => None,
            I::C15 => None,
            I::C16 => None,
            I::C17 => None,
            I::C18 => None,
            I::C19 => None,
            I::C20 => None,
            I::C21 => None,
            I::C22 => None,
            I::C23 => None,
            I::N10 => None,
            I::N11 => None,
            I::N12 => None,
            I::N13 => None,
            I::N14 => pct!(99.6205),
            I::N15 => pct!(0.3795),
            I::N16 => None,
            I::N17 => None,
            I::N18 => None,
            I::N19 => None,
            I::N20 => None,
            I::N21 => None,
            I::N22 => None,
            I::N23 => None,
            I::N24 => None,
            I::N25 => None,
            I::O11 => None,
            I::O12 => None,
            I::O13 => None,
            I::O14 => None,
            I::O15 => None,
            I::O16 => pct!(99.757),
            I::O17 => pct!(0.03835),
            I::O18 => pct!(0.2045),
            I::O19 => None,
            I::O20 => None,
            I::O21 => None,
            I::O22 => None,
            I::O23 => None,
            I::O24 => None,
            I::O25 => None,
            I::O26 => None,
            I::O27 => None,
            I::O28 => None,
            I::F13 => None,
            I::F14 => None,
            I::F15 => None,
            I::F16 => None,
            I::F17 => None,
            I::F18 => None,
            I::F19 => pct!(100.0),
            I::F20 => None,
            I::F21 => None,
            I::F22 => None,
            I::F23 => None,
            I::F24 => None,
            I::F25 => None,
            I::F26 => None,
            I::F27 => None,
            I::F28 => None,
            I::F29 => None,
            I::F30 => None,
            I::F31 => None,
            I::Ne15 => None,
            I::Ne16 => None,
            I::Ne17 => None,
            I::Ne18 => None,
            I::Ne19 => None,
            I::Ne20 => pct!(90.48),
            I::Ne21 => pct!(0.27),
            I::Ne22 => pct!(9.25),
            I::Ne23 => None,
            I::Ne24 => None,
            I::Ne25 => None,
            I::Ne26 => None,
            I::Ne27 => None,
            I::Ne28 => None,
            I::Ne29 => None,
            I::Ne30 => None,
            I::Ne31 => None,
            I::Ne32 => None,
            I::Ne33 => None,
            I::Ne34 => None,
            I::Na17 => None,
            I::Na18 => None,
            I::Na19 => None,
            I::Na20 => None,
            I::Na21 => None,
            I::Na22 => None,
            I::Na23 => pct!(100.0),
            I::Na24 => None,
            I::Na25 => None,
            I::Na26 => None,
            I::Na27 => None,
            I::Na28 => None,
            I::Na29 => None,
            I::Na30 => None,
            I::Na31 => None,
            I::Na32 => None,
            I::Na33 => None,
            I::Na34 => None,
            I::Na35 => None,
            I::Na36 => None,
            I::Na37 => None,
            I::Na38 => None,
            I::Na39 => None,
            I::Mg19 => None,
            I::Mg20 => None,
            I::Mg21 => None,
            I::Mg22 => None,
            I::Mg23 => None,
            I::Mg24 => pct!(78.965),
            I::Mg25 => pct!(10.011),
            I::Mg26 => pct!(11.025),
            I::Mg27 => None,
            I::Mg28 => None,
            I::Mg29 => None,
            I::Mg30 => None,
            I::Mg31 => None,
            I::Mg32 => None,
            I::Mg33 => None,
            I::Mg34 => None,
            I::Mg35 => None,
            I::Mg36 => None,
            I::Mg37 => None,
            I::Mg38 => None,
            I::Mg39 => None,
            I::Mg40 => None,
            I::Mg41 => None,
            I::Al21 => None,
            I::Al22 => None,
            I::Al23 => None,
            I::Al24 => None,
            I::Al25 => None,
            I::Al26 => None,
            I::Al27 => pct!(100.0),
            I::Al28 => None,
            I::Al29 => None,
            I::Al30 => None,
            I::Al31 => None,
            I::Al32 => None,
            I::Al33 => None,
            I::Al34 => None,
            I::Al35 => None,
            I::Al36 => None,
            I::Al37 => None,
            I::Al38 => None,
            I::Al39 => None,
            I::Al40 => None,
            I::Al41 => None,
            I::Al42 => None,
            I::Al43 => None,
            I::Si22 => None,
            I::Si23 => None,
            I::Si24 => None,
            I::Si25 => None,
            I::Si26 => None,
            I::Si27 => None,
            I::Si28 => pct!(92.2545),
            I::Si29 => pct!(4.672),
            I::Si30 => pct!(3.0735),
            I::Si31 => None,
            I::Si32 => None,
            I::Si33 => None,
            I::Si34 => None,
            I::Si35 => None,
            I::Si36 => None,
            I::Si37 => None,
            I::Si38 => None,
            I::Si39 => None,
            I::Si40 => None,
            I::Si41 => None,
            I::Si42 => None,
            I::Si43 => None,
            I::Si44 => None,
            I::Si45 => None,
            I::P24 => None,
            I::P25 => None,
            I::P26 => None,
            I::P27 => None,
            I::P28 => None,
            I::P29 => None,
            I::P30 => None,
            I::P31 => pct!(100.0),
            I::P32 => None,
            I::P33 => None,
            I::P34 => None,
            I::P35 => None,
            I::P36 => None,
            I::P37 => None,
            I::P38 => None,
            I::P39 => None,
            I::P40 => None,
            I::P41 => None,
            I::P42 => None,
            I::P43 => None,
            I::P44 => None,
            I::P45 => None,
            I::P46 => None,
            I::P47 => None,
            I::S26 => None,
            I::S27 => None,
            I::S28 => None,
            I::S29 => None,
            I::S30 => None,
            I::S31 => None,
            I::S32 => pct!(94.85),
            I::S33 => pct!(0.763),
            I::S34 => pct!(4.365),
            I::S35 => None,
            I::S36 => pct!(0.0158),
            I::S37 => None,
            I::S38 => None,
            I::S39 => None,
            I::S40 => None,
            I::S41 => None,
            I::S42 => None,
            I::S43 => None,
            I::S44 => None,
            I::S45 => None,
            I::S46 => None,
            I::S47 => None,
            I::S48 => None,
            I::S49 => None,
            I::Cl28 => None,
            I::Cl29 => None,
            I::Cl30 => None,
            I::Cl31 => None,
            I::Cl32 => None,
            I::Cl33 => None,
            I::Cl34 => None,
            I::Cl35 => pct!(75.8),
            I::Cl36 => None,
            I::Cl37 => pct!(24.2),
            I::Cl38 => None,
            I::Cl39 => None,
            I::Cl40 => None,
            I::Cl41 => None,
            I::Cl42 => None,
            I::Cl43 => None,
            I::Cl44 => None,
            I::Cl45 => None,
            I::Cl46 => None,
            I::Cl47 => None,
            I::Cl48 => None,
            I::Cl49 => None,
            I::Cl50 => None,
            I::Cl51 => None,
            I::Cl52 => None,
            I::Ar29 => None,
            I::Ar30 => None,
            I::Ar31 => None,
            I::Ar32 => None,
            I::Ar33 => None,
            I::Ar34 => None,
            I::Ar35 => None,
            I::Ar36 => pct!(0.3336),
            I::Ar37 => None,
            I::Ar38 => pct!(0.0629),
            I::Ar39 => None,
            I::Ar40 => pct!(99.6035),
            I::Ar41 => None,
            I::Ar42 => None,
            I::Ar43 => None,
            I::Ar44 => None,
            I::Ar45 => None,
            I::Ar46 => None,
            I::Ar47 => None,
            I::Ar48 => None,
            I::Ar49 => None,
            I::Ar50 => None,
            I::Ar51 => None,
            I::Ar52 => None,
            I::Ar53 => None,
            I::Ar54 => None,
            I::K31 => None,
            I::K32 => None,
            I::K33 => None,
            I::K34 => None,
            I::K35 => None,
            I::K36 => None,
            I::K37 => None,
            I::K38 => None,
            I::K39 => pct!(93.2581),
            I::K40 => pct!(0.0117),
            I::K41 => pct!(6.7302),
            I::K42 => None,
            I::K43 => None,
            I::K44 => None,
            I::K45 => None,
            I::K46 => None,
            I::K47 => None,
            I::K48 => None,
            I::K49 => None,
            I::K50 => None,
            I::K51 => None,
            I::K52 => None,
            I::K53 => None,
            I::K54 => None,
            I::K55 => None,
            I::K56 => None,
            I::K57 => None,
            I::K58 => None,
            I::K59 => None,
            I::Ca33 => None,
            I::Ca34 => None,
            I::Ca35 => None,
            I::Ca36 => None,
            I::Ca37 => None,
            I::Ca38 => None,
            I::Ca39 => None,
            I::Ca40 => pct!(96.941),
            I::Ca41 => None,
            I::Ca42 => pct!(0.647),
            I::Ca43 => pct!(0.135),
            I::Ca44 => pct!(2.086),
            I::Ca45 => None,
            I::Ca46 => pct!(0.004),
            I::Ca47 => None,
            I::Ca48 => pct!(0.187),
            I::Ca49 => None,
            I::Ca50 => None,
            I::Ca51 => None,
            I::Ca52 => None,
            I::Ca53 => None,
            I::Ca54 => None,
            I::Ca55 => None,
            I::Ca56 => None,
            I::Ca57 => None,
            I::Ca58 => None,
            I::Ca59 => None,
            I::Ca60 => None,
            I::Ca61 => None,
            I::Sc35 => None,
            I::Sc36 => None,
            I::Sc37 => None,
            I::Sc38 => None,
            I::Sc39 => None,
            I::Sc40 => None,
            I::Sc41 => None,
            I::Sc42 => None,
            I::Sc43 => None,
            I::Sc44 => None,
            I::Sc45 => pct!(100.0),
            I::Sc46 => None,
            I::Sc47 => None,
            I::Sc48 => None,
            I::Sc49 => None,
            I::Sc50 => None,
            I::Sc51 => None,
            I::Sc52 => None,
            I::Sc53 => None,
            I::Sc54 => None,
            I::Sc55 => None,
            I::Sc56 => None,
            I::Sc57 => None,
            I::Sc58 => None,
            I::Sc59 => None,
            I::Sc60 => None,
            I::Sc61 => None,
            I::Sc62 => None,
            I::Sc63 => None,
            I::Ti37 => None,
            I::Ti38 => None,
            I::Ti39 => None,
            I::Ti40 => None,
            I::Ti41 => None,
            I::Ti42 => None,
            I::Ti43 => None,
            I::Ti44 => None,
            I::Ti45 => None,
            I::Ti46 => pct!(8.25),
            I::Ti47 => pct!(7.44),
            I::Ti48 => pct!(73.72),
            I::Ti49 => pct!(5.41),
            I::Ti50 => pct!(5.18),
            I::Ti51 => None,
            I::Ti52 => None,
            I::Ti53 => None,
            I::Ti54 => None,
            I::Ti55 => None,
            I::Ti56 => None,
            I::Ti57 => None,
            I::Ti58 => None,
            I::Ti59 => None,
            I::Ti60 => None,
            I::Ti61 => None,
            I::Ti62 => None,
            I::Ti63 => None,
            I::Ti64 => None,
            I::Ti65 => None,
            I::V39 => None,
            I::V40 => None,
            I::V41 => None,
            I::V42 => None,
            I::V43 => None,
            I::V44 => None,
            I::V45 => None,
            I::V46 => None,
            I::V47 => None,
            I::V48 => None,
            I::V49 => None,
            I::V50 => pct!(0.25),
            I::V51 => pct!(99.75),
            I::V52 => None,
            I::V53 => None,
            I::V54 => None,
            I::V55 => None,
            I::V56 => None,
            I::V57 => None,
            I::V58 => None,
            I::V59 => None,
            I::V60 => None,
            I::V61 => None,
            I::V62 => None,
            I::V63 => None,
            I::V64 => None,
            I::V65 => None,
            I::V66 => None,
            I::V67 => None,
            I::Cr41 => None,
            I::Cr42 => None,
            I::Cr43 => None,
            I::Cr44 => None,
            I::Cr45 => None,
            I::Cr46 => None,
            I::Cr47 => None,
            I::Cr48 => None,
            I::Cr49 => None,
            I::Cr50 => pct!(4.345),
            I::Cr51 => None,
            I::Cr52 => pct!(83.789),
            I::Cr53 => pct!(9.501),
            I::Cr54 => pct!(2.365),
            I::Cr55 => None,
            I::Cr56 => None,
            I::Cr57 => None,
            I::Cr58 => None,
            I::Cr59 => None,
            I::Cr60 => None,
            I::Cr61 => None,
            I::Cr62 => None,
            I::Cr63 => None,
            I::Cr64 => None,
            I::Cr65 => None,
            I::Cr66 => None,
            I::Cr67 => None,
            I::Cr68 => None,
            I::Cr69 => None,
            I::Cr70 => None,
            I::Mn43 => None,
            I::Mn44 => None,
            I::Mn45 => None,
            I::Mn46 => None,
            I::Mn47 => None,
            I::Mn48 => None,
            I::Mn49 => None,
            I::Mn50 => None,
            I::Mn51 => None,
            I::Mn52 => None,
            I::Mn53 => None,
            I::Mn54 => None,
            I::Mn55 => pct!(100.0),
            I::Mn56 => None,
            I::Mn57 => None,
            I::Mn58 => None,
            I::Mn59 => None,
            I::Mn60 => None,
            I::Mn61 => None,
            I::Mn62 => None,
            I::Mn63 => None,
            I::Mn64 => None,
            I::Mn65 => None,
            I::Mn66 => None,
            I::Mn67 => None,
            I::Mn68 => None,
            I::Mn69 => None,
            I::Mn70 => None,
            I::Mn71 => None,
            I::Mn72 => None,
            I::Mn73 => None,
            I::Fe45 => None,
            I::Fe46 => None,
            I::Fe47 => None,
            I::Fe48 => None,
            I::Fe49 => None,
            I::Fe50 => None,
            I::Fe51 => None,
            I::Fe52 => None,
            I::Fe53 => None,
            I::Fe54 => pct!(5.845),
            I::Fe55 => None,
            I::Fe56 => pct!(91.754),
            I::Fe57 => pct!(2.119),
            I::Fe58 => pct!(0.282),
            I::Fe59 => None,
            I::Fe60 => None,
            I::Fe61 => None,
            I::Fe62 => None,
            I::Fe63 => None,
            I::Fe64 => None,
            I::Fe65 => None,
            I::Fe66 => None,
            I::Fe67 => None,
            I::Fe68 => None,
            I::Fe69 => None,
            I::Fe70 => None,
            I::Fe71 => None,
            I::Fe72 => None,
            I::Fe73 => None,
            I::Fe74 => None,
            I::Fe75 => None,
            I::Fe76 => None,
            I::Co47 => None,
            I::Co48 => None,
            I::Co49 => None,
            I::Co50 => None,
            I::Co51 => None,
            I::Co52 => None,
            I::Co53 => None,
            I::Co54 => None,
            I::Co55 => None,
            I::Co56 => None,
            I::Co57 => None,
            I::Co58 => None,
            I::Co59 => pct!(100.0),
            I::Co60 => None,
            I::Co61 => None,
            I::Co62 => None,
            I::Co63 => None,
            I::Co64 => None,
            I::Co65 => None,
            I::Co66 => None,
            I::Co67 => None,
            I::Co68 => None,
            I::Co69 => None,
            I::Co70 => None,
            I::Co71 => None,
            I::Co72 => None,
            I::Co73 => None,
            I::Co74 => None,
            I::Co75 => None,
            I::Co76 => None,
            I::Co77 => None,
            I::Co78 => None,
            I::Ni48 => None,
            I::Ni49 => None,
            I::Ni50 => None,
            I::Ni51 => None,
            I::Ni52 => None,
            I::Ni53 => None,
            I::Ni54 => None,
            I::Ni55 => None,
            I::Ni56 => None,
            I::Ni57 => None,
            I::Ni58 => pct!(68.0769),
            I::Ni59 => None,
            I::Ni60 => pct!(26.2231),
            I::Ni61 => pct!(1.1399),
            I::Ni62 => pct!(3.6345),
            I::Ni63 => None,
            I::Ni64 => pct!(0.9256),
            I::Ni65 => None,
            I::Ni66 => None,
            I::Ni67 => None,
            I::Ni68 => None,
            I::Ni69 => None,
            I::Ni70 => None,
            I::Ni71 => None,
            I::Ni72 => None,
            I::Ni73 => None,
            I::Ni74 => None,
            I::Ni75 => None,
            I::Ni76 => None,
            I::Ni77 => None,
            I::Ni78 => None,
            I::Ni79 => None,
            I::Ni80 => None,
            I::Ni81 => None,
            I::Ni82 => None,
            I::Cu52 => None,
            I::Cu53 => None,
            I::Cu54 => None,
            I::Cu55 => None,
            I::Cu56 => None,
            I::Cu57 => None,
            I::Cu58 => None,
            I::Cu59 => None,
            I::Cu60 => None,
            I::Cu61 => None,
            I::Cu62 => None,
            I::Cu63 => pct!(69.15),
            I::Cu64 => None,
            I::Cu65 => pct!(30.85),
            I::Cu66 => None,
            I::Cu67 => None,
            I::Cu68 => None,
            I::Cu69 => None,
            I::Cu70 => None,
            I::Cu71 => None,
            I::Cu72 => None,
            I::Cu73 => None,
            I::Cu74 => None,
            I::Cu75 => None,
            I::Cu76 => None,
            I::Cu77 => None,
            I::Cu78 => None,
            I::Cu79 => None,
            I::Cu80 => None,
            I::Cu81 => None,
            I::Cu82 => None,
            I::Cu83 => None,
            I::Cu84 => None,
            I::Zn54 => None,
            I::Zn55 => None,
            I::Zn56 => None,
            I::Zn57 => None,
            I::Zn58 => None,
            I::Zn59 => None,
            I::Zn60 => None,
            I::Zn61 => None,
            I::Zn62 => None,
            I::Zn63 => None,
            I::Zn64 => pct!(49.17),
            I::Zn65 => None,
            I::Zn66 => pct!(27.73),
            I::Zn67 => pct!(4.04),
            I::Zn68 => pct!(18.45),
            I::Zn69 => None,
            I::Zn70 => pct!(0.61),
            I::Zn71 => None,
            I::Zn72 => None,
            I::Zn73 => None,
            I::Zn74 => None,
            I::Zn75 => None,
            I::Zn76 => None,
            I::Zn77 => None,
            I::Zn78 => None,
            I::Zn79 => None,
            I::Zn80 => None,
            I::Zn81 => None,
            I::Zn82 => None,
            I::Zn83 => None,
            I::Zn84 => None,
            I::Zn85 => None,
            I::Zn86 => None,
            I::Ga56 => None,
            I::Ga57 => None,
            I::Ga58 => None,
            I::Ga59 => None,
            I::Ga60 => None,
            I::Ga61 => None,
            I::Ga62 => None,
            I::Ga63 => None,
            I::Ga64 => None,
            I::Ga65 => None,
            I::Ga66 => None,
            I::Ga67 => None,
            I::Ga68 => None,
            I::Ga69 => pct!(60.108),
            I::Ga70 => None,
            I::Ga71 => pct!(39.892),
            I::Ga72 => None,
            I::Ga73 => None,
            I::Ga74 => None,
            I::Ga75 => None,
            I::Ga76 => None,
            I::Ga77 => None,
            I::Ga78 => None,
            I::Ga79 => None,
            I::Ga80 => None,
            I::Ga81 => None,
            I::Ga82 => None,
            I::Ga83 => None,
            I::Ga84 => None,
            I::Ga85 => None,
            I::Ga86 => None,
            I::Ga87 => None,
            I::Ga88 => None,
            I::Ge58 => None,
            I::Ge59 => None,
            I::Ge60 => None,
            I::Ge61 => None,
            I::Ge62 => None,
            I::Ge63 => None,
            I::Ge64 => None,
            I::Ge65 => None,
            I::Ge66 => None,
            I::Ge67 => None,
            I::Ge68 => None,
            I::Ge69 => None,
            I::Ge70 => pct!(20.52),
            I::Ge71 => None,
            I::Ge72 => pct!(27.45),
            I::Ge73 => pct!(7.76),
            I::Ge74 => pct!(36.52),
            I::Ge75 => None,
            I::Ge76 => pct!(7.75),
            I::Ge77 => None,
            I::Ge78 => None,
            I::Ge79 => None,
            I::Ge80 => None,
            I::Ge81 => None,
            I::Ge82 => None,
            I::Ge83 => None,
            I::Ge84 => None,
            I::Ge85 => None,
            I::Ge86 => None,
            I::Ge87 => None,
            I::Ge88 => None,
            I::Ge89 => None,
            I::Ge90 => None,
            I::As60 => None,
            I::As61 => None,
            I::As62 => None,
            I::As63 => None,
            I::As64 => None,
            I::As65 => None,
            I::As66 => None,
            I::As67 => None,
            I::As68 => None,
            I::As69 => None,
            I::As70 => None,
            I::As71 => None,
            I::As72 => None,
            I::As73 => None,
            I::As74 => None,
            I::As75 => pct!(100.0),
            I::As76 => None,
            I::As77 => None,
            I::As78 => None,
            I::As79 => None,
            I::As80 => None,
            I::As81 => None,
            I::As82 => None,
            I::As83 => None,
            I::As84 => None,
            I::As85 => None,
            I::As86 => None,
            I::As87 => None,
            I::As88 => None,
            I::As89 => None,
            I::As90 => None,
            I::As91 => None,
            I::As92 => None,
            I::Se63 => None,
            I::Se64 => None,
            I::Se65 => None,
            I::Se66 => None,
            I::Se67 => None,
            I::Se68 => None,
            I::Se69 => None,
            I::Se70 => None,
            I::Se71 => None,
            I::Se72 => None,
            I::Se73 => None,
            I::Se74 => pct!(0.86),
            I::Se75 => None,
            I::Se76 => pct!(9.23),
            I::Se77 => pct!(7.6),
            I::Se78 => pct!(23.69),
            I::Se79 => None,
            I::Se80 => pct!(49.8),
            I::Se81 => None,
            I::Se82 => pct!(8.82),
            I::Se83 => None,
            I::Se84 => None,
            I::Se85 => None,
            I::Se86 => None,
            I::Se87 => None,
            I::Se88 => None,
            I::Se89 => None,
            I::Se90 => None,
            I::Se91 => None,
            I::Se92 => None,
            I::Se93 => None,
            I::Se94 => None,
            I::Se95 => None,
            I::Br65 => None,
            I::Br66 => None,
            I::Br67 => None,
            I::Br68 => None,
            I::Br69 => None,
            I::Br70 => None,
            I::Br71 => None,
            I::Br72 => None,
            I::Br73 => None,
            I::Br74 => None,
            I::Br75 => None,
            I::Br76 => None,
            I::Br77 => None,
            I::Br78 => None,
            I::Br79 => pct!(50.65),
            I::Br80 => None,
            I::Br81 => pct!(49.35),
            I::Br82 => None,
            I::Br83 => None,
            I::Br84 => None,
            I::Br85 => None,
            I::Br86 => None,
            I::Br87 => None,
            I::Br88 => None,
            I::Br89 => None,
            I::Br90 => None,
            I::Br91 => None,
            I::Br92 => None,
            I::Br93 => None,
            I::Br94 => None,
            I::Br95 => None,
            I::Br96 => None,
            I::Br97 => None,
            I::Br98 => None,
            I::Kr67 => None,
            I::Kr68 => None,
            I::Kr69 => None,
            I::Kr70 => None,
            I::Kr71 => None,
            I::Kr72 => None,
            I::Kr73 => None,
            I::Kr74 => None,
            I::Kr75 => None,
            I::Kr76 => None,
            I::Kr77 => None,
            I::Kr78 => pct!(0.355),
            I::Kr79 => None,
            I::Kr80 => pct!(2.286),
            I::Kr81 => None,
            I::Kr82 => pct!(11.593),
            I::Kr83 => pct!(11.5),
            I::Kr84 => pct!(56.987),
            I::Kr85 => None,
            I::Kr86 => pct!(17.279),
            I::Kr87 => None,
            I::Kr88 => None,
            I::Kr89 => None,
            I::Kr90 => None,
            I::Kr91 => None,
            I::Kr92 => None,
            I::Kr93 => None,
            I::Kr94 => None,
            I::Kr95 => None,
            I::Kr96 => None,
            I::Kr97 => None,
            I::Kr98 => None,
            I::Kr99 => None,
            I::Kr100 => None,
            I::Kr101 => None,
            I::Rb71 => None,
            I::Rb72 => None,
            I::Rb73 => None,
            I::Rb74 => None,
            I::Rb75 => None,
            I::Rb76 => None,
            I::Rb77 => None,
            I::Rb78 => None,
            I::Rb79 => None,
            I::Rb80 => None,
            I::Rb81 => None,
            I::Rb82 => None,
            I::Rb83 => None,
            I::Rb84 => None,
            I::Rb85 => pct!(72.17),
            I::Rb86 => None,
            I::Rb87 => pct!(27.83),
            I::Rb88 => None,
            I::Rb89 => None,
            I::Rb90 => None,
            I::Rb91 => None,
            I::Rb92 => None,
            I::Rb93 => None,
            I::Rb94 => None,
            I::Rb95 => None,
            I::Rb96 => None,
            I::Rb97 => None,
            I::Rb98 => None,
            I::Rb99 => None,
            I::Rb100 => None,
            I::Rb101 => None,
            I::Rb102 => None,
            I::Rb103 => None,
            I::Rb104 => None,
            I::Sr73 => None,
            I::Sr74 => None,
            I::Sr75 => None,
            I::Sr76 => None,
            I::Sr77 => None,
            I::Sr78 => None,
            I::Sr79 => None,
            I::Sr80 => None,
            I::Sr81 => None,
            I::Sr82 => None,
            I::Sr83 => None,
            I::Sr84 => pct!(0.56),
            I::Sr85 => None,
            I::Sr86 => pct!(9.86),
            I::Sr87 => pct!(7.0),
            I::Sr88 => pct!(82.58),
            I::Sr89 => None,
            I::Sr90 => None,
            I::Sr91 => None,
            I::Sr92 => None,
            I::Sr93 => None,
            I::Sr94 => None,
            I::Sr95 => None,
            I::Sr96 => None,
            I::Sr97 => None,
            I::Sr98 => None,
            I::Sr99 => None,
            I::Sr100 => None,
            I::Sr101 => None,
            I::Sr102 => None,
            I::Sr103 => None,
            I::Sr104 => None,
            I::Sr105 => None,
            I::Sr106 => None,
            I::Sr107 => None,
            I::Y75 => None,
            I::Y76 => None,
            I::Y77 => None,
            I::Y78 => None,
            I::Y79 => None,
            I::Y80 => None,
            I::Y81 => None,
            I::Y82 => None,
            I::Y83 => None,
            I::Y84 => None,
            I::Y85 => None,
            I::Y86 => None,
            I::Y87 => None,
            I::Y88 => None,
            I::Y89 => pct!(100.0),
            I::Y90 => None,
            I::Y91 => None,
            I::Y92 => None,
            I::Y93 => None,
            I::Y94 => None,
            I::Y95 => None,
            I::Y96 => None,
            I::Y97 => None,
            I::Y98 => None,
            I::Y99 => None,
            I::Y100 => None,
            I::Y101 => None,
            I::Y102 => None,
            I::Y103 => None,
            I::Y104 => None,
            I::Y105 => None,
            I::Y106 => None,
            I::Y107 => None,
            I::Y108 => None,
            I::Y109 => None,
            I::Zr77 => None,
            I::Zr78 => None,
            I::Zr79 => None,
            I::Zr80 => None,
            I::Zr81 => None,
            I::Zr82 => None,
            I::Zr83 => None,
            I::Zr84 => None,
            I::Zr85 => None,
            I::Zr86 => None,
            I::Zr87 => None,
            I::Zr88 => None,
            I::Zr89 => None,
            I::Zr90 => pct!(51.45),
            I::Zr91 => pct!(11.22),
            I::Zr92 => pct!(17.15),
            I::Zr93 => None,
            I::Zr94 => pct!(17.38),
            I::Zr95 => None,
            I::Zr96 => pct!(2.8),
            I::Zr97 => None,
            I::Zr98 => None,
            I::Zr99 => None,
            I::Zr100 => None,
            I::Zr101 => None,
            I::Zr102 => None,
            I::Zr103 => None,
            I::Zr104 => None,
            I::Zr105 => None,
            I::Zr106 => None,
            I::Zr107 => None,
            I::Zr108 => None,
            I::Zr109 => None,
            I::Zr110 => None,
            I::Zr111 => None,
            I::Zr112 => None,
            I::Zr113 => None,
            I::Nb79 => None,
            I::Nb80 => None,
            I::Nb81 => None,
            I::Nb82 => None,
            I::Nb83 => None,
            I::Nb84 => None,
            I::Nb85 => None,
            I::Nb86 => None,
            I::Nb87 => None,
            I::Nb88 => None,
            I::Nb89 => None,
            I::Nb90 => None,
            I::Nb91 => None,
            I::Nb92 => None,
            I::Nb93 => pct!(100.0),
            I::Nb94 => None,
            I::Nb95 => None,
            I::Nb96 => None,
            I::Nb97 => None,
            I::Nb98 => None,
            I::Nb99 => None,
            I::Nb100 => None,
            I::Nb101 => None,
            I::Nb102 => None,
            I::Nb103 => None,
            I::Nb104 => None,
            I::Nb105 => None,
            I::Nb106 => None,
            I::Nb107 => None,
            I::Nb108 => None,
            I::Nb109 => None,
            I::Nb110 => None,
            I::Nb111 => None,
            I::Nb112 => None,
            I::Nb113 => None,
            I::Nb114 => None,
            I::Nb115 => None,
            I::Nb116 => None,
            I::Mo81 => None,
            I::Mo82 => None,
            I::Mo83 => None,
            I::Mo84 => None,
            I::Mo85 => None,
            I::Mo86 => None,
            I::Mo87 => None,
            I::Mo88 => None,
            I::Mo89 => None,
            I::Mo90 => None,
            I::Mo91 => None,
            I::Mo92 => pct!(14.649),
            I::Mo93 => None,
            I::Mo94 => pct!(9.187),
            I::Mo95 => pct!(15.873),
            I::Mo96 => pct!(16.673),
            I::Mo97 => pct!(9.582),
            I::Mo98 => pct!(24.292),
            I::Mo99 => None,
            I::Mo100 => pct!(9.744),
            I::Mo101 => None,
            I::Mo102 => None,
            I::Mo103 => None,
            I::Mo104 => None,
            I::Mo105 => None,
            I::Mo106 => None,
            I::Mo107 => None,
            I::Mo108 => None,
            I::Mo109 => None,
            I::Mo110 => None,
            I::Mo111 => None,
            I::Mo112 => None,
            I::Mo113 => None,
            I::Mo114 => None,
            I::Mo115 => None,
            I::Mo116 => None,
            I::Mo117 => None,
            I::Mo118 => None,
            I::Mo119 => None,
            I::Tc83 => None,
            I::Tc84 => None,
            I::Tc85 => None,
            I::Tc86 => None,
            I::Tc87 => None,
            I::Tc88 => None,
            I::Tc89 => None,
            I::Tc90 => None,
            I::Tc91 => None,
            I::Tc92 => None,
            I::Tc93 => None,
            I::Tc94 => None,
            I::Tc95 => None,
            I::Tc96 => None,
            I::Tc97 => None,
            I::Tc98 => None,
            I::Tc99 => None,
            I::Tc100 => None,
            I::Tc101 => None,
            I::Tc102 => None,
            I::Tc103 => None,
            I::Tc104 => None,
            I::Tc105 => None,
            I::Tc106 => None,
            I::Tc107 => None,
            I::Tc108 => None,
            I::Tc109 => None,
            I::Tc110 => None,
            I::Tc111 => None,
            I::Tc112 => None,
            I::Tc113 => None,
            I::Tc114 => None,
            I::Tc115 => None,
            I::Tc116 => None,
            I::Tc117 => None,
            I::Tc118 => None,
            I::Tc119 => None,
            I::Tc120 => None,
            I::Tc121 => None,
            I::Tc122 => None,
            I::Ru85 => None,
            I::Ru86 => None,
            I::Ru87 => None,
            I::Ru88 => None,
            I::Ru89 => None,
            I::Ru90 => None,
            I::Ru91 => None,
            I::Ru92 => None,
            I::Ru93 => None,
            I::Ru94 => None,
            I::Ru95 => None,
            I::Ru96 => pct!(5.54),
            I::Ru97 => None,
            I::Ru98 => pct!(1.87),
            I::Ru99 => pct!(12.76),
            I::Ru100 => pct!(12.6),
            I::Ru101 => pct!(17.06),
            I::Ru102 => pct!(31.55),
            I::Ru103 => None,
            I::Ru104 => pct!(18.62),
            I::Ru105 => None,
            I::Ru106 => None,
            I::Ru107 => None,
            I::Ru108 => None,
            I::Ru109 => None,
            I::Ru110 => None,
            I::Ru111 => None,
            I::Ru112 => None,
            I::Ru113 => None,
            I::Ru114 => None,
            I::Ru115 => None,
            I::Ru116 => None,
            I::Ru117 => None,
            I::Ru118 => None,
            I::Ru119 => None,
            I::Ru120 => None,
            I::Ru121 => None,
            I::Ru122 => None,
            I::Ru123 => None,
            I::Ru124 => None,
            I::Ru125 => None,
            I::Rh88 => None,
            I::Rh89 => None,
            I::Rh90 => None,
            I::Rh91 => None,
            I::Rh92 => None,
            I::Rh93 => None,
            I::Rh94 => None,
            I::Rh95 => None,
            I::Rh96 => None,
            I::Rh97 => None,
            I::Rh98 => None,
            I::Rh99 => None,
            I::Rh100 => None,
            I::Rh101 => None,
            I::Rh102 => None,
            I::Rh103 => pct!(100.0),
            I::Rh104 => None,
            I::Rh105 => None,
            I::Rh106 => None,
            I::Rh107 => None,
            I::Rh108 => None,
            I::Rh109 => None,
            I::Rh110 => None,
            I::Rh111 => None,
            I::Rh112 => None,
            I::Rh113 => None,
            I::Rh114 => None,
            I::Rh115 => None,
            I::Rh116 => None,
            I::Rh117 => None,
            I::Rh118 => None,
            I::Rh119 => None,
            I::Rh120 => None,
            I::Rh121 => None,
            I::Rh122 => None,
            I::Rh123 => None,
            I::Rh124 => None,
            I::Rh125 => None,
            I::Rh126 => None,
            I::Rh127 => None,
            I::Rh128 => None,
            I::Pd90 => None,
            I::Pd91 => None,
            I::Pd92 => None,
            I::Pd93 => None,
            I::Pd94 => None,
            I::Pd95 => None,
            I::Pd96 => None,
            I::Pd97 => None,
            I::Pd98 => None,
            I::Pd99 => None,
            I::Pd100 => None,
            I::Pd101 => None,
            I::Pd102 => pct!(1.02),
            I::Pd103 => None,
            I::Pd104 => pct!(11.14),
            I::Pd105 => pct!(22.33),
            I::Pd106 => pct!(27.33),
            I::Pd107 => None,
            I::Pd108 => pct!(26.46),
            I::Pd109 => None,
            I::Pd110 => pct!(11.72),
            I::Pd111 => None,
            I::Pd112 => None,
            I::Pd113 => None,
            I::Pd114 => None,
            I::Pd115 => None,
            I::Pd116 => None,
            I::Pd117 => None,
            I::Pd118 => None,
            I::Pd119 => None,
            I::Pd120 => None,
            I::Pd121 => None,
            I::Pd122 => None,
            I::Pd123 => None,
            I::Pd124 => None,
            I::Pd125 => None,
            I::Pd126 => None,
            I::Pd127 => None,
            I::Pd128 => None,
            I::Pd129 => None,
            I::Pd130 => None,
            I::Pd131 => None,
            I::Ag92 => None,
            I::Ag93 => None,
            I::Ag94 => None,
            I::Ag95 => None,
            I::Ag96 => None,
            I::Ag97 => None,
            I::Ag98 => None,
            I::Ag99 => None,
            I::Ag100 => None,
            I::Ag101 => None,
            I::Ag102 => None,
            I::Ag103 => None,
            I::Ag104 => None,
            I::Ag105 => None,
            I::Ag106 => None,
            I::Ag107 => pct!(51.839),
            I::Ag108 => None,
            I::Ag109 => pct!(48.161),
            I::Ag110 => None,
            I::Ag111 => None,
            I::Ag112 => None,
            I::Ag113 => None,
            I::Ag114 => None,
            I::Ag115 => None,
            I::Ag116 => None,
            I::Ag117 => None,
            I::Ag118 => None,
            I::Ag119 => None,
            I::Ag120 => None,
            I::Ag121 => None,
            I::Ag122 => None,
            I::Ag123 => None,
            I::Ag124 => None,
            I::Ag125 => None,
            I::Ag126 => None,
            I::Ag127 => None,
            I::Ag128 => None,
            I::Ag129 => None,
            I::Ag130 => None,
            I::Ag131 => None,
            I::Ag132 => None,
            I::Ag133 => None,
            I::Cd94 => None,
            I::Cd95 => None,
            I::Cd96 => None,
            I::Cd97 => None,
            I::Cd98 => None,
            I::Cd99 => None,
            I::Cd100 => None,
            I::Cd101 => None,
            I::Cd102 => None,
            I::Cd103 => None,
            I::Cd104 => None,
            I::Cd105 => None,
            I::Cd106 => pct!(1.245),
            I::Cd107 => None,
            I::Cd108 => pct!(0.888),
            I::Cd109 => None,
            I::Cd110 => pct!(12.47),
            I::Cd111 => pct!(12.795),
            I::Cd112 => pct!(24.109),
            I::Cd113 => pct!(12.227),
            I::Cd114 => pct!(28.754),
            I::Cd115 => None,
            I::Cd116 => pct!(7.512),
            I::Cd117 => None,
            I::Cd118 => None,
            I::Cd119 => None,
            I::Cd120 => None,
            I::Cd121 => None,
            I::Cd122 => None,
            I::Cd123 => None,
            I::Cd124 => None,
            I::Cd125 => None,
            I::Cd126 => None,
            I::Cd127 => None,
            I::Cd128 => None,
            I::Cd129 => None,
            I::Cd130 => None,
            I::Cd131 => None,
            I::Cd132 => None,
            I::Cd133 => None,
            I::Cd134 => None,
            I::Cd135 => None,
            I::In96 => None,
            I::In97 => None,
            I::In98 => None,
            I::In99 => None,
            I::In100 => None,
            I::In101 => None,
            I::In102 => None,
            I::In103 => None,
            I::In104 => None,
            I::In105 => None,
            I::In106 => None,
            I::In107 => None,
            I::In108 => None,
            I::In109 => None,
            I::In110 => None,
            I::In111 => None,
            I::In112 => None,
            I::In113 => pct!(4.281),
            I::In114 => None,
            I::In115 => pct!(95.719),
            I::In116 => None,
            I::In117 => None,
            I::In118 => None,
            I::In119 => None,
            I::In120 => None,
            I::In121 => None,
            I::In122 => None,
            I::In123 => None,
            I::In124 => None,
            I::In125 => None,
            I::In126 => None,
            I::In127 => None,
            I::In128 => None,
            I::In129 => None,
            I::In130 => None,
            I::In131 => None,
            I::In132 => None,
            I::In133 => None,
            I::In134 => None,
            I::In135 => None,
            I::In136 => None,
            I::In137 => None,
            I::Sn99 => None,
            I::Sn100 => None,
            I::Sn101 => None,
            I::Sn102 => None,
            I::Sn103 => None,
            I::Sn104 => None,
            I::Sn105 => None,
            I::Sn106 => None,
            I::Sn107 => None,
            I::Sn108 => None,
            I::Sn109 => None,
            I::Sn110 => None,
            I::Sn111 => None,
            I::Sn112 => pct!(0.97),
            I::Sn113 => None,
            I::Sn114 => pct!(0.66),
            I::Sn115 => pct!(0.34),
            I::Sn116 => pct!(14.54),
            I::Sn117 => pct!(7.68),
            I::Sn118 => pct!(24.22),
            I::Sn119 => pct!(8.59),
            I::Sn120 => pct!(32.58),
            I::Sn121 => None,
            I::Sn122 => pct!(4.63),
            I::Sn123 => None,
            I::Sn124 => pct!(5.79),
            I::Sn125 => None,
            I::Sn126 => None,
            I::Sn127 => None,
            I::Sn128 => None,
            I::Sn129 => None,
            I::Sn130 => None,
            I::Sn131 => None,
            I::Sn132 => None,
            I::Sn133 => None,
            I::Sn134 => None,
            I::Sn135 => None,
            I::Sn136 => None,
            I::Sn137 => None,
            I::Sn138 => None,
            I::Sn139 => None,
            I::Sn140 => None,
            I::Sb102 => None,
            I::Sb103 => None,
            I::Sb104 => None,
            I::Sb105 => None,
            I::Sb106 => None,
            I::Sb107 => None,
            I::Sb108 => None,
            I::Sb109 => None,
            I::Sb110 => None,
            I::Sb111 => None,
            I::Sb112 => None,
            I::Sb113 => None,
            I::Sb114 => None,
            I::Sb115 => None,
            I::Sb116 => None,
            I::Sb117 => None,
            I::Sb118 => None,
            I::Sb119 => None,
            I::Sb120 => None,
            I::Sb121 => pct!(57.21),
            I::Sb122 => None,
            I::Sb123 => pct!(42.79),
            I::Sb124 => None,
            I::Sb125 => None,
            I::Sb126 => None,
            I::Sb127 => None,
            I::Sb128 => None,
            I::Sb129 => None,
            I::Sb130 => None,
            I::Sb131 => None,
            I::Sb132 => None,
            I::Sb133 => None,
            I::Sb134 => None,
            I::Sb135 => None,
            I::Sb136 => None,
            I::Sb137 => None,
            I::Sb138 => None,
            I::Sb139 => None,
            I::Sb140 => None,
            I::Sb141 => None,
            I::Sb142 => None,
            I::Te104 => None,
            I::Te105 => None,
            I::Te106 => None,
            I::Te107 => None,
            I::Te108 => None,
            I::Te109 => None,
            I::Te110 => None,
            I::Te111 => None,
            I::Te112 => None,
            I::Te113 => None,
            I::Te114 => None,
            I::Te115 => None,
            I::Te116 => None,
            I::Te117 => None,
            I::Te118 => None,
            I::Te119 => None,
            I::Te120 => pct!(0.09),
            I::Te121 => None,
            I::Te122 => pct!(2.55),
            I::Te123 => pct!(0.89),
            I::Te124 => pct!(4.74),
            I::Te125 => pct!(7.07),
            I::Te126 => pct!(8.84),
            I::Te127 => None,
            I::Te128 => pct!(31.74),
            I::Te129 => None,
            I::Te130 => pct!(34.08),
            I::Te131 => None,
            I::Te132 => None,
            I::Te133 => None,
            I::Te134 => None,
            I::Te135 => None,
            I::Te136 => None,
            I::Te137 => None,
            I::Te138 => None,
            I::Te139 => None,
            I::Te140 => None,
            I::Te141 => None,
            I::Te142 => None,
            I::Te143 => None,
            I::Te144 => None,
            I::Te145 => None,
            I::I106 => None,
            I::I107 => None,
            I::I108 => None,
            I::I109 => None,
            I::I110 => None,
            I::I111 => None,
            I::I112 => None,
            I::I113 => None,
            I::I114 => None,
            I::I115 => None,
            I::I116 => None,
            I::I117 => None,
            I::I118 => None,
            I::I119 => None,
            I::I120 => None,
            I::I121 => None,
            I::I122 => None,
            I::I123 => None,
            I::I124 => None,
            I::I125 => None,
            I::I126 => None,
            I::I127 => pct!(100.0),
            I::I128 => None,
            I::I129 => None,
            I::I130 => None,
            I::I131 => None,
            I::I132 => None,
            I::I133 => None,
            I::I134 => None,
            I::I135 => None,
            I::I136 => None,
            I::I137 => None,
            I::I138 => None,
            I::I139 => None,
            I::I140 => None,
            I::I141 => None,
            I::I142 => None,
            I::I143 => None,
            I::I144 => None,
            I::I145 => None,
            I::I146 => None,
            I::I147 => None,
            I::Xe108 => None,
            I::Xe109 => None,
            I::Xe110 => None,
            I::Xe111 => None,
            I::Xe112 => None,
            I::Xe113 => None,
            I::Xe114 => None,
            I::Xe115 => None,
            I::Xe116 => None,
            I::Xe117 => None,
            I::Xe118 => None,
            I::Xe119 => None,
            I::Xe120 => None,
            I::Xe121 => None,
            I::Xe122 => None,
            I::Xe123 => None,
            I::Xe124 => pct!(0.095),
            I::Xe125 => None,
            I::Xe126 => pct!(0.089),
            I::Xe127 => None,
            I::Xe128 => pct!(1.91),
            I::Xe129 => pct!(26.401),
            I::Xe130 => pct!(4.071),
            I::Xe131 => pct!(21.232),
            I::Xe132 => pct!(26.909),
            I::Xe133 => None,
            I::Xe134 => pct!(10.436),
            I::Xe135 => None,
            I::Xe136 => pct!(8.857),
            I::Xe137 => None,
            I::Xe138 => None,
            I::Xe139 => None,
            I::Xe140 => None,
            I::Xe141 => None,
            I::Xe142 => None,
            I::Xe143 => None,
            I::Xe144 => None,
            I::Xe145 => None,
            I::Xe146 => None,
            I::Xe147 => None,
            I::Xe148 => None,
            I::Xe149 => None,
            I::Xe150 => None,
            I::Cs111 => None,
            I::Cs112 => None,
            I::Cs113 => None,
            I::Cs114 => None,
            I::Cs115 => None,
            I::Cs116 => None,
            I::Cs117 => None,
            I::Cs118 => None,
            I::Cs119 => None,
            I::Cs120 => None,
            I::Cs121 => None,
            I::Cs122 => None,
            I::Cs123 => None,
            I::Cs124 => None,
            I::Cs125 => None,
            I::Cs126 => None,
            I::Cs127 => None,
            I::Cs128 => None,
            I::Cs129 => None,
            I::Cs130 => None,
            I::Cs131 => None,
            I::Cs132 => None,
            I::Cs133 => pct!(100.0),
            I::Cs134 => None,
            I::Cs135 => None,
            I::Cs136 => None,
            I::Cs137 => None,
            I::Cs138 => None,
            I::Cs139 => None,
            I::Cs140 => None,
            I::Cs141 => None,
            I::Cs142 => None,
            I::Cs143 => None,
            I::Cs144 => None,
            I::Cs145 => None,
            I::Cs146 => None,
            I::Cs147 => None,
            I::Cs148 => None,
            I::Cs149 => None,
            I::Cs150 => None,
            I::Cs151 => None,
            I::Cs152 => None,
            I::Ba113 => None,
            I::Ba114 => None,
            I::Ba115 => None,
            I::Ba116 => None,
            I::Ba117 => None,
            I::Ba118 => None,
            I::Ba119 => None,
            I::Ba120 => None,
            I::Ba121 => None,
            I::Ba122 => None,
            I::Ba123 => None,
            I::Ba124 => None,
            I::Ba125 => None,
            I::Ba126 => None,
            I::Ba127 => None,
            I::Ba128 => None,
            I::Ba129 => None,
            I::Ba130 => pct!(0.11),
            I::Ba131 => None,
            I::Ba132 => pct!(0.1),
            I::Ba133 => None,
            I::Ba134 => pct!(2.42),
            I::Ba135 => pct!(6.59),
            I::Ba136 => pct!(7.85),
            I::Ba137 => pct!(11.23),
            I::Ba138 => pct!(71.7),
            I::Ba139 => None,
            I::Ba140 => None,
            I::Ba141 => None,
            I::Ba142 => None,
            I::Ba143 => None,
            I::Ba144 => None,
            I::Ba145 => None,
            I::Ba146 => None,
            I::Ba147 => None,
            I::Ba148 => None,
            I::Ba149 => None,
            I::Ba150 => None,
            I::Ba151 => None,
            I::Ba152 => None,
            I::Ba153 => None,
            I::Ba154 => None,
            I::La116 => None,
            I::La117 => None,
            I::La118 => None,
            I::La119 => None,
            I::La120 => None,
            I::La121 => None,
            I::La122 => None,
            I::La123 => None,
            I::La124 => None,
            I::La125 => None,
            I::La126 => None,
            I::La127 => None,
            I::La128 => None,
            I::La129 => None,
            I::La130 => None,
            I::La131 => None,
            I::La132 => None,
            I::La133 => None,
            I::La134 => None,
            I::La135 => None,
            I::La136 => None,
            I::La137 => None,
            I::La138 => pct!(0.08881),
            I::La139 => pct!(99.91119),
            I::La140 => None,
            I::La141 => None,
            I::La142 => None,
            I::La143 => None,
            I::La144 => None,
            I::La145 => None,
            I::La146 => None,
            I::La147 => None,
            I::La148 => None,
            I::La149 => None,
            I::La150 => None,
            I::La151 => None,
            I::La152 => None,
            I::La153 => None,
            I::La154 => None,
            I::La155 => None,
            I::La156 => None,
            I::La157 => None,
            I::Ce119 => None,
            I::Ce120 => None,
            I::Ce121 => None,
            I::Ce122 => None,
            I::Ce123 => None,
            I::Ce124 => None,
            I::Ce125 => None,
            I::Ce126 => None,
            I::Ce127 => None,
            I::Ce128 => None,
            I::Ce129 => None,
            I::Ce130 => None,
            I::Ce131 => None,
            I::Ce132 => None,
            I::Ce133 => None,
            I::Ce134 => None,
            I::Ce135 => None,
            I::Ce136 => pct!(0.186),
            I::Ce137 => None,
            I::Ce138 => pct!(0.251),
            I::Ce139 => None,
            I::Ce140 => pct!(88.449),
            I::Ce141 => None,
            I::Ce142 => pct!(11.114),
            I::Ce143 => None,
            I::Ce144 => None,
            I::Ce145 => None,
            I::Ce146 => None,
            I::Ce147 => None,
            I::Ce148 => None,
            I::Ce149 => None,
            I::Ce150 => None,
            I::Ce151 => None,
            I::Ce152 => None,
            I::Ce153 => None,
            I::Ce154 => None,
            I::Ce155 => None,
            I::Ce156 => None,
            I::Ce157 => None,
            I::Ce158 => None,
            I::Ce159 => None,
            I::Pr121 => None,
            I::Pr122 => None,
            I::Pr123 => None,
            I::Pr124 => None,
            I::Pr125 => None,
            I::Pr126 => None,
            I::Pr127 => None,
            I::Pr128 => None,
            I::Pr129 => None,
            I::Pr130 => None,
            I::Pr131 => None,
            I::Pr132 => None,
            I::Pr133 => None,
            I::Pr134 => None,
            I::Pr135 => None,
            I::Pr136 => None,
            I::Pr137 => None,
            I::Pr138 => None,
            I::Pr139 => None,
            I::Pr140 => None,
            I::Pr141 => pct!(100.0),
            I::Pr142 => None,
            I::Pr143 => None,
            I::Pr144 => None,
            I::Pr145 => None,
            I::Pr146 => None,
            I::Pr147 => None,
            I::Pr148 => None,
            I::Pr149 => None,
            I::Pr150 => None,
            I::Pr151 => None,
            I::Pr152 => None,
            I::Pr153 => None,
            I::Pr154 => None,
            I::Pr155 => None,
            I::Pr156 => None,
            I::Pr157 => None,
            I::Pr158 => None,
            I::Pr159 => None,
            I::Pr160 => None,
            I::Pr161 => None,
            I::Nd124 => None,
            I::Nd125 => None,
            I::Nd126 => None,
            I::Nd127 => None,
            I::Nd128 => None,
            I::Nd129 => None,
            I::Nd130 => None,
            I::Nd131 => None,
            I::Nd132 => None,
            I::Nd133 => None,
            I::Nd134 => None,
            I::Nd135 => None,
            I::Nd136 => None,
            I::Nd137 => None,
            I::Nd138 => None,
            I::Nd139 => None,
            I::Nd140 => None,
            I::Nd141 => None,
            I::Nd142 => pct!(27.153),
            I::Nd143 => pct!(12.173),
            I::Nd144 => pct!(23.798),
            I::Nd145 => pct!(8.293),
            I::Nd146 => pct!(17.189),
            I::Nd147 => None,
            I::Nd148 => pct!(5.756),
            I::Nd149 => None,
            I::Nd150 => pct!(5.638),
            I::Nd151 => None,
            I::Nd152 => None,
            I::Nd153 => None,
            I::Nd154 => None,
            I::Nd155 => None,
            I::Nd156 => None,
            I::Nd157 => None,
            I::Nd158 => None,
            I::Nd159 => None,
            I::Nd160 => None,
            I::Nd161 => None,
            I::Nd162 => None,
            I::Nd163 => None,
            I::Pm126 => None,
            I::Pm127 => None,
            I::Pm128 => None,
            I::Pm129 => None,
            I::Pm130 => None,
            I::Pm131 => None,
            I::Pm132 => None,
            I::Pm133 => None,
            I::Pm134 => None,
            I::Pm135 => None,
            I::Pm136 => None,
            I::Pm137 => None,
            I::Pm138 => None,
            I::Pm139 => None,
            I::Pm140 => None,
            I::Pm141 => None,
            I::Pm142 => None,
            I::Pm143 => None,
            I::Pm144 => None,
            I::Pm145 => None,
            I::Pm146 => None,
            I::Pm147 => None,
            I::Pm148 => None,
            I::Pm149 => None,
            I::Pm150 => None,
            I::Pm151 => None,
            I::Pm152 => None,
            I::Pm153 => None,
            I::Pm154 => None,
            I::Pm155 => None,
            I::Pm156 => None,
            I::Pm157 => None,
            I::Pm158 => None,
            I::Pm159 => None,
            I::Pm160 => None,
            I::Pm161 => None,
            I::Pm162 => None,
            I::Pm163 => None,
            I::Pm164 => None,
            I::Pm165 => None,
            I::Sm128 => None,
            I::Sm129 => None,
            I::Sm130 => None,
            I::Sm131 => None,
            I::Sm132 => None,
            I::Sm133 => None,
            I::Sm134 => None,
            I::Sm135 => None,
            I::Sm136 => None,
            I::Sm137 => None,
            I::Sm138 => None,
            I::Sm139 => None,
            I::Sm140 => None,
            I::Sm141 => None,
            I::Sm142 => None,
            I::Sm143 => None,
            I::Sm144 => pct!(3.08),
            I::Sm145 => None,
            I::Sm146 => None,
            I::Sm147 => pct!(15.0),
            I::Sm148 => pct!(11.25),
            I::Sm149 => pct!(13.82),
            I::Sm150 => pct!(7.37),
            I::Sm151 => None,
            I::Sm152 => pct!(26.74),
            I::Sm153 => None,
            I::Sm154 => pct!(22.74),
            I::Sm155 => None,
            I::Sm156 => None,
            I::Sm157 => None,
            I::Sm158 => None,
            I::Sm159 => None,
            I::Sm160 => None,
            I::Sm161 => None,
            I::Sm162 => None,
            I::Sm163 => None,
            I::Sm164 => None,
            I::Sm165 => None,
            I::Sm166 => None,
            I::Sm167 => None,
            I::Sm168 => None,
            I::Eu130 => None,
            I::Eu131 => None,
            I::Eu132 => None,
            I::Eu133 => None,
            I::Eu134 => None,
            I::Eu135 => None,
            I::Eu136 => None,
            I::Eu137 => None,
            I::Eu138 => None,
            I::Eu139 => None,
            I::Eu140 => None,
            I::Eu141 => None,
            I::Eu142 => None,
            I::Eu143 => None,
            I::Eu144 => None,
            I::Eu145 => None,
            I::Eu146 => None,
            I::Eu147 => None,
            I::Eu148 => None,
            I::Eu149 => None,
            I::Eu150 => None,
            I::Eu151 => pct!(47.81),
            I::Eu152 => None,
            I::Eu153 => pct!(52.19),
            I::Eu154 => None,
            I::Eu155 => None,
            I::Eu156 => None,
            I::Eu157 => None,
            I::Eu158 => None,
            I::Eu159 => None,
            I::Eu160 => None,
            I::Eu161 => None,
            I::Eu162 => None,
            I::Eu163 => None,
            I::Eu164 => None,
            I::Eu165 => None,
            I::Eu166 => None,
            I::Eu167 => None,
            I::Eu168 => None,
            I::Eu169 => None,
            I::Eu170 => None,
            I::Gd133 => None,
            I::Gd134 => None,
            I::Gd135 => None,
            I::Gd136 => None,
            I::Gd137 => None,
            I::Gd138 => None,
            I::Gd139 => None,
            I::Gd140 => None,
            I::Gd141 => None,
            I::Gd142 => None,
            I::Gd143 => None,
            I::Gd144 => None,
            I::Gd145 => None,
            I::Gd146 => None,
            I::Gd147 => None,
            I::Gd148 => None,
            I::Gd149 => None,
            I::Gd150 => None,
            I::Gd151 => None,
            I::Gd152 => pct!(0.2),
            I::Gd153 => None,
            I::Gd154 => pct!(2.18),
            I::Gd155 => pct!(14.8),
            I::Gd156 => pct!(20.47),
            I::Gd157 => pct!(15.65),
            I::Gd158 => pct!(24.84),
            I::Gd159 => None,
            I::Gd160 => pct!(21.86),
            I::Gd161 => None,
            I::Gd162 => None,
            I::Gd163 => None,
            I::Gd164 => None,
            I::Gd165 => None,
            I::Gd166 => None,
            I::Gd167 => None,
            I::Gd168 => None,
            I::Gd169 => None,
            I::Gd170 => None,
            I::Gd171 => None,
            I::Gd172 => None,
            I::Tb135 => None,
            I::Tb136 => None,
            I::Tb137 => None,
            I::Tb138 => None,
            I::Tb139 => None,
            I::Tb140 => None,
            I::Tb141 => None,
            I::Tb142 => None,
            I::Tb143 => None,
            I::Tb144 => None,
            I::Tb145 => None,
            I::Tb146 => None,
            I::Tb147 => None,
            I::Tb148 => None,
            I::Tb149 => None,
            I::Tb150 => None,
            I::Tb151 => None,
            I::Tb152 => None,
            I::Tb153 => None,
            I::Tb154 => None,
            I::Tb155 => None,
            I::Tb156 => None,
            I::Tb157 => None,
            I::Tb158 => None,
            I::Tb159 => pct!(100.0),
            I::Tb160 => None,
            I::Tb161 => None,
            I::Tb162 => None,
            I::Tb163 => None,
            I::Tb164 => None,
            I::Tb165 => None,
            I::Tb166 => None,
            I::Tb167 => None,
            I::Tb168 => None,
            I::Tb169 => None,
            I::Tb170 => None,
            I::Tb171 => None,
            I::Tb172 => None,
            I::Tb173 => None,
            I::Tb174 => None,
            I::Dy138 => None,
            I::Dy139 => None,
            I::Dy140 => None,
            I::Dy141 => None,
            I::Dy142 => None,
            I::Dy143 => None,
            I::Dy144 => None,
            I::Dy145 => None,
            I::Dy146 => None,
            I::Dy147 => None,
            I::Dy148 => None,
            I::Dy149 => None,
            I::Dy150 => None,
            I::Dy151 => None,
            I::Dy152 => None,
            I::Dy153 => None,
            I::Dy154 => None,
            I::Dy155 => None,
            I::Dy156 => pct!(0.056),
            I::Dy157 => None,
            I::Dy158 => pct!(0.095),
            I::Dy159 => None,
            I::Dy160 => pct!(2.329),
            I::Dy161 => pct!(18.889),
            I::Dy162 => pct!(25.475),
            I::Dy163 => pct!(24.896),
            I::Dy164 => pct!(28.26),
            I::Dy165 => None,
            I::Dy166 => None,
            I::Dy167 => None,
            I::Dy168 => None,
            I::Dy169 => None,
            I::Dy170 => None,
            I::Dy171 => None,
            I::Dy172 => None,
            I::Dy173 => None,
            I::Dy174 => None,
            I::Dy175 => None,
            I::Dy176 => None,
            I::Ho140 => None,
            I::Ho141 => None,
            I::Ho142 => None,
            I::Ho143 => None,
            I::Ho144 => None,
            I::Ho145 => None,
            I::Ho146 => None,
            I::Ho147 => None,
            I::Ho148 => None,
            I::Ho149 => None,
            I::Ho150 => None,
            I::Ho151 => None,
            I::Ho152 => None,
            I::Ho153 => None,
            I::Ho154 => None,
            I::Ho155 => None,
            I::Ho156 => None,
            I::Ho157 => None,
            I::Ho158 => None,
            I::Ho159 => None,
            I::Ho160 => None,
            I::Ho161 => None,
            I::Ho162 => None,
            I::Ho163 => None,
            I::Ho164 => None,
            I::Ho165 => pct!(100.0),
            I::Ho166 => None,
            I::Ho167 => None,
            I::Ho168 => None,
            I::Ho169 => None,
            I::Ho170 => None,
            I::Ho171 => None,
            I::Ho172 => None,
            I::Ho173 => None,
            I::Ho174 => None,
            I::Ho175 => None,
            I::Ho176 => None,
            I::Ho177 => None,
            I::Ho178 => None,
            I::Er142 => None,
            I::Er143 => None,
            I::Er144 => None,
            I::Er145 => None,
            I::Er146 => None,
            I::Er147 => None,
            I::Er148 => None,
            I::Er149 => None,
            I::Er150 => None,
            I::Er151 => None,
            I::Er152 => None,
            I::Er153 => None,
            I::Er154 => None,
            I::Er155 => None,
            I::Er156 => None,
            I::Er157 => None,
            I::Er158 => None,
            I::Er159 => None,
            I::Er160 => None,
            I::Er161 => None,
            I::Er162 => pct!(0.139),
            I::Er163 => None,
            I::Er164 => pct!(1.601),
            I::Er165 => None,
            I::Er166 => pct!(33.503),
            I::Er167 => pct!(22.869),
            I::Er168 => pct!(26.978),
            I::Er169 => None,
            I::Er170 => pct!(14.91),
            I::Er171 => None,
            I::Er172 => None,
            I::Er173 => None,
            I::Er174 => None,
            I::Er175 => None,
            I::Er176 => None,
            I::Er177 => None,
            I::Er178 => None,
            I::Er179 => None,
            I::Er180 => None,
            I::Tm144 => None,
            I::Tm145 => None,
            I::Tm146 => None,
            I::Tm147 => None,
            I::Tm148 => None,
            I::Tm149 => None,
            I::Tm150 => None,
            I::Tm151 => None,
            I::Tm152 => None,
            I::Tm153 => None,
            I::Tm154 => None,
            I::Tm155 => None,
            I::Tm156 => None,
            I::Tm157 => None,
            I::Tm158 => None,
            I::Tm159 => None,
            I::Tm160 => None,
            I::Tm161 => None,
            I::Tm162 => None,
            I::Tm163 => None,
            I::Tm164 => None,
            I::Tm165 => None,
            I::Tm166 => None,
            I::Tm167 => None,
            I::Tm168 => None,
            I::Tm169 => pct!(100.0),
            I::Tm170 => None,
            I::Tm171 => None,
            I::Tm172 => None,
            I::Tm173 => None,
            I::Tm174 => None,
            I::Tm175 => None,
            I::Tm176 => None,
            I::Tm177 => None,
            I::Tm178 => None,
            I::Tm179 => None,
            I::Tm180 => None,
            I::Tm181 => None,
            I::Tm182 => None,
            I::Yb148 => None,
            I::Yb149 => None,
            I::Yb150 => None,
            I::Yb151 => None,
            I::Yb152 => None,
            I::Yb153 => None,
            I::Yb154 => None,
            I::Yb155 => None,
            I::Yb156 => None,
            I::Yb157 => None,
            I::Yb158 => None,
            I::Yb159 => None,
            I::Yb160 => None,
            I::Yb161 => None,
            I::Yb162 => None,
            I::Yb163 => None,
            I::Yb164 => None,
            I::Yb165 => None,
            I::Yb166 => None,
            I::Yb167 => None,
            I::Yb168 => pct!(0.123),
            I::Yb169 => None,
            I::Yb170 => pct!(2.982),
            I::Yb171 => pct!(14.086),
            I::Yb172 => pct!(21.686),
            I::Yb173 => pct!(16.103),
            I::Yb174 => pct!(32.025),
            I::Yb175 => None,
            I::Yb176 => pct!(12.995),
            I::Yb177 => None,
            I::Yb178 => None,
            I::Yb179 => None,
            I::Yb180 => None,
            I::Yb181 => None,
            I::Yb182 => None,
            I::Yb183 => None,
            I::Yb184 => None,
            I::Yb185 => None,
            I::Lu150 => None,
            I::Lu151 => None,
            I::Lu152 => None,
            I::Lu153 => None,
            I::Lu154 => None,
            I::Lu155 => None,
            I::Lu156 => None,
            I::Lu157 => None,
            I::Lu158 => None,
            I::Lu159 => None,
            I::Lu160 => None,
            I::Lu161 => None,
            I::Lu162 => None,
            I::Lu163 => None,
            I::Lu164 => None,
            I::Lu165 => None,
            I::Lu166 => None,
            I::Lu167 => None,
            I::Lu168 => None,
            I::Lu169 => None,
            I::Lu170 => None,
            I::Lu171 => None,
            I::Lu172 => None,
            I::Lu173 => None,
            I::Lu174 => None,
            I::Lu175 => pct!(97.401),
            I::Lu176 => pct!(2.599),
            I::Lu177 => None,
            I::Lu178 => None,
            I::Lu179 => None,
            I::Lu180 => None,
            I::Lu181 => None,
            I::Lu182 => None,
            I::Lu183 => None,
            I::Lu184 => None,
            I::Lu185 => None,
            I::Lu186 => None,
            I::Lu187 => None,
            I::Lu188 => None,
            I::Hf153 => None,
            I::Hf154 => None,
            I::Hf155 => None,
            I::Hf156 => None,
            I::Hf157 => None,
            I::Hf158 => None,
            I::Hf159 => None,
            I::Hf160 => None,
            I::Hf161 => None,
            I::Hf162 => None,
            I::Hf163 => None,
            I::Hf164 => None,
            I::Hf165 => None,
            I::Hf166 => None,
            I::Hf167 => None,
            I::Hf168 => None,
            I::Hf169 => None,
            I::Hf170 => None,
            I::Hf171 => None,
            I::Hf172 => None,
            I::Hf173 => None,
            I::Hf174 => pct!(0.16),
            I::Hf175 => None,
            I::Hf176 => pct!(5.26),
            I::Hf177 => pct!(18.6),
            I::Hf178 => pct!(27.28),
            I::Hf179 => pct!(13.62),
            I::Hf180 => pct!(35.08),
            I::Hf181 => None,
            I::Hf182 => None,
            I::Hf183 => None,
            I::Hf184 => None,
            I::Hf185 => None,
            I::Hf186 => None,
            I::Hf187 => None,
            I::Hf188 => None,
            I::Hf189 => None,
            I::Hf190 => None,
            I::Ta155 => None,
            I::Ta156 => None,
            I::Ta157 => None,
            I::Ta158 => None,
            I::Ta159 => None,
            I::Ta160 => None,
            I::Ta161 => None,
            I::Ta162 => None,
            I::Ta163 => None,
            I::Ta164 => None,
            I::Ta165 => None,
            I::Ta166 => None,
            I::Ta167 => None,
            I::Ta168 => None,
            I::Ta169 => None,
            I::Ta170 => None,
            I::Ta171 => None,
            I::Ta172 => None,
            I::Ta173 => None,
            I::Ta174 => None,
            I::Ta175 => None,
            I::Ta176 => None,
            I::Ta177 => None,
            I::Ta178 => None,
            I::Ta179 => None,
            I::Ta180 => None,
            I::Ta181 => pct!(99.98799),
            I::Ta182 => None,
            I::Ta183 => None,
            I::Ta184 => None,
            I::Ta185 => None,
            I::Ta186 => None,
            I::Ta187 => None,
            I::Ta188 => None,
            I::Ta189 => None,
            I::Ta190 => None,
            I::Ta191 => None,
            I::Ta192 => None,
            I::Ta193 => None,
            I::Ta194 => None,
            I::W157 => None,
            I::W158 => None,
            I::W159 => None,
            I::W160 => None,
            I::W161 => None,
            I::W162 => None,
            I::W163 => None,
            I::W164 => None,
            I::W165 => None,
            I::W166 => None,
            I::W167 => None,
            I::W168 => None,
            I::W169 => None,
            I::W170 => None,
            I::W171 => None,
            I::W172 => None,
            I::W173 => None,
            I::W174 => None,
            I::W175 => None,
            I::W176 => None,
            I::W177 => None,
            I::W178 => None,
            I::W179 => None,
            I::W180 => pct!(0.12),
            I::W181 => None,
            I::W182 => pct!(26.5),
            I::W183 => pct!(14.31),
            I::W184 => pct!(30.64),
            I::W185 => None,
            I::W186 => pct!(28.43),
            I::W187 => None,
            I::W188 => None,
            I::W189 => None,
            I::W190 => None,
            I::W191 => None,
            I::W192 => None,
            I::W193 => None,
            I::W194 => None,
            I::W195 => None,
            I::W196 => None,
            I::W197 => None,
            I::Re159 => None,
            I::Re160 => None,
            I::Re161 => None,
            I::Re162 => None,
            I::Re163 => None,
            I::Re164 => None,
            I::Re165 => None,
            I::Re166 => None,
            I::Re167 => None,
            I::Re168 => None,
            I::Re169 => None,
            I::Re170 => None,
            I::Re171 => None,
            I::Re172 => None,
            I::Re173 => None,
            I::Re174 => None,
            I::Re175 => None,
            I::Re176 => None,
            I::Re177 => None,
            I::Re178 => None,
            I::Re179 => None,
            I::Re180 => None,
            I::Re181 => None,
            I::Re182 => None,
            I::Re183 => None,
            I::Re184 => None,
            I::Re185 => pct!(37.4),
            I::Re186 => None,
            I::Re187 => pct!(62.6),
            I::Re188 => None,
            I::Re189 => None,
            I::Re190 => None,
            I::Re191 => None,
            I::Re192 => None,
            I::Re193 => None,
            I::Re194 => None,
            I::Re195 => None,
            I::Re196 => None,
            I::Re197 => None,
            I::Re198 => None,
            I::Re199 => None,
            I::Os161 => None,
            I::Os162 => None,
            I::Os163 => None,
            I::Os164 => None,
            I::Os165 => None,
            I::Os166 => None,
            I::Os167 => None,
            I::Os168 => None,
            I::Os169 => None,
            I::Os170 => None,
            I::Os171 => None,
            I::Os172 => None,
            I::Os173 => None,
            I::Os174 => None,
            I::Os175 => None,
            I::Os176 => None,
            I::Os177 => None,
            I::Os178 => None,
            I::Os179 => None,
            I::Os180 => None,
            I::Os181 => None,
            I::Os182 => None,
            I::Os183 => None,
            I::Os184 => pct!(0.02),
            I::Os185 => None,
            I::Os186 => pct!(1.59),
            I::Os187 => pct!(1.96),
            I::Os188 => pct!(13.24),
            I::Os189 => pct!(16.15),
            I::Os190 => pct!(26.26),
            I::Os191 => None,
            I::Os192 => pct!(40.78),
            I::Os193 => None,
            I::Os194 => None,
            I::Os195 => None,
            I::Os196 => None,
            I::Os197 => None,
            I::Os198 => None,
            I::Os199 => None,
            I::Os200 => None,
            I::Os201 => None,
            I::Os202 => None,
            I::Os203 => None,
            I::Ir163 => None,
            I::Ir164 => None,
            I::Ir165 => None,
            I::Ir166 => None,
            I::Ir167 => None,
            I::Ir168 => None,
            I::Ir169 => None,
            I::Ir170 => None,
            I::Ir171 => None,
            I::Ir172 => None,
            I::Ir173 => None,
            I::Ir174 => None,
            I::Ir175 => None,
            I::Ir176 => None,
            I::Ir177 => None,
            I::Ir178 => None,
            I::Ir179 => None,
            I::Ir180 => None,
            I::Ir181 => None,
            I::Ir182 => None,
            I::Ir183 => None,
            I::Ir184 => None,
            I::Ir185 => None,
            I::Ir186 => None,
            I::Ir187 => None,
            I::Ir188 => None,
            I::Ir189 => None,
            I::Ir190 => None,
            I::Ir191 => pct!(37.3),
            I::Ir192 => None,
            I::Ir193 => pct!(62.7),
            I::Ir194 => None,
            I::Ir195 => None,
            I::Ir196 => None,
            I::Ir197 => None,
            I::Ir198 => None,
            I::Ir199 => None,
            I::Ir200 => None,
            I::Ir201 => None,
            I::Ir202 => None,
            I::Ir203 => None,
            I::Ir204 => None,
            I::Ir205 => None,
            I::Pt165 => None,
            I::Pt166 => None,
            I::Pt167 => None,
            I::Pt168 => None,
            I::Pt169 => None,
            I::Pt170 => None,
            I::Pt171 => None,
            I::Pt172 => None,
            I::Pt173 => None,
            I::Pt174 => None,
            I::Pt175 => None,
            I::Pt176 => None,
            I::Pt177 => None,
            I::Pt178 => None,
            I::Pt179 => None,
            I::Pt180 => None,
            I::Pt181 => None,
            I::Pt182 => None,
            I::Pt183 => None,
            I::Pt184 => None,
            I::Pt185 => None,
            I::Pt186 => None,
            I::Pt187 => None,
            I::Pt188 => None,
            I::Pt189 => None,
            I::Pt190 => pct!(0.012),
            I::Pt191 => None,
            I::Pt192 => pct!(0.782),
            I::Pt193 => None,
            I::Pt194 => pct!(32.864),
            I::Pt195 => pct!(33.775),
            I::Pt196 => pct!(25.211),
            I::Pt197 => None,
            I::Pt198 => pct!(7.356),
            I::Pt199 => None,
            I::Pt200 => None,
            I::Pt201 => None,
            I::Pt202 => None,
            I::Pt203 => None,
            I::Pt204 => None,
            I::Pt205 => None,
            I::Pt206 => None,
            I::Pt207 => None,
            I::Pt208 => None,
            I::Au168 => None,
            I::Au169 => None,
            I::Au170 => None,
            I::Au171 => None,
            I::Au172 => None,
            I::Au173 => None,
            I::Au174 => None,
            I::Au175 => None,
            I::Au176 => None,
            I::Au177 => None,
            I::Au178 => None,
            I::Au179 => None,
            I::Au180 => None,
            I::Au181 => None,
            I::Au182 => None,
            I::Au183 => None,
            I::Au184 => None,
            I::Au185 => None,
            I::Au186 => None,
            I::Au187 => None,
            I::Au188 => None,
            I::Au189 => None,
            I::Au190 => None,
            I::Au191 => None,
            I::Au192 => None,
            I::Au193 => None,
            I::Au194 => None,
            I::Au195 => None,
            I::Au196 => None,
            I::Au197 => pct!(100.0),
            I::Au198 => None,
            I::Au199 => None,
            I::Au200 => None,
            I::Au201 => None,
            I::Au202 => None,
            I::Au203 => None,
            I::Au204 => None,
            I::Au205 => None,
            I::Au206 => None,
            I::Au207 => None,
            I::Au208 => None,
            I::Au209 => None,
            I::Au210 => None,
            I::Hg170 => None,
            I::Hg171 => None,
            I::Hg172 => None,
            I::Hg173 => None,
            I::Hg174 => None,
            I::Hg175 => None,
            I::Hg176 => None,
            I::Hg177 => None,
            I::Hg178 => None,
            I::Hg179 => None,
            I::Hg180 => None,
            I::Hg181 => None,
            I::Hg182 => None,
            I::Hg183 => None,
            I::Hg184 => None,
            I::Hg185 => None,
            I::Hg186 => None,
            I::Hg187 => None,
            I::Hg188 => None,
            I::Hg189 => None,
            I::Hg190 => None,
            I::Hg191 => None,
            I::Hg192 => None,
            I::Hg193 => None,
            I::Hg194 => None,
            I::Hg195 => None,
            I::Hg196 => pct!(0.15),
            I::Hg197 => None,
            I::Hg198 => pct!(10.04),
            I::Hg199 => pct!(16.94),
            I::Hg200 => pct!(23.14),
            I::Hg201 => pct!(13.17),
            I::Hg202 => pct!(29.74),
            I::Hg203 => None,
            I::Hg204 => pct!(6.82),
            I::Hg205 => None,
            I::Hg206 => None,
            I::Hg207 => None,
            I::Hg208 => None,
            I::Hg209 => None,
            I::Hg210 => None,
            I::Hg211 => None,
            I::Hg212 => None,
            I::Hg213 => None,
            I::Hg214 => None,
            I::Hg215 => None,
            I::Hg216 => None,
            I::Tl176 => None,
            I::Tl177 => None,
            I::Tl178 => None,
            I::Tl179 => None,
            I::Tl180 => None,
            I::Tl181 => None,
            I::Tl182 => None,
            I::Tl183 => None,
            I::Tl184 => None,
            I::Tl185 => None,
            I::Tl186 => None,
            I::Tl187 => None,
            I::Tl188 => None,
            I::Tl189 => None,
            I::Tl190 => None,
            I::Tl191 => None,
            I::Tl192 => None,
            I::Tl193 => None,
            I::Tl194 => None,
            I::Tl195 => None,
            I::Tl196 => None,
            I::Tl197 => None,
            I::Tl198 => None,
            I::Tl199 => None,
            I::Tl200 => None,
            I::Tl201 => None,
            I::Tl202 => None,
            I::Tl203 => pct!(29.515),
            I::Tl204 => None,
            I::Tl205 => pct!(70.485),
            I::Tl206 => None,
            I::Tl207 => None,
            I::Tl208 => None,
            I::Tl209 => None,
            I::Tl210 => None,
            I::Tl211 => None,
            I::Tl212 => None,
            I::Tl213 => None,
            I::Tl214 => None,
            I::Tl215 => None,
            I::Tl216 => None,
            I::Tl217 => None,
            I::Tl218 => None,
            I::Pb178 => None,
            I::Pb179 => None,
            I::Pb180 => None,
            I::Pb181 => None,
            I::Pb182 => None,
            I::Pb183 => None,
            I::Pb184 => None,
            I::Pb185 => None,
            I::Pb186 => None,
            I::Pb187 => None,
            I::Pb188 => None,
            I::Pb189 => None,
            I::Pb190 => None,
            I::Pb191 => None,
            I::Pb192 => None,
            I::Pb193 => None,
            I::Pb194 => None,
            I::Pb195 => None,
            I::Pb196 => None,
            I::Pb197 => None,
            I::Pb198 => None,
            I::Pb199 => None,
            I::Pb200 => None,
            I::Pb201 => None,
            I::Pb202 => None,
            I::Pb203 => None,
            I::Pb204 => pct!(1.4),
            I::Pb205 => None,
            I::Pb206 => pct!(24.1),
            I::Pb207 => pct!(22.1),
            I::Pb208 => pct!(52.4),
            I::Pb209 => None,
            I::Pb210 => None,
            I::Pb211 => None,
            I::Pb212 => None,
            I::Pb213 => None,
            I::Pb214 => None,
            I::Pb215 => None,
            I::Pb216 => None,
            I::Pb217 => None,
            I::Pb218 => None,
            I::Pb219 => None,
            I::Pb220 => None,
            I::Bi184 => None,
            I::Bi185 => None,
            I::Bi186 => None,
            I::Bi187 => None,
            I::Bi188 => None,
            I::Bi189 => None,
            I::Bi190 => None,
            I::Bi191 => None,
            I::Bi192 => None,
            I::Bi193 => None,
            I::Bi194 => None,
            I::Bi195 => None,
            I::Bi196 => None,
            I::Bi197 => None,
            I::Bi198 => None,
            I::Bi199 => None,
            I::Bi200 => None,
            I::Bi201 => None,
            I::Bi202 => None,
            I::Bi203 => None,
            I::Bi204 => None,
            I::Bi205 => None,
            I::Bi206 => None,
            I::Bi207 => None,
            I::Bi208 => None,
            I::Bi209 => pct!(100.0),
            I::Bi210 => None,
            I::Bi211 => None,
            I::Bi212 => None,
            I::Bi213 => None,
            I::Bi214 => None,
            I::Bi215 => None,
            I::Bi216 => None,
            I::Bi217 => None,
            I::Bi218 => None,
            I::Bi219 => None,
            I::Bi220 => None,
            I::Bi221 => None,
            I::Bi222 => None,
            I::Bi223 => None,
            I::Bi224 => None,
            I::Po186 => None,
            I::Po187 => None,
            I::Po188 => None,
            I::Po189 => None,
            I::Po190 => None,
            I::Po191 => None,
            I::Po192 => None,
            I::Po193 => None,
            I::Po194 => None,
            I::Po195 => None,
            I::Po196 => None,
            I::Po197 => None,
            I::Po198 => None,
            I::Po199 => None,
            I::Po200 => None,
            I::Po201 => None,
            I::Po202 => None,
            I::Po203 => None,
            I::Po204 => None,
            I::Po205 => None,
            I::Po206 => None,
            I::Po207 => None,
            I::Po208 => None,
            I::Po209 => None,
            I::Po210 => None,
            I::Po211 => None,
            I::Po212 => None,
            I::Po213 => None,
            I::Po214 => None,
            I::Po215 => None,
            I::Po216 => None,
            I::Po217 => None,
            I::Po218 => None,
            I::Po219 => None,
            I::Po220 => None,
            I::Po221 => None,
            I::Po222 => None,
            I::Po223 => None,
            I::Po224 => None,
            I::Po225 => None,
            I::Po226 => None,
            I::Po227 => None,
            I::At191 => None,
            I::At192 => None,
            I::At193 => None,
            I::At194 => None,
            I::At195 => None,
            I::At196 => None,
            I::At197 => None,
            I::At198 => None,
            I::At199 => None,
            I::At200 => None,
            I::At201 => None,
            I::At202 => None,
            I::At203 => None,
            I::At204 => None,
            I::At205 => None,
            I::At206 => None,
            I::At207 => None,
            I::At208 => None,
            I::At209 => None,
            I::At210 => None,
            I::At211 => None,
            I::At212 => None,
            I::At213 => None,
            I::At214 => None,
            I::At215 => None,
            I::At216 => None,
            I::At217 => None,
            I::At218 => None,
            I::At219 => None,
            I::At220 => None,
            I::At221 => None,
            I::At222 => None,
            I::At223 => None,
            I::At224 => None,
            I::At225 => None,
            I::At226 => None,
            I::At227 => None,
            I::At228 => None,
            I::At229 => None,
            I::Rn193 => None,
            I::Rn194 => None,
            I::Rn195 => None,
            I::Rn196 => None,
            I::Rn197 => None,
            I::Rn198 => None,
            I::Rn199 => None,
            I::Rn200 => None,
            I::Rn201 => None,
            I::Rn202 => None,
            I::Rn203 => None,
            I::Rn204 => None,
            I::Rn205 => None,
            I::Rn206 => None,
            I::Rn207 => None,
            I::Rn208 => None,
            I::Rn209 => None,
            I::Rn210 => None,
            I::Rn211 => None,
            I::Rn212 => None,
            I::Rn213 => None,
            I::Rn214 => None,
            I::Rn215 => None,
            I::Rn216 => None,
            I::Rn217 => None,
            I::Rn218 => None,
            I::Rn219 => None,
            I::Rn220 => None,
            I::Rn221 => None,
            I::Rn222 => None,
            I::Rn223 => None,
            I::Rn224 => None,
            I::Rn225 => None,
            I::Rn226 => None,
            I::Rn227 => None,
            I::Rn228 => None,
            I::Rn229 => None,
            I::Rn230 => None,
            I::Rn231 => None,
            I::Fr197 => None,
            I::Fr198 => None,
            I::Fr199 => None,
            I::Fr200 => None,
            I::Fr201 => None,
            I::Fr202 => None,
            I::Fr203 => None,
            I::Fr204 => None,
            I::Fr205 => None,
            I::Fr206 => None,
            I::Fr207 => None,
            I::Fr208 => None,
            I::Fr209 => None,
            I::Fr210 => None,
            I::Fr211 => None,
            I::Fr212 => None,
            I::Fr213 => None,
            I::Fr214 => None,
            I::Fr215 => None,
            I::Fr216 => None,
            I::Fr217 => None,
            I::Fr218 => None,
            I::Fr219 => None,
            I::Fr220 => None,
            I::Fr221 => None,
            I::Fr222 => None,
            I::Fr223 => None,
            I::Fr224 => None,
            I::Fr225 => None,
            I::Fr226 => None,
            I::Fr227 => None,
            I::Fr228 => None,
            I::Fr229 => None,
            I::Fr230 => None,
            I::Fr231 => None,
            I::Fr232 => None,
            I::Fr233 => None,
            I::Ra201 => None,
            I::Ra202 => None,
            I::Ra203 => None,
            I::Ra204 => None,
            I::Ra205 => None,
            I::Ra206 => None,
            I::Ra207 => None,
            I::Ra208 => None,
            I::Ra209 => None,
            I::Ra210 => None,
            I::Ra211 => None,
            I::Ra212 => None,
            I::Ra213 => None,
            I::Ra214 => None,
            I::Ra215 => None,
            I::Ra216 => None,
            I::Ra217 => None,
            I::Ra218 => None,
            I::Ra219 => None,
            I::Ra220 => None,
            I::Ra221 => None,
            I::Ra222 => None,
            I::Ra223 => None,
            I::Ra224 => None,
            I::Ra225 => None,
            I::Ra226 => None,
            I::Ra227 => None,
            I::Ra228 => None,
            I::Ra229 => None,
            I::Ra230 => None,
            I::Ra231 => None,
            I::Ra232 => None,
            I::Ra233 => None,
            I::Ra234 => None,
            I::Ra235 => None,
            I::Ac205 => None,
            I::Ac206 => None,
            I::Ac207 => None,
            I::Ac208 => None,
            I::Ac209 => None,
            I::Ac210 => None,
            I::Ac211 => None,
            I::Ac212 => None,
            I::Ac213 => None,
            I::Ac214 => None,
            I::Ac215 => None,
            I::Ac216 => None,
            I::Ac217 => None,
            I::Ac218 => None,
            I::Ac219 => None,
            I::Ac220 => None,
            I::Ac221 => None,
            I::Ac222 => None,
            I::Ac223 => None,
            I::Ac224 => None,
            I::Ac225 => None,
            I::Ac226 => None,
            I::Ac227 => None,
            I::Ac228 => None,
            I::Ac229 => None,
            I::Ac230 => None,
            I::Ac231 => None,
            I::Ac232 => None,
            I::Ac233 => None,
            I::Ac234 => None,
            I::Ac235 => None,
            I::Ac236 => None,
            I::Ac237 => None,
            I::Th208 => None,
            I::Th209 => None,
            I::Th210 => None,
            I::Th211 => None,
            I::Th212 => None,
            I::Th213 => None,
            I::Th214 => None,
            I::Th215 => None,
            I::Th216 => None,
            I::Th217 => None,
            I::Th218 => None,
            I::Th219 => None,
            I::Th220 => None,
            I::Th221 => None,
            I::Th222 => None,
            I::Th223 => None,
            I::Th224 => None,
            I::Th225 => None,
            I::Th226 => None,
            I::Th227 => None,
            I::Th228 => None,
            I::Th229 => None,
            I::Th230 => pct!(0.02),
            I::Th231 => None,
            I::Th232 => pct!(99.98),
            I::Th233 => None,
            I::Th234 => None,
            I::Th235 => None,
            I::Th236 => None,
            I::Th237 => None,
            I::Th238 => None,
            I::Th239 => None,
            I::Pa211 => None,
            I::Pa212 => None,
            I::Pa213 => None,
            I::Pa214 => None,
            I::Pa215 => None,
            I::Pa216 => None,
            I::Pa217 => None,
            I::Pa218 => None,
            I::Pa219 => None,
            I::Pa220 => None,
            I::Pa221 => None,
            I::Pa222 => None,
            I::Pa223 => None,
            I::Pa224 => None,
            I::Pa225 => None,
            I::Pa226 => None,
            I::Pa227 => None,
            I::Pa228 => None,
            I::Pa229 => None,
            I::Pa230 => None,
            I::Pa231 => pct!(100.0),
            I::Pa232 => None,
            I::Pa233 => None,
            I::Pa234 => None,
            I::Pa235 => None,
            I::Pa236 => None,
            I::Pa237 => None,
            I::Pa238 => None,
            I::Pa239 => None,
            I::Pa240 => None,
            I::Pa241 => None,
            I::U215 => None,
            I::U216 => None,
            I::U217 => None,
            I::U218 => None,
            I::U219 => None,
            I::U220 => None,
            I::U221 => None,
            I::U222 => None,
            I::U223 => None,
            I::U224 => None,
            I::U225 => None,
            I::U226 => None,
            I::U227 => None,
            I::U228 => None,
            I::U229 => None,
            I::U230 => None,
            I::U231 => None,
            I::U232 => None,
            I::U233 => None,
            I::U234 => pct!(0.0054),
            I::U235 => pct!(0.7204),
            I::U236 => None,
            I::U237 => None,
            I::U238 => pct!(99.2742),
            I::U239 => None,
            I::U240 => None,
            I::U241 => None,
            I::U242 => None,
            I::U243 => None,
            I::Np219 => None,
            I::Np220 => None,
            I::Np221 => None,
            I::Np222 => None,
            I::Np223 => None,
            I::Np224 => None,
            I::Np225 => None,
            I::Np226 => None,
            I::Np227 => None,
            I::Np228 => None,
            I::Np229 => None,
            I::Np230 => None,
            I::Np231 => None,
            I::Np232 => None,
            I::Np233 => None,
            I::Np234 => None,
            I::Np235 => None,
            I::Np236 => None,
            I::Np237 => None,
            I::Np238 => None,
            I::Np239 => None,
            I::Np240 => None,
            I::Np241 => None,
            I::Np242 => None,
            I::Np243 => None,
            I::Np244 => None,
            I::Np245 => None,
            I::Pu221 => None,
            I::Pu222 => None,
            I::Pu223 => None,
            I::Pu224 => None,
            I::Pu225 => None,
            I::Pu226 => None,
            I::Pu227 => None,
            I::Pu228 => None,
            I::Pu229 => None,
            I::Pu230 => None,
            I::Pu231 => None,
            I::Pu232 => None,
            I::Pu233 => None,
            I::Pu234 => None,
            I::Pu235 => None,
            I::Pu236 => None,
            I::Pu237 => None,
            I::Pu238 => None,
            I::Pu239 => None,
            I::Pu240 => None,
            I::Pu241 => None,
            I::Pu242 => None,
            I::Pu243 => None,
            I::Pu244 => None,
            I::Pu245 => None,
            I::Pu246 => None,
            I::Pu247 => None,
            I::Am223 => None,
            I::Am224 => None,
            I::Am225 => None,
            I::Am226 => None,
            I::Am227 => None,
            I::Am228 => None,
            I::Am229 => None,
            I::Am230 => None,
            I::Am231 => None,
            I::Am232 => None,
            I::Am233 => None,
            I::Am234 => None,
            I::Am235 => None,
            I::Am236 => None,
            I::Am237 => None,
            I::Am238 => None,
            I::Am239 => None,
            I::Am240 => None,
            I::Am241 => None,
            I::Am242 => None,
            I::Am243 => None,
            I::Am244 => None,
            I::Am245 => None,
            I::Am246 => None,
            I::Am247 => None,
            I::Am248 => None,
            I::Am249 => None,
            I::Cm231 => None,
            I::Cm232 => None,
            I::Cm233 => None,
            I::Cm234 => None,
            I::Cm235 => None,
            I::Cm236 => None,
            I::Cm237 => None,
            I::Cm238 => None,
            I::Cm239 => None,
            I::Cm240 => None,
            I::Cm241 => None,
            I::Cm242 => None,
            I::Cm243 => None,
            I::Cm244 => None,
            I::Cm245 => None,
            I::Cm246 => None,
            I::Cm247 => None,
            I::Cm248 => None,
            I::Cm249 => None,
            I::Cm250 => None,
            I::Cm251 => None,
            I::Cm252 => None,
            I::Bk233 => None,
            I::Bk234 => None,
            I::Bk235 => None,
            I::Bk236 => None,
            I::Bk237 => None,
            I::Bk238 => None,
            I::Bk239 => None,
            I::Bk240 => None,
            I::Bk241 => None,
            I::Bk242 => None,
            I::Bk243 => None,
            I::Bk244 => None,
            I::Bk245 => None,
            I::Bk246 => None,
            I::Bk247 => None,
            I::Bk248 => None,
            I::Bk249 => None,
            I::Bk250 => None,
            I::Bk251 => None,
            I::Bk252 => None,
            I::Bk253 => None,
            I::Bk254 => None,
            I::Cf237 => None,
            I::Cf238 => None,
            I::Cf239 => None,
            I::Cf240 => None,
            I::Cf241 => None,
            I::Cf242 => None,
            I::Cf243 => None,
            I::Cf244 => None,
            I::Cf245 => None,
            I::Cf246 => None,
            I::Cf247 => None,
            I::Cf248 => None,
            I::Cf249 => None,
            I::Cf250 => None,
            I::Cf251 => None,
            I::Cf252 => None,
            I::Cf253 => None,
            I::Cf254 => None,
            I::Cf255 => None,
            I::Cf256 => None,
            I::Es239 => None,
            I::Es240 => None,
            I::Es241 => None,
            I::Es242 => None,
            I::Es243 => None,
            I::Es244 => None,
            I::Es245 => None,
            I::Es246 => None,
            I::Es247 => None,
            I::Es248 => None,
            I::Es249 => None,
            I::Es250 => None,
            I::Es251 => None,
            I::Es252 => None,
            I::Es253 => None,
            I::Es254 => None,
            I::Es255 => None,
            I::Es256 => None,
            I::Es257 => None,
            I::Es258 => None,
            I::Fm241 => None,
            I::Fm242 => None,
            I::Fm243 => None,
            I::Fm244 => None,
            I::Fm245 => None,
            I::Fm246 => None,
            I::Fm247 => None,
            I::Fm248 => None,
            I::Fm249 => None,
            I::Fm250 => None,
            I::Fm251 => None,
            I::Fm252 => None,
            I::Fm253 => None,
            I::Fm254 => None,
            I::Fm255 => None,
            I::Fm256 => None,
            I::Fm257 => None,
            I::Fm258 => None,
            I::Fm259 => None,
            I::Fm260 => None,
            I::Md244 => None,
            I::Md245 => None,
            I::Md246 => None,
            I::Md247 => None,
            I::Md248 => None,
            I::Md249 => None,
            I::Md250 => None,
            I::Md251 => None,
            I::Md252 => None,
            I::Md253 => None,
            I::Md254 => None,
            I::Md255 => None,
            I::Md256 => None,
            I::Md257 => None,
            I::Md258 => None,
            I::Md259 => None,
            I::Md260 => None,
            I::Md261 => None,
            I::Md262 => None,
            I::No248 => None,
            I::No249 => None,
            I::No250 => None,
            I::No251 => None,
            I::No252 => None,
            I::No253 => None,
            I::No254 => None,
            I::No255 => None,
            I::No256 => None,
            I::No257 => None,
            I::No258 => None,
            I::No259 => None,
            I::No260 => None,
            I::No261 => None,
            I::No262 => None,
            I::No263 => None,
            I::No264 => None,
            I::Lr251 => None,
            I::Lr252 => None,
            I::Lr253 => None,
            I::Lr254 => None,
            I::Lr255 => None,
            I::Lr256 => None,
            I::Lr257 => None,
            I::Lr258 => None,
            I::Lr259 => None,
            I::Lr260 => None,
            I::Lr261 => None,
            I::Lr262 => None,
            I::Lr263 => None,
            I::Lr264 => None,
            I::Lr265 => None,
            I::Lr266 => None,
            I::Rf253 => None,
            I::Rf254 => None,
            I::Rf255 => None,
            I::Rf256 => None,
            I::Rf257 => None,
            I::Rf258 => None,
            I::Rf259 => None,
            I::Rf260 => None,
            I::Rf261 => None,
            I::Rf262 => None,
            I::Rf263 => None,
            I::Rf264 => None,
            I::Rf265 => None,
            I::Rf266 => None,
            I::Rf267 => None,
            I::Rf268 => None,
            I::Db255 => None,
            I::Db256 => None,
            I::Db257 => None,
            I::Db258 => None,
            I::Db259 => None,
            I::Db260 => None,
            I::Db261 => None,
            I::Db262 => None,
            I::Db263 => None,
            I::Db264 => None,
            I::Db265 => None,
            I::Db266 => None,
            I::Db267 => None,
            I::Db268 => None,
            I::Db269 => None,
            I::Db270 => None,
            I::Sg258 => None,
            I::Sg259 => None,
            I::Sg260 => None,
            I::Sg261 => None,
            I::Sg262 => None,
            I::Sg263 => None,
            I::Sg264 => None,
            I::Sg265 => None,
            I::Sg266 => None,
            I::Sg267 => None,
            I::Sg268 => None,
            I::Sg269 => None,
            I::Sg270 => None,
            I::Sg271 => None,
            I::Sg272 => None,
            I::Sg273 => None,
            I::Bh260 => None,
            I::Bh261 => None,
            I::Bh262 => None,
            I::Bh263 => None,
            I::Bh264 => None,
            I::Bh265 => None,
            I::Bh266 => None,
            I::Bh267 => None,
            I::Bh268 => None,
            I::Bh269 => None,
            I::Bh270 => None,
            I::Bh271 => None,
            I::Bh272 => None,
            I::Bh273 => None,
            I::Bh274 => None,
            I::Bh275 => None,
            I::Bh276 => None,
            I::Bh277 => None,
            I::Bh278 => None,
            I::Hs263 => None,
            I::Hs264 => None,
            I::Hs265 => None,
            I::Hs266 => None,
            I::Hs267 => None,
            I::Hs268 => None,
            I::Hs269 => None,
            I::Hs270 => None,
            I::Hs271 => None,
            I::Hs272 => None,
            I::Hs273 => None,
            I::Hs274 => None,
            I::Hs275 => None,
            I::Hs276 => None,
            I::Hs277 => None,
            I::Hs278 => None,
            I::Hs279 => None,
            I::Hs280 => None,
            I::Mt265 => None,
            I::Mt266 => None,
            I::Mt267 => None,
            I::Mt268 => None,
            I::Mt269 => None,
            I::Mt270 => None,
            I::Mt271 => None,
            I::Mt272 => None,
            I::Mt273 => None,
            I::Mt274 => None,
            I::Mt275 => None,
            I::Mt276 => None,
            I::Mt277 => None,
            I::Mt278 => None,
            I::Mt279 => None,
            I::Mt280 => None,
            I::Mt281 => None,
            I::Mt282 => None,
            I::Ds267 => None,
            I::Ds268 => None,
            I::Ds269 => None,
            I::Ds270 => None,
            I::Ds271 => None,
            I::Ds272 => None,
            I::Ds273 => None,
            I::Ds274 => None,
            I::Ds275 => None,
            I::Ds276 => None,
            I::Ds277 => None,
            I::Ds278 => None,
            I::Ds279 => None,
            I::Ds280 => None,
            I::Ds281 => None,
            I::Ds282 => None,
            I::Ds283 => None,
            I::Ds284 => None,
            I::Rg272 => None,
            I::Rg273 => None,
            I::Rg274 => None,
            I::Rg275 => None,
            I::Rg276 => None,
            I::Rg277 => None,
            I::Rg278 => None,
            I::Rg279 => None,
            I::Rg280 => None,
            I::Rg281 => None,
            I::Rg282 => None,
            I::Rg283 => None,
            I::Rg284 => None,
            I::Rg285 => None,
            I::Rg286 => None,
            I::Cn276 => None,
            I::Cn277 => None,
            I::Cn278 => None,
            I::Cn279 => None,
            I::Cn280 => None,
            I::Cn281 => None,
            I::Cn282 => None,
            I::Cn283 => None,
            I::Cn284 => None,
            I::Cn285 => None,
            I::Cn286 => None,
            I::Cn287 => None,
            I::Cn288 => None,
            I::Nh278 => None,
            I::Nh279 => None,
            I::Nh280 => None,
            I::Nh281 => None,
            I::Nh282 => None,
            I::Nh283 => None,
            I::Nh284 => None,
            I::Nh285 => None,
            I::Nh286 => None,
            I::Nh287 => None,
            I::Nh288 => None,
            I::Nh289 => None,
            I::Nh290 => None,
            I::Fl284 => None,
            I::Fl285 => None,
            I::Fl286 => None,
            I::Fl287 => None,
            I::Fl288 => None,
            I::Fl289 => None,
            I::Fl290 => None,
            I::Fl291 => None,
            I::Mc287 => None,
            I::Mc288 => None,
            I::Mc289 => None,
            I::Mc290 => None,
            I::Mc291 => None,
            I::Mc292 => None,
            I::Lv289 => None,
            I::Lv290 => None,
            I::Lv291 => None,
            I::Lv292 => None,
            I::Lv293 => None,
            I::Ts291 => None,
            I::Ts292 => None,
            I::Ts293 => None,
            I::Ts294 => None,
            I::Og293 => None,
            I::Og294 => None,
            I::Og295 => None,
        }
    }
}
