﻿// A set of regression tests for equality/relational operators and the IComparer<> and IEqualityComparer<>
// implementation provided by ComparisonIdentity.Structural and HashIdentity.Structural

namespace FSharp.Core.Unittests.FSharp_Core.Microsoft_FSharp_Core

open System
open System.Numerics 
open FSharp.Core.Unittests.LibraryTestFx
open NUnit.Framework

module ComparersRegression =
    type RefWrap<'item> = { Item : 'item }

    [<Struct>]
    type ValueWrap<'item> =
        val Item : 'item
        new(item) = { Item = item }

    type UnionWrap<'item> =
    | UnionRaw of 'item
    | UnionRefWrap of RefWrap<'item>
    | UnionValueWrap of ValueWrap<'item>
    | UnionUnion of UnionWrap<'item>

    let createUnionWrap s =
        s
        |> Seq.collect (fun item ->
            [ UnionRaw item
              UnionRefWrap {Item = item}
              UnionValueWrap (ValueWrap item)
              UnionUnion (UnionRaw item)
              UnionUnion (UnionRefWrap {Item = item})
              UnionUnion (UnionValueWrap (ValueWrap item))
              UnionUnion (UnionUnion (UnionRaw item)) ])
        |> Array.ofSeq

    let createNullables s =
        seq {
            yield Nullable ()
            yield! s |> Seq.map (fun x -> Nullable x)
        }
        |> Array.ofSeq

    let createUnionTypes raw ref value union item =
        [| raw item
           ref item
           value item
           union (raw item)
           union (ref item)
           union (value item)
           union (union (raw item)) |]

    type Collection<'item, 'reftype, 'valuetype, 'uniontype> = {
        Array        : array<'item>
        ToRefType    : 'item -> 'reftype
        ToValueType  : 'item -> 'valuetype
        ToUnionTypes : 'item -> array<'uniontype>
    } with
        member this.ValueWrapArray =
            this.Array
            |> Array.map (fun item -> ValueWrap item)

        member this.RefWrapArray =
            this.Array
            |> Array.map (fun item -> { RefWrap.Item = item })

        member this.UnionWrapArray =
            this.Array
            |> createUnionWrap

        member this.ValueArray =
            this.Array
            |> Array.map this.ToValueType

        member this.RefArray =
            this.Array
            |> Array.map this.ToRefType

        member this.UnionArray =
            this.Array
            |> Array.collect this.ToUnionTypes

        member this.OptionArray =
            [|  yield None
                yield! this.Array |> Array.map Some |]

        member this.ArrayArray =
            [| yield! this.Array |> Array.map (fun x -> [| x |])
               yield! this.Array |> Array.mapi (fun i _ -> [| this.Array.[i]; this.Array.[(i+1)%this.Array.Length] |]) |]

        member this.ListArray =
            this.ArrayArray
            |> Array.map Array.toList

    module Bools =
        type TestType = bool

        let Values : array<TestType> = [| true; false|]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    module NullableBools =
        type TestType = Nullable<bool>

        let Values : array<TestType> = createNullables Bools.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module SBytes =
        type TestType = sbyte

        let Values : array<TestType> = [| SByte.MinValue; SByte.MaxValue; -1y; 0y; +1y |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableSbytes =
        type TestType = Nullable<sbyte>

        let Values : array<TestType> = createNullables SBytes.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Int16s =
        type TestType = int16

        let Values : array<TestType> = [| Int16.MaxValue; Int16.MaxValue; -1s; 0s; +1s |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableInt16s =
        type TestType = Nullable<int16>

        let Values : array<TestType> = createNullables Int16s.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Int32s =
        type TestType = int32

        let Values : array<TestType> = [| Int32.MinValue; Int32.MaxValue; -1; 0; +1 |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableInt32s =
        type TestType = Nullable<int32>

        let Values : array<TestType> = createNullables Int32s.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Int64s =
        type TestType = int64

        let Values : array<TestType> = [| Int64.MinValue; Int64.MaxValue; -1L; 0L; +1L |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableInt64s =
        type TestType = Nullable<int64>

        let Values : array<TestType> = createNullables Int64s.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NativeInts =
        type TestType = nativeint

        let Values : array<TestType> = [| -1n; 0n; +1n |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableNativeInts =
        type TestType = Nullable<nativeint>

        let Values : array<TestType> = createNullables NativeInts.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Bytes =
        type TestType = byte

        let Values : array<TestType> = [| Byte.MinValue; Byte.MaxValue; 0uy; 1uy; 2uy |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableBytes =
        type TestType = Nullable<byte>

        let Values : array<TestType> = createNullables Bytes.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Uint16s =
        type TestType = uint16

        let Values : array<TestType> = [| UInt16.MinValue; UInt16.MaxValue; 0us; 1us; 2us |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableUInt16s =
        type TestType = Nullable<uint16>

        let Values : array<TestType> = createNullables Uint16s.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module UInt32s =
        type TestType = uint32

        let Values : array<TestType> = [| UInt32.MinValue; UInt32.MaxValue; 0u; 1u; 2u|]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableUInt32s =
        type TestType = Nullable<uint32>

        let Values : array<TestType> = createNullables UInt32s.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module UInt64s =
        type TestType = uint64

        let Values : array<TestType> = [| UInt64.MinValue; UInt64.MaxValue; 0UL; 1UL; 2UL|]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableUInt64s =
        type TestType = Nullable<uint64>

        let Values : array<TestType> = createNullables UInt64s.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module UNativeInts =
        type TestType = unativeint

        let Values : array<TestType> = [| 0un; 1un; 2un |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableUNativeInts =
        type TestType = Nullable<unativeint>

        let Values : array<TestType> = createNullables UNativeInts.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Chars =
        type TestType = char

        let Values : array<TestType> = [| Char.MinValue; Char.MaxValue; '0'; '1'; '2' |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableChars =
        type TestType = Nullable<char>

        let Values : array<TestType> = createNullables Chars.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Strings =
        type TestType = string

        let Values : array<TestType> = [| null; String.Empty; "Hello, world!"; String('\u0000', 3); "\u0061\u030a"; "\u00e5" |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Decimals =
        type TestType = decimal

        let Values : array<TestType> = [| Decimal.MinValue; Decimal.MaxValue; Decimal.MinusOne; Decimal.Zero; Decimal.One |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableDecimals =
        type TestType = Nullable<decimal>

        let Values : array<TestType> = createNullables Decimals.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Floats =
        type TestType = float

        let Values : array<TestType> = [| Double.MinValue; Double.MaxValue; Double.Epsilon; Double.NaN; Double.NegativeInfinity; Double.PositiveInfinity; -1.; 0.; 1. |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableFloats =
        type TestType = Nullable<float>

        let Values : array<TestType> = createNullables Floats.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Float32s =
        type TestType = float32

        let Values : array<TestType> = [| Single.MinValue; Single.MaxValue; Single.Epsilon; Single.NaN; Single.NegativeInfinity; Single.PositiveInfinity; -1.f; 0.f; 1.f |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableFloat32s =
        type TestType = Nullable<float32>

        let Values : array<TestType> = createNullables Float32s.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module DateTimes =
        type TestType = System.DateTime

        let Values : array<TestType> = [| DateTime.MinValue; DateTime.MaxValue; DateTime(2015, 10, 8, 5, 39, 23) |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module NullableDateTimes =
        type TestType = Nullable<DateTime>

        let Values : array<TestType> = createNullables DateTimes.Values
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Tuple2s =
        type TestType = float*float

        let Values : array<TestType> = [| (nan, nan); (nan, 0.0); (0.0, nan); (0.0, 0.0) |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }
    // ----------------------------------------------------------------------------

    module Tuple3s =
        type TestType = float*float*float

        let Values : array<TestType> = [|
            (nan, nan, nan); (nan, nan, 0.0); (nan, 0.0, nan); (nan, 0.0, 0.0);
            (0.0, nan, nan); (0.0, nan, 0.0); (0.0, 0.0, nan); (0.0, 0.0, 0.0) |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Tuple4s =
        type TestType = float*float*float*float

        let Values : array<TestType> = [|
            (nan, nan, nan, nan); (nan, nan, nan, 0.0); (nan, nan, 0.0, nan); (nan, nan, 0.0, 0.0);
            (nan, 0.0, nan, nan); (nan, 0.0, nan, 0.0); (nan, 0.0, 0.0, nan); (nan, 0.0, 0.0, 0.0);
            (0.0, nan, nan, nan); (0.0, nan, nan, 0.0); (0.0, nan, 0.0, nan); (0.0, nan, 0.0, 0.0);
            (0.0, 0.0, nan, nan); (0.0, 0.0, nan, 0.0); (0.0, 0.0, 0.0, nan); (0.0, 0.0, 0.0, 0.0);
        |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    module Tuple5s =
        type TestType = float*float*float*float*float

        let Values : array<TestType> = [|
            (nan, nan, nan, nan, nan); (nan, nan, nan, nan, 0.0); (nan, nan, nan, 0.0, nan); (nan, nan, nan, 0.0, 0.0);
            (nan, nan, 0.0, nan, nan); (nan, nan, 0.0, nan, 0.0); (nan, nan, 0.0, 0.0, nan); (nan, nan, 0.0, 0.0, 0.0);
            (nan, 0.0, nan, nan, nan); (nan, 0.0, nan, nan, 0.0); (nan, 0.0, nan, 0.0, nan); (nan, 0.0, nan, 0.0, 0.0);
            (nan, 0.0, 0.0, nan, nan); (nan, 0.0, 0.0, nan, 0.0); (nan, 0.0, 0.0, 0.0, nan); (nan, 0.0, 0.0, 0.0, 0.0);
            (0.0, nan, nan, nan, nan); (0.0, nan, nan, nan, 0.0); (0.0, nan, nan, 0.0, nan); (0.0, nan, nan, 0.0, 0.0);
            (0.0, nan, 0.0, nan, nan); (0.0, nan, 0.0, nan, 0.0); (0.0, nan, 0.0, 0.0, nan); (0.0, nan, 0.0, 0.0, 0.0);
            (0.0, 0.0, nan, nan, nan); (0.0, 0.0, nan, nan, 0.0); (0.0, 0.0, nan, 0.0, nan); (0.0, 0.0, nan, 0.0, 0.0);
            (0.0, 0.0, 0.0, nan, nan); (0.0, 0.0, 0.0, nan, 0.0); (0.0, 0.0, 0.0, 0.0, nan); (0.0, 0.0, 0.0, 0.0, 0.0);
        |]
    
        type RefType = {
            Item : TestType
        }
    
        [<Struct>]
        type ValueType =
            val Item : TestType
            new(item) = { Item = item }

        type UnionType =
        | UnionRaw of TestType
        | UnionRefType of RefType
        | UnionValueType of ValueType
        | UnionUnion of UnionType

        let toRefType = fun x -> { Item = x}
        let toValueType = fun x -> ValueType x

        let createUnion =
            let raw x = UnionRaw x
            let ref x = UnionRefType (toRefType x)
            let value x = UnionValueType (toValueType x)
            let union x = UnionUnion x
            createUnionTypes raw ref value union

        let Collection = {
            Array        = Values
            ToRefType    = fun x -> { Item = x}
            ToValueType  = fun x -> ValueType x
            ToUnionTypes = createUnion
        }

    // ----------------------------------------------------------------------------

    type IOperation<'a> = abstract Exec : 'a -> 'a -> int

    exception ValidationException of lhs:obj * rhs:obj * expected:obj * received:obj

    let make_result_set<'a,'b when 'b : equality> (f:IOperation<'a>) (items:array<'a>) (validation_set:option<array<int>>)=
        let results = Array.zeroCreate (items.Length*items.Length)
        for i = 0 to items.Length-1 do
            for j = 0 to items.Length-1 do
                let index = i * items.Length + j
                let lhs = items.[i] 
                let rhs = items.[j]
                let result = f.Exec lhs rhs

                validation_set
                |> Option.iter (fun validation_set ->
                    let expected = validation_set.[index]
                    if expected <> result then
                        raise (ValidationException (box lhs, box rhs, expected, result)))

                results.[index] <- result
        results

    let toint b = if b then 1 else 0

    type EqualityOperations<'a when 'a : equality>() =
        member inline __.equals     = { new IOperation<'a> with member __.Exec lhs rhs = toint (HashIdentity.Structural.Equals(lhs,rhs)) }
        member inline __.equal      = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs = rhs) }
        member inline __.not_equal  = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs <> rhs) }

    type ComparisonOperations<'a when 'a : comparison>() =
        member inline __.equals             = { new IOperation<'a> with member __.Exec lhs rhs = toint (HashIdentity.Structural.Equals(lhs,rhs)) }
        member inline __.equal              = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs = rhs) }
        member inline __.not_equal          = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs <> rhs) }
        member inline __.compare            = { new IOperation<'a> with member __.Exec lhs rhs = ComparisonIdentity.Structural.Compare(lhs,rhs) }
        member inline __.less_than          = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs < rhs) }
        member inline __.less_or_equal      = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs <= rhs) }
        member inline __.greater_than       = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs > rhs) }
        member inline __.greater_or_equal   = { new IOperation<'a> with member __.Exec lhs rhs = toint (lhs >= rhs) }

    type NoninlinableEqualityOperations<'a when 'a : equality>() =
        let operations = 
            let underlyingType = typedefof<EqualityOperations<_>>.MakeGenericType [| typeof<'a> |]
            match System.Activator.CreateInstance underlyingType with
            | :? EqualityOperations<'a> as operations -> operations
            | _ -> failwith ""

        member __.equals = operations.equals
        member __.equal      = operations.equal
        member __.not_equal     = operations.not_equal

    type NoninlinableComparisonOperations<'a when 'a : comparison>() =
        let operations = 
            let underlyingType = typedefof<ComparisonOperations<_>>.MakeGenericType [| typeof<'a> |]
            match System.Activator.CreateInstance underlyingType with
            | :? ComparisonOperations<'a> as operations -> operations
            | _ -> failwith ""

        member __.equals           = operations.equals
        member __.equal            = operations.equal
        member __.not_equal        = operations.not_equal
        member __.compare          = operations.compare
        member __.less_than        = operations.less_than
        member __.less_or_equal    = operations.less_or_equal
        member __.greater_than     = operations.greater_than
        member __.greater_or_equal = operations.greater_or_equal

    type E<'a when 'a : equality>() =
        static let inlinable    = EqualityOperations<'a> ()
        static let noninlinable = NoninlinableEqualityOperations<'a> ()

        static member I = inlinable
        static member N = noninlinable

    type C<'a when 'a : comparison>() =
        static let inlinable    = ComparisonOperations<'a> ()
        static let noninlinable = NoninlinableComparisonOperations<'a> ()
        
        static member I = inlinable
        static member N = noninlinable

#if FX_ATLEAST_45
    let create<'a,'b when 'b : equality> name operation (f:IOperation<'a>) (items:array<'a>) =
        printf """ [<Test>]
 member __.``%s %s``() =
  validate (%s) %s """ name operation name operation

        make_result_set f items None
        |> Seq.iteri (fun n result ->
            if n = 0 
                then printf "[|"
                else printf ";"
            if n % 40 = 0 then printf "\n   "
            printf "%d" result)
        printfn "\n  |]\n"

    let create_inequalities name (items:array<'a>) =
        create name "C.I.equals"           C.I.equals           items
        create name "C.I.equal"            C.I.equal            items
        create name "C.I.not_equal"        C.I.not_equal        items
        create name "C.I.compare"          C.I.compare          items
        create name "C.I.less_than"        C.I.less_than        items
        create name "C.I.less_or_equal"    C.I.less_or_equal    items
        create name "C.I.greater_than"     C.I.greater_than     items
        create name "C.I.greater_or_equal" C.I.greater_or_equal items
        create name "C.N.equals"           C.N.equals           items
        create name "C.N.equal"            C.N.equal            items
        create name "C.N.not_equal"        C.N.not_equal        items
        create name "C.N.compare"          C.N.compare          items
        create name "C.N.less_than"        C.N.less_than        items
        create name "C.N.less_or_equal"    C.N.less_or_equal    items
        create name "C.N.greater_than"     C.N.greater_than     items
        create name "C.N.greater_or_equal" C.N.greater_or_equal items

    let create_equalities name (items:array<'a>) =
        create name "E.I.equals"    E.I.equals    items
        create name "E.I.equal"     E.I.equal     items
        create name "E.I.not_equal" E.I.not_equal items
        create name "E.N.equals"    E.N.equals    items
        create name "E.N.equal"     E.N.equal     items
        create name "E.N.not_equal" E.N.not_equal items

    let create_collection_inequalities name (collection:Collection<_,_,_,_>) =
        create_inequalities (name + ".Array")          collection.Array
        create_inequalities (name + ".OptionArray")    collection.OptionArray
        create_inequalities (name + ".RefArray")       collection.RefArray
        create_inequalities (name + ".RefWrapArray")   collection.RefWrapArray
        create_inequalities (name + ".UnionArray")     collection.UnionArray
        create_inequalities (name + ".UnionWrapArray") collection.UnionWrapArray
        create_inequalities (name + ".ValueArray")     collection.ValueArray
        create_inequalities (name + ".ValueWrapArray") collection.ValueWrapArray
        create_inequalities (name + ".ArrayArray")     collection.ArrayArray
        create_inequalities (name + ".ListArray")      collection.ListArray
        create_inequalities (name + ".ArrayArray |> Array.map Set.ofArray") (collection.ArrayArray |> Array.map Set.ofArray)

    let create_tuples_tests name (collection:Collection<_,_,_,_>) =
        create_inequalities (name + ".Array")          collection.Array

    let create_collection_equalities name (collection:Collection<_,_,_,_>) =
        create_equalities (name + ".Array")          collection.Array
        create_equalities (name + ".OptionArray")    collection.OptionArray
        create_equalities (name + ".RefArray")       collection.RefArray
        create_equalities (name + ".RefWrapArray")   collection.RefWrapArray
        create_equalities (name + ".UnionArray")     collection.UnionArray
        create_equalities (name + ".UnionWrapArray") collection.UnionWrapArray
        create_equalities (name + ".ValueArray")     collection.ValueArray
        create_equalities (name + ".ValueWrapArray") collection.ValueWrapArray
        create_equalities (name + ".ArrayArray")     collection.ArrayArray
        create_equalities (name + ".ListArray")      collection.ListArray

    let createData () =
        create_collection_inequalities "Bools.Collection"               Bools.Collection
        create_collection_equalities   "NullableBools.Collection"       NullableBools.Collection
        create_collection_inequalities "SBytes.Collection"              SBytes.Collection
        create_collection_equalities   "NullableSbytes.Collection"      NullableSbytes.Collection
        create_collection_inequalities "Int16s.Collection"              Int16s.Collection
        create_collection_equalities   "NullableInt16s.Collection"      NullableInt16s.Collection
        create_collection_inequalities "Int32s.Collection"              Int32s.Collection
        create_collection_equalities   "NullableInt32s.Collection"      NullableInt32s.Collection
        create_collection_inequalities "Int64s.Collection"              Int64s.Collection
        create_collection_equalities   "NullableInt64s.Collection"      NullableInt64s.Collection
        create_collection_inequalities "NativeInts.Collection"          NativeInts.Collection
        create_collection_equalities   "NullableNativeInts.Collection"  NullableNativeInts.Collection
        create_collection_inequalities "Bytes.Collection"               Bytes.Collection
        create_collection_equalities   "NullableBytes.Collection"       NullableBytes.Collection
        create_collection_inequalities "Uint16s.Collection"             Uint16s.Collection
        create_collection_equalities   "NullableUInt16s.Collection"     NullableUInt16s.Collection
        create_collection_inequalities "UInt32s.Collection"             UInt32s.Collection
        create_collection_equalities   "NullableUInt32s.Collection"     NullableUInt32s.Collection
        create_collection_inequalities "UInt64s.Collection"             UInt64s.Collection
        create_collection_equalities   "NullableUInt64s.Collection"     NullableUInt64s.Collection
        create_collection_inequalities "UNativeInts.Collection"         UNativeInts.Collection
        create_collection_equalities   "NullableUNativeInts.Collection" NullableUNativeInts.Collection
        create_collection_inequalities "Chars.Collection"               Chars.Collection
        create_collection_equalities   "NullableChars.Collection"       NullableChars.Collection
        create_collection_inequalities "Strings.Collection"             Strings.Collection
        create_collection_inequalities "Decimals.Collection"            Decimals.Collection
        create_collection_equalities   "NullableDecimals.Collection"    NullableDecimals.Collection
        create_collection_inequalities "Floats.Collection"              Floats.Collection
        create_collection_equalities   "NullableFloats.Collection"      NullableFloats.Collection
        create_collection_inequalities "Float32s.Collection"            Float32s.Collection
        create_collection_equalities   "NullableFloat32s.Collection"    NullableFloat32s.Collection
        create_collection_inequalities "DateTimes.Collection"           DateTimes.Collection
        create_collection_equalities   "NullableDateTimes.Collection"   NullableDateTimes.Collection
        create_collection_inequalities "Tuple2s.Collection"             Tuple2s.Collection
        create_tuples_tests            "Tuple3s.Collection"             Tuple3s.Collection
        create_tuples_tests            "Tuple4s.Collection"             Tuple4s.Collection
        create_tuples_tests            "Tuple5s.Collection"             Tuple5s.Collection
#endif

    let validate (items:array<'a>) (f:IOperation<'a>) (expected:array<int>) =
        try
            make_result_set f items (Some expected) |> ignore
        with
        | ValidationException(lhs=lhs; rhs=rhs; expected=expected; received=received) ->
            failwith <| sprintf "args(%O, %O) Expected=%O. Received=%O." lhs rhs expected received

open ComparersRegression

[<TestFixture>]
type GeneratedTestSuite () =
 let _ = ()
// ------------------------------------------------------------------------------
// -- The following should be generated by running CreateComparersRegression.fsx
// ------------------------------------------------------------------------------
 [<Test>]
 member __.``Bools.Collection.Array C.I.equals``() =
  validate (Bools.Collection.Array) C.I.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.I.equal``() =
  validate (Bools.Collection.Array) C.I.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.I.not_equal``() =
  validate (Bools.Collection.Array) C.I.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.I.compare``() =
  validate (Bools.Collection.Array) C.I.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.I.less_than``() =
  validate (Bools.Collection.Array) C.I.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.I.less_or_equal``() =
  validate (Bools.Collection.Array) C.I.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.I.greater_than``() =
  validate (Bools.Collection.Array) C.I.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.I.greater_or_equal``() =
  validate (Bools.Collection.Array) C.I.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.equals``() =
  validate (Bools.Collection.Array) C.N.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.equal``() =
  validate (Bools.Collection.Array) C.N.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.not_equal``() =
  validate (Bools.Collection.Array) C.N.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.compare``() =
  validate (Bools.Collection.Array) C.N.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.less_than``() =
  validate (Bools.Collection.Array) C.N.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.less_or_equal``() =
  validate (Bools.Collection.Array) C.N.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.greater_than``() =
  validate (Bools.Collection.Array) C.N.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.Array C.N.greater_or_equal``() =
  validate (Bools.Collection.Array) C.N.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.equals``() =
  validate (Bools.Collection.OptionArray) C.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.equal``() =
  validate (Bools.Collection.OptionArray) C.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.not_equal``() =
  validate (Bools.Collection.OptionArray) C.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.compare``() =
  validate (Bools.Collection.OptionArray) C.I.compare [|
   0;-1;-1;1;0;1;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.less_than``() =
  validate (Bools.Collection.OptionArray) C.I.less_than [|
   0;1;1;0;0;0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.less_or_equal``() =
  validate (Bools.Collection.OptionArray) C.I.less_or_equal [|
   1;1;1;0;1;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.greater_than``() =
  validate (Bools.Collection.OptionArray) C.I.greater_than [|
   0;0;0;1;0;1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.I.greater_or_equal``() =
  validate (Bools.Collection.OptionArray) C.I.greater_or_equal [|
   1;0;0;1;1;1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.equals``() =
  validate (Bools.Collection.OptionArray) C.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.equal``() =
  validate (Bools.Collection.OptionArray) C.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.not_equal``() =
  validate (Bools.Collection.OptionArray) C.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.compare``() =
  validate (Bools.Collection.OptionArray) C.N.compare [|
   0;-1;-1;1;0;1;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.less_than``() =
  validate (Bools.Collection.OptionArray) C.N.less_than [|
   0;1;1;0;0;0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.less_or_equal``() =
  validate (Bools.Collection.OptionArray) C.N.less_or_equal [|
   1;1;1;0;1;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.greater_than``() =
  validate (Bools.Collection.OptionArray) C.N.greater_than [|
   0;0;0;1;0;1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.OptionArray C.N.greater_or_equal``() =
  validate (Bools.Collection.OptionArray) C.N.greater_or_equal [|
   1;0;0;1;1;1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.equals``() =
  validate (Bools.Collection.RefArray) C.I.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.equal``() =
  validate (Bools.Collection.RefArray) C.I.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.not_equal``() =
  validate (Bools.Collection.RefArray) C.I.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.compare``() =
  validate (Bools.Collection.RefArray) C.I.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.less_than``() =
  validate (Bools.Collection.RefArray) C.I.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.less_or_equal``() =
  validate (Bools.Collection.RefArray) C.I.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.greater_than``() =
  validate (Bools.Collection.RefArray) C.I.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.I.greater_or_equal``() =
  validate (Bools.Collection.RefArray) C.I.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.equals``() =
  validate (Bools.Collection.RefArray) C.N.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.equal``() =
  validate (Bools.Collection.RefArray) C.N.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.not_equal``() =
  validate (Bools.Collection.RefArray) C.N.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.compare``() =
  validate (Bools.Collection.RefArray) C.N.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.less_than``() =
  validate (Bools.Collection.RefArray) C.N.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.less_or_equal``() =
  validate (Bools.Collection.RefArray) C.N.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.greater_than``() =
  validate (Bools.Collection.RefArray) C.N.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefArray C.N.greater_or_equal``() =
  validate (Bools.Collection.RefArray) C.N.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.equals``() =
  validate (Bools.Collection.RefWrapArray) C.I.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.equal``() =
  validate (Bools.Collection.RefWrapArray) C.I.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.not_equal``() =
  validate (Bools.Collection.RefWrapArray) C.I.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.compare``() =
  validate (Bools.Collection.RefWrapArray) C.I.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.less_than``() =
  validate (Bools.Collection.RefWrapArray) C.I.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.less_or_equal``() =
  validate (Bools.Collection.RefWrapArray) C.I.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.greater_than``() =
  validate (Bools.Collection.RefWrapArray) C.I.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.I.greater_or_equal``() =
  validate (Bools.Collection.RefWrapArray) C.I.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.equals``() =
  validate (Bools.Collection.RefWrapArray) C.N.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.equal``() =
  validate (Bools.Collection.RefWrapArray) C.N.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.not_equal``() =
  validate (Bools.Collection.RefWrapArray) C.N.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.compare``() =
  validate (Bools.Collection.RefWrapArray) C.N.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.less_than``() =
  validate (Bools.Collection.RefWrapArray) C.N.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.less_or_equal``() =
  validate (Bools.Collection.RefWrapArray) C.N.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.greater_than``() =
  validate (Bools.Collection.RefWrapArray) C.N.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.RefWrapArray C.N.greater_or_equal``() =
  validate (Bools.Collection.RefWrapArray) C.N.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.equals``() =
  validate (Bools.Collection.UnionArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.equal``() =
  validate (Bools.Collection.UnionArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.not_equal``() =
  validate (Bools.Collection.UnionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.compare``() =
  validate (Bools.Collection.UnionArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;
   -1;-1;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;
   2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;
   0;-1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-1;
   -2;3;2;1;1;0;-1;-2;3;2;1;2;1;-1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.less_than``() =
  validate (Bools.Collection.UnionArray) C.I.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.less_or_equal``() =
  validate (Bools.Collection.UnionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.greater_than``() =
  validate (Bools.Collection.UnionArray) C.I.greater_than [|
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.I.greater_or_equal``() =
  validate (Bools.Collection.UnionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.equals``() =
  validate (Bools.Collection.UnionArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.equal``() =
  validate (Bools.Collection.UnionArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.not_equal``() =
  validate (Bools.Collection.UnionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.compare``() =
  validate (Bools.Collection.UnionArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;
   -1;-1;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;
   2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;
   0;-1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-1;
   -2;3;2;1;1;0;-1;-2;3;2;1;2;1;-1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.less_than``() =
  validate (Bools.Collection.UnionArray) C.N.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.less_or_equal``() =
  validate (Bools.Collection.UnionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.greater_than``() =
  validate (Bools.Collection.UnionArray) C.N.greater_than [|
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionArray C.N.greater_or_equal``() =
  validate (Bools.Collection.UnionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.equals``() =
  validate (Bools.Collection.UnionWrapArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.equal``() =
  validate (Bools.Collection.UnionWrapArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.not_equal``() =
  validate (Bools.Collection.UnionWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.compare``() =
  validate (Bools.Collection.UnionWrapArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;
   -1;-1;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;
   2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;
   0;-1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-1;
   -2;3;2;1;1;0;-1;-2;3;2;1;2;1;-1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.less_than``() =
  validate (Bools.Collection.UnionWrapArray) C.I.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.less_or_equal``() =
  validate (Bools.Collection.UnionWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.greater_than``() =
  validate (Bools.Collection.UnionWrapArray) C.I.greater_than [|
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.I.greater_or_equal``() =
  validate (Bools.Collection.UnionWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.equals``() =
  validate (Bools.Collection.UnionWrapArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.equal``() =
  validate (Bools.Collection.UnionWrapArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.not_equal``() =
  validate (Bools.Collection.UnionWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.compare``() =
  validate (Bools.Collection.UnionWrapArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;
   -1;-1;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;
   2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;
   0;-1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;-1;-1;
   -2;3;2;1;1;0;-1;-2;3;2;1;2;1;-1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.less_than``() =
  validate (Bools.Collection.UnionWrapArray) C.N.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.less_or_equal``() =
  validate (Bools.Collection.UnionWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.greater_than``() =
  validate (Bools.Collection.UnionWrapArray) C.N.greater_than [|
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.UnionWrapArray C.N.greater_or_equal``() =
  validate (Bools.Collection.UnionWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.equals``() =
  validate (Bools.Collection.ValueArray) C.I.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.equal``() =
  validate (Bools.Collection.ValueArray) C.I.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.not_equal``() =
  validate (Bools.Collection.ValueArray) C.I.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.compare``() =
  validate (Bools.Collection.ValueArray) C.I.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.less_than``() =
  validate (Bools.Collection.ValueArray) C.I.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.less_or_equal``() =
  validate (Bools.Collection.ValueArray) C.I.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.greater_than``() =
  validate (Bools.Collection.ValueArray) C.I.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.I.greater_or_equal``() =
  validate (Bools.Collection.ValueArray) C.I.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.equals``() =
  validate (Bools.Collection.ValueArray) C.N.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.equal``() =
  validate (Bools.Collection.ValueArray) C.N.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.not_equal``() =
  validate (Bools.Collection.ValueArray) C.N.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.compare``() =
  validate (Bools.Collection.ValueArray) C.N.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.less_than``() =
  validate (Bools.Collection.ValueArray) C.N.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.less_or_equal``() =
  validate (Bools.Collection.ValueArray) C.N.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.greater_than``() =
  validate (Bools.Collection.ValueArray) C.N.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueArray C.N.greater_or_equal``() =
  validate (Bools.Collection.ValueArray) C.N.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.equals``() =
  validate (Bools.Collection.ValueWrapArray) C.I.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.equal``() =
  validate (Bools.Collection.ValueWrapArray) C.I.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.not_equal``() =
  validate (Bools.Collection.ValueWrapArray) C.I.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.compare``() =
  validate (Bools.Collection.ValueWrapArray) C.I.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.less_than``() =
  validate (Bools.Collection.ValueWrapArray) C.I.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.less_or_equal``() =
  validate (Bools.Collection.ValueWrapArray) C.I.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.greater_than``() =
  validate (Bools.Collection.ValueWrapArray) C.I.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.I.greater_or_equal``() =
  validate (Bools.Collection.ValueWrapArray) C.I.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.equals``() =
  validate (Bools.Collection.ValueWrapArray) C.N.equals [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.equal``() =
  validate (Bools.Collection.ValueWrapArray) C.N.equal [|
   1;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.not_equal``() =
  validate (Bools.Collection.ValueWrapArray) C.N.not_equal [|
   0;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.compare``() =
  validate (Bools.Collection.ValueWrapArray) C.N.compare [|
   0;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.less_than``() =
  validate (Bools.Collection.ValueWrapArray) C.N.less_than [|
   0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.less_or_equal``() =
  validate (Bools.Collection.ValueWrapArray) C.N.less_or_equal [|
   1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.greater_than``() =
  validate (Bools.Collection.ValueWrapArray) C.N.greater_than [|
   0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ValueWrapArray C.N.greater_or_equal``() =
  validate (Bools.Collection.ValueWrapArray) C.N.greater_or_equal [|
   1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.equals``() =
  validate (Bools.Collection.ArrayArray) C.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.equal``() =
  validate (Bools.Collection.ArrayArray) C.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.not_equal``() =
  validate (Bools.Collection.ArrayArray) C.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.compare``() =
  validate (Bools.Collection.ArrayArray) C.I.compare [|
   0;1;-1;-1;-1;0;-1;-1;1;1;0;1;1;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.less_than``() =
  validate (Bools.Collection.ArrayArray) C.I.less_than [|
   0;0;1;1;1;0;1;1;0;0;0;0;0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.less_or_equal``() =
  validate (Bools.Collection.ArrayArray) C.I.less_or_equal [|
   1;0;1;1;1;1;1;1;0;0;1;0;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.greater_than``() =
  validate (Bools.Collection.ArrayArray) C.I.greater_than [|
   0;1;0;0;0;0;0;0;1;1;0;1;1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.I.greater_or_equal``() =
  validate (Bools.Collection.ArrayArray) C.I.greater_or_equal [|
   1;1;0;0;0;1;0;0;1;1;1;1;1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.equals``() =
  validate (Bools.Collection.ArrayArray) C.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.equal``() =
  validate (Bools.Collection.ArrayArray) C.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.not_equal``() =
  validate (Bools.Collection.ArrayArray) C.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.compare``() =
  validate (Bools.Collection.ArrayArray) C.N.compare [|
   0;1;-1;-1;-1;0;-1;-1;1;1;0;1;1;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.less_than``() =
  validate (Bools.Collection.ArrayArray) C.N.less_than [|
   0;0;1;1;1;0;1;1;0;0;0;0;0;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.less_or_equal``() =
  validate (Bools.Collection.ArrayArray) C.N.less_or_equal [|
   1;0;1;1;1;1;1;1;0;0;1;0;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.greater_than``() =
  validate (Bools.Collection.ArrayArray) C.N.greater_than [|
   0;1;0;0;0;0;0;0;1;1;0;1;1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray C.N.greater_or_equal``() =
  validate (Bools.Collection.ArrayArray) C.N.greater_or_equal [|
   1;1;0;0;0;1;0;0;1;1;1;1;1;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.equals``() =
  validate (Bools.Collection.ListArray) C.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.equal``() =
  validate (Bools.Collection.ListArray) C.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.not_equal``() =
  validate (Bools.Collection.ListArray) C.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.compare``() =
  validate (Bools.Collection.ListArray) C.I.compare [|
   0;1;-1;1;-1;0;-1;-1;1;1;0;1;-1;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.less_than``() =
  validate (Bools.Collection.ListArray) C.I.less_than [|
   0;0;1;0;1;0;1;1;0;0;0;0;1;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.less_or_equal``() =
  validate (Bools.Collection.ListArray) C.I.less_or_equal [|
   1;0;1;0;1;1;1;1;0;0;1;0;1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.greater_than``() =
  validate (Bools.Collection.ListArray) C.I.greater_than [|
   0;1;0;1;0;0;0;0;1;1;0;1;0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.I.greater_or_equal``() =
  validate (Bools.Collection.ListArray) C.I.greater_or_equal [|
   1;1;0;1;0;1;0;0;1;1;1;1;0;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.equals``() =
  validate (Bools.Collection.ListArray) C.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.equal``() =
  validate (Bools.Collection.ListArray) C.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.not_equal``() =
  validate (Bools.Collection.ListArray) C.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.compare``() =
  validate (Bools.Collection.ListArray) C.N.compare [|
   0;1;-1;1;-1;0;-1;-1;1;1;0;1;-1;1;-1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.less_than``() =
  validate (Bools.Collection.ListArray) C.N.less_than [|
   0;0;1;0;1;0;1;1;0;0;0;0;1;0;1;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.less_or_equal``() =
  validate (Bools.Collection.ListArray) C.N.less_or_equal [|
   1;0;1;0;1;1;1;1;0;0;1;0;1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.greater_than``() =
  validate (Bools.Collection.ListArray) C.N.greater_than [|
   0;1;0;1;0;0;0;0;1;1;0;1;0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ListArray C.N.greater_or_equal``() =
  validate (Bools.Collection.ListArray) C.N.greater_or_equal [|
   1;1;0;1;0;1;0;0;1;1;1;1;0;1;0;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.equals``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;1;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;1;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.not_equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;0;1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.compare``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.compare [|
   0;1;1;1;-1;0;-1;-1;-1;1;0;0;-1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_than``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_than [|
   0;0;0;0;1;0;1;1;1;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_or_equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_or_equal [|
   1;0;0;0;1;1;1;1;1;0;1;1;1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_than``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_than [|
   0;1;1;1;0;0;0;0;0;1;0;0;0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_or_equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_or_equal [|
   1;1;1;1;0;1;0;0;0;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.equals``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;1;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;1;0;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.not_equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;0;1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.compare``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.compare [|
   0;1;1;1;-1;0;-1;-1;-1;1;0;0;-1;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_than``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_than [|
   0;0;0;0;1;0;1;1;1;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_or_equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_or_equal [|
   1;0;0;0;1;1;1;1;1;0;1;1;1;0;1;1
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_than``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_than [|
   0;1;1;1;0;0;0;0;0;1;0;0;0;1;0;0
  |]

 [<Test>]
 member __.``Bools.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_or_equal``() =
  validate (Bools.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_or_equal [|
   1;1;1;1;0;1;0;0;0;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.Array E.I.equals``() =
  validate (NullableBools.Collection.Array) E.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.Array E.I.equal``() =
  validate (NullableBools.Collection.Array) E.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.Array E.I.not_equal``() =
  validate (NullableBools.Collection.Array) E.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.Array E.N.equals``() =
  validate (NullableBools.Collection.Array) E.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.Array E.N.equal``() =
  validate (NullableBools.Collection.Array) E.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.Array E.N.not_equal``() =
  validate (NullableBools.Collection.Array) E.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.OptionArray E.I.equals``() =
  validate (NullableBools.Collection.OptionArray) E.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.OptionArray E.I.equal``() =
  validate (NullableBools.Collection.OptionArray) E.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.OptionArray E.I.not_equal``() =
  validate (NullableBools.Collection.OptionArray) E.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.OptionArray E.N.equals``() =
  validate (NullableBools.Collection.OptionArray) E.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.OptionArray E.N.equal``() =
  validate (NullableBools.Collection.OptionArray) E.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.OptionArray E.N.not_equal``() =
  validate (NullableBools.Collection.OptionArray) E.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefArray E.I.equals``() =
  validate (NullableBools.Collection.RefArray) E.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefArray E.I.equal``() =
  validate (NullableBools.Collection.RefArray) E.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefArray E.I.not_equal``() =
  validate (NullableBools.Collection.RefArray) E.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefArray E.N.equals``() =
  validate (NullableBools.Collection.RefArray) E.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefArray E.N.equal``() =
  validate (NullableBools.Collection.RefArray) E.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefArray E.N.not_equal``() =
  validate (NullableBools.Collection.RefArray) E.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefWrapArray E.I.equals``() =
  validate (NullableBools.Collection.RefWrapArray) E.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefWrapArray E.I.equal``() =
  validate (NullableBools.Collection.RefWrapArray) E.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefWrapArray E.I.not_equal``() =
  validate (NullableBools.Collection.RefWrapArray) E.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefWrapArray E.N.equals``() =
  validate (NullableBools.Collection.RefWrapArray) E.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefWrapArray E.N.equal``() =
  validate (NullableBools.Collection.RefWrapArray) E.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.RefWrapArray E.N.not_equal``() =
  validate (NullableBools.Collection.RefWrapArray) E.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionArray E.I.equals``() =
  validate (NullableBools.Collection.UnionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionArray E.I.equal``() =
  validate (NullableBools.Collection.UnionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionArray E.I.not_equal``() =
  validate (NullableBools.Collection.UnionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionArray E.N.equals``() =
  validate (NullableBools.Collection.UnionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionArray E.N.equal``() =
  validate (NullableBools.Collection.UnionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionArray E.N.not_equal``() =
  validate (NullableBools.Collection.UnionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionWrapArray E.I.equals``() =
  validate (NullableBools.Collection.UnionWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionWrapArray E.I.equal``() =
  validate (NullableBools.Collection.UnionWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionWrapArray E.I.not_equal``() =
  validate (NullableBools.Collection.UnionWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionWrapArray E.N.equals``() =
  validate (NullableBools.Collection.UnionWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionWrapArray E.N.equal``() =
  validate (NullableBools.Collection.UnionWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NullableBools.Collection.UnionWrapArray E.N.not_equal``() =
  validate (NullableBools.Collection.UnionWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueArray E.I.equals``() =
  validate (NullableBools.Collection.ValueArray) E.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueArray E.I.equal``() =
  validate (NullableBools.Collection.ValueArray) E.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueArray E.I.not_equal``() =
  validate (NullableBools.Collection.ValueArray) E.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueArray E.N.equals``() =
  validate (NullableBools.Collection.ValueArray) E.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueArray E.N.equal``() =
  validate (NullableBools.Collection.ValueArray) E.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueArray E.N.not_equal``() =
  validate (NullableBools.Collection.ValueArray) E.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueWrapArray E.I.equals``() =
  validate (NullableBools.Collection.ValueWrapArray) E.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueWrapArray E.I.equal``() =
  validate (NullableBools.Collection.ValueWrapArray) E.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueWrapArray E.I.not_equal``() =
  validate (NullableBools.Collection.ValueWrapArray) E.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueWrapArray E.N.equals``() =
  validate (NullableBools.Collection.ValueWrapArray) E.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueWrapArray E.N.equal``() =
  validate (NullableBools.Collection.ValueWrapArray) E.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ValueWrapArray E.N.not_equal``() =
  validate (NullableBools.Collection.ValueWrapArray) E.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ArrayArray E.I.equals``() =
  validate (NullableBools.Collection.ArrayArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ArrayArray E.I.equal``() =
  validate (NullableBools.Collection.ArrayArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ArrayArray E.I.not_equal``() =
  validate (NullableBools.Collection.ArrayArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ArrayArray E.N.equals``() =
  validate (NullableBools.Collection.ArrayArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ArrayArray E.N.equal``() =
  validate (NullableBools.Collection.ArrayArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ArrayArray E.N.not_equal``() =
  validate (NullableBools.Collection.ArrayArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ListArray E.I.equals``() =
  validate (NullableBools.Collection.ListArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ListArray E.I.equal``() =
  validate (NullableBools.Collection.ListArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ListArray E.I.not_equal``() =
  validate (NullableBools.Collection.ListArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableBools.Collection.ListArray E.N.equals``() =
  validate (NullableBools.Collection.ListArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ListArray E.N.equal``() =
  validate (NullableBools.Collection.ListArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableBools.Collection.ListArray E.N.not_equal``() =
  validate (NullableBools.Collection.ListArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.equals``() =
  validate (SBytes.Collection.Array) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.equal``() =
  validate (SBytes.Collection.Array) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.not_equal``() =
  validate (SBytes.Collection.Array) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.compare``() =
  validate (SBytes.Collection.Array) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.less_than``() =
  validate (SBytes.Collection.Array) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.less_or_equal``() =
  validate (SBytes.Collection.Array) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.greater_than``() =
  validate (SBytes.Collection.Array) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.I.greater_or_equal``() =
  validate (SBytes.Collection.Array) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.equals``() =
  validate (SBytes.Collection.Array) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.equal``() =
  validate (SBytes.Collection.Array) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.not_equal``() =
  validate (SBytes.Collection.Array) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.compare``() =
  validate (SBytes.Collection.Array) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.less_than``() =
  validate (SBytes.Collection.Array) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.less_or_equal``() =
  validate (SBytes.Collection.Array) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.greater_than``() =
  validate (SBytes.Collection.Array) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.Array C.N.greater_or_equal``() =
  validate (SBytes.Collection.Array) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.equals``() =
  validate (SBytes.Collection.OptionArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.equal``() =
  validate (SBytes.Collection.OptionArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.not_equal``() =
  validate (SBytes.Collection.OptionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.compare``() =
  validate (SBytes.Collection.OptionArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;-255;-127;-128;-129;1;255;0;128;127;126;1;127;-128;0;-1;-2;1;128;-127;1;0;-1;1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.less_than``() =
  validate (SBytes.Collection.OptionArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.less_or_equal``() =
  validate (SBytes.Collection.OptionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;0;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.greater_than``() =
  validate (SBytes.Collection.OptionArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;1;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.OptionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.equals``() =
  validate (SBytes.Collection.OptionArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.equal``() =
  validate (SBytes.Collection.OptionArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.not_equal``() =
  validate (SBytes.Collection.OptionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.compare``() =
  validate (SBytes.Collection.OptionArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;-255;-127;-128;-129;1;255;0;128;127;126;1;127;-128;0;-1;-2;1;128;-127;1;0;-1;1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.less_than``() =
  validate (SBytes.Collection.OptionArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.less_or_equal``() =
  validate (SBytes.Collection.OptionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;0;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.greater_than``() =
  validate (SBytes.Collection.OptionArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;1;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.OptionArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.OptionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.equals``() =
  validate (SBytes.Collection.RefArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.equal``() =
  validate (SBytes.Collection.RefArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.not_equal``() =
  validate (SBytes.Collection.RefArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.compare``() =
  validate (SBytes.Collection.RefArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.less_than``() =
  validate (SBytes.Collection.RefArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.less_or_equal``() =
  validate (SBytes.Collection.RefArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.greater_than``() =
  validate (SBytes.Collection.RefArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.RefArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.equals``() =
  validate (SBytes.Collection.RefArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.equal``() =
  validate (SBytes.Collection.RefArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.not_equal``() =
  validate (SBytes.Collection.RefArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.compare``() =
  validate (SBytes.Collection.RefArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.less_than``() =
  validate (SBytes.Collection.RefArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.less_or_equal``() =
  validate (SBytes.Collection.RefArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.greater_than``() =
  validate (SBytes.Collection.RefArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.RefArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.equals``() =
  validate (SBytes.Collection.RefWrapArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.equal``() =
  validate (SBytes.Collection.RefWrapArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.not_equal``() =
  validate (SBytes.Collection.RefWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.compare``() =
  validate (SBytes.Collection.RefWrapArray) C.I.compare [|
   0;-255;-127;-128;-129;255;0;128;127;126;127;-128;0;-1;-2;128;-127;1;0;-1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.less_than``() =
  validate (SBytes.Collection.RefWrapArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.less_or_equal``() =
  validate (SBytes.Collection.RefWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.greater_than``() =
  validate (SBytes.Collection.RefWrapArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.RefWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.equals``() =
  validate (SBytes.Collection.RefWrapArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.equal``() =
  validate (SBytes.Collection.RefWrapArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.not_equal``() =
  validate (SBytes.Collection.RefWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.compare``() =
  validate (SBytes.Collection.RefWrapArray) C.N.compare [|
   0;-255;-127;-128;-129;255;0;128;127;126;127;-128;0;-1;-2;128;-127;1;0;-1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.less_than``() =
  validate (SBytes.Collection.RefWrapArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.less_or_equal``() =
  validate (SBytes.Collection.RefWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.greater_than``() =
  validate (SBytes.Collection.RefWrapArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.RefWrapArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.RefWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.equals``() =
  validate (SBytes.Collection.UnionArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.equal``() =
  validate (SBytes.Collection.UnionArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.not_equal``() =
  validate (SBytes.Collection.UnionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.compare``() =
  validate (SBytes.Collection.UnionArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.less_than``() =
  validate (SBytes.Collection.UnionArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.less_or_equal``() =
  validate (SBytes.Collection.UnionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.greater_than``() =
  validate (SBytes.Collection.UnionArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.UnionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.equals``() =
  validate (SBytes.Collection.UnionArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.equal``() =
  validate (SBytes.Collection.UnionArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.not_equal``() =
  validate (SBytes.Collection.UnionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.compare``() =
  validate (SBytes.Collection.UnionArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.less_than``() =
  validate (SBytes.Collection.UnionArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.less_or_equal``() =
  validate (SBytes.Collection.UnionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.greater_than``() =
  validate (SBytes.Collection.UnionArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.UnionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.equals``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.not_equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.compare``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-255;-1;-2;-3;-3;-3;-3;-127;-1;-2;-3;-3;-3;-3;-128;-1;-2;-3;-3;-3;-3;-129;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-255;-1;-2;-2;-2;-2;1;-127;-1;-2;-2;-2;-2;1;-128;-1;-2;-2;-2;-2;1;-129;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-255;
   -1;-1;-1;-1;2;1;-127;-1;-1;-1;-1;2;1;-128;-1;-1;-1;-1;2;1;-129;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-255;-1;-2;-3;3;
   2;1;-127;-1;-2;-3;3;2;1;-128;-1;-2;-3;3;2;1;-129;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-255;-1;-2;3;2;1;1;-127;-1;
   -2;3;2;1;1;-128;-1;-2;3;2;1;1;-129;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-255;-1;3;2;1;2;1;-127;-1;3;2;1;2;
   1;-128;-1;3;2;1;2;1;-129;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-255;3;2;1;3;2;1;-127;3;2;1;3;2;1;-128;3;2;
   1;3;2;1;-129;255;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;128;-1;-2;-3;-3;-3;-3;127;-1;-2;-3;-3;-3;-3;126;-1;-2;-3;-3;-3;-3;
   1;255;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;128;-1;-2;-2;-2;-2;1;127;-1;-2;-2;-2;-2;1;126;-1;-2;-2;-2;-2;2;1;255;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;128;-1;-1;-1;-1;2;1;127;-1;-1;-1;-1;2;1;126;-1;-1;-1;-1;3;2;1;255;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;128;-1;-2;-3;3;2;1;127;-1;-2;-3;3;2;1;126;-1;-2;-3;3;2;1;1;255;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;128;-1;-2;3;2;1;1;127;-1;-2;3;2;1;1;126;-1;-2;3;2;1;2;1;255;-1;3;2;1;2;1;0;-1;3;2;1;2;1;128;
   -1;3;2;1;2;1;127;-1;3;2;1;2;1;126;-1;3;2;1;3;2;1;255;3;2;1;3;2;1;0;3;2;1;3;2;1;128;3;2;1;3;
   2;1;127;3;2;1;3;2;1;126;127;-1;-2;-3;-3;-3;-3;-128;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-2;-1;
   -2;-3;-3;-3;-3;1;127;-1;-2;-2;-2;-2;1;-128;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-2;-1;-2;-2;-2;-2;
   2;1;127;-1;-1;-1;-1;2;1;-128;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-2;-1;-1;-1;-1;3;2;1;127;-1;
   -2;-3;3;2;1;-128;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-2;-1;-2;-3;3;2;1;1;127;-1;-2;3;2;1;
   1;-128;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-2;-1;-2;3;2;1;2;1;127;-1;3;2;1;2;1;-128;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-2;-1;3;2;1;3;2;1;127;3;2;1;3;2;1;-128;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-2;128;-1;-2;-3;-3;-3;-3;-127;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;128;-1;-2;-2;-2;-2;1;-127;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;128;-1;-1;-1;-1;2;1;-127;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;128;-1;-2;-3;3;2;1;-127;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;128;
   -1;-2;3;2;1;1;-127;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;128;-1;3;2;1;
   2;1;-127;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;128;3;2;1;3;2;1;-127;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;129;-1;-2;-3;-3;-3;-3;-126;-1;-2;-3;-3;-3;-3;2;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;129;-1;-2;-2;-2;-2;1;-126;-1;-2;-2;-2;-2;1;2;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;129;-1;-1;-1;-1;2;1;-126;-1;-1;-1;-1;2;1;2;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;129;-1;-2;-3;3;2;1;-126;-1;-2;-3;3;2;1;2;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;129;-1;-2;3;2;1;1;-126;-1;-2;3;2;1;1;2;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   129;-1;3;2;1;2;1;-126;-1;3;2;1;2;1;2;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;129;3;2;1;
   3;2;1;-126;3;2;1;3;2;1;2;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.less_than``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.less_or_equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.greater_than``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.equals``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.not_equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.compare``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-255;-1;-2;-3;-3;-3;-3;-127;-1;-2;-3;-3;-3;-3;-128;-1;-2;-3;-3;-3;-3;-129;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-255;-1;-2;-2;-2;-2;1;-127;-1;-2;-2;-2;-2;1;-128;-1;-2;-2;-2;-2;1;-129;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-255;
   -1;-1;-1;-1;2;1;-127;-1;-1;-1;-1;2;1;-128;-1;-1;-1;-1;2;1;-129;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-255;-1;-2;-3;3;
   2;1;-127;-1;-2;-3;3;2;1;-128;-1;-2;-3;3;2;1;-129;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-255;-1;-2;3;2;1;1;-127;-1;
   -2;3;2;1;1;-128;-1;-2;3;2;1;1;-129;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-255;-1;3;2;1;2;1;-127;-1;3;2;1;2;
   1;-128;-1;3;2;1;2;1;-129;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-255;3;2;1;3;2;1;-127;3;2;1;3;2;1;-128;3;2;
   1;3;2;1;-129;255;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;128;-1;-2;-3;-3;-3;-3;127;-1;-2;-3;-3;-3;-3;126;-1;-2;-3;-3;-3;-3;
   1;255;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;128;-1;-2;-2;-2;-2;1;127;-1;-2;-2;-2;-2;1;126;-1;-2;-2;-2;-2;2;1;255;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;128;-1;-1;-1;-1;2;1;127;-1;-1;-1;-1;2;1;126;-1;-1;-1;-1;3;2;1;255;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;128;-1;-2;-3;3;2;1;127;-1;-2;-3;3;2;1;126;-1;-2;-3;3;2;1;1;255;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;128;-1;-2;3;2;1;1;127;-1;-2;3;2;1;1;126;-1;-2;3;2;1;2;1;255;-1;3;2;1;2;1;0;-1;3;2;1;2;1;128;
   -1;3;2;1;2;1;127;-1;3;2;1;2;1;126;-1;3;2;1;3;2;1;255;3;2;1;3;2;1;0;3;2;1;3;2;1;128;3;2;1;3;
   2;1;127;3;2;1;3;2;1;126;127;-1;-2;-3;-3;-3;-3;-128;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-2;-1;
   -2;-3;-3;-3;-3;1;127;-1;-2;-2;-2;-2;1;-128;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-2;-1;-2;-2;-2;-2;
   2;1;127;-1;-1;-1;-1;2;1;-128;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-2;-1;-1;-1;-1;3;2;1;127;-1;
   -2;-3;3;2;1;-128;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-2;-1;-2;-3;3;2;1;1;127;-1;-2;3;2;1;
   1;-128;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-2;-1;-2;3;2;1;2;1;127;-1;3;2;1;2;1;-128;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-2;-1;3;2;1;3;2;1;127;3;2;1;3;2;1;-128;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-2;128;-1;-2;-3;-3;-3;-3;-127;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;128;-1;-2;-2;-2;-2;1;-127;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;128;-1;-1;-1;-1;2;1;-127;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;128;-1;-2;-3;3;2;1;-127;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;128;
   -1;-2;3;2;1;1;-127;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;128;-1;3;2;1;
   2;1;-127;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;128;3;2;1;3;2;1;-127;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;129;-1;-2;-3;-3;-3;-3;-126;-1;-2;-3;-3;-3;-3;2;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;129;-1;-2;-2;-2;-2;1;-126;-1;-2;-2;-2;-2;1;2;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;129;-1;-1;-1;-1;2;1;-126;-1;-1;-1;-1;2;1;2;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;129;-1;-2;-3;3;2;1;-126;-1;-2;-3;3;2;1;2;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;129;-1;-2;3;2;1;1;-126;-1;-2;3;2;1;1;2;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   129;-1;3;2;1;2;1;-126;-1;3;2;1;2;1;2;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;129;3;2;1;
   3;2;1;-126;3;2;1;3;2;1;2;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.less_than``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.less_or_equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.greater_than``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.UnionWrapArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.UnionWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.equals``() =
  validate (SBytes.Collection.ValueArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.equal``() =
  validate (SBytes.Collection.ValueArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.not_equal``() =
  validate (SBytes.Collection.ValueArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.compare``() =
  validate (SBytes.Collection.ValueArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.less_than``() =
  validate (SBytes.Collection.ValueArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.less_or_equal``() =
  validate (SBytes.Collection.ValueArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.greater_than``() =
  validate (SBytes.Collection.ValueArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.ValueArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.equals``() =
  validate (SBytes.Collection.ValueArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.equal``() =
  validate (SBytes.Collection.ValueArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.not_equal``() =
  validate (SBytes.Collection.ValueArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.compare``() =
  validate (SBytes.Collection.ValueArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.less_than``() =
  validate (SBytes.Collection.ValueArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.less_or_equal``() =
  validate (SBytes.Collection.ValueArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.greater_than``() =
  validate (SBytes.Collection.ValueArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.ValueArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.equals``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.not_equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.compare``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.compare [|
   0;-255;-127;-128;-129;255;0;128;127;126;127;-128;0;-1;-2;128;-127;1;0;-1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.less_than``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.less_or_equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.greater_than``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.equals``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.not_equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.compare``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.compare [|
   0;-255;-127;-128;-129;255;0;128;127;126;127;-128;0;-1;-2;128;-127;1;0;-1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.less_than``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.less_or_equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.greater_than``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ValueWrapArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.ValueWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.equals``() =
  validate (SBytes.Collection.ArrayArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.equal``() =
  validate (SBytes.Collection.ArrayArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.not_equal``() =
  validate (SBytes.Collection.ArrayArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.compare``() =
  validate (SBytes.Collection.ArrayArray) C.I.compare [|
   0;1;-1;1;1;-1;-1;-1;-1;-1;-1;0;-1;1;1;-1;-1;-1;-1;-1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0;-1;-1;-1;-1;-1;-1;
   -1;-1;-1;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;1;-1;1;1;1;1;1;1;1;-1;0;-1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;-1;-1;-1;0;-1;1;1;1;1;1;-1;-1;-1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.less_than``() =
  validate (SBytes.Collection.ArrayArray) C.I.less_than [|
   0;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;1;0;1;0;0;0;0;0;1;1;1;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.less_or_equal``() =
  validate (SBytes.Collection.ArrayArray) C.I.less_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;1;1;1;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.greater_than``() =
  validate (SBytes.Collection.ArrayArray) C.I.greater_than [|
   0;1;0;1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;0;0;0;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.ArrayArray) C.I.greater_or_equal [|
   1;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;0;1;0;1;1;1;1;1;0;0;0;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.equals``() =
  validate (SBytes.Collection.ArrayArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.equal``() =
  validate (SBytes.Collection.ArrayArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.not_equal``() =
  validate (SBytes.Collection.ArrayArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.compare``() =
  validate (SBytes.Collection.ArrayArray) C.N.compare [|
   0;1;-1;1;1;-1;-1;-1;-1;-1;-1;0;-1;1;1;-1;-1;-1;-1;-1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0;-1;-1;-1;-1;-1;-1;
   -1;-1;-1;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;1;-1;1;1;1;1;1;1;1;-1;0;-1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;-1;-1;-1;0;-1;1;1;1;1;1;-1;-1;-1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.less_than``() =
  validate (SBytes.Collection.ArrayArray) C.N.less_than [|
   0;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;1;0;1;0;0;0;0;0;1;1;1;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.less_or_equal``() =
  validate (SBytes.Collection.ArrayArray) C.N.less_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;1;1;1;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.greater_than``() =
  validate (SBytes.Collection.ArrayArray) C.N.greater_than [|
   0;1;0;1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;0;0;0;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.ArrayArray) C.N.greater_or_equal [|
   1;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;0;1;0;1;1;1;1;1;0;0;0;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.equals``() =
  validate (SBytes.Collection.ListArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.equal``() =
  validate (SBytes.Collection.ListArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.not_equal``() =
  validate (SBytes.Collection.ListArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.compare``() =
  validate (SBytes.Collection.ListArray) C.I.compare [|
   0;-255;-127;-128;-129;-1;-255;-127;-128;-129;255;0;128;127;126;255;-1;128;127;126;127;-128;0;-1;-2;127;-128;-1;-1;-2;128;-127;1;0;-1;128;-127;1;-1;-1;
   129;-126;2;1;0;129;-126;2;1;-1;1;-255;-127;-128;-129;0;-255;-127;-128;-129;255;1;128;127;126;255;0;128;127;126;127;-128;1;-1;-2;127;-128;0;-1;-2;
   128;-127;1;1;-1;128;-127;1;0;-1;129;-126;2;1;1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.less_than``() =
  validate (SBytes.Collection.ListArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;0;1;0;0;1;0;1;0;1;1;
   0;1;0;0;0;0;1;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.less_or_equal``() =
  validate (SBytes.Collection.ListArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;
   0;1;0;0;1;0;1;0;1;1;0;1;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.greater_than``() =
  validate (SBytes.Collection.ListArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;
   1;0;1;1;0;1;0;1;0;0;1;0;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.ListArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;1;0;1;1;0;1;0;1;0;0;
   1;0;1;1;1;1;0;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.equals``() =
  validate (SBytes.Collection.ListArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.equal``() =
  validate (SBytes.Collection.ListArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.not_equal``() =
  validate (SBytes.Collection.ListArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.compare``() =
  validate (SBytes.Collection.ListArray) C.N.compare [|
   0;-255;-127;-128;-129;-1;-255;-127;-128;-129;255;0;128;127;126;255;-1;128;127;126;127;-128;0;-1;-2;127;-128;-1;-1;-2;128;-127;1;0;-1;128;-127;1;-1;-1;
   129;-126;2;1;0;129;-126;2;1;-1;1;-255;-127;-128;-129;0;-255;-127;-128;-129;255;1;128;127;126;255;0;128;127;126;127;-128;1;-1;-2;127;-128;0;-1;-2;
   128;-127;1;1;-1;128;-127;1;0;-1;129;-126;2;1;1;129;-126;2;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.less_than``() =
  validate (SBytes.Collection.ListArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;0;1;0;0;1;0;1;0;1;1;
   0;1;0;0;0;0;1;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.less_or_equal``() =
  validate (SBytes.Collection.ListArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;
   0;1;0;0;1;0;1;0;1;1;0;1;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.greater_than``() =
  validate (SBytes.Collection.ListArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;
   1;0;1;1;0;1;0;1;0;0;1;0;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ListArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.ListArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;1;0;1;1;0;1;0;1;0;0;
   1;0;1;1;1;1;0;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.equals``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.not_equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.compare``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;1;1;-1;1;0;-1;1;1;1;-1;1;
   1;-1;1;1;0;1;1;1;1;1;1;-1;-1;-1;-1;0;-1;-1;-1;1;1;-1;1;-1;-1;1;0;1;-1;1;1;-1;1;-1;-1;1;-1;0;-1;1;
   1;-1;1;1;-1;1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_than``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;1;1;0;0;1;0;0;1;0;0;0;1;0;
   0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;0;1;0;1;1;0;1;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_or_equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0;0;1;0;1;1;0;1;0;1;0;0;1;0;1;1;0;1;1;1;0;
   0;1;0;0;1;0;0;0;1;0;0;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_than``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1;1;0;1;0;0;1;0;1;0;1;1;0;1;0;0;1;0;0;0;1;
   1;0;1;1;0;1;1;1;0;1;1;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_or_equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;0;0;1;1;0;1;1;0;1;1;1;0;1;
   1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;1;0;1;0;0;1;0;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.equals``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.not_equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.compare``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;1;1;-1;1;0;-1;1;1;1;-1;1;
   1;-1;1;1;0;1;1;1;1;1;1;-1;-1;-1;-1;0;-1;-1;-1;1;1;-1;1;-1;-1;1;0;1;-1;1;1;-1;1;-1;-1;1;-1;0;-1;1;
   1;-1;1;1;-1;1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_than``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;1;1;0;0;1;0;0;1;0;0;0;1;0;
   0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;0;1;0;1;1;0;1;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_or_equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0;0;1;0;1;1;0;1;0;1;0;0;1;0;1;1;0;1;1;1;0;
   0;1;0;0;1;0;0;0;1;0;0;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_than``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1;1;0;1;0;0;1;0;1;0;1;1;0;1;0;0;1;0;0;0;1;
   1;0;1;1;0;1;1;1;0;1;1;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``SBytes.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_or_equal``() =
  validate (SBytes.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;0;0;1;1;0;1;1;0;1;1;1;0;1;
   1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;1;0;1;0;0;1;0;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.Array E.I.equals``() =
  validate (NullableSbytes.Collection.Array) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.Array E.I.equal``() =
  validate (NullableSbytes.Collection.Array) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.Array E.I.not_equal``() =
  validate (NullableSbytes.Collection.Array) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.Array E.N.equals``() =
  validate (NullableSbytes.Collection.Array) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.Array E.N.equal``() =
  validate (NullableSbytes.Collection.Array) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.Array E.N.not_equal``() =
  validate (NullableSbytes.Collection.Array) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.OptionArray E.I.equals``() =
  validate (NullableSbytes.Collection.OptionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.OptionArray E.I.equal``() =
  validate (NullableSbytes.Collection.OptionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.OptionArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.OptionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.OptionArray E.N.equals``() =
  validate (NullableSbytes.Collection.OptionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.OptionArray E.N.equal``() =
  validate (NullableSbytes.Collection.OptionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.OptionArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.OptionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefArray E.I.equals``() =
  validate (NullableSbytes.Collection.RefArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefArray E.I.equal``() =
  validate (NullableSbytes.Collection.RefArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.RefArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefArray E.N.equals``() =
  validate (NullableSbytes.Collection.RefArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefArray E.N.equal``() =
  validate (NullableSbytes.Collection.RefArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.RefArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefWrapArray E.I.equals``() =
  validate (NullableSbytes.Collection.RefWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefWrapArray E.I.equal``() =
  validate (NullableSbytes.Collection.RefWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefWrapArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.RefWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefWrapArray E.N.equals``() =
  validate (NullableSbytes.Collection.RefWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefWrapArray E.N.equal``() =
  validate (NullableSbytes.Collection.RefWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.RefWrapArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.RefWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionArray E.I.equals``() =
  validate (NullableSbytes.Collection.UnionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionArray E.I.equal``() =
  validate (NullableSbytes.Collection.UnionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.UnionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionArray E.N.equals``() =
  validate (NullableSbytes.Collection.UnionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionArray E.N.equal``() =
  validate (NullableSbytes.Collection.UnionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.UnionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionWrapArray E.I.equals``() =
  validate (NullableSbytes.Collection.UnionWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionWrapArray E.I.equal``() =
  validate (NullableSbytes.Collection.UnionWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionWrapArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.UnionWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionWrapArray E.N.equals``() =
  validate (NullableSbytes.Collection.UnionWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionWrapArray E.N.equal``() =
  validate (NullableSbytes.Collection.UnionWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.UnionWrapArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.UnionWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueArray E.I.equals``() =
  validate (NullableSbytes.Collection.ValueArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueArray E.I.equal``() =
  validate (NullableSbytes.Collection.ValueArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.ValueArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueArray E.N.equals``() =
  validate (NullableSbytes.Collection.ValueArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueArray E.N.equal``() =
  validate (NullableSbytes.Collection.ValueArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.ValueArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueWrapArray E.I.equals``() =
  validate (NullableSbytes.Collection.ValueWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueWrapArray E.I.equal``() =
  validate (NullableSbytes.Collection.ValueWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueWrapArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.ValueWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueWrapArray E.N.equals``() =
  validate (NullableSbytes.Collection.ValueWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueWrapArray E.N.equal``() =
  validate (NullableSbytes.Collection.ValueWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ValueWrapArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.ValueWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ArrayArray E.I.equals``() =
  validate (NullableSbytes.Collection.ArrayArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ArrayArray E.I.equal``() =
  validate (NullableSbytes.Collection.ArrayArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ArrayArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.ArrayArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ArrayArray E.N.equals``() =
  validate (NullableSbytes.Collection.ArrayArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ArrayArray E.N.equal``() =
  validate (NullableSbytes.Collection.ArrayArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ArrayArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.ArrayArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ListArray E.I.equals``() =
  validate (NullableSbytes.Collection.ListArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ListArray E.I.equal``() =
  validate (NullableSbytes.Collection.ListArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ListArray E.I.not_equal``() =
  validate (NullableSbytes.Collection.ListArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ListArray E.N.equals``() =
  validate (NullableSbytes.Collection.ListArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ListArray E.N.equal``() =
  validate (NullableSbytes.Collection.ListArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableSbytes.Collection.ListArray E.N.not_equal``() =
  validate (NullableSbytes.Collection.ListArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.equals``() =
  validate (Int16s.Collection.Array) C.I.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.equal``() =
  validate (Int16s.Collection.Array) C.I.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.not_equal``() =
  validate (Int16s.Collection.Array) C.I.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.compare``() =
  validate (Int16s.Collection.Array) C.I.compare [|
   0;0;1;1;1;0;0;1;1;1;-1;-1;0;-1;-1;-1;-1;1;0;-1;-1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.less_than``() =
  validate (Int16s.Collection.Array) C.I.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.less_or_equal``() =
  validate (Int16s.Collection.Array) C.I.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.greater_than``() =
  validate (Int16s.Collection.Array) C.I.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.I.greater_or_equal``() =
  validate (Int16s.Collection.Array) C.I.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.equals``() =
  validate (Int16s.Collection.Array) C.N.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.equal``() =
  validate (Int16s.Collection.Array) C.N.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.not_equal``() =
  validate (Int16s.Collection.Array) C.N.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.compare``() =
  validate (Int16s.Collection.Array) C.N.compare [|
   0;0;1;1;1;0;0;1;1;1;-1;-1;0;-1;-1;-1;-1;1;0;-1;-1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.less_than``() =
  validate (Int16s.Collection.Array) C.N.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.less_or_equal``() =
  validate (Int16s.Collection.Array) C.N.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.greater_than``() =
  validate (Int16s.Collection.Array) C.N.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.Array C.N.greater_or_equal``() =
  validate (Int16s.Collection.Array) C.N.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.equals``() =
  validate (Int16s.Collection.OptionArray) C.I.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.equal``() =
  validate (Int16s.Collection.OptionArray) C.I.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.not_equal``() =
  validate (Int16s.Collection.OptionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.compare``() =
  validate (Int16s.Collection.OptionArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;0;32768;32767;32766;1;0;0;32768;32767;32766;1;-32768;-32768;0;-1;-2;1;-32767;-32767;1;0;-1;1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.less_than``() =
  validate (Int16s.Collection.OptionArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;0;1;1;0;0;1;0;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.less_or_equal``() =
  validate (Int16s.Collection.OptionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;0;1;1;0;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.greater_than``() =
  validate (Int16s.Collection.OptionArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;1;0;0;1;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.OptionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;1;0;0;1;1;0;1;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.equals``() =
  validate (Int16s.Collection.OptionArray) C.N.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.equal``() =
  validate (Int16s.Collection.OptionArray) C.N.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.not_equal``() =
  validate (Int16s.Collection.OptionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.compare``() =
  validate (Int16s.Collection.OptionArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;0;32768;32767;32766;1;0;0;32768;32767;32766;1;-32768;-32768;0;-1;-2;1;-32767;-32767;1;0;-1;1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.less_than``() =
  validate (Int16s.Collection.OptionArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;0;1;1;0;0;1;0;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.less_or_equal``() =
  validate (Int16s.Collection.OptionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;0;1;1;0;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.greater_than``() =
  validate (Int16s.Collection.OptionArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;1;0;0;1;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.OptionArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.OptionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;1;0;0;1;1;0;1;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.equals``() =
  validate (Int16s.Collection.RefArray) C.I.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.equal``() =
  validate (Int16s.Collection.RefArray) C.I.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.not_equal``() =
  validate (Int16s.Collection.RefArray) C.I.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.compare``() =
  validate (Int16s.Collection.RefArray) C.I.compare [|
   0;0;1;1;1;0;0;1;1;1;-1;-1;0;-1;-1;-1;-1;1;0;-1;-1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.less_than``() =
  validate (Int16s.Collection.RefArray) C.I.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.less_or_equal``() =
  validate (Int16s.Collection.RefArray) C.I.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.greater_than``() =
  validate (Int16s.Collection.RefArray) C.I.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.RefArray) C.I.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.equals``() =
  validate (Int16s.Collection.RefArray) C.N.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.equal``() =
  validate (Int16s.Collection.RefArray) C.N.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.not_equal``() =
  validate (Int16s.Collection.RefArray) C.N.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.compare``() =
  validate (Int16s.Collection.RefArray) C.N.compare [|
   0;0;1;1;1;0;0;1;1;1;-1;-1;0;-1;-1;-1;-1;1;0;-1;-1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.less_than``() =
  validate (Int16s.Collection.RefArray) C.N.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.less_or_equal``() =
  validate (Int16s.Collection.RefArray) C.N.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.greater_than``() =
  validate (Int16s.Collection.RefArray) C.N.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.RefArray) C.N.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.equals``() =
  validate (Int16s.Collection.RefWrapArray) C.I.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.equal``() =
  validate (Int16s.Collection.RefWrapArray) C.I.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.not_equal``() =
  validate (Int16s.Collection.RefWrapArray) C.I.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.compare``() =
  validate (Int16s.Collection.RefWrapArray) C.I.compare [|
   0;0;32768;32767;32766;0;0;32768;32767;32766;-32768;-32768;0;-1;-2;-32767;-32767;1;0;-1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.less_than``() =
  validate (Int16s.Collection.RefWrapArray) C.I.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.less_or_equal``() =
  validate (Int16s.Collection.RefWrapArray) C.I.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.greater_than``() =
  validate (Int16s.Collection.RefWrapArray) C.I.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.RefWrapArray) C.I.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.equals``() =
  validate (Int16s.Collection.RefWrapArray) C.N.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.equal``() =
  validate (Int16s.Collection.RefWrapArray) C.N.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.not_equal``() =
  validate (Int16s.Collection.RefWrapArray) C.N.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.compare``() =
  validate (Int16s.Collection.RefWrapArray) C.N.compare [|
   0;0;32768;32767;32766;0;0;32768;32767;32766;-32768;-32768;0;-1;-2;-32767;-32767;1;0;-1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.less_than``() =
  validate (Int16s.Collection.RefWrapArray) C.N.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.less_or_equal``() =
  validate (Int16s.Collection.RefWrapArray) C.N.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.greater_than``() =
  validate (Int16s.Collection.RefWrapArray) C.N.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.RefWrapArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.RefWrapArray) C.N.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.equals``() =
  validate (Int16s.Collection.UnionArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.equal``() =
  validate (Int16s.Collection.UnionArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.not_equal``() =
  validate (Int16s.Collection.UnionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.compare``() =
  validate (Int16s.Collection.UnionArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;0;
   -1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;0;-1;-2;-3;3;
   2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;1;-1;
   -2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;-1;3;2;1;2;
   1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;
   1;3;2;1;1;0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;0;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;-1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;-1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   -1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.less_than``() =
  validate (Int16s.Collection.UnionArray) C.I.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.less_or_equal``() =
  validate (Int16s.Collection.UnionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.greater_than``() =
  validate (Int16s.Collection.UnionArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.UnionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.equals``() =
  validate (Int16s.Collection.UnionArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.equal``() =
  validate (Int16s.Collection.UnionArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.not_equal``() =
  validate (Int16s.Collection.UnionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.compare``() =
  validate (Int16s.Collection.UnionArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;0;
   -1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;0;-1;-2;-3;3;
   2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;1;-1;
   -2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;-1;3;2;1;2;
   1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;
   1;3;2;1;1;0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;0;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;0;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;-1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;-1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   -1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.less_than``() =
  validate (Int16s.Collection.UnionArray) C.N.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.less_or_equal``() =
  validate (Int16s.Collection.UnionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.greater_than``() =
  validate (Int16s.Collection.UnionArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.UnionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.equals``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.not_equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.compare``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;32768;-1;-2;-3;-3;-3;-3;32767;-1;-2;-3;-3;-3;-3;32766;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;0;-1;-2;-2;-2;-2;1;32768;-1;-2;-2;-2;-2;1;32767;-1;-2;-2;-2;-2;1;32766;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;0;
   -1;-1;-1;-1;2;1;32768;-1;-1;-1;-1;2;1;32767;-1;-1;-1;-1;2;1;32766;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;0;-1;-2;-3;3;
   2;1;32768;-1;-2;-3;3;2;1;32767;-1;-2;-3;3;2;1;32766;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;32768;-1;
   -2;3;2;1;1;32767;-1;-2;3;2;1;1;32766;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;32768;-1;3;2;1;2;
   1;32767;-1;3;2;1;2;1;32766;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;32768;3;2;1;3;2;1;32767;3;2;
   1;3;2;1;32766;0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;32768;-1;-2;-3;-3;-3;-3;32767;-1;-2;-3;-3;-3;-3;32766;-1;-2;-3;-3;-3;-3;
   1;0;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;32768;-1;-2;-2;-2;-2;1;32767;-1;-2;-2;-2;-2;1;32766;-1;-2;-2;-2;-2;2;1;0;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;32768;-1;-1;-1;-1;2;1;32767;-1;-1;-1;-1;2;1;32766;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;32768;-1;-2;-3;3;2;1;32767;-1;-2;-3;3;2;1;32766;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;32768;-1;-2;3;2;1;1;32767;-1;-2;3;2;1;1;32766;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;32768;
   -1;3;2;1;2;1;32767;-1;3;2;1;2;1;32766;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;32768;3;2;1;3;
   2;1;32767;3;2;1;3;2;1;32766;-32768;-1;-2;-3;-3;-3;-3;-32768;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-2;-1;
   -2;-3;-3;-3;-3;1;-32768;-1;-2;-2;-2;-2;1;-32768;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-2;-1;-2;-2;-2;-2;
   2;1;-32768;-1;-1;-1;-1;2;1;-32768;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-2;-1;-1;-1;-1;3;2;1;-32768;-1;
   -2;-3;3;2;1;-32768;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-2;-1;-2;-3;3;2;1;1;-32768;-1;-2;3;2;1;
   1;-32768;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-2;-1;-2;3;2;1;2;1;-32768;-1;3;2;1;2;1;-32768;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-2;-1;3;2;1;3;2;1;-32768;3;2;1;3;2;1;-32768;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-2;-32767;-1;-2;-3;-3;-3;-3;-32767;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-32767;-1;-2;-2;-2;-2;1;-32767;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;-32767;-1;-1;-1;-1;2;1;-32767;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;-32767;-1;-2;-3;3;2;1;-32767;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-32767;
   -1;-2;3;2;1;1;-32767;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;-32767;-1;3;2;1;
   2;1;-32767;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;-32767;3;2;1;3;2;1;-32767;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;-32766;-1;-2;-3;-3;-3;-3;-32766;-1;-2;-3;-3;-3;-3;2;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-32766;-1;-2;-2;-2;-2;1;-32766;-1;-2;-2;-2;-2;1;2;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;-32766;-1;-1;-1;-1;2;1;-32766;-1;-1;-1;-1;2;1;2;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;-32766;-1;-2;-3;3;2;1;-32766;-1;-2;-3;3;2;1;2;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;-32766;-1;-2;3;2;1;1;-32766;-1;-2;3;2;1;1;2;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   -32766;-1;3;2;1;2;1;-32766;-1;3;2;1;2;1;2;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-32766;3;2;1;
   3;2;1;-32766;3;2;1;3;2;1;2;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.less_than``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.less_or_equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.greater_than``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.equals``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.not_equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.compare``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;32768;-1;-2;-3;-3;-3;-3;32767;-1;-2;-3;-3;-3;-3;32766;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;0;-1;-2;-2;-2;-2;1;32768;-1;-2;-2;-2;-2;1;32767;-1;-2;-2;-2;-2;1;32766;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;0;
   -1;-1;-1;-1;2;1;32768;-1;-1;-1;-1;2;1;32767;-1;-1;-1;-1;2;1;32766;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;0;-1;-2;-3;3;
   2;1;32768;-1;-2;-3;3;2;1;32767;-1;-2;-3;3;2;1;32766;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;32768;-1;
   -2;3;2;1;1;32767;-1;-2;3;2;1;1;32766;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;32768;-1;3;2;1;2;
   1;32767;-1;3;2;1;2;1;32766;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;32768;3;2;1;3;2;1;32767;3;2;
   1;3;2;1;32766;0;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;32768;-1;-2;-3;-3;-3;-3;32767;-1;-2;-3;-3;-3;-3;32766;-1;-2;-3;-3;-3;-3;
   1;0;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;32768;-1;-2;-2;-2;-2;1;32767;-1;-2;-2;-2;-2;1;32766;-1;-2;-2;-2;-2;2;1;0;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;32768;-1;-1;-1;-1;2;1;32767;-1;-1;-1;-1;2;1;32766;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;32768;-1;-2;-3;3;2;1;32767;-1;-2;-3;3;2;1;32766;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;32768;-1;-2;3;2;1;1;32767;-1;-2;3;2;1;1;32766;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;0;-1;3;2;1;2;1;32768;
   -1;3;2;1;2;1;32767;-1;3;2;1;2;1;32766;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;0;3;2;1;3;2;1;32768;3;2;1;3;
   2;1;32767;3;2;1;3;2;1;32766;-32768;-1;-2;-3;-3;-3;-3;-32768;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-2;-1;
   -2;-3;-3;-3;-3;1;-32768;-1;-2;-2;-2;-2;1;-32768;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-2;-1;-2;-2;-2;-2;
   2;1;-32768;-1;-1;-1;-1;2;1;-32768;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-2;-1;-1;-1;-1;3;2;1;-32768;-1;
   -2;-3;3;2;1;-32768;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-2;-1;-2;-3;3;2;1;1;-32768;-1;-2;3;2;1;
   1;-32768;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-2;-1;-2;3;2;1;2;1;-32768;-1;3;2;1;2;1;-32768;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-2;-1;3;2;1;3;2;1;-32768;3;2;1;3;2;1;-32768;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-2;-32767;-1;-2;-3;-3;-3;-3;-32767;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-32767;-1;-2;-2;-2;-2;1;-32767;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;-32767;-1;-1;-1;-1;2;1;-32767;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;-32767;-1;-2;-3;3;2;1;-32767;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-32767;
   -1;-2;3;2;1;1;-32767;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;-32767;-1;3;2;1;
   2;1;-32767;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;-32767;3;2;1;3;2;1;-32767;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;-32766;-1;-2;-3;-3;-3;-3;-32766;-1;-2;-3;-3;-3;-3;2;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-32766;-1;-2;-2;-2;-2;1;-32766;-1;-2;-2;-2;-2;1;2;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;-32766;-1;-1;-1;-1;2;1;-32766;-1;-1;-1;-1;2;1;2;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;-32766;-1;-2;-3;3;2;1;-32766;-1;-2;-3;3;2;1;2;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;-32766;-1;-2;3;2;1;1;-32766;-1;-2;3;2;1;1;2;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   -32766;-1;3;2;1;2;1;-32766;-1;3;2;1;2;1;2;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;-32766;3;2;1;
   3;2;1;-32766;3;2;1;3;2;1;2;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.less_than``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.less_than [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.less_or_equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;
   1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.greater_than``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.UnionWrapArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.UnionWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;
   0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.equals``() =
  validate (Int16s.Collection.ValueArray) C.I.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.equal``() =
  validate (Int16s.Collection.ValueArray) C.I.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.not_equal``() =
  validate (Int16s.Collection.ValueArray) C.I.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.compare``() =
  validate (Int16s.Collection.ValueArray) C.I.compare [|
   0;0;1;1;1;0;0;1;1;1;-1;-1;0;-1;-1;-1;-1;1;0;-1;-1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.less_than``() =
  validate (Int16s.Collection.ValueArray) C.I.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.less_or_equal``() =
  validate (Int16s.Collection.ValueArray) C.I.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.greater_than``() =
  validate (Int16s.Collection.ValueArray) C.I.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.ValueArray) C.I.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.equals``() =
  validate (Int16s.Collection.ValueArray) C.N.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.equal``() =
  validate (Int16s.Collection.ValueArray) C.N.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.not_equal``() =
  validate (Int16s.Collection.ValueArray) C.N.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.compare``() =
  validate (Int16s.Collection.ValueArray) C.N.compare [|
   0;0;1;1;1;0;0;1;1;1;-1;-1;0;-1;-1;-1;-1;1;0;-1;-1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.less_than``() =
  validate (Int16s.Collection.ValueArray) C.N.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.less_or_equal``() =
  validate (Int16s.Collection.ValueArray) C.N.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.greater_than``() =
  validate (Int16s.Collection.ValueArray) C.N.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.ValueArray) C.N.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.equals``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.not_equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.compare``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.compare [|
   0;0;32768;32767;32766;0;0;32768;32767;32766;-32768;-32768;0;-1;-2;-32767;-32767;1;0;-1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.less_than``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.less_or_equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.greater_than``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.I.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.equals``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.equals [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.equal [|
   1;1;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.not_equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.not_equal [|
   0;0;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.compare``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.compare [|
   0;0;32768;32767;32766;0;0;32768;32767;32766;-32768;-32768;0;-1;-2;-32767;-32767;1;0;-1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.less_than``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.less_than [|
   0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;0;0;1;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.less_or_equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.greater_than``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ValueWrapArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.ValueWrapArray) C.N.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;1;1;0;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.equals``() =
  validate (Int16s.Collection.ArrayArray) C.I.equals [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.equal``() =
  validate (Int16s.Collection.ArrayArray) C.I.equal [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.not_equal``() =
  validate (Int16s.Collection.ArrayArray) C.I.not_equal [|
   0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.compare``() =
  validate (Int16s.Collection.ArrayArray) C.I.compare [|
   0;0;32768;32767;32766;-1;-1;-1;-1;-1;0;0;32768;32767;32766;-1;-1;-1;-1;-1;-32768;-32768;0;-1;-2;-1;-1;-1;-1;-1;-32767;-32767;1;0;-1;-1;-1;-1;-1;-1;
   -32766;-32766;2;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;32768;32768;32767;32766;1;1;1;1;1;-32768;0;32768;32767;32766;1;1;1;1;1;-32768;-32768;0;-1;-2;
   1;1;1;1;1;-32767;-32767;1;0;-1;1;1;1;1;1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.less_than``() =
  validate (Int16s.Collection.ArrayArray) C.I.less_than [|
   0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;
   1;1;0;0;0;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;1;1;
   0;0;0;0;0;1;1;0;0;1;0;0;0;0;0;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.less_or_equal``() =
  validate (Int16s.Collection.ArrayArray) C.I.less_or_equal [|
   1;1;0;0;0;1;1;1;1;1;1;1;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;1;1;1;1;1;
   0;0;0;0;0;1;1;0;1;1;0;0;0;0;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.greater_than``() =
  validate (Int16s.Collection.ArrayArray) C.I.greater_than [|
   0;0;1;1;1;0;0;0;0;0;0;0;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;0;0;0;0;0;
   1;1;1;1;1;0;0;1;0;0;1;1;1;1;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.ArrayArray) C.I.greater_or_equal [|
   1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;
   0;0;1;1;1;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;0;0;
   1;1;1;1;1;0;0;1;1;0;1;1;1;1;1;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.equals``() =
  validate (Int16s.Collection.ArrayArray) C.N.equals [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.equal``() =
  validate (Int16s.Collection.ArrayArray) C.N.equal [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.not_equal``() =
  validate (Int16s.Collection.ArrayArray) C.N.not_equal [|
   0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.compare``() =
  validate (Int16s.Collection.ArrayArray) C.N.compare [|
   0;0;32768;32767;32766;-1;-1;-1;-1;-1;0;0;32768;32767;32766;-1;-1;-1;-1;-1;-32768;-32768;0;-1;-2;-1;-1;-1;-1;-1;-32767;-32767;1;0;-1;-1;-1;-1;-1;-1;
   -32766;-32766;2;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;32768;32768;32767;32766;1;1;1;1;1;-32768;0;32768;32767;32766;1;1;1;1;1;-32768;-32768;0;-1;-2;
   1;1;1;1;1;-32767;-32767;1;0;-1;1;1;1;1;1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.less_than``() =
  validate (Int16s.Collection.ArrayArray) C.N.less_than [|
   0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;
   1;1;0;0;0;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;1;1;
   0;0;0;0;0;1;1;0;0;1;0;0;0;0;0;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.less_or_equal``() =
  validate (Int16s.Collection.ArrayArray) C.N.less_or_equal [|
   1;1;0;0;0;1;1;1;1;1;1;1;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;1;1;1;1;1;
   0;0;0;0;0;1;1;0;1;1;0;0;0;0;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.greater_than``() =
  validate (Int16s.Collection.ArrayArray) C.N.greater_than [|
   0;0;1;1;1;0;0;0;0;0;0;0;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;0;0;0;0;0;
   1;1;1;1;1;0;0;1;0;0;1;1;1;1;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.ArrayArray) C.N.greater_or_equal [|
   1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;
   0;0;1;1;1;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;0;0;
   1;1;1;1;1;0;0;1;1;0;1;1;1;1;1;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.equals``() =
  validate (Int16s.Collection.ListArray) C.I.equals [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.equal``() =
  validate (Int16s.Collection.ListArray) C.I.equal [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.not_equal``() =
  validate (Int16s.Collection.ListArray) C.I.not_equal [|
   0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.compare``() =
  validate (Int16s.Collection.ListArray) C.I.compare [|
   0;0;32768;32767;32766;-1;-1;32768;32767;32766;0;0;32768;32767;32766;-1;-1;32768;32767;32766;-32768;-32768;0;-1;-2;-32768;-32768;-1;-1;-2;-32767;-32767;1;0;-1;-32767;-32767;1;-1;-1;
   -32766;-32766;2;1;0;-32766;-32766;2;1;-1;1;1;32768;32767;32766;0;32768;32768;32767;32766;1;1;32768;32767;32766;-32768;0;32768;32767;32766;-32768;-32768;1;-1;-2;-32768;-32768;0;-1;-2;
   -32767;-32767;1;1;-1;-32767;-32767;1;0;-1;-32766;-32766;2;1;1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.less_than``() =
  validate (Int16s.Collection.ListArray) C.I.less_than [|
   0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;0;1;1;
   1;1;0;0;0;1;1;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0;1;1;1;1;0;1;1;
   1;1;0;0;1;1;1;0;0;1;1;1;0;0;0;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.less_or_equal``() =
  validate (Int16s.Collection.ListArray) C.I.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;0;1;1;
   1;1;0;0;1;1;1;0;0;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;1;1;0;1;1;1;1;1;1;1;
   1;1;0;0;1;1;1;0;1;1;1;1;0;0;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.greater_than``() =
  validate (Int16s.Collection.ListArray) C.I.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;1;0;0;
   0;0;1;1;0;0;0;1;1;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;0;0;1;0;0;0;0;0;0;0;
   0;0;1;1;0;0;0;1;0;0;0;0;1;1;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.ListArray) C.I.greater_or_equal [|
   1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;1;0;0;
   0;0;1;1;1;0;0;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1;0;0;0;0;1;0;0;
   0;0;1;1;0;0;0;1;1;0;0;0;1;1;1;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.equals``() =
  validate (Int16s.Collection.ListArray) C.N.equals [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.equal``() =
  validate (Int16s.Collection.ListArray) C.N.equal [|
   1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.not_equal``() =
  validate (Int16s.Collection.ListArray) C.N.not_equal [|
   0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.compare``() =
  validate (Int16s.Collection.ListArray) C.N.compare [|
   0;0;32768;32767;32766;-1;-1;32768;32767;32766;0;0;32768;32767;32766;-1;-1;32768;32767;32766;-32768;-32768;0;-1;-2;-32768;-32768;-1;-1;-2;-32767;-32767;1;0;-1;-32767;-32767;1;-1;-1;
   -32766;-32766;2;1;0;-32766;-32766;2;1;-1;1;1;32768;32767;32766;0;32768;32768;32767;32766;1;1;32768;32767;32766;-32768;0;32768;32767;32766;-32768;-32768;1;-1;-2;-32768;-32768;0;-1;-2;
   -32767;-32767;1;1;-1;-32767;-32767;1;0;-1;-32766;-32766;2;1;1;-32766;-32766;2;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.less_than``() =
  validate (Int16s.Collection.ListArray) C.N.less_than [|
   0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;1;1;0;0;0;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;0;1;1;
   1;1;0;0;0;1;1;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;1;1;0;1;1;1;1;0;1;1;
   1;1;0;0;1;1;1;0;0;1;1;1;0;0;0;1;1;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.less_or_equal``() =
  validate (Int16s.Collection.ListArray) C.N.less_or_equal [|
   1;1;0;0;0;1;1;0;0;0;1;1;0;0;0;1;1;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;0;1;1;
   1;1;0;0;1;1;1;0;0;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;1;1;0;1;1;1;1;1;1;1;
   1;1;0;0;1;1;1;0;1;1;1;1;0;0;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.greater_than``() =
  validate (Int16s.Collection.ListArray) C.N.greater_than [|
   0;0;1;1;1;0;0;1;1;1;0;0;1;1;1;0;0;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;1;0;0;
   0;0;1;1;0;0;0;1;1;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;1;0;0;1;0;0;0;0;0;0;0;
   0;0;1;1;0;0;0;1;0;0;0;0;1;1;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ListArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.ListArray) C.N.greater_or_equal [|
   1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;0;0;1;1;1;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;0;1;0;0;
   0;0;1;1;1;0;0;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;0;0;1;0;0;0;0;1;0;0;
   0;0;1;1;0;0;0;1;1;0;0;0;1;1;1;0;0;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.equals``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equals [|
   1;1;0;0;0;1;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equal [|
   1;1;0;0;0;1;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.not_equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.not_equal [|
   0;0;1;1;1;0;1;1;1;1;0;0;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.compare``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.compare [|
   0;0;1;1;1;0;1;1;1;1;0;0;1;1;1;0;1;1;1;1;-1;-1;0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;-1;-1;1;1;-1;-1;
   -1;-1;1;1;0;-1;1;1;1;-1;0;0;1;1;1;0;1;1;1;1;-1;-1;1;-1;-1;-1;0;1;-1;-1;-1;-1;1;-1;-1;-1;-1;0;-1;-1;
   -1;-1;1;1;-1;-1;1;1;0;-1;-1;-1;1;1;1;-1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_than``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;0;0;1;1;
   1;1;0;0;0;1;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;0;0;1;1;1;1;0;1;1;1;1;0;1;1;
   1;1;0;0;1;1;0;0;0;1;1;1;0;0;0;1;0;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_or_equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_or_equal [|
   1;1;0;0;0;1;0;0;0;0;1;1;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;0;0;1;1;
   1;1;0;0;1;1;0;0;0;1;1;1;0;0;0;1;0;0;0;0;1;1;0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;0;0;1;1;0;0;1;1;1;1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_than``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_than [|
   0;0;1;1;1;0;1;1;1;1;0;0;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;1;1;0;0;
   0;0;1;1;0;0;1;1;1;0;0;0;1;1;1;0;1;1;1;1;0;0;1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;1;1;0;0;1;1;0;0;0;0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_or_equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;1;1;0;0;
   0;0;1;1;1;0;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;1;1;0;0;0;0;1;0;0;0;0;1;0;0;
   0;0;1;1;0;0;1;1;1;0;0;0;1;1;1;0;1;1;1;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.equals``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equals [|
   1;1;0;0;0;1;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equal [|
   1;1;0;0;0;1;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.not_equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.not_equal [|
   0;0;1;1;1;0;1;1;1;1;0;0;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.compare``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.compare [|
   0;0;1;1;1;0;1;1;1;1;0;0;1;1;1;0;1;1;1;1;-1;-1;0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;-1;-1;1;1;-1;-1;
   -1;-1;1;1;0;-1;1;1;1;-1;0;0;1;1;1;0;1;1;1;1;-1;-1;1;-1;-1;-1;0;1;-1;-1;-1;-1;1;-1;-1;-1;-1;0;-1;-1;
   -1;-1;1;1;-1;-1;1;1;0;-1;-1;-1;1;1;1;-1;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_than``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;1;1;1;1;1;1;0;0;1;1;0;0;1;1;
   1;1;0;0;0;1;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;1;0;1;1;1;0;0;1;1;1;1;0;1;1;1;1;0;1;1;
   1;1;0;0;1;1;0;0;0;1;1;1;0;0;0;1;0;0;0;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_or_equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_or_equal [|
   1;1;0;0;0;1;0;0;0;0;1;1;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;0;0;1;1;
   1;1;0;0;1;1;0;0;0;1;1;1;0;0;0;1;0;0;0;0;1;1;0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;0;0;1;1;0;0;1;1;1;1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_than``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_than [|
   0;0;1;1;1;0;1;1;1;1;0;0;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;1;1;0;0;
   0;0;1;1;0;0;1;1;1;0;0;0;1;1;1;0;1;1;1;1;0;0;1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;1;1;0;0;1;1;0;0;0;0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``Int16s.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_or_equal``() =
  validate (Int16s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;0;0;0;0;0;0;1;1;0;0;1;1;0;0;
   0;0;1;1;1;0;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;0;1;0;0;0;1;1;0;0;0;0;1;0;0;0;0;1;0;0;
   0;0;1;1;0;0;1;1;1;0;0;0;1;1;1;0;1;1;1;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.Array E.I.equals``() =
  validate (NullableInt16s.Collection.Array) E.I.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.Array E.I.equal``() =
  validate (NullableInt16s.Collection.Array) E.I.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.Array E.I.not_equal``() =
  validate (NullableInt16s.Collection.Array) E.I.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.Array E.N.equals``() =
  validate (NullableInt16s.Collection.Array) E.N.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.Array E.N.equal``() =
  validate (NullableInt16s.Collection.Array) E.N.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.Array E.N.not_equal``() =
  validate (NullableInt16s.Collection.Array) E.N.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.OptionArray E.I.equals``() =
  validate (NullableInt16s.Collection.OptionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.OptionArray E.I.equal``() =
  validate (NullableInt16s.Collection.OptionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.OptionArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.OptionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.OptionArray E.N.equals``() =
  validate (NullableInt16s.Collection.OptionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.OptionArray E.N.equal``() =
  validate (NullableInt16s.Collection.OptionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.OptionArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.OptionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefArray E.I.equals``() =
  validate (NullableInt16s.Collection.RefArray) E.I.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefArray E.I.equal``() =
  validate (NullableInt16s.Collection.RefArray) E.I.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.RefArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefArray E.N.equals``() =
  validate (NullableInt16s.Collection.RefArray) E.N.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefArray E.N.equal``() =
  validate (NullableInt16s.Collection.RefArray) E.N.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.RefArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefWrapArray E.I.equals``() =
  validate (NullableInt16s.Collection.RefWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefWrapArray E.I.equal``() =
  validate (NullableInt16s.Collection.RefWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefWrapArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.RefWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefWrapArray E.N.equals``() =
  validate (NullableInt16s.Collection.RefWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefWrapArray E.N.equal``() =
  validate (NullableInt16s.Collection.RefWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.RefWrapArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.RefWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionArray E.I.equals``() =
  validate (NullableInt16s.Collection.UnionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionArray E.I.equal``() =
  validate (NullableInt16s.Collection.UnionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.UnionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionArray E.N.equals``() =
  validate (NullableInt16s.Collection.UnionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionArray E.N.equal``() =
  validate (NullableInt16s.Collection.UnionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.UnionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionWrapArray E.I.equals``() =
  validate (NullableInt16s.Collection.UnionWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionWrapArray E.I.equal``() =
  validate (NullableInt16s.Collection.UnionWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionWrapArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.UnionWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionWrapArray E.N.equals``() =
  validate (NullableInt16s.Collection.UnionWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionWrapArray E.N.equal``() =
  validate (NullableInt16s.Collection.UnionWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.UnionWrapArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.UnionWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueArray E.I.equals``() =
  validate (NullableInt16s.Collection.ValueArray) E.I.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueArray E.I.equal``() =
  validate (NullableInt16s.Collection.ValueArray) E.I.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.ValueArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueArray E.N.equals``() =
  validate (NullableInt16s.Collection.ValueArray) E.N.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueArray E.N.equal``() =
  validate (NullableInt16s.Collection.ValueArray) E.N.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.ValueArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueWrapArray E.I.equals``() =
  validate (NullableInt16s.Collection.ValueWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueWrapArray E.I.equal``() =
  validate (NullableInt16s.Collection.ValueWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueWrapArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.ValueWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueWrapArray E.N.equals``() =
  validate (NullableInt16s.Collection.ValueWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueWrapArray E.N.equal``() =
  validate (NullableInt16s.Collection.ValueWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;1;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ValueWrapArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.ValueWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;0;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ArrayArray E.I.equals``() =
  validate (NullableInt16s.Collection.ArrayArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ArrayArray E.I.equal``() =
  validate (NullableInt16s.Collection.ArrayArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ArrayArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.ArrayArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ArrayArray E.N.equals``() =
  validate (NullableInt16s.Collection.ArrayArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ArrayArray E.N.equal``() =
  validate (NullableInt16s.Collection.ArrayArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ArrayArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.ArrayArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ListArray E.I.equals``() =
  validate (NullableInt16s.Collection.ListArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ListArray E.I.equal``() =
  validate (NullableInt16s.Collection.ListArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ListArray E.I.not_equal``() =
  validate (NullableInt16s.Collection.ListArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ListArray E.N.equals``() =
  validate (NullableInt16s.Collection.ListArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ListArray E.N.equal``() =
  validate (NullableInt16s.Collection.ListArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt16s.Collection.ListArray E.N.not_equal``() =
  validate (NullableInt16s.Collection.ListArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.equals``() =
  validate (Int32s.Collection.Array) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.equal``() =
  validate (Int32s.Collection.Array) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.not_equal``() =
  validate (Int32s.Collection.Array) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.compare``() =
  validate (Int32s.Collection.Array) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.less_than``() =
  validate (Int32s.Collection.Array) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.less_or_equal``() =
  validate (Int32s.Collection.Array) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.greater_than``() =
  validate (Int32s.Collection.Array) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.I.greater_or_equal``() =
  validate (Int32s.Collection.Array) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.equals``() =
  validate (Int32s.Collection.Array) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.equal``() =
  validate (Int32s.Collection.Array) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.not_equal``() =
  validate (Int32s.Collection.Array) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.compare``() =
  validate (Int32s.Collection.Array) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.less_than``() =
  validate (Int32s.Collection.Array) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.less_or_equal``() =
  validate (Int32s.Collection.Array) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.greater_than``() =
  validate (Int32s.Collection.Array) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.Array C.N.greater_or_equal``() =
  validate (Int32s.Collection.Array) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.equals``() =
  validate (Int32s.Collection.OptionArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.equal``() =
  validate (Int32s.Collection.OptionArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.not_equal``() =
  validate (Int32s.Collection.OptionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.compare``() =
  validate (Int32s.Collection.OptionArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;-1;-1;-1;1;1;0;1;1;1;1;1;-1;0;-1;-1;1;1;-1;1;0;-1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.less_than``() =
  validate (Int32s.Collection.OptionArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.less_or_equal``() =
  validate (Int32s.Collection.OptionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;0;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.greater_than``() =
  validate (Int32s.Collection.OptionArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;1;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.OptionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.equals``() =
  validate (Int32s.Collection.OptionArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.equal``() =
  validate (Int32s.Collection.OptionArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.not_equal``() =
  validate (Int32s.Collection.OptionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.compare``() =
  validate (Int32s.Collection.OptionArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;-1;-1;-1;1;1;0;1;1;1;1;1;-1;0;-1;-1;1;1;-1;1;0;-1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.less_than``() =
  validate (Int32s.Collection.OptionArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.less_or_equal``() =
  validate (Int32s.Collection.OptionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;0;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.greater_than``() =
  validate (Int32s.Collection.OptionArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;1;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.OptionArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.OptionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.equals``() =
  validate (Int32s.Collection.RefArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.equal``() =
  validate (Int32s.Collection.RefArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.not_equal``() =
  validate (Int32s.Collection.RefArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.compare``() =
  validate (Int32s.Collection.RefArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.less_than``() =
  validate (Int32s.Collection.RefArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.less_or_equal``() =
  validate (Int32s.Collection.RefArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.greater_than``() =
  validate (Int32s.Collection.RefArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.RefArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.equals``() =
  validate (Int32s.Collection.RefArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.equal``() =
  validate (Int32s.Collection.RefArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.not_equal``() =
  validate (Int32s.Collection.RefArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.compare``() =
  validate (Int32s.Collection.RefArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.less_than``() =
  validate (Int32s.Collection.RefArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.less_or_equal``() =
  validate (Int32s.Collection.RefArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.greater_than``() =
  validate (Int32s.Collection.RefArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.RefArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.equals``() =
  validate (Int32s.Collection.RefWrapArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.equal``() =
  validate (Int32s.Collection.RefWrapArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.not_equal``() =
  validate (Int32s.Collection.RefWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.compare``() =
  validate (Int32s.Collection.RefWrapArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.less_than``() =
  validate (Int32s.Collection.RefWrapArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.less_or_equal``() =
  validate (Int32s.Collection.RefWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.greater_than``() =
  validate (Int32s.Collection.RefWrapArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.RefWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.equals``() =
  validate (Int32s.Collection.RefWrapArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.equal``() =
  validate (Int32s.Collection.RefWrapArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.not_equal``() =
  validate (Int32s.Collection.RefWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.compare``() =
  validate (Int32s.Collection.RefWrapArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.less_than``() =
  validate (Int32s.Collection.RefWrapArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.less_or_equal``() =
  validate (Int32s.Collection.RefWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.greater_than``() =
  validate (Int32s.Collection.RefWrapArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.RefWrapArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.RefWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.equals``() =
  validate (Int32s.Collection.UnionArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.equal``() =
  validate (Int32s.Collection.UnionArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.not_equal``() =
  validate (Int32s.Collection.UnionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.compare``() =
  validate (Int32s.Collection.UnionArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.less_than``() =
  validate (Int32s.Collection.UnionArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.less_or_equal``() =
  validate (Int32s.Collection.UnionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.greater_than``() =
  validate (Int32s.Collection.UnionArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.UnionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.equals``() =
  validate (Int32s.Collection.UnionArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.equal``() =
  validate (Int32s.Collection.UnionArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.not_equal``() =
  validate (Int32s.Collection.UnionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.compare``() =
  validate (Int32s.Collection.UnionArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.less_than``() =
  validate (Int32s.Collection.UnionArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.less_or_equal``() =
  validate (Int32s.Collection.UnionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.greater_than``() =
  validate (Int32s.Collection.UnionArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.UnionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.equals``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.not_equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.compare``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.less_than``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.less_or_equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.greater_than``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.equals``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.not_equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.compare``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.less_than``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.less_or_equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.greater_than``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.UnionWrapArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.UnionWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.equals``() =
  validate (Int32s.Collection.ValueArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.equal``() =
  validate (Int32s.Collection.ValueArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.not_equal``() =
  validate (Int32s.Collection.ValueArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.compare``() =
  validate (Int32s.Collection.ValueArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.less_than``() =
  validate (Int32s.Collection.ValueArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.less_or_equal``() =
  validate (Int32s.Collection.ValueArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.greater_than``() =
  validate (Int32s.Collection.ValueArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.ValueArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.equals``() =
  validate (Int32s.Collection.ValueArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.equal``() =
  validate (Int32s.Collection.ValueArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.not_equal``() =
  validate (Int32s.Collection.ValueArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.compare``() =
  validate (Int32s.Collection.ValueArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.less_than``() =
  validate (Int32s.Collection.ValueArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.less_or_equal``() =
  validate (Int32s.Collection.ValueArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.greater_than``() =
  validate (Int32s.Collection.ValueArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.ValueArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.equals``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.not_equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.compare``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.less_than``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.less_or_equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.greater_than``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.equals``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.not_equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.compare``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.less_than``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.less_or_equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.greater_than``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ValueWrapArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.ValueWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.equals``() =
  validate (Int32s.Collection.ArrayArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.equal``() =
  validate (Int32s.Collection.ArrayArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.not_equal``() =
  validate (Int32s.Collection.ArrayArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.compare``() =
  validate (Int32s.Collection.ArrayArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;-1;-1;-1;-1;-1;1;-1;0;-1;-1;-1;-1;-1;-1;-1;1;-1;1;0;-1;-1;-1;-1;-1;-1;
   1;-1;1;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;
   1;1;1;1;1;1;-1;1;0;-1;1;1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.less_than``() =
  validate (Int32s.Collection.ArrayArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;
   0;1;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;
   0;0;0;0;0;0;1;0;0;1;0;0;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.less_or_equal``() =
  validate (Int32s.Collection.ArrayArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;
   0;1;0;0;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;
   0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.greater_than``() =
  validate (Int32s.Collection.ArrayArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;
   1;0;1;1;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;
   1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.ArrayArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;
   1;0;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;
   1;1;1;1;1;1;0;1;1;0;1;1;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.equals``() =
  validate (Int32s.Collection.ArrayArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.equal``() =
  validate (Int32s.Collection.ArrayArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.not_equal``() =
  validate (Int32s.Collection.ArrayArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.compare``() =
  validate (Int32s.Collection.ArrayArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;-1;-1;-1;-1;-1;1;-1;0;-1;-1;-1;-1;-1;-1;-1;1;-1;1;0;-1;-1;-1;-1;-1;-1;
   1;-1;1;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;
   1;1;1;1;1;1;-1;1;0;-1;1;1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.less_than``() =
  validate (Int32s.Collection.ArrayArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;
   0;1;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;
   0;0;0;0;0;0;1;0;0;1;0;0;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.less_or_equal``() =
  validate (Int32s.Collection.ArrayArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;
   0;1;0;0;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;
   0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.greater_than``() =
  validate (Int32s.Collection.ArrayArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;
   1;0;1;1;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;
   1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.ArrayArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;
   1;0;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;
   1;1;1;1;1;1;0;1;1;0;1;1;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.equals``() =
  validate (Int32s.Collection.ListArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.equal``() =
  validate (Int32s.Collection.ListArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.not_equal``() =
  validate (Int32s.Collection.ListArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.compare``() =
  validate (Int32s.Collection.ListArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;-1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;-1;1;-1;1;0;-1;1;-1;1;-1;-1;
   1;-1;1;1;0;1;-1;1;1;-1;1;-1;-1;-1;-1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;-1;1;-1;-1;1;-1;0;-1;-1;
   1;-1;1;1;-1;1;-1;1;0;-1;1;-1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.less_than``() =
  validate (Int32s.Collection.ListArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;0;1;0;0;1;0;1;0;1;1;
   0;1;0;0;0;0;1;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.less_or_equal``() =
  validate (Int32s.Collection.ListArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;
   0;1;0;0;1;0;1;0;1;1;0;1;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.greater_than``() =
  validate (Int32s.Collection.ListArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;
   1;0;1;1;0;1;0;1;0;0;1;0;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.ListArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;1;0;1;1;0;1;0;1;0;0;
   1;0;1;1;1;1;0;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.equals``() =
  validate (Int32s.Collection.ListArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.equal``() =
  validate (Int32s.Collection.ListArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.not_equal``() =
  validate (Int32s.Collection.ListArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.compare``() =
  validate (Int32s.Collection.ListArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;-1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;-1;1;-1;1;0;-1;1;-1;1;-1;-1;
   1;-1;1;1;0;1;-1;1;1;-1;1;-1;-1;-1;-1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;-1;1;-1;-1;1;-1;0;-1;-1;
   1;-1;1;1;-1;1;-1;1;0;-1;1;-1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.less_than``() =
  validate (Int32s.Collection.ListArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;0;1;0;0;1;0;1;0;1;1;
   0;1;0;0;0;0;1;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.less_or_equal``() =
  validate (Int32s.Collection.ListArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;
   0;1;0;0;1;0;1;0;1;1;0;1;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.greater_than``() =
  validate (Int32s.Collection.ListArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;
   1;0;1;1;0;1;0;1;0;0;1;0;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ListArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.ListArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;1;0;1;1;0;1;0;1;0;0;
   1;0;1;1;1;1;0;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.equals``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.not_equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.compare``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;1;1;-1;1;0;-1;1;1;1;-1;1;
   1;-1;1;1;0;1;1;1;1;1;1;-1;-1;-1;-1;0;-1;-1;-1;1;1;-1;1;-1;-1;1;0;1;-1;1;1;-1;1;-1;-1;1;-1;0;-1;1;
   1;-1;1;1;-1;1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_than``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;1;1;0;0;1;0;0;1;0;0;0;1;0;
   0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;0;1;0;1;1;0;1;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_or_equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0;0;1;0;1;1;0;1;0;1;0;0;1;0;1;1;0;1;1;1;0;
   0;1;0;0;1;0;0;0;1;0;0;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_than``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1;1;0;1;0;0;1;0;1;0;1;1;0;1;0;0;1;0;0;0;1;
   1;0;1;1;0;1;1;1;0;1;1;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_or_equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;0;0;1;1;0;1;1;0;1;1;1;0;1;
   1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;1;0;1;0;0;1;0;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.equals``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.not_equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.compare``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;1;1;-1;1;0;-1;1;1;1;-1;1;
   1;-1;1;1;0;1;1;1;1;1;1;-1;-1;-1;-1;0;-1;-1;-1;1;1;-1;1;-1;-1;1;0;1;-1;1;1;-1;1;-1;-1;1;-1;0;-1;1;
   1;-1;1;1;-1;1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_than``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;1;1;0;0;1;0;0;1;0;0;0;1;0;
   0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;0;1;0;1;1;0;1;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_or_equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0;0;1;0;1;1;0;1;0;1;0;0;1;0;1;1;0;1;1;1;0;
   0;1;0;0;1;0;0;0;1;0;0;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_than``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1;1;0;1;0;0;1;0;1;0;1;1;0;1;0;0;1;0;0;0;1;
   1;0;1;1;0;1;1;1;0;1;1;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int32s.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_or_equal``() =
  validate (Int32s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;0;0;1;1;0;1;1;0;1;1;1;0;1;
   1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;1;0;1;0;0;1;0;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.Array E.I.equals``() =
  validate (NullableInt32s.Collection.Array) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.Array E.I.equal``() =
  validate (NullableInt32s.Collection.Array) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.Array E.I.not_equal``() =
  validate (NullableInt32s.Collection.Array) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.Array E.N.equals``() =
  validate (NullableInt32s.Collection.Array) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.Array E.N.equal``() =
  validate (NullableInt32s.Collection.Array) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.Array E.N.not_equal``() =
  validate (NullableInt32s.Collection.Array) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.OptionArray E.I.equals``() =
  validate (NullableInt32s.Collection.OptionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.OptionArray E.I.equal``() =
  validate (NullableInt32s.Collection.OptionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.OptionArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.OptionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.OptionArray E.N.equals``() =
  validate (NullableInt32s.Collection.OptionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.OptionArray E.N.equal``() =
  validate (NullableInt32s.Collection.OptionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.OptionArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.OptionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefArray E.I.equals``() =
  validate (NullableInt32s.Collection.RefArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefArray E.I.equal``() =
  validate (NullableInt32s.Collection.RefArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.RefArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefArray E.N.equals``() =
  validate (NullableInt32s.Collection.RefArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefArray E.N.equal``() =
  validate (NullableInt32s.Collection.RefArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.RefArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefWrapArray E.I.equals``() =
  validate (NullableInt32s.Collection.RefWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefWrapArray E.I.equal``() =
  validate (NullableInt32s.Collection.RefWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefWrapArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.RefWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefWrapArray E.N.equals``() =
  validate (NullableInt32s.Collection.RefWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefWrapArray E.N.equal``() =
  validate (NullableInt32s.Collection.RefWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.RefWrapArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.RefWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionArray E.I.equals``() =
  validate (NullableInt32s.Collection.UnionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionArray E.I.equal``() =
  validate (NullableInt32s.Collection.UnionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.UnionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionArray E.N.equals``() =
  validate (NullableInt32s.Collection.UnionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionArray E.N.equal``() =
  validate (NullableInt32s.Collection.UnionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.UnionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionWrapArray E.I.equals``() =
  validate (NullableInt32s.Collection.UnionWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionWrapArray E.I.equal``() =
  validate (NullableInt32s.Collection.UnionWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionWrapArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.UnionWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionWrapArray E.N.equals``() =
  validate (NullableInt32s.Collection.UnionWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionWrapArray E.N.equal``() =
  validate (NullableInt32s.Collection.UnionWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.UnionWrapArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.UnionWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueArray E.I.equals``() =
  validate (NullableInt32s.Collection.ValueArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueArray E.I.equal``() =
  validate (NullableInt32s.Collection.ValueArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.ValueArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueArray E.N.equals``() =
  validate (NullableInt32s.Collection.ValueArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueArray E.N.equal``() =
  validate (NullableInt32s.Collection.ValueArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.ValueArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueWrapArray E.I.equals``() =
  validate (NullableInt32s.Collection.ValueWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueWrapArray E.I.equal``() =
  validate (NullableInt32s.Collection.ValueWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueWrapArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.ValueWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueWrapArray E.N.equals``() =
  validate (NullableInt32s.Collection.ValueWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueWrapArray E.N.equal``() =
  validate (NullableInt32s.Collection.ValueWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ValueWrapArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.ValueWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ArrayArray E.I.equals``() =
  validate (NullableInt32s.Collection.ArrayArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ArrayArray E.I.equal``() =
  validate (NullableInt32s.Collection.ArrayArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ArrayArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.ArrayArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ArrayArray E.N.equals``() =
  validate (NullableInt32s.Collection.ArrayArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ArrayArray E.N.equal``() =
  validate (NullableInt32s.Collection.ArrayArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ArrayArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.ArrayArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ListArray E.I.equals``() =
  validate (NullableInt32s.Collection.ListArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ListArray E.I.equal``() =
  validate (NullableInt32s.Collection.ListArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ListArray E.I.not_equal``() =
  validate (NullableInt32s.Collection.ListArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ListArray E.N.equals``() =
  validate (NullableInt32s.Collection.ListArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ListArray E.N.equal``() =
  validate (NullableInt32s.Collection.ListArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt32s.Collection.ListArray E.N.not_equal``() =
  validate (NullableInt32s.Collection.ListArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.equals``() =
  validate (Int64s.Collection.Array) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.equal``() =
  validate (Int64s.Collection.Array) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.not_equal``() =
  validate (Int64s.Collection.Array) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.compare``() =
  validate (Int64s.Collection.Array) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.less_than``() =
  validate (Int64s.Collection.Array) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.less_or_equal``() =
  validate (Int64s.Collection.Array) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.greater_than``() =
  validate (Int64s.Collection.Array) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.I.greater_or_equal``() =
  validate (Int64s.Collection.Array) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.equals``() =
  validate (Int64s.Collection.Array) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.equal``() =
  validate (Int64s.Collection.Array) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.not_equal``() =
  validate (Int64s.Collection.Array) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.compare``() =
  validate (Int64s.Collection.Array) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.less_than``() =
  validate (Int64s.Collection.Array) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.less_or_equal``() =
  validate (Int64s.Collection.Array) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.greater_than``() =
  validate (Int64s.Collection.Array) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.Array C.N.greater_or_equal``() =
  validate (Int64s.Collection.Array) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.equals``() =
  validate (Int64s.Collection.OptionArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.equal``() =
  validate (Int64s.Collection.OptionArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.not_equal``() =
  validate (Int64s.Collection.OptionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.compare``() =
  validate (Int64s.Collection.OptionArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;-1;-1;-1;1;1;0;1;1;1;1;1;-1;0;-1;-1;1;1;-1;1;0;-1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.less_than``() =
  validate (Int64s.Collection.OptionArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.less_or_equal``() =
  validate (Int64s.Collection.OptionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;0;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.greater_than``() =
  validate (Int64s.Collection.OptionArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;1;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.OptionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.equals``() =
  validate (Int64s.Collection.OptionArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.equal``() =
  validate (Int64s.Collection.OptionArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.not_equal``() =
  validate (Int64s.Collection.OptionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.compare``() =
  validate (Int64s.Collection.OptionArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;-1;-1;-1;1;1;0;1;1;1;1;1;-1;0;-1;-1;1;1;-1;1;0;-1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.less_than``() =
  validate (Int64s.Collection.OptionArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.less_or_equal``() =
  validate (Int64s.Collection.OptionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;0;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.greater_than``() =
  validate (Int64s.Collection.OptionArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;1;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.OptionArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.OptionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.equals``() =
  validate (Int64s.Collection.RefArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.equal``() =
  validate (Int64s.Collection.RefArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.not_equal``() =
  validate (Int64s.Collection.RefArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.compare``() =
  validate (Int64s.Collection.RefArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.less_than``() =
  validate (Int64s.Collection.RefArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.less_or_equal``() =
  validate (Int64s.Collection.RefArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.greater_than``() =
  validate (Int64s.Collection.RefArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.RefArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.equals``() =
  validate (Int64s.Collection.RefArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.equal``() =
  validate (Int64s.Collection.RefArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.not_equal``() =
  validate (Int64s.Collection.RefArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.compare``() =
  validate (Int64s.Collection.RefArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.less_than``() =
  validate (Int64s.Collection.RefArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.less_or_equal``() =
  validate (Int64s.Collection.RefArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.greater_than``() =
  validate (Int64s.Collection.RefArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.RefArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.equals``() =
  validate (Int64s.Collection.RefWrapArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.equal``() =
  validate (Int64s.Collection.RefWrapArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.not_equal``() =
  validate (Int64s.Collection.RefWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.compare``() =
  validate (Int64s.Collection.RefWrapArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.less_than``() =
  validate (Int64s.Collection.RefWrapArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.less_or_equal``() =
  validate (Int64s.Collection.RefWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.greater_than``() =
  validate (Int64s.Collection.RefWrapArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.RefWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.equals``() =
  validate (Int64s.Collection.RefWrapArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.equal``() =
  validate (Int64s.Collection.RefWrapArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.not_equal``() =
  validate (Int64s.Collection.RefWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.compare``() =
  validate (Int64s.Collection.RefWrapArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.less_than``() =
  validate (Int64s.Collection.RefWrapArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.less_or_equal``() =
  validate (Int64s.Collection.RefWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.greater_than``() =
  validate (Int64s.Collection.RefWrapArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.RefWrapArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.RefWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.equals``() =
  validate (Int64s.Collection.UnionArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.equal``() =
  validate (Int64s.Collection.UnionArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.not_equal``() =
  validate (Int64s.Collection.UnionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.compare``() =
  validate (Int64s.Collection.UnionArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.less_than``() =
  validate (Int64s.Collection.UnionArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.less_or_equal``() =
  validate (Int64s.Collection.UnionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.greater_than``() =
  validate (Int64s.Collection.UnionArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.UnionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.equals``() =
  validate (Int64s.Collection.UnionArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.equal``() =
  validate (Int64s.Collection.UnionArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.not_equal``() =
  validate (Int64s.Collection.UnionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.compare``() =
  validate (Int64s.Collection.UnionArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.less_than``() =
  validate (Int64s.Collection.UnionArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.less_or_equal``() =
  validate (Int64s.Collection.UnionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.greater_than``() =
  validate (Int64s.Collection.UnionArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.UnionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.equals``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.not_equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.compare``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.less_than``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.less_or_equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.greater_than``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.equals``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.not_equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.compare``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.less_than``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.less_or_equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.greater_than``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.UnionWrapArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.UnionWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.equals``() =
  validate (Int64s.Collection.ValueArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.equal``() =
  validate (Int64s.Collection.ValueArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.not_equal``() =
  validate (Int64s.Collection.ValueArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.compare``() =
  validate (Int64s.Collection.ValueArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.less_than``() =
  validate (Int64s.Collection.ValueArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.less_or_equal``() =
  validate (Int64s.Collection.ValueArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.greater_than``() =
  validate (Int64s.Collection.ValueArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.ValueArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.equals``() =
  validate (Int64s.Collection.ValueArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.equal``() =
  validate (Int64s.Collection.ValueArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.not_equal``() =
  validate (Int64s.Collection.ValueArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.compare``() =
  validate (Int64s.Collection.ValueArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.less_than``() =
  validate (Int64s.Collection.ValueArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.less_or_equal``() =
  validate (Int64s.Collection.ValueArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.greater_than``() =
  validate (Int64s.Collection.ValueArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.ValueArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.equals``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.not_equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.compare``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.less_than``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.less_or_equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.greater_than``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.equals``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.not_equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.compare``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.compare [|
   0;-1;-1;-1;-1;1;0;1;1;1;1;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.less_than``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.less_than [|
   0;1;1;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.less_or_equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;0;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.greater_than``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;1;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ValueWrapArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.ValueWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.equals``() =
  validate (Int64s.Collection.ArrayArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.equal``() =
  validate (Int64s.Collection.ArrayArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.not_equal``() =
  validate (Int64s.Collection.ArrayArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.compare``() =
  validate (Int64s.Collection.ArrayArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;-1;-1;-1;-1;-1;1;-1;0;-1;-1;-1;-1;-1;-1;-1;1;-1;1;0;-1;-1;-1;-1;-1;-1;
   1;-1;1;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;
   1;1;1;1;1;1;-1;1;0;-1;1;1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.less_than``() =
  validate (Int64s.Collection.ArrayArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;
   0;1;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;
   0;0;0;0;0;0;1;0;0;1;0;0;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.less_or_equal``() =
  validate (Int64s.Collection.ArrayArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;
   0;1;0;0;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;
   0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.greater_than``() =
  validate (Int64s.Collection.ArrayArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;
   1;0;1;1;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;
   1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.ArrayArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;
   1;0;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;
   1;1;1;1;1;1;0;1;1;0;1;1;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.equals``() =
  validate (Int64s.Collection.ArrayArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.equal``() =
  validate (Int64s.Collection.ArrayArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.not_equal``() =
  validate (Int64s.Collection.ArrayArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.compare``() =
  validate (Int64s.Collection.ArrayArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;-1;-1;-1;-1;-1;1;-1;0;-1;-1;-1;-1;-1;-1;-1;1;-1;1;0;-1;-1;-1;-1;-1;-1;
   1;-1;1;1;0;-1;-1;-1;-1;-1;1;1;1;1;1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;
   1;1;1;1;1;1;-1;1;0;-1;1;1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.less_than``() =
  validate (Int64s.Collection.ArrayArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;
   0;1;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;
   0;0;0;0;0;0;1;0;0;1;0;0;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.less_or_equal``() =
  validate (Int64s.Collection.ArrayArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;1;0;1;1;1;1;1;1;1;
   0;1;0;0;1;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;
   0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.greater_than``() =
  validate (Int64s.Collection.ArrayArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;
   1;0;1;1;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;
   1;1;1;1;1;1;0;1;0;0;1;1;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.ArrayArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;0;0;0;0;0;1;0;1;0;0;0;0;0;0;0;1;0;1;1;0;0;0;0;0;0;
   1;0;1;1;1;0;0;0;0;0;1;1;1;1;1;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;
   1;1;1;1;1;1;0;1;1;0;1;1;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.equals``() =
  validate (Int64s.Collection.ListArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.equal``() =
  validate (Int64s.Collection.ListArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.not_equal``() =
  validate (Int64s.Collection.ListArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.compare``() =
  validate (Int64s.Collection.ListArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;-1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;-1;1;-1;1;0;-1;1;-1;1;-1;-1;
   1;-1;1;1;0;1;-1;1;1;-1;1;-1;-1;-1;-1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;-1;1;-1;-1;1;-1;0;-1;-1;
   1;-1;1;1;-1;1;-1;1;0;-1;1;-1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.less_than``() =
  validate (Int64s.Collection.ListArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;0;1;0;0;1;0;1;0;1;1;
   0;1;0;0;0;0;1;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.less_or_equal``() =
  validate (Int64s.Collection.ListArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;
   0;1;0;0;1;0;1;0;1;1;0;1;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.greater_than``() =
  validate (Int64s.Collection.ListArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;
   1;0;1;1;0;1;0;1;0;0;1;0;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.ListArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;1;0;1;1;0;1;0;1;0;0;
   1;0;1;1;1;1;0;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.equals``() =
  validate (Int64s.Collection.ListArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.equal``() =
  validate (Int64s.Collection.ListArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.not_equal``() =
  validate (Int64s.Collection.ListArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.compare``() =
  validate (Int64s.Collection.ListArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;-1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;-1;1;-1;1;0;-1;1;-1;1;-1;-1;
   1;-1;1;1;0;1;-1;1;1;-1;1;-1;-1;-1;-1;0;-1;-1;-1;-1;1;1;1;1;1;1;0;1;1;1;1;-1;1;-1;-1;1;-1;0;-1;-1;
   1;-1;1;1;-1;1;-1;1;0;-1;1;-1;1;1;1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.less_than``() =
  validate (Int64s.Collection.ListArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;0;1;0;0;1;0;1;0;1;1;
   0;1;0;0;0;0;1;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;0;0;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.less_or_equal``() =
  validate (Int64s.Collection.ListArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;0;1;0;1;1;
   0;1;0;0;1;0;1;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;1;0;0;0;0;1;0;1;1;0;1;1;1;1;
   0;1;0;0;1;0;1;0;1;1;0;1;0;0;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.greater_than``() =
  validate (Int64s.Collection.ListArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;0;1;1;1;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;
   1;0;1;1;0;1;0;1;0;0;1;0;1;1;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ListArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.ListArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;0;1;1;1;1;0;1;0;0;1;0;0;0;0;1;0;1;1;0;1;0;1;0;0;
   1;0;1;1;1;1;0;1;1;0;1;0;0;0;0;1;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;1;0;0;
   1;0;1;1;0;1;0;1;1;0;1;0;1;1;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.equals``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.not_equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.compare``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;1;1;-1;1;0;-1;1;1;1;-1;1;
   1;-1;1;1;0;1;1;1;1;1;1;-1;-1;-1;-1;0;-1;-1;-1;1;1;-1;1;-1;-1;1;0;1;-1;1;1;-1;1;-1;-1;1;-1;0;-1;1;
   1;-1;1;1;-1;1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_than``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;1;1;0;0;1;0;0;1;0;0;0;1;0;
   0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;0;1;0;1;1;0;1;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_or_equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0;0;1;0;1;1;0;1;0;1;0;0;1;0;1;1;0;1;1;1;0;
   0;1;0;0;1;0;0;0;1;0;0;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_than``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1;1;0;1;0;0;1;0;1;0;1;1;0;1;0;0;1;0;0;0;1;
   1;0;1;1;0;1;1;1;0;1;1;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_or_equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;0;0;1;1;0;1;1;0;1;1;1;0;1;
   1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;1;0;1;0;0;1;0;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.equals``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.not_equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.compare``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;-1;-1;-1;-1;1;0;1;1;1;1;1;1;1;1;1;-1;0;-1;-1;1;-1;-1;-1;1;1;-1;1;0;-1;1;1;1;-1;1;
   1;-1;1;1;0;1;1;1;1;1;1;-1;-1;-1;-1;0;-1;-1;-1;1;1;-1;1;-1;-1;1;0;1;-1;1;1;-1;1;-1;-1;1;-1;0;-1;1;
   1;-1;1;1;-1;1;1;1;0;1;1;-1;-1;-1;-1;-1;-1;-1;-1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_than``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;0;1;1;1;0;0;1;0;0;1;0;0;0;1;0;
   0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;0;1;0;1;1;0;1;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_or_equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;1;1;1;1;0;1;1;1;0;0;1;0;1;1;0;0;0;1;0;
   0;1;0;0;1;0;0;0;0;0;0;1;1;1;1;1;1;1;1;0;0;1;0;1;1;0;1;0;1;0;0;1;0;1;1;0;1;1;1;0;
   0;1;0;0;1;0;0;0;1;0;0;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_than``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1;1;0;1;0;0;1;0;1;0;1;1;0;1;0;0;1;0;0;0;1;
   1;0;1;1;0;1;1;1;0;1;1;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Int64s.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_or_equal``() =
  validate (Int64s.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;1;0;0;0;1;1;0;1;1;0;1;1;1;0;1;
   1;0;1;1;1;1;1;1;1;1;1;0;0;0;0;1;0;0;0;1;1;0;1;0;0;1;1;1;0;1;1;0;1;0;0;1;0;1;0;1;
   1;0;1;1;0;1;1;1;1;1;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.Array E.I.equals``() =
  validate (NullableInt64s.Collection.Array) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.Array E.I.equal``() =
  validate (NullableInt64s.Collection.Array) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.Array E.I.not_equal``() =
  validate (NullableInt64s.Collection.Array) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.Array E.N.equals``() =
  validate (NullableInt64s.Collection.Array) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.Array E.N.equal``() =
  validate (NullableInt64s.Collection.Array) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.Array E.N.not_equal``() =
  validate (NullableInt64s.Collection.Array) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.OptionArray E.I.equals``() =
  validate (NullableInt64s.Collection.OptionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.OptionArray E.I.equal``() =
  validate (NullableInt64s.Collection.OptionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.OptionArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.OptionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.OptionArray E.N.equals``() =
  validate (NullableInt64s.Collection.OptionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.OptionArray E.N.equal``() =
  validate (NullableInt64s.Collection.OptionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.OptionArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.OptionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefArray E.I.equals``() =
  validate (NullableInt64s.Collection.RefArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefArray E.I.equal``() =
  validate (NullableInt64s.Collection.RefArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.RefArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefArray E.N.equals``() =
  validate (NullableInt64s.Collection.RefArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefArray E.N.equal``() =
  validate (NullableInt64s.Collection.RefArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.RefArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefWrapArray E.I.equals``() =
  validate (NullableInt64s.Collection.RefWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefWrapArray E.I.equal``() =
  validate (NullableInt64s.Collection.RefWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefWrapArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.RefWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefWrapArray E.N.equals``() =
  validate (NullableInt64s.Collection.RefWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefWrapArray E.N.equal``() =
  validate (NullableInt64s.Collection.RefWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.RefWrapArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.RefWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionArray E.I.equals``() =
  validate (NullableInt64s.Collection.UnionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionArray E.I.equal``() =
  validate (NullableInt64s.Collection.UnionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.UnionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionArray E.N.equals``() =
  validate (NullableInt64s.Collection.UnionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionArray E.N.equal``() =
  validate (NullableInt64s.Collection.UnionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.UnionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionWrapArray E.I.equals``() =
  validate (NullableInt64s.Collection.UnionWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionWrapArray E.I.equal``() =
  validate (NullableInt64s.Collection.UnionWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionWrapArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.UnionWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionWrapArray E.N.equals``() =
  validate (NullableInt64s.Collection.UnionWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionWrapArray E.N.equal``() =
  validate (NullableInt64s.Collection.UnionWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.UnionWrapArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.UnionWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueArray E.I.equals``() =
  validate (NullableInt64s.Collection.ValueArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueArray E.I.equal``() =
  validate (NullableInt64s.Collection.ValueArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.ValueArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueArray E.N.equals``() =
  validate (NullableInt64s.Collection.ValueArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueArray E.N.equal``() =
  validate (NullableInt64s.Collection.ValueArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.ValueArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueWrapArray E.I.equals``() =
  validate (NullableInt64s.Collection.ValueWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueWrapArray E.I.equal``() =
  validate (NullableInt64s.Collection.ValueWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueWrapArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.ValueWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueWrapArray E.N.equals``() =
  validate (NullableInt64s.Collection.ValueWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueWrapArray E.N.equal``() =
  validate (NullableInt64s.Collection.ValueWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ValueWrapArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.ValueWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ArrayArray E.I.equals``() =
  validate (NullableInt64s.Collection.ArrayArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ArrayArray E.I.equal``() =
  validate (NullableInt64s.Collection.ArrayArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ArrayArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.ArrayArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ArrayArray E.N.equals``() =
  validate (NullableInt64s.Collection.ArrayArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ArrayArray E.N.equal``() =
  validate (NullableInt64s.Collection.ArrayArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ArrayArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.ArrayArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ListArray E.I.equals``() =
  validate (NullableInt64s.Collection.ListArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ListArray E.I.equal``() =
  validate (NullableInt64s.Collection.ListArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ListArray E.I.not_equal``() =
  validate (NullableInt64s.Collection.ListArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ListArray E.N.equals``() =
  validate (NullableInt64s.Collection.ListArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ListArray E.N.equal``() =
  validate (NullableInt64s.Collection.ListArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableInt64s.Collection.ListArray E.N.not_equal``() =
  validate (NullableInt64s.Collection.ListArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.equals``() =
  validate (NativeInts.Collection.Array) C.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.equal``() =
  validate (NativeInts.Collection.Array) C.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.not_equal``() =
  validate (NativeInts.Collection.Array) C.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.compare``() =
  validate (NativeInts.Collection.Array) C.I.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.less_than``() =
  validate (NativeInts.Collection.Array) C.I.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.less_or_equal``() =
  validate (NativeInts.Collection.Array) C.I.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.greater_than``() =
  validate (NativeInts.Collection.Array) C.I.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.I.greater_or_equal``() =
  validate (NativeInts.Collection.Array) C.I.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.equals``() =
  validate (NativeInts.Collection.Array) C.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.equal``() =
  validate (NativeInts.Collection.Array) C.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.not_equal``() =
  validate (NativeInts.Collection.Array) C.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.compare``() =
  validate (NativeInts.Collection.Array) C.N.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.less_than``() =
  validate (NativeInts.Collection.Array) C.N.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.less_or_equal``() =
  validate (NativeInts.Collection.Array) C.N.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.greater_than``() =
  validate (NativeInts.Collection.Array) C.N.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.Array C.N.greater_or_equal``() =
  validate (NativeInts.Collection.Array) C.N.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.equals``() =
  validate (NativeInts.Collection.OptionArray) C.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.equal``() =
  validate (NativeInts.Collection.OptionArray) C.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.not_equal``() =
  validate (NativeInts.Collection.OptionArray) C.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.compare``() =
  validate (NativeInts.Collection.OptionArray) C.I.compare [|
   0;-1;-1;-1;1;0;-1;-1;1;1;0;-1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.less_than``() =
  validate (NativeInts.Collection.OptionArray) C.I.less_than [|
   0;1;1;1;0;0;1;1;0;0;0;1;0;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.OptionArray) C.I.less_or_equal [|
   1;1;1;1;0;1;1;1;0;0;1;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.greater_than``() =
  validate (NativeInts.Collection.OptionArray) C.I.greater_than [|
   0;0;0;0;1;0;0;0;1;1;0;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.OptionArray) C.I.greater_or_equal [|
   1;0;0;0;1;1;0;0;1;1;1;0;1;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.equals``() =
  validate (NativeInts.Collection.OptionArray) C.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.equal``() =
  validate (NativeInts.Collection.OptionArray) C.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.not_equal``() =
  validate (NativeInts.Collection.OptionArray) C.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.compare``() =
  validate (NativeInts.Collection.OptionArray) C.N.compare [|
   0;-1;-1;-1;1;0;-1;-1;1;1;0;-1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.less_than``() =
  validate (NativeInts.Collection.OptionArray) C.N.less_than [|
   0;1;1;1;0;0;1;1;0;0;0;1;0;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.OptionArray) C.N.less_or_equal [|
   1;1;1;1;0;1;1;1;0;0;1;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.greater_than``() =
  validate (NativeInts.Collection.OptionArray) C.N.greater_than [|
   0;0;0;0;1;0;0;0;1;1;0;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.OptionArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.OptionArray) C.N.greater_or_equal [|
   1;0;0;0;1;1;0;0;1;1;1;0;1;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.equals``() =
  validate (NativeInts.Collection.RefArray) C.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.equal``() =
  validate (NativeInts.Collection.RefArray) C.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.not_equal``() =
  validate (NativeInts.Collection.RefArray) C.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.compare``() =
  validate (NativeInts.Collection.RefArray) C.I.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.less_than``() =
  validate (NativeInts.Collection.RefArray) C.I.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.RefArray) C.I.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.greater_than``() =
  validate (NativeInts.Collection.RefArray) C.I.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.RefArray) C.I.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.equals``() =
  validate (NativeInts.Collection.RefArray) C.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.equal``() =
  validate (NativeInts.Collection.RefArray) C.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.not_equal``() =
  validate (NativeInts.Collection.RefArray) C.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.compare``() =
  validate (NativeInts.Collection.RefArray) C.N.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.less_than``() =
  validate (NativeInts.Collection.RefArray) C.N.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.RefArray) C.N.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.greater_than``() =
  validate (NativeInts.Collection.RefArray) C.N.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.RefArray) C.N.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.equals``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.not_equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.compare``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.less_than``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.greater_than``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.I.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.equals``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.not_equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.compare``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.less_than``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.greater_than``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.RefWrapArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.RefWrapArray) C.N.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.equals``() =
  validate (NativeInts.Collection.UnionArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.equal``() =
  validate (NativeInts.Collection.UnionArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.not_equal``() =
  validate (NativeInts.Collection.UnionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.compare``() =
  validate (NativeInts.Collection.UnionArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;
   -2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;
   -1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;
   -3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;0;-1;
   -1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;
   1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;
   3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;
   -2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;
   1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.less_than``() =
  validate (NativeInts.Collection.UnionArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.UnionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.greater_than``() =
  validate (NativeInts.Collection.UnionArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.UnionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.equals``() =
  validate (NativeInts.Collection.UnionArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.equal``() =
  validate (NativeInts.Collection.UnionArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.not_equal``() =
  validate (NativeInts.Collection.UnionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.compare``() =
  validate (NativeInts.Collection.UnionArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;
   -2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;
   -1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;
   -3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;0;-1;
   -1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;
   1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;
   3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;
   -2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;
   1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.less_than``() =
  validate (NativeInts.Collection.UnionArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.UnionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.greater_than``() =
  validate (NativeInts.Collection.UnionArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.UnionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.equals``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.not_equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.compare``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;
   -2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;
   -1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;
   -3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;0;-1;
   -1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;
   1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;
   3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;
   -2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;
   1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.less_than``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.greater_than``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.equals``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.not_equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.compare``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;
   -2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;
   -1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;
   -3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;0;-1;
   -1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;
   1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;
   3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;
   -2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;3;2;1;
   1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.less_than``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.greater_than``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0
  |]

 [<Test>]
 member __.``NativeInts.Collection.UnionWrapArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.UnionWrapArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.equals``() =
  validate (NativeInts.Collection.ValueArray) C.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.equal``() =
  validate (NativeInts.Collection.ValueArray) C.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.not_equal``() =
  validate (NativeInts.Collection.ValueArray) C.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.compare``() =
  validate (NativeInts.Collection.ValueArray) C.I.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.less_than``() =
  validate (NativeInts.Collection.ValueArray) C.I.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.ValueArray) C.I.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.greater_than``() =
  validate (NativeInts.Collection.ValueArray) C.I.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.ValueArray) C.I.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.equals``() =
  validate (NativeInts.Collection.ValueArray) C.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.equal``() =
  validate (NativeInts.Collection.ValueArray) C.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.not_equal``() =
  validate (NativeInts.Collection.ValueArray) C.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.compare``() =
  validate (NativeInts.Collection.ValueArray) C.N.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.less_than``() =
  validate (NativeInts.Collection.ValueArray) C.N.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.ValueArray) C.N.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.greater_than``() =
  validate (NativeInts.Collection.ValueArray) C.N.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.ValueArray) C.N.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.equals``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.not_equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.compare``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.less_than``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.greater_than``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.I.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.equals``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.equals [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.equal [|
   1;0;0;0;1;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.not_equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.not_equal [|
   0;1;1;1;0;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.compare``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.compare [|
   0;-1;-1;1;0;-1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.less_than``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.less_than [|
   0;1;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.less_or_equal [|
   1;1;1;0;1;1;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.greater_than``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.greater_than [|
   0;0;0;1;0;0;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ValueWrapArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.ValueWrapArray) C.N.greater_or_equal [|
   1;0;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.equals``() =
  validate (NativeInts.Collection.ArrayArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.equal``() =
  validate (NativeInts.Collection.ArrayArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.not_equal``() =
  validate (NativeInts.Collection.ArrayArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.compare``() =
  validate (NativeInts.Collection.ArrayArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;-1;-1;-1;1;1;0;-1;-1;-1;1;1;1;0;-1;-1;1;1;1;1;0;-1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.less_than``() =
  validate (NativeInts.Collection.ArrayArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.ArrayArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.greater_than``() =
  validate (NativeInts.Collection.ArrayArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.ArrayArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.equals``() =
  validate (NativeInts.Collection.ArrayArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.equal``() =
  validate (NativeInts.Collection.ArrayArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.not_equal``() =
  validate (NativeInts.Collection.ArrayArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.compare``() =
  validate (NativeInts.Collection.ArrayArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;-1;-1;-1;1;1;0;-1;-1;-1;1;1;1;0;-1;-1;1;1;1;1;0;-1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.less_than``() =
  validate (NativeInts.Collection.ArrayArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.ArrayArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.greater_than``() =
  validate (NativeInts.Collection.ArrayArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.ArrayArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.equals``() =
  validate (NativeInts.Collection.ListArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.equal``() =
  validate (NativeInts.Collection.ListArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.not_equal``() =
  validate (NativeInts.Collection.ListArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.compare``() =
  validate (NativeInts.Collection.ListArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;1;-1;-1;1;1;0;1;1;-1;1;-1;-1;0;-1;-1;1;1;-1;1;0;-1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.less_than``() =
  validate (NativeInts.Collection.ListArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;1;0;1;1;0;0;0;0;0;1;0;1;1;0;1;1;0;0;1;0;0;1;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.ListArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;0;1;1;0;0;1;0;0;1;0;1;1;1;1;1;0;0;1;0;1;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.greater_than``() =
  validate (NativeInts.Collection.ListArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;1;0;0;1;1;0;1;1;0;1;0;0;0;0;0;1;1;0;1;0;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.ListArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;1;0;0;1;1;1;1;1;0;1;0;0;1;0;0;1;1;0;1;1;0;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.equals``() =
  validate (NativeInts.Collection.ListArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.equal``() =
  validate (NativeInts.Collection.ListArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.not_equal``() =
  validate (NativeInts.Collection.ListArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.compare``() =
  validate (NativeInts.Collection.ListArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;1;-1;-1;1;1;0;1;1;-1;1;-1;-1;0;-1;-1;1;1;-1;1;0;-1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.less_than``() =
  validate (NativeInts.Collection.ListArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;1;0;1;1;0;0;0;0;0;1;0;1;1;0;1;1;0;0;1;0;0;1;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.ListArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;0;1;1;0;0;1;0;0;1;0;1;1;1;1;1;0;0;1;0;1;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.greater_than``() =
  validate (NativeInts.Collection.ListArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;1;0;0;1;1;0;1;1;0;1;0;0;0;0;0;1;1;0;1;0;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ListArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.ListArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;1;0;0;1;1;1;1;1;0;1;0;0;1;0;0;1;1;0;1;1;0;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.equals``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.not_equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.compare``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;1;-1;1;1;1;0;1;1;1;1;-1;-1;0;-1;-1;1;1;-1;1;0;1;1;-1;-1;1;-1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_than``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;1;0;1;0;0;0;0;0;0;0;0;1;1;0;1;1;0;0;1;0;0;0;0;1;1;0;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.less_or_equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;0;1;0;0;0;1;0;0;0;0;1;1;1;1;1;0;0;1;0;1;0;0;1;1;0;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_than``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;1;0;1;1;1;0;1;1;1;1;0;0;0;0;0;1;1;0;1;0;1;1;0;0;1;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.I.greater_or_equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;1;0;1;1;1;1;1;1;1;1;0;0;1;0;0;1;1;0;1;1;1;1;0;0;1;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.equals``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.not_equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.compare``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;-1;1;-1;1;1;1;0;1;1;1;1;-1;-1;0;-1;-1;1;1;-1;1;0;1;1;-1;-1;1;-1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_than``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;1;0;1;0;0;0;0;0;0;0;0;1;1;0;1;1;0;0;1;0;0;0;0;1;1;0;1;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.less_or_equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;0;1;0;0;0;1;0;0;0;0;1;1;1;1;1;0;0;1;0;1;0;0;1;1;0;1;1
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_than``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;1;0;1;1;1;0;1;1;1;1;0;0;0;0;0;1;1;0;1;0;1;1;0;0;1;0;0
  |]

 [<Test>]
 member __.``NativeInts.Collection.ArrayArray |> Array.map Set.ofArray C.N.greater_or_equal``() =
  validate (NativeInts.Collection.ArrayArray |> Array.map Set.ofArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;1;0;1;1;1;1;1;1;1;1;0;0;1;0;0;1;1;0;1;1;1;1;0;0;1;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.Array E.I.equals``() =
  validate (NullableNativeInts.Collection.Array) E.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.Array E.I.equal``() =
  validate (NullableNativeInts.Collection.Array) E.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.Array E.I.not_equal``() =
  validate (NullableNativeInts.Collection.Array) E.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.Array E.N.equals``() =
  validate (NullableNativeInts.Collection.Array) E.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.Array E.N.equal``() =
  validate (NullableNativeInts.Collection.Array) E.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.Array E.N.not_equal``() =
  validate (NullableNativeInts.Collection.Array) E.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.OptionArray E.I.equals``() =
  validate (NullableNativeInts.Collection.OptionArray) E.I.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.OptionArray E.I.equal``() =
  validate (NullableNativeInts.Collection.OptionArray) E.I.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.OptionArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.OptionArray) E.I.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.OptionArray E.N.equals``() =
  validate (NullableNativeInts.Collection.OptionArray) E.N.equals [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.OptionArray E.N.equal``() =
  validate (NullableNativeInts.Collection.OptionArray) E.N.equal [|
   1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.OptionArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.OptionArray) E.N.not_equal [|
   0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefArray E.I.equals``() =
  validate (NullableNativeInts.Collection.RefArray) E.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefArray E.I.equal``() =
  validate (NullableNativeInts.Collection.RefArray) E.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.RefArray) E.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefArray E.N.equals``() =
  validate (NullableNativeInts.Collection.RefArray) E.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefArray E.N.equal``() =
  validate (NullableNativeInts.Collection.RefArray) E.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.RefArray) E.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefWrapArray E.I.equals``() =
  validate (NullableNativeInts.Collection.RefWrapArray) E.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefWrapArray E.I.equal``() =
  validate (NullableNativeInts.Collection.RefWrapArray) E.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefWrapArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.RefWrapArray) E.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefWrapArray E.N.equals``() =
  validate (NullableNativeInts.Collection.RefWrapArray) E.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefWrapArray E.N.equal``() =
  validate (NullableNativeInts.Collection.RefWrapArray) E.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.RefWrapArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.RefWrapArray) E.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionArray E.I.equals``() =
  validate (NullableNativeInts.Collection.UnionArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionArray E.I.equal``() =
  validate (NullableNativeInts.Collection.UnionArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.UnionArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionArray E.N.equals``() =
  validate (NullableNativeInts.Collection.UnionArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionArray E.N.equal``() =
  validate (NullableNativeInts.Collection.UnionArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.UnionArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionWrapArray E.I.equals``() =
  validate (NullableNativeInts.Collection.UnionWrapArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionWrapArray E.I.equal``() =
  validate (NullableNativeInts.Collection.UnionWrapArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionWrapArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.UnionWrapArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionWrapArray E.N.equals``() =
  validate (NullableNativeInts.Collection.UnionWrapArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionWrapArray E.N.equal``() =
  validate (NullableNativeInts.Collection.UnionWrapArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.UnionWrapArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.UnionWrapArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueArray E.I.equals``() =
  validate (NullableNativeInts.Collection.ValueArray) E.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueArray E.I.equal``() =
  validate (NullableNativeInts.Collection.ValueArray) E.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.ValueArray) E.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueArray E.N.equals``() =
  validate (NullableNativeInts.Collection.ValueArray) E.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueArray E.N.equal``() =
  validate (NullableNativeInts.Collection.ValueArray) E.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.ValueArray) E.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueWrapArray E.I.equals``() =
  validate (NullableNativeInts.Collection.ValueWrapArray) E.I.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueWrapArray E.I.equal``() =
  validate (NullableNativeInts.Collection.ValueWrapArray) E.I.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueWrapArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.ValueWrapArray) E.I.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueWrapArray E.N.equals``() =
  validate (NullableNativeInts.Collection.ValueWrapArray) E.N.equals [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueWrapArray E.N.equal``() =
  validate (NullableNativeInts.Collection.ValueWrapArray) E.N.equal [|
   1;0;0;0;0;1;0;0;0;0;1;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ValueWrapArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.ValueWrapArray) E.N.not_equal [|
   0;1;1;1;1;0;1;1;1;1;0;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ArrayArray E.I.equals``() =
  validate (NullableNativeInts.Collection.ArrayArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ArrayArray E.I.equal``() =
  validate (NullableNativeInts.Collection.ArrayArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ArrayArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.ArrayArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ArrayArray E.N.equals``() =
  validate (NullableNativeInts.Collection.ArrayArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ArrayArray E.N.equal``() =
  validate (NullableNativeInts.Collection.ArrayArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ArrayArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.ArrayArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ListArray E.I.equals``() =
  validate (NullableNativeInts.Collection.ListArray) E.I.equals [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ListArray E.I.equal``() =
  validate (NullableNativeInts.Collection.ListArray) E.I.equal [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ListArray E.I.not_equal``() =
  validate (NullableNativeInts.Collection.ListArray) E.I.not_equal [|
   0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ListArray E.N.equals``() =
  validate (NullableNativeInts.Collection.ListArray) E.N.equals [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ListArray E.N.equal``() =
  validate (NullableNativeInts.Collection.ListArray) E.N.equal [|
   1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``NullableNativeInts.Collection.ListArray E.N.not_equal``() =
  validate (NullableNativeInts.Collection.ListArray) E.N.not_equal [|
   0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.equals``() =
  validate (Bytes.Collection.Array) C.I.equals [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.equal``() =
  validate (Bytes.Collection.Array) C.I.equal [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.not_equal``() =
  validate (Bytes.Collection.Array) C.I.not_equal [|
   0;1;0;1;1;1;0;1;1;1;0;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.compare``() =
  validate (Bytes.Collection.Array) C.I.compare [|
   0;-1;0;-1;-1;1;0;1;1;1;0;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.less_than``() =
  validate (Bytes.Collection.Array) C.I.less_than [|
   0;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.less_or_equal``() =
  validate (Bytes.Collection.Array) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.greater_than``() =
  validate (Bytes.Collection.Array) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.I.greater_or_equal``() =
  validate (Bytes.Collection.Array) C.I.greater_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.equals``() =
  validate (Bytes.Collection.Array) C.N.equals [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.equal``() =
  validate (Bytes.Collection.Array) C.N.equal [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.not_equal``() =
  validate (Bytes.Collection.Array) C.N.not_equal [|
   0;1;0;1;1;1;0;1;1;1;0;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.compare``() =
  validate (Bytes.Collection.Array) C.N.compare [|
   0;-1;0;-1;-1;1;0;1;1;1;0;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.less_than``() =
  validate (Bytes.Collection.Array) C.N.less_than [|
   0;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.less_or_equal``() =
  validate (Bytes.Collection.Array) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.greater_than``() =
  validate (Bytes.Collection.Array) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.Array C.N.greater_or_equal``() =
  validate (Bytes.Collection.Array) C.N.greater_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.equals``() =
  validate (Bytes.Collection.OptionArray) C.I.equals [|
   1;0;0;0;0;0;0;1;0;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.equal``() =
  validate (Bytes.Collection.OptionArray) C.I.equal [|
   1;0;0;0;0;0;0;1;0;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.not_equal``() =
  validate (Bytes.Collection.OptionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;0;1;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.compare``() =
  validate (Bytes.Collection.OptionArray) C.I.compare [|
   0;-1;-1;-1;-1;-1;1;0;-255;0;-1;-2;1;255;0;255;254;253;1;0;-255;0;-1;-2;1;1;-254;1;0;-1;1;2;-253;2;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.less_than``() =
  validate (Bytes.Collection.OptionArray) C.I.less_than [|
   0;1;1;1;1;1;0;0;1;0;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.less_or_equal``() =
  validate (Bytes.Collection.OptionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;1;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.greater_than``() =
  validate (Bytes.Collection.OptionArray) C.I.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;0;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.I.greater_or_equal``() =
  validate (Bytes.Collection.OptionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;1;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.equals``() =
  validate (Bytes.Collection.OptionArray) C.N.equals [|
   1;0;0;0;0;0;0;1;0;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.equal``() =
  validate (Bytes.Collection.OptionArray) C.N.equal [|
   1;0;0;0;0;0;0;1;0;1;0;0;0;0;1;0;0;0;0;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.not_equal``() =
  validate (Bytes.Collection.OptionArray) C.N.not_equal [|
   0;1;1;1;1;1;1;0;1;0;1;1;1;1;0;1;1;1;1;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.compare``() =
  validate (Bytes.Collection.OptionArray) C.N.compare [|
   0;-1;-1;-1;-1;-1;1;0;-255;0;-1;-2;1;255;0;255;254;253;1;0;-255;0;-1;-2;1;1;-254;1;0;-1;1;2;-253;2;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.less_than``() =
  validate (Bytes.Collection.OptionArray) C.N.less_than [|
   0;1;1;1;1;1;0;0;1;0;1;1;0;0;0;0;0;0;0;0;1;0;1;1;0;0;1;0;0;1;0;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.less_or_equal``() =
  validate (Bytes.Collection.OptionArray) C.N.less_or_equal [|
   1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;0;0;0;0;1;1;1;1;1;0;0;1;0;1;1;0;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.greater_than``() =
  validate (Bytes.Collection.OptionArray) C.N.greater_than [|
   0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;1;1;1;1;0;0;0;0;0;1;1;0;1;0;0;1;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.OptionArray C.N.greater_or_equal``() =
  validate (Bytes.Collection.OptionArray) C.N.greater_or_equal [|
   1;0;0;0;0;0;1;1;0;1;0;0;1;1;1;1;1;1;1;1;0;1;0;0;1;1;0;1;1;0;1;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.equals``() =
  validate (Bytes.Collection.RefArray) C.I.equals [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.equal``() =
  validate (Bytes.Collection.RefArray) C.I.equal [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.not_equal``() =
  validate (Bytes.Collection.RefArray) C.I.not_equal [|
   0;1;0;1;1;1;0;1;1;1;0;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.compare``() =
  validate (Bytes.Collection.RefArray) C.I.compare [|
   0;-1;0;-1;-1;1;0;1;1;1;0;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.less_than``() =
  validate (Bytes.Collection.RefArray) C.I.less_than [|
   0;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.less_or_equal``() =
  validate (Bytes.Collection.RefArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.greater_than``() =
  validate (Bytes.Collection.RefArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.I.greater_or_equal``() =
  validate (Bytes.Collection.RefArray) C.I.greater_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.equals``() =
  validate (Bytes.Collection.RefArray) C.N.equals [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.equal``() =
  validate (Bytes.Collection.RefArray) C.N.equal [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.not_equal``() =
  validate (Bytes.Collection.RefArray) C.N.not_equal [|
   0;1;0;1;1;1;0;1;1;1;0;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.compare``() =
  validate (Bytes.Collection.RefArray) C.N.compare [|
   0;-1;0;-1;-1;1;0;1;1;1;0;-1;0;-1;-1;1;-1;1;0;-1;1;-1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.less_than``() =
  validate (Bytes.Collection.RefArray) C.N.less_than [|
   0;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.less_or_equal``() =
  validate (Bytes.Collection.RefArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.greater_than``() =
  validate (Bytes.Collection.RefArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefArray C.N.greater_or_equal``() =
  validate (Bytes.Collection.RefArray) C.N.greater_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.equals``() =
  validate (Bytes.Collection.RefWrapArray) C.I.equals [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.equal``() =
  validate (Bytes.Collection.RefWrapArray) C.I.equal [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.not_equal``() =
  validate (Bytes.Collection.RefWrapArray) C.I.not_equal [|
   0;1;0;1;1;1;0;1;1;1;0;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.compare``() =
  validate (Bytes.Collection.RefWrapArray) C.I.compare [|
   0;-255;0;-1;-2;255;0;255;254;253;0;-255;0;-1;-2;1;-254;1;0;-1;2;-253;2;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.less_than``() =
  validate (Bytes.Collection.RefWrapArray) C.I.less_than [|
   0;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.less_or_equal``() =
  validate (Bytes.Collection.RefWrapArray) C.I.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.greater_than``() =
  validate (Bytes.Collection.RefWrapArray) C.I.greater_than [|
   0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.I.greater_or_equal``() =
  validate (Bytes.Collection.RefWrapArray) C.I.greater_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.equals``() =
  validate (Bytes.Collection.RefWrapArray) C.N.equals [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.equal``() =
  validate (Bytes.Collection.RefWrapArray) C.N.equal [|
   1;0;1;0;0;0;1;0;0;0;1;0;1;0;0;0;0;0;1;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.not_equal``() =
  validate (Bytes.Collection.RefWrapArray) C.N.not_equal [|
   0;1;0;1;1;1;0;1;1;1;0;1;0;1;1;1;1;1;0;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.compare``() =
  validate (Bytes.Collection.RefWrapArray) C.N.compare [|
   0;-255;0;-1;-2;255;0;255;254;253;0;-255;0;-1;-2;1;-254;1;0;-1;2;-253;2;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.less_than``() =
  validate (Bytes.Collection.RefWrapArray) C.N.less_than [|
   0;1;0;1;1;0;0;0;0;0;0;1;0;1;1;0;1;0;0;1;0;1;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.less_or_equal``() =
  validate (Bytes.Collection.RefWrapArray) C.N.less_or_equal [|
   1;1;1;1;1;0;1;0;0;0;1;1;1;1;1;0;1;0;1;1;0;1;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.greater_than``() =
  validate (Bytes.Collection.RefWrapArray) C.N.greater_than [|
   0;0;0;0;0;1;0;1;1;1;0;0;0;0;0;1;0;1;0;0;1;0;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.RefWrapArray C.N.greater_or_equal``() =
  validate (Bytes.Collection.RefWrapArray) C.N.greater_or_equal [|
   1;0;1;0;0;1;1;1;1;1;1;0;1;0;0;1;0;1;1;0;1;0;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.equals``() =
  validate (Bytes.Collection.UnionArray) C.I.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.equal``() =
  validate (Bytes.Collection.UnionArray) C.I.equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.not_equal``() =
  validate (Bytes.Collection.UnionArray) C.I.not_equal [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.compare``() =
  validate (Bytes.Collection.UnionArray) C.I.compare [|
   0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;0;-1;-2;-2;
   -2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;2;1;0;-1;-1;-1;-1;2;1;-1;
   -1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;
   2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;0;-1;
   -2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;0;-1;3;2;1;2;
   1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;
   1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;
   1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;2;1;1;-1;-1;
   -1;-1;2;1;0;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;
   0;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;
   2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;1;
   -1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;1;3;2;1;3;
   2;1;1;3;2;1;3;2;1;1;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;-1;-1;
   -2;-3;-3;-3;-3;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;
   2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;3;2;1;0;-1;
   -2;-3;3;2;1;-1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;0;-1;-2;3;2;1;
   1;-1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;
   2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;3;2;1;3;2;1;
   0;3;2;1;3;2;1;-1;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;
   -3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;0;-1;-2;-2;-2;-2;1;-1;
   -1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;0;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;
   3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;1;
   -1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;2;1;1;-1;3;2;1;
   2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;2;1;-1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;-1;3;
   2;1;3;2;1;1;3;2;1;3;2;1;0;3;2;1;3;2;1;-1;1;-1;-2;-3;-3;-3;-3;-1;-1;-2;-3;-3;-3;-3;1;-1;-2;-3;-3;-3;
   -3;1;-1;-2;-3;-3;-3;-3;0;-1;-2;-3;-3;-3;-3;1;1;-1;-2;-2;-2;-2;1;-1;-1;-2;-2;-2;-2;1;1;-1;-2;-2;-2;-2;1;1;-1;-2;
   -2;-2;-2;1;0;-1;-2;-2;-2;-2;2;1;1;-1;-1;-1;-1;2;1;-1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;1;-1;-1;-1;-1;2;1;
   0;-1;-1;-1;-1;3;2;1;1;-1;-2;-3;3;2;1;-1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;1;-1;-2;-3;3;2;1;0;-1;-2;-3;
   3;2;1;1;1;-1;-2;3;2;1;1;-1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;1;-1;-2;3;2;1;1;0;-1;-2;3;2;1;2;1;
   1;-1;3;2;1;2;1;-1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;1;-1;3;2;1;2;1;0;-1;3;2;1;3;2;1;1;3;2;1;
   3;2;1;-1;3;2;1;3;2;1;1;3;2;1;3;2;1;1;3;2;1;3;2;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.less_than``() =
  validate (Bytes.Collection.UnionArray) C.I.less_than [|
   0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;
   1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;
   1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;
   0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;0;1;
   1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;1;1;0;0;0;
   0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;
   0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;
   0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.less_or_equal``() =
  validate (Bytes.Collection.UnionArray) C.I.less_or_equal [|
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;
   1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;
   0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;
   0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;
   0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;
   1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;
   0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;1;
   1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;0;0;
   1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;0;0;1;0;0;0;
   0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;
   1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;0;1;1;1;1;1;0;1;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;
   1;1;1;0;1;1;1;1;1;1;0;0;0;1;1;1;1;0;0;1;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   1;1;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;1;1;1;1;
   0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;1;1;1;0;0;0;0;0;
   0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;1;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.greater_than``() =
  validate (Bytes.Collection.UnionArray) C.I.greater_than [|
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;
   0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;
   0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;
   1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.I.greater_or_equal``() =
  validate (Bytes.Collection.UnionArray) C.I.greater_or_equal [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;
   0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;
   0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;
   1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;
   0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;
   1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;
   1;1;1;1;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;
   0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;
   1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;
   1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;
   0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;
   1;1;1;1;1;1;1;1;1;1;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;
   1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;1;1;1;0;
   0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;0;0;1;1;1;
   1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;
   1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;1;1;1;1;1;
   1;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;0;0;0;1;0;
   0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;
   1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;1;
   0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;1;0;1;1;1;
   1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;
   1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;1;0;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;0;0;0;0;0;0;1;1;0;0;0;0;0;1;1;0;0;
   0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;0;0;0;0;0;1;1;1;0;0;0;0;1;1;1;0;0;0;0;1;1;
   1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;0;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;1;1;1;1;0;0;0;
   1;1;1;1;1;0;0;1;1;1;1;0;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;0;0;1;1;1;1;1;
   1;0;1;1;1;1;1;0;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;0;1;1;1;1;1;1;1;1;1;1;
   1;1;1;0;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1
  |]

 [<Test>]
 member __.``Bytes.Collection.UnionArray C.N.equals``() =
  validate (Bytes.Collection.UnionArray) C.N.equals [|
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;
   0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;
   0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;
   1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;0;
   0;0;0;0;0;0;0;0;0;0;0;0;1;0;0;0;0;0;0;