/////////////////////////////////////////////////////////////////////////////
// Name:        log.h
// Purpose:     interface of wxLog* classes
// Author:      wxWidgets team
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

#if wxUSE_BASE

/**
    Different standard log levels (you may also define your own) used with
    by standard wxLog functions wxLogGeneric(), wxLogError(), wxLogWarning(), etc...
*/
enum wxLogLevelValues
{
    wxLOG_FatalError, //!< program can't continue, abort immediately
    wxLOG_Error,      //!< a serious error, user must be informed about it
    wxLOG_Warning,    //!< user is normally informed about it but may be ignored
    wxLOG_Message,    //!< normal message (i.e. normal output of a non GUI app)
    wxLOG_Status,     //!< informational: might go to the status line of GUI app
    wxLOG_Info,       //!< informational message (a.k.a. 'Verbose')
    wxLOG_Debug,      //!< never shown to the user, disabled in release mode
    wxLOG_Trace,      //!< trace messages are also only enabled in debug mode
    wxLOG_Progress,   //!< used for progress indicator (not yet)
    wxLOG_User = 100, //!< user defined levels start here
    wxLOG_Max = 10000
};

/**
    The type used to specify a log level.

    Default values of ::wxLogLevel used by wxWidgets are contained in the
    ::wxLogLevelValues enumeration.
*/
typedef unsigned long wxLogLevel;

/**
    Information about a log record (unit of the log output).
 */
class wxLogRecordInfo
{
public:
    /// The name of the file where this log message was generated.
    const char *filename;

    /// The line number at which this log message was generated.
    int line;

    /**
        The name of the function where the log record was generated.

        This field may be @NULL if the compiler doesn't support @c \__FUNCTION__
        (but most modern compilers do).
     */
    const char *func;

    /**
        Time in milliseconds since Epoch when the log message was generated.

        @since 3.1.5
     */
    wxLongLong_t timestampMS;

    /**
        Time when the log message was generated.

        @deprecated Please use timestampMS instead.
     */
    time_t timestamp;

    /**
        Id of the thread in which the message was generated.

        This field is only available if wxWidgets was built with threads
        support (<code>wxUSE_THREADS == 1</code>).

        @see wxThread::GetCurrentId()
     */
    wxThreadIdType threadId;
};

/**
    @class wxLogFormatter

    wxLogFormatter class is used to format the log messages. It implements the
    default formatting and can be derived from to create custom formatters.

    The default implementation formats the message into a string containing
    the time stamp, level-dependent prefix and the message itself.

    To change it, you can derive from it and override its Format() method. For
    example, to include the thread id in the log messages you can use
    @code
        class LogFormatterWithThread : public wxLogFormatter
        {
            virtual wxString Format(wxLogLevel level,
                                    const wxString& msg,
                                    const wxLogRecordInfo& info) const
            {
                return wxString::Format("[%d] %s(%d) : %s",
                    info.threadId, info.filename, info.line, msg);
            }
        };
    @endcode
    And then associate it with wxLog instance using its SetFormatter(). Then,
    if you call:

    @code
        wxLogMessage(_("*** Application started ***"));
    @endcode

    the log output could be something like:

    @verbatim
        [7872] d:\testApp\src\testApp.cpp(85) : *** Application started ***
    @endverbatim

    @library{wxbase}
    @category{logging}

    @see @ref overview_log

    @since 2.9.4
*/
class wxLogFormatter
{
public:
    /**
        The default ctor does nothing.
    */
    wxLogFormatter();


    /**
        This function creates the full log message string.

        Override it to customize the output string format.

        @param level
            The level of this log record, e.g. ::wxLOG_Error.
        @param msg
            The log message itself.
        @param info
            All the other information (such as time, component, location...)
            associated with this log record.

        @return
            The formatted message.

        @note
            Time stamping is disabled for Visual C++ users in debug builds by
            default because otherwise it would be impossible to directly go to the line
            from which the log message was generated by simply clicking in the debugger
            window on the corresponding error message. If you wish to enable it, override
            FormatTime().
    */
    virtual wxString Format(wxLogLevel level,
                            const wxString& msg,
                            const wxLogRecordInfo& info) const;

protected:
    /**
        This function formats the time stamp part of the log message including
        milliseconds.

        Override this function if you need to customize just the time stamp
        formatting in the log messages.

        @param msec
            Time to format as the number of milliseconds since
            1970-01-01T00:00:00.

        @return
            The formatted time string, may be empty.

        @since 3.1.5
    */
    virtual wxString FormatTimeMS(wxLongLong_t msec) const;

    /**
        This function formats the time stamp part of the log message.

        @deprecated This function only exists for compatibility, please
        override FormatTimeMS() in the new code.

        Override this function if you need to customize just the time stamp.

        @param time
            Time to format.

        @return
            The formatted time string, may be empty.
    */
    virtual wxString FormatTime(time_t time) const;
};


/**
    @class wxLog

    wxLog class defines the interface for the <em>log targets</em> used by wxWidgets
    logging functions as explained in the @ref overview_log.

    The only situations when you need to directly use this class is when you want
    to derive your own log target because the existing ones don't satisfy your
    needs.

    Otherwise, it is completely hidden behind the @ref group_funcmacro_log "wxLogXXX() functions"
    and you may not even know about its existence.

    @note For console-mode applications, the default target is wxLogStderr, so
          that all @e wxLogXXX() functions print on @c stderr when @c wxUSE_GUI = 0.

    @library{wxbase}
    @category{logging}

    @see @ref overview_log, @ref group_funcmacro_log "wxLogXXX() functions"
*/
class wxLog
{
public:
    /**
        @name Trace mask functions
    */
    ///@{

    /**
        Add the @a mask to the list of allowed masks for wxLogTrace().

        @see RemoveTraceMask(), GetTraceMasks()
    */
    static void AddTraceMask(const wxString& mask);

    /**
        Removes all trace masks previously set with AddTraceMask().

        @see RemoveTraceMask()
    */
    static void ClearTraceMasks();

    /**
        Returns the currently allowed list of string trace masks.

        @see AddTraceMask().
    */
    static const wxArrayString& GetTraceMasks();

    /**
        Returns @true if the @a mask is one of allowed masks for wxLogTrace().

        See also: AddTraceMask(), RemoveTraceMask()
    */
    static bool IsAllowedTraceMask(const wxString& mask);

    /**
        Remove the @a mask from the list of allowed masks for
        wxLogTrace().

        @see AddTraceMask()
    */
    static void RemoveTraceMask(const wxString& mask);

    ///@}



    /**
        @name Log target functions
    */
    ///@{

    /**
        Instructs wxLog to not create new log targets on the fly if there is none
        currently (see GetActiveTarget()).

        (Almost) for internal use only: it is supposed to be called by the
        application shutdown code (where you don't want the log target to be
        automatically created anymore).

        Note that this function also calls ClearTraceMasks().
    */
    static void DontCreateOnDemand();

    /**
        Returns the pointer to the active log target (may be @NULL).

        Notice that if SetActiveTarget() hadn't been previously explicitly
        called, this function will by default try to create a log target by
        calling wxAppTraits::CreateLogTarget() which may be overridden in a
        user-defined traits class to change the default behaviour. You may also
        call DontCreateOnDemand() to disable this behaviour.

        When this function is called from threads other than main one,
        auto-creation doesn't happen. But if the thread has a thread-specific
        log target previously set by SetThreadActiveTarget(), it is returned
        instead of the global one. Otherwise, the global log target is
        returned.
    */
    static wxLog* GetActiveTarget();

    /**
        Sets the specified log target as the active one.

        Returns the pointer to the previous active log target (may be @NULL).
        To suppress logging use a new instance of wxLogNull not @NULL.  If the
        active log target is set to @NULL a new default log target will be
        created when logging occurs.

        @see SetThreadActiveTarget()
    */
    static wxLog* SetActiveTarget(wxLog* logtarget);

    /**
        Sets a thread-specific log target.

        The log target passed to this function will be used for all messages
        logged by the current thread using the usual wxLog functions. This
        shouldn't be called from the main thread which never uses a thread-
        specific log target but can be used for the other threads to handle
        thread logging completely separately; instead of buffering thread log
        messages in the main thread logger.

        Notice that unlike for SetActiveTarget(), wxWidgets does not destroy
        the thread-specific log targets when the thread terminates so doing
        this is your responsibility.

        This method is only available if @c wxUSE_THREADS is 1, i.e. wxWidgets
        was compiled with threads support.

        @param logger
            The new thread-specific log target, possibly @NULL.
        @return
            The previous thread-specific log target, initially @NULL.

        @since 2.9.1
     */
    static wxLog *SetThreadActiveTarget(wxLog *logger);

    /**
        Flushes the current log target if any, does nothing if there is none.

        When this method is called from the main thread context, it also
        flushes any previously buffered messages logged by the other threads.
        When it is called from the other threads it simply calls Flush() on the
        currently active log target, so it mostly makes sense to do this if a
        thread has its own logger set with SetThreadActiveTarget().
    */
    static void FlushActive();

    /**
        Resumes logging previously suspended by a call to Suspend().
        All messages logged in the meanwhile will be flushed soon.
    */
    static void Resume();

    /**
        Suspends the logging until Resume() is called.

        Note that the latter must be called the same number of times as the former
        to undo it, i.e. if you call Suspend() twice you must call Resume() twice as well.

        Note that suspending the logging means that the log sink won't be flushed
        periodically, it doesn't have any effect if the current log target does the
        logging immediately without waiting for Flush() to be called (the standard
        GUI log target only shows the log dialog when it is flushed, so Suspend()
        works as expected with it).

        @see Resume(), wxLogNull
    */
    static void Suspend();

    ///@}



    /**
        @name Log level functions
    */
    ///@{

    /**
        Returns the current log level limit.

        All messages at levels strictly greater than the value returned by this
        function are not logged at all.

        @see SetLogLevel(), IsLevelEnabled()
    */
    static wxLogLevel GetLogLevel();

    /**
        Returns true if logging at this level is enabled for the current thread.

        This function only returns @true if logging is globally enabled and if
        @a level is less than or equal to the maximal log level enabled for the
        given @a component.

        @see IsEnabled(), SetLogLevel(), GetLogLevel(), SetComponentLevel()

        @since 2.9.1
     */
    static bool IsLevelEnabled(wxLogLevel level, wxString component);

    /**
        Sets the log level for the given component.

        For example, to disable all but error messages from wxWidgets network
        classes you may use
        @code
            wxLog::SetComponentLevel("wx/net", wxLOG_Error);
        @endcode

        SetLogLevel() may be used to set the global log level.

        @param component
            Non-empty component name, possibly using slashes (@c /) to separate
            it into several parts.
        @param level
            Maximal level of log messages from this component which will be
            handled instead of being simply discarded.

        @since 2.9.1
     */
    static void SetComponentLevel(const wxString& component, wxLogLevel level);

    /**
        Specifies that log messages with level greater (numerically) than
        @a logLevel should be ignored and not sent to the active log target.

        @see SetComponentLevel()
    */
    static void SetLogLevel(wxLogLevel logLevel);

    ///@}



    /**
        @name Enable/disable features functions
    */
    ///@{

    /**
        Globally enable or disable logging.

        Calling this function with @false argument disables all log messages
        for the current thread.

        @see wxLogNull, IsEnabled()

        @return
            The old state, i.e. @true if logging was previously enabled and
            @false if it was disabled.
     */
    static bool EnableLogging(bool enable = true);

    /**
        Returns true if logging is enabled at all now.

        @see IsLevelEnabled(), EnableLogging()
     */
    static bool IsEnabled();

    /**
        Returns whether the repetition counting mode is enabled.
    */
    static bool GetRepetitionCounting();

    /**
        Enables logging mode in which a log message is logged once, and in case exactly
        the same message successively repeats one or more times, only the number of
        repetitions is logged.
    */
    static void SetRepetitionCounting(bool repetCounting = true);

    /**
        Returns the current timestamp format string.

        Notice that the current time stamp is only used by the default log
        formatter and custom formatters may ignore this format.
    */
    static const wxString& GetTimestamp();

    /**
        Sets the timestamp format prepended by the default log targets to all
        messages. The string may contain any normal characters as well as %
        prefixed format specifiers, see @e strftime() manual for details.
        Passing an empty string to this function disables message time stamping.

        Notice that the current time stamp is only used by the default log
        formatter and custom formatters may ignore this format. You can also
        define a custom wxLogFormatter to customize the time stamp handling
        beyond changing its format.
    */
    static void SetTimestamp(const wxString& format);

    /**
        Disables time stamping of the log messages.

        Notice that the current time stamp is only used by the default log
        formatter and custom formatters may ignore calls to this function.

        @since 2.9.0
    */
    static void DisableTimestamp();

    /**
        Returns whether the verbose mode is currently active.
    */
    static bool GetVerbose();

    /**
        Activates or deactivates verbose mode in which the verbose messages are
        logged as the normal ones instead of being silently dropped.

        The verbose messages are the trace messages which are not disabled in the
        release mode and are generated by wxLogVerbose().

        @see @ref overview_log
    */
    static void SetVerbose(bool verbose = true);

    ///@}


    /**
        Sets the specified formatter as the active one.

        @param formatter
            The new formatter. If @NULL, reset to the default formatter.

        Returns the pointer to the previous formatter. You must delete it
        if you don't plan to attach it again to a wxLog object later.

        @since 2.9.4
    */
    wxLogFormatter *SetFormatter(wxLogFormatter* formatter);


    /**
        Show all pending output and clear the buffer.

        Some of wxLog implementations, most notably the standard wxLogGui class,
        buffer the messages (for example, to avoid showing the user a zillion of modal
        message boxes one after another -- which would be really annoying).
        This function shows them all and clears the buffer contents.
        If the buffer is already empty, nothing happens.

        If you override this method in a derived class, call the base class
        version first, before doing anything else.
    */
    virtual void Flush();

    /**
        Log the given record.

        This function should only be called from the DoLog() implementations in
        the derived classes if they need to call DoLogRecord() on another log
        object (they can, of course, just use wxLog::DoLogRecord() call syntax
        to call it on the object itself). It should not be used for logging new
        messages which can be only sent to the currently active logger using
        OnLog() which also checks if the logging (for this level) is enabled
        while this method just directly calls DoLog().

        Example of use of this class from wxLogChain:
        @code
        void wxLogChain::DoLogRecord(wxLogLevel level,
                                     const wxString& msg,
                                     const wxLogRecordInfo& info)
        {
            // let the previous logger show it
            if ( m_logOld && IsPassingMessages() )
                m_logOld->LogRecord(level, msg, info);

            // and also send it to the new one
            if ( m_logNew && m_logNew != this )
                m_logNew->LogRecord(level, msg, info);
        }
        @endcode

        @since 2.9.1
     */
    void LogRecord(wxLogLevel level, const wxString& msg, const wxLogRecordInfo& info);

protected:
    /**
        @name Logging callbacks.

        The functions which should be overridden by custom log targets.

        When defining a new log target, you have a choice between overriding
        DoLogRecord(), which provides maximal flexibility, DoLogTextAtLevel()
        which can be used if you don't intend to change the default log
        messages formatting but want to handle log messages of different levels
        differently or, in the simplest case, DoLogText().
     */
    ///@{

    /**
        Called to log a new record.

        Any log message created by wxLogXXX() functions is passed to this
        method of the active log target. The default implementation prepends
        the timestamp and, for some log levels (e.g. error and warning), the
        corresponding prefix to @a msg and passes it to DoLogTextAtLevel().

        You may override this method to implement custom formatting of the
        log messages or to implement custom filtering of log messages (e.g. you
        could discard all log messages coming from the given source file).
     */
    virtual void DoLogRecord(wxLogLevel level,
                             const wxString& msg,
                             const wxLogRecordInfo& info);

    /**
        Called to log the specified string at given level.

        The base class versions logs debug and trace messages on the system
        default debug output channel and passes all the other messages to
        DoLogText().
    */
    virtual void DoLogTextAtLevel(wxLogLevel level, const wxString& msg);

    /**
        Called to log the specified string.

        A simple implementation might just send the string to @c stdout or
        @c stderr or save it in a file (of course, the already existing
        wxLogStderr can be used for this).

        The base class version of this function asserts so it must be
        overridden if you don't override DoLogRecord() or DoLogTextAtLevel().
    */
    virtual void DoLogText(const wxString& msg);

    ///@}
};



/**
    @class wxLogChain

    This simple class allows you to chain log sinks, that is to install a new sink but
    keep passing log messages to the old one instead of replacing it completely as
    wxLog::SetActiveTarget does.

    It is especially useful when you want to divert the logs somewhere (for
    example to a file or a log window) but also keep showing the error messages
    using the standard dialogs as wxLogGui does by default.

    Example of usage:

    @code
    wxLogChain *logChain = new wxLogChain(new wxLogStderr);

    // all the log messages are sent to stderr and also processed as usually
    ...

    // don't delete logChain directly as this would leave a dangling
    // pointer as active log target, use SetActiveTarget() instead
    delete wxLog::SetActiveTarget(...something else or NULL...);
    @endcode

    @library{wxbase}
    @category{logging}
*/
class wxLogChain : public wxLog
{
public:
    /**
        Sets the specified @c logger (which may be @NULL) as the default log
        target but the log messages are also passed to the previous log target if any.
    */
    wxLogChain(wxLog* logger);

    /**
        Destroys the previous log target.
    */
    virtual ~wxLogChain();

    /**
        Detaches the old log target so it won't be destroyed when the wxLogChain object
        is destroyed.
    */
    void DetachOldLog();

    /**
        Returns the pointer to the previously active log target (which may be @NULL).
    */
    wxLog* GetOldLog() const;

    /**
        Returns @true if the messages are passed to the previously active log
        target (default) or @false if PassMessages() had been called.
    */
    bool IsPassingMessages() const;

    /**
        By default, the log messages are passed to the previously active log target.
        Calling this function with @false parameter disables this behaviour
        (presumably temporarily, as you shouldn't use wxLogChain at all otherwise) and
        it can be re-enabled by calling it again with @a passMessages set to @true.
    */
    void PassMessages(bool passMessages);

    /**
        Sets another log target to use (may be @NULL).

        The log target specified in the wxLogChain(wxLog*) constructor or in a
        previous call to this function is deleted.
        This doesn't change the old log target value (the one the messages are
        forwarded to) which still remains the same as was active when wxLogChain
        object was created.
    */
    void SetLog(wxLog* logger);
};



/**
    @class wxLogInterposer

    A special version of wxLogChain which uses itself as the new log target.
    It forwards log messages to the previously installed one in addition to
    processing them itself.

    Unlike wxLogChain which is usually used directly as is, this class must be
    derived from to implement wxLog::DoLog and/or wxLog::DoLogString methods.

    wxLogInterposer destroys the previous log target in its destructor.
    If you don't want this to happen, use wxLogInterposerTemp instead.

    @library{wxbase}
    @category{logging}
*/
class wxLogInterposer : public wxLogChain
{
public:
    /**
        The default constructor installs this object as the current active log target.
    */
    wxLogInterposer();
};



/**
    @class wxLogInterposerTemp

    Legacy class which should not be used any longer.

    @deprecated
    This class is only preserved for compatibility, but using it can result in
    unexpected behaviour and memory leaks.

    @library{wxbase}
    @category{logging}
*/
class wxLogInterposerTemp : public wxLogChain
{
public:
    /**
        The default constructor installs this object as the current active log target.
    */
    wxLogInterposerTemp();
};


/**
    @class wxLogStream

    This class can be used to redirect the log messages to a C++ stream.

    Please note that this class is only available if wxWidgets was compiled with
    the standard iostream library support (@c wxUSE_STD_IOSTREAM must be on).

    @library{wxbase}
    @category{logging}

    @see wxLogStderr, wxStreamToTextRedirector
*/
class wxLogStream : public wxLog
{
public:
    /**
        Constructs a log target which sends all the log messages to the given
        output stream. If it is @NULL, the messages are sent to @c cerr.
        The messages will be written in the encoding specified by the
        given @c wxMBConv.

        The @a conv argument is only available in wxWidgets 3.1.1 and later.

        @note
            In practice, it is only advisable to specify @c wxConvUTF8 as
            the @a conv.
            If using @c wxMBConvUTF16(), the file should be opened in
            @c std::ios::binary mode.

        @warning
            If a log message contains any characters that cannot be converted
            to the character set given by @a conv, that message will be
            silently ignored, i.e. it will not be written at all.
    */
    wxLogStream(std::ostream *ostr = NULL,
                const wxMBConv &conv = wxConvWhateverWorks);
};



/**
    @class wxLogStderr

    This class can be used to redirect the log messages to a C file stream (not to
    be confused with C++ streams).

    It is the default log target for the non-GUI wxWidgets applications which
    send all the output to @c stderr.

    @library{wxbase}
    @category{logging}

    @see wxLogStream
*/
class wxLogStderr : public wxLog
{
public:
    /**
        Constructs a log target which sends all the log messages to the given
        @c FILE. If it is @NULL, the messages are sent to @c stderr.
        The messages will be written in the encoding specified by the
        given @c wxMBConv.

        The @a conv argument is only available in wxWidgets 3.1.1 and later.

        @note
            In practice, it is only advisable to specify @c wxConvUTF8 as
            the @a conv.
            If using @c wxMBConvUTF16(), the file should be opened in
            @c "wb" mode.

        @warning
            If a log message contains any characters that cannot be converted
            to the character set given by @a conv, that message will be
            silently ignored, i.e. it will not be written at all.
    */
    wxLogStderr(FILE *fp = NULL,
                const wxMBConv &conv = wxConvWhateverWorks);
};



/**
    @class wxLogBuffer

    wxLogBuffer is a very simple implementation of log sink which simply collects
    all the logged messages in a string (except the debug messages which are output
    in the usual way immediately as we're presumably not interested in collecting
    them for later). The messages from different log function calls are separated
    by the new lines.

    All the messages collected so far can be shown to the user (and the current
    buffer cleared) by calling the overloaded wxLogBuffer::Flush method.

    @library{wxbase}
    @category{logging}
*/
class wxLogBuffer : public wxLog
{
public:
    /**
        The default ctor does nothing.
    */
    wxLogBuffer();

    /**
        Shows all the messages collected so far to the user (using a message box in the
        GUI applications or by printing them out to the console in text mode) and
        clears the internal buffer.
    */
    virtual void Flush();

    /**
        Returns the current buffer contains. Messages from different log function calls
        are separated with the new lines in the buffer.
        The buffer can be cleared by Flush() which will also show the current
        contents to the user.
    */
    const wxString& GetBuffer() const;
};



/**
    @class wxLogNull

    This class allows you to temporarily suspend logging. All calls to the log
    functions during the life time of an object of this class are just ignored.

    In particular, it can be used to suppress the log messages given by wxWidgets
    itself but it should be noted that it is rarely the best way to cope with this
    problem as @b all log messages are suppressed, even if they indicate a
    completely different error than the one the programmer wanted to suppress.

    For instance, the example of the overview:

    @code
      wxFile file;

      // wxFile.Open() normally complains if file can't be opened, we don't want it
      {
        wxLogNull logNo;
        if ( !file.Open("bar") )
          ... process error ourselves ...
      } // ~wxLogNull called, old log sink restored

      wxLogMessage("..."); // ok
    @endcode

    would be better written as:

    @code
      wxFile file;

      // don't try to open file if it doesn't exist, we are prepared to deal with
      // this ourselves - but all other errors are not expected
      if ( wxFile::Exists("bar") )
      {
          // gives an error message if the file couldn't be opened
          file.Open("bar");
      }
      else
      {
          ...
      }
    @endcode


    @library{wxbase}
    @category{logging}
*/
class wxLogNull
{
public:
    /**
        Suspends logging.
    */
    wxLogNull();

    /**
        Resumes logging.
    */
    ~wxLogNull();
};

#endif // wxUSE_BASE

#if wxUSE_GUI

/**
    @class wxLogWindow

    This class represents a background log window: to be precise, it collects all
    log messages in the log frame which it manages but also passes them on to the
    log target which was active at the moment of its creation. This allows you, for
    example, to show all the log messages in a frame but still continue to process
    them normally by showing the standard log dialog.

    @library{wxcore}
    @category{logging}

    @see wxLogTextCtrl
*/
class wxLogWindow : public wxLogInterposer
{
public:
    /**
        Creates the log frame window and starts collecting the messages in it.

        @param pParent
            The parent window for the log frame, may be @NULL
        @param szTitle
            The title for the log frame
        @param show
            @true to show the frame initially (default), otherwise
            Show() must be called later.
        @param passToOld
            @true to process the log messages normally in addition to logging them
            in the log frame (default), @false to only log them in the log frame.
            Note that if no targets were set using wxLog::SetActiveTarget() then
            wxLogWindow simply becomes the active one and messages won't be passed
            to other targets.
    */
    wxLogWindow(wxWindow* pParent, const wxString& szTitle, bool show = true,
                bool passToOld = true);

    /**
        Returns the associated log frame window. This may be used to position or resize
        it but use Show() to show or hide it.
    */
    wxFrame* GetFrame() const;

    /**
        Called if the user closes the window interactively, will not be
        called if it is destroyed for another reason (such as when program
        exits).

        Return @true from here to allow the frame to close, @false to
        prevent this from happening.

        @see OnFrameDelete()
    */
    virtual bool OnFrameClose(wxFrame* frame);

    /**
        Called right before the log frame is going to be deleted: will
        always be called unlike OnFrameClose().
    */
    virtual void OnFrameDelete(wxFrame* frame);

    /**
        Shows or hides the frame.
    */
    void Show(bool show = true);
};



/**
    @class wxLogGui

    This is the default log target for the GUI wxWidgets applications.

    Please see @ref overview_log_customize for explanation of how to change the
    default log target.

    An object of this class is used by default to show the log messages created
    by using wxLogMessage(), wxLogError() and other logging functions. It
    doesn't display the messages logged by them immediately however but
    accumulates all messages logged during an event handler execution and then
    shows them all at once when its Flush() method is called during the idle
    time processing. This has the important advantage of showing only a single
    dialog to the user even if several messages were logged because of a single
    error as it often happens (e.g. a low level function could log a message
    because it failed to open a file resulting in its caller logging another
    message due to the failure of higher level operation requiring the use of
    this file). If you need to force the display of all previously logged
    messages immediately you can use wxLog::FlushActive() to force the dialog
    display.

    Also notice that if an error message is logged when several informative
    messages had been already logged before, the informative messages are
    discarded on the assumption that they are not useful -- and may be
    confusing and hence harmful -- any more after the error. The warning
    and error messages are never discarded however and any informational
    messages logged after the first error one are also kept (as they may
    contain information about the error recovery). You may override DoLog()
    method to change this behaviour.

    At any rate, it is possible that that several messages were accumulated
    before this class Flush() method is called. If this is the case, Flush()
    uses a custom dialog which shows the last message directly and allows the
    user to view the previously logged ones by expanding the "Details"
    wxCollapsiblePane inside it. This custom dialog also provides the buttons
    for copying the log messages to the clipboard and saving them to a file.

    However if only a single message is present when Flush() is called, just a
    wxMessageBox() is used to show it. This has the advantage of being closer
    to the native behaviour but it doesn't give the user any possibility to
    copy or save the message (except for the recent Windows versions where @c
    Ctrl-C may be pressed in the message box to copy its contents to the
    clipboard) so you may want to override DoShowSingleLogMessage() to
    customize wxLogGui -- the dialogs sample shows how to do this.

    @library{wxcore}
    @category{logging}
*/
class wxLogGui : public wxLog
{
public:
    /**
        Default constructor.
    */
    wxLogGui();

    /**
        Presents the accumulated log messages, if any, to the user.

        This method is called during the idle time and should show any messages
        accumulated in wxLogGui#m_aMessages field to the user.
     */
    virtual void Flush();

protected:
    /**
        Returns the appropriate title for the dialog.

        The title is constructed from wxApp::GetAppDisplayName() and the
        severity string (e.g. "error" or "warning") appropriate for the current
        wxLogGui#m_bErrors and wxLogGui#m_bWarnings values.
     */
    wxString GetTitle() const;

    /**
        Returns wxICON_ERROR, wxICON_WARNING or wxICON_INFORMATION depending on
        the current maximal severity.

        This value is suitable to be used in the style parameter of
        wxMessageBox() function.
     */
    int GetSeverityIcon() const;

    /**
        Forgets all the currently stored messages.

        If you override Flush() (and don't call the base class version), you
        must call this method to avoid messages being logged over and over
        again.
     */
    void Clear();


    /**
        All currently accumulated messages.

        This array may be empty if no messages were logged.

        @see m_aSeverity, m_aTimes
     */
    wxArrayString m_aMessages;

    /**
        The severities of each logged message.

        This array is synchronized with wxLogGui#m_aMessages, i.e. the n-th
        element of this array corresponds to the severity of the n-th message.
        The possible severity values are @c wxLOG_XXX constants, e.g.
        wxLOG_Error, wxLOG_Warning, wxLOG_Message etc.
     */
    wxArrayInt m_aSeverity;

    /**
        The time stamps of each logged message.

        The elements of this array are time_t values corresponding to the time
        when the message was logged.
     */
    wxArrayLong m_aTimes;

    /**
        True if there any error messages.
     */
    bool m_bErrors;

    /**
        True if there any warning messages.

        If both wxLogGui#m_bErrors and this member are false, there are only
        informational messages to be shown.
     */
    bool m_bWarnings;

    /**
        True if there any messages to be shown to the user.

        This variable is used instead of simply checking whether
        wxLogGui#m_aMessages array is empty to allow blocking further calls to
        Flush() while a log dialog is already being shown, even if the messages
        array hasn't been emptied yet.
     */
    bool m_bHasMessages;

private:
    /**
        Method called by Flush() to show a single log message.

        This function can be overridden to show the message in a different way.
        By default a simple wxMessageBox() call is used.

        @param message
            The message to show (it can contain multiple lines).
        @param title
            The suggested title for the dialog showing the message, see
            GetTitle().
        @param style
            One of @c wxICON_XXX constants, see GetSeverityIcon().
     */
    virtual void DoShowSingleLogMessage(const wxString& message,
                                        const wxString& title,
                                        int style);

    /**
        Method called by Flush() to show multiple log messages.

        This function can be overridden to show the messages in a different way.
        By default a special log dialog showing the most recent message and
        allowing the user to expand it to view the previously logged ones is
        used.

        @param messages
            Array of messages to show; it contains more than one element.
        @param severities
            Array of message severities containing @c wxLOG_XXX values.
        @param times
            Array of time_t values indicating when each message was logged.
        @param title
            The suggested title for the dialog showing the message, see
            GetTitle().
        @param style
            One of @c wxICON_XXX constants, see GetSeverityIcon().
     */
    virtual void DoShowMultipleLogMessages(const wxArrayString& messages,
                                           const wxArrayInt& severities,
                                           const wxArrayLong& times,
                                           const wxString& title,
                                           int style);
};



/**
    @class wxLogTextCtrl

    Using these target all the log messages can be redirected to a text control.
    The text control must have been created with @c wxTE_MULTILINE style by the
    caller previously.

    @library{wxcore}
    @category{logging}

    @see wxTextCtrl, wxStreamToTextRedirector
*/
class wxLogTextCtrl : public wxLog
{
public:
    /**
        Constructs a log target which sends all the log messages to the given text
        control. The @a textctrl parameter cannot be @NULL.
    */
    wxLogTextCtrl(wxTextCtrl* pTextCtrl);
};


#endif // wxUSE_GUI

#if wxUSE_BASE


// ============================================================================
// Global functions/macros
// ============================================================================

/** @addtogroup group_funcmacro_log */
///@{

/**
    This function shows a message to the user in a safe way and should be safe
    to call even before the application has been initialized or if it is
    currently in some other strange state (for example, about to crash). Under
    Windows this function shows a message box using a native dialog instead of
    wxMessageBox() (which might be unsafe to call), elsewhere it simply prints
    the message to the standard output using the title as prefix.

    @param title
        The title of the message box shown to the user or the prefix of the
        message string.
    @param text
        The text to show to the user.
    @return
        @true If a message box was actually shown or @false if the message was
        logged to the console because there is no safe to show it currently
        (the return value is only available since wxWidgets 3.1.5, the function
        doesn't return anything in the previous versions).

    @see wxLogFatalError()

    @header{wx/log.h}
*/
bool wxSafeShowMessage(const wxString& title, const wxString& text);

/**
    Returns the error code from the last system call. This function uses
    @c errno on Unix platforms and @c GetLastError under Win32.

    @see wxSysErrorMsgStr(), wxLogSysError()

    @header{wx/log.h}
*/
unsigned long wxSysErrorCode();

/**
    Returns the error message corresponding to the given system error code. If
    @a errCode is 0 (default), the last error code (as returned by
    wxSysErrorCode()) is used.

    Use this function instead of wxSysErrorMsg(), as the latter one is not
    thread-safe.

    @since 3.1.0

    @see wxSysErrorCode(), wxLogSysError()

    @header{wx/log.h}
*/
wxString wxSysErrorMsgStr(unsigned long errCode = 0);

/**
    Returns the error message corresponding to the given system error code. If
    @a errCode is 0 (default), the last error code (as returned by
    wxSysErrorCode()) is used.

    Use wxSysErrorMsgStr() instead of this function especially in a
    multi-threaded application.

    @see wxSysErrorCode(), wxLogSysError()

    @header{wx/log.h}
*/
const wxChar* wxSysErrorMsg(unsigned long errCode = 0);

///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    Logs a message with the given wxLogLevel.
    E.g. using @c wxLOG_Message as first argument, this function behaves like wxLogMessage().

    @header{wx/log.h}
*/
void wxLogGeneric(wxLogLevel level, const char* formatString, ... );
void wxVLogGeneric(wxLogLevel level, const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    For all normal, informational messages. They also appear in a message box
    by default (but it can be changed).

    @header{wx/log.h}
*/
void wxLogMessage(const char* formatString, ... );
void wxVLogMessage(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    For low priority messages.

    They are handled in the same way as messages logged by wxLogMessage() by
    the default logger but could be handled differently by the custom loggers.

    @header{wx/log.h}
*/
void wxLogInfo(const char* formatString, ... );
void wxVLogInfo(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    For verbose output.

    Messages generated by these functions are suppressed by default, even if
    the log level is higher than ::wxLOG_Info and need to be explicitly
    activated by calling wxLog::SetVerbose().

    Notice that this is done automatically by wxWidgets, unless the standard
    command line handling is overridden, if @c \--verbose option is specified on
    the program command line, so using these functions provides a simple way of
    having some diagnostic messages not shown by default but which can be
    easily shown by the user if needed.

    @header{wx/log.h}
*/
void wxLogVerbose(const char* formatString, ... );
void wxVLogVerbose(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    For warnings - they are also normally shown to the user, but don't
    interrupt the program work.

    @header{wx/log.h}
*/
void wxLogWarning(const char* formatString, ... );
void wxVLogWarning(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    Like wxLogError(), but also terminates the program with the exit code 3.
    Using @e abort() standard function also terminates the program with this
    exit code.

    @header{wx/log.h}
*/
void wxLogFatalError(const char* formatString, ... );
void wxVLogFatalError(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    The functions to use for error messages, i.e. the messages that must be
    shown to the user. The default processing is to pop up a message box to
    inform the user about it.

    @header{wx/log.h}
*/
void wxLogError(const char* formatString, ... );
void wxVLogError(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    Log a message at @c wxLOG_Trace log level (see ::wxLogLevelValues enum).

    Notice that the use of trace masks is not recommended any more as setting
    the log components (please see @ref overview_log_enable) provides a way to
    do the same thing for log messages of any level, and not just the tracing
    ones.

    Like wxLogDebug(), trace functions only do something in debug builds and
    expand to nothing in the release one. The reason for making it a separate
    function is that usually there are a lot of trace messages, so it might
    make sense to separate them from other debug messages.

    Trace messages can be separated into different categories; these functions in facts
    only log the message if the given @a mask is currently enabled in wxLog.
    This lets you selectively trace only some operations and not others by enabling the
    desired trace masks with wxLog::AddTraceMask() or by setting the
    @ref overview_envvars "@c WXTRACE environment variable".

    The predefined string trace masks used by wxWidgets are:

    @beginDefList
    @itemdef{ wxTRACE_MemAlloc, Trace memory allocation (new/delete) }
    @itemdef{ wxTRACE_Messages, Trace window messages/X callbacks }
    @itemdef{ wxTRACE_ResAlloc, Trace GDI resource allocation }
    @itemdef{ wxTRACE_RefCount, Trace various ref counting operations }
    @itemdef{ wxTRACE_OleCalls, Trace OLE method calls (Win32 only) }
    @endDefList

    @header{wx/log.h}
*/
void wxLogTrace(const char* mask, const char* formatString, ... );
void wxVLogTrace(const char* mask, const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    Like wxLogDebug(), trace functions only do something in debug builds and
    expand to nothing in the release one. The reason for making it a separate
    function is that usually there are a lot of trace messages, so it might
    make sense to separate them from other debug messages.

    @deprecated
    This version of wxLogTrace() only logs the message if all the bits
    corresponding to the @a mask are set in the wxLog trace mask which can be
    set by calling wxLog::SetTraceMask(). This version is less flexible than
    wxLogTrace(const char*,const char*,...) because it doesn't allow defining
    the user trace masks easily. This is why it is deprecated in favour of
    using string trace masks.

    The following bitmasks are defined for wxTraceMask:

    @beginDefList
    @itemdef{ wxTraceMemAlloc, Trace memory allocation (new/delete) }
    @itemdef{ wxTraceMessages, Trace window messages/X callbacks }
    @itemdef{ wxTraceResAlloc, Trace GDI resource allocation }
    @itemdef{ wxTraceRefCount, Trace various ref counting operations }
    @itemdef{ wxTraceOleCalls, Trace OLE method calls (Win32 only) }
    @endDefList

    @header{wx/log.h}
*/
void wxLogTrace(wxTraceMask mask, const char* formatString, ... );
void wxVLogTrace(wxTraceMask mask, const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    The right functions for debug output. They only do something in debug mode
    (when the preprocessor symbol @c \__WXDEBUG__ is defined) and expand to
    nothing in release mode (otherwise).

    @header{wx/log.h}
*/
void wxLogDebug(const char* formatString, ... );
void wxVLogDebug(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    Messages logged by this function will appear in the statusbar of the
    @a frame or of the top level application window by default (i.e. when using
    the second version of the functions).

    If the target frame doesn't have a statusbar, the message will be lost.

    @header{wx/log.h}
*/
void wxLogStatus(wxFrame* frame, const char* formatString, ... );
void wxVLogStatus(wxFrame* frame, const char* formatString, va_list argPtr);
void wxLogStatus(const char* formatString, ... );
void wxVLogStatus(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_log */
///@{
/**
    Mostly used by wxWidgets itself, but might be handy for logging errors
    after system call (API function) failure. It logs the specified message
    text as well as the last system error code (@e errno or @e GetLastError()
    depending on the platform) and the corresponding error message. The second
    form of this function takes the error code explicitly as the first
    argument.

    @see wxSysErrorCode(), wxSysErrorMsgStr()

    @header{wx/log.h}
*/
void wxLogSysError(const char* formatString, ... );
void wxVLogSysError(const char* formatString, va_list argPtr);
///@}

/** @addtogroup group_funcmacro_debug */
///@{

/**
    @def wxDISABLE_DEBUG_LOGGING_IN_RELEASE_BUILD()

    Use this macro to disable logging at debug and trace levels in release
    build when not using wxIMPLEMENT_APP().

    @see wxDISABLE_DEBUG_SUPPORT(),
         wxDISABLE_ASSERTS_IN_RELEASE_BUILD(),
         @ref overview_debugging

    @since 2.9.1

    @header{wx/log.h}
 */
#define wxDISABLE_DEBUG_LOGGING_IN_RELEASE_BUILD()

///@}

#endif // wxUSE_BASE
